﻿namespace G3dCore.Entities
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using nw.g3d.nw4f_3dif;
    using Opal.Security.Cryptography;

    /// <summary>
    /// シェーダーソース定義のエンティティクラスです。
    /// </summary>
    public class ShaderSrc : Entity<shader_srcType>
    {
        private int srcIndex = 0;
        private string includePath = string.Empty;
        private string path = string.Empty;
        private int streamIndex = 0;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public ShaderSrc ()
        {
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="data">設定するデータです。</param>
        public ShaderSrc(shader_srcType data)
            : this()
        {
            this.srcIndex = data.src_index;
            this.includePath = data.include_path;
            this.path = data.path;
            this.streamIndex = data.stream_index;
        }

        /// <summary>
        /// インデックスを取得設定します。
        /// </summary>
        public int SrcIndex
        {
            get
            {
                return this.srcIndex;
            }

            set
            {
                this.SetProperty(ref this.srcIndex, value);
            }
        }

        /// <summary>
        /// シェーダソースのインクルードパスを取得設定します。
        /// </summary>
        public string IncludePath
        {
            get
            {
                return this.includePath;
            }

            set
            {
                this.SetProperty(ref this.includePath, value ?? string.Empty, () => this.CalcCRC());
            }
        }

        /// <summary>
        /// シェーダソースのインクルードパスからの相対パスを取得設定します。
        /// </summary>
        public string Path
        {
            get
            {
                return this.path;
            }

            set
            {
                this.SetProperty(ref this.path, value ?? string.Empty, () => this.CalcCRC());
            }
        }

        /// <summary>
        /// stream のインデックスを取得設定します。
        /// </summary>
        public int StreamIndex
        {
            get
            {
                return this.streamIndex;
            }

            set
            {
                this.SetProperty(ref this.streamIndex, value, () => this.CalcCRC());
            }
        }

        /// <summary>
        /// 現在のインスタンスのコピーである新しいオブジェクトを作成します。
        /// </summary>
        /// <returns>現在のインスタンスのコピーである新しいオブジェクトを返します。</returns>
        public override IEntity Clone()
        {
            var instance = new ShaderSrc();
            instance.srcIndex = this.srcIndex;
            instance.includePath = this.includePath;
            instance.path = this.path;
            instance.streamIndex = this.streamIndex;

            return instance;
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override shader_srcType CreateWriteData()
        {
            var instance = new shader_srcType();
            instance.src_index = this.srcIndex;
            instance.include_path = this.includePath;
            instance.path = this.path;
            instance.stream_index = this.streamIndex;

            return instance;
        }

        /// <summary>
        /// エンティティのCRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRCの値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.srcIndex)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.includePath)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.path)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.streamIndex)));

            var hashValue = crc.ComputeHashUInt32(buffers.ToArray());
            return hashValue;
        }

    }
}
