﻿namespace ShaderAssistAddons.Modules.ShaderConfig.ViewModels
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using G3dCore.Messaging;
    using G3dCore.Resources;
    using G3dCore.Windows.Actions;
    using Opal.ViewModels;
    using ShaderAssistAddons.Modules.ShaderConfig.Commands;
    using ShaderAssistAddons.Modules.ShaderConfig.Utilities;
    using ShaderAssistAddons.Resources;

    /// <summary>
    /// シェーダソースファイルの設定を行うビューモデルです。
    /// </summary>
    public class ShaderSourceSettingViewModel : ViewModel
    {
        public enum EnumShaderType
        {
            Vertex,
            Geometry,
            Fragment,
            Compute
        }

        public EnumShaderType ShaderType { get; protected set; }

        ErrorMessageTextBlockViewModel _errorShaderSourceViewModel = new ErrorMessageTextBlockViewModel();

        string _includePath;
        string _shaderSource;
        string _shaderSourceFullPath;

        DelegateCommand _refCommand;

        Messenger _openFileDialogMessenger = new Messenger();

        //---------------------------------------------------------------------

        public string IncludePath
        {
            get { return _includePath; }
            protected set
            {
                if (value != _includePath)
                {
                    _includePath = value;
                    this.RaisePropertyChanged("IncludePath");
                }
            }
        }

        public string ShaderSource
        {
            get { return _shaderSource; }
            set
            {
                string fileName = System.IO.Path.GetFileName(value);
                string fullPath = null;
                string includePath = null;
                string relativePath = null;

                bool result =
                    ShaderConfigUtility.CheckShaderPath(
                        fileName,
                        System.IO.Path.GetDirectoryName(ShaderConfigPath),
                        IncludeFullPaths,
                        out fullPath,
                        out includePath,
                        out relativePath);
                if (result)
                {
                    IncludePath = includePath + "\\";
                    ShaderSourceFullPath = fullPath;
                }
                else
                {
                    IncludePath = string.Empty;
                    ShaderSourceFullPath = string.Empty;
                }

                ErrorShaderSourceViewModel.IsShown = !result;
                if (value == string.Empty)
                {
                    ErrorShaderSourceViewModel.IsShown = false;
                }

                if (relativePath != null)
                {
                    if (relativePath != _shaderSource)
                    {
                        _shaderSource = relativePath;
                        this.RaisePropertyChanged("ShaderSource");
                    }
                }
                else
                {
                    if (value != _shaderSource)
                    {
                        _shaderSource = value;
                        this.RaisePropertyChanged("ShaderSource");
                    }
                }
            }
        }

        public string ShaderSourceFullPath
        {
            get { return _shaderSourceFullPath; }
            protected set
            {
                if (value != _shaderSourceFullPath)
                {
                    _shaderSourceFullPath = value;
                    this.RaisePropertyChanged("ShaderSourceFullPath");
                }
            }
        }

        public string Label { get; set; }

        string ShaderConfigPath { get; set; }
        string[] IncludeFullPaths { get; set; }

        public ErrorMessageTextBlockViewModel ErrorShaderSourceViewModel
        {
            get { return _errorShaderSourceViewModel; }
        }

        //---------------------------------------------------------------------

        public DelegateCommand RefCommand
        {
            get { return this._refCommand; }
        }

        //---------------------------------------------------------------------

        public Messenger OpenFileDialogMessenger
        {
            get { return this._openFileDialogMessenger; }
        }

        //---------------------------------------------------------------------

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public ShaderSourceSettingViewModel(EnumShaderType type, string shaderConfigPath, string[] includeFullPaths)
        {
            _refCommand = new DelegateCommand(RefExecute);
            ShaderType = type;
            ShaderConfigPath = shaderConfigPath;
            IncludeFullPaths = includeFullPaths;
            ShaderSourceFullPath = string.Empty;

            switch (type)
            {
                case EnumShaderType.Vertex:
                    Label = ShaderConfigMessage.ShaderSourceSettingVertex;
                    break;
                case EnumShaderType.Geometry:
                    Label = ShaderConfigMessage.ShaderSourceSettingGeometry;
                    break;
                case EnumShaderType.Fragment:
                    Label = ShaderConfigMessage.ShaderSourceSettingFragment;
                    break;
                case EnumShaderType.Compute:
                    Label = ShaderConfigMessage.ShaderSourceSettingCompute;
                    break;
            }

            ErrorShaderSourceViewModel.Message = ShaderConfigMessage.WizardShaderSettingShaderPathError;
        }

        /// <summary>
        /// パスを追加します。
        /// </summary>
        public void AddPath(string path)
        {
            if (System.IO.File.Exists(path))
            {
                ShaderSource = path;
            }
        }

        //---------------------------------------------------------------------

        /// <summary>
        /// 参照ボタン
        /// </summary>
        private void RefExecute()
        {
            OpenFileDialogAction.InputArg argInput = new OpenFileDialogAction.InputArg
            {
                Title = ShaderConfigMessage.ChooseShaderSource,
                Filter = Labels.FilterShaderSource,
                FilterIndex = 2,
                InitialDirectory = System.IO.Path.GetDirectoryName(ShaderConfigPath)
            };
            OpenFileDialogMessenger.Raise(
                new Message(argInput),
                m =>
                {
                    OpenFileDialogAction.OutputArg argOutput = (OpenFileDialogAction.OutputArg)m.Response;

                    if (argOutput.Result)
                    {
                        this.ShaderSource = argOutput.FileName;
                    }
                }
            );
        }

    }
}
