﻿namespace ShaderAssistAddons.Modules.ShaderConfig.ViewModels
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using G3dCore.Converters;
    using G3dCore.Entities;
    using G3dCore.Extensions;
    using G3dCore.Modules.ErrorView;
    using G3dCore.Resources;
    using ShaderAssistAddons.Modules.ShaderConfig.Commands;
    using ShaderAssistAddons.Resources;

    /// <summary>
    /// シェーダコンバータを実行するビューモデルです。
    /// </summary>
    public class ShaderConverterViewModel : WizardPageViewModel
    {
        private bool isCheckOnly;
        private bool isProcessing;
        private bool isConvertCheckOK;
        private int progressBarValue;
        private string status = string.Empty;
        private ShaderConvertExecutor executor = null;
        private ShaderConverterResult result = null;
        private Shader convertedShader = null;
        private ErrorViewLogger logger = null;

        /// <summary>
        /// 処理中かどうかを取得します。
        /// </summary>
        public bool IsProcessing
        {
            get
            {
                return this.isProcessing;
            }

            protected set
            {
                if (value != this.isProcessing)
                {
                    this.isProcessing = value;
                    this.RaisePropertyChanged();
                    this.RaisePropertyChanged(() => this.ShaderConvertContent);
                }
            }
        }

        /// <summary>
        /// コンバートチェックで問題なかったかを取得します。
        /// </summary>
        public bool IsConvertCheckOK
        {
            get
            {
                return this.isConvertCheckOK;
            }

            protected set
            {
                if (value != this.isConvertCheckOK)
                {
                    this.isConvertCheckOK = value;
                    this.RaisePropertyChanged();

                    this.ProgressBarValue = this.isConvertCheckOK ? 100 : 0;
                }
            }
        }

        /// <summary>
        /// ヒントメッセージを取得します。
        /// </summary>
        public string Hint
        {
            get
            {
                return this.isCheckOnly ?
                    ShaderConfigMessage.WizardShaderConvertHintCheck : ShaderConfigMessage.WizardShaderConvertHintExtractVariation;
            }
        }

        /// <summary>
        /// シェーダコンバート実行ボタンのコンテントを取得します。
        /// </summary>
        public string ShaderConvertContent
        {
            get
            {
                return this.IsProcessing ?
                    Labels.Converting : Labels.ExecuteConvert;
            }
        }

        /// <summary>
        /// プログレスバーの値を取得します。
        /// </summary>
        public int ProgressBarValue
        {
            get
            {
                return this.progressBarValue;
            }

            protected set
            {
                this.SetProperty(ref this.progressBarValue, value);
            }
        }

        /// <summary>
        /// シェーダコンバート実行コマンドを取得します。
        /// </summary>
        public DelegateCommand ShaderConvertCommand
        {
            get
            {
                return new DelegateCommand(this.ShaderConvertCommandExecute, this.CanShaderConvert);
            }
        }

        /// <summary>
        /// ログを開く実行コマンドを取得します。
        /// </summary>
        public DelegateCommand OpenLogCommand
        {
            get
            {
                return new DelegateCommand(this.OpenLogCommandExecute, this.CanOpenLog);
            }
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="wizardViewModel">ウィザードビューモデルです。</param>
        /// <param name="isCheckOnly">チェックのみを行う場合 true を指定します。</param>
        public ShaderConverterViewModel(WizardViewModel wizardViewModel, bool isCheckOnly)
            : base(wizardViewModel)
        {
            this.isCheckOnly = isCheckOnly;

            if (isCheckOnly)
            {
                this.Description = ShaderConfigMessage.WizardDescriptionCheck;
            }
            else
            {
                this.Description = ShaderConfigMessage.WizardDescriptionExtractVariation;
            }
        }

        /// <summary>
        /// 状態をリセットします。
        /// </summary>
        public void Reset()
        {
            this.IsConvertCheckOK = false;

            if (this.logger != null)
            {
                this.logger.Dispose();
                this.logger = null;
            }
        }

        /// <summary>
        /// 次ページに進めるかを取得します。
        /// </summary>
        /// <returns>進める場合 true を返します。</returns>
        public override bool CanGoNext()
        {
            return !this.IsProcessing && this.IsConvertCheckOK;
        }

        /// <summary>
        /// 前ページに進めるかを取得します。
        /// </summary>
        /// <returns>進める場合 true を返します。</returns>
        public override bool CanGoBack()
        {
            return !this.IsProcessing;
        }

        /// <summary>
        /// コンバート結果のシェーダ定義を取得します。
        /// </summary>
        /// <returns>シェーダ定義を返します。</returns>
        public ShaderDefinition GetConvertedShaderDefinition()
        {
            if (this.result != null)
            {
                return this.result.GetShaderDefinition();
            }

            return null;
        }

        /// <summary>
        /// コンバートに使用した Shader を取得します。
        /// </summary>
        /// <returns>Shader を返します。</returns>
        public Shader GetConvertedShader()
        {
            return this.convertedShader;
        }

        private void ShaderConvertCommandExecute()
        {
            this.IsProcessing = true;

            if (!this.isCheckOnly)
            {
                this.GenerateShaderDefinitionFromEditShader();
            }
            else
            {
                this.TryShaderConvertFromShaderSetting();
            }
        }

        private bool CanShaderConvert()
        {
            return !this.IsProcessing;
        }

        private void OpenLogCommandExecute()
        {
            this.logger.Show();
        }

        private bool CanOpenLog()
        {
            return this.logger != null;
        }

        private void GenerateShaderDefinitionFromEditShader()
        {
            var viewModel = (WizardShaderViewModel)this.WizardViewModel;

            G3dFile shaderConfigFile = new G3dFile(G3dKind.ShaderConfig);
            ShaderConfig shaderConfig = shaderConfigFile.GetRootEntity<ShaderConfig>();
            shaderConfig.CreateEmptyShaderConfig(viewModel.ShaderConfigViewModel.ShaderConfigData);

            this.convertedShader = this.CreateEditShader();
            shaderConfig.Shaders.Add(this.convertedShader);

            this.executor = new ShaderConvertExecutor();
            this.executor.BeginTryShaderConvert(shaderConfigFile, viewModel.FileDir, this.ShaderConvertResult);
        }

        private void TryShaderConvertFromShaderSetting()
        {
            var viewModel = (WizardShaderConfigViewModel)this.WizardViewModel;
            var shaderConfigFile = viewModel.ShaderConfigViewModel.FileViewModel.File.Clone() as G3dFile;

            var shaderConfig = shaderConfigFile.GetRootEntity<ShaderConfig>();
            shaderConfig.ShaderConfigInfo.CodePage = viewModel.CodePage;
            shaderConfig.SetIncludePaths(viewModel.GetIncludePaths(viewModel.FileDir));
            shaderConfig.SetForceIncludeFiles(viewModel.GetForceIncludeFiles());

            // 各シェーダファイルパスを設定され直したインクルードパス相対に直す。
            viewModel.CreateAdjustShaderPathOperation(shaderConfig.Shaders).Execute();

            this.executor = new ShaderConvertExecutor();
            this.executor.BeginTryShaderConvert(shaderConfigFile, viewModel.FileDir, this.ShaderConvertResult);
        }

        private void ShaderConvertResult(bool isSuccessful, ShaderConverterResult result)
        {
            this.result = result;
            this.IsConvertCheckOK = isSuccessful;

            ErrorViewLog log = result.CreateErrorViewLog();
            this.logger = new ErrorViewLogger();
            this.logger.Add(log);
            this.logger.Show();

            this.executor = null;
            this.IsProcessing = false;
            this.ShaderConvertCommand.RaiseCanExecuteChanged();
        }

        private Shader CreateEditShader()
        {
            var viewModel = (WizardShaderViewModel)this.WizardViewModel;

            var shader = new Shader();
            shader.Name = viewModel.ShaderName;
            shader.MaterialShader = viewModel.IsMaterialShader;
            shader.VertexShader.Path = viewModel.VertexShaderPath;
            shader.GeometryShader.Path = viewModel.GeometryShaderPath;
            shader.FragmentShader.Path = viewModel.FragmentShaderPath;
            shader.ComputeShader.Path = viewModel.ComputeShaderPath;

            foreach (var macroViewModel in viewModel.MacroViewModels)
            {
                var macro = new Macro()
                {
                    Name = macroViewModel.Name,
                    Value = macroViewModel.Value
                };

                shader.Macros.Add(macro);
            }

            return shader;
        }
    }
}
