﻿namespace ShaderAssistAddons.Modules.ShaderConfig.ViewModels
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using G3dCore.IO;
    using G3dCore.Messaging;
    using G3dCore.Modules.ErrorView;
    using G3dCore.Resources;
    using G3dCore.Windows.Actions;
    using ShaderAssistAddons.Modules.ShaderConfig.Utilities;
    using ShaderAssistAddons.Resources;

    /// <summary>
    /// 共通設定２を行うビューモデルです。
    /// </summary>
    public class PathSettingViewModel : WizardPageViewModel
    {
        private readonly EditableListBoxViewModel includePathListViewModel = new EditableListBoxViewModel(true);
        private readonly EditableListBoxViewModel forceIncludeListViewModel = new EditableListBoxViewModel(false);
        private readonly Messenger openFileDialogMessenger = new Messenger();
        private readonly Messenger folderBrowserDialogMessenger = new Messenger();
        private ErrorViewLogger errorViewLogger = null;

        /// <summary>
        /// インクルードパスリストボックスビューモデルを取得します。
        /// </summary>
        public EditableListBoxViewModel IncludePathListViewModel
        {
            get
            {
                return this.includePathListViewModel;
            }
        }

        /// <summary>
        /// 強制インクルードファイルリストボックスビューモデルを取得します。
        /// </summary>
        public EditableListBoxViewModel ForceIncludeListViewModel
        {
            get
            {
                return this.forceIncludeListViewModel;
            }
        }

        /// <summary>
        /// オープンファイルダイアログ表示メッセンジャーを取得します。
        /// </summary>
        public Messenger OpenFileDialogMessenger
        {
            get
            {
                return this.openFileDialogMessenger;
            }
        }

        /// <summary>
        /// フォルダ選択ダイアログ表示メッセンジャーを取得します。
        /// </summary>
        public Messenger FolderBrowserDialogMessenger
        {
            get
            {
                return this.folderBrowserDialogMessenger;
            }
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="wizardViewModel">ウィザードビューモデルです。</param>
        public PathSettingViewModel(WizardViewModel wizardViewModel)
            : base(wizardViewModel)
        {
            this.Description = ShaderConfigMessage.WizardDescriptionPath;

            this.IncludePathListViewModel.OnAdding = this.CheckIncludePaths;
            this.IncludePathListViewModel.AddExecute = this.AddIncludePathExecute;
            this.IncludePathListViewModel.ItemBeforeEdit = this.BeforeEditIncludePath;
            this.IncludePathListViewModel.ItemAfterEdit = this.AfterEditIncludePath;
            this.IncludePathListViewModel.OnListCollectionChanged = this.OnIncludePathsListCollectionChanged;

            this.ForceIncludeListViewModel.OnAdding = this.CheckForceIncludeFiles;
            this.ForceIncludeListViewModel.AddExecute = this.AddForceIncludeFileExecute;
        }

        /// <summary>
        /// リストボックスを構築します。
        /// </summary>
        public void BuildListBox()
        {
            this.IncludePathListViewModel.ClearItems();
            this.ForceIncludeListViewModel.ClearItems();

            if (this.GetParentViewModel().IncludeFullPaths != null)
            {
                this.IncludePathListViewModel.AddToList(this.GetParentViewModel().IncludeFullPaths);
                this.ForceIncludeListViewModel.AddToList(this.GetParentViewModel().ForceIncludeFilePaths);
            }
        }

        /// <summary>
        /// 次ページに進めるかを取得します。
        /// </summary>
        /// <returns>進める場合 true を返します。</returns>
        public override bool CanGoNext()
        {
            return this.IncludePathListViewModel.ListBoxItems.Count > 0 &&
                   !this.IncludePathListViewModel.ListBoxItems.Any(x => x.HasError) &&
                   !this.IncludePathListViewModel.IsEditing &&
                   !this.ForceIncludeListViewModel.ListBoxItems.Any(x => x.HasError);
        }

        /// <summary>
        /// 前ページに進めるかを取得します。
        /// </summary>
        /// <returns>進める場合 true を返します。</returns>
        public override bool CanGoBack()
        {
            return !this.IncludePathListViewModel.IsEditing;
        }

        private string BeforeEditIncludePath(EditableListBoxItemViewModel item)
        {
            return item.Tag != null ? item.Tag.ToString() : string.Empty;
        }

        private void AfterEditIncludePath(EditableListBoxItemViewModel item, string editText)
        {
            item.Label = editText;
            item.Tag = null;
            this.CheckIncludePaths(item);
        }

        private void OnIncludePathsListCollectionChanged()
        {
            this.ForceIncludeListViewModel.CanAdd = IncludePathListViewModel.ListBoxItems.Count > 0;
            this.CheckIncludePaths(this.ForceIncludeListViewModel.ListBoxItems.ToArray());
        }

        private void AddIncludePathExecute()
        {
            var input = new FolderBrowserDialogAction.InputArg
            {
                SelectedPath = Path.GetDirectoryName(this.GetParentViewModel().FilePath),
                Description = Labels.ChooseIncludePath
            };

            this.FolderBrowserDialogMessenger.Raise(
                new Message(input),
                m =>
                {
                    var output = (FolderBrowserDialogAction.OutputArg)m.Response;

                    if (output.Result)
                    {
                        this.errorViewLogger = new ErrorViewLogger();
                        this.IncludePathListViewModel.AddToList(output.SelectedPath);
                    }
                }
            );
        }

        private void CheckIncludePaths(params EditableListBoxItemViewModel[] items)
        {
            Debug.Assert(!string.IsNullOrEmpty(this.GetParentViewModel().FilePath));

            using (ErrorViewLogger logger = this.errorViewLogger)
            {
                foreach (EditableListBoxItemViewModel item in items)
                {
                    // フルパスを保存
                    item.Tag = item.Label;

                    if (File.Exists(item.Label))
                    {
                        item.Label = Path.GetDirectoryName(item.Label);
                        item.Tag = item.Label;
                    }

                    if (!Directory.Exists(PathUtility.GetExpandEnvironmentVariable(item.Label)))
                    {
                        item.HasError = true;
                        continue;
                    }

                    if (this.CheckIncludePath(item, logger))
                    {
                        if (!item.Label.Contains('%'))
                        {
                            item.Label = PathUtility.MakeRelativePath(
                                Path.GetDirectoryName(this.GetParentViewModel().FilePath), item.Label);
                        }

                        Debug.Assert(item.Label != string.Empty);
                        item.HasError = false;
                    }
                    else
                    {
                        item.HasError = true;
                        if (logger != null)
                        {
                            item.Label = string.Empty;
                        }
                    }
                }
            }

            this.errorViewLogger = null;
        }

        private bool CheckIncludePath(EditableListBoxItemViewModel item, ErrorViewLogger errLogger)
        {
            // シェーダソースのインクルードパスのチェック。
            string path = PathUtility.GetExpandEnvironmentVariable(item.Label);
            if (path != null)
            {
                if (!this.CheckIncludePathForFscOutput(path))
                {
                    this.AddErrorLog(errLogger, ShaderConfigMessage.ErrorIncludePathUnmatchedFscDir, path);
                    return false;
                }

                if (!this.CheckDuplicatedIncludePath(path, item))
                {
                    this.AddErrorLog(errLogger, ShaderConfigMessage.ErrorIncludePathDuplicated, path);
                    return false;
                }
            }
            else
            {
                this.AddErrorLog(errLogger, ShaderConfigMessage.ErrorIncludePathCannotExpandEnvironmentVariable, path);
                return false;
            }

            return true;
        }

        private bool CheckIncludePathForFscOutput(string includePath)
        {
            // path が fsc 出力先に対してインクルードパスとして問題ないかチェック
            string shaderConfigDir = Path.GetDirectoryName(this.GetParentViewModel().FilePath);

            // fsc 出力先から上位ディレクトリは NG。
            if (shaderConfigDir.StartsWith(includePath) && shaderConfigDir != includePath)
            {
                return false;
            }

            return true;
        }

        private bool CheckDuplicatedIncludePath(string path, EditableListBoxItemViewModel srcItem)
        {
            // path が既に追加されているインクルードパスと競合しないかをチェック
            // 既に追加したパスに包含されている（またはしている）場合は NG。
            bool isFound = this.IncludePathListViewModel.ListBoxItems.Any(
                delegate(EditableListBoxItemViewModel item)
                {
                    if (item == srcItem || item.HasError)
                    {
                        return false;
                    }

                    string fullpath = PathUtility.GetExpandEnvironmentVariable(item.Tag.ToString());

                    if (fullpath.StartsWith(path) || path.StartsWith(fullpath))
                    {
                        return true;
                    }

                    return false;
                }
            );

            return !isFound;
        }

        private void AddForceIncludeFileExecute()
        {
            var input = new OpenFileDialogAction.InputArg
            {
                Title = Labels.ChooseShaderSource,
                Filter = Labels.FilterShaderSource,
                FilterIndex = 2,
                InitialDirectory = Path.GetDirectoryName(this.GetParentViewModel().FilePath)
            };

            this.OpenFileDialogMessenger.Raise(
                new Message(input),
                m =>
                {
                    var output = (OpenFileDialogAction.OutputArg)m.Response;

                    if (output.Result)
                    {
                        this.errorViewLogger = new ErrorViewLogger();
                        this.ForceIncludeListViewModel.AddToList(output.FileName);
                    }
                }
            );
        }

        private void CheckForceIncludeFiles(EditableListBoxItemViewModel[] items)
        {
            Debug.Assert(!string.IsNullOrEmpty(this.GetParentViewModel().FilePath));

            var includeFullPaths = this.IncludePathListViewModel.ListBoxItems.Select(x => x.Tag.ToString());

            using (ErrorViewLogger logger = this.errorViewLogger)
            {
                foreach (EditableListBoxItemViewModel item in items)
                {
                    string fileName = item.Label;
                    if (item.Tag == null)
                    {
                        // 初回追加時には Label にフルパスが入っていて Tag が null。
                        fileName = Path.GetFileName(item.Label);

                        // フルパスを保存
                        item.Tag = item.Label;
                        item.Label = fileName;
                    }

                    string fullPath = null;
                    string includePath = null;
                    string relativePath = null;

                    if (ShaderConfigUtility.CheckShaderPath(
                            fileName,
                            Path.GetDirectoryName(this.GetParentViewModel().FilePath),
                            includeFullPaths.ToArray(),
                            out fullPath,
                            out includePath,
                            out relativePath)
                       )
                    {
                        if (this.ForceIncludeListViewModel.ListBoxItems.Any(x => x != item && x.Label == item.Label))
                        {
                            // 既に追加されている。
                            if (logger != null)
                            {
                                this.AddErrorLog(logger, ShaderConfigMessage.ErrorForceIncludeFileDuplicated, item.Tag.ToString());
                                item.Label = string.Empty;
                            }
                            else
                            {
                                item.HasError = true;
                            }
                        }
                        else
                        {
                            item.HasError = false;
                        }
                    }
                    else
                    {
                        // インクルードパス内に含まれていない。
                        if (logger != null)
                        {
                            this.AddErrorLog(logger, ShaderConfigMessage.ErrorForceIncludeFileBadFile, item.Tag.ToString());
                            item.Label = string.Empty;
                        }
                        else
                        {
                            item.HasError = true;
                        }
                    }
                }
            }

            this.errorViewLogger = null;
        }

        private void AddErrorLog(ErrorViewLogger logger, string summary, string details = null)
        {
            if (logger != null)
            {
                var log = new ErrorViewLog()
                {
                    Category = ErrorViewLog.CategoryType.Error,
                    Summary = summary,
                    Details = details ?? string.Empty
                };

                logger.Add(log);
            }
        }

        private WizardShaderConfigViewModel GetParentViewModel()
        {
            return this.WizardViewModel as WizardShaderConfigViewModel;
        }
    }
}
