﻿namespace ShaderAssistAddons.Modules.ShaderConfig.ViewModels
{
    using System;
    using System.Collections.Generic;
    using System.ComponentModel;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using System.Windows.Data;
    using G3dCore.Configurations;
    using G3dCore.Entities;
    using Opal.App;
    using Opal.ViewModels;
    using ShaderAssistAddons.Modules.ShaderConfig.Commands;

    /// <summary>
    /// シェーダソース一覧のビューモデルです。
    /// </summary>
    public class ShaderSourceListViewModel : ViewModel
    {
        /// <summary>
        /// 列のタイプです。
        /// </summary>
        public enum ColumnTypes
        {
            /// <summary>
            /// ファイルの種類です。
            /// </summary>
            FileType,

            /// <summary>
            /// インクルードパスです。
            /// </summary>
            IncludePath,

            /// <summary>
            /// ファイルパスです。
            /// </summary>
            FilePath,

            /// <summary>
            /// 最大値
            /// </summary>
            MaxCount
        }

        private enum SortDescriptionTypes
        {
            Default,
            Ascending,
            Descending,
            MaxCount
        }

        private ShaderSourceListItemViewModel[] shaderSourceListItems;
        private ShaderSourceListItemViewModel selectedShaderSourceListItem = null;
        private string strShaderSource = string.Empty;
        private SortDescriptionTypes[] sortDescriptions = new SortDescriptionTypes[(int)ColumnTypes.MaxCount];
        private ShaderConfigViewModel currentShaderConfigViewModel = null;

        /// <summary>
        /// シェーダソースリストを取得設定します。
        /// </summary>
        public ShaderSourceListItemViewModel[] ShaderSourceListItems
        {
            get
            {
                return this.shaderSourceListItems;
            }

            set
            {
                this.shaderSourceListItems = value;
                this.RaisePropertyChanged();
            }
        }

        /// <summary>
        /// 選択シェーダソースリストを取得設定します。
        /// </summary>
        public ShaderSourceListItemViewModel SelectedShaderSourceListItem
        {
            get
            {
                return this.selectedShaderSourceListItem;
            }

            set
            {
                if (value != this.selectedShaderSourceListItem)
                {
                    this.selectedShaderSourceListItem = value;
                    this.ReadShaderSourceText(this.selectedShaderSourceListItem);
                    this.RaisePropertyChanged();
                }
            }
        }

        /// <summary>
        /// シェーダソースを取得設定します。
        /// </summary>
        public string StrShaderSource
        {
            get
            {
                return this.strShaderSource;
            }

            set
            {
                this.SetProperty(ref this.strShaderSource, value);
            }
        }

#region DelegateCommands

        /// <summary>
        /// シェーダソースを開く実行時コマンドを取得します。
        /// </summary>
        public DelegateCommand OpenShaderSourceCommand
        {
            get
            {
                return new DelegateCommand(this.OpenShaderSourceExecute, this.CanOpenShaderSource);
            }
        }

        /// <summary>
        /// インクルードパスを開く実行時コマンドを取得します。
        /// </summary>
        public DelegateCommand OpenIncludePathCommand
        {
            get
            {
                return new DelegateCommand(this.OpenIncludePathExecute, this.CanOpenShaderSource);
            }
        }

        /// <summary>
        /// ファイルパスを開く実行時コマンドを取得します。
        /// </summary>
        public DelegateCommand OpenFilePathCommand
        {
            get
            {
                return new DelegateCommand(this.OpenFilePathExecute, this.CanOpenShaderSource);
            }
        }

#endregion

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public ShaderSourceListViewModel()
        {
        }

        /// <summary>
        /// シェーダビューモデルを設定します。
        /// </summary>
        /// <param name="shaderViewModel">シェーダビューモデルです。</param>
        /// <param name="shaderConfigViewModel">シェーダ設定ビューモデルです。</param>
        public void SetCurrentShaderViewModel(ShaderViewModel shaderViewModel, ShaderConfigViewModel shaderConfigViewModel)
        {
            this.currentShaderConfigViewModel = shaderConfigViewModel;
            if (shaderViewModel == null)
            {
                this.ShaderSourceListItems = null;
                return;
            }

            var items = new List<ShaderSourceListItemViewModel>();

            if (!string.IsNullOrWhiteSpace(shaderViewModel.ShaderData.VertexShader.Path))
            {
                ShaderSourceListItemViewModel data = new ShaderSourceListItemViewModel()
                {
                    FileType = ShaderSourceListItemViewModel.ItemTypes.VertexShader,
                    FilePath = shaderViewModel.ShaderData.VertexShader.Path
                };

                items.Add(data);
            }

            if (!string.IsNullOrWhiteSpace(shaderViewModel.ShaderData.GeometryShader.Path))
            {
                ShaderSourceListItemViewModel data = new ShaderSourceListItemViewModel()
                {
                    FileType = ShaderSourceListItemViewModel.ItemTypes.GeometryShader,
                    FilePath = shaderViewModel.ShaderData.GeometryShader.Path
                };

                items.Add(data);
            }

            if (!string.IsNullOrWhiteSpace(shaderViewModel.ShaderData.FragmentShader.Path))
            {
                ShaderSourceListItemViewModel data = new ShaderSourceListItemViewModel()
                {
                    FileType = ShaderSourceListItemViewModel.ItemTypes.FragmentShader,
                    FilePath = shaderViewModel.ShaderData.FragmentShader.Path
                };

                items.Add(data);
            }

            if (!string.IsNullOrWhiteSpace(shaderViewModel.ShaderData.ComputeShader.Path))
            {
                ShaderSourceListItemViewModel data = new ShaderSourceListItemViewModel()
                {
                    FileType = ShaderSourceListItemViewModel.ItemTypes.ComputeShader,
                    FilePath = shaderViewModel.ShaderData.ComputeShader.Path
                };

                items.Add(data);
            }

            if (shaderViewModel.ShaderDefinitionData != null)
            {
                foreach (ShaderSrc shaderSrc in shaderViewModel.ShaderDefinitionData.ShaderSrcs)
                {
                    string fileName = Path.GetFileName(shaderSrc.Path);

                    {
                        var listData = items.FindAll(x => Path.GetFileName(x.FilePath) == fileName);
                        if (listData.Count > 0)
                        {
                            foreach (var data in listData)
                            {
                                if (data.FileType == ShaderSourceListItemViewModel.ItemTypes.VertexShader ||
                                    data.FileType == ShaderSourceListItemViewModel.ItemTypes.GeometryShader ||
                                    data.FileType == ShaderSourceListItemViewModel.ItemTypes.FragmentShader ||
                                    data.FileType == ShaderSourceListItemViewModel.ItemTypes.ComputeShader)
                                {
                                    data.IncludePath = shaderSrc.IncludePath;
                                }
                            }
                            continue;
                        }
                    }

                    if (shaderConfigViewModel.ShaderConfigData.ForceIncludeFiles.FirstOrDefault(
                        x => Path.GetFileName(x.Path) == fileName) != null)
                    {
                        var data = new ShaderSourceListItemViewModel()
                        {
                            FileType = ShaderSourceListItemViewModel.ItemTypes.ForceIncludeFile,
                            IncludePath = shaderSrc.IncludePath,
                            FilePath = shaderSrc.Path
                        };

                        items.Add(data);
                        continue;
                    }

                    {
                        var data = new ShaderSourceListItemViewModel()
                        {
                            FileType = ShaderSourceListItemViewModel.ItemTypes.IncludeFile,
                            IncludePath = shaderSrc.IncludePath,
                            FilePath = shaderSrc.Path
                        };

                        items.Add(data);
                    }
                }

                this.ShaderSourceListItems = items.ToArray();
            }
            else
            {
                this.ShaderSourceListItems = null;
            }
        }

        /// <summary>
        /// 次のソート方法に設定します。
        /// </summary>
        /// <param name="column">設定する列です。</param>
        public void SetNextSortDescription(ColumnTypes column)
        {
            this.sortDescriptions[(int)column] =
                (SortDescriptionTypes)(((int)this.sortDescriptions[(int)column] + 1) % (int)SortDescriptionTypes.MaxCount);
        }

        /// <summary>
        /// ソート方法を設定します。
        /// </summary>
        public void SetSortDescriptions()
        {
            if (ShaderSourceListItems == null)
            {
                return;
            }

            var collectionView = (CollectionView)CollectionViewSource.GetDefaultView(ShaderSourceListItems);
            collectionView.SortDescriptions.Clear();

            for (int idx = 0; idx < (int)ColumnTypes.MaxCount; ++idx)
            {
                string propertyName = string.Empty;
                switch (idx)
                {
                    case (int)ColumnTypes.FileType:
                        propertyName = "StrFileType";
                        break;
                    case (int)ColumnTypes.IncludePath:
                        propertyName = "IncludePath";
                        break;
                    case (int)ColumnTypes.FilePath:
                        propertyName = "FilePath";
                        break;
                }

                if (sortDescriptions[idx] == SortDescriptionTypes.Ascending)
                {
                    collectionView.SortDescriptions.Add(new SortDescription(propertyName, ListSortDirection.Ascending));
                }
                else if (sortDescriptions[idx] == SortDescriptionTypes.Descending)
                {
                    collectionView.SortDescriptions.Add(new SortDescription(propertyName, ListSortDirection.Descending));
                }
            }

            collectionView.Refresh();
        }

        private void ReadShaderSourceText(ShaderSourceListItemViewModel data)
        {
            if (data == null || this.currentShaderConfigViewModel == null)
            {
                this.StrShaderSource = string.Empty;
                return;
            }

            string fileFullPath = Path.Combine(
                Path.GetDirectoryName(this.currentShaderConfigViewModel.FileViewModel.FilePath), data.IncludePath, data.FilePath);

            if (File.Exists(fileFullPath))
            {
                var reader = new StreamReader(
                    fileFullPath, Encoding.GetEncoding(this.currentShaderConfigViewModel.ShaderConfigData.ShaderConfigInfo.CodePage));
                this.StrShaderSource = reader.ReadToEnd();
                reader.Close();
            }
            else
            {
                this.StrShaderSource = string.Empty;
            }
        }

        private void OpenShaderSourceExecute()
        {
            if (this.selectedShaderSourceListItem == null || this.currentShaderConfigViewModel == null)
            {
                return;
            }

            string fileFullPath = Path.Combine(
                Path.GetDirectoryName(this.currentShaderConfigViewModel.FileViewModel.FilePath), this.selectedShaderSourceListItem.IncludePath, this.selectedShaderSourceListItem.FilePath);

            if (File.Exists(fileFullPath))
            {
                string path = string.Empty;
                string option = string.Empty;

                var userConfigRef = AppManager.GetConfig<UserConfig>();

                UserConfig userConfig;
                if (userConfigRef.TryGetTarget(out userConfig))
                {
                    var config = userConfig.GetSubConfig<G3dCore.Configurations.UserComfigs.CommonConfig>();
                    if (config != null)
                    {
                        path = config.TextEditor.Path;
                        option = config.TextEditor.Option;
                    }
                }

                if (!string.IsNullOrWhiteSpace(path))
                {
                    Process process = new Process();
                    process.StartInfo.FileName = path;
                    process.StartInfo.Arguments = option.Replace("%1", fileFullPath);
                    process.Start();
                }
                else
                {
                    Process.Start("Notepad", fileFullPath);
                }
            }
        }

        private bool CanOpenShaderSource()
        {
            return this.selectedShaderSourceListItem != null;
        }

        private void OpenIncludePathExecute()
        {
            if (this.selectedShaderSourceListItem == null || this.currentShaderConfigViewModel == null)
            {
                return;
            }

            string path = Path.Combine(
                Path.GetDirectoryName(this.currentShaderConfigViewModel.FileViewModel.FilePath), this.selectedShaderSourceListItem.IncludePath);

            if (Directory.Exists(path))
            {
                Process.Start(path);
            }
        }

        private void OpenFilePathExecute()
        {
            if (this.selectedShaderSourceListItem == null || this.currentShaderConfigViewModel == null)
            {
                return;
            }

            string path = Path.Combine(
                Path.GetDirectoryName(this.currentShaderConfigViewModel.FileViewModel.FilePath), this.selectedShaderSourceListItem.IncludePath, Path.GetDirectoryName(this.selectedShaderSourceListItem.FilePath));

            if (Directory.Exists(path))
            {
                Process.Start(path);
            }
        }
    }
}
