﻿namespace ShaderAssistAddons.Modules.ShaderConfig.Utilities
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using G3dCore.Configurations;
    using G3dCore.Configurations.TeamConfigs;
    using G3dCore.IO;
    using Opal.App;

    /// <summary>
    /// シェーダ設定に関するユーティリティクラスです。
    /// </summary>
    public class ShaderConfigUtility
    {
        /// <summary>
        /// choice の種類です。
        /// </summary>
        public enum ChoiceType
        {
            /// <summary>
            /// bool です。
            /// </summary>
            Bool,

            /// <summary>
            /// 範囲です。
            /// </summary>
            Range,

            /// <summary>
            /// リストです。
            /// </summary>
            List
        }

        /// <summary>
        /// シェーダソースがインクルードパスに対して正しいかチェックします。
        /// </summary>
        /// <param name="fileName">シェーダソースのファイル名です。</param>
        /// <param name="shaderConfigDir">シェーダ設定ファイルのあるディレクトリです。</param>
        /// <param name="includeFullPaths">インクルードパスのフルパスです。</param>
        /// <param name="fullPath">シェーダソースのフルパスを返します。</param>
        /// <param name="includePath">適合するインクルードパスを返します。</param>
        /// <param name="relativePath">シェーダソースのインクルードパスからの相対パスを返します。</param>
        /// <returns>シェーダソースがインクルードパスに対して正しい場合、true を返します。</returns>
        public static bool CheckShaderPath(
            string fileName,
            string shaderConfigDir,
            string[] includeFullPaths,
            out string fullPath,
            out string includePath,
            out string relativePath
        )
        {
            fullPath = null;
            includePath = null;
            relativePath = null;

            foreach (string path in includeFullPaths)
            {
                string includeFullPath = PathUtility.GetExpandEnvironmentVariable(path);
                if (includeFullPath == null)
                {
                    return false;
                }

                string[] files = Directory.GetFiles(includeFullPath, fileName, SearchOption.AllDirectories);

                if (files.Length > 0)
                {
                    if (fullPath != null || files.Length >= 2)
                    {
                        // インクルードパス内に指定した名前のファイルはただ一つしか存在してはいけない。
                        //MessageBox.Show("選択したファイルと同名のファイルが登録されているインクルードパス内に含まれるため設定できません。", "Shader Config Editor", MessageBoxButtons.OK, MessageBoxIcon.Error);
                        return false;
                    }

                    fullPath = files[0];

                    if (fullPath.StartsWith(includeFullPath))
                    {
                        includePath = PathUtility.MakeRelativePath(shaderConfigDir, includeFullPath);
                        relativePath = fullPath.Substring(includeFullPath.Length + 1);
                    }
                }
            }

            return fullPath != null;
        }

        /// <summary>
        /// インクルードパスからの相対シェーダパスを取得します。
        /// </summary>
        /// <param name="shaderPath">相対シェーダパスを取得する元になるシェーダパスです。</param>
        /// <param name="shaderConfigDir">シェーダ設定ファイルのあるディレクトリです。</param>
        /// <param name="includeFullPaths">インクルードパスのフルパスです。</param>
        /// <returns>インクルードパスからの相対シェーダパスを返します。</returns>
        public static string GetRelativeShaderPath(string shaderPath, string shaderConfigDir, string[] includeFullPaths)
        {
            string fileName = Path.GetFileName(shaderPath);
            string fullPath = null;
            string includePath = null;
            string relativePath = null;

            bool result = CheckShaderPath(
                fileName,
                shaderConfigDir,
                includeFullPaths,
                out fullPath,
                out includePath,
                out relativePath);

            if (!result)
            {
                return null;
            }

            return relativePath;
        }

        /// <summary>
        /// ラベル部分を除いた choice を返します。
        /// </summary>
        /// <param name="choice">ラベルが含まれる choice の文字列です。</param>
        /// <returns>ラベル部分を除いた choice を返します。</returns>
        public static string GetChoiceWithoutLabels(string choice)
        {
            if (choice == "bool" || choice.StartsWith("["))
            {
                return choice;
            }

            string result = string.Empty;

            string[] choices = choice.Split(',');
            for (int i = 0; i < choices.Length; ++i)
            {
                choices[i] = choices[i].Split(':')[0].Trim();
            }

            for (int i = 0; i < choices.Length; ++i)
            {
                result += choices[i];
                if (i < choices.Length - 1)
                {
                    result += ", ";
                }
            }

            return result;
        }

        /// <summary>
        /// ラベル付き choice から値に対応するラベルを検索します。
        /// </summary>
        /// <param name="value">値です。</param>
        /// <param name="choiceLabels">ラベル付き choice の配列です。</param>
        /// <returns>ラベルを返します。ラベルが無い場合には値を返します。</returns>
        public static string FindLabelFromChoices(string value, string[] choiceLabels)
        {
            foreach (string choiceLabel in choiceLabels)
            {
                string[] separatedChoiceLabel = choiceLabel.Split(':');

                if (value == separatedChoiceLabel[0])
                {
                    return (separatedChoiceLabel.Length >= 2 ? separatedChoiceLabel[1] : value);
                }
            }

            return string.Empty;
        }

        /// <summary>
        /// choice の配列から値部分またはラベル部分を取得します。
        /// </summary>
        /// <param name="choices">choice の配列です。</param>
        /// <param name="isLabels">ラベル部分を取得する場合には true を指定します。</param>
        /// <returns>値部分またはラベル部分を返します。</returns>
        public static string[] GetValuesOrLabelsFromChoice(string[] choices, bool isLabels)
        {
            if (choices == null || choices.Length == 0)
            {
                return null;
            }

            string[] enums = new string[choices.Length];
            for (int i = 0; i < choices.Length; ++i)
            {
                string[] enumLabel = choices[i].Split(':');

                if (isLabels && enumLabel.Length >= 2)
                {
                    enums[i] = enumLabel[1];
                }
                else
                {
                    enums[i] = enumLabel[0];
                }
            }

            return enums;
        }

        /// <summary>
        /// 文字列 choice を展開してラベル付き choice の配列を取得します。
        /// </summary>
        /// <param name="choice">choice の文字列です。</param>
        /// <param name="choicesList">ラベル付き choice の配列を返します。</param>
        /// <returns>choice の種類を返します。</returns>
        public static ChoiceType ExpandChoice(string choice, out string[] choicesList)
        {
            ChoiceType optionVarChoiceType = ChoiceType.List;
            choicesList = null;

            if (choice == "bool")
            {
                // choice の種類：bool
                optionVarChoiceType = ChoiceType.Bool;
                choicesList = new string[2] { "0:False", "1:True" };
            }
            else if (choice.StartsWith("["))
            {
                // choice の種類：range
                optionVarChoiceType = ChoiceType.Range;
                string[] ranges = choice.Substring(1, choice.Length - 2).Split(',');
                Debug.Assert(ranges.Length == 2);

                int lower = int.Parse(ranges[0]);
                int upper = int.Parse(ranges[1]);
                Debug.Assert(lower < upper);
                choicesList = new string[upper - lower + 1];
                int cnt = 0;
                for (int i = lower; i <= upper; ++i)
                {
                    string val = string.Format("{0}", i);
                    choicesList[cnt++] = val;
                }
            }
            else
            {
                // choice の種類：list
                optionVarChoiceType = ChoiceType.List;
                choicesList = choice.Split(',');
                for (int i = 0; i < choicesList.Length; ++i)
                {
                    choicesList[i] = choicesList[i].Trim();
                }
            }

            return optionVarChoiceType;
        }

        /// <summary>
        /// choices 配列から Variation 用 choice 文字列を生成します。
        /// </summary>
        public static string GetVariationChoiceFromArray(ChoiceType choiceType, string[] choices)
        {
            if (choices == null || choices.Length == 0)
            {
                return string.Empty;
            }

            // シェーダ定義の Choice の種類が bool で両方指定されている場合。
            if (choiceType == ChoiceType.Bool)
            {
                if (choices.Contains("0") && choices.Contains("1"))
                {
                    return "bool";
                }
            }

            // 値が連続している場合は "[0, 8]" のような書き方にします。
            {
                bool newChoiceDecided = false;
                int lower = int.MinValue;
                int upper = 0;
                int currentNum = 0;

                foreach (string s in choices)
                {
                    int resultInt;
                    if (int.TryParse(s, out resultInt))
                    {
                        if (lower == int.MinValue)
                        {
                            lower = resultInt;
                            currentNum = resultInt;
                            continue;
                        }

                        upper = resultInt;
                        newChoiceDecided = (currentNum + 1 == upper ? true : false);
                        currentNum = resultInt;
                        if (!newChoiceDecided)
                        {
                            break;
                        }
                    }
                    else
                    {
                        newChoiceDecided = false;
                        break;
                    }
                }

                if (newChoiceDecided)
                {
                    return string.Format("[{0}, {1}]", lower, upper);
                }
            }

            // "add, sub, mul" のような書き方をします。
            string newChoice = string.Empty;
            foreach (string s in choices)
            {
                newChoice += s + ", ";
            }
            newChoice = newChoice.Substring(0, newChoice.Length - 2);

            return newChoice;
        }

        /// <summary>
        /// アプリケーションモードを取得します。
        /// </summary>
        /// <returns>アプリケーションモードを返します。</returns>
        public static AppMode GetAppMode()
        {
            try
            {
                var teamConfigRef = AppManager.GetConfig<TeamConfig>();

                TeamConfig teamConfig;
                teamConfigRef.TryGetTarget(out teamConfig);

                var config = teamConfig.GetSubConfig<ShaderAssistConfig>();
                if (config != null)
                {
                    return config.AppMode;
                }
            }
            catch (Exception)
            {
            }

            return AppMode.Default;
        }
    }
}
