﻿namespace ShaderAssistAddons.Modules.ShaderConfig.Commands
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using System.Windows;
    using System.Windows.Input;
    using System.Windows.Threading;

    /// <summary>
    /// デリゲートを受け取るICommandの実装
    /// </summary>
    public class DelegateCommand : ICommand
    {
        readonly Action _execute;
        readonly Func<bool> _canExecute;
        Dispatcher _dispatcher;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public DelegateCommand(Action execute)
            : this(execute, () => true)
        {
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public DelegateCommand(Action execute, Func<bool> canExecute)
        {
            if (execute == null)
            {
                throw new ArgumentNullException("execute");
            }

            if (canExecute == null)
            {
                throw new ArgumentNullException("canExecute");
            }

            this._execute = execute;
            this._canExecute = canExecute;
            this._dispatcher = Application.Current.Dispatcher;
        }

        /// <summary>
        /// コマンドが実行可能な状態かどうか問い合わせます。
        /// </summary>
        public bool CanExecute()
        {
            if (System.Threading.Thread.CurrentThread != this._dispatcher.Thread)
            {
                return this._dispatcher.Invoke(_canExecute);
            }
            return this._canExecute();
        }

        /// <summary>
        /// コマンドを実行します。
        /// </summary>
        public void Execute()
        {
            if (System.Threading.Thread.CurrentThread != this._dispatcher.Thread)
            {
                this._dispatcher.Invoke(_execute);
            }
            else
            {
                this._execute();
            }
        }

        /// <summary>
        /// ICommand.CanExecuteの明示的な実装。
        /// </summary>
        bool ICommand.CanExecute(object parameter)
        {
            return this.CanExecute();
        }

        /// <summary>
        /// ICommand.Executeの明示的な実装。
        /// </summary>
        void ICommand.Execute(object parameter)
        {
            this.Execute();
        }

        /// <summary>
        /// CanExecuteの結果に変更があったことを通知するイベント。
        /// </summary>
        public event EventHandler CanExecuteChanged
        {
            add { CommandManager.RequerySuggested += value; }
            remove { CommandManager.RequerySuggested -= value; }
        }

        /// <summary>
        /// コマンドの実行可否状態が変更された事を知らせるイベントを発生させます。
        /// </summary>
        public void RaiseCanExecuteChanged()
        {
            if (System.Threading.Thread.CurrentThread != this._dispatcher.Thread)
            {
                this._dispatcher.Invoke(CommandManager.InvalidateRequerySuggested);
            }
            else
            {
                CommandManager.InvalidateRequerySuggested();
            }
        }
    }
}
