﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Globalization;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Runtime.Remoting;
using System.Runtime.Remoting.Channels;
using System.Runtime.Remoting.Channels.Ipc;
using System.Threading;
using System.Windows.Forms;

using Bridge.res;

using Nintendo.Foundation.IO;

namespace Bridge
{
    enum ReturnCode
    {
        Success = 0,
        CannotFound3DEditor,
        InvalidOption,
        FailedCommunication,
        BridgeInfoError,
        DisconnectedCommunication,
        AlreadyStarted3DEditor,
        CannotStart3DEditor,
        CannotClose3DEditor,

        UndefinedError = 255,
    }
    class Bridge
    {
        // プロセス名
        private static string processName_ = "3dEditor";

        private static readonly string BridgeMutexName = "3dEditorBridge_Mutex";

        static int Main(string[] args)
        {
            try
            {
                var executingAssemblyName = Path.GetFileNameWithoutExtension(Assembly.GetExecutingAssembly().Location);

                if (executingAssemblyName.StartsWith("NW4F"))
                {
                    processName_ = "NW4F_3DEditor";
                }

                // ＵＩ言語初期化
                if (File.Exists(Application.StartupPath + "\\en\\" + executingAssemblyName + ".resources.dll"))
                {
                    CultureInfo cultureInfo = CultureInfo.CreateSpecificCulture("en-us");
                    Thread.CurrentThread.CurrentCulture = cultureInfo;
                    Thread.CurrentThread.CurrentUICulture = cultureInfo;
                }

                CommandLineParameters parameters;
                try
                {
                    if (!(new CommandLineParser().ParseArgs(args, out parameters)))
                    {
                        // ヘルプ表示、バージョン表示
                        return (int)ReturnCode.Success;
                    }
                }
                catch
                {
                    // 不正なオプション指定
                    return (int)ReturnCode.InvalidOption;
                }

                var bridgeInfo = MakeBridgeInfo(parameters);
                if (bridgeInfo == null)
                {
                    // 不正なオプション指定
                   ShowMessage(Strings.InvalidOption);
                   return (int)ReturnCode.InvalidOption;
                }

                // 3DEditor を起動
                if (bridgeInfo.Mode == BridgeInfo.ModeType.Start3DEditor)
                {
                    if (IsFoundWindow())
                    {
                        // 3DEditorが既に起動済み
                        ShowMessage(Strings.AlreadyStarted3DEditor);
                        return (int)ReturnCode.AlreadyStarted3DEditor;
                    }
                    // パスの設定、bridgeと同じディレクトリにある3DEditorを起動する
                    var location = Assembly.GetExecutingAssembly().Location;
                    var path = Path.GetDirectoryName(location);

                    // プロセス起動
                    var option = bridgeInfo.Path == null ? "" : "--team-settings-file=\"" + bridgeInfo.Path + "\"";
                    if (!Start3DEditor(processName_, path, option))
                    {
                        ShowMessage(Strings.CannotExecute3DEditor);
                        return (int)ReturnCode.CannotStart3DEditor;
                    }

                    return (int)ReturnCode.Success;
                }

                // 3DEditor を終了
                if (bridgeInfo.Mode == BridgeInfo.ModeType.Close3DEditor)
                {
                    // 3DEditorが起動していない場合はそのまま成功として終了
                    if (!IsFoundWindow())
                    {
                        return (int)ReturnCode.Success;
                    }
                }

                if (IsFoundWindow() == false)
                {
                    // 3DEditorが見つからない
                    ShowMessage(Strings.CannotFound3DEditor);
                    return (int)ReturnCode.CannotFound3DEditor;
                }

                bool isSuccess = false;
                try
                {
                    isSuccess = WriteBridgeInfo(bridgeInfo);
                }
                catch (RemotingException e)
                {
                    // 通信が途切れたとき?
                    ShowMessage(Strings.FailedCommunication);

                    string message = e.Message??"" + "\n" + e.StackTrace;
#if DEBUG
                    ShowMessage(message);
#endif
                    return (int)ReturnCode.DisconnectedCommunication;
                }

                if (isSuccess == false)
                {
                    // TODO: メッセージ修正
                    // 通信に失敗しました。
                    ShowMessage(Strings.FailedCommunication);
                    return (int)ReturnCode.FailedCommunication;
                }

                // 3DEditor の終了を待機
                if (bridgeInfo.Mode == BridgeInfo.ModeType.Close3DEditor)
                {
                    var waitCount = 0;
                    while (IsFoundWindow())
                    {
                        if (waitCount++ > 100)
                        {
                            ShowMessage(Strings.Bridge_CannotClose3DEditor);
                            return (int)ReturnCode.CannotClose3DEditor;
                        }
                        Thread.Sleep(100);
                    }
                }

                if (bridgeInfo.Results.Any())
                {
                    ShowResult(bridgeInfo);
                    if (bridgeInfo.Results.Any(x => x.error))
                    {
                        return (int)ReturnCode.BridgeInfoError;
                    }
                }

                // 正常に終了
                // query_connection_state のときは結果表示と紛らわしいので成功メッセージは表示しない
                if (bridgeInfo.Mode != BridgeInfo.ModeType.QueryConnectionState)
                {
                    ShowMessage(Strings.SuccessfulFinished);
                }
                return (int)ReturnCode.Success;
            }
            catch(Exception e)
            {
                // 不正なオプション指定
                ShowMessage(Strings.UndefinedError);
                string message = (e.Message??"") + "\n" + e.StackTrace;
#if DEBUG
                ShowMessage(message);
#endif
                return (int)ReturnCode.UndefinedError;
            }
        }

        // プロセス開始の待ち時間(ms)
        private const int StartProcessWaitTime = 30000;
        // 起動チェックインターバル
        private const int StartCheckInterval = 500;
        // 起動チェック回数
        private const int MaxStartCheckCount = 120; // 最大1分

        // 3DEditor の起動
        private static bool Start3DEditor(string fileName, string path, string arguments)
        {
            var process = new Process();
            process.StartInfo.WorkingDirectory = path;
            process.StartInfo.FileName = Path.Combine(path, fileName);
            process.StartInfo.Arguments = arguments;
            try
            {
                process.Start();
            }
            catch (Win32Exception)
            {
                return false;
            }
            // プロセスがアイドル状態になるまで待つ
            process.WaitForInputIdle(StartProcessWaitTime);

            int startCheckCount = 0;
            while (!IsFoundWindow())
            {
                startCheckCount++;
                if (startCheckCount == MaxStartCheckCount) { break; }
                Thread.Sleep(StartCheckInterval);
            }

            if (!IsFoundWindow())
            {
                return false;
            }

            // 実際にbridgeInfoが取得できるまで待機する
            // 3DEditor の Bridge機能の同時実行を抑制する
            var mutex = new Mutex(false, BridgeMutexName);
            mutex.WaitOne();
            var channel = new IpcClientChannel();
            ChannelServices.RegisterChannel(channel, true);

            startCheckCount = 0;
            var connected = false;

            while (!connected)
            {
                startCheckCount++;
                if (startCheckCount > MaxStartCheckCount) { break; }
                try
                {
                    var remoteBridgeInfo =
                        Activator.GetObject(
                            typeof(BridgeInfo),
                            string.Format("ipc://{0}/{1}", BridgeInfo.IpcChannelName, BridgeInfo.IpcUrl)) as BridgeInfo;
                    if (remoteBridgeInfo == null)
                    {
                        continue;
                    }

                    var dummy = remoteBridgeInfo.IsReceived; // IPCで接続できなければ、ここで例外が起きる

                    connected = true;
                    break;
                }
                catch(Exception)
                {
                    ;
                }
                Thread.Sleep(StartCheckInterval);
            }

            // チャンネルを解除
            ChannelServices.UnregisterChannel(channel);
            mutex.ReleaseMutex();

            return connected;
        }

        // ウィンドウの検索
        private static bool IsFoundWindow()
        {
            return Process.GetProcessesByName(processName_).Any();
        }

        // メッセージ表示
        private static void ShowMessage(string message, string title = null)
        {
            if (string.IsNullOrEmpty(title))
            {
                Debug.Assert(!string.IsNullOrEmpty(message));
                Console.WriteLine(message);
            }
            else if (string.IsNullOrEmpty(message))
            {
                Console.WriteLine(title);
            }
            else
            {
                Console.WriteLine(string.Format("{0} : {1}", title, message));
            }
        }

        private static BridgeInfo MakeBridgeInfo(CommandLineParameters ps)
        {
            BridgeInfo info = new BridgeInfo();
            if (ps.ExportInfo != null)
            {
                info.Mode = BridgeInfo.ModeType.ExportInfo;
                info.Output = ps.ExportInfo.OutputPath;
            }
            else if (ps.ExportTemporaryFiles != null)
            {
                info.Mode = BridgeInfo.ModeType.ExportTemporaryFiles;
                info.Path = ps.ExportTemporaryFiles.Path;
                info.Files = ps.ExportTemporaryFiles.FileNames.ToList();
            }
            else if (ps.ReloadTemporaryFiles != null)
            {
                info.Mode = BridgeInfo.ModeType.ReloadTemporaryFiles;
                info.ForceUnloadAndLoad = ps.ReloadTemporaryFiles.ForceUnloadAndLoad;
                info.Path = ps.ReloadTemporaryFiles.Path;
                info.Files = ps.ReloadTemporaryFiles.FileNames.ToList();
            }
            else if (ps.LoadFiles != null)
            {
                info.Mode = BridgeInfo.ModeType.LoadFiles;
                info.Files = ps.LoadFiles.Paths.ToList();
            }
            else if (ps.CloseAll != null)
            {
                info.Mode = BridgeInfo.ModeType.CloseAll;
            }
            else if (ps.SaveFiles != null)
            {
                info.Mode = BridgeInfo.ModeType.SaveFiles;
                info.Files = ps.SaveFiles.Paths.ToList();
                info.DisableFileInfo = ps.SaveFiles.DisableFileInfo;
            }
            else if (ps.SaveAll != null)
            {
                info.Mode = BridgeInfo.ModeType.SaveAll;
                info.DisableFileInfo = ps.SaveAll.DisableFileInfo;
            }
            else if (ps.OptimizeShader != null)
            {
                info.Mode = BridgeInfo.ModeType.OptimizeShader;
                info.DisableOptimizeShader = ps.OptimizeShader.Disable;
            }
            else if (ps.ConnectRuntime != null)
            {
                info.Mode = BridgeInfo.ModeType.Connect;
                switch (ps.ConnectRuntime.Target.ToLower())
                {
                    case "pc":
                        info.IsDeviceTarget = false;
                        break;
                    case "device":
                        info.IsDeviceTarget = true;
                        break;
                    case "catdev":
                        Console.WriteLine(Strings.Bridge_NoticeCatDevObsolete);
                        info.IsDeviceTarget = true;
                        break;
                    default:
                        Console.WriteLine(Strings.Bridge_UnknownTarget, ps.ConnectRuntime.Target);
                        return null;
                }
                info.IsDeviceTarget =  string.Compare(ps.ConnectRuntime.Target, "Device", StringComparison.OrdinalIgnoreCase) == 0 ||
                                        string.Compare(ps.ConnectRuntime.Target, "CatDev", StringComparison.OrdinalIgnoreCase) == 0; //古いオプション名
                info.Platform = ps.ConnectRuntime.Platform ?? string.Empty;
            }
            else if (ps.DisconnectRuntime != null)
            {
                info.Mode = BridgeInfo.ModeType.Disconnect;
            }
            else if (ps.ClearHistory != null)
            {
                info.Mode = BridgeInfo.ModeType.ClearHistory;
            }
            else if (ps.Start3DEditor != null)
            {
                info.Mode = BridgeInfo.ModeType.Start3DEditor;
                info.Path = ps.Start3DEditor.TeamSettingsFilePath;
            }
            else if (ps.Close3DEditor != null)
            {
                info.Mode = BridgeInfo.ModeType.Close3DEditor;
            }
            else if (ps.QueryConnectionState != null)
            {
                info.Mode = BridgeInfo.ModeType.QueryConnectionState;
            }
            else
            {
                // 指定なし
                return null;
            }

            return info;
        }

        private static bool WriteBridgeInfo(BridgeInfo bridgeInfo)
        {
            var isSuccess = false;

            // 3DEditor の Bridge機能の同時実行を抑制する
            var mutex = new Mutex(false, BridgeMutexName);

            try
            {
                mutex.WaitOne();

                var channel = new IpcClientChannel();
                ChannelServices.RegisterChannel(channel, true);

                // リモートオブジェクトを取得
                var remoteBridgeInfo =
                    Activator.GetObject(
                        typeof(BridgeInfo),
                        string.Format("ipc://{0}/{1}", BridgeInfo.IpcChannelName, BridgeInfo.IpcUrl)) as BridgeInfo;

                // 情報をコピーする
                {
                    remoteBridgeInfo.Mode = bridgeInfo.Mode;
                    remoteBridgeInfo.Output = bridgeInfo.Output;
                    remoteBridgeInfo.Path = bridgeInfo.Path;
                    remoteBridgeInfo.Files = bridgeInfo.Files != null ? new List<string>(bridgeInfo.Files) : null;
                    remoteBridgeInfo.ForceUnloadAndLoad = bridgeInfo.ForceUnloadAndLoad;
                    remoteBridgeInfo.WorkingDirectory = Directory.GetCurrentDirectory();
                    remoteBridgeInfo.IsDeviceTarget = bridgeInfo.IsDeviceTarget;
                    remoteBridgeInfo.Platform = bridgeInfo.Platform;
                    remoteBridgeInfo.DisableOptimizeShader = bridgeInfo.DisableOptimizeShader;
                    remoteBridgeInfo.DisableFileInfo = bridgeInfo.DisableFileInfo;
                }

                // 3DEditor に通知
                remoteBridgeInfo.Results = new List<BridgeInfo.Result>();
                remoteBridgeInfo.IsReceived = false;
                remoteBridgeInfo.OnWrote();

                // 受信待ち
                // 少なくとも3DEditor が終了すれば例外が発生して終了する。
                for (int i = 0;; i++)
                {
                    if (remoteBridgeInfo.IsReceived)
                    {
                        isSuccess = true;
                        break;
                    }

                    Thread.Sleep(100);
                }

                // 結果を戻す
                bridgeInfo.Results = new List<BridgeInfo.Result>(remoteBridgeInfo.Results);

                // チャンネルを解除
                ChannelServices.UnregisterChannel(channel);
            }
            finally
            {
                mutex.ReleaseMutex();
            }

            return isSuccess;
        }

        // 結果表示
        private static void ShowResult(BridgeInfo bridgeInfo)
        {
            var titles = new Dictionary<BridgeInfo.Result.ResultType, string>()
            {
                {BridgeInfo.Result.ResultType.CannotWrite,	Strings.CannotWrite},
                {BridgeInfo.Result.ResultType.FileNotFoundIn3DEditor,	Strings.FileNotFoundIn3DEditor},
                {BridgeInfo.Result.ResultType.CannotRead, Strings.CannotRead},
                {BridgeInfo.Result.ResultType.FileDuplicated, Strings.FileDuplicated},
                {BridgeInfo.Result.ResultType.FailedToConnect, Strings.FailedToConnect},
                {BridgeInfo.Result.ResultType.AlreadyConnected, Strings.AlreadyConnected},
                {BridgeInfo.Result.ResultType.FailedToDisconnect, Strings.FailedToDisconnect},
                {BridgeInfo.Result.ResultType.AlreadyDisconnect, Strings.AlreadyDisconnected},
                {BridgeInfo.Result.ResultType.Information, null},
                {BridgeInfo.Result.ResultType.Warning, null},
                {BridgeInfo.Result.ResultType.Error, null},
                {BridgeInfo.Result.ResultType.ConnectionIdle, Strings.ConnectionIdle},
                {BridgeInfo.Result.ResultType.ConnectionBusy, Strings.ConnectionBusy},
                {BridgeInfo.Result.ResultType.Disconnected, Strings.Disconnected},
            };

            Debug.Assert(titles.Count == Enum.GetValues(typeof(BridgeInfo.Result.ResultType)).Length);

            foreach(var result in bridgeInfo.Results)
            {
                Debug.Assert(titles.ContainsKey(result.Type));

                ShowMessage(result.Message, titles[result.Type]);
            }
        }
    }
}
