﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Diagnostics;
using System.Linq;
using App;
using App.Data;
using App.Utility;
using ConfigCommon;

namespace Viewer
{
    using IronPython.Modules;

    using nw.g3d.iflib;

    /// <summary>
    /// システムメッセージの送信。
    /// </summary>
    public abstract class SystemMessage : BaseMessage
    {
        /// <summary>メッセージカテゴリー</summary>
        protected override MessageCategory _messageCategory { get { return MessageCategory.System; } }
        // 送信できるタイプかどうか？の判定を行う。
        public static bool IsValidSelectType(GuiObjectID id)
        {
            return HioUtility.IsValidSelectType(id);
        }

        /// <summary>
        /// シーンアニメーションの場合に、オブジェクトのインデックスを計算する。
        /// </summary>
        protected static uint GetObjectIndex(EditTargetType objectType, GuiObject guiObject)
        {
            uint index = 0;
            if (objectType == EditTargetType.CameraAnim)
            {
                foreach (CameraAnimation camera in ((SceneAnimation)guiObject.OwnerDocument).CameraAnims)
                {
                    if (guiObject == camera)
                    {
                        return index;
                    }
                    index++;
                }
            }
            else if (objectType == EditTargetType.LightAnim)
            {
                foreach (LightAnimation light in ((SceneAnimation)guiObject.OwnerDocument).LightAnims)
                {
                    if (guiObject == light)
                    {
                        return index;
                    }
                    index++;
                }
            }
            else if (objectType == EditTargetType.FogAnim)
            {
                foreach (FogAnimation fog in ((SceneAnimation)guiObject.OwnerDocument).FogAnims)
                {
                    if (guiObject == fog)
                    {
                        return index;
                    }
                    index++;
                }
            }
            else
            {
                return (uint)guiObject.Index;
            }

            return index;
        }
    }

    /// <summary>
    /// 指定されたアクションを実行する
    /// </summary>
    public sealed class GeneralActionMessage : SystemMessage
    {
        public GeneralActionMessage(Action action)
        {
            this.action = action;
        }

        private readonly Action action;
        public override void Execute()
        {
            if (Viewer.Manager.Instance.IsConnected)
            {
                action();
            }
        }

        public static void Send(Action action)
        {
            (new GeneralActionMessage(action)).Push();
        }
    }

    /// <summary>
    /// オブジェクトを選択します。
    /// </summary>
    public sealed class Select : SystemMessage
    {
        readonly ArrayList _targets;
        readonly GuiObjectID _guiObjectId;
        // カレントオブジェクトタイプ
        public static GuiObjectID _currentType = GuiObjectID.DummyObject;
        // カレント選択オブジェクト
        private static ArrayList _currentSelectedObjects;

        /// <summary>コンストラクタ</summary>
        private Select(ArrayList targets, GuiObjectID guiObjectId)
        {
            _targets = targets;
            _guiObjectId = guiObjectId;
        }

        /// <summary>送信しても良いターゲットのリストを作成する</summary>
        public static ArrayList MakeSendObjectList(ArrayList targets, GuiObjectID guiObjectId)
        {
            ArrayList sendTargets = new ArrayList();
            foreach (var obj in targets)
            {
                switch (guiObjectId)
                {
                    case GuiObjectID.Material:
                        Material material = obj as Material;
                        if (material != null)
                        {
                            if (material.Referrers.Any(x => x.IsAttached))
                            {
                                sendTargets.Add(obj);
                            }
                        }
                        break;
                    case GuiObjectID.Bone:
                        Bone bone = obj as Bone;
                        if (bone != null)
                        {
                            if (bone.Owner.IsAttached)
                            {
                                sendTargets.Add(obj);
                            }
                        }
                        break;
                    case GuiObjectID.Shape:
                        Shape shape = obj as Shape;
                        if (shape != null)
                        {
                            if (shape.Owner.IsAttached)
                            {
                                sendTargets.Add(obj);
                            }
                        }
                        break;
                    case GuiObjectID.Model:
                        Model model = obj as Model;
                        if (model != null)
                        {
                            if (model.IsAttached)
                            {
                                sendTargets.Add(obj);
                            }
                        }
                        break;
                    case GuiObjectID.ShaderDefinition:
                        // 現状SELECTしない
                        //_sendTargets.Add(obj);
                        break;
                    case GuiObjectID.BoneVisibilityAnimation:
                    case GuiObjectID.MaterialAnimation:
                    case GuiObjectID.ColorAnimation:
                    case GuiObjectID.MaterialVisibilityAnimation:
                    case GuiObjectID.ShaderParameterAnimation:
                    case GuiObjectID.ShapeAnimation:
                    case GuiObjectID.SkeletalAnimation:
                    case GuiObjectID.TexturePatternAnimation:
                    case GuiObjectID.TextureSrtAnimation:
                    case GuiObjectID.CameraAnimation:
                    case GuiObjectID.FogAnimation:
                    case GuiObjectID.LightAnimation:
                    case GuiObjectID.SceneAnimation:
                        AnimationDocument animDoc = obj as AnimationDocument;
                        if (animDoc != null)
                        {
                            if (animDoc.IsAttached)
                            {
                                sendTargets.Add(obj);
                            }
                        }
                        break;
                    default:
                        sendTargets.Add(obj);
                        break;
                }

            }

            return sendTargets;
        }

        /// <summary>実行</summary>
        public override void Execute()
        {
            if (Viewer.Manager.Instance.IsConnected)
            {
                SelectInternal(_targets, _guiObjectId);
            }
        }

        /// <summary>
        /// 送信スレッド用の Select の実装
        /// </summary>
        public static void SelectInternal(ArrayList targets, GuiObjectID guiObjectId)
        {
            // Attatchしているもののみ送るようにターゲットリストの更新
            var _sendTargets = MakeSendObjectList(targets, guiObjectId);

            if (_currentType == guiObjectId &&
                _currentSelectedObjects != null &&
                _currentSelectedObjects.Count == _sendTargets.Count &&
                _currentSelectedObjects.OfType<object>().SequenceEqual(_sendTargets.OfType<object>()))
            {
                // 同じなら送らない
                return;
            }

            if (_sendTargets.Count == 0)
            {
                // デバッグログ
                DebugConsole.WriteLine("Select Execute  : num = {0}, ", _sendTargets.Count);

                // UI 上で選択を外した後に再選択できるようにする。
                // ランタイム側の選択状態とは異なったものになる。
                _currentSelectedObjects = _sendTargets;
                _currentType = guiObjectId;
                return;
            }

            HioUtility.Select(_sendTargets, guiObjectId);

            // デバッグログ
            DebugConsole.WriteLine("Select Execute  : num = {0}, ", _sendTargets.Count);
            int index = 0;
            foreach (var target in _sendTargets)
            {
                DebugConsole.WriteLine("({0}), type = {1}, name = {2}", index, target.GetType().ToString(), target.ToString());
                index++;
            }

            _currentSelectedObjects = _sendTargets;
            _currentType = guiObjectId;
        }

        /// <summary>送る</summary>
        public static void Send(GuiObject selectedObject)
        {
            ArrayList guiObjects = new ArrayList();
            guiObjects.Add(selectedObject);

            Send(guiObjects, selectedObject.ObjectID);
        }

        /// <summary>送る</summary>
        public static void Send(GuiObjectGroup targets, GuiObjectID type)
        {
            ArrayList guiObjects = new ArrayList();
            foreach (GuiObject guiObject in targets.GetObjects(type))
            {
                guiObjects.Add(guiObject);
            }

            Send(guiObjects, type);
        }

        /// <summary>送る</summary>
        public static void Send(ArrayList selectedObjects, GuiObjectID type)
        {
            (new Select(selectedObjects, type)).Push();
        }
    }

    public sealed class EditSelectTarget : SystemMessage
    {
        GuiObjectID id;
        ArrayList objects;
        public EditSelectTarget(ArrayList selectedObjects, GuiObjectID type)
        {
            id = type;
            objects = selectedObjects;
        }

        public override void Execute()
        {
            if (!Viewer.Manager.Instance.IsConnected)
            {
                return;
            }

            // 何も選択できないなら送らない
 			if (Select.MakeSendObjectList(objects, id).Count == 0)
            {
                return;
            }

            NintendoWare.G3d.Edit.SelectTargetKind kind;
            switch (id)
            {
                case GuiObjectID.Model:
                    kind = NintendoWare.G3d.Edit.SelectTargetKind.Model;
                    break;
                case GuiObjectID.Material:
                    kind = NintendoWare.G3d.Edit.SelectTargetKind.Material;
                    break;
                case GuiObjectID.Bone:
                    kind = NintendoWare.G3d.Edit.SelectTargetKind.Bone;
                    break;
                case GuiObjectID.Shape:
                    kind = NintendoWare.G3d.Edit.SelectTargetKind.Shape;
                    break;
                case GuiObjectID.Texture:
                    // そもそも選択対象でないので何もしない
                    return;
                default:
                    // 何もしない
                    // kind が用意されていないので何もしない
                    return;
            }

            DebugConsole.WriteLine("Hio.EditSelectTarget: {0}", kind);
            G3dHioLibProxy.Hio.EditSelectTarget(kind);
            return;
        }

        public static void Send(ArrayList selectedObjects, GuiObjectID type)
        {
            Select.Send(selectedObjects, type);
            (new EditSelectTarget(selectedObjects, type)).Push();
        }
    }

    /// <summary>
    /// 再生フレーム設定。
    /// </summary>
    public sealed class PlayFrameCtrl : SystemMessage
    {
        /// <summary>コンストラクタ</summary>
        private PlayFrameCtrl(float currentFrame)
        {
            IsCompressible = true;
        }

        //public static double diff = 0;

        /// <summary>実行</summary>
        public override void Execute()
        {
            if (Viewer.Manager.Instance.IsConnected)
            {
                var frame = AnimationMessage.CurrentFrame;
                /* デバッグ用
                var time = System.Environment.TickCount;
                diff = time * 60 - frame * 1000;
                 */
                G3dHioLibProxy.Hio.PlayFrameCtrl(frame);
                AnimationMessage.IsPlaying = true;
                DebugConsole.WriteLine("PlayFrameCtrl " + frame);
            }
        }

        /// <summary>送る</summary>
        public static void Send(float currentFrame)
        {
            (new PlayFrameCtrl(currentFrame)).Push();
            Connecter.PulseMessageThread();
        }
    }

    /// <summary>
    /// 停止フレーム設定。
    /// </summary>
    public sealed class StopFrameCtrl : SystemMessage
    {
        /// <summary>コンストラクタ</summary>
        private StopFrameCtrl(float currentFrame)
        {
            IsCompressible = true;
        }

        /// <summary>実行</summary>
        public override void Execute()
        {
            if (Viewer.Manager.Instance.IsConnected)
            {

                var frame = AnimationMessage.CurrentFrame;
                /* デバッグ用
                var time = System.Environment.TickCount;
                var totalFrame = 248;
                DebugConsole.WriteLine("StopFrameCtrl Execute  : frame {0}", Math.IEEERemainder(time * 60 - frame * 1000 - PlayFrameCtrl.diff, totalFrame * 1000) / 1000);
                */
                G3dHioLibProxy.Hio.StopFrameCtrl(frame);
                AnimationMessage.IsPlaying = false;
                DebugConsole.WriteLine("StopFrameCtrl " + frame);
            }
        }

        /// <summary>送る</summary>
        public static void Send(float currentFrame)
        {
            (new StopFrameCtrl(currentFrame)).Push();
        }
    }

    /// <summary>
    /// 送信状態更新
    /// メッセージは送らない
    /// </summary>
    public sealed class UpdatePlaying : SystemMessage
    {
        private bool _playing;
        private UpdatePlaying(bool playing)
        {
            IsCompressible = true;
            _playing = playing;
        }

        public override void Execute()
        {
            AnimationMessage.IsPlaying = _playing;
        }

        public static void Send(bool playing)
        {
            (new UpdatePlaying(playing)).Push();
        }
    }

    public sealed class SetPlayPolicy : SystemMessage
    {
        private readonly NintendoWare.G3d.Edit.PlayPolicy	_playPolicy;
        private readonly float								_currentFrame;

        /// <summary>コンストラクタ</summary>
        private SetPlayPolicy(NintendoWare.G3d.Edit.PlayPolicy playPolicy, float currentFrame)
        {
            _playPolicy = playPolicy;
            _currentFrame = currentFrame;
            IsCompressible = true;
        }

        /// <summary>実行</summary>
        public override void Execute()
        {
            if (Viewer.Manager.Instance.IsConnected)
            {
                G3dHioLibProxy.Hio.StopFrameCtrl(_currentFrame);
                AnimationMessage.IsPlaying = false;
                DebugConsole.WriteLine("StopFrameCtrl Execute  : frame {0}", _currentFrame);

                G3dHioLibProxy.Hio.SetPlayPolicy(_playPolicy);
                DebugConsole.WriteLine("SetPlayPolicy Execute  : {0}", _playPolicy);
            }
        }

        /// <summary>送る</summary>
        public static void Send(NintendoWare.G3d.Edit.PlayPolicy playPolicy, float currentFrame)
        {
            (new SetPlayPolicy(playPolicy, currentFrame)).Push();
        }
    }

    public sealed class SetFrameStep : SystemMessage
    {
        private readonly float _frameScale;

        /// <summary>コンストラクタ</summary>
        private SetFrameStep(float frameScale)
        {
            _frameScale = frameScale;
            IsCompressible = true;
        }

        /// <summary>実行</summary>
        public override void Execute()
        {
            if (Viewer.Manager.Instance.IsConnected)
            {
                G3dHioLibProxy.Hio.SetFrameStep(_frameScale);
                DebugConsole.WriteLine("SetFrameStep Execute  : frame {0}", _frameScale);
            }
        }

        /// <summary>送る</summary>
        public static void Send(float frameScale)
        {
            (new SetFrameStep(frameScale)).Push();
        }
    }

    public sealed class SetFrameCount : SystemMessage
    {
        private readonly float _frameCount;
        public static float lastFrameCount = -1; // 切断時に -1 に戻る

        private SetFrameCount(float frameCount)
        {
            _frameCount = frameCount;
            IsCompressible = true;
        }

        public override void Execute()
        {
            if (Viewer.Manager.Instance.IsConnected)
            {
                if (lastFrameCount != _frameCount)
                {
                    G3dHioLibProxy.Hio.SetFrameCount(_frameCount);
                    DebugConsole.WriteLine("SetFrameCount Execute  : frame {0}", _frameCount);
                    lastFrameCount = _frameCount;
                }
            }
        }

        public static void Send(float frameCount)
        {
            (new SetFrameCount(frameCount)).Push();
        }
    }

    public sealed class ModelRuntimeUserScript : SystemMessage
    {
        private readonly Model model_;
        private readonly string script_;

        private ModelRuntimeUserScript(Model model, string script)
        {
            model_ = model;
            script_ = script;
        }

        public override void Execute()
        {
            if (Viewer.Manager.Instance.IsConnected)
            {
                G3dHioLibProxy.Hio.ExecuteModelUserScript(model_, script_);
            }
        }

        public static void Send(Model model, string script)
        {
            // 選択状態を再転送する(選択解除が送られないため、選択中の状態が同期していない可能性がある)
            var materials = App.AppContext.SelectedTarget.GetObjects(GuiObjectID.Material);
            if (materials.Any())
            {
                Select.Send(new ArrayList(materials), GuiObjectID.Material);
            }
            else
            {
                GeneralActionMessage.Send(() => G3dHioLibProxy.Hio.ClearSelectedMaterials());
            }

            var bones = App.AppContext.SelectedTarget.GetObjects(GuiObjectID.Bone);
            if (bones.Any())
            {
                Select.Send(new ArrayList(bones), GuiObjectID.Bone);
            }
            else
            {
                GeneralActionMessage.Send(() => G3dHioLibProxy.Hio.ClearSelectedBones());
            }

            var shapes = App.AppContext.SelectedTarget.GetObjects(GuiObjectID.Shape);
            if (shapes.Any())
            {
                Select.Send(new ArrayList(shapes), GuiObjectID.Shape);
            }
            else
            {
                GeneralActionMessage.Send(() => G3dHioLibProxy.Hio.ClearSelectedShapes());
            }

            (new ModelRuntimeUserScript(model, script)).Push();
        }
    }

    /// <summary>
    /// ランタイムの反映の区切りを定める
    /// </summary>
    public sealed class TransactionMessage : SystemMessage
    {
        // Poll と　transaction Dispose を同時させないための同期オブジェクト
        public static object sync = new object();

        public static IDisposable transaction = null;
        readonly bool begin;

        private readonly bool _waitRuntimeState = false;
        public TransactionMessage(bool begin, bool waitRuntimeState)
        {
            this.begin = begin;
            _waitRuntimeState = waitRuntimeState;
            IsCompressible = true;
        }

        public override bool IsSameTarget(BaseMessage msg)
        {
            return false;
        }

        public override bool Anihhilate(BaseMessage msg)
        {
            // 終了, 開始または開始, 終了と続いたら消去
            return msg is TransactionMessage && ((TransactionMessage)msg).begin != begin &&
                ((TransactionMessage)msg)._waitRuntimeState ==_waitRuntimeState;
        }

        public override void Execute()
        {
            if (begin)
            {
                BeginTransaction(_waitRuntimeState);
            }
            else
            {
                EndTransaction();
            }
        }

        private static int count = 0;
        public static bool needToWaitRuntimeStatus;
        public static void BeginTransaction(bool wait)
        {
            if (count == 0)
            {
                // 開始

                // Normal になるまで待つ
                {
                    using (var watch = new DebugStopWatch("Wait Normal"))
                    {
                        while (G3dHioLibProxy.Hio.GetRuntimeState() != NintendoWare.G3d.Edit.RuntimeState.Normal &&
                            (System.Threading.Thread.CurrentThread.ThreadState & System.Threading.ThreadState.AbortRequested) == 0 &&
                            G3dHioLibProxy.Hio.IsConnected)
                        {
                            lock (Connecter.SyncPollPing)
                            {
                                System.Threading.Monitor.Pulse(Connecter.SyncPollPing);
                            }
                            // 計測した結果 1 や 20 より少し早く処理が終わる
                            System.Threading.Thread.Sleep(10);
                        }
                    }
                }

                // Normal に戻っているはず
                Debug.Assert(G3dHioLibProxy.Hio.GetRuntimeState() == NintendoWare.G3d.Edit.RuntimeState.Normal);

                DebugConsole.WriteLine("TransactionMessage start");
                needToWaitRuntimeStatus = wait;
                transaction = G3dHioLibProxy.Hio.BeginCommand(needToWaitRuntimeStatus);
            }

            // wait = false の後で wait = true が呼ばれることはないはず!
            Debug.Assert(count == 0 || wait != true || needToWaitRuntimeStatus);

            count++;
        }

        public static void EndTransaction()
        {
            Debug.Assert(transaction != null);
            if (transaction != null)
            {
                count--;
                if (count == 0)
                {
                    // 終了
                    lock (sync)
                    {
                        DebugConsole.WriteLine("TransactionMessage end");
                        transaction.Dispose();
                        transaction = null;
                    }

                    if (needToWaitRuntimeStatus)
                    {
                        HioUtility.WaitRuntimeState();
                    }
                }
            }
        }

        public static void Send(bool begin, bool overInterval = false, bool waitRuntimeState = false)
        {
            (new TransactionMessage(begin, waitRuntimeState)).Push();
        }

        public static void Initialize()
        {
            if (transaction != null)
            {
                transaction.Dispose();
                transaction = null;
            }
            count = 0;
            needToWaitRuntimeStatus = false;
        }
    }

    /// <summary>
    /// ランタイムデバッグメッセージ出力の切り替え
    /// </summary>
    public sealed class EnableRuntimeDebugMessage : SystemMessage
    {
        private bool enable;

        public EnableRuntimeDebugMessage(bool enable)
        {
            this.enable = enable;
        }

        public override void Execute()
        {
            if (Viewer.Manager.Instance.IsConnected)
            {
                if (enable)
                {
                    G3dHioLibProxy.Hio.EnableRuntimeDebugLog();
                }
                else
                {
                    G3dHioLibProxy.Hio.DisableRuntimeDebugLog();
                }
            }
        }

        public static void Send(bool enable)
        {
            (new EnableRuntimeDebugMessage(enable)).Push();
        }
    }
}
