﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using App;
using App.Data;
using App.Utility;
using ConfigCommon;
using nw.g3d.nw4f_3dif;

namespace Viewer
{
    /// <summary>
    /// マテリアル用メッセージです。
    /// </summary>
    public abstract class MaterialMessage : BaseMessage
    {
        /// <summary>どのプロパティが編集されたかのフラグ。</summary>
        protected uint _modifiedProperties;
        /// <summary>プロパティの中での要素編集フラグ(RGBAのRとか)。</summary>
        protected uint _modifiedPropertyElements;
        /// <summary>メッセージカテゴリー</summary>
        protected override MessageCategory _messageCategory { get { return MessageCategory.Material; } }
        /// <summary>対象マテリアルのリスト。</summary>
        protected ArrayList _targetMaterialList;
        /// <summary>実行</summary>
        protected abstract void ExecuteInternal();
        //---------------------------------------------------------------------
        /// <summary>
        /// 選択。
        /// </summary>
        protected static void Select(GuiObject material)
        {
            Viewer.Select.Send(material);
        }

        /// <summary>
        /// 選択。
        /// </summary>
        protected static void Select(GuiObjectGroup targets)
        {
#if true
            Viewer.Select.Send(targets, GuiObjectID.Material);
#endif
        }

        /// <summary>
        /// 選択。
        /// </summary>
        protected static void Select(ArrayList targets)
        {
            Viewer.Select.Send(targets, GuiObjectID.Material);
        }

        protected static ArrayList GetTargetList(GuiObjectGroup targets)
        {
            ArrayList targetList = new ArrayList();
            foreach (var target in targets.GetObjects(GuiObjectID.Material))
            {
                targetList.Add(target);
            }
            return targetList;
        }

        /// <summary>
        /// 対象マテリアルを持つモデルがアタッチされていれば、送信可。
        /// </summary>
        protected bool CanSend()
        {
            if (_targetMaterialList == null)
            {
                //// デバッグメッセージ
                //DebugConsole.WriteLine("MaterialMessage Execute not send Message : Execute() is not impliment. ");
                return false;
            }

            foreach (var mat in _targetMaterialList)
            {
                Material material = mat as Material;
                if (material != null)
                {
                    if (material.Referrers.All(x => x.IsAttached))
                    {
                        return true;
                    }
                }
            }

            // デバッグメッセージ
            DebugConsole.WriteLine("MaterialMessage Execute not send Message : Owner model is not attatched. ");
            return false;
        }

        /// <summary>実行</summary>
        public override void Execute()
        {
            if (!Viewer.Manager.Instance.IsConnected)
            {
                return;
            }

            // モデルファイルがアタッチされていない場合は送信しない。
            if (!CanSend())
            {
                return;
            }

            // 実際の実行
            ExecuteInternal();
        }

    }

    /// <summary>
    /// マテリアルの表示/非表示を設定します。
    /// </summary>
    public sealed class SetMaterialVisibility : MaterialMessage
    {
        private readonly bool _data;

        /// <summary>コンストラクタ</summary>
        private SetMaterialVisibility(bool src, ArrayList targetMaterialList)
        {
            _data = src;
            _targetMaterialList = targetMaterialList;
        }

        /// <summary>実行</summary>
        protected override void ExecuteInternal()
        {
            HioUtility.EditMaterialVisibility(_data);
            // デバッグメッセージ
            DebugConsole.WriteLine("SetMaterialVisibility Execute  : ({0})", _data);
        }

        /// <summary>送る</summary>
        public static void Send(GuiObject target, bool src, uint modifiedProperties, uint modifiedPropertyElements)
        {
            Select(target);

            ArrayList targetList = new ArrayList();
            targetList.Add(target);
            (new SetMaterialVisibility(src, targetList)).Push();
        }

        /// <summary>送る</summary>
        public static void Send(GuiObjectGroup targets, bool src, uint modifiedProperties, uint modifiedPropertyElements)
        {
            Select(targets);

            ArrayList targetList = GetTargetList(targets);
            (new SetMaterialVisibility(src, targetList)).Push();
        }

        /// <summary>送る</summary>
        public static void Send(ArrayList targets, object[] _data, uint modifiedProperties, uint modifiedPropertyElements)
        {
            object sameData = CheckSameData(targets, _data);
            if (sameData != null)
            {
                Select(targets);

                (new SetMaterialVisibility((bool)sameData, targets)).Push();
            }
            else
            {
                for (int i = 0; i < _data.Length; i++)
                {
                    Send((GuiObject)targets[i], (bool)_data[i], modifiedProperties, modifiedPropertyElements);
                }
            }
        }
    }

    public sealed class BlinkMaterial : MaterialMessage
    {
        private BlinkMaterial(ArrayList targetMaterials)
        {
            _targetMaterialList = targetMaterials;
        }

        protected override void ExecuteInternal()
        {
            G3dHioLibProxy.Hio.EditBlinkMaterials();
        }

        public static void Send(ArrayList targets)
        {
            Select(targets);
            (new BlinkMaterial(targets)).Push();
        }
    }

    public class SetRenderState<T> : MaterialMessage
    {
        private readonly Action<T> _HioEdit;
        private T[] values;
        private Dictionary<Model,LoadOrReloadModel> loadModels;
        private bool _reloadIfOptimized;
        private bool[] _maybeOptimized;
        private SetRenderState(ArrayList targetMaterials, Action<T> HioEdit, object[] data, bool[] maybeOptimized, bool reloadModelIfOptimized)
        {
            values = data.OfType<T>().ToArray();
            Debug.Assert(values.Length == targetMaterials.Count);
            Debug.Assert(maybeOptimized.Length == targetMaterials.Count);
            _HioEdit = HioEdit;
            _targetMaterialList = targetMaterials;
            _reloadIfOptimized = reloadModelIfOptimized;
            _maybeOptimized = maybeOptimized;
            loadModels = new Dictionary<Model, LoadOrReloadModel>();
            if (reloadModelIfOptimized)
            {
                for (int i = 0; i < targetMaterials.Count; i++)
                {
                    if (maybeOptimized[i])
                    {
                        foreach (var model in ((Material)targetMaterials[i]).Referrers)
                        {
                            if (!loadModels.ContainsKey(model))
                            {
                                loadModels.Add(model, new LoadOrReloadModel(model));
                            }
                        }
                    }
                }
                /*
                foreach (var model in targetMaterials.OfType<Material>().Select(x => x.Owner).Distinct())
                {
                    loadModels.Add(new Tuple<Model, LoadModel2>(model, new LoadModel2(model)));
                }*/
            }
            targetMaterials.OfType<Material>();
        }

        protected override void ExecuteInternal()
        {
            ArrayList optimized = new ArrayList();
            ArrayList notOptimized = new ArrayList();
            List<T> notOptimizedData = new List<T>();
            for (int i = 0; i < _targetMaterialList.Count; i++)
            {
                var material = (Material)_targetMaterialList[i];

                var numOptimized = optimized.Count;
                if (_maybeOptimized[i])
                {
                    foreach (var model in material.Referrers.Where(x => x.lastMaterialOptimizeData != null))
                    {
                        var data = model.lastMaterialOptimizeData[material.Index];
                        if (data != null && data.Item1 != null)
                        {
                            var materialData = data.Item1.materials.FirstOrDefault(x => x.index == material.Index);
                            Debug.Assert(materialData != null && materialData.reusedData == null);
                            if (materialData != null && materialData.optimize)
                            {
                                optimized.Add(material);
                            }
                        }
                    }
                }
                if (numOptimized != optimized.Count)
                {
                    continue;
                }

                notOptimized.Add(material);
                notOptimizedData.Add(values[i]);
            }

            var v = notOptimizedData.Distinct().Take(2).ToArray();
            if (v.Length == 1)
            {
                Viewer.Select.SelectInternal(notOptimized, GuiObjectID.Material);
                _HioEdit(v[0]);
            }
            else
            {
                for (int i = 0; i < notOptimized.Count; i++)
                {
                    var array = new ArrayList();
                    array.Add(notOptimized[i]);
                    Viewer.Select.SelectInternal(array, GuiObjectID.Material);
                    _HioEdit(notOptimizedData[i]);
                }
            }

            if (_reloadIfOptimized)
            {
                foreach (var model in optimized.OfType<Material>().SelectMany(x => x.Referrers).Distinct())
                {
                    loadModels[model].Execute();
                }
            }
        }

        public static void Send(ArrayList targetMaterials, Action<T> HioEdit, object[] data, bool[] maybeOptimized, bool reload)
        {
            (new SetRenderState<T>(targetMaterials, HioEdit, data, maybeOptimized, reload)).Push();
        }
    }

    public class SetMaterialRenderStateAllOrReload
    {
        public static void Send(Material[] materials)
        {
            SetRenderState<render_stateType>.Send(
                new ArrayList(materials),
                x =>
                {
                    DebugConsole.WriteLine("Send render state parematers");
                    HioUtility.EditMaterialRenderStateMode(x.mode);
                    HioUtility.EditMaterialDepthTestEnable(x.depth_test.enable);
                    HioUtility.EditMaterialDepthTestWriteEnable(x.depth_test.write);
                    HioUtility.EditMaterialDepthTestFunc(x.depth_test.func);
                    HioUtility.EditMaterialAlphaTestEnable(x.alpha_test.enable);
                    HioUtility.EditMaterialAlphaTestFunc(x.alpha_test.func);
                    //HioUtility.EditMaterialAlphaTestValue(x.alpha_test.value);
                    HioUtility.EditMaterialBlendMode(x.blend_mode);
                    HioUtility.EditMaterialColorCombine(x.color_blend.rgb_op);
                    HioUtility.EditMaterialColorSrcBlend(x.color_blend.rgb_src_func);
                    HioUtility.EditMaterialColorDstBlend(x.color_blend.rgb_dst_func);
                    HioUtility.EditMaterialAlphaCombine(x.color_blend.alpha_op);
                    HioUtility.EditMaterialAlphaSrcBlend(x.color_blend.alpha_src_func);
                    HioUtility.EditMaterialAlphaDstBlend(x.color_blend.alpha_dst_func);
                    //HioUtility.EditMaterialConstantColor(
                    HioUtility.EditMaterialLogicOp(x.logical_blend.op);
                },
                materials.Select(x => ObjectUtility.Clone(x.Data.render_state)).ToArray(),
                materials.Select(x => true).ToArray(),
                true);
        }
    }

    /// <summary>
    /// マテリアルのデプステストの有効/無効を設定します。
    /// </summary>
    public sealed class SetMaterialDepthTestEnable
    {
        /// <summary>送る</summary>
        public static void Send(ArrayList targets, object[] data, bool[] maybeOptimized, bool reloadIfOptimized)
        {
            SetRenderState<bool>.Send(
                targets,
                _data =>
                {
                    HioUtility.EditMaterialDepthTestEnable(_data);
                    // デバッグメッセージ
                    DebugConsole.WriteLine("SetMaterialDepthTestEnable Execute  : ({0})", _data);
                },
                data,
                maybeOptimized,
                reloadIfOptimized);
        }
    }

    /// <summary>
    /// マテリアルのデプスへの書き込みの有効/無効を設定します。
    /// </summary>
    public sealed class SetMaterialDepthWriteEnable
    {
        /// <summary>送る</summary>
        public static void Send(ArrayList targets, object[] data, bool[] maybeOptimized, bool reloadIfOptimized)
        {
            SetRenderState<bool>.Send(
                targets,
                _data =>
                {
                    HioUtility.EditMaterialDepthTestWriteEnable(_data);
                    // デバッグメッセージ
                    DebugConsole.WriteLine("EditMaterialDepthTestWriteEnable Execute  : ({0})", _data);
                },
                data,
                maybeOptimized,
                reloadIfOptimized);
        }
    }

    /// <summary>
    /// マテリアルのデプス関数を設定します。
    /// </summary>
    public sealed class SetMaterialDepthFunc
    {
        /// <summary>送る</summary>
        public static void Send(ArrayList targets, object[] data, bool[] maybeOptimized, bool reloadIfOptimized)
        {
            SetRenderState<depth_test_funcType>.Send(
                targets,
                _data =>
                {
                    HioUtility.EditMaterialDepthTestFunc(_data);
                    // デバッグメッセージ
                    DebugConsole.WriteLine("EditMaterialDepthTestWriteEnable Execute  : ({0})", _data.ToString());
                },
                data,
                maybeOptimized,
                reloadIfOptimized);
        }
    }

    /// <summary>
    /// マテリアルのアルファテストの有効/無効を設定します。
    /// </summary>
    public sealed class SetMaterialAlphaTestEnable
    {
        /// <summary>送る</summary>
        public static void Send(ArrayList targets, object[] data, bool[] maybeOptimized, bool reloadIfOptimized)
        {
            SetRenderState<bool>.Send(
                targets,
                _data =>
                {
                    HioUtility.EditMaterialAlphaTestEnable(_data);
                    // デバッグメッセージ
                    DebugConsole.WriteLine("SetMaterialAlphaTestEnable Execute  : ({0})", _data);
                },
                data,
                maybeOptimized,
                reloadIfOptimized);
        }
    }

    /// <summary>
    /// マテリアルのアルファテストの比較関数を設定します。
    /// </summary>
    public sealed class SetMaterialAlphaFunc
    {
        /// <summary>送る</summary>
        public static void Send(ArrayList targets, object[] data, bool[] maybeOptimized, bool reloadIfOptimized)
        {
            SetRenderState<alpha_test_funcType>.Send(
                targets,
                _data =>
                {
                    HioUtility.EditMaterialAlphaTestFunc(_data);
                    // デバッグメッセージ
                    DebugConsole.WriteLine("SetMaterialAlphaFunc Execute  : ({0})", _data.ToString());
                },
                data,
                maybeOptimized,
                reloadIfOptimized);
        }
    }

    /// <summary>
    /// マテリアルのアルファテストの参照値を設定します。
    /// </summary>
    public sealed class SetMaterialAlphaRefValue : MaterialMessage
    {
        private readonly float _data;

        /// <summary>コンストラクタ</summary>
        private SetMaterialAlphaRefValue(float src, uint modifiedProperties, uint modifiedPropertyElements, ArrayList targetMaterialList)
        {
            _data = src;
            _modifiedProperties = modifiedProperties;
            _modifiedPropertyElements = modifiedPropertyElements;
            _targetMaterialList = targetMaterialList;

            // このメッセージは間引き対象
            IsCompressible = true;
        }

        /// <summary>実行</summary>
        protected override void ExecuteInternal()
        {
            HioUtility.EditMaterialAlphaTestValue(_data);
            // デバッグメッセージ
            DebugConsole.WriteLine("SetMaterialAlphaRefValue Execute  : ({0})", _data);
        }

        /// <summary>送る</summary>
        public static void Send(GuiObject target, float src, uint modifiedProperties, uint modifiedPropertyElements)
        {
            Select(target);

            ArrayList targetList = new ArrayList();
            targetList.Add(target);
            (new SetMaterialAlphaRefValue(src, modifiedProperties, modifiedPropertyElements, targetList)).Push();
        }

        /// <summary>送る</summary>
        public static void Send(GuiObjectGroup targets, float src, uint modifiedProperties, uint modifiedPropertyElements)
        {
            Select(targets);

            ArrayList targetList = GetTargetList(targets);
            (new SetMaterialAlphaRefValue(src, modifiedProperties, modifiedPropertyElements, targetList)).Push();
        }

        /// <summary>送る</summary>
        public static void Send(ArrayList targets, object[] _data, uint modifiedProperties, uint modifiedPropertyElements)
        {
            object sameData = CheckSameData(targets, _data);
            if (sameData != null)
            {
                Select(targets);

                (new SetMaterialAlphaRefValue((float)sameData, modifiedProperties, modifiedPropertyElements, targets)).Push();
            }
            else
            {
                for (int i = 0; i < _data.Length; i++)
                {
                    Send((GuiObject)targets[i], (float)_data[i], modifiedProperties, modifiedPropertyElements);
                }
            }
        }
    }

    /// <summary>
    /// マテリアルの論理演算の関数を設定します。
    /// </summary>
    public sealed class SetMaterialLogicOp
    {
        /// <summary>送る</summary>
        public static void Send(ArrayList targets, object[] data, bool[] maybeOptimized, bool reloadIfOptimized)
        {
            SetRenderState<logical_blend_opType>.Send(
                targets,
                _data =>
                {
                    HioUtility.EditMaterialLogicOp(_data);
                    // デバッグメッセージ
                    DebugConsole.WriteLine("SetMaterialLogicOp Execute  : ({0})", _data.ToString());
                },
                data,
                maybeOptimized,
                reloadIfOptimized);
        }
    }

    /// <summary>
    /// マテリアルのブレンドモードを設定します。
    /// </summary>
    public sealed class SetMaterialBlendMode
    {
        /// <summary>送る</summary>
        public static void Send(ArrayList targets, object[] data, bool[] maybeOptimized, bool reloadIfOptimized)
        {
            SetRenderState<render_state_blend_modeType>.Send(
                targets,
                _data =>
                {
                    HioUtility.EditMaterialBlendMode(_data);
                    // デバッグメッセージ
                    DebugConsole.WriteLine("SetMaterialBlendMode Execute  : ({0})", _data.ToString());
                },
                data,
                maybeOptimized,
                reloadIfOptimized);
        }
    }

    /// <summary>
    /// マテリアルのブレンドのソースに用いる式を設定します。
    /// </summary>
    public sealed class SetMaterialColorSrcBlend
    {
        /// <summary>送る</summary>
        public static void Send(ArrayList targets, object[] data, bool[] maybeOptimized, bool reloadIfOptimized)
        {
            SetRenderState<color_blend_rgb_funcType>.Send(
                targets,
                _data =>
                {
                    HioUtility.EditMaterialColorSrcBlend(_data);
                    // デバッグメッセージ
                    DebugConsole.WriteLine("SetMaterialColorSrcBlend Execute  : ({0})", _data.ToString());
                },
                data,
                maybeOptimized,
                reloadIfOptimized);
        }
    }

    /// <summary>
    /// マテリアルのブレンドのデスティネーションに用いる式を設定します。
    /// </summary>
    public sealed class SetMaterialColorDstBlend
    {
        /// <summary>送る</summary>
        public static void Send(ArrayList targets, object[] data, bool[] maybeOptimized, bool reloadIfOptimized)
        {
            SetRenderState<color_blend_rgb_funcType>.Send(
                targets,
                _data =>
                {
                    HioUtility.EditMaterialColorDstBlend(_data);
                    // デバッグメッセージ
                    DebugConsole.WriteLine("SetMaterialColorDstBlend Execute  : ({0})", _data.ToString());
                },
                data,
                maybeOptimized,
                reloadIfOptimized);
        }
    }

    /// <summary>
    /// マテリアルのコンバイナ式を設定します。
    /// </summary>
    public sealed class SetMaterialColorCombine
    {
        /// <summary>送る</summary>
        public static void Send(ArrayList targets, object[] data, bool[] maybeOptimized, bool reloadIfOptimized)
        {
            SetRenderState<color_blend_opType>.Send(
                targets,
                _data =>
                {
                    HioUtility.EditMaterialColorCombine(_data);
                    // デバッグメッセージ
                    DebugConsole.WriteLine("SetMaterialColorCombine Execute  : ({0})", _data.ToString());
                },
                data,
                maybeOptimized,
                reloadIfOptimized);
        }
    }

    /// <summary>
    /// マテリアルのアルファのブレンドのソースに用いる式を設定します。
    /// </summary>
    public sealed class SetMaterialAlphaSrcBlend
    {
        /// <summary>送る</summary>
        public static void Send(ArrayList targets, object[] data, bool[] maybeOptimized, bool reloadIfOptimized)
        {
            SetRenderState<color_blend_alpha_funcType>.Send(
                targets,
                _data =>
                {
                    HioUtility.EditMaterialAlphaSrcBlend(_data);
                    // デバッグメッセージ
                    DebugConsole.WriteLine("SetMaterialAlphaSrcBlend Execute  : ({0})", _data.ToString());
                },
                data,
                maybeOptimized,
                reloadIfOptimized);
        }
    }

    /// <summary>
    /// マテリアルのアルファのブレンドのディスティネーションに用いる式を設定します。
    /// </summary>
    public sealed class SetMaterialAlphaDstBlend
    {
        /// <summary>送る</summary>
        public static void Send(ArrayList targets, object[] data, bool[] maybeOptimized, bool reloadIfOptimized)
        {
            SetRenderState<color_blend_alpha_funcType>.Send(
                targets,
                _data =>
                {
                    HioUtility.EditMaterialAlphaDstBlend(_data);
                    // デバッグメッセージ
                    DebugConsole.WriteLine("SetMaterialAlphaDstBlend Execute  : ({0})", _data.ToString());
                },
                data,
                maybeOptimized,
                reloadIfOptimized);
        }
    }

    /// <summary>
    /// マテリアルのアルファのコンバイナ式を設定します。
    /// </summary>
    public sealed class SetMaterialAlphaCombine
    {
        /// <summary>送る</summary>
        public static void Send(ArrayList targets, object[] data, bool[] maybeOptimized, bool reloadIfOptimized)
        {
            SetRenderState<color_blend_opType>.Send(
                targets,
                _data =>
                {
                    HioUtility.EditMaterialAlphaCombine(_data);
                    // デバッグメッセージ
                    DebugConsole.WriteLine("SetMaterialAlphaCombine Execute  : ({0})", _data.ToString());
                },
                data,
                maybeOptimized,
                reloadIfOptimized);
        }
    }

    /// <summary>
    /// マテリアルのコンスタントカラーを設定します。
    /// </summary>
    public sealed class SetMaterialConstantColor : MaterialMessage
    {
        private RgbaColor _data;

        /// <summary>コンストラクタ</summary>
        private SetMaterialConstantColor(RgbaColor src, uint modifiedProperties, uint modifiedPropertyElements, ArrayList targetMaterialList)
        {
            _data = src;
            _modifiedProperties = modifiedProperties;
            _modifiedPropertyElements = modifiedPropertyElements;
            _targetMaterialList = targetMaterialList;

            // このメッセージは間引き対象
            IsCompressible = true;
        }

        public override bool IsSameTarget(BaseMessage msg)
        {
            SetMaterialConstantColor target = msg as SetMaterialConstantColor;
            if (target == null)
            {
                return false;
            }

            if (target._modifiedPropertyElements == _modifiedPropertyElements)
            {
                return true;
            }

            return false;
        }

        /// <summary>実行</summary>
        protected override void ExecuteInternal()
        {
            HioUtility.EditMaterialConstantColor(_data.R, _data.G, _data.B, _data.A, _modifiedPropertyElements);
            // デバッグメッセージ
            DebugConsole.WriteLine("SetMaterialConstantColor Execute  : ({0})", _data.ToString());
        }

        /// <summary>送る</summary>
        public static void Send(GuiObject target, RgbaColor src, uint modifiedProperties, uint modifiedPropertyElements)
        {
            Select(target);

            ArrayList targetList = new ArrayList();
            targetList.Add(target);
            (new SetMaterialConstantColor(src, modifiedProperties, modifiedPropertyElements, targetList)).Push();
        }

        /// <summary>送る</summary>
        public static void Send(GuiObjectGroup targets, RgbaColor src, uint modifiedProperties, uint modifiedPropertyElements)
        {
            Select(targets);

            ArrayList targetList = GetTargetList(targets);
            (new SetMaterialConstantColor(src, modifiedProperties, modifiedPropertyElements, targetList)).Push();
        }

        /// <summary>送る</summary>
        public static void Send(ArrayList targets, object[] _data, uint modifiedProperties, uint modifiedPropertyElements)
        {
            object sameData = CheckSameData(targets, _data);
            if (sameData != null)
            {
                Select(targets);

                (new SetMaterialConstantColor((RgbaColor)sameData, modifiedProperties, modifiedPropertyElements, targets)).Push();
            }
            else
            {
                for (int i = 0; i < _data.Length; i++)
                {
                    Send((GuiObject)targets[i], (RgbaColor)_data[i], modifiedProperties, modifiedPropertyElements);
                }
            }
        }
    }

    /// <summary>
    /// マテリアルの描画ステートモードを設定します。
    /// </summary>
    public sealed class SetMaterialRenderStateMode
    {
        /// <summary>送る</summary>
        public static void Send(ArrayList targets, object[] data, bool[] maybeOptimized, bool reloadIfOptimized)
        {
            SetRenderState<render_state_modeType>.Send(
                targets,
                _data =>
                {
                    HioUtility.EditMaterialRenderStateMode(_data);
                    // デバッグメッセージ
                    DebugConsole.WriteLine("SetMaterialRenderStateMode Execute  : ({0})", _data.ToString());
                },
                data,
                maybeOptimized,
                reloadIfOptimized);
        }
    }

    /// <summary>
    /// マテリアルのテクスチャサンプラのX方向のラップを設定します。
    /// </summary>
    public sealed class SetMaterialSamplerClampX : MaterialMessage
    {
        public struct PacketData
        {
            public uint Index;
            public wrap_uvwType Clamp;
            public PacketData(uint index, wrap_uvwType clamp)
            {
                Index = index;
                Clamp = clamp;
            }
        }

        private PacketData _data;


        /// <summary>コンストラクタ</summary>
        private SetMaterialSamplerClampX(PacketData src, uint modifiedProperties, uint modifiedPropertyElements, ArrayList targetMaterialList)
        {
            _data = src;
            _modifiedProperties = modifiedProperties;
            _modifiedPropertyElements = modifiedPropertyElements;
            _targetMaterialList = targetMaterialList;
        }

        /// <summary>実行</summary>
        protected override void ExecuteInternal()
        {
            HioUtility.EditMaterialSamplerWrapU((int)_data.Index, _data.Clamp);
            // デバッグメッセージ
            DebugConsole.WriteLine("SetMaterialSamplerClampX Execute  : Index = ({0}), Clamp = ({1})", _data.Index, _data.Clamp.ToString());
        }

        /// <summary>送る</summary>
        public static void Send(GuiObject target, PacketData src, uint modifiedProperties, uint modifiedPropertyElements)
        {
            Select(target);

            ArrayList targetList = new ArrayList();
            targetList.Add(target);
            (new SetMaterialSamplerClampX(src, modifiedProperties, modifiedPropertyElements, targetList)).Push();
        }

        /// <summary>送る</summary>
        public static void Send(GuiObjectGroup targets, PacketData src, uint modifiedProperties, uint modifiedPropertyElements)
        {
            Select(targets);

            ArrayList targetList = GetTargetList(targets);
            (new SetMaterialSamplerClampX(src, modifiedProperties, modifiedPropertyElements, targetList)).Push();
        }

        /// <summary>送る</summary>
        public static void Send(ArrayList targets, object[] _data, uint modifiedProperties, uint modifiedPropertyElements)
        {
            object sameData = CheckSameData(targets, _data);
            if (sameData != null)
            {
                Select(targets);

                (new SetMaterialSamplerClampX((PacketData)sameData, modifiedProperties, modifiedPropertyElements, targets)).Push();
            }
            else
            {
                for (int i = 0; i < _data.Length; i++)
                {
                    Send((GuiObject)targets[i], (PacketData)_data[i], modifiedProperties, modifiedPropertyElements);
                }
            }
        }
    }

    /// <summary>
    /// マテリアルのテクスチャサンプラのY方向のラップを設定します。
    /// </summary>
    public sealed class SetMaterialSamplerClampY : MaterialMessage
    {
        public struct PacketData
        {
            public uint Index;
            public wrap_uvwType Clamp;
            public PacketData(uint index, wrap_uvwType clamp)
            {
                Index = index;
                Clamp = clamp;
            }
        }

        private PacketData _data;

        /// <summary>コンストラクタ</summary>
        private SetMaterialSamplerClampY(PacketData src, uint modifiedProperties, uint modifiedPropertyElements, ArrayList targetMaterialList)
        {
            _data = src;
            _modifiedProperties = modifiedProperties;
            _modifiedPropertyElements = modifiedPropertyElements;
            _targetMaterialList = targetMaterialList;
        }

        /// <summary>実行</summary>
        protected override void ExecuteInternal()
        {
            HioUtility.EditMaterialSamplerWrapV((int)_data.Index, _data.Clamp);
            // デバッグメッセージ
            DebugConsole.WriteLine("SetMaterialSamplerClampY Execute  : Index = ({0}), Clamp = ({1})", _data.Index, _data.Clamp.ToString());
        }

        /// <summary>送る</summary>
        public static void Send(GuiObject target, PacketData src, uint modifiedProperties, uint modifiedPropertyElements)
        {
            Select(target);

            ArrayList targetList = new ArrayList();
            targetList.Add(target);
            (new SetMaterialSamplerClampY(src, modifiedProperties, modifiedPropertyElements, targetList)).Push();
        }

        /// <summary>送る</summary>
        public static void Send(GuiObjectGroup targets, PacketData src, uint modifiedProperties, uint modifiedPropertyElements)
        {
            Select(targets);

            ArrayList targetList = GetTargetList(targets);
            (new SetMaterialSamplerClampY(src, modifiedProperties, modifiedPropertyElements, targetList)).Push();
        }

        /// <summary>送る</summary>
        public static void Send(ArrayList targets, object[] _data, uint modifiedProperties, uint modifiedPropertyElements)
        {
            object sameData = CheckSameData(targets, _data);
            if (sameData != null)
            {
                Select(targets);

                (new SetMaterialSamplerClampY((PacketData)sameData, modifiedProperties, modifiedPropertyElements, targets)).Push();
            }
            else
            {
                for (int i = 0; i < _data.Length; i++)
                {
                    Send((GuiObject)targets[i], (PacketData)_data[i], modifiedProperties, modifiedPropertyElements);
                }
            }
        }
    }

    /// <summary>
    /// マテリアルのテクスチャサンプラのZ方向のラップを設定します。
    /// </summary>
    public sealed class SetMaterialSamplerClampZ : MaterialMessage
    {
        public struct PacketData
        {
            public uint Index;
            public wrap_uvwType Clamp;
            public PacketData(uint index, wrap_uvwType clamp)
            {
                Index = index;
                Clamp = clamp;
            }
        }

        private PacketData _data;

        /// <summary>コンストラクタ</summary>
        private SetMaterialSamplerClampZ(PacketData src, uint modifiedProperties, uint modifiedPropertyElements, ArrayList targetMaterialList)
        {
            _data = src;
            _modifiedProperties = modifiedProperties;
            _modifiedPropertyElements = modifiedPropertyElements;
            _targetMaterialList = targetMaterialList;
        }

        /// <summary>実行</summary>
        protected override void ExecuteInternal()
        {
            HioUtility.EditMaterialSamplerWrapW((int)_data.Index, _data.Clamp);
            // デバッグメッセージ
            DebugConsole.WriteLine("SetMaterialSamplerClampZ Execute  : Index = ({0}), Clamp = ({1})", _data.Index, _data.Clamp.ToString());
        }

        /// <summary>送る</summary>
        public static void Send(GuiObject target, PacketData src, uint modifiedProperties, uint modifiedPropertyElements)
        {
            Select(target);

            ArrayList targetList = new ArrayList();
            targetList.Add(target);
            (new SetMaterialSamplerClampZ(src, modifiedProperties, modifiedPropertyElements, targetList)).Push();
        }

        /// <summary>送る</summary>
        public static void Send(GuiObjectGroup targets, PacketData src, uint modifiedProperties, uint modifiedPropertyElements)
        {
            Select(targets);

            ArrayList targetList = GetTargetList(targets);
            (new SetMaterialSamplerClampZ(src, modifiedProperties, modifiedPropertyElements, targetList)).Push();
        }

        /// <summary>送る</summary>
        public static void Send(ArrayList targets, object[] _data, uint modifiedProperties, uint modifiedPropertyElements)
        {
            object sameData = CheckSameData(targets, _data);
            if (sameData != null)
            {
                Select(targets);

                (new SetMaterialSamplerClampZ((PacketData)sameData, modifiedProperties, modifiedPropertyElements, targets)).Push();
            }
            else
            {
                for (int i = 0; i < _data.Length; i++)
                {
                    Send((GuiObject)targets[i], (PacketData)_data[i], modifiedProperties, modifiedPropertyElements);
                }
            }
        }
    }

    public sealed class SetMaterialSamplerFilter : MaterialMessage
    {
        public struct Data
        {
            public int Index;
            public filter_mag_minType minFilter;
            public filter_mag_minType magFilter;
            public filter_mipType mipmapFilter;
            public filter_max_anisoType maxAnisotropy;
        }

        private Data _data;

        private SetMaterialSamplerFilter(Data src, ArrayList targets)
        {
            _data = src;
            _targetMaterialList = targets;
        }

        protected override void ExecuteInternal()
        {
            DebugConsole.WriteLine("SetMaterialSamplerFilter Execute : Index = ({0}), minFilter = ({1}), magFilter = ({2}), mipmapFilter = ({3}), maxAnisotropy = ({4})",
                _data.Index,
                _data.minFilter,
                _data.magFilter,
                _data.mipmapFilter,
                _data.maxAnisotropy);
            HioUtility.EditmaterialSamplerFilter(
                _data.Index,
                _data.minFilter,
                _data.magFilter,
                _data.mipmapFilter,
                _data.maxAnisotropy);
        }

        public static void Send(IEnumerable<Tuple<Material, Data>> data)
        {
            var groups = from item in data
                        group item.Item1 by item.Item2;

            foreach (var group in groups)
            {
                var targets = new ArrayList(group.ToArray());
                Select(targets);
                (new SetMaterialSamplerFilter(group.Key, targets)).Push();
            }
        }
    }

    /// <summary>
    /// マテリアルのテクスチャサンプラの最小LODレベルを設定します。
    /// </summary>
    public sealed class SetMaterialSamplerMinLOD : MaterialMessage
    {
        public struct PacketData
        {
            public uint Index;
            public float Lod;
            public PacketData(uint index, float lod)
            {
                Index = index;
                Lod = lod;
            }
        }

        private PacketData _data;
        public uint GetIndex() { return _data.Index; }

        /// <summary>コンストラクタ</summary>
        private SetMaterialSamplerMinLOD(PacketData src, uint modifiedProperties, uint modifiedPropertyElements, ArrayList targetMaterialList)
        {
            _data = src;
            _modifiedProperties = modifiedProperties;
            _modifiedPropertyElements = modifiedPropertyElements;
            _targetMaterialList = targetMaterialList;

            // このメッセージは間引き対象
            IsCompressible = true;
        }

        public override bool IsSameTarget(BaseMessage msg)
        {
            SetMaterialSamplerMinLOD target = msg as SetMaterialSamplerMinLOD;
            if (target == null)
            {
                return false;
            }

            if (target.GetIndex() == _data.Index)
            {
                return true;
            }

            return false;
        }

        /// <summary>実行</summary>
        protected override void ExecuteInternal()
        {
            HioUtility.EditMaterialSamplerMinLOD((int)_data.Index, _data.Lod);
            // デバッグメッセージ
            DebugConsole.WriteLine("EditMaterialSamplerMinLOD Execute  : Index = ({0}), Lod = ({1})", _data.Index, _data.Lod.ToString());
        }

        /// <summary>送る</summary>
        public static void Send(GuiObject target, PacketData src, uint modifiedProperties, uint modifiedPropertyElements)
        {
            Select(target);

            ArrayList targetList = new ArrayList();
            targetList.Add(target);
            (new SetMaterialSamplerMinLOD(src, modifiedProperties, modifiedPropertyElements, targetList)).Push();
        }

        /// <summary>送る</summary>
        public static void Send(GuiObjectGroup targets, PacketData src, uint modifiedProperties, uint modifiedPropertyElements)
        {
            Select(targets);

            ArrayList targetList = GetTargetList(targets);
            (new SetMaterialSamplerMinLOD(src, modifiedProperties, modifiedPropertyElements, targetList)).Push();
        }

        /// <summary>送る</summary>
        public static void Send(ArrayList targets, object[] _data, uint modifiedProperties, uint modifiedPropertyElements)
        {
            object sameData = CheckSameData(targets, _data);
            if (sameData != null)
            {
                Select(targets);

                (new SetMaterialSamplerMinLOD((PacketData)sameData, modifiedProperties, modifiedPropertyElements, targets)).Push();
            }
            else
            {
                for (int i = 0; i < _data.Length; i++)
                {
                    Send((GuiObject)targets[i], (PacketData)_data[i], modifiedProperties, modifiedPropertyElements);
                }
            }
        }
    }

    /// <summary>
    /// マテリアルのテクスチャサンプラの最大LODレベルを設定します。
    /// </summary>
    public sealed class SetMaterialSamplerMaxLOD : MaterialMessage
    {
        public struct PacketData
        {
            public uint Index;
            public float Lod;
            public PacketData(uint index, float lod)
            {
                Index = index;
                Lod = lod;
            }
        }

        private PacketData _data;
        public uint GetIndex() { return _data.Index; }

        /// <summary>コンストラクタ</summary>
        private SetMaterialSamplerMaxLOD(PacketData src, uint modifiedProperties, uint modifiedPropertyElements, ArrayList targetMaterialList)
        {
            _data = src;
            _modifiedProperties = modifiedProperties;
            _modifiedPropertyElements = modifiedPropertyElements;
            _targetMaterialList = targetMaterialList;

            // このメッセージは間引き対象
            IsCompressible = true;
        }

        public override bool IsSameTarget(BaseMessage msg)
        {
            SetMaterialSamplerMaxLOD target = msg as SetMaterialSamplerMaxLOD;
            if (target == null)
            {
                return false;
            }

            if (target.GetIndex() == _data.Index)
            {
                return true;
            }

            return false;
        }

        /// <summary>実行</summary>
        protected override void ExecuteInternal()
        {
            HioUtility.EditMaterialSamplerMaxLOD((int)_data.Index, _data.Lod);
            // デバッグメッセージ
            DebugConsole.WriteLine("SetMaterialSamplerMaxLOD Execute  : Index = ({0}), Lod = ({1})", _data.Index, _data.Lod.ToString());
        }

        /// <summary>送る</summary>
        public static void Send(GuiObject target, PacketData src, uint modifiedProperties, uint modifiedPropertyElements)
        {
            Select(target);

            ArrayList targetList = new ArrayList();
            targetList.Add(target);
            (new SetMaterialSamplerMaxLOD(src, modifiedProperties, modifiedPropertyElements, targetList)).Push();
        }

        /// <summary>送る</summary>
        public static void Send(GuiObjectGroup targets, PacketData src, uint modifiedProperties, uint modifiedPropertyElements)
        {
            Select(targets);

            ArrayList targetList = GetTargetList(targets);
            (new SetMaterialSamplerMaxLOD(src, modifiedProperties, modifiedPropertyElements, targetList)).Push();
        }

        /// <summary>送る</summary>
        public static void Send(ArrayList targets, object[] _data, uint modifiedProperties, uint modifiedPropertyElements)
        {
            object sameData = CheckSameData(targets, _data);
            if (sameData != null)
            {
                Select(targets);

                (new SetMaterialSamplerMaxLOD((PacketData)sameData, modifiedProperties, modifiedPropertyElements, targets)).Push();
            }
            else
            {
                for (int i = 0; i < _data.Length; i++)
                {
                    Send((GuiObject)targets[i], (PacketData)_data[i], modifiedProperties, modifiedPropertyElements);
                }
            }
        }
    }

    /// <summary>
    /// マテリアルのテクスチャサンプラのLODバイアスを設定します。
    /// </summary>
    public sealed class SetMaterialSamplerLODBias : MaterialMessage
    {
        public struct PacketData
        {
            public uint Index;
            public float Lod;
            public PacketData(uint index, float lod)
            {
                Index = index;
                Lod = lod;
            }
        }

        private PacketData _data;
        public uint GetIndex() { return _data.Index; }

        /// <summary>コンストラクタ</summary>
        private SetMaterialSamplerLODBias(PacketData src, uint modifiedProperties, uint modifiedPropertyElements, ArrayList targetMaterialList)
        {
            _data = src;
            _modifiedProperties = modifiedProperties;
            _modifiedPropertyElements = modifiedPropertyElements;
            _targetMaterialList = targetMaterialList;

            // このメッセージは間引き対象
            IsCompressible = true;
        }

        public override bool IsSameTarget(BaseMessage msg)
        {
            SetMaterialSamplerLODBias target = msg as SetMaterialSamplerLODBias;
            if (target == null)
            {
                return false;
            }

            if (target.GetIndex() == _data.Index)
            {
                return true;
            }

            return false;
        }

        /// <summary>実行</summary>
        protected override void ExecuteInternal()
        {
            HioUtility.EditMaterialSamplerLODBias((int)_data.Index, _data.Lod);
            // デバッグメッセージ
            DebugConsole.WriteLine("SetMaterialSamplerLODBias Execute  : Index = ({0}), Lod = ({1})", _data.Index, _data.Lod.ToString());
        }

        /// <summary>送る</summary>
        public static void Send(GuiObject target, PacketData src, uint modifiedProperties, uint modifiedPropertyElements)
        {
            Select(target);

            ArrayList targetList = new ArrayList();
            targetList.Add(target);
            (new SetMaterialSamplerLODBias(src, modifiedProperties, modifiedPropertyElements, targetList)).Push();
        }

        /// <summary>送る</summary>
        public static void Send(GuiObjectGroup targets, PacketData src, uint modifiedProperties, uint modifiedPropertyElements)
        {
            Select(targets);

            ArrayList targetList = GetTargetList(targets);
            (new SetMaterialSamplerLODBias(src, modifiedProperties, modifiedPropertyElements, targetList)).Push();
        }

        /// <summary>送る</summary>
        public static void Send(ArrayList targets, object[] _data, uint modifiedProperties, uint modifiedPropertyElements)
        {
            object sameData = CheckSameData(targets, _data);
            if (sameData != null)
            {
                Select(targets);

                (new SetMaterialSamplerLODBias((PacketData)sameData, modifiedProperties, modifiedPropertyElements, targets)).Push();
            }
            else
            {
                for (int i = 0; i < _data.Length; i++)
                {
                    Send((GuiObject)targets[i], (PacketData)_data[i], modifiedProperties, modifiedPropertyElements);
                }
            }
        }
    }

    /// <summary>
    /// マテリアルのシェーダーパラメーター用メッセージです。
    /// </summary>
    public abstract class MaterialShaderParameterMessage : MaterialMessage
    {
    }

    /// <summary>
    /// アトリビュートアサインを設定します。
    /// </summary>
    public sealed class SetMaterialAttribAssign : MaterialShaderParameterMessage
    {
        public sealed class PacketData : object
        {
            public string Id;
            public string Parameter;
            public PacketData(string id, string name)
            {
                Id = id;
                Parameter = name;
            }
        }

        /// <summary>コンストラクタ</summary>
        private SetMaterialAttribAssign(PacketData src, uint modifiedProperties, uint modifiedPropertyElements)
        {
            _modifiedProperties = modifiedProperties;
            _modifiedPropertyElements = modifiedPropertyElements;
        }

        /// <summary>実行</summary>
        protected override void ExecuteInternal()
        {
            ;
        }

        /// <summary>送る</summary>
        public static void Send(GuiObject target, PacketData src, uint modifiedProperties, uint modifiedPropertyElements)
        {
            Select(target);

            (new SetMaterialAttribAssign(src, modifiedProperties, modifiedPropertyElements)).Push();
        }

        /// <summary>送る</summary>
        public static void Send(GuiObjectGroup targets, PacketData src, uint modifiedProperties, uint modifiedPropertyElements)
        {
            Select(targets);

            (new SetMaterialAttribAssign(src, modifiedProperties, modifiedPropertyElements)).Push();
        }

        /// <summary>送る</summary>
        public static void Send(ArrayList targets, object[] _data, uint modifiedProperties, uint modifiedPropertyElements)
        {
            object sameData = CheckSameData(targets, _data);
            if (sameData != null)
            {
                Select(targets);

                (new SetMaterialAttribAssign((PacketData)sameData, modifiedProperties, modifiedPropertyElements)).Push();
            }
            else
            {
                for (int i = 0; i < _data.Length; i++)
                {
                    Send((GuiObject)targets[i], (PacketData)_data[i], modifiedProperties, modifiedPropertyElements);
                }
            }
        }
    }

    /// <summary>
    /// シェーダーオプションを設定します。
    /// </summary>
    public sealed class SetMaterialShaderOption : MaterialShaderParameterMessage
    {
        public sealed class PacketData : object
        {
            public string Id;
            public string Parameter;
            public PacketData(string id, string Value)
            {
                Id = id;
                Parameter = Value;
            }
        }

        /// <summary>コンストラクタ</summary>
        private SetMaterialShaderOption(PacketData src, uint modifiedProperties, uint modifiedPropertyElements)
        {
            _modifiedProperties = modifiedProperties;
            _modifiedPropertyElements = modifiedPropertyElements;
        }

        /// <summary>実行</summary>
        protected override void ExecuteInternal()
        {
            ;
        }

        /// <summary>送る</summary>
        public static void Send(GuiObject target, PacketData src, uint modifiedProperties, uint modifiedPropertyElements)
        {
            Select(target);

            (new SetMaterialShaderOption(src, modifiedProperties, modifiedPropertyElements)).Push();
        }

        /// <summary>送る</summary>
        public static void Send(GuiObjectGroup targets, PacketData src, uint modifiedProperties, uint modifiedPropertyElements)
        {
            Select(targets);

            (new SetMaterialShaderOption(src, modifiedProperties, modifiedPropertyElements)).Push();
        }

        /// <summary>送る</summary>
        public static void Send(ArrayList targets, object[] _data, uint modifiedProperties, uint modifiedPropertyElements)
        {
            object sameData = CheckSameData(targets, _data);
            if (sameData != null)
            {
                Select(targets);

                (new SetMaterialShaderOption((PacketData)sameData, modifiedProperties, modifiedPropertyElements)).Push();
            }
            else
            {
                for (int i = 0; i < _data.Length; i++)
                {
                    Send((GuiObject)targets[i], (PacketData)_data[i], modifiedProperties, modifiedPropertyElements);
                }
            }
        }
    }

    public sealed class SetMaterial_render_state_display_face : MaterialMessage
    {
        private readonly render_state_display_faceType _data;

        /// <summary>コンストラクタ</summary>
        private SetMaterial_render_state_display_face(render_state_display_faceType src, uint modifiedProperties, uint modifiedPropertyElements, ArrayList targetMaterialList)
        {
            _data = src;
            _modifiedProperties = modifiedProperties;
            _modifiedPropertyElements = modifiedPropertyElements;
            _targetMaterialList = targetMaterialList;
        }

        /// <summary>実行</summary>
        protected override void ExecuteInternal()
        {
            HioUtility.EditMaterialDisplayFace(_data);
            // デバッグメッセージ
            DebugConsole.WriteLine("HIO_SetMaterial_render_state_display_face Execute  : ({0})", _data.ToString());
        }

        /// <summary>送る</summary>
        public static void Send(GuiObject target, render_state_display_faceType src, uint modifiedProperties, uint modifiedPropertyElements)
        {
            Select(target);

            ArrayList targetList = new ArrayList();
            targetList.Add(target);
            (new SetMaterial_render_state_display_face(src, modifiedProperties, modifiedPropertyElements, targetList)).Push();
        }

        /// <summary>送る</summary>
        public static void Send(GuiObjectGroup targets, render_state_display_faceType src, uint modifiedProperties, uint modifiedPropertyElements)
        {
            Select(targets);

            ArrayList targetList = GetTargetList(targets);
            (new SetMaterial_render_state_display_face(src, modifiedProperties, modifiedPropertyElements, targetList)).Push();
        }

        /// <summary>送る</summary>
        public static void Send(ArrayList targets, object[] _data, uint modifiedProperties, uint modifiedPropertyElements)
        {
            object sameData = CheckSameData(targets, _data);
            if (sameData != null)
            {
                Select(targets);

                (new SetMaterial_render_state_display_face((render_state_display_faceType)sameData, modifiedProperties, modifiedPropertyElements, targets)).Push();
            }
            else
            {
                for (int i = 0; i < _data.Length; i++)
                {
                    Send((GuiObject)targets[i], (render_state_display_faceType)_data[i], modifiedProperties, modifiedPropertyElements);
                }
            }
        }
    }

    public sealed class MaterialShaderParameter : MaterialMessage
    {
        public class Data
        {
            public string					ParamName{			get; set; }
            public shader_param_typeType	ShaderParamType{	get; set; }
            public object					Params{				get; set; }
        }

        private readonly Data _data;
        public string GetParamName() { return _data.ParamName; }

        /// <summary>コンストラクタ</summary>
        private MaterialShaderParameter(MaterialShaderParameter.Data src, uint modifiedProperties, uint modifiedPropertyElements, ArrayList targetMaterialList)
        {
            _data = src;
            _modifiedProperties = modifiedProperties;
            _modifiedPropertyElements = modifiedPropertyElements;
            _targetMaterialList = targetMaterialList;

            // このメッセージは間引き対象
            IsCompressible = true;
        }

        public override bool IsSameTarget(BaseMessage msg)
        {
            MaterialShaderParameter target = msg as MaterialShaderParameter;
            if (target == null)
            {
                return false;
            }

            if (target.GetParamName() == _data.ParamName)
            {
                return true;
            }

            return false;
        }

        /// <summary>実行</summary>
        protected override void ExecuteInternal()
        {
            var paramsType = _data.Params.GetType();

            switch(ShaderTypeUtility.ShaderParamTypeKindFromType(_data.ShaderParamType))
            {
                // スカラとベクトル
                case ShaderTypeUtility.ShaderParamTypeKind.Scalar:
                case ShaderTypeUtility.ShaderParamTypeKind.Vector:
                {
                            if (_data.Params is float[]) {			HioUtility.EditMaterialShaderParameter_VectorOrScalar(_data.ParamName, _data.Params as float[],	_modifiedPropertyElements);	}
                    else if (paramsType.Name == "Int32[]") {	HioUtility.EditMaterialShaderParameter_VectorOrScalar(_data.ParamName, _data.Params as int[],	_modifiedPropertyElements);	}
                    else if (paramsType.Name == "UInt32[]") {	HioUtility.EditMaterialShaderParameter_VectorOrScalar(_data.ParamName, _data.Params as uint[],	_modifiedPropertyElements);	}
                    else if (_data.Params is bool[]) {			HioUtility.EditMaterialShaderParameter_VectorOrScalar(_data.ParamName, _data.Params as bool[],	_modifiedPropertyElements);	}
                    else {										Debug.Assert(false);																										}
                    break;
                }

                // マトリックス
                case ShaderTypeUtility.ShaderParamTypeKind.Matrix:
                {
                    HioUtility.EditMaterialShaderParameter_Matrix(_data.ShaderParamType, _data.ParamName, _data.Params as float[]);
                    break;
                }

                // テクスチャSRT
                case ShaderTypeUtility.ShaderParamTypeKind.TextureSrt:
                {
                    HioUtility.EditMaterialShaderParameter_TextureSrt(_data.ShaderParamType, _data.ParamName, _data.Params as float[]);
                    break;
                }

                default:
                    Debug.Assert(false);
                    break;
            }

            // デバッグメッセージ
            DebugConsole.WriteLine("HIO_MaterialShaderParameter Execute  : ParamName = ({0})", _data.ParamName);
        }

        /// <summary>送る</summary>
        private static void Send(GuiObject target, MaterialShaderParameter.Data src, uint modifiedProperties, uint modifiedPropertyElements)
        {
            Select(target);

            ArrayList targetList = new ArrayList();
            targetList.Add(target);
            (new MaterialShaderParameter(src, modifiedProperties, modifiedPropertyElements, targetList)).Push();
        }

        /// <summary>送る</summary>
        public static void Send(GuiObjectGroup targets, MaterialShaderParameter.Data src, uint modifiedProperties, uint modifiedPropertyElements)
        {
            TransactionMessage.Send(true);
            Select(targets);

            ArrayList targetList = GetTargetList(targets);
            (new MaterialShaderParameter(src, modifiedProperties, modifiedPropertyElements, targetList)).Push();
            TransactionMessage.Send(false);
        }

        /// <summary>送る</summary>
        public static void Send(ArrayList targets, object[] _data, uint modifiedProperties, uint modifiedPropertyElements)
        {
            TransactionMessage.Send(true);
            object sameData = CheckSameData(targets, _data);
            if (sameData != null)
            {
                Select(targets);

                (new MaterialShaderParameter((MaterialShaderParameter.Data)sameData, modifiedProperties, modifiedPropertyElements, targets)).Push();
            }
            else
            {
                for (int i = 0; i < _data.Length; i++)
                {
                    Send((GuiObject)targets[i], (MaterialShaderParameter.Data)_data[i], modifiedProperties, modifiedPropertyElements);
                }
            }
            TransactionMessage.Send(false);
        }
    }

    public sealed class SetRenderInfo : MaterialMessage
    {
        public class Data
        {
            public Material material;
            public RenderInfo[] renderInfos;
            public string name;
            public render_info_typeType type;
            public string[] strings;
            public int[] ints;
            public float[] floats;
            public bool? queryRenderInfo; // null のときはEdit***RenderInfo が行われた場合のみ
            public HashSet<string> defaultEmpty;
        }

        private readonly Data _data;
        //public string GetParamName() { return _data.ParamName; }

        /// <summary>コンストラクタ</summary>
        private SetRenderInfo(Data src, ArrayList targetMaterialList)
        {
            _data = src;
            //_modifiedProperties = modifiedProperties;
            //_modifiedPropertyElements = modifiedPropertyElements;
            _targetMaterialList = targetMaterialList;

            // このメッセージは間引き対象
            IsCompressible = true;
        }

        public override bool IsSameTarget(BaseMessage msg)
        {
            var target = msg as SetRenderInfo;
            if (target == null)
            {
                return false;
            }

            return target._data.material == _data.material &&
                target._data.name == _data.name &&
                target._data.type == _data.type &&
                target._data.queryRenderInfo == _data.queryRenderInfo;
        }

        /// <summary>実行</summary>
        protected override void ExecuteInternal()
        {
            bool updated = _data.material.LastRenderInfo.DefaultEmptyRenderInfoHio.Count != _data.defaultEmpty.Count || _data.material.LastRenderInfo.DefaultEmptyRenderInfoHio.Any(x => !_data.defaultEmpty.Contains(x));
            _data.material.LastRenderInfo.DefaultEmptyRenderInfoHio = _data.defaultEmpty;

            var stringTable = _data.material.LastRenderInfo.StringRenderInfoHio;
            var intTable = _data.material.LastRenderInfo.IntRenderInfoHio;
            var floatTable = _data.material.LastRenderInfo.FloatRenderInfoHio;
            switch (_data.type)
            {
                case render_info_typeType.@string:
                    {
                        var current = _data.material.LastRenderInfo.StringRenderInfoHio[_data.name];
                        if (!current.SequenceEqual(_data.strings))
                        {
                            updated = true;
                            stringTable[_data.name] = _data.strings;
                            var renderInfoArray = HioUtility.CreateIRenderInfoArrayFromTable(_data.renderInfos, stringTable, intTable, floatTable);
                            HioUtility.DebugUpdateRenderInfo(renderInfoArray);
                            G3dHioLibProxy.Hio.UpdateRenderInfo(renderInfoArray);
                            DebugConsole.WriteLine("Hio.EditStringRenderInfo: " + _data.name);
                            G3dHioLibProxy.Hio.EditStringRenderInfo(_data.name);
                        }
                    }
                    break;
                case render_info_typeType.@int:
                    {
                        var current = _data.material.LastRenderInfo.IntRenderInfoHio[_data.name];
                        List<int> pos = new List<int>();
                        {
                            int i;
                            for (i = 0; i < Math.Min(current.Length, _data.ints.Length); i++)
                            {
                                if (current[i] != _data.ints[i])
                                {
                                    pos.Add(i);
                                }
                            }

                            for (; i < Math.Max(current.Length, _data.ints.Length); i++)
                            {
                                pos.Add(i);
                            }
                        }

                        if (pos.Any())
                        {
                            updated = true;
                            intTable[_data.name] = _data.ints;
                            if (current.Length != _data.ints.Length)
                            {
                                var renderInfoArray = HioUtility.CreateIRenderInfoArrayFromTable(_data.renderInfos, stringTable, intTable, floatTable);
                                HioUtility.DebugUpdateRenderInfo(renderInfoArray);
                                G3dHioLibProxy.Hio.UpdateRenderInfo(renderInfoArray);
                            }

                            foreach (var index in pos)
                            {
                                DebugConsole.WriteLine("Hio.EditIntRenderInfo: " + _data.name + ", " + index + ", " + (index < _data.ints.Length ? _data.ints[index] : 0));
                                G3dHioLibProxy.Hio.EditIntRenderInfo(_data.name, index, index < _data.ints.Length ? _data.ints[index] : 0);
                            }
                        }
                    }
                    break;
                case render_info_typeType.@float:
                    {
                        var current = _data.material.LastRenderInfo.FloatRenderInfoHio[_data.name];
                        List<int> pos = new List<int>();
                        {
                            int i;
                            for (i = 0; i < Math.Min(current.Length, _data.floats.Length); i++)
                            {
                                if (current[i] != _data.floats[i])
                                {
                                    pos.Add(i);
                                }
                            }

                            for (; i < Math.Max(current.Length, _data.floats.Length); i++)
                            {
                                pos.Add(i);
                            }
                        }

                        if (pos.Any())
                        {
                            updated = true;
                            floatTable[_data.name] = _data.floats;
                            if (current.Length != _data.floats.Length)
                            {
                                var renderInfoArray = HioUtility.CreateIRenderInfoArrayFromTable(_data.renderInfos, stringTable, intTable, floatTable);
                                HioUtility.DebugUpdateRenderInfo(renderInfoArray);
                                G3dHioLibProxy.Hio.UpdateRenderInfo(renderInfoArray);
                            }

                            foreach (var index in pos)
                            {
                                DebugConsole.WriteLine("Hio.EditFloatRenderInfo: " + _data.name + ", " + index + ", " + (index < _data.floats.Length ? _data.floats[index] : 0));
                                G3dHioLibProxy.Hio.EditFloatRenderInfo(_data.name, index, index < _data.floats.Length ? _data.floats[index] : 0);
                            }
                        }
                    }
                    break;
            }

            if (_data.queryRenderInfo.HasValue)
            {
                if (_data.queryRenderInfo.Value)
                {
                    _data.material.RenderInfoQueried = true;
                    DebugConsole.WriteLine("QueryRenderInfoQueue.Push");
                    QueryRenderInfoQueue.Push(_data.material);
                }
            }
            else
            {
                if (updated)
                {
                    _data.material.RenderInfoQueried = true;
                    DebugConsole.WriteLine("QueryRenderInfoQueue.Push");
                    QueryRenderInfoQueue.Push(_data.material);
                }
            }

            // デバッグメッセージ
            DebugConsole.WriteLine("HIO_MaterialShaderParameter Execute  : ParamName = ({0}), Query = {1}", _data.name, _data.queryRenderInfo);
        }

        /// <summary>送る</summary>
        public static void Send(Material material, string name, render_info_typeType type, List<string> values, bool? queryRenderInfo)
        {
            Data data = new Data()
            {
                material = material,
                renderInfos = material.MaterialShaderAssign.RenderInfos.ToArray(),
                name = name,
                type = type,
                queryRenderInfo = queryRenderInfo,
                defaultEmpty = new HashSet<string>(material.MaterialShaderAssign.RenderInfos.Where(x => x.IsDefaultEmpty).Select(x => x.name)),
            };

            switch (type)
            {
                case render_info_typeType.@string:
                    data.strings = values.Where(x =>
                        {
                            if (string.IsNullOrEmpty(x))
                            {
                                return false;
                            }

                            if (!material.RenderInfoPackFromHio.IsNull)
                            {
                                var candidate = material.RenderInfoPackFromHio.StringItems.FirstOrDefault(y => y.name == name);
                                if (candidate != null)
                                {
                                    if (!candidate.Choices.Contains(x))
                                    {
                                        return false;
                                    }
                                }
                            }

                            return true;
                        }).ToArray();
                    break;
                case render_info_typeType.@int:
                    data.ints = values.Where(x => !string.IsNullOrEmpty(x)).Select(x => { int v; int.TryParse(x, out v); return v; }).ToArray();
                    break;
                case render_info_typeType.@float:
                    data.floats = values.Where(x => !string.IsNullOrEmpty(x)).Select(x => { float v; float.TryParse(x, out v); return v; }).ToArray();
                    break;

            }

            if (!QueryRenderInfoMessage.NeedToQueryRenderInfo(material))
            {
                data.queryRenderInfo = false;
            }

            Select(material);
            ArrayList targetList = new ArrayList();
            targetList.Add(material);
            (new SetRenderInfo(data, targetList)).Push();
        }
    }

    public sealed class SetRenderInfoAll : MaterialMessage
    {
        public class Data
        {
            public Material material;
            public RenderInfo[] renderInfos;
            public Dictionary<string, string[]> strings;
            public Dictionary<string, int[]> ints;
            public Dictionary<string, float[]> floats;
            public HashSet<string> defaultEmpty;
            public bool queryRenderInfo;
        }

        private readonly Data _data;
        //public string GetParamName() { return _data.ParamName; }

        /// <summary>コンストラクタ</summary>
        private SetRenderInfoAll(Data src, ArrayList targetMaterialList)
        {
            _data = src;
            _targetMaterialList = targetMaterialList;

            // このメッセージは間引き対象
            IsCompressible = true;
        }

        public override bool IsSameTarget(BaseMessage msg)
        {
            var target = msg as SetRenderInfoAll;
            if (target == null)
            {
                return false;
            }

            return target._data.material == _data.material;
        }

        /// <summary>実行</summary>
        protected override void ExecuteInternal()
        {
            bool inTransaction = true;
            HioUtility.SendRenderInfoDiff(_data.material, _data.strings, _data.ints, _data.floats, _data.defaultEmpty,_data.renderInfos, _data.queryRenderInfo, ref inTransaction);
        }

        /// <summary>送る</summary>
        public static void Send(Material material)
        {
            Data data = new Data()
            {
                material = material,
                renderInfos = material.MaterialShaderAssign.RenderInfos.ToArray(),
                queryRenderInfo = QueryRenderInfoMessage.NeedToQueryRenderInfo(material),
                defaultEmpty = new HashSet<string>(material.MaterialShaderAssign.RenderInfos.Where(x => x.IsDefaultEmpty).Select(x => x.name)),
                //renderInfoArray = CreateIRenderInfoArray(material),
            };

            CreateRenderInfoTable(material, out data.strings, out data.ints, out data.floats);
            ArrayList targetList = new ArrayList();
            targetList.Add(material);
            (new SetRenderInfoAll(data, targetList)).Push();
        }

        public static void CreateRenderInfoTable(Material material, out Dictionary<string, string[]> stringTable, out Dictionary<string, int[]> intTable, out Dictionary<string, float[]> floatTable)
        {
            stringTable = new Dictionary<string, string[]>();
            intTable = new Dictionary<string, int[]>();
            floatTable = new Dictionary<string, float[]>();
            foreach (var info in material.MaterialShaderAssign.RenderInfos)
            {
                switch (info.type)
                {
                    case render_info_typeType.@string:
                        {
                            var item = new List<string>();
                            foreach (var value in info.values)
                            {
                                if (!string.IsNullOrEmpty(value))
                                {
                                    if (!material.RenderInfoPackFromHio.IsNull)
                                    {
                                        var candidate = material.RenderInfoPackFromHio.StringItems.FirstOrDefault(y => y.name == info.name);
                                        if (candidate != null)
                                        {
                                            if (!candidate.Choices.Contains(value))
                                            {
                                                continue;
                                            }
                                        }
                                    }

                                    item.Add(value);
                                }
                            }
                            stringTable.Add(info.name, item.ToArray());
                        }
                        break;
                    case render_info_typeType.@int:
                        {
                            var item = new List<int>();
                            foreach (var value in info.values.Where(x => !string.IsNullOrEmpty(x)))
                            {
                                int v;
                                int.TryParse(value, out v);
                                item.Add(v);
                            }
                            intTable.Add(info.name, item.ToArray());
                        }
                        break;
                    case render_info_typeType.@float:
                        {
                            var item = new List<float>();
                            foreach (var value in info.values.Where(x => !string.IsNullOrEmpty(x)))
                            {
                                float v;
                                float.TryParse(value, out v);
                                item.Add(v);
                            }
                            floatTable.Add(info.name, item.ToArray());
                        }
                        break;
                }
            }
        }
    }

    public sealed class QueryRenderInfoQueue : SystemMessage
    {
        private static readonly HashSet<Material> materials = new HashSet<Material>();
        private static readonly List<Material> materialArray = new List<Material>();
        public static void Push(Material material)
        {
            if (materials.Add(material))
            {
                materialArray.Add(material);
            }
        }

        public static void Clear()
        {
            materials.Clear();
            materialArray.Clear();
        }

        public static int Count()
        {
            return materialArray.Count;
        }

        public static void Query()
        {
            if (materials.Any())
            {
                var arrayList = new ArrayList();
                foreach (var item in materialArray)
                {
                    if (item.Referrers.Any(x => x.IsAttached))
                    {
                        arrayList.Add(item);
                    }
                }
                if (arrayList.Count > 0)
                {
                    Select.SelectInternal(arrayList, GuiObjectID.Material);
                    DebugConsole.WriteLine("Hio.QueryRenderInfo");
                    G3dHioLibProxy.Hio.QueryRenderInfo(null);
                }
                Clear();
            }
        }

        public override void Execute()
        {
            if (Viewer.Manager.Instance.IsConnected)
            {
                Query();
            }
        }

        public static void Send()
        {
            (new QueryRenderInfoQueue()).Push();
        }
    }

    public sealed class QueryRenderInfoMessage : MaterialMessage
    {
        /// <summary>コンストラクタ</summary>
        private QueryRenderInfoMessage(ArrayList targetMaterialList)
        {
            _targetMaterialList = targetMaterialList;
        }

        public bool skipIfQueried = false;

        private Material[] Materials;

        /// <summary>実行</summary>
        protected override void ExecuteInternal()
        {
            foreach (var material in Materials)
            {
                if (skipIfQueried && material.RenderInfoQueried)
                {
                    continue;
                }
                material.RenderInfoQueried = true;
                DebugConsole.WriteLine("QueryRenderInfoQueue.Push");
                QueryRenderInfoQueue.Push(material);
            }

            if (QueryRenderInfoQueue.Count() > 0)
            {
                TransactionMessage.BeginTransaction(true);
                QueryRenderInfoQueue.Query();
                TransactionMessage.EndTransaction();
            }
        }

        /// <summary>送る</summary>
        public static void Send(GuiObjectGroup targets)
        {
            if (!Viewer.Manager.Instance.IsConnected)
            {
                return;
            }

            var newTargets = new GuiObjectGroup();
            var updateTargets = new List<Material>();
            foreach (var target in targets.Objects.OfType<Material>())
            {
                //if (!target.RenderInfoQueried)
                {
                    if (NeedToQueryRenderInfo(target))
                    {
                        newTargets.Add(target);
                    }
                    else
                    {
                        updateTargets.Add(target);
                    }
                }
            }

            targets = newTargets;

            var materials = targets.Objects.OfType<Material>().ToArray();
            if (materials.Length > 0)
            {

                DebugConsole.WriteLine("QueryRenderInfoMessage");
                Select(targets);
                ArrayList targetList = GetTargetList(targets);
                (new QueryRenderInfoMessage(targetList)
                    {
                        Materials = materials,
                        skipIfQueried = true,
                    }).Push();
            }
        }

        /// <summary>
        /// choice 未指定のものがあるかどうか
        /// </summary>
        public static bool NeedToQueryRenderInfo(Material material)
        {
            var shadingModel = material.MaterialShaderAssign.ShadingModel;
            if (shadingModel != null)
            {
                foreach (var item in shadingModel.RenderInfoSlots())
                {
                    if (string.IsNullOrWhiteSpace(item.choice))
                    {
                        return true;
                    }
                }
            }

            return false;
        }
    }
}
