﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading;
using System.Windows.Forms;
using App;
using App.Data;
using App.Utility;
using App.res;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;
using App.ConfigData;

namespace Viewer
{
    /// <summary>
    /// バイナリデータを読み込みます。
    /// </summary>
    public abstract class FileMessage : BaseMessage
    {
        /// <summary>メッセージカテゴリー</summary>
        protected override MessageCategory _messageCategory { get { return MessageCategory.File; } }

        /// <summary>
        /// ドキュメント(モデル、テクスチャ、アニメーション等)からnw4f_3difとBinaryStreamをバイナリデータとして出力する。
        /// </summary>
        protected static byte[] CreateBinaryData(IntermediateFileDocument intermediateFile, string filePath)
        {
            byte[] fileImage;

            using (var watch = new DebugStopWatch("Create_nw4f_3difType " + filePath))
            {
                nw4f_3difType nw4f_3dif = intermediateFile.Create_nw4f_3difType(true);
                nw4f_3dif.file_info = intermediateFile.file_info;

                // テキスト中間ファイルはバイナリ中間ファイルにする。
                List<G3dStream> streams = intermediateFile.BinaryStreams;
                // バイナリ中間ファイルの場合、nw4f_3difからストリームを削除する。
                if (G3dStreamUtility.HasStreamArray(nw4f_3dif))
                {
                    nw4f_3dif.RootElement.stream_array = null;
                }

                // バイナリデータに変換する。
                lock (Viewer.Connecter.AbortLock)
                {
                    fileImage = IfBinaryFormatter.FormatStream(nw4f_3dif, streams, null);
                }

                //////////////////////////////////////////////////////////////////////////
                // 現状バイナリコンバーターがUserDataを含む場合、アスキーにしか対応していないので、
                // アスキーでの出力に変更します。
                //////////////////////////////////////////////////////////////////////////
                //if (!G3dStreamUtility.HasStreamArray(nw4f_3dif))
                //{
                //    nw4f_3dif.RootElement.stream_array =
                //        G3dStreamUtility.ToStreamArray(intermediateFile.BinaryStreams);
                //}
                //fileImage = IfTextFormatter.FormatBytes(nw4f_3dif);

            }

            return fileImage;
        }

        static public void DisconnectOnMainThreadAsync()
        {
            {
                TheApp.MainFrame.BeginInvoke(
                    new MethodInvoker(
                        () =>
                        {
                            // キャンセル時に自動接続を無効にする
                            G3dHioLibProxy.Hio.IsAutoConnection = false;
                            Thread.MemoryBarrier();

                            App.TheApp.MainFrame.ConnectToHio(false);
                        }
                    )
                );
            }
        }
    }

    /// <summary>
    /// ファイルを閉じます。
    /// </summary>
    public sealed class Close : FileMessage
    {
        private readonly Document target_;
        private readonly Action afterExecute_;

        /// <summary>コンストラクタ</summary>
        private Close(Document target, Action afterExecute)
        {
            target_ = target;
            afterExecute_ = afterExecute;
        }

        /// <summary>実行</summary>
        public override void Execute()
        {
            if (Viewer.Manager.Instance.IsConnected)
            {
                // http://www-sdd.zelda.nintendo.co.jp/project/nintendoware3/kagemai/html/user.cgi?project=nw3_3de&action=view_report&id=802
                var targetModel = target_ as Model;
                var targetShaderDef = target_ as ShaderDefinition;
                if (targetModel != null)
                {
                    if (targetModel.PreviewingAnimations.Any())
                    {
                        using (G3dHioLibProxy.Hio.BeginCommand(false))
                        {
                            G3dHioLibProxy.Hio.UnbindAnimations(targetModel, targetModel.PreviewingAnimations.Select(x => x.Item2).ToArray());
                            targetModel.PreviewingAnimations.Clear();
                        }
                    }

                    // リターゲット情報を削除
                    foreach (var animation in targetModel.RetargetHostingAnimations)
                    {
                        G3dHioLibProxy.Hio.SetRetargetingHostModel(animation, null);
                        animation.PreviewingRetargetHost = null;
                    }
                    targetModel.RetargetHostingAnimations.Clear();

                    if (targetModel.IsSendAttached_BeforeResetStatus)
                    {
                        // SendAttach時はアンロードを呼ばない
                        ;

                        targetModel.IsSendAttached_BeforeResetStatus = false;
                    }
                    else
                    {
                        HioUtility.UnloadResource(target_);
                    }

                    HioUtility.UnloadUnusedTextures(target_);
                }
                else if (targetShaderDef != null)
                {
                    HioUtility.UnloadShaderArchive(targetShaderDef, false);
                }
                else
                {
                    var targetAnimation = target_ as AnimationDocument;

                    // リターゲット情報を削除
                    if (targetAnimation != null && targetAnimation.PreviewingRetargetHost != null)
                    {
                        G3dHioLibProxy.Hio.SetRetargetingHostModel(targetAnimation, null);
                        targetAnimation.PreviewingRetargetHost.RetargetHostingAnimations.Remove(targetAnimation);
                        targetAnimation.PreviewingRetargetHost = null;
                    }

                    HioUtility.UnloadResource(target_);

                    HioUtility.UnloadUnusedTextures(target_);
                }

                // g3d::editとの連携情報をリセット
                //target_.ResetStatus();
                target_.UnloadedFromHio();
            }
            else
            {
                // g3d::editとの連携情報をリセット
                target_.ResetStatus();
                target_.UnloadedFromHio();
            }

            if (afterExecute_ != null)
            {
                afterExecute_();
            }
        }

        /// <summary>送る</summary>
        public static void Send(Document target, Action afterExecute = null)
        {
            (new Close(target, afterExecute)).Push();
        }
    }

    /// <summary>
    /// モデルデータを読み込みます。
    /// </summary>
    public class LoadOrReloadModel : FileMessage
    {
        private readonly List<Tuple<IntermediateFileDocument, nw4f_3difType, List<G3dStream>, string>> _modelDataList = new List<Tuple<IntermediateFileDocument, nw4f_3difType, List<G3dStream>, string>>();

        // 同一確認用
        public Model model;

        // RenderInfo 用
        public Material[] materials;
        public bool[] queryRenderInfo;
        public HashSet<string>[] defaultEmpty;

        public bool canOptimizeShader;
        readonly List<Model.ShaderOptimizeData> ShaderOptimizeData = null;
        readonly List<ShaderDefinition> shaderDefinitions;
        readonly List<string> shadingModels;
        private readonly string modelName;
        public string modelFileName;
        public string dumpShaderSourcePath;
        private readonly bool optimizeShader;

        public bool ignoreLastErrorOfShaderBinarization = false;

        // リターゲット用
        private List<AnimationDocument> RetargetHostAnimations;

        /// <summary>コンストラクタ</summary>
        public LoadOrReloadModel(IntermediateFileDocument document)
        {
            DebugConsole.WriteLine("LoadOrReloadModel " + document.Name);

            model = (Model)document;
            materials = model.Materials.ToArray();
            defaultEmpty = new HashSet<string>[materials.Length];
            for (int i = 0; i < materials.Length; i++)
            {
                defaultEmpty[i] = new HashSet<string>(materials[i].MaterialShaderAssign.RenderInfos.Where(x => x.IsDefaultEmpty).Select(x => x.name));
            }

            queryRenderInfo = materials.Select(x => Viewer.QueryRenderInfoMessage.NeedToQueryRenderInfo(x)).ToArray();

            // 参照ファイル(テクスチャ)を取得する。
            var docs = GetDocuments(document).Where(x => !(x is AnimationDocument));
            foreach(var doc in docs)
            {
                // スレッドセーフにするために、IntermediateFileDocument, nw4f_3difType, List<G3dStream>を保持しておく。
                var nw4F3Dif = CopyNW4F3DIF((IntermediateFileDocument)doc);
                var g3DStreams = GetStream((IntermediateFileDocument)doc);

                // 頂点の量子化無効オプションがONの時
                if (model.DisableVertexQuantize && nw4F3Dif.Item is modelType)
                {
                    var modeltype = nw4F3Dif.Item as modelType;
                    {
                        if (modeltype.vertex_array != null)
                        {
                            foreach (var vtx in modeltype.vertex_array.vertex)
                            {
                                foreach (var attrib in vtx.vtx_attrib_array.Items)
                                {
                                    attrib.quantize_type = vtx_attrib_quantize_typeType.none;
                                }
                            }
                        }
                    }
                }

                var data =
                    new Tuple<IntermediateFileDocument, nw4f_3difType, List<G3dStream>, string>(
                        (IntermediateFileDocument)doc,
                        nw4F3Dif,
                        g3DStreams,
                        doc.Name
                        );

                _modelDataList.Add(data);
            }

            // マテリアル参照を解決する
            if (model.Materials.Any(x => x.ParentMaterials.Any()))
            {
                var resolvedModel = _modelDataList[0].Item2.Item as modelType;
                Debug.Assert(resolvedModel != null, "resolvedModel != null");
                Debug.Assert(materials.Length == resolvedModel.material_array.length);
                ResolveParentMaterials(model, resolvedModel);
            }

            // PreBinalize command
            if (Viewer.Manager.Instance.IsConnected && ApplicationConfig.FileIo.PreBinarizeCommand.HasCommand)
            {
                var newlist = PrePostIO.ExecutePreBinarize(_modelDataList);
                if (newlist != null && newlist.Any())
                {
                    _modelDataList = newlist;
                }
            }

            ShaderOptimizeData = HioUtility.CreateShaderOptimizeData(model, out shaderDefinitions, out shadingModels);

            canOptimizeShader = true;
            modelName = model.Name;

            dumpShaderSourcePath = ApplicationConfig.UserSetting.IO.DumpShaderSource ? TemporaryFileUtility.ShaderFolderPath : null;

            RetargetHostAnimations = DocumentManager.Animations.Where(x => x.RetargetingHostModelName == model.Name).ToList();
            optimizeShader = DocumentManager.OptimizeShader;
        }

        /// <summary>実行</summary>
        public override void Execute()
        {
            if (Viewer.Manager.Instance.IsConnected)
            {
                using (var block = new LoadProgressDialog.DialogBlock(Strings.Viewer_Loading, () => DisconnectOnMainThreadAsync(), true))
                {
                    var model = _modelDataList[0].Item1 as Model;

                    if (model.HioLoaded)
                    {
                        Debug.Assert(model.IsAttached);

                        if (canOptimizeShader)
                        {
                            bool binarizeError = false;
                            if (HioUtility.LoadOptimizedShader(model, ShaderOptimizeData, _modelDataList, out binarizeError, true, shaderDefinitions, shadingModels, block, modelName, dumpShaderSourcePath, optimizeShader, ignoreLastErrorOfShaderBinarization))
                            {
                                // シェーダーの最適化時にモデルは再転送済み

                                // RenderInfo を送る
                                HioUtility.SendRenderInfoAfterReloadModel(_modelDataList[0].Item2.Item as modelType, materials, queryRenderInfo, defaultEmpty);
                                return;
                            }

                            if (binarizeError)
                            {
                                return;
                            }
                        }

                        bool unloaded;
                        if (HioUtility.ReloadModel(_modelDataList, out unloaded, block, modelName))
                        {
                            // RenderInfo を送る
                            HioUtility.SendRenderInfoAfterReloadModel(_modelDataList[0].Item2.Item as modelType, materials, queryRenderInfo, defaultEmpty);
                        }
                    }
                    else
                    {
                        Action<Model> action;
                        lock (Connecter.BeforeLoadModelActions)
                        {
                            if (Connecter.BeforeLoadModelActions.TryGetValue(modelName, out action))
                            {
                                Connecter.BeforeLoadModelActions.Remove(modelName);
                            }
                        }
                        if (action != null)
                        {
                            action(model);
                        }

                        var modelBinaryPath = HioUtility.LoadModel(_modelDataList, materials, defaultEmpty, () => LoadProgressDialog.Hide(), block, modelName);

                        if (modelBinaryPath == null)
                        {
                            return;
                        }

                        // LODが選択されているときは送る
                        if (model.PreviewLodLevel >= 0)
                        {
                            EditSetShapeLodLevel.Send(model, model.PreviewLodLevel);
                        }
                        // 必要な時はシェーダーバイナリを送る
                        if (canOptimizeShader && ShaderOptimizeData.Any())
                        {
                            foreach (var material in ShaderOptimizeData.SelectMany(x => x.materials))
                            {
                                material.forceOptimize = material.optimize;
                            }
                            bool binarizeError;
                            HioUtility.LoadOptimizedShader(model, ShaderOptimizeData, _modelDataList, out binarizeError, true, shaderDefinitions, shadingModels, block, modelName, dumpShaderSourcePath, optimizeShader, ignoreLastErrorOfShaderBinarization);
                        }

                        // リターゲットの設定
                        foreach (var animation in RetargetHostAnimations)
                        {
                            if (animation.HioLoaded)
                            {
                                G3dHioLibProxy.Hio.SetRetargetingHostModel(animation, model);
                                animation.PreviewingRetargetHost = model;
                                model.RetargetHostingAnimations.Add(animation);
                            }
                        }

                        // RenderInfo は送らない
                    }
                }
            }
        }

        /// <summary>送る</summary>
        public static void Send(Model document)
        {
            if (DocumentManager.Models.Contains((Model)document) && document.IsVisible)
            {
                (new LoadOrReloadModel(document)).Push();
                Connecter.PulseMessageThread();
            }
        }

        public static void Send(Model document, bool ignoreLastErrorOfShaderBinarization)
        {
            if (DocumentManager.Models.Contains((Model)document) && document.IsVisible)
            {
                (new LoadOrReloadModel(document) { ignoreLastErrorOfShaderBinarization = ignoreLastErrorOfShaderBinarization }).Push();
                Connecter.PulseMessageThread();
            }
        }

        internal static void ResolveParentMaterials(Model model, modelType resolvedModel)
        {
            Debug.Assert(model.Materials.Count == resolvedModel.material_array.length);

            for (var i = 0; i < model.Materials.Count; i++)
            {
                var material = model.Materials[i];
                var resolvedMaterialType = resolvedModel.material_array.material[i];
                if (material.ParentMaterials == null || material.ParentMaterials.Count == 0 || !material.ParentMaterialsValidity())
                {
                    continue;
                }
                var resolvedShaderAssign = resolvedMaterialType.shader_assign;
                var materialShaderAssign = material.MaterialShaderAssign;
                if (resolvedShaderAssign == null || materialShaderAssign == null)
                {
                    continue;
                }

                foreach (var option in materialShaderAssign.ShaderOptions)
                {
                    var target =
                        resolvedShaderAssign.shader_option_array?.shader_option?.FirstOrDefault(
                            x => x.id == option.id);
                    if (target != null)
                    {
                        target.value = material.GetResolvedOptionValue(option);
                    }
                }

                var resolvedSamplerAssigns = materialShaderAssign.SamplerAssigns
                    .Select(x => new sampler_assignType() {id = x.id, sampler_name = material.GetResolvedSamplerAssignValue(x)})
                    .Where(x => !string.IsNullOrEmpty(x.sampler_name)).ToArray();
                if (resolvedSamplerAssigns.Any())
                {
                    if (resolvedShaderAssign.sampler_assign_array == null)
                    {
                        resolvedShaderAssign.sampler_assign_array = new sampler_assign_arrayType();
                    }
                    resolvedShaderAssign.sampler_assign_array.sampler_assign = resolvedSamplerAssigns;
                }
                else
                {
                    resolvedShaderAssign.sampler_assign_array = null;
                }

                foreach (var shaderParam in materialShaderAssign.ShaderParams)
                {
                    var target =
                        resolvedShaderAssign.shader_param_array?.shader_param?.FirstOrDefault(
                            x => x.id == shaderParam.id);
                    if (target != null)
                    {
                        target.Value = material.GetResolvedShaderParamValue(shaderParam);
                    }
                }

                var resolvedAttribAssigns = materialShaderAssign.AttribAssigns.Select(x =>
                {
                    var attribName = material.GetResolvedAttribAssignValue(x);
                    return string.IsNullOrEmpty(attribName) ? null : new attrib_assignType() {id = x.id, attrib_name = attribName};
                }).Where(x => x != null).ToArray();
                if (resolvedAttribAssigns.Any())
                {
                    if (resolvedShaderAssign.attrib_assign_array == null)
                    {
                        resolvedShaderAssign.attrib_assign_array = new attrib_assign_arrayType() {attrib_assign = resolvedAttribAssigns};
                    }
                    else
                    {
                        resolvedShaderAssign.attrib_assign_array.attrib_assign = resolvedAttribAssigns;
                    }
                }

                foreach (var renderInfo in materialShaderAssign.RenderInfos)
                {
                    var target =
                        resolvedShaderAssign.render_info_array?.render_info?.FirstOrDefault(
                            x => x.name == renderInfo.name);
                    if (target != null)
                    {
                        var values = material.GetResolvedRenderInfoValues(renderInfo);
                        var renderInfoType = ShaderAssignUtility.To_render_info(renderInfo, values.ToArray());
                        target.Value = renderInfoType.Value;
                    }
                }

                var samplers = ObjectUtility.Clone(material.ResolvedSamplers);
                if (samplers.Any())
                {
                    if (resolvedMaterialType.sampler_array == null)
                    {
                        resolvedMaterialType.sampler_array = new sampler_arrayType();
                    }
                    resolvedMaterialType.sampler_array.sampler = samplers;
                }
                else
                {
                    resolvedMaterialType.sampler_array = null;
                }
            }
        }
    }

    /// <summary>
    /// アニメーションデータを読み込みます。
    /// </summary>
    public sealed class LoadOrReloadAnimation : FileMessage
    {
        public AnimationDocument OriginalTarget;
        private readonly List<Tuple<IntermediateFileDocument, nw4f_3difType, List<G3dStream>, string>> _AnimDataList = new List<Tuple<IntermediateFileDocument, nw4f_3difType, List<G3dStream>, string>>();

        private readonly List<Tuple<Model, List<Tuple<string, AnimationDocument>>, Dictionary<AnimationDocument, float>>> parentModels_ = new List<Tuple<Model, List<Tuple<string, AnimationDocument>>, Dictionary<AnimationDocument, float>>>();

        private readonly bool canBind = true;

        private Model retargetHostModel = null;





        #region 暫定対処
        private readonly Dictionary<AnimationDocument, List<Tuple<IntermediateFileDocument, nw4f_3difType, List<G3dStream>, string>>> _AnimDataLists
    = new Dictionary<AnimationDocument, List<Tuple<IntermediateFileDocument, nw4f_3difType, List<G3dStream>, string>>>();
        private readonly Dictionary<Model, bool> pauses = new Dictionary<Model, bool>();
        private readonly Dictionary<Model, float> pausingFrames = new Dictionary<Model, float>();
        private Tuple<string, AnimationDocument> animations_;
        #endregion

        /// <summary>コンストラクタ</summary>
        public LoadOrReloadAnimation(AnimationDocument target)
        {
            OriginalTarget = target;

            var docs = GetDocuments(target);
            foreach (var doc in docs)
            {
                // スレッドセーフにするために、IntermediateFileDocument, nw4f_3difType, List<G3dStream>を保持しておく。
                var data =
                    new Tuple<IntermediateFileDocument, nw4f_3difType, List<G3dStream>, string>(
                        (IntermediateFileDocument)doc, CopyNW4F3DIF((IntermediateFileDocument)doc), GetStream((IntermediateFileDocument)doc), doc.Name
                        );
                switch (data.Item1.ObjectID)
                {
                    case ConfigCommon.GuiObjectID.MaterialAnimation:
                        // オリジナル情報がありカーブがないとバイナライズでエラーになる? ので、ランタイム連携向けにはオリジナルを削除
                        ((material_animType)data.Item2.Item).original_per_material_anim_array = null;
                        break;
                    case ConfigCommon.GuiObjectID.ShaderParameterAnimation:
                    case ConfigCommon.GuiObjectID.ColorAnimation:
                    case ConfigCommon.GuiObjectID.TextureSrtAnimation:
                        // オリジナル情報がありカーブがないとバイナライズでエラーになるので、ランタイム連携向けにはオリジナルを削除
                        ((shader_param_animType)data.Item2.Item).original_material_anim_array = null;
                        break;
                }
                _AnimDataList.Add(data);
            }

            // PreBinalize command
            if (Viewer.Manager.Instance.IsConnected && ApplicationConfig.FileIo.PreBinarizeCommand.HasCommand)
            {
                var newlist = PrePostIO.ExecutePreBinarize(_AnimDataList);
                if (newlist != null && newlist.Any())
                {
                    _AnimDataList = newlist;
                }
            }

            //pauses[target] = pause;
            //pausingFrames[target] = pausingFrame;
            _AnimDataLists[target] = _AnimDataList;
            animations_ = new Tuple<string, AnimationDocument>(target.FileName, target);

            canBind = AnimationMessage.CanBind(target);

            // バインドの更新
            foreach (var doc in DocumentManager.Models)
            {
                if (doc.PreviewAnimSet != null &&
                    doc.PreviewAnimSet.Animations.Contains(new AnimationSetItem(target.FileName, target.FileLocation)) &&
                    !target.Pause.InvisibleBinds.Contains(new KeyValuePair<object,string>(doc.ModelId, doc.PreviewAnimSet.Name)))
                {
                    var animations = DocumentManager.Animations.Where(x => AnimationMessage.CanBind(x) && doc.PreviewAnimSet.Animations.Contains(new AnimationSetItem(x.FileName, x.FileLocation)))
                        .Select(x => new Tuple<string, AnimationDocument>(x.FileName, x)).ToList();
                    var pausingFrames = new Dictionary<AnimationDocument,float>();
                    foreach (var animation in animations.Select(x => x.Item2))
                    {
                        double frame;
                        if (animation.PauseFrames.TryGetValue(new KeyValuePair<object,string>(doc.ModelId, doc.PreviewAnimSet.Name), out frame))
                        {
                            pausingFrames[animation] = (float)frame;
                        }
                    }
                    parentModels_.Add(new Tuple<Model, List<Tuple<string, AnimationDocument>>, Dictionary<AnimationDocument, float>>(
                        doc,
                        animations,
                        pausingFrames
                        ));
                }
            }

            retargetHostModel = target.GetRetargetHost();
        }

        /// <summary>実行</summary>
        public override void Execute()
        {
            if (Viewer.Manager.Instance.IsConnected)
            {
                // アニメーション転送時はダイアログを表示しない
                //using (var block = new LoadProgressDialog.DialogBlock(Strings.Viewer_Loading, () => DisconnectOnMainThreadSync(), true))
                {
                    if (OriginalTarget.HioLoaded)
                    {
                        // バインド可能かどうかで送る順番を変更
                        if (!canBind)
                        {
                            SendBinds2();
                            //SendBinds();
                        }

                        var result = HioUtility.ReloadAnimation(_AnimDataList);

                        // 必要ならリターゲットの設定を更新
                        HioUtility.SetRetargetHostModel(OriginalTarget, retargetHostModel);

                        if (canBind)
                        {
                            SendBinds2();
                        }
                        // フレームを設定する
                        if (AnimationMessage.IsPlaying)
                        {
                            G3dHioLibProxy.Hio.PlayFrameCtrl(AnimationMessage.CurrentFrame);
                        }
                        else
                        {
                            G3dHioLibProxy.Hio.StopFrameCtrl(AnimationMessage.CurrentFrame);
                        }
                    }
                    else
                    {
                        var result = HioUtility.LoadAnim(_AnimDataList, () => LoadProgressDialog.Hide());

                        // リターゲットの設定
                        if (retargetHostModel != null)
                        {
                            if (retargetHostModel.HioLoaded)
                            {
                                G3dHioLibProxy.Hio.SetRetargetingHostModel(OriginalTarget, retargetHostModel);
                                OriginalTarget.PreviewingRetargetHost = retargetHostModel;
                                retargetHostModel.RetargetHostingAnimations.Add(OriginalTarget);
                            }
                        }
                    }
                }
            }
        }

        // 複数モデルへのバインドの仮対応。SendBinds の代わり
        private void SendBinds2()
        {
            foreach (var tuple in parentModels_)
            {
                var model = tuple.Item1;
                if (!canBind)
                {
                    var newAnimations = model.PreviewingAnimations.Where(x => x.Item2 != OriginalTarget).ToArray();
                    HioUtility.UpdateAnimationBinds(model, newAnimations, tuple.Item3, false);
                }
                else if (!model.PreviewingAnimations.Any(x => x.Item2 == OriginalTarget))
                {
                    var newAnimations = model.PreviewingAnimations.Concat(Enumerable.Repeat(animations_,1)).ToArray();
                    HioUtility.UpdateAnimationBinds(model, newAnimations, tuple.Item3, false);
                }
            }
        }

        /// <summary>送る</summary>
        public static void Send(AnimationDocument document)
        {
            // 不正アニメーションチェック
            document.CheckAndDisConnect();

            if (Viewer.Manager.Instance.IsConnected)
            {
                if (!document.IsFileOutputable)
                {
                    // 先に切断してから
                    TheApp.MainFrame.ConnectToHio(false);

                    var builder = new StringBuilder();

                    builder.AppendFormat(Strings.Viewer_InvalidAnimationDisconnect, document.FileName);

                    builder.AppendLine();
                    builder.Append(document.FileNotOutputErrorMessage);

                    // メッセージ表示する。
                    App.AppContext.NotificationHandler.MessageBoxError(builder.ToString());

                    return;
                }
            }

            (new LoadOrReloadAnimation(document)).Push();
            Connecter.PulseMessageThread();
        }
    }

    /// <summary>
    /// テクスチャのアンロードを一時停止を設定する
    /// 接続先に直接情報を送信するわけではない
    /// </summary>
    public sealed class SuspendUnloadTexture : SystemMessage
    {
        /// <summary>
        /// テクスチャのアンロードが一時停止されているか？
        /// </summary>
        public static bool Suspended
        {
            get
            {
                lock (CountLock)
                {
                    return Count > 0;
                }
            }
        }

        /// <summary>
        /// 一時停止を解除する
        /// </summary>
        public static void Reset()
        {
            lock (CountLock)
            {
                Count = 0;
            }
        }

        private static object CountLock = new object();
        private static int Count = 0;
        private bool suspend;
        private SuspendUnloadTexture(bool suspend)
        {
            this.suspend = suspend;
        }

        public override void Execute()
        {
            if (Viewer.Manager.Instance.IsConnected)
            {
                lock (CountLock)
                {
                    if (suspend)
                    {
                        Count++;
                    }
                    else
                    {
                        Debug.Assert(Count > 0);
                        Count = Math.Max(0, Count - 1);

                        // ロックが解除されたら不要なテクスチャを除く
                        if (Count == 0)
                        {
                            HioUtility.UnloadUnusedTexturesAll();
                        }
                    }
                }
            }
        }

        public static void Send(bool suspend)
        {
            new SuspendUnloadTexture(suspend).Push();
        }

    }
}
