﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading;
using App;
using App.Controls;
using App.Data;
using App.Utility;
using App.res;
using ConfigCommon;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;
using System.IO;

namespace Viewer
{
    static partial class HioUtility
    {
        // HIO専用メッセージブロック
        public class HioExclusiveMessageBlock : IDisposable
        {
            private static int depth_ = 0;

            public static bool IsEnabled
            {
                get
                {
                    return depth_ > 0;
                }
            }
            public HioExclusiveMessageBlock()
            {
                ++ depth_;
            }

            public void Dispose()
            {
                -- depth_;
            }
        }

        public static bool IsValidSelectType(GuiObjectID id)
        {
        	return
        		(id == GuiObjectID.Material) ||
        		IsHioSupportedAnimation(id);
        }

        public static void Initialize()
        {
            ;
        }

        public static void Destroy()
        {
        }

        // ビューアの描画抑制ブロックに入っている最中か？
        private static int inViewerDrawSuppressBlock_ = 0;

        // ビューアの描画抑制ブロックに入っているときに読みこもうとしたモデル。
        private static readonly List<Model>					viewerDrawSuppressBlockModel_			= new List<Model>();

        // ビューアの描画抑制ブロックに入っているときに読みこもうとしたアニメーション。
        private static readonly List<AnimationDocument>		viewerDrawSuppressBlockAnim_			= new List<AnimationDocument>();

        // ビューアの描画抑制ブロックに入っているときにバインドしようとしたアニメーション。
        private static readonly List<Model>
                                                    viewerDrawSuppressBlockBindAnim_		= new List<Model>();

        // ビューアの描画抑制ブロックに入っているときにバインドしようとしたシーンアニメーション。
        private static bool viewerDrawSuppressBlockBindSceneAnim_ = false;

        public static void ClearBlockedMessages()
        {
            viewerDrawSuppressBlockModel_.Clear();
            viewerDrawSuppressBlockAnim_.Clear();
            viewerDrawSuppressBlockBindAnim_.Clear();
            viewerDrawSuppressBlockBindSceneAnim_ = false;
        }

        public static void BeginViewerDrawSuppressBlock()
        {
            inViewerDrawSuppressBlock_ ++;
        }

        public static void EndViewerDrawSuppressBlock()
        {
            inViewerDrawSuppressBlock_--;
            if (inViewerDrawSuppressBlock_ != 0)
            {
                return;
            }

            // ビューアの描画抑制ブロックに入っているときに読みこもうとしたモデルを実際に読み込む
            foreach(var model in viewerDrawSuppressBlockModel_)
            {
                //				LoadModel(model);
                Viewer.LoadOrReloadModel.Send(model);
            }

            // ビューアの描画抑制ブロックに入っているときに読みこもうとしたアニメーションを実際に読み込む
            foreach(var anim in viewerDrawSuppressBlockAnim_)
            {
                //				LoadAnim(anim);

                // 不正アニメーションチェック
                anim.CheckAndDisConnect();
                Viewer.LoadOrReloadAnimation.Send(anim);
            }

            // 閉じられるモデルはここでバインドを更新しなくても更新される。
            // ビューアの描画抑制ブロックに入っているときにバインドしようとしたアニメーションを実際にバインドする
#if false // 暫定対応
            Viewer.TransactionMessage.Send(true, overInterval: true);
#endif
            DebugConsole.WriteLine("viewerDrawSuppressBlockBindAnim_ " + viewerDrawSuppressBlockBindAnim_.Count);
            foreach (var model in viewerDrawSuppressBlockBindAnim_.Distinct().Intersect(DocumentManager.Models))
            {
                AnimationDocument[] animations;
                if (model.PreviewAnimSet != null)
                {
                    var key = new KeyValuePair<object, string>(model.ModelId, model.PreviewAnimSet.Name);
                    animations = DocumentManager.GetAnimations(model.PreviewAnimSet.Animations).Where(x => !x.Pause.InvisibleBinds.Contains(key)).ToArray();
                }
                else
                {
                    animations = new AnimationDocument[0];
                }

                // アニメーションを追加
                foreach (var animation in animations)
                {
                    animation.SetPreviewAnimationSet(model.ModelId, model.PreviewAnimSet.Name);
                }

                // 対象外のアニメーションを更新
                foreach (var animation in DocumentManager.Animations.Except(animations))
                {
                    animation.ClearPreviewAnimationSet(model.ModelId);
                }

                BindAnimations(model, animations);
            }
#if false // 暫定対応
            Viewer.TransactionMessage.Send(false, overInterval: true);
#endif
            // ビューアの描画抑制ブロックに入っているときにバインドしようとしたアニメーションを実際にバインドする
            if (viewerDrawSuppressBlockBindSceneAnim_)
            {
                if (DocumentManager.PreviewSceneAnimSet != null)
                {
//					var animations = DocumentManager.Animations.Where(x => DocumentManager.PreviewSceneAnimSet.Animations.Contains(x.FileName)).ToArray();
                    var animations = DocumentManager.GetAnimations(DocumentManager.PreviewSceneAnimSet.Animations).OfType<SceneAnimation>().Where(x => !x.InvisibleBinds.Contains(DocumentManager.PreviewSceneAnimSet)).ToArray();
                    BindSceneAnimations(animations);
                }
                else
                {
                    BindSceneAnimations(Enumerable.Empty<AnimationDocument>());
                }
            }

            viewerDrawSuppressBlockModel_.Clear();
            viewerDrawSuppressBlockAnim_.Clear();
            viewerDrawSuppressBlockBindAnim_.Clear();
            viewerDrawSuppressBlockBindSceneAnim_ = false;
        }

        public static void AddViewerDrawSuppressBlockModel(Model model)
        {
            if (inViewerDrawSuppressBlock_ > 0)
            {
                viewerDrawSuppressBlockModel_.Add(model);
            }
        }
        public static void AddViewerDrawSuppressBlockAnim(AnimationDocument anim)
        {
            if (inViewerDrawSuppressBlock_ > 0)
            {
                viewerDrawSuppressBlockAnim_.Add(anim);
            }
        }

        public static void AddViewerDrawSuppressBlockBindAnim(Model model)
        {
            Debug.Assert(inViewerDrawSuppressBlock_ > 0);
            if (inViewerDrawSuppressBlock_ > 0)
            {
                viewerDrawSuppressBlockBindAnim_.Add(model);
            }
        }

        public static void AddViewerDrawSuppressBlockBindSceneAnim()
        {
            if (inViewerDrawSuppressBlock_ > 0)
            {
                // 入っていなかったら、入れる
                viewerDrawSuppressBlockBindSceneAnim_ = true;
            }
        }

        // バイナリリソースを作る
        // isIncludeDependData: 依存するデータも含めるかどうか？
        private static string MakeBinaryResource(
            List<Tuple<IntermediateFileDocument, nw4f_3difType, List<G3dStream>, string>> dataList,
            List<string> errMsgs,
            bool reloadModel,
            TeamConfig.PlatformPreset platform,
            out uint alignmentSize)
        {
            string fileName = null;

            // Load 時と変わらない Texture は送らない
            var filtered = dataList.Where(x =>
            {
                if (!reloadModel)
                {
                    DebugConsole.WriteLine("Binarize {0}", x.Item4);
                    return true;
                }

                if (x.Item1.ObjectID != GuiObjectID.Texture)
                {
                    DebugConsole.WriteLine("Binarize {0}", x.Item4);
                    return true;
                }

                // Nw 向け以外は bfres にテクスチャを含めない
                if (!platform.UseNw)
                {
                    return false;
                }

                var model = dataList[0].Item1 as Model;
                Tuple<byte[], List<G3dStreamCachedComparer>> tuple;
                if (model.loadedTextures.TryGetValue(x.Item4, out tuple))
                {
                    if (tuple.Item2.Count == x.Item3.Count &&
                        tuple.Item2.Zip(x.Item3, (y, z) => y.IsSame(z)).All(y => y))
                    {
                        var bytes = ObjectUtility.ToBytes(x.Item2.Item);
                        if (bytes.Length == tuple.Item1.Length &&
                            bytes.SequenceEqual(tuple.Item1))
                        {
                            DebugConsole.WriteLine("Don't binarize {0}", x.Item4);
                            return false;
                        }
                    }
                }

                DebugConsole.WriteLine("Binarize {0}", x.Item4);
                return true;
            }).ToList();

            {
                Document doc = dataList[0].Item1;
                fileName = TemporaryFileUtility.MakeTemporaryFileName(".bfres");
                {
                    var filePath = App.BinaryConverterManager.ConvertToBinary(filtered, fileName, errMsgs, true, platform, out alignmentSize);

                    if (string.IsNullOrEmpty(filePath) || errMsgs.Any())
                    {
                        // エラー
                        return null;
                    }
                }
            }

            return fileName;
        }

        public static string LoadModel(
            List<Tuple<IntermediateFileDocument, nw4f_3difType, List<G3dStream>, string>> modelDataList,
            Material[] materials,
            HashSet<string>[] defaultEmptyRenderInfo,
            Action onSkip,
            LoadProgressDialog.DialogBlock block,
            string modelName)
        {
            var model = (Model)(modelDataList[0].Item1);

            DebugConsole.WriteLine("G3dHioLibProxy.LoadModel(); ---------------------------------------------{0}", model.FileName);

            if (model.IsAllMaterialShaderAssigned == false)
            {
                if (onSkip != null)
                {
                    onSkip();
                }

                UIMessageBox.Warning(Strings.HIO_IsNotAllMaterialShaderAssigned);
                return null;
            }

            var errMsgs = new List<string>();
            var tmp = block.Message;
            block.Message = string.Format(Strings.Viewer_ConvertingModel, modelName);

            // バイナリ生成
            uint alignmentSize;
            var fileName = MakeBinaryResource(modelDataList, errMsgs, false, Connecter.Platform, out alignmentSize);

            if (fileName == null)
            {
                if (onSkip != null)
                {
                    onSkip();
                }

                // エラーログ出力
                ErrorLog.WriteBinaryErrorLog(errMsgs);

                // バイナリ化失敗エラーメッセージ
                BinaryConverterManager.ShowBinalizeErrorMsg(errMsgs);

#if true
                // TODO: 問題なければ削除
                if (model.IsAttached)
                {
                    // 来ないはず
                    Debug.Assert(false);
                    UnloadModelInternal(model);

                    //model.ResetStatus();
                    model.UnloadedFromHio();
                }
#endif
            }
            else
            {
                var textures = new List<Texture>();
                if (Connecter.Platform.UseNw)
                {
                    model.loadedTextures.Clear();
                    foreach (var item in modelDataList)
                    {
                        if (item.Item1.ObjectID == GuiObjectID.Texture)
                        {
                            model.loadedTextures.Add(item.Item4, new Tuple<byte[], List<G3dStreamCachedComparer>>(
                                ObjectUtility.ToBytes(item.Item2.Item),
                                item.Item3.Select(x => new G3dStreamCachedComparer(x)).ToList()));
                        }
                    }
                }
                else
                {
                    // テクスチャを送る
                    var convertErrors = new List<string>();
                    foreach (var item in modelDataList)
                    {
                        if (item.Item1.ObjectID == GuiObjectID.Texture)
                        {
                            if (LoadOrReloadOrSkipTexture((Texture) item.Item1, item.Item2, item.Item3, item.Item4,
                                block))
                            {
                                HioUtility.AddTextureBinding((Texture)item.Item1, model);
                                textures.Add((Texture)item.Item1);
                            }
                            else
                            {
                                DebugConsole.WriteLine("ConvertTextureBinary Failed  --------------- " + item.Item4);
                                convertErrors.Add(string.IsNullOrEmpty(item.Item1?.FilePath) ? item.Item4 : item.Item1.FilePath);
                            }
                        }
                    }
                    if (convertErrors.Any())
                    {
                        TheApp.MainFrame.BeginInvoke((Action)(() =>
                        {
                            using (var dialog = new OkListBoxDialog())
                            {
                                dialog.Text = Strings.TextureBinariesError;
                                dialog.lblDescription.Text = null;
                                foreach (var error in convertErrors)
                                {
                                    dialog.AddLine(error);
                                }
                                dialog.ShowDialog();
                            }
                        }));
                    }
                }

                model.HioLoaded = true;
                if (model.IsSendAttached)
                {
                    DebugConsole.WriteLine("G3dHioLibProxy.Hio.SendAttachModel(); ---------------------------------------------");

                    //LoadCounter.BeginLoad();
                    G3dHioLibProxy.Hio.SendAttachModel(model, new NintendoWare.G3d.Edit.FileData() { FileName = fileName, Alignment = alignmentSize });

                    if (!Connecter.Platform.UseNw)
                    {
                        // TODO: 必要なら更新があったときだけ送る
                        DebugConsole.WriteLine("HIO.SetTextureBindings --------------- " + model.Name + " " + textures.Count());
                        G3dHioLibProxy.Hio.SetTextureBindingsForModel(model, textures.OfType<NintendoWare.G3d.Edit.IEditTextureTarget>().ToArray());
                    }

                    lock (DocumentManager.PreviewingModels)
                    {
                        DocumentManager.PreviewingModels.Add(model);
                    }
                }
                else
                {
                    using (var waitEvent = new System.Threading.ManualResetEventSlim())
                    {
                        uint Key = 0;
                        try
                        {
                            // 下位2bit が 00 にならない値
                            Key = model.Key = (keyCount << 2) | 1;
                            keyCount++;

                            //model.WaitEventIsAttached = waitEvent;

                            DebugConsole.WriteLine("G3dHioLibProxy.Hio.LoadModel(): {0} {1}", modelName, alignmentSize);
                            block.Message = string.Format(Strings.Viewer_LoadingModel, modelName);


                            lock (Connecter.WaitingModels)
                            {
                                Connecter.WaitingModels[model.Key] = new Tuple<Model, ManualResetEventSlim>(model, waitEvent);
                            }

                            G3dHioLibProxy.Hio.LoadModel(model, new NintendoWare.G3d.Edit.FileData() { FileName = fileName, Alignment = alignmentSize });

                            while (!waitEvent.Wait(1000) && Manager.Instance.IsConnected){}

                            if (Manager.Instance.IsConnected)
                            {
                                DebugConsole.WriteLine("G3dHioLibProxy.Hio.LoadModel() : --------------------------------------------- ok");
                            }

                            if (!Connecter.Platform.UseNw)
                            {
                                // TODO: 必要なら更新があったときだけ送る
                                DebugConsole.WriteLine("HIO.SetTextureBindings --------------- " + model.Name + " " + textures.Count());
                                G3dHioLibProxy.Hio.SetTextureBindingsForModel(model, textures.OfType<NintendoWare.G3d.Edit.IEditTextureTarget>().ToArray());
                            }

                            lock (DocumentManager.PreviewingModels)
                            {
                                DocumentManager.PreviewingModels.Add(model);
                            }
                        }
                        finally
                        {
                            lock (Connecter.WaitingModels)
                            {
                                Connecter.WaitingModels.Remove(Key);
                            }
                        }
                    }

                }

                modelType modelData = modelDataList[0].Item2.Item as modelType;
                // RenderInfo のテーブルを更新
                int i = 0;
                foreach (var material in materials)
                {
                    var materialData = modelData.material_array.GetItems().FirstOrDefault(x => x.name == material.Name);
                    CreateRenderInfoTable(materialData, out material.LastRenderInfo.StringRenderInfoHio, out material.LastRenderInfo.IntRenderInfoHio, out material.LastRenderInfo.FloatRenderInfoHio);
                    material.LastRenderInfo.DefaultEmptyRenderInfoHio = defaultEmptyRenderInfo[i];
                    i++;
                }
            }
            block.Message = tmp;

            return fileName;
        }

        public static void CreateRenderInfoTable(materialType material, out Dictionary<string, string[]> stringTable, out Dictionary<string, int[]> intTable, out Dictionary<string, float[]> floatTable)
        {
            stringTable = new Dictionary<string,string[]>();
            intTable = new Dictionary<string,int[]>();
            floatTable = new Dictionary<string,float[]>();
            Debug.Assert(material.shader_assign != null);
            foreach (var info in material.shader_assign.render_info_array.GetItems())
            {
                var tokens = G3dDataParser.Tokenize(info.Value ?? "");
                switch (info.type)
                {
                    case render_info_typeType.@string:
                        stringTable.Add(info.name, tokens.Where(x => !string.IsNullOrEmpty(x)).ToArray());
                        break;
                    case render_info_typeType.@int:
                        int[] ints = tokens.Select(x => { int v; int.TryParse(x, out v); return v; }).ToArray();
                        intTable.Add(info.name, ints);
                        break;
                    case render_info_typeType.@float:
                        float[] floats = tokens.Select(x => { float v; float.TryParse(x, out v); return v; }).ToArray();
                        floatTable.Add(info.name, floats);
                        break;
                }
            }
        }

        private static uint keyCount = 0;
        public static bool LoadAnim(List<Tuple<IntermediateFileDocument, nw4f_3difType, List<G3dStream>, string>> animDataList, Action onSkip)
        {
            AnimationDocument anim = (AnimationDocument)(animDataList[0].Item1);
            bool result = false;
            var errMsgs = new List<string>();
            uint alignmentSize;
            var fileName = MakeBinaryResource(animDataList, errMsgs, false, Connecter.Platform, out alignmentSize);
            if (fileName == null)
            {
                if (onSkip != null)
                {
                    onSkip();
                }

                // エラーログ出力
                ErrorLog.WriteBinaryErrorLog(errMsgs);

                // バイナリ化失敗エラーメッセージ
                BinaryConverterManager.ShowBinalizeErrorMsg(errMsgs);

                // 自動接続を無効にする
                G3dHioLibProxy.Hio.IsAutoConnection = false;

                // TODO: ランタイム連携切断する必要があるか検証
                TheApp.MainFrame.Invoke(new System.Windows.Forms.MethodInvoker(()=>
                    Viewer.ViewerUtility.Disconnect(Strings.BinalizeError_Disconnect, false, false)));
            }
            else
            {
                var textures = new List<Texture>();
                if (!Connecter.Platform.UseNw)
                {
                    // テクスチャを送る
                    foreach (var item in animDataList)
                    {
                        if (item.Item1.ObjectID == GuiObjectID.Texture)
                        {
                            LoadOrReloadOrSkipTexture((Texture)item.Item1, item.Item2, item.Item3, item.Item4, null);
                            HioUtility.AddTextureBinding((Texture)item.Item1, anim);
                            textures.Add((Texture)item.Item1);
                        }
                    }
                }

                var resetEvent = new ManualResetEventSlim();
                Action fileLoaded = () => resetEvent.Set();
                Viewer.Manager.Instance.AddFileLoadedAction(fileLoaded);

                uint Key = 0;
                try
                {
                    DebugConsole.WriteLine("G3dHioLibProxy.Hio.Load***Animation(); ---------------------------------------------");

                    // 下位2bit が 00 にならない値
                    Key = anim.Key = (keyCount << 2) | 1;
                    keyCount++;

                    lock (Connecter.WaitingLoadFiles)
                    {
                        Connecter.WaitingLoadFiles[Key] = anim;
                    }

                    //G3dHioLibProxy.Hio.LoadTexture(null, new NintendoWare.G3d.Edit.FileData() { });
                    var fileData = new NintendoWare.G3d.Edit.FileData() { FileName = fileName, Alignment = alignmentSize };
                    switch (anim.ObjectID)
                    {
                        case GuiObjectID.MaterialAnimation: result = G3dHioLibProxy.Hio.LoadMaterialAnimation(anim, fileData); break;
                        case GuiObjectID.ShaderParameterAnimation: result = G3dHioLibProxy.Hio.LoadShaderParamAnimation(anim, fileData); break;
                        case GuiObjectID.ColorAnimation: result = G3dHioLibProxy.Hio.LoadColorAnimation(anim, fileData); break;
                        case GuiObjectID.TextureSrtAnimation: result = G3dHioLibProxy.Hio.LoadTextureSRTAnimation(anim, fileData); break;
                        case GuiObjectID.BoneVisibilityAnimation: result = G3dHioLibProxy.Hio.LoadBoneVisibilityAnimation(anim, fileData); break;
                        case GuiObjectID.MaterialVisibilityAnimation: result = G3dHioLibProxy.Hio.LoadMaterialVisibilityAnimation(anim, fileData); break;
                        case GuiObjectID.SkeletalAnimation: result = G3dHioLibProxy.Hio.LoadSkeletalAnimation(anim, fileData); break;
                        case GuiObjectID.TexturePatternAnimation: result = G3dHioLibProxy.Hio.LoadTexturePatternAnimation(anim, fileData); break;
                        case GuiObjectID.SceneAnimation: result = G3dHioLibProxy.Hio.LoadSceneAnimation((NintendoWare.G3d.Edit.IEditSceneAnimTarget)anim, fileData); break;
                        case GuiObjectID.ShapeAnimation: G3dHioLibProxy.Hio.LoadShapeAnimation(anim, fileData); break;
                        //DebugConsole.WriteLine("LoadAnim : Loading shape animations is not supported."); break;
                        default:
                            Debug.Assert(false);
                            return false;
                    }
                    anim.HioLoaded = true;

                    // Load されて ResKey が設定されるのを待つ
                    while (!resetEvent.Wait(1000) && Viewer.Manager.Instance.IsConnected)
                    {
                        ;
                    }

                    if (!Connecter.Platform.UseNw && anim is IReferTexture)
                    {
                        // TODO: 必要なら更新があったときだけ送る
                        DebugConsole.WriteLine("HIO.SetTextureBindings --------------- " + anim.Name + " " + textures.Count());
                        G3dHioLibProxy.Hio.SetTextureBindingsForAnimation(anim, textures.OfType<NintendoWare.G3d.Edit.IEditTextureTarget>().ToArray());
                    }
                }
                finally
                {
                    lock (Connecter.WaitingLoadFiles)
                    {
                        Connecter.WaitingLoadFiles.Remove(Key);
                    }
                    Viewer.Manager.Instance.RemoveFileLoadedAction(fileLoaded);
                }
            }

            return result;
        }

        // HIO専用メッセージブロック
        public class ReloadModelSuppressBlock : IDisposable
        {
            private static int depth_ = 0;

            public static bool IsEnabled
            {
                get
                {
                    return depth_ > 0;
                }
            }

            public ReloadModelSuppressBlock()
            {
                ++ depth_;
            }

            public void Dispose()
            {
                -- depth_;
            }
        };

        public static bool ReloadModel(List<Tuple<IntermediateFileDocument, nw4f_3difType, List<G3dStream>, string>> modelDataList, out bool unloaded, LoadProgressDialog.DialogBlock block, string modelName)
        {
            unloaded = false;
            Model model = (Model)modelDataList[0].Item1;

            if (ReloadModelSuppressBlock.IsEnabled == false)
            {
                var errMsgs = new List<string>();
                uint alignmentSize;
                var fileName = MakeBinaryResource(modelDataList, errMsgs, true, Connecter.Platform, out alignmentSize);
                if (fileName == null)
                {
                    // エラーログ出力
                    ErrorLog.WriteBinaryErrorLog(errMsgs);

                    // バイナリ化失敗エラーメッセージ
                    BinaryConverterManager.ShowBinalizeErrorMsg(errMsgs);

                    if (model.IsAttached)
                    {
                        if (model.PreviewingAnimations.Any())
                        {
                            G3dHioLibProxy.Hio.UnbindAnimations(model, model.PreviewingAnimations.Select(x => x.Item2).OfType<NintendoWare.G3d.Edit.IEditTarget>().ToArray());
                        }
                        UnloadModelInternal(model);

                        //model.ResetStatus();
                        model.UnloadedFromHio();

                        UnloadUnusedTextures(model);

                        unloaded = true;
                    }
                }
                else
                {
                    var textures = new List<Texture>();
                    if (!Connecter.Platform.UseNw)
                    {
                        // テクスチャを送る
                        foreach (var item in modelDataList)
                        {
                            if (item.Item1.ObjectID == GuiObjectID.Texture)
                            {
                                LoadOrReloadOrSkipTexture((Texture)item.Item1, item.Item2, item.Item3, item.Item4, block);
                                HioUtility.AddTextureBinding((Texture)item.Item1, model);
                                textures.Add((Texture)item.Item1);
                            }
                        }
                    }

                    // この処理はいらない。
                    // http://www-sdd.zelda.nintendo.co.jp/project/nintendoware3/kagemai/html/user.cgi?project=nw3_3de&action=view_report&id=844#6
                    //				LoadCounter.BeginLoad();

                    var binName = fileName;

                    DebugConsole.WriteLine("G3dHioLibProxy.Hio.ReloadModel()----------------------------------------------------------");
                    var tmp = block.Message;
                    block.Message = string.Format(Strings.Viewer_LoadingModel, modelName);
                    TransactionMessage.BeginTransaction(true);
                    var materialCount = ((modelType)modelDataList[0].Item2.Item).material_array.GetItems().Count();
                    G3dHioLibProxy.Hio.ReloadModel(
                        model,
                        new NintendoWare.G3d.Edit.FileData() { FileName = binName, Alignment = alignmentSize });

                    TransactionMessage.EndTransaction();
                    block.Message = tmp;

                    if (!Connecter.Platform.UseNw)
                    {
                        // TODO: 必要なら更新があったときだけ送る
                        DebugConsole.WriteLine("HIO.SetTextureBindings --------------- " + model.Name + " " + textures.Count());
                        G3dHioLibProxy.Hio.SetTextureBindingsForModel(model, textures.OfType<NintendoWare.G3d.Edit.IEditTextureTarget>().ToArray());
                    }

                    UnloadUnusedTextures(model, textures);

                    return true;
                }
            }

            return false;
        }

        public static void SendRenderInfoAfterReloadModel(modelType modelData, Material[] materials, bool[] queryRenderInfo, HashSet<string>[] defaultEmpty)
        {
            bool inTransaction = false;
            // RenderInfo のテーブルを更新
            for (int i = 0; i < materials.Length; i++)
            {
                var material = materials[i];
                var materialData = modelData.material_array.GetItems().FirstOrDefault(x => x.name == material.Name);
                Dictionary<string, string[]> strings;
                Dictionary<string, int[]> ints;
                Dictionary<string, float[]> floats;
                CreateRenderInfoTable(materialData, out strings, out ints, out floats);
                SendRenderInfoDiff(material, strings, ints, floats, defaultEmpty[i], null, queryRenderInfo[i], ref inTransaction);
            }

            // キューを送信
            QueryRenderInfoQueue.Query();

            if (inTransaction)
            {
                TransactionMessage.EndTransaction();
            }
        }

        public static void SendRenderInfoDiff(Material material, Dictionary<string, string[]> strings, Dictionary<string, int[]> ints, Dictionary<string, float[]> floats, HashSet<string> defaultEmpty, RenderInfo[] renderInfosForUpdate, bool query, ref bool inTransaction)
        {
            bool changedALot = false;
            List<string> stringDiff = new List<string>();
            foreach (var item in material.LastRenderInfo.StringRenderInfoHio)
            {
                if (!strings.ContainsKey(item.Key))
                {
                    stringDiff.Add(item.Key);
                }
            }

            foreach (var item in strings)
            {
                if (!material.LastRenderInfo.StringRenderInfoHio.ContainsKey(item.Key))
                {
                    stringDiff.Add(item.Key);
                }
                else
                {
                    var values = material.LastRenderInfo.StringRenderInfoHio[item.Key];
                    if (!values.SequenceEqual(item.Value))
                    {
                        stringDiff.Add(item.Key);
                    }
                }
            }

            changedALot = stringDiff.Any();

            List<Tuple<string, int, int>> intDiff = new List<Tuple<string, int, int>>();
            foreach (var item in material.LastRenderInfo.IntRenderInfoHio)
            {
                if (!ints.ContainsKey(item.Key))
                {
                    for (int i=0; i < item.Value.Length; i++)
                    {
                        intDiff.Add(new Tuple<string, int, int>(item.Key, i, item.Value[i]));
                        changedALot = true;
                    }
                }
            }

            foreach (var item in ints)
            {
                if (!material.LastRenderInfo.IntRenderInfoHio.ContainsKey(item.Key))
                {
                    for (int i = 0; i < item.Value.Length; i++)
                    {
                        intDiff.Add(new Tuple<string, int, int>(item.Key, i, item.Value[i]));
                        changedALot = true;
                    }
                }
                else
                {
                    var values = material.LastRenderInfo.IntRenderInfoHio[item.Key];
                    int i;
                    for (i = 0; i < Math.Min(item.Value.Length, values.Length); i++)
                    {
                        if (item.Value[i] != values[i])
                        {
                            intDiff.Add(new Tuple<string, int, int>(item.Key, i, item.Value[i]));
                        }
                    }

                    for (; i < values.Length; i++)
                    {
                        intDiff.Add(new Tuple<string, int, int>(item.Key, i, values[i]));
                        changedALot = true;
                    }

                    for (; i < item.Value.Length; i++)
                    {
                        intDiff.Add(new Tuple<string, int, int>(item.Key, i, item.Value[i]));
                        changedALot = true;
                    }
                }
            }

            List<Tuple<string, int, float>> floatDiff = new List<Tuple<string, int, float>>();
            foreach (var item in material.LastRenderInfo.FloatRenderInfoHio)
            {
                if (!floats.ContainsKey(item.Key))
                {
                    for (int i = 0; i < item.Value.Length; i++)
                    {
                        floatDiff.Add(new Tuple<string, int, float>(item.Key, i, item.Value[i]));
                        changedALot = true;
                    }
                }
            }

            foreach (var item in floats)
            {
                if (!material.LastRenderInfo.FloatRenderInfoHio.ContainsKey(item.Key))
                {
                    for (int i = 0; i < item.Value.Length; i++)
                    {
                        floatDiff.Add(new Tuple<string, int, float>(item.Key, i, item.Value[i]));
                        changedALot = true;
                    }
                }
                else
                {
                    var values = material.LastRenderInfo.FloatRenderInfoHio[item.Key];
                    int i;
                    for (i = 0; i < Math.Min(item.Value.Length, values.Length); i++)
                    {
                        if (item.Value[i] != values[i])
                        {
                            floatDiff.Add(new Tuple<string, int, float>(item.Key, i, item.Value[i]));
                        }
                    }

                    for (; i < values.Length; i++)
                    {
                        floatDiff.Add(new Tuple<string, int, float>(item.Key, i, values[i]));
                        changedALot = true;
                    }

                    for (; i < item.Value.Length; i++)
                    {
                        floatDiff.Add(new Tuple<string, int, float>(item.Key, i, item.Value[i]));
                        changedALot = true;
                    }
                }
            }

            material.LastRenderInfo.StringRenderInfoHio = strings;
            material.LastRenderInfo.IntRenderInfoHio = ints;
            material.LastRenderInfo.FloatRenderInfoHio = floats;

            bool defaultChanged = material.LastRenderInfo.DefaultEmptyRenderInfoHio.Count != defaultEmpty.Count || material.LastRenderInfo.DefaultEmptyRenderInfoHio.Any(x => !defaultEmpty.Contains(x));
            material.LastRenderInfo.DefaultEmptyRenderInfoHio = defaultEmpty;

            if (stringDiff.Any() || intDiff.Any() || floatDiff.Any())
            {
                if (!inTransaction)
                {
                    TransactionMessage.BeginTransaction(true);
                    inTransaction = true;
                }

                var objects = new ArrayList();
                objects.Add(material);
                Viewer.Select.SelectInternal(objects, GuiObjectID.Material);

                if (renderInfosForUpdate != null && changedALot)
                {
                    var renderInfoArray = CreateIRenderInfoArrayFromTable(renderInfosForUpdate, material.LastRenderInfo.StringRenderInfoHio, material.LastRenderInfo.IntRenderInfoHio, material.LastRenderInfo.FloatRenderInfoHio);
                    DebugUpdateRenderInfo(renderInfoArray);
                    G3dHioLibProxy.Hio.UpdateRenderInfo(renderInfoArray);
                }

                foreach (var diff in stringDiff)
                {
                    DebugConsole.WriteLine("Hio.EditStringRenderInfo: " + diff);
                    G3dHioLibProxy.Hio.EditStringRenderInfo(diff);
                }

                foreach (var diff in intDiff)
                {
                    DebugConsole.WriteLine("Hio.EditIntRenderInfo: " + diff.Item1 + ", " + diff.Item2 + ", " + diff.Item3);
                    G3dHioLibProxy.Hio.EditIntRenderInfo(diff.Item1, diff.Item2, diff.Item3);
                }

                foreach (var diff in floatDiff)
                {
                    DebugConsole.WriteLine("Hio.EditFloatRenderInfo: " + diff.Item1 + ", " + diff.Item2 + ", " + diff.Item3);
                    G3dHioLibProxy.Hio.EditFloatRenderInfo(diff.Item1, diff.Item2, diff.Item3);
                }

                if (query)
                {
                    material.RenderInfoQueried = true;
                    DebugConsole.WriteLine("QueryRenderInfoQueue.Push");
                    QueryRenderInfoQueue.Push(material);
                }
            }
            else if (query && defaultChanged)
            {
                material.RenderInfoQueried = true;
                DebugConsole.WriteLine("QueryRenderInfoQueue.Push");
                QueryRenderInfoQueue.Push(material);
            }
        }

        public static void DebugUpdateRenderInfo(NintendoWare.G3d.Edit.IRenderInfo[] renderInfoArray)
        {
#if DEBUG
            var stringBuilder = new StringBuilder();
            if (renderInfoArray == null)
            {
                DebugConsole.WriteLine("renderInfoArray == null");
                return;
            }

            stringBuilder.Append("Hio.UpdateRenderInfo: ");
            for (int i=0;i < renderInfoArray.Length;i++)
            {
                var item = renderInfoArray[i];
                stringBuilder.Append("(" + item.Name + ")");
                switch (item.Kind)
                {
                    case NintendoWare.G3d.Edit.RenderInfoKind.String:
                        {
                            var info = item as NintendoWare.G3d.Edit.StringRenderInfo;
                            var del = "";
                            foreach (var elem in info.Values)
                            {
                                stringBuilder.Append(del + elem.EditedValue);
                                del = ", ";
                            }
                        }
                        break;
                    case NintendoWare.G3d.Edit.RenderInfoKind.Int:
                        {
                            var info = item as NintendoWare.G3d.Edit.IntRenderInfo;
                            var del = "";
                            foreach (var elem in info.Values)
                            {
                                stringBuilder.Append(del + elem.EditedValue);
                                del = ", ";
                            }
                        }
                        break;
                    case NintendoWare.G3d.Edit.RenderInfoKind.Float:
                        {
                            var info = item as NintendoWare.G3d.Edit.FloatRenderInfo;
                            var del = "";
                            foreach (var elem in info.Values)
                            {
                                stringBuilder.Append(del + elem.EditedValue);
                                del = ", ";
                            }
                        }
                        break;
                }
                stringBuilder.Append(": ");
            }

            DebugConsole.WriteLine(stringBuilder.ToString());
#endif
        }

        public static NintendoWare.G3d.Edit.IRenderInfo[] CreateIRenderInfoArrayFromTable(RenderInfo[] renderInfo, Dictionary<string, string[]> stringTable, Dictionary<string, int[]> intTable, Dictionary<string, float[]> floatTable)
        {
            List<NintendoWare.G3d.Edit.IRenderInfo> renderInfos = new List<NintendoWare.G3d.Edit.IRenderInfo>();
            foreach (var info in renderInfo)
            {
                switch (info.type)
                {
                    case render_info_typeType.@string:
                        {
                            var item = new NintendoWare.G3d.Edit.StringRenderInfo(info.name);
                            foreach (var value in stringTable[info.name])
                            {
                                item.AddValue(new NintendoWare.G3d.Edit.StringRenderInfo.Value("", value));
                            }
                            renderInfos.Add(item);
                        }
                        break;
                    case render_info_typeType.@int:
                        {
                            var item = new NintendoWare.G3d.Edit.IntRenderInfo(info.name);
                            foreach (var value in intTable[info.name])
                            {
                                item.AddValue(new NintendoWare.G3d.Edit.IntRenderInfo.Value(0, value));
                            }
                            renderInfos.Add(item);
                        }
                        break;
                    case render_info_typeType.@float:
                        {
                            var item = new NintendoWare.G3d.Edit.FloatRenderInfo(info.name);
                            foreach (var value in floatTable[info.name])
                            {
                                item.AddValue(new NintendoWare.G3d.Edit.FloatRenderInfo.Value(0, value));
                            }
                            renderInfos.Add(item);
                        }
                        break;
                }
            }
            return renderInfos.ToArray();
        }

        public static string CreateModelBinaryFile(List<Tuple<IntermediateFileDocument, nw4f_3difType, List<G3dStream>, string>> modelDataList, bool load, out uint alignmentSize)
        {
            Model model = (Model)modelDataList[0].Item1;


            var errMsgs = new List<string>();
            var fileName = MakeBinaryResource(modelDataList, errMsgs, !load, Connecter.Platform, out alignmentSize);
            if (fileName == null)
            {
                // エラーログ出力
                ErrorLog.WriteBinaryErrorLog(errMsgs);

                // バイナリ化失敗エラーメッセージ
                BinaryConverterManager.ShowBinalizeErrorMsg(errMsgs);

                if (model.IsAttached)
                {
                    UnloadModelInternal(model);

                    //model.ResetStatus();
                    model.UnloadedFromHio();

                    UnloadUnusedTextures(model);
                }
            }

            return fileName;
        }

        public static bool ReloadAnimation(List<Tuple<IntermediateFileDocument, nw4f_3difType, List<G3dStream>, string>> animDataList)
        {
            AnimationDocument anim = (AnimationDocument)(animDataList[0].Item1);
            bool result = false;

            var errMsgs = new List<string>();
            uint alignmentSize;
            var fileName = MakeBinaryResource(animDataList, errMsgs, false, Connecter.Platform, out alignmentSize);
            if (fileName == null)
            {
                // エラーログ出力
                ErrorLog.WriteBinaryErrorLog(errMsgs);

                // バイナリ化失敗エラーメッセージ
                BinaryConverterManager.ShowBinalizeErrorMsg(errMsgs);

                // 自動接続を無効にする
                G3dHioLibProxy.Hio.IsAutoConnection = false;

                // ランタイム連携切断
                TheApp.MainFrame.Invoke(new System.Windows.Forms.MethodInvoker(() =>
                        Viewer.ViewerUtility.Disconnect(Strings.BinalizeError_Disconnect, false, false)));
            }
            else
            {
                var textures = new List<Texture>();
                if (!Connecter.Platform.UseNw)
                {
                    // テクスチャを送る
                    foreach (var item in animDataList)
                    {
                        if (item.Item1.ObjectID == GuiObjectID.Texture)
                        {
                            LoadOrReloadOrSkipTexture((Texture)item.Item1, item.Item2, item.Item3, item.Item4, null);
                            HioUtility.AddTextureBinding((Texture)item.Item1, anim);
                            textures.Add((Texture)item.Item1);
                        }
                    }
                }

                var binName = fileName;
                var fileData = new NintendoWare.G3d.Edit.FileData() { FileName = binName, Alignment = alignmentSize };

                // G3dHioLibProxy.RecvInfo.IsFileReloaded が true になるまで待つ
                var resetEvent = new ManualResetEventSlim();
                Action onFileReloaded = () => resetEvent.Set();
                Viewer.Manager.Instance.AddFileReloadedAction(onFileReloaded);
                var oldKey = anim.ResFileKey;
                try
                {
                    lock (Connecter.WaitingReloads)
                    {
                        Connecter.WaitingReloads.Add(oldKey, anim);
                    }
                    switch (anim.ObjectID)
                    {
                        case GuiObjectID.MaterialAnimation: result = G3dHioLibProxy.Hio.ReloadMaterialAnimation(anim, fileData); break;
                        case GuiObjectID.ShaderParameterAnimation: result = G3dHioLibProxy.Hio.ReloadShaderParamAnimation(anim, fileData); break;
                        case GuiObjectID.ColorAnimation: result = G3dHioLibProxy.Hio.ReloadColorAnimation(anim, fileData); break;
                        case GuiObjectID.TextureSrtAnimation: result = G3dHioLibProxy.Hio.ReloadTextureSRTAnimation(anim, fileData); break;
                        case GuiObjectID.BoneVisibilityAnimation: result = G3dHioLibProxy.Hio.ReloadBoneVisibilityAnimation(anim, fileData); break;
                        case GuiObjectID.MaterialVisibilityAnimation: result = G3dHioLibProxy.Hio.ReloadMaterialVisibilityAnimation(anim, fileData); break;
                        case GuiObjectID.SkeletalAnimation: result = G3dHioLibProxy.Hio.ReloadSkeletalAnimation(anim, fileData); break;
                        case GuiObjectID.TexturePatternAnimation: result = G3dHioLibProxy.Hio.ReloadTexturePatternAnimation(anim, fileData); break;
                        case GuiObjectID.SceneAnimation: result = G3dHioLibProxy.Hio.ReloadSceneAnimation((NintendoWare.G3d.Edit.IEditSceneAnimTarget)anim, fileData); break;
                        case GuiObjectID.ShapeAnimation: result = G3dHioLibProxy.Hio.ReloadShapeAnimation(anim, fileData); break;
                        default:
                            Debug.Assert(false);
                            break;
                    }
                    if (result)
                    {
                        while (!resetEvent.Wait(1000) && Viewer.Manager.Instance.IsConnected)
                        {
                            ;
                        }
                    }
                }
                finally
                {
                    Viewer.Manager.Instance.RemoveFileReloadedAction(onFileReloaded);
                    lock (Connecter.WaitingReloads)
                    {
                        Connecter.WaitingReloads.Remove(oldKey);
                    }
                }

                DebugConsole.WriteLine("G3dHioLibProxy.Hio.ReloadAnimation();");

                if (!Connecter.Platform.UseNw && anim is IReferTexture)
                {
                    // TODO: 必要なら更新があったときだけ送る
                    DebugConsole.WriteLine("HIO.SetTextureBindings --------------- " + anim.Name + " " + textures.Count());
                    G3dHioLibProxy.Hio.SetTextureBindingsForAnimation(anim, textures.OfType<NintendoWare.G3d.Edit.IEditTextureTarget>().ToArray());
                }

                UnloadUnusedTextures(anim, textures);
            }

            return result;
        }

        public static void UnloadShaderArchive(ShaderDefinition doc, bool updateUI)
        {
            DebugConsole.WriteLine("HioUtility.UnloadShaderArchive()----------------------------------------------------------");

            if (!Viewer.Manager.Instance.IsConnected)
            {
                return;
            }

            // アタッチされているもの場合のみ送る。
            if (doc.IsAttached && doc.ShaderArchiveKey != 0)
            {
                // HIOに送る
                G3dHioLibProxy.Hio.UnloadShaderArchive(doc);

                if (updateUI)
                {
                    // UI 更新用に変更通知
                    TheApp.MainFrame.BeginInvokeOrExecute(
                        () => App.AppContext.NotifyPropertyChanged(null, new DocumentContentsChangedArgs(doc, null)));
                }
            }
        }

        public static void EditShadingModels(ShaderDefinition doc, int[] shadingModelIndices)
        {
            if (!Viewer.Manager.Instance.IsConnected)
            {
                return;
            }

            if (doc.IsAttached && doc.ShaderArchiveKey != 0 && !doc.FailedToBinarize)
            {
                DebugConsole.WriteLine("HioUtility.EditShadingModels()-------------------- targetShaderArchive = {0} ---", doc.FileName);
                foreach (var index in shadingModelIndices)
                {
                    DebugConsole.WriteLine(" shadingModelIndices = {0}", index);
                }

                // HIOに送る
                G3dHioLibProxy.Hio.EditShadingModels(doc, shadingModelIndices);
            }
        }

        private static bool IsHioSupportedAnimation(GuiObjectID guiObjectId)
        {
            switch (guiObjectId)
            {
                case GuiObjectID.SkeletalAnimation:
                case GuiObjectID.MaterialAnimation:
                case GuiObjectID.ShaderParameterAnimation:
                case GuiObjectID.ColorAnimation:
                case GuiObjectID.TextureSrtAnimation:
                case GuiObjectID.MaterialVisibilityAnimation:
                case GuiObjectID.BoneVisibilityAnimation:
                case GuiObjectID.TexturePatternAnimation:
                case GuiObjectID.SceneAnimation:
                case GuiObjectID.CameraAnimation:
                case GuiObjectID.LightAnimation:
                case GuiObjectID.FogAnimation:
                case GuiObjectID.ShapeAnimation:
                    return true;
                default:
                    // テクスチャでもここにくるので Assert は一旦無効にする
                    // Debug.Assert(false);
                    return false;
            }
        }

#region アニメーション・カーブ関係
        private static void BindAnimations(Model target, IEnumerable<AnimationDocument> animations)
        {
            // 現時点でHIOランタイム連携で対応しているアニメーションのみにする
            animations = animations.Where(x => IsHioSupportedAnimation(x.ObjectID));

            Debug.Assert(inViewerDrawSuppressBlock_ == 0);
            Viewer.BindAnimations.Send(target, animations.ToArray());
        }

        public static void BindSceneAnimations(IEnumerable<AnimationDocument> animations)
        {
            // 現時点でHIOランタイム連携で対応しているアニメーションのみにする
            animations = animations.Where(x => IsHioSupportedAnimation(x.ObjectID));

            // SceneAnimation.IsVisible でプレビュー対象のものだけを送信する。
            Viewer.BindSceneAnimations.Send(animations.OfType<SceneAnimation>().ToArray());	// ToArray()で実体化
        }

        public static void FlushCurves(GuiObjectGroup targets, GuiObjectID guiObjectId)
        {
            // TODO: ?

            if ((guiObjectId == GuiObjectID.ShaderParameterAnimation) ||
                (guiObjectId == GuiObjectID.ColorAnimation			) ||
                (guiObjectId == GuiObjectID.TextureSrtAnimation		))
            {
                G3dHioLibProxy.Hio.FlushShaderParamCurves();
            }
            else
            {
                DebugConsole.WriteLine("HioUtility.FlushCurves(); 未対応---------------------------------------------");
            }
        }
#endregion

#region HIO に対応するメソッド
    #region リソース読み込み

        public static void UnloadResource(Document target)
        {
            // ロードされていなければスキップ
            if (!target.HioLoaded)
            {
                return;
            }

            if (target is AnimationDocument)
            {
                if (target is SceneAnimation)
                {
                    UnbindSceneAnimationOnUnload((SceneAnimation)target);
                }
                else
                {
                    UnbindModelAnimationOnUnload((AnimationDocument)target);
                }
            }
            switch(target.ObjectID)
            {
                case GuiObjectID.Model:
                    UnloadModelInternal(target as Model);							break;
                case GuiObjectID.Texture:
                    UnloadTextureInternal(target as Texture);                       break;
                // 以下アニメーション
                    // TODO:
                case GuiObjectID.MaterialAnimation:
                    G3dHioLibProxy.Hio.UnloadMaterialAnimation(target);             break;
                case GuiObjectID.ShaderParameterAnimation:
                    G3dHioLibProxy.Hio.UnloadShaderParamAnimation(target);			break;
                case GuiObjectID.ColorAnimation:
                    G3dHioLibProxy.Hio.UnloadColorAnimation(target); break;
                case GuiObjectID.TextureSrtAnimation:
                    G3dHioLibProxy.Hio.UnloadTextureSRTAnimation(target);			break;
                case GuiObjectID.SkeletalAnimation:
                    G3dHioLibProxy.Hio.UnloadSkeletalAnimation(target);				break;
                case GuiObjectID.TexturePatternAnimation:
                    G3dHioLibProxy.Hio.UnloadTexturePatternAnimation(target);		break;
                case GuiObjectID.BoneVisibilityAnimation:
                    G3dHioLibProxy.Hio.UnloadBoneVisibilityAnimation(target);		break;
                case GuiObjectID.MaterialVisibilityAnimation:
                    G3dHioLibProxy.Hio.UnloadMaterialVisibilityAnimation(target);	break;
                case GuiObjectID.ShapeAnimation:
                    G3dHioLibProxy.Hio.UnloadShapeAnimation(target); break;
                case GuiObjectID.SceneAnimation:
                    {
                        DebugConsole.WriteLine("Unload scene animation");
                        G3dHioLibProxy.Hio.UnloadSceneAnimation((NintendoWare.G3d.Edit.IEditSceneAnimTarget)target);
                        break;
                    }
            }
        }

        public static void UnbindModelAnimationOnUnload(AnimationDocument animation)
        {
            bool unbined = false;
            using (G3dHioLibProxy.Hio.BeginCommand(false))
            {
                lock (DocumentManager.PreviewingModels)
                {
                    foreach (var model in DocumentManager.PreviewingModels)
                    {
                        var target = model.PreviewingAnimations.FirstOrDefault(x => x.Item2 == animation);
                        if (target != null)
                        {
                            DebugConsole.Write("UnbindModelAnimationOnUnload: {0} ", target.Item1);
                            var pausingFrame = new Dictionary<AnimationDocument, float>();
                            var animations = model.PreviewingAnimations.Where(x => x != target).ToArray();
                            foreach (var anim in animations.Select(x => x.Item2))
                            {
                                double frame;
                                if (anim.PauseFrames.TryGetValue(new KeyValuePair<object, string>(model.ModelId, model.PreviewAnimSet.Name), out frame))
                                {
                                    pausingFrame[anim] = (float)frame;
                                }
                            }
                            UpdateAnimationBinds(model, animations, pausingFrame, false);
                            unbined = true;
                        }

                    }
                }

                if (unbined)
                {
                    // フレームを設定する
                    if (AnimationMessage.IsPlaying)
                    {
                        G3dHioLibProxy.Hio.PlayFrameCtrl(AnimationMessage.CurrentFrame);
                    }
                    else
                    {
                        G3dHioLibProxy.Hio.StopFrameCtrl(AnimationMessage.CurrentFrame);
                    }
                }
            }
        }

        public static void UnbindSceneAnimationOnUnload(SceneAnimation sceneAnimation)
        {
            var target = DocumentManager.PreviewingSceneAnimations.FirstOrDefault(x => x.Item2 == sceneAnimation);
            if (target != null)
            {
                DebugConsole.Write("UnbindModelAnimationOnUnload: {0} ", target.Item1);
                G3dHioLibProxy.Hio.UnbindSceneAnimations(DocumentManager.PreviewingSceneAnimations.Select(x => x.Item2).ToArray());
                DocumentManager.PreviewingSceneAnimations.Remove(target);
                G3dHioLibProxy.Hio.BindSceneAnimations(DocumentManager.PreviewingSceneAnimations.Select(x => x.Item2).ToArray());
            }
        }

        public static void UpdateAnimationBinds(
            Model model,
            Tuple<string, AnimationDocument>[] animations,
            Dictionary<AnimationDocument, float> pausingFrame,
            bool sendFrame)
        {
            var unbindAnimations = model.PreviewingAnimations.Where(x => x.Item2.HioLoaded).ToArray();

            if (!model.HioLoaded)
            {
                return;
            }

            if (unbindAnimations.SequenceEqual(animations))
            {
                DebugConsole.WriteLine("BindAnimations Skipped c");
                return;
            }

            using (G3dHioLibProxy.Hio.BeginCommand(false))
            {
                G3dHioLibProxy.Hio.UnbindAnimations(model, unbindAnimations.Select(x => x.Item2).OfType<NintendoWare.G3d.Edit.IEditTarget>().ToArray());

                // 順番があるので全てバインドしなおす
                G3dHioLibProxy.Hio.BindAnimations(model, animations.Select(x => x.Item2).OfType<NintendoWare.G3d.Edit.IEditTarget>().ToArray());

                // バインドしたアニメーションをモデルに保存する。
                model.PreviewingAnimations.Clear();
                model.PreviewingAnimations.AddRange(animations);

                // バインドしなおしたので一時停止情報を送る
                foreach (var item in animations.Select(x => x.Item2))
                {
                    float frame;
                    if (pausingFrame.TryGetValue(item, out frame) && item.IsAttached)
                    {
                        G3dHioLibProxy.Hio.EditAnimationPause(model, item, frame, true);
                    }
                }

                if (sendFrame)
                {
                    // フレームを設定する
                    if (AnimationMessage.IsPlaying)
                    {
                        G3dHioLibProxy.Hio.PlayFrameCtrl(AnimationMessage.CurrentFrame);
                    }
                    else
                    {
                        G3dHioLibProxy.Hio.StopFrameCtrl(AnimationMessage.CurrentFrame);
                    }
                }
            }
        }
    #endregion

    #region ボーン
        public static void EditBoneVisibility(bool isVisible)
        {
            G3dHioLibProxy.Hio.EditBoneVisibility(isVisible);
        }

        public static void EditBoneBillboard(NintendoWare.G3d.Edit.BillboardKind billboardKind)
        {
            G3dHioLibProxy.Hio.EditBoneBillboard(billboardKind);
        }
    #endregion


    #region マテリアル - レンダーステート
        public static void EditMaterialDisplayFace(render_state_display_faceType kind)
        {
            Debug.Assert(materialDisplayFaceKindDict_.ContainsKey(kind));

            G3dHioLibProxy.Hio.EditMaterialDisplayFace(materialDisplayFaceKindDict_[kind]);
        }

        public static void EditMaterialRenderStateMode(render_state_modeType mode)
        {
            Debug.Assert(renderStateModeDict_.ContainsKey(mode));

            G3dHioLibProxy.Hio.EditMaterialRenderStateMode(renderStateModeDict_[mode]);
        }

        public static void EditMaterialBlendMode(render_state_blend_modeType mode)
        {
            Debug.Assert(blendModeDict_.ContainsKey(mode));

            G3dHioLibProxy.Hio.EditMaterialBlendMode(blendModeDict_[mode]);
        }

        public static void EditMaterialLogicOp(logical_blend_opType kind)
        {
            Debug.Assert(logicOpKindDict_.ContainsKey(kind));

            G3dHioLibProxy.Hio.EditMaterialLogicOp(logicOpKindDict_[kind]);
        }

        public static void EditMaterialVisibility(bool isVisible)
        {
            G3dHioLibProxy.Hio.EditMaterialVisibility(isVisible);
        }

        // アルファテスト
        public static void EditMaterialAlphaTestEnable(bool enable)
        {
            G3dHioLibProxy.Hio.EditMaterialAlphaTestEnable(enable);
        }

        public static void EditMaterialAlphaTestFunc(alpha_test_funcType kind)
        {
            Debug.Assert(alphaTestFuncKindDict_.ContainsKey(kind));

            G3dHioLibProxy.Hio.EditMaterialAlphaTestFunc(alphaTestFuncKindDict_[kind]);
        }

        public static void EditMaterialAlphaTestValue(float value)
        {
            G3dHioLibProxy.Hio.EditMaterialAlphaTestValue(value);
        }

        // デプステスト
        public static void EditMaterialDepthTestEnable(bool enable)
        {
            G3dHioLibProxy.Hio.EditMaterialDepthTestEnable(enable);
        }

        public static void EditMaterialDepthTestWriteEnable(bool enable)
        {
            G3dHioLibProxy.Hio.EditMaterialDepthTestWriteEnable(enable);
        }

        public static void EditMaterialDepthTestFunc(depth_test_funcType kind)
        {
            Debug.Assert(depthTestFuncKindDict_.ContainsKey(kind));

            G3dHioLibProxy.Hio.EditMaterialDepthTestFunc(depthTestFuncKindDict_[kind]);
        }

        // カラーブレンド
        public static void EditMaterialColorCombine(color_blend_opType kind)
        {
            Debug.Assert(blendCombineKindDict_.ContainsKey(kind));

            G3dHioLibProxy.Hio.EditMaterialColorCombine(blendCombineKindDict_[kind]);
        }

        public static void EditMaterialAlphaCombine(color_blend_opType kind)
        {
            Debug.Assert(blendCombineKindDict_.ContainsKey(kind));

            G3dHioLibProxy.Hio.EditMaterialAlphaCombine(blendCombineKindDict_[kind]);
        }

        public static void EditMaterialColorSrcBlend(color_blend_rgb_funcType kind)
        {
            Debug.Assert(colorBlendFuncKindDict_.ContainsKey(kind));

            G3dHioLibProxy.Hio.EditMaterialColorSrcBlend(colorBlendFuncKindDict_[kind]);
        }

        public static void EditMaterialColorDstBlend(color_blend_rgb_funcType kind)
        {
            Debug.Assert(colorBlendFuncKindDict_.ContainsKey(kind));

            G3dHioLibProxy.Hio.EditMaterialColorDstBlend(colorBlendFuncKindDict_[kind]);
        }

        public static void EditMaterialAlphaSrcBlend(color_blend_alpha_funcType kind)
        {
            Debug.Assert(alphaBlendFuncKindDict_.ContainsKey(kind));

            G3dHioLibProxy.Hio.EditMaterialAlphaSrcBlend(alphaBlendFuncKindDict_[kind]);
        }

        public static void EditMaterialAlphaDstBlend(color_blend_alpha_funcType kind)
        {
            Debug.Assert(alphaBlendFuncKindDict_.ContainsKey(kind));

            G3dHioLibProxy.Hio.EditMaterialAlphaDstBlend(alphaBlendFuncKindDict_[kind]);
        }

        public static void EditMaterialConstantColor(float r, float g, float b, float a, uint elementBits)
        {
            G3dHioLibProxy.Hio.EditMaterialConstantColor(
                r, g, b, a,
                G3dHioLibProxy.ColorChannelFlag(elementBits)
            );
        }
    #endregion
    #region マテリアル - サンプラ
        public static void EditMaterialSamplerWrapU(int samplerIndex, wrap_uvwType kind)
        {
            Debug.Assert(textureClampKindDict_.ContainsKey(kind));
            G3dHioLibProxy.Hio.EditMaterialSamplerWrapU(samplerIndex, textureClampKindDict_[kind]);
        }

        public static void EditMaterialSamplerWrapV(int samplerIndex, wrap_uvwType kind)
        {
            Debug.Assert(textureClampKindDict_.ContainsKey(kind));
            G3dHioLibProxy.Hio.EditMaterialSamplerWrapV(samplerIndex, textureClampKindDict_[kind]);
        }

        public static void EditMaterialSamplerWrapW(int samplerIndex, wrap_uvwType kind)
        {
            Debug.Assert(textureClampKindDict_.ContainsKey(kind));
            G3dHioLibProxy.Hio.EditMaterialSamplerWrapW(samplerIndex, textureClampKindDict_[kind]);
        }

        public static void EditMaterialSamplerMinLOD(int samplerIndex, float minLOD)
        {
            G3dHioLibProxy.Hio.EditMaterialSamplerMinLOD(samplerIndex, minLOD);
        }

        public static void EditMaterialSamplerMaxLOD(int samplerIndex, float maxLOD)
        {
            G3dHioLibProxy.Hio.EditMaterialSamplerMaxLOD(samplerIndex, maxLOD);
        }

        public static void EditMaterialSamplerLODBias(int samplerIndex, float bias)
        {
            G3dHioLibProxy.Hio.EditMaterialSamplerLODBias(samplerIndex, bias);
        }

        public static void EditmaterialSamplerFilter(
            int samplerIndex,
            filter_mag_minType minFilter,
            filter_mag_minType magFilter,
            filter_mipType mipmaptype,
            filter_max_anisoType maxAnisotropy)
        {
            G3dHioLibProxy.Hio.EditMaterialSamplerFilter(
                samplerIndex,
                texturFilterKindDict_[minFilter],
                texturFilterKindDict_[magFilter],
                textureMipFilterKindDict_[mipmaptype],
                textureAnisoRationKindToInt_[maxAnisotropy]);
        }
    #endregion
    #region マテリアル - シェーダー割り当て関連
        public static void EditMaterialShaderParameter_VectorOrScalar(string parameterName, float[] value, uint elementBits)
        {
            Debug.Assert(value != null);
            Debug.Assert((value.Length >= 1) && (value.Length <= 4));

            if (value.Length == 1)
            {
                G3dHioLibProxy.Hio.EditMaterialShaderParameter(parameterName, value[0]);
            }
            else
            {
                var vector = G3dHioLibProxy.CreateVector(value);
                var flag   = G3dHioLibProxy.VectorChannelFlag(elementBits);

                switch (value.Length)
                {
                    case 2:
                        G3dHioLibProxy.Hio.EditMaterialShaderParameter(parameterName, (NintendoWare.G3d.Edit.Math.Vector2)vector, flag);
                        break;

                    case 3:
                        G3dHioLibProxy.Hio.EditMaterialShaderParameter(parameterName, (NintendoWare.G3d.Edit.Math.Vector3)vector, flag);
                        break;

                    case 4:
                        G3dHioLibProxy.Hio.EditMaterialShaderParameter(parameterName, (NintendoWare.G3d.Edit.Math.Vector4)vector, flag);
                        break;

                    default:
                        Debug.Assert(false);
                        break;
                }
            }
        }

        public static void EditMaterialShaderParameter_VectorOrScalar(string parameterName, int[] value, uint elementBits)
        {
            Debug.Assert(value != null);
            Debug.Assert((value.Length >= 1) && (value.Length <= 4));

            if (value.Length == 1)
            {
                G3dHioLibProxy.Hio.EditMaterialShaderParameter(parameterName, value[0]);
            }
            else
            {
                var vector = G3dHioLibProxy.CreateVectorI(value);
                var flag   = G3dHioLibProxy.VectorChannelFlag(elementBits);

                switch (value.Length)
                {
                    case 2:
                        G3dHioLibProxy.Hio.EditMaterialShaderParameter(parameterName, (NintendoWare.G3d.Edit.Math.Vector2i)vector, flag);
                        break;

                    case 3:
                        G3dHioLibProxy.Hio.EditMaterialShaderParameter(parameterName, (NintendoWare.G3d.Edit.Math.Vector3i)vector, flag);
                        break;

                    case 4:
                        G3dHioLibProxy.Hio.EditMaterialShaderParameter(parameterName, (NintendoWare.G3d.Edit.Math.Vector4i)vector, flag);
                        break;

                    default:
                        Debug.Assert(false);
                        break;
                }
            }
        }

        public static void EditMaterialShaderParameter_VectorOrScalar(string parameterName, uint[] value, uint elementBits)
        {
            Debug.Assert(value != null);
            Debug.Assert((value.Length >= 1) && (value.Length <= 4));

            if (value.Length == 1)
            {
                G3dHioLibProxy.Hio.EditMaterialShaderParameter(parameterName, value[0]);
            }
            else
            {
                var vector = G3dHioLibProxy.CreateVectorU(value);
                var flag   = G3dHioLibProxy.VectorChannelFlag(elementBits);

                switch (value.Length)
                {
                    case 2:
                        G3dHioLibProxy.Hio.EditMaterialShaderParameter(parameterName, (NintendoWare.G3d.Edit.Math.Vector2u)vector, flag);
                        break;

                    case 3:
                        G3dHioLibProxy.Hio.EditMaterialShaderParameter(parameterName, (NintendoWare.G3d.Edit.Math.Vector3u)vector, flag);
                        break;

                    case 4:
                        G3dHioLibProxy.Hio.EditMaterialShaderParameter(parameterName, (NintendoWare.G3d.Edit.Math.Vector4u)vector, flag);
                        break;

                    default:
                        Debug.Assert(false);
                        break;
                }
            }
        }

        public static void EditMaterialShaderParameter_VectorOrScalar(string parameterName, bool[] value, uint elementBits)
        {
            Debug.Assert(value != null);
            Debug.Assert((value.Length >= 1) && (value.Length <= 4));

            if (value.Length == 1)
            {
                G3dHioLibProxy.Hio.EditMaterialShaderParameter(parameterName, value[0]);
            }
            else
            {
                var vector = G3dHioLibProxy.CreateVectorB(value);
                var flag   = G3dHioLibProxy.VectorChannelFlag(elementBits);

                switch (value.Length)
                {
                    case 2:
                        G3dHioLibProxy.Hio.EditMaterialShaderParameter(parameterName, (NintendoWare.G3d.Edit.Math.Vector2b)vector, flag);
                        break;

                    case 3:
                        G3dHioLibProxy.Hio.EditMaterialShaderParameter(parameterName, (NintendoWare.G3d.Edit.Math.Vector3b)vector, flag);
                        break;

                    case 4:
                        G3dHioLibProxy.Hio.EditMaterialShaderParameter(parameterName, (NintendoWare.G3d.Edit.Math.Vector4b)vector, flag);
                        break;

                    default:
                        Debug.Assert(false);
                        break;
                }
            }
        }

        public static void EditMaterialShaderParameter_Matrix(shader_param_typeType type, string parameterName, float[] value)
        {
            Debug.Assert(value != null);
            Debug.Assert(ShaderTypeUtility.ShaderParamTypeKindFromType(type) == ShaderTypeUtility.ShaderParamTypeKind.Matrix);

            var matrix = G3dHioLibProxy.CreateMatrix(type, value);

            switch(type)
            {
                case shader_param_typeType.float2x2:
                    G3dHioLibProxy.Hio.EditMaterialShaderParameter(parameterName, (NintendoWare.G3d.Edit.Math.Matrix22)matrix);
                    break;

                case shader_param_typeType.float2x3:
                    G3dHioLibProxy.Hio.EditMaterialShaderParameter(parameterName, (NintendoWare.G3d.Edit.Math.Matrix23)matrix);
                    break;

                case shader_param_typeType.float2x4:
                    G3dHioLibProxy.Hio.EditMaterialShaderParameter(parameterName, (NintendoWare.G3d.Edit.Math.Matrix24)matrix);
                    break;

                case shader_param_typeType.float3x2:
                    G3dHioLibProxy.Hio.EditMaterialShaderParameter(parameterName, (NintendoWare.G3d.Edit.Math.Matrix32)matrix);
                    break;

                case shader_param_typeType.float3x3:
                    G3dHioLibProxy.Hio.EditMaterialShaderParameter(parameterName, (NintendoWare.G3d.Edit.Math.Matrix33)matrix);
                    break;

                case shader_param_typeType.float3x4:
                    G3dHioLibProxy.Hio.EditMaterialShaderParameter(parameterName, (NintendoWare.G3d.Edit.Math.Matrix34)matrix);
                    break;

                case shader_param_typeType.float4x2:
                    G3dHioLibProxy.Hio.EditMaterialShaderParameter(parameterName, (NintendoWare.G3d.Edit.Math.Matrix42)matrix);
                    break;

                case shader_param_typeType.float4x3:
                    G3dHioLibProxy.Hio.EditMaterialShaderParameter(parameterName, (NintendoWare.G3d.Edit.Math.Matrix43)matrix);
                    break;

                case shader_param_typeType.float4x4:
                    G3dHioLibProxy.Hio.EditMaterialShaderParameter(parameterName, (NintendoWare.G3d.Edit.Math.Matrix44)matrix);
                    break;

                default:
                    Debug.Assert(false);
                    break;
            }
        }

        public static void EditMaterialShaderParameter_TextureSrt(shader_param_typeType type, string parameterName, float[] value)
        {
            Debug.Assert(value != null);
            Debug.Assert(ShaderTypeUtility.ShaderParamTypeKindFromType(type) == ShaderTypeUtility.ShaderParamTypeKind.TextureSrt);

            var textureSrt = G3dHioLibProxy.CreateTextureSrt(type, value);

            switch(type)
            {
                case shader_param_typeType.srt2d:
                    G3dHioLibProxy.Hio.EditMaterialShaderParameter(parameterName, (NintendoWare.G3d.Edit.Math.TextureSRT2D)textureSrt);
                    break;

                case shader_param_typeType.texsrt:
                    G3dHioLibProxy.Hio.EditMaterialShaderParameter(parameterName, (NintendoWare.G3d.Edit.Math.TextureSRT2D)textureSrt);
                    break;

                case shader_param_typeType.srt3d:
                    G3dHioLibProxy.Hio.EditMaterialShaderParameter(parameterName, (NintendoWare.G3d.Edit.Math.TextureSRT3D)textureSrt);
                    break;

                default:
                    Debug.Assert(false);
                    break;
            }
        }

    #endregion
#endregion

#region 選択オブジェクト関係
        public static void Select(ArrayList targets, GuiObjectID guiObjectId)
        {
            switch (guiObjectId)
            {
                case GuiObjectID.Material:
                    Select_Material(targets);
                    break;
                case GuiObjectID.Bone:
                    Select_Bone(targets);
                    break;
                case GuiObjectID.Shape:
                    Select_Shape(targets);
                    break;
                case GuiObjectID.Model:
                    Select_Model(targets);
                    break;
                default:
                    if (IsHioSupportedAnimation(guiObjectId))
                    {
                        Select_Animation(targets);
                    }
                    break;
            }
        }

        private static void Select_Material(ArrayList targets)
        {
            // 選択されているマテリアルをモデルごとに束ねて作っておく
            var selectedMaterialsDict = new Dictionary<Model, List<int>>();
            {
                foreach (Material material in targets)
                {
                    foreach (var model in material.Referrers)
                    {
                        List<int> indices;
                        if (selectedMaterialsDict.TryGetValue(model, out indices) == false)
                        {
                            selectedMaterialsDict[model] = new List<int>()
                            {
                                material.Index
                            };
                        }
                        else
                        {
                            indices.Add(material.Index);
                        }
                    }
                }
            }

            // HIOにおくる
            {
                G3dHioLibProxy.Hio.ClearSelectedMaterials();

                foreach(var si in selectedMaterialsDict)
                {
                    G3dHioLibProxy.Hio.SelectTargetMaterials(
                        si.Key,
                        si.Value.ToArray()
                    );
                }
            }
        }

        private static void Select_Bone(ArrayList targets)
        {
            // 選択されているボーンをモデルごとに束ねて作っておく
            var selectedBonesDict = new Dictionary<Model, List<int>>();
            {
                foreach(Bone bone in targets)
                {
                    List<int> indices;
                    if (selectedBonesDict.TryGetValue(bone.Owner, out indices) == false)
                    {
                        selectedBonesDict[bone.Owner] = new List<int>()
                        {
                            bone.Index
                        };
                    }
                    else
                    {
                        indices.Add(bone.Index);
                    }
                }
            }

            // HIOにおくる
            {
                G3dHioLibProxy.Hio.ClearSelectedBones();

                foreach(var si in selectedBonesDict)
                {
                    G3dHioLibProxy.Hio.SelectTargetBones(
                        si.Key,
                        si.Value.ToArray()
                    );
                }
            }
        }

        private static void Select_Shape(ArrayList targets)
        {
            // 選択されているボーンをモデルごとに束ねて作っておく
            var selectedShapesDict = new Dictionary<Model, List<int>>();
            {
                foreach(Shape shape in targets)
                {
                    List<int> indices;
                    if (selectedShapesDict.TryGetValue(shape.Owner, out indices) == false)
                    {
                        selectedShapesDict[shape.Owner] = new List<int>()
                        {
                            shape.Index
                        };
                    }
                    else
                    {
                        indices.Add(shape.Index);
                    }
                }
            }

            // HIOにおくる
            {
                G3dHioLibProxy.Hio.ClearSelectedShapes();

                foreach(var si in selectedShapesDict)
                {
                    G3dHioLibProxy.Hio.SelectTargetShapes(
                        si.Key,
                        si.Value.ToArray()
                    );
                }
            }
        }

        private static void Select_Model(ArrayList targets)
        {
            // HIOにおくる
            {
                G3dHioLibProxy.Hio.ClearSelectedModels();

                foreach(NintendoWare.G3d.Edit.IEditModelTarget model in targets)
                {
                    G3dHioLibProxy.Hio.SelectTargetModel(model);
                }
            }
        }

        private static void Select_Animation(ArrayList targets)
        {
            foreach(NintendoWare.G3d.Edit.IEditTarget target in targets)
            {
                G3dHioLibProxy.Hio.SelectTargetAnimation(target);
            }
        }
#endregion

        private static void UnloadModelInternal(Model target)
        {
            // バインド元になっているモデルが閉じられたら、HIO.ClearBoneBind() を呼ぶ
            // http://www-sdd.zelda.nintendo.co.jp/project/nintendoware3/kagemai/html/user.cgi?project=nw3_3de&action=view_report&id=1174
            if (target.IsAttached)
            {
                foreach(var model in DocumentManager.Models)
                {
                    if (model.PreviewInfo.BindModelName != null)
                    {
                        if (model.PreviewInfo.BindModelName == target.Name)
                        {
                            DebugConsole.WriteLine("G3dHioLibProxy.Hio.ClearBoneBind ---------------------------------------------");

                            G3dHioLibProxy.Hio.ClearBoneBind(new[]{model});
                        }
                    }
                }
            }

            lock (DocumentManager.PreviewingModels)
            {
                DocumentManager.PreviewingModels.Remove(target);
            }
            DebugConsole.WriteLine("G3dHioLibProxy.Hio.UnloadModel(); ---------------------------------------------");

            if (target.IsAttached)
            {
                G3dHioLibProxy.Hio.UnloadModel(target);
            }
            else
            {
                DebugConsole.WriteLine("UnloadModel skipped");
            }
        }

#region シェーダー最適化
        public static List<Model.ShaderOptimizeData> CreateShaderOptimizeData(Model model, out List<ShaderDefinition> shaderDefinitions, out List<string> shadingModels, bool forceOptimize = false)
        {
            shaderDefinitions = (from material in model.Materials
                                 select material.MaterialShaderAssign.ShaderDefinition).ToList();
            shadingModels = (from material in model.Materials
                             select material.MaterialShaderAssign.ShaderName).ToList();

            if (!G3dHioLibProxy.Hio.IsConnected)
            {
                return new List<Model.ShaderOptimizeData>();
            }

            bool IsDeviceTargeted = Viewer.Manager.IsDeviceTargeted;
            var list = new List<Model.ShaderOptimizeData>();

            // 最適化シェーダー
            {
                var materialGroup = (from material in model.Materials.Where(
                                         x => DocumentManager.OptimizeShader &&
                                             x.OptimizeShader)
                                     let shaderDefinition = material.MaterialShaderAssign.ShaderDefinition
                                     where shaderDefinition != null
                                     let shadingModelIndex = shaderDefinition.Definitions.FindIndex(x => x.Name == material.MaterialShaderAssign.ShaderName)
                                     where !(shaderDefinition.IsAttached && shaderDefinition.ModifiedNotMaterialShadingModelIndices.Contains(shadingModelIndex)) // アタッチされかつ編集されていたら対象外
                                     group material by material.MaterialShaderAssign.ShaderDefinition).ToArray();

                list.AddRange(from materials in materialGroup.OrderBy(x => x.Key.Name)
                        let definition = materials.Key
                        let shader_definition = definition.NoBranchData
                        let shadingModelNames = materials.Select(x => x.MaterialShaderAssign.ShaderName).ToArray()
                        select new Model.ShaderOptimizeData()
                        {
                            name = definition.Name,
                            document = definition,
                            definition = shader_definition,
                            streams = definition.BinaryStreams.ToList(),
                            optimize = true,
                            //shader_variation = default_shader_variation,
                            DeviceTargeted = IsDeviceTargeted,
                            platform = Connecter.Platform,
                            useConverter32 = G3dHioLibProxy.Hio.GetRuntimeAddressType() == NintendoWare.G3d.Edit.PlatformAddressType.Adress32,
                            materials = materials.Select(x => new Model.ShaderOptimizeData.MaterialData(x, forceOptimize, true, definition)).ToArray(),
                        });
            }

            // 最適化しない
            {
                var notOptimized = (from material in model.Materials.Where(x =>
                    !(DocumentManager.OptimizeShader &&
                     x.OptimizeShader))
                            let shaderDefinition = material.MaterialShaderAssign.ShaderDefinition
                            let shadingModelName = material.MaterialShaderAssign.ShaderName
                            where shaderDefinition != null && !shaderDefinition.IsAttached // アタッチされていたら対象外
                            let shadingModel = shaderDefinition.Data.shading_model_array.GetItems().FirstOrDefault(x => x.name == shadingModelName)
                            where shadingModel != null && shadingModel.IsMaterialShader()
                            select new Model.ShaderOptimizeData()
                            {
                                name = shaderDefinition.Name,
                                document = shaderDefinition,
                                materials = new Model.ShaderOptimizeData.MaterialData[] { new Model.ShaderOptimizeData.MaterialData(material, false, false, shaderDefinition) },
                                DeviceTargeted = IsDeviceTargeted,
                                platform = Connecter.Platform,
                                useConverter32 = G3dHioLibProxy.Hio.GetRuntimeAddressType() == NintendoWare.G3d.Edit.PlatformAddressType.Adress32,
                            }).ToArray();

                // 変更されたものやコンバイナーを含んでいれば追加する
                var combinerMaterials = model.Materials.Where(x => x.GetCombiners().Any()).ToDictionary(x => x.Index);
                if (notOptimized.Any(x => x.materials.Any(y => (y.modifiedCount > 0) || combinerMaterials.ContainsKey(y.index))))
                {
                    list.AddRange(notOptimized);
                }

                return list;
            }
        }

        public static bool SameShaderOptimizeData(List<Model.ShaderOptimizeData> lhs, List<Model.ShaderOptimizeData> rhs)
        {
            if (lhs.Count != rhs.Count)
            {
                return false;
            }

            // 暫定対処
            if (lhs.All(x => x.shader_variation != null) && rhs.All(x => x.shader_variation != null))
            {
                return false;
            }

            return lhs.Zip(rhs,
                (x, y) => x.document == y.document &&
                    x.definition == y.definition && // streams の比較は不要
                    (x.shader_variation == y.shader_variation ||
                     (x.shader_variation.Length == y.shader_variation.Length &&
                      x.shader_variation.SequenceEqual(y.shader_variation))) &&
                    x.DeviceTargeted == y.DeviceTargeted &&
                    x.platform == y.platform &&
                    x.useConverter32 == y.useConverter32)
                    .All(x => x);
        }

        public static bool SameShaderOptimizeData(Model.ShaderOptimizeData x, Model.ShaderOptimizeData y)
        {
            return x.document == y.document &&
                    x.definition == y.definition && // streams の比較は不要
                    x.shader_variation.Length == y.shader_variation.Length &&
                    x.shader_variation.SequenceEqual(y.shader_variation) &&
                    x.DeviceTargeted == y.DeviceTargeted &&
                    x.platform == y.platform &&
                    x.useConverter32 == y.useConverter32;
        }

        public static bool ContainsOptimizeData(Model.ShaderOptimizeData x, Model.ShaderOptimizeData y)
        {
            return x != null &&
                x.document == y.document &&
                x.definition == y.definition && // streams の比較は不要
                x.shadingModelSets.Count >= y.shadingModelSets.Count &&
                x.DeviceTargeted == y.DeviceTargeted &&
                x.platform == y.platform &&
                x.useConverter32 == y.useConverter32 &&
                y.shadingModelSets.All(z =>
                    {

                        HashSet<shader_programType> shaderProgramSet;
                        x.shadingModelSets.TryGetValue(z.Key, out shaderProgramSet);
                        return shaderProgramSet != null &&
                            shaderProgramSet.Count >= z.Value.Count &&
                            z.Value.All(w => shaderProgramSet.Contains(w));
                    }
                );
        }

        public static string[] CreateOptimizeBfsdArray(
            List<Model.ShaderOptimizeData> data,
            Viewer.LoadProgressDialog.DialogBlock block,
            Model model,
            int materialCount,
            string dumpShaderSourceDirectory,
            bool ignoreLastErrorOfShaderBinarization)
        {
            string[] bfshas = new string[materialCount];
            var materialOptimizeData = new Tuple<Model.ShaderOptimizeData, string>[materialCount];
            using (var stopWatch = new DebugStopWatch("CreateOptimzieBfsdArray"))
            {
                bool optimized = false;
                var tmp = block.Message;
                foreach (var item in data)
                {
                    if (item.materials.Any(x => x.optimize))
                    {
                        string path;
                        Debug.Assert(item.shader_variation != null);
                        Tuple<Model.ShaderOptimizeData, string> lastItem = null;
                        if (model.lastMaterialOptimizeData != null)
                        {
                            if (Viewer.Manager.IsDeviceTargeted)
                            {
                                lastItem = model.deviceFirstOptimizedData
                                    .Concat(model.lastMaterialOptimizeData)
                                    .Distinct().FirstOrDefault(x => x != null && ContainsOptimizeData(x.Item1, item));
                            }
                            else
                            {
                                lastItem = model.pcFirstOptimizedData
                                    .Concat(model.lastMaterialOptimizeData)
                                    .Distinct().FirstOrDefault(x => x != null && ContainsOptimizeData(x.Item1, item));
                            }
                        }

                        // キャッシュに変更は無くてもログを作成するために再出力する場合がある
                        if (lastItem != null && !model.ForcedOptimized)
                        {
                            // 包含されていたら以前のものを使う
                            path = lastItem.Item2;
                        }
                        else
                        {
                            block.Message = Strings.Viewer_OptimizeShaderConverting;

                            // 最適化
                            string optimizedLogFolder = null;
                            path = GetOptimizedBfsdFromFsv(item, model, dumpShaderSourceDirectory, out optimizedLogFolder, ignoreLastErrorOfShaderBinarization);
                            if (path != null)
                            {
                                optimized = true;

                                if (dumpShaderSourceDirectory != null && optimizedLogFolder != null)
                                {
                                    // キャッシュと最適化ログの関連付け
                                    string folder = dumpShaderSourceDirectory;
                                    folder += "\\";
                                    folder += optimizedLogFolder;
                                    model.AddOptimizedLogFolder(path, folder);

                                    // キャッシュが変更されずに最適化ログだけ作成した場合、直前のキャッシュにもログのパスを関連付ける
                                    if (model.ForcedOptimized && lastItem != null)
                                    {
                                        model.AddOptimizedLogFolder(lastItem.Item2, folder);
                                    }
                                }
                            }
                        }

                        model.ForcedOptimized = false;

                        var tuple = new Tuple<Model.ShaderOptimizeData, string>(item, path);
                        foreach (var material in item.materials.Where(x => x.optimize))
                        {
                            materialOptimizeData[material.index] = tuple;
                            material.bfshaPath = bfshas[material.index] = path;
                        }
                    }

                    foreach (var material in item.materials.Where(x => !x.optimize))
                    {
                        if (material.reusedData != null)
                        {
                            // マテリアルが変更されていなければ再利用
                            Debug.Assert(model.lastMaterialOptimizeData != null);
                            materialOptimizeData[material.index] = material.reusedData;//model.lastMaterialOptimizeData[material.index];
                            material.bfshaPath = bfshas[material.index] = materialOptimizeData[material.index].Item2;
                        }
                        else
                        {
                            // 非最適化時
                            material.bfshaPath = bfshas[material.index] = item.document.GetShaderBinaryPath(material.shadingModelName, material.modifiedCount, !item.DeviceTargeted, ignoreLastErrorOfShaderBinarization, Connecter.Platform, block);
                        }
                    }
                }

                block.Message = tmp;

                lock (Viewer.Connecter.AbortLock)
                {
                    if (model.lastMaterialOptimizeData == null || model.lastMaterialOptimizeData.Length != model.Materials.Count)
                    {
                        model.lastMaterialOptimizeData = new Tuple<Model.ShaderOptimizeData, string>[model.Materials.Count];
                    }

                    if (model.MaterialOptimizeDataCache == null || model.MaterialOptimizeDataCache.Length != model.Materials.Count)
                    {
                        model.MaterialOptimizeDataCache = new Tuple<Model.ShaderOptimizeData, string>[model.Materials.Count];
                    }

                    for (int i = 0; i < model.Materials.Count; i++)
                    {
                        if (model.lastMaterialOptimizeData[i] != materialOptimizeData[i])
                        {
                            model.MaterialOptimizeDataCache[i] = model.lastMaterialOptimizeData[i];
                        }
                    }

                    model.lastMaterialOptimizeData = materialOptimizeData;
                    if (optimized)
                    {
                        if (Viewer.Manager.IsDeviceTargeted)
                        {
                            if (model.deviceFirstOptimizedData.Length == 0)
                            {
                                model.deviceFirstOptimizedData = model.lastMaterialOptimizeData.Where(x => x != null).Distinct().ToArray();
                            }
                        }
                        else
                        {
                            if (model.pcFirstOptimizedData.Length == 0)
                            {
                                model.pcFirstOptimizedData = model.lastMaterialOptimizeData.Where(x => x != null).Distinct().ToArray();
                            }
                        }
                    }
                }

                return bfshas;
            }
        }

        // 最適化シェーダー作成時のコンバイナーシェーダーファイル書き込み日時
        // 親キー：モデル
        // 子キー：コンバイナーシェーダーファイル
        private static Dictionary<Model, Dictionary<string, DateTime>> combinerShaderFileLastWriteTimes = new Dictionary<Model, Dictionary<string, DateTime>>();

        // 最適化シェーダー再作成のためのコンバイナーシェーダーディレクトリウォッチャー
        // キー：モデル
        private static Dictionary<Model, FileSystemWatcher> combinerShaderDirectoryWatchers = new Dictionary<Model, FileSystemWatcher>();

        // 最適化シェーダー再作成のためのコンバイナーシェーダーファイルウォッチャー
        // 親キー：コンバイナーシェーダーディレクトリウォッチャー
        // 子キー：コンバイナーシェーダーファイルパス
        private static Dictionary<FileSystemWatcher, Dictionary<string, FileSystemWatcher>> combinerShaderFileWatchers = new Dictionary<FileSystemWatcher, Dictionary<string, FileSystemWatcher>>();

        private static string GetOptimizedBfsdFromFsv(Model.ShaderOptimizeData data, Model model, string dumpShaderSourceDirectory, out string directoryName, bool ignoreLastErrorOfShaderBinarization)
        {
            directoryName = null;

            lock (data.document)
            {
                if (data.document.FailedToBinarize && !ignoreLastErrorOfShaderBinarization)
                {
                    return null;
                }
                List<Tuple<byte[], string>> inputFiles = new List<Tuple<byte[], string>>();

                string tempDir = TemporaryFileUtility.MakeTempDirectory();
                byte[] fsvImage = data.shader_variation;
                string fsvFilePath = Path.Combine(tempDir, data.name + ".fsva");
                inputFiles.Add(new Tuple<byte[], string>(fsvImage, fsvFilePath));

                // デバッグ用
                DebugConsole.WriteLine(UTF8Encoding.UTF8.GetString(fsvImage, 0, fsvImage.Length));
                //File.WriteAllBytes(fsvFilePath, fsvImage);

                // fsd
                nw4f_3difType fsd = new nw4f_3difType()
                {
                    Item = data.definition,
                };

                // バイナリ中間ファイルの場合、nw4f_3difからストリームを削除する。
                if (G3dStreamUtility.HasStreamArray(fsd))
                {
                    fsd.RootElement.stream_array = null;
                }

                byte[] fileImage;
                lock (Viewer.Connecter.AbortLock)
                {
                    fileImage = IfBinaryFormatter.FormatStream(fsd, data.streams, null);
                }

                int size = fileImage.Count();

                string fsdFilePath = Path.Combine(tempDir, data.name + ".fsdb");
                var ecmbFiles = new List<string>((model != null) ? model.Materials.SelectMany(x => x.GetCombiners().Where(y => !CombinerShaderConverterManager.ReservedValue.IsNullOrEmptyOrUnset(y.Item2.value)).Select(y => y.Item1.CreateCombinerShaderFilePath(y.Item2)).Where(y => !string.IsNullOrEmpty(y))).Distinct() : Enumerable.Empty<string>());
                var ecmbLastWriteTimes = ecmbFiles.ToDictionary(x => x, x => !string.IsNullOrEmpty(x) ? (new System.IO.FileInfo(x)).LastWriteTime : default(DateTime));
                bool combinerFailure = false;
                if (ecmbFiles.Any())
                {
                    // fileImage を ecmb を埋め込んだものに変換する。
                    // 1. fileImage をファイルに保存
                    // 2. 保存したファイルに ecmb ファイルを埋め込む
                    // 3. 埋め込んだファイルイメージを取得する
                    try
                    {
                        File.WriteAllBytes(fsdFilePath, fileImage);
                        if (!CombinerShaderConverterManager.ConvertShader(fsdFilePath, ecmbFiles, true))
                        {
                            throw new Exception();
                        }
                        fileImage = File.ReadAllBytes(fsdFilePath);
                    }
                    catch
                    {
                        combinerFailure = true;
                    }
                }
                inputFiles.Add(new Tuple<byte[], string>(fileImage, fsdFilePath));

                string additionalOption = null;
                if (dumpShaderSourceDirectory != null)
                {
                    dumpShaderSourceDirectory = Environment.ExpandEnvironmentVariables(dumpShaderSourceDirectory);
                    bool ok = false;
                    try
                    {
                        ok = Path.IsPathRooted(dumpShaderSourceDirectory) && Directory.Exists(dumpShaderSourceDirectory);
                    }
                    catch
                    {
                    }

                    if (ok)
                    {
                        var time = System.DateTime.Now.ToString("yyyyMMdd_HHmmss");
                        string outputFolder = null;
                        int index = 0;
                        while (outputFolder == null || Directory.Exists(outputFolder) || File.Exists(outputFolder))
                        {
                            directoryName = string.Format("{0}_{1}_{2}", data.name, time, index);
                            outputFolder = Path.Combine(dumpShaderSourceDirectory, directoryName);
                            index++;
                        }

                        Directory.CreateDirectory(outputFolder);
                        additionalOption = string.Format("--dump-statistics-information=\"{0}\" ", outputFolder);
                    }
                    else
                    {
                        if (dumpShaderSourceDirectory == string.Empty)
                        {
                            TheApp.MainFrame.BeginInvokeOrExecute(
                                () => UIMessageBox.Error(Strings.Viewer_ShaderSourceDirectoryEmpty));
                        }
                        else
                        {
                            TheApp.MainFrame.BeginInvokeOrExecute(
                                () => UIMessageBox.Error(Strings.Viewer_ShaderSourceDirectory, dumpShaderSourceDirectory));
                        }
                    }
                }

                string BfsdPath = Path.Combine(tempDir, data.name + ".bfsha");

                if (combinerFailure || !ShdrcvtrManager.ConvertShaderToBinaryForViewer(
                    inputFiles,
                    BfsdPath,
                    data.DeviceTargeted ? ShdrcvtrManager.TargetType.Cafe: ShdrcvtrManager.TargetType.PC,
                    Connecter.Platform,
                    G3dHioLibProxy.Hio.GetRuntimeAddressType() == NintendoWare.G3d.Edit.PlatformAddressType.Adress32,
                    additionalOption))
                {
                    BfsdPath = null;
                    combinerShaderFileLastWriteTimes.Remove(model);
                    data.document.FailedToBinarize = true;

                    // Unload は行わない
                    //if (data.document.IsAttached)
                    //{
                    //		HioUtility.UnloadShaderArchive(data.document, false);
                    //}

                    // UI 更新用に変更通知
                    TheApp.MainFrame.BeginInvokeOrExecute(
                        () => App.AppContext.NotifyPropertyChanged(null, new DocumentContentsChangedArgs(data.document, null)));
                }
                else if (!combinerFailure)
                {
                    FileSystemWatcher ecmbDirWatcher;
                    if (!combinerShaderDirectoryWatchers.TryGetValue(model, out ecmbDirWatcher))
                    {
                        // ディレクトリウォッチャーを作成。
                        ecmbDirWatcher = new FileSystemWatcher(model.FileLocation, "combiners");
                        ecmbDirWatcher.NotifyFilter = System.IO.NotifyFilters.DirectoryName | NotifyFilters.LastWrite;
                        ecmbDirWatcher.IncludeSubdirectories = false;
                        ecmbDirWatcher.SynchronizingObject = TheApp.MainFrame;
                        Action dirChanged = () =>
                        {
                            Dictionary<string, FileSystemWatcher> fileWatchers;
                            if (combinerShaderFileWatchers.TryGetValue(ecmbDirWatcher, out fileWatchers))
                            {
                                bool updated = false;

                                // コンバイナーシェーダーディレクトリ下の未作成ファイルウォッチャーの作成を試みる。
                                foreach (var item in fileWatchers.Where(x => x.Value == null).ToArray())
                                {
                                    FileSystemWatcher ecmbFileWatcher = null;
                                    try
                                    {
                                        ecmbFileWatcher = new FileSystemWatcher(Path.Combine(ecmbDirWatcher.Path, ecmbDirWatcher.Filter), System.IO.Path.GetFileName(item.Key));
                                        ecmbFileWatcher.NotifyFilter = System.IO.NotifyFilters.FileName;
                                        ecmbFileWatcher.IncludeSubdirectories = false;
                                        ecmbFileWatcher.SynchronizingObject = TheApp.MainFrame;
                                        Action fileChanged = () =>
                                        {
                                            model.ForcedOptimized = true;
                                            Viewer.LoadOrReloadModel.Send(model, true);
                                        };
                                        ecmbFileWatcher.Changed += (ss, ee) => { fileChanged(); };
                                        ecmbFileWatcher.Created += (ss, ee) => { fileChanged(); };
                                        ecmbFileWatcher.Deleted += (ss, ee) => { fileChanged(); };
                                        ecmbFileWatcher.Renamed += (ss, ee) => { fileChanged(); };
                                        ecmbFileWatcher.EnableRaisingEvents = true;
                                    }
                                    catch
                                    {
                                        ecmbFileWatcher?.Dispose();
                                        ecmbFileWatcher = null;
                                    }
                                    fileWatchers[item.Key] = ecmbFileWatcher;

                                    if (ecmbDirWatcher != item.Value)
                                    {
                                        DebugConsole.WriteLine(string.Format("Reload {0}", item.Key));
                                        updated |= true;
                                    }
                                }

                                if (updated)
                                {
                                    model.ForcedOptimized = true;
                                    Viewer.LoadOrReloadModel.Send(model, true);
                                }
                            }
                        };
                        ecmbDirWatcher.Changed += (ss, ee) => { dirChanged(); };
                        ecmbDirWatcher.Created += (ss, ee) => { dirChanged(); };
                        ecmbDirWatcher.Deleted += (ss, ee) => { dirChanged(); };
                        ecmbDirWatcher.Renamed += (ss, ee) => { dirChanged(); };
                        ecmbDirWatcher.EnableRaisingEvents = true;
                        combinerShaderDirectoryWatchers.Add(model, ecmbDirWatcher);
                    }

                    // ファイルウォッチャーを更新。
                    Dictionary<string, FileSystemWatcher> ecmbFileWatchers;
                    if (combinerShaderFileWatchers.TryGetValue(ecmbDirWatcher, out ecmbFileWatchers))
                    {
                        combinerShaderFileWatchers.Remove(ecmbDirWatcher);
                        foreach (var watcher in ecmbFileWatchers.Values)
                        {
                            watcher?.Dispose();
                        }
                        ecmbFileWatchers.Clear();
                    }
                    else
                    {
                        ecmbFileWatchers = new Dictionary<string, FileSystemWatcher>();
                    }
                    foreach (var ecmbFilePath in ecmbFiles)
                    {
                        // コンバイナーシェーダーファイルウォッチャーの作成を試みる。
                        // 存在しないディレクトリのウォッチャーは作成できないので、
                        // ここで作成に失敗したものは、ディレクトリウォッチャー側で再作成を試みる。
                        FileSystemWatcher ecmbFileWatcher = null;
                        try
                        {
                            ecmbFileWatcher = new FileSystemWatcher(Path.Combine(ecmbDirWatcher.Path, ecmbDirWatcher.Filter), System.IO.Path.GetFileName(ecmbFilePath));
                            ecmbFileWatcher.NotifyFilter = System.IO.NotifyFilters.FileName | NotifyFilters.LastWrite;
                            ecmbFileWatcher.IncludeSubdirectories = false;
                            ecmbFileWatcher.SynchronizingObject = TheApp.MainFrame;
                            Action fileChanged = () =>
                            {
                                DebugConsole.WriteLine(string.Format("Reload {0}", ecmbFilePath));
                                model.ForcedOptimized = true;
                                Viewer.LoadOrReloadModel.Send(model, true);
                            };
                            ecmbFileWatcher.Changed += (ss, ee) => { fileChanged(); };
                            ecmbFileWatcher.Created += (ss, ee) => { fileChanged(); };
                            ecmbFileWatcher.Deleted += (ss, ee) => { fileChanged(); };
                            ecmbFileWatcher.Renamed += (ss, ee) => { fileChanged(); };
                            ecmbFileWatcher.EnableRaisingEvents = true;
                        }
                        catch
                        {
                            ecmbFileWatcher?.Dispose();
                            ecmbFileWatcher = null;
                        }
                        ecmbFileWatchers.Add(ecmbFilePath, ecmbFileWatcher);
                    }
                    combinerShaderFileWatchers.Add(ecmbDirWatcher, ecmbFileWatchers);

                    combinerShaderFileLastWriteTimes[model] = ecmbLastWriteTimes;
                }

                // バイナライズ成功時のフラグを設定。
                data.document.SucceededInBinarize();

                return BfsdPath;
            }
        }

        public static bool LoadOptimizedShader(
            Model target,
            List<Model.ShaderOptimizeData> ShaderOptimizeData,
            List<Tuple<IntermediateFileDocument, nw4f_3difType, List<G3dStream>, string>> _modelDataList,
            out bool binarizeError,
            bool checkMaterial,
            List<ShaderDefinition> shaderDefinitions,
            List<string> shadingModels,
            LoadProgressDialog.DialogBlock block,
            string modelName,
            string dumpShaderSourceDirectory,
            bool enableShaderOptimize,
            bool ignoreLastErrorOfShaderBinarization
            )
        {
            binarizeError = false;

            binarizeError = false;

            if (!G3dHioLibProxy.Hio.IsConnected)
            {
                return false;
            }

            // モデルバイナリ生成
            string tmp1 = block.Message;
            block.Message = string.Format(Strings.Viewer_ConvertingModel, modelName);
            uint aligntmentSize;
            string modelBinaryPath = HioUtility.CreateModelBinaryFile(_modelDataList, false, out aligntmentSize);
            block.Message = tmp1;

            if (modelBinaryPath == null)
            {
                binarizeError = true;
                return false;
            }

            // 必要なら fsv を生成
            CreateFsv(
                _modelDataList,
                modelName,
                ShaderOptimizeData.Where(x => x.optimize).ToList(),
                target,
                block);

            int materialCount = ((modelType)_modelDataList[0].Item2.Item).material_array.GetItems().Count();
            string[] bfshaPaths = HioUtility.CreateOptimizeBfsdArray(ShaderOptimizeData, block, target, materialCount, dumpShaderSourceDirectory, ignoreLastErrorOfShaderBinarization);
            int[] materialShaderIndices = new int[materialCount];
            var materialData = ShaderOptimizeData.SelectMany(x => x.materials).ToList();
            materialData.Sort((x, y) => x.index.CompareTo(y.index));

            var bfshaDictionary = new Dictionary<string, int>();
            var bfshaList = new List<string>();
            for (int i = 0; i < materialCount; i++)
            {
                int bfshaIndex;
                if (bfshaPaths[i] == null)
                {
                    bfshaIndex = -1;
                }
                else if (!bfshaDictionary.TryGetValue(bfshaPaths[i], out bfshaIndex))
                {
                    bfshaIndex = bfshaDictionary.Count;
                    bfshaDictionary[bfshaPaths[i]] = bfshaIndex;
                    bfshaList.Add(bfshaPaths[i]);
                }

                materialShaderIndices[i] = bfshaIndex;
            }

            string[] bfshas = bfshaList.ToArray();
            if (bfshas.Length == 0)
            {
                bfshas = null;
            }

            var builder = new StringBuilder();
            builder.AppendFormat("ReloadModel ({0} {1})", target.FileName, modelBinaryPath);
            string sp = "\r\n";
            if (bfshas != null)
            {
                for (int i = 0; i < bfshas.Length; i++)
                {
                    builder.AppendFormat("{0}{1}:{2}", sp, i, bfshas[i]);
                }
            }

            foreach (var index in materialShaderIndices)
            {
                builder.AppendFormat("{0} {1}", sp, index);
                sp = ", ";
            }
            DebugConsole.WriteLine(builder.ToString());

            var textures = new List<Texture>();
            if (!Connecter.Platform.UseNw)
            {
                // テクスチャを送る
                foreach (var item in _modelDataList)
                {
                    if (item.Item1.ObjectID == GuiObjectID.Texture)
                    {
                        LoadOrReloadOrSkipTexture((Texture)item.Item1, item.Item2, item.Item3, item.Item4, block);
                        HioUtility.AddTextureBinding((Texture)item.Item1, target);
                        textures.Add((Texture)item.Item1);
                    }
                }
            }

            var tmp = block.Message;
            block.Message = string.Format(Strings.Viewer_LoadingModel, modelName);
            TransactionMessage.BeginTransaction(true);

            var bfresFileData = new NintendoWare.G3d.Edit.FileData() { FileName = modelBinaryPath, Alignment = aligntmentSize };
            // シェーダーバリエーションが生成された場合、またはシェーダー最適化がON && AllowNoOutputがON && シェーダー最適化対象のマテリアルが有る場合
            if (bfshas != null ||
                    (enableShaderOptimize
                    && App.ConfigData.ApplicationConfig.FileIo.CreateShaderVariationCommand.AllowNoOutput
                    && materialData.Any(x => x.optimizeEnabled)))
            {
                var shaderInfos = new List<NintendoWare.G3d.Edit.MaterialShaderInfo>();
                for (var i = 0; i < materialShaderIndices.Length; ++i)
                {
                    var material = materialData.FirstOrDefault(x => x.index == i);
                    var info = new NintendoWare.G3d.Edit.MaterialShaderInfo();
                    info.ShaderArchiveIndex = materialShaderIndices[i];
                    if (material != null)
                    {
                        info.IsOptimized = material.optimizeEnabled;
                        info.IsOptimizationSkipped = !material.optimizeOnReloadEnabled && !material.forceOptimize;
                    }
                    else
                    {
                        info.IsOptimized = info.IsOptimizationSkipped = false;
                    }

                    shaderInfos.Add(info);
                }

                G3dHioLibProxy.Hio.ReloadModel(
                    target,
                    bfresFileData,
                    bfshas,
                    shaderInfos.ToArray());
            }
            else
            {
                G3dHioLibProxy.Hio.ReloadModel(target, bfresFileData);
            }

            TransactionMessage.EndTransaction();
            block.Message = tmp;

            if (!Connecter.Platform.UseNw)
            {
                // TODO: 必要なら更新があったときだけ送る
                DebugConsole.WriteLine("HIO.SetTextureBindings --------------- " + target.Name + " " + textures.Count());
                G3dHioLibProxy.Hio.SetTextureBindingsForModel(target, textures.OfType<NintendoWare.G3d.Edit.IEditTextureTarget>().ToArray());
            }

            UnloadUnusedTextures(target, textures);

            if (TheApp.MainFrame != null && TheApp.MainFrame.IsHandleCreated)
            {
                TheApp.MainFrame.BeginInvoke((Action)(() => target.AfterModelReloaded(ShaderOptimizeData)));
            }


            return true;
        }

        private static string g3dfilterExePath
        {
            get
            {
                // SIGLO-61624 SIGLO-61638 対応。
                string path = System.IO.Path.GetFullPath(string.Format(
                    @"{0}\..\3dTools\{1}",
                    Environment.GetEnvironmentVariable("NW4F_3DEDITOR_ROOT"),
                    "3dIntermediateFileFilter.exe"));

                return path;
            }
        }

        private static string _createFsvBat;
        public static string CreateFsvBat
        {
            get
            {
                if (_createFsvBat != null)
                {
                    return _createFsvBat;
                }

                if (!App.ConfigData.ApplicationConfig.FileIo.CreateShaderVariationCommand.HasCommand)
                {
                    return null;
                }

                _createFsvBat = TemporaryFileUtility.MakeViewerTempFileName(".bat");
                File.WriteAllText(_createFsvBat, App.ConfigData.ApplicationConfig.FileIo.CreateShaderVariationCommand.PathXml);
                return _createFsvBat;
            }
        }


        // TODO: そのうち廃止
        private static string _scriptFilePath;
        public static string scriptFilePath
        {
            get
            {
                if (_scriptFilePath != null)
                {
                    return _scriptFilePath;
                }

                if (string.IsNullOrEmpty(App.ConfigData.ApplicationConfig.FileIo.BuildFsv.ScriptFilePath))
                {
                    return null;
                }

                _scriptFilePath = Environment.ExpandEnvironmentVariables(App.ConfigData.ApplicationConfig.FileIo.BuildFsv.ScriptFilePath);
                return _scriptFilePath;
            }
        }

        // TODO: そのうち廃止
        private static string _templateFilePath;
        public static string templateFilePath
        {
            get
            {
                if (_templateFilePath != null)
                {
                    return _templateFilePath;
                }

                if (string.IsNullOrEmpty(App.ConfigData.ApplicationConfig.FileIo.BuildFsv.TemplateFilePath))
                {
                    return null;
                }

                _templateFilePath = Environment.ExpandEnvironmentVariables(App.ConfigData.ApplicationConfig.FileIo.BuildFsv.TemplateFilePath);
                return _templateFilePath;
            }
        }

        private static void CreateFsv(
            List<Tuple<IntermediateFileDocument, nw4f_3difType, List<G3dStream>, string>> modelDataList,
            string modelName,
            List<Model.ShaderOptimizeData> shaderOptimizeData,
            Model model,
            LoadProgressDialog.DialogBlock block)
        {
            if (!shaderOptimizeData.Any())
            {
                return;
            }

            bool hasScript = (!string.IsNullOrEmpty(scriptFilePath) && !string.IsNullOrEmpty(templateFilePath)) ||
                !string.IsNullOrEmpty(CreateFsvBat);

            var outputBuilder = new StringBuilder();
            var errorBuilder = new StringBuilder();
            var disposableDirectory = TemporaryFileUtility.MakeDisposableDirectoryName();
            try
            {
                using (var timer = new DebugStopWatch("CreateFsvByScript"))
                {
                    var tempDirectory = disposableDirectory.Path;
                    if (tempDirectory.EndsWith("\\"))
                    {
                        tempDirectory = tempDirectory.TrimEnd("\\".ToArray());
                    }
                    var fmdb = System.IO.Path.Combine(tempDirectory, modelName + ".fmdb");

                    // モデル出力
                    var nw4f_3dif = modelDataList[0].Item2;

                    // マテリアルを除いた分
                    byte[] modelBytes;
                    {
                        var tmp = ((modelType)nw4f_3dif.Item).material_array;
                        ((modelType)nw4f_3dif.Item).material_array = null;
                        modelBytes = ObjectUtility.ToBytes((modelType)nw4f_3dif.Item);
                        ((modelType)nw4f_3dif.Item).material_array = tmp;
                    }

                    // null に書き換える部分の shader_assign を保存する
                    var tmp_shader_assigns = ((modelType)nw4f_3dif.Item).material_array.GetItems().Select(x => x.shader_assign).ToArray();
                    var optimizeMaterials = new bool[tmp_shader_assigns.Length];
                    var materials = ((modelType)nw4f_3dif.Item).material_array.GetItems().ToArray();

                    if (model.lastMaterialOptimizeData == null || model.lastMaterialOptimizeData.Length != materials.Length)
                    {
                        model.lastMaterialOptimizeData = new Tuple<Model.ShaderOptimizeData, string>[materials.Length];
                    }

                    if (model.MaterialOptimizeDataCache == null || model.MaterialOptimizeDataCache.Length != materials.Length)
                    {
                        model.MaterialOptimizeDataCache = new Tuple<Model.ShaderOptimizeData, string>[materials.Length];
                    }

                    foreach (var index in shaderOptimizeData.SelectMany(x => x.materials.Select(y => y.index)))
                    {
                        optimizeMaterials[index] = true;
                    }

                    List<byte[]> materialBytes = new List<byte[]>();
                    var modelStreams = modelDataList[0].Item3.ToList();

                    // 最適化に含めない場合は material_array を null に設定
                    for (int i = 0; i < tmp_shader_assigns.Length; i++)
                    {
                        if (!optimizeMaterials[i])
                        {
                            materialBytes.Add(null);
                            materials[i].shader_assign = null;
                        }
                        else
                        {
                            materialBytes.Add(ObjectUtility.ToBytes((materialType)materials[i]));
                        }
                    }

                    Dictionary<string, DateTime> lastWriteTimes;
                    if (!combinerShaderFileLastWriteTimes.TryGetValue(model, out lastWriteTimes))
                    {
                        lastWriteTimes = new Dictionary<string, DateTime>();
                    }

                    bool runScript = false;
                    // シェーダー定義出力
                    foreach (var item in shaderOptimizeData)
                    {
                        bool createFsv = false;
                        foreach (var material in item.materials)
                        {
                            var tuple = model.lastMaterialOptimizeData[material.index];
                            if (!model.ForcedOptimized && tuple != null && ObjectUtility.IsSameBytes(tuple.Item1.lastMaterialData[material.index], materialBytes[material.index]))
                            {
                                if (tuple.Item1.document == item.document &&
                                    tuple.Item1.definition == item.definition && // 必要?
                                    ObjectUtility.IsSameBytes(tuple.Item1.lastModelData, modelBytes) && // 比較結果をキャッシュする余地あり
                                    tuple.Item1.lastModelStream.Count == modelStreams.Count &&
                                    tuple.Item1.lastModelStream.Zip(modelStreams, (x, y) => IfUtility.IsSameStream(x, y)).All(x => x) &&
                                    tuple.Item1.DeviceTargeted == item.DeviceTargeted &&
                                    tuple.Item1.platform == item.platform &&
                                    tuple.Item1.useConverter32 == item.useConverter32 &&
                                    model.Materials[material.index].GetCombiners()
                                        .Where(x => !CombinerShaderConverterManager.ReservedValue.IsNullOrEmptyOrUnset(x.Item2.value))
                                        .Select(x => x.Item1.CreateCombinerShaderFilePath(x.Item2))
                                        .Where(x => !string.IsNullOrEmpty(x))
                                        .Distinct()
                                        .All(x =>
                                        {
                                            DateTime lastWriteTime;
                                            return lastWriteTimes.TryGetValue(x, out lastWriteTime) && (DateTime.Compare(lastWriteTime, !string.IsNullOrEmpty(x) ? (new System.IO.FileInfo(x)).LastWriteTime : default(DateTime)) == 0);
                                        }))
                                {
                                    material.reusedData = tuple;
                                    materials[material.index].shader_assign = null;
                                    materialBytes[material.index] = null;
                                    material.optimize = false;
                                    continue;
                                }
                            }

                            tuple = model.MaterialOptimizeDataCache[material.index];
                            if (!model.ForcedOptimized && tuple != null && ObjectUtility.IsSameBytes(tuple.Item1.lastMaterialData[material.index], materialBytes[material.index]))
                            {
                                if (tuple.Item1.document == item.document &&
                                    tuple.Item1.definition == item.definition && // 必要?
                                    ObjectUtility.IsSameBytes(tuple.Item1.lastModelData, modelBytes) && // 比較結果をキャッシュする余地あり
                                    tuple.Item1.lastModelStream.Count == modelStreams.Count &&
                                    tuple.Item1.lastModelStream.Zip(modelStreams, (x, y) => IfUtility.IsSameStream(x, y)).All(x => x) &&
                                    tuple.Item1.DeviceTargeted == item.DeviceTargeted &&
                                    tuple.Item1.platform == item.platform &&
                                    tuple.Item1.useConverter32 == item.useConverter32 &&
                                    model.Materials[material.index].GetCombiners()
                                        .Where(x => !CombinerShaderConverterManager.ReservedValue.IsNullOrEmptyOrUnset(x.Item2.value))
                                        .Select(x => x.Item1.CreateCombinerShaderFilePath(x.Item2))
                                        .Where(x => !string.IsNullOrEmpty(x))
                                        .Distinct()
                                        .All(x =>
                                        {
                                            DateTime lastWriteTime;
                                            return lastWriteTimes.TryGetValue(x, out lastWriteTime) && (DateTime.Compare(lastWriteTime, !string.IsNullOrEmpty(x) ? (new System.IO.FileInfo(x)).LastWriteTime : default(DateTime)) == 0);
                                        }))
                                {
                                    material.reusedData = tuple;
                                    materials[material.index].shader_assign = null;
                                    materialBytes[material.index] = null;
                                    material.optimize = false;
                                    continue;
                                }
                            }

                            Debug.Assert(material.reusedData == null);
                            if (material.reusedData == null)
                            {
                                if (material.forceOptimize)
                                {
                                    // 最適化する
                                    createFsv = true;
                                }
                                else
                                {
                                    // もう最適化しない
                                    material.optimize = false;
                                }
                            }
                        }

                        if (createFsv)
                        {
                            if (hasScript)
                            {
                                var nw4f_3dif1 = new nw4f_3difType()
                                {
                                    Item = item.definition,
                                };
                                var fsdb = System.IO.Path.Combine(tempDirectory, item.name + ".fsdb");

                                // 中断すると危険なのでロック
                                lock (Viewer.Connecter.AbortLock)
                                {
                                    IfWriteUtility.Write(nw4f_3dif1, item.streams, fsdb);
                                }
                                runScript = true;
                                item.runScript = true;
                            }
                            else
                            {
                                item.lastModelData = modelBytes;
                                item.lastModelStream = modelStreams;
                                item.lastMaterialData = materialBytes;
                                item.shader_variation = item.document.CreateDefaultFsvImage(
                                    item.materials.Where(x => x.optimize).Select(x => materials[x.index]).ToArray());
                            }
                        }
                    }

                    var tmpMessage = block.Message;
                    if (runScript)
                    {
                        block.Message = Strings.Viewer_FsvScript;
                        // モデル出力
                        List<G3dStream> streams = modelDataList[0].Item3;
                        if (G3dStreamUtility.HasStreamArray(nw4f_3dif))
                        {
                            nw4f_3dif.RootElement.stream_array = null;
                        }

                        // 中断すると危険なのでロック
                        lock (Viewer.Connecter.AbortLock)
                        {
                            IfWriteUtility.Write(nw4f_3dif, streams, fmdb);
                        }
                    }

                    // shader_assign を元に戻す
                    for (int i = 0; i < tmp_shader_assigns.Length; i++)
                    {
                        if (tmp_shader_assigns[i] != null)
                        {
                            materials[i].shader_assign = tmp_shader_assigns[i];
                        }
                    }

                    if (!runScript)
                    {
                        DebugConsole.WriteLine("Filtering skipped");
                        return;
                    }

                    var logStopWatch = new MessageLogStopWatch(App.res.Strings.Log_CreateFsv, App.ConfigData.ApplicationConfig.FileIo.LogShaderConvertTime);
                    var fsvCreated = CreateFsv(tempDirectory, outputBuilder, errorBuilder);
                    if (fsvCreated && !App.ConfigData.ApplicationConfig.FileIo.CreateShaderVariationCommand.AllowNoOutput)
                    {
                        // fsv未出力を許可しない設定の場合はすべてのfsvがあるかチェックする
                        if(shaderOptimizeData.Any(x => !File.Exists(Path.Combine(tempDirectory, x.name + ".fsva"))))
                        {
                            fsvCreated = false;
                        }
                    }

                    if (!fsvCreated)
                    {
                        // エラーのときはすぐにディレクトリを破棄しない。
                        // プログラム終了時のテンポラリディレクトリ以下の破棄時に破棄される。
                        disposableDirectory = null;

                        // 必要ならエラー通知
                        TheApp.MainFrame.BeginInvoke((Action)(()=>
                        {
                            using (var dialog = new OkListBoxDialog())
                            {
                                dialog.Text = Strings.General_Error;
                                dialog.lblDescription.Text = Strings.Viewer_FsvError;
                                dialog.AddLine(errorBuilder.ToString());
                                dialog.AddLine(outputBuilder.ToString());
                                dialog.ShowDialog();
                            }
                        }));

                        foreach (var material in shaderOptimizeData.SelectMany(x => x.materials))
                        {
                            material.optimize = false;
                        }
                        return;
                    }

                    var bytes = new List<byte[]>();
                    // fsva 読み込み
                    foreach (var data in shaderOptimizeData)
                    {
                        var fsva = System.IO.Path.Combine(tempDirectory, data.name + ".fsva");
                        if (data.runScript && File.Exists(fsva))
                        {
                            bytes.Add(System.IO.File.ReadAllBytes(fsva));
                        }
                        else
                        {
                            bytes.Add(null);
                        }
                    }

                    // 例外が起きたときに部分的に更新されるのを防ぐ
                    for (int i = 0; i < shaderOptimizeData.Count; i++)
                    {
                        var data = shaderOptimizeData[i];
                        if (data.runScript)
                        {
                            if (bytes[i] != null)
                            {
                                data.shader_variation = bytes[i];
                                data.lastModelData = modelBytes;
                                data.lastModelStream = modelStreams;
                                data.lastMaterialData = materialBytes;
                                foreach (var material in data.materials.Where(x => x.reusedData == null))
                                {
                                    material.optimize = true;
                                }
                            }
                            else // スクリプトを実行しfsvaが使われなかった場合はoptimizeをfalseにする
                            {

                                foreach (var material in data.materials.Where(x => x.reusedData == null))
                                {
                                    material.optimize = false;
                                }
                            }
                        }
                    }

                    block.Message = tmpMessage;

                    // fsv の生成から読み込みまでの時間を出力
                    // エラーや例外が起きたときはログ出力しない
                    logStopWatch.WriteLog();
                }
            }
            catch (ThreadAbortException)
            {
                // Abortに対しては特に何も行わない
            }
            catch (Exception exception)
            {
                // エラーのときはすぐにディレクトリを破棄しない。
                // プログラム終了時のテンポラリディレクトリ以下の破棄時に破棄される。
                disposableDirectory = null;

                var builder = new StringBuilder();
                builder.AppendLine(Strings.Viewer_FsvException);
                var e=exception;
                while (e != null)
                {
                    builder.AppendLine(e.Message);
                    e = e.InnerException;
                }

                TheApp.MainFrame.BeginInvoke((Action)(()=>UIMessageBox.Warning(builder.ToString())));

                foreach (var material in shaderOptimizeData.SelectMany(x => x.materials))
                {
                    material.optimize = false;
                }
            }
            finally
            {
                /*
                if (process != null)
                {
                    try
                    {
                        // プロセスが走っているかもしれないので止める
                        process.Kill();
                    }
                    catch
                    {
                        // 何もしない
                        DebugConsole.WriteLine("Failed to Kill process");
                    }
                }*/

                // デバッグログ出力
                var error = errorBuilder.ToString();
                if (error.Length > 0)
                {
                    DebugConsole.WriteLine(error);
                }

                var output = outputBuilder.ToString();
                if (output.Length > 0)
                {
                    DebugConsole.WriteLine(output);
                }

                if (disposableDirectory != null)
                {
                    disposableDirectory.Dispose();
                }
            }
        }

        public static bool CreateFsv(string tempDirectory, StringBuilder outputBuilder, StringBuilder errorBuilder)
        {
            if (string.IsNullOrEmpty(CreateFsvBat))
            {
                return RunFsvFilter(tempDirectory, outputBuilder, errorBuilder);
            }

            if (tempDirectory.Contains(' '))
            {
                // 一時ディレクトリのパスが空白を含む
                Debug.Assert(false);
                tempDirectory = "\"" + tempDirectory + "\"";
            }

            int? result = ProcessUtility.ProcessStart(
                CreateFsvBat,
                tempDirectory,
                (s, e) => { outputBuilder.AppendLine(e.Data); },
                (s, e) => { errorBuilder.AppendLine(e.Data ?? ""); },
                e => { errorBuilder.AppendLine(e.Message); });

            return result.HasValue && result.Value == 0;
        }

        public static bool RunFsvFilter(string tempDirectory, StringBuilder outputBuilder, StringBuilder errorBuilder)
        {
            bool done = false;
            Process process = null;
            try
            {
                var builder = new StringBuilder();
                // オプション
                builder.Append("--enable-extension=\"fmdb,fmda,fsdb,fsda\" ");

                // スクリプトファイルパス
                builder.AppendFormat("--script=\"{0}\" ", scriptFilePath);

                // 引数
                builder.AppendFormat("--argument=\"{0} {1}\" ", templateFilePath, tempDirectory);

                // モデルファイルパス
                builder.AppendFormat("{0} ", tempDirectory);

                DebugConsole.WriteLine(string.Format("{0} {1}", g3dfilterExePath, builder.ToString()));

                // プロセス情報設定
                var info = new ProcessStartInfo(g3dfilterExePath, builder.ToString());
                info.CreateNoWindow = true;
                info.UseShellExecute = false;
                info.RedirectStandardError = true;
                info.StandardErrorEncoding = Encoding.Default;
                info.RedirectStandardOutput = true;
                info.StandardOutputEncoding = Encoding.Default;
                info.WorkingDirectory = System.IO.Path.GetDirectoryName(scriptFilePath);

                using (new DebugStopWatch("g3dFilter"))
                {
                    // プロセス起動
                    process = Process.Start(info);
                    process.OutputDataReceived += (s, a) =>
                    {
                        if (!string.IsNullOrEmpty(a.Data))
                        {
                            outputBuilder.AppendLine(a.Data);
                        }
                    };
                    process.BeginOutputReadLine();
                    process.ErrorDataReceived += (s, a) =>
                    {
                        if (!string.IsNullOrEmpty(a.Data))
                        {
                            errorBuilder.AppendLine(a.Data);
                        }
                    };
                    process.BeginErrorReadLine();

                    // (未確認) process.WaitForExit(); だと Abort されない？
                    while (!process.WaitForExit(10000)) { }
                }

                done = true;
                return process.ExitCode == 0;
            }
            finally
            {
                if (!done && process != null)
                {
                    try
                    {
                        process.Kill();
                    }
                    catch
                    {
                        // 何もしない
                    }
                }
            }
        }
#endregion
        /// <summary>
        /// Hio.RuntimeState が初期化されるのを待つ
        /// </summary>
        public static void WaitRuntimeState()
        {
            using (var block = new LoadProgressDialog.DialogBlock(App.res.Strings.Viewer_Loading, () => FileMessage.DisconnectOnMainThreadAsync(), true))
            {
                using (var watch = new DebugStopWatch("Wait Unlock"))
                {
                    while (!(G3dHioLibProxy.Hio.GetRuntimeState().HasFlag(NintendoWare.G3d.Edit.RuntimeState.Unlock)) &&
                        (System.Threading.Thread.CurrentThread.ThreadState & System.Threading.ThreadState.AbortRequested) == 0 &&
                        G3dHioLibProxy.Hio.IsConnected)
                    {
                        lock (Connecter.SyncPollPing)
                        {
                            System.Threading.Monitor.Pulse(Connecter.SyncPollPing);
                        }
                        // 計測した結果 1 や 20 より少し早く処理が終わる
                        System.Threading.Thread.Sleep(10);
                    }
                }

                G3dHioLibProxy.Hio.ResetRuntimeState();

                using (var watch = new DebugStopWatch("Wait Normal"))
                {
                    while (G3dHioLibProxy.Hio.GetRuntimeState() != NintendoWare.G3d.Edit.RuntimeState.Normal &&
                        (System.Threading.Thread.CurrentThread.ThreadState & System.Threading.ThreadState.AbortRequested) == 0 &&
                        G3dHioLibProxy.Hio.IsConnected)
                    {
                        lock (Connecter.SyncPollPing)
                        {
                            System.Threading.Monitor.Pulse(Connecter.SyncPollPing);
                        }
                        // 計測した結果 1 や 20 より少し早く処理が終わる
                        System.Threading.Thread.Sleep(10);
                    }
                }
            }
        }

        /// <summary>
        /// リターゲットの設定の更新
        /// </summary>
        public static void SetRetargetHostModel(AnimationDocument target, Model model)
        {
            if (target.HioLoaded)
            {
                if (model == null || !model.HioLoaded)
                {
                    if (target.PreviewingRetargetHost != null)
                    {
                        G3dHioLibProxy.Hio.SetRetargetingHostModel(target, null);
                        target.PreviewingRetargetHost.RetargetHostingAnimations.Remove(target);
                        target.PreviewingRetargetHost = null;
                    }
                }
                else if (target.PreviewingRetargetHost != model)
                {
                    if (target.PreviewingRetargetHost != null)
                    {
                        target.PreviewingRetargetHost.RetargetHostingAnimations.Remove(target);
                    }

                    G3dHioLibProxy.Hio.SetRetargetingHostModel(target, model);
                    model.RetargetHostingAnimations.Add(target);
                    target.PreviewingRetargetHost = model;
                }
            }
        }

        public static bool LoadOrReloadOrSkipTexture(Texture texture, nw4f_3difType data, List<G3dStream> streams, string name, LoadProgressDialog.DialogBlock block)
        {
            string originalMessage = block != null ? block.Message : null;
            try
            {
                Tuple<byte[], List<G3dStreamCachedComparer>> tuple;
                bool reload = false;
                lock (DocumentManager.PreviewingTextures)
                {
                    DocumentManager.PreviewingTextures.TryGetValue(texture, out tuple);
                }

                if (tuple != null)
                {
                    if (tuple.Item2.Count == streams.Count &&
                        tuple.Item2.Zip(streams, (y, z) => y.IsSame(z)).All(y => y))
                    {
                        var bytes = ObjectUtility.ToBytes(data);
                        if (bytes.Length == tuple.Item1.Length &&
                            bytes.SequenceEqual(tuple.Item1))
                        {
                            // Skip
                            return true;
                        }
                    }

                    reload = true;
                }

                if (block != null)
                {
                    block.Message = string.Format(Strings.Viewer_ConvertingTexture, name);
                }

                // テクスチャが TGA から作られる場合、空白の file_info タグが作成されないように無効化する
                string lowerExt = texture.FileExt.ToLower();
                if (lowerExt == "tga")
                {
                    data.file_info = null;
                }

                // バイナリ生成
                var filePath = TemporaryFileUtility.MakeTemporaryFileName(".bntx");
                string error;
                uint alignment;
                if (!App.BinaryConverterManager.ConvertTextureBinary(data, streams, name, Connecter.Platform, filePath, out alignment, out error))
                {
                    return false;
                }

                if (block != null)
                {
                    block.Message = string.Format(Strings.Viewer_LoadingTexture, name);
                }

                if (reload)
                {
                    DebugConsole.WriteLine("HIO.ReloadTexture ---------- " + filePath);
                    G3dHioLibProxy.Hio.ReloadTexture(texture, new NintendoWare.G3d.Edit.FileData() { FileName = filePath, Alignment = alignment });
                }
                else
                {
                    var resetEvent = new ManualResetEventSlim();
                    Action fileLoaded = () => resetEvent.Set();
                    Viewer.Manager.Instance.AddFileLoadedAction(fileLoaded);
                    // 下位2bit が 00 にならない値
                    uint Key = texture.Key = (keyCount << 2) | 1;
                    keyCount++;

                    try
                    {
                        lock (Connecter.WaitingLoadFiles)
                        {
                            Connecter.WaitingLoadFiles[Key] = texture;
                        }

                        DebugConsole.WriteLine("HIO.LoadTexture ---------- " + filePath);
                        G3dHioLibProxy.Hio.LoadTexture(texture, new NintendoWare.G3d.Edit.FileData() { FileName = filePath, Alignment = alignment });
                        texture.HioLoaded = true;

                        // TODO: もう少し後で待つようにしたほうが効率がいいかもしれない
                        while (!resetEvent.Wait(1000) && Viewer.Manager.Instance.IsConnected)
                        {
                            ;
                        }
                    }
                    finally
                    {
                        lock (Connecter.WaitingLoadFiles)
                        {
                            Connecter.WaitingLoadFiles.Remove(Key);
                        }
                        Viewer.Manager.Instance.RemoveFileLoadedAction(fileLoaded);
                    }
                }

                {
                    // BinaryConverterManager.ConvertTextureBinary() 内部で、コンバート直前に data が更新 (data.UpdateArrayHint()) されるので、
                    // ConvertTextureBinary() 後の data をキャッシュする。
                    var bytes = ObjectUtility.ToBytes(data);

                    lock (DocumentManager.PreviewingTextures)
                    {
                        DocumentManager.PreviewingTextures[texture] = new Tuple<byte[], List<G3dStreamCachedComparer>>(bytes, streams.Select(x => new G3dStreamCachedComparer(x)).ToList());
                    }
                }

                return true;
            }
            finally
            {
                if (block != null)
                {
                    block.Message = originalMessage;
                }

            }
        }

        // 内部の Binding の追加、実際のバインドは後で行う
        public static void AddTextureBinding(Texture texture, GuiObject target)
        {
            lock (DocumentManager.TextureBindings)
            {
                HashSet<GuiObject> values;
                if (!DocumentManager.TextureBindings.TryGetValue(texture, out values))
                {
                    values = new HashSet<GuiObject>();
                    DocumentManager.TextureBindings[texture] = values;
                }

                if (values.Add(target))
                {
                    List<Texture> textures;
                    if (!DocumentManager.BindingTextures.TryGetValue(target, out textures))
                    {
                        textures = new List<Texture>();
                        DocumentManager.BindingTextures[target] = textures;
                    }

                    textures.Add(texture);
                }
            }
        }

        public static void UnloadUnusedTextures(GuiObject target)
        {
            UnloadUnusedTextures(target, Enumerable.Empty<Texture>());
        }

        public static void UnloadUnusedTextures(GuiObject target, IEnumerable<Texture> bindingTextures)
        {
            var notBoundTextures = new HashSet<Texture>();
            lock (DocumentManager.TextureBindings)
            {
                List<Texture> textures;
                if (DocumentManager.BindingTextures.TryGetValue(target, out textures))
                {
                    foreach (var texture in textures.Except(bindingTextures).ToArray())
                    {
                        HashSet<GuiObject> values;
                        if (DocumentManager.TextureBindings.TryGetValue(texture, out values))
                        {
                            values.Remove(target);
                            var count = values.Count;
                            if (count == 0)
                            {
                                DocumentManager.TextureBindings.Remove(texture);
                                notBoundTextures.Add(texture);
                            }
                        }
                        else
                        {
                            Debug.Assert(false);
                        }

                        textures.Remove(texture);
                    }

                    if (textures.Count == 0)
                    {
                        DocumentManager.BindingTextures.Remove(target);
                    }
                }
            }

            if (!SuspendUnloadTexture.Suspended)
            {
                foreach (var texture in notBoundTextures)
                {
                    DebugConsole.WriteLine("HIO.UnloadTexture ---------- " + texture.Name);
                    G3dHioLibProxy.Hio.UnloadTexture(texture);
                    texture.UnloadedFromHio();
                }

                lock (DocumentManager.PreviewingTextures)
                {
                    foreach (var texture in notBoundTextures)
                    {
                        DocumentManager.PreviewingTextures.Remove(texture);
                    }
                }
            }
        }

        public static void UnloadUnusedTexturesAll()
        {
            Texture[] previewingTextures;
            lock (DocumentManager.PreviewingTextures)
            {
                previewingTextures = DocumentManager.PreviewingTextures.Select(x => x.Key).ToArray();
            }

            Texture[] notBoundTextures;
            lock (DocumentManager.TextureBindings)
            {
                notBoundTextures = previewingTextures.Where(x => !DocumentManager.TextureBindings.ContainsKey(x)).ToArray();
            }

            foreach (var texture in notBoundTextures)
            {
                DebugConsole.WriteLine("HIO.UnloadTexture ---------- " + texture.Name);
                G3dHioLibProxy.Hio.UnloadTexture(texture);
                texture.UnloadedFromHio();
            }

            lock (DocumentManager.PreviewingTextures)
            {
                foreach (var texture in notBoundTextures)
                {
                    DocumentManager.PreviewingTextures.Remove(texture);
                }
            }
        }

        private static void UnloadTextureInternal(Texture texture)
        {
            lock (DocumentManager.TextureBindings)
            {
                HashSet<GuiObject> values;
                if (DocumentManager.TextureBindings.TryGetValue(texture, out values))
                {
                    foreach (var target in values)
                    {
                        DocumentManager.BindingTextures.Remove(target);
                    }
                    DocumentManager.TextureBindings.Remove(texture);
                }
            }

            G3dHioLibProxy.Hio.UnloadTexture(texture);
            DocumentManager.PreviewingTextures.Remove(texture);
        }
    }
}
