﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using App.Data;
using nw.g3d.nw4f_3dif;

namespace App.Utility
{
    static class VertexUtility
    {
        public static IEnumerable<Tuple<vertexType, string[]>> RelatedVerticesAndKeyShapeAttribNames(Model model, Material material)
        {
            return from shape in model.Shapes
                   where shape.Data.shape_info.mat_name == material.Name
                   let key_shape_attribs = KeyShapeAttribNames(shape.Data)
                   select new Tuple<vertexType, string[]>(model.Data.vertex_array.vertex[shape.Data.shape_info.vertex_index], key_shape_attribs);
        }

        public static string[] KeyShapeAttribNames(shapeType shape)
        {
            if (shape.key_shape_array == null)
            {
                return new string[0];
            }

            return (from key_shape in shape.key_shape_array.key_shape
                    from target_attrib in key_shape.target_attrib_array.target_attrib
                    select target_attrib.attrib_name).Distinct().ToArray();
        }

        /// <summary>
        /// 頂点バッファの設定をやり直す必要があるか？
        /// </summary>
        public static bool IsConsistent(vertexType vertex, IEnumerable<attrib_assignType> attrib_assigns, IEnumerable<interleaveType> interleaves, string[] keyShapeAttribNames)
        {
            int[][] indices = GetVtxBufferInputs(vertex, attrib_assigns, interleaves, keyShapeAttribNames);
            return SequenceEquals(vertex, indices);
        }

        public static bool SequenceEquals(vertexType vertex, int[][] indices)
        {
            if (indices.Length != vertex.VtxBuffers().Length)
            {
                return false;
            }

            var attribIndices = from buffer in vertex.VtxBuffers()
                                select from input in buffer.input_array.input select input.attrib_index;

            return indices.Zip(attribIndices, (x, y) => x.SequenceEqual(y)).All(x => x);
        }

        public static vtx_bufferType[] ToBufferArray(int[][] indices)
        {
            return (from buf in indices
                    select new vtx_bufferType()
                    {
                        input_array = new input_arrayType()
                        {
                            input = (from index in buf select new inputType() { attrib_index = index }).ToArray()
                        }
                    }).ToArray();
        }

        /// <summary>
        /// 頂点バッファの設定を生成する
        /// </summary>
        public static int[][] GetVtxBufferInputs(
            vertexType vertex,
            IEnumerable<attrib_assignType> attrib_assigns,
            IEnumerable<interleaveType> interleaves,
            string[] keyShapeAttribNames)
        {
            // バイナリ化対象
            var binarizedIndices = new HashSet<int>();

            // バイナリ化対象外
            var notBinarizedIndices = new HashSet<int>();

            var keyShapeIndices = new HashSet<int>();
            for (int i = 0; i < vertex.vtx_attrib_array.vtx_attrib.Length; i++)
            {
                if (keyShapeAttribNames.Contains(vertex.vtx_attrib_array.vtx_attrib[i].name))
                {
                    keyShapeIndices.Add(i);
                }
            }

            // 出力
            List<int[]> buffer = new List<int[]>();

            foreach (var interleave in interleaves)
            {
                var indices = GetVtxAttribIndices(vertex, attrib_assigns, interleave);

                if (!interleave.binarize)
                {
                    List<int> tmp = new List<int>();
                    foreach (var index in indices)
                    {
                        if (keyShapeIndices.Contains(index))
                        {
                            tmp.Add(index);
                        }
                        else
                        {
                            // バイナリ化対象外
                            notBinarizedIndices.Add(index);
                        }
                    }
                    indices = tmp.ToArray();
                }

                var list = new List<int>();

                // 割り当てがされているもの
                foreach (var index in indices.Where(x => x != -1))
                {
                    // 既にバイナリ化対象に含まれているものは除く
                    if (binarizedIndices.Contains(index))
                    {
                        continue;
                    }

                    list.Add(index);
                    binarizedIndices.Add(index);
                }

                // 空の場合は出力しない
                if (list.Count > 0)
                {
                    buffer.Add(list.ToArray());
                }
            }

            if (interleaves.Any(x => string.IsNullOrWhiteSpace(x.id_set) && x.binarize) // 陽に id_set="" のバイナリ出力が指定がされている場合
                || interleaves.All(x => !string.IsNullOrWhiteSpace(x.id_set))) // id_set="" の指定がない場合
            {
                var remain = Enumerable.Range(0, vertex.vtx_attrib_array.vtx_attrib.Length)
                    .Where(x => !binarizedIndices.Contains(x) && !notBinarizedIndices.Contains(x)).ToArray();

                // 残りを出力
                if (remain.Length > 0)
                {
                    buffer.Add(remain);
                }
            }
            else
            {
                // id_set="" binarize=false のとき

                // キーシェイプ関連の残りを出力
                var remain = Enumerable.Range(0, vertex.vtx_attrib_array.vtx_attrib.Length)
                    .Where(x => !binarizedIndices.Contains(x) && keyShapeIndices.Contains(x)).ToArray();

                if (remain.Length > 0)
                {
                    buffer.Add(remain);
                }
            }

            return buffer.ToArray();
        }

        /// <summary>
        /// interleave と attrib_assigns からvtx_attrib の index 列を生成する。
        /// attrib_assigns の未割り当て部分は -1 になる。
        /// id_set="" の場合は空の配列を返す。
        /// </summary>
        public static int[] GetVtxAttribIndices(vertexType vertex, IEnumerable<attrib_assignType> attrib_assigns, interleaveType interleave)
        {
                var ids = interleave.id_set.Split(" ".ToArray(), StringSplitOptions.RemoveEmptyEntries);
                return (from id in ids
                        let name = (from assign in attrib_assigns where assign.id == id select assign.attrib_name).FirstOrDefault()
                        select (from index in Enumerable.Range(0, vertex.vtx_attrib_array.vtx_attrib.Length)
                                let vtx_attrib = vertex.vtx_attrib_array.vtx_attrib[index]
                                where vtx_attrib.name == name
                                select index).DefaultIfEmpty(-1).FirstOrDefault()).ToArray();
        }

        // VtxBuffer
        public static vtx_bufferType[] VtxBuffers(this vertexType vertex)
        {
            return vertex.vtx_buffer_array == null ? new vtx_bufferType[0] : vertex.vtx_buffer_array.vtx_buffer;
        }
    }
}
