﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using ConfigCommon;
using nw.g3d.nw4f_3dif;

namespace App.Utility
{
    public static class ObjectIDUtility
    {
        private static readonly List<KeyValuePair<string, GuiObjectID>> extIdPairs;

        private static readonly Dictionary<string, GuiObjectID> extToID;

        private static readonly Dictionary<GuiObjectID, List<string>> idToExt;

        private static readonly GuiObjectID[] ModelAnimationIDs;

        /// <summary>
        /// ファイルの保存順
        /// </summary>
        /// <remarks>
        /// ファイルを新規に追加するときは、ObjectIDUitility 以外に
        /// Resource.Strings.FileFilter_***
        /// DocumentCheckedListView, BinarySaveDialog, SaveDirectoryDialog を更新する必要あり
        /// </remarks>
        public static readonly Dictionary<GuiObjectID, int> CloseOrder;
        static ObjectIDUtility()
        {
            extIdPairs = new List<KeyValuePair<string, GuiObjectID>>()
            {
                // http://www-sdd.zelda.nintendo.co.jp/project/nintendoware3/kagemai/html/user.cgi?project=nw3_3de&action=view_report&id=1380
                // fsda は対象外とする
                new KeyValuePair<string, GuiObjectID>(WithoutDot(G3dPath.ModelTextExtension),				GuiObjectID.Model),
                new KeyValuePair<string, GuiObjectID>(WithoutDot(G3dPath.ModelBinaryExtension),				GuiObjectID.Model),
                new KeyValuePair<string, GuiObjectID>(WithoutDot(G3dPath.TextureTextExtension),				GuiObjectID.Texture),
                new KeyValuePair<string, GuiObjectID>(WithoutDot(G3dPath.TextureBinaryExtension),			GuiObjectID.Texture),
                new KeyValuePair<string, GuiObjectID>(WithoutDot(G3dPath.SkeletalAnimTextExtension),		GuiObjectID.SkeletalAnimation),
                new KeyValuePair<string, GuiObjectID>(WithoutDot(G3dPath.SkeletalAnimBinaryExtension),		GuiObjectID.SkeletalAnimation),
                new KeyValuePair<string, GuiObjectID>(WithoutDot(G3dPath.ShapeAnimTextExtension),			GuiObjectID.ShapeAnimation),
                new KeyValuePair<string, GuiObjectID>(WithoutDot(G3dPath.ShapeAnimBinaryExtension),			GuiObjectID.ShapeAnimation),
                new KeyValuePair<string, GuiObjectID>(WithoutDot(G3dPath.MaterialAnimTextExtension),		GuiObjectID.MaterialAnimation),
                new KeyValuePair<string, GuiObjectID>(WithoutDot(G3dPath.MaterialAnimBinaryExtension),	    GuiObjectID.MaterialAnimation),
                new KeyValuePair<string, GuiObjectID>(WithoutDot(G3dPath.ShaderParamAnimTextExtension),		GuiObjectID.ShaderParameterAnimation),
                new KeyValuePair<string, GuiObjectID>(WithoutDot(G3dPath.ShaderParamAnimBinaryExtension),	GuiObjectID.ShaderParameterAnimation),
                new KeyValuePair<string, GuiObjectID>(WithoutDot(G3dPath.ColorAnimTextExtension),			GuiObjectID.ColorAnimation),
                new KeyValuePair<string, GuiObjectID>(WithoutDot(G3dPath.ColorAnimBinaryExtension),			GuiObjectID.ColorAnimation),
                new KeyValuePair<string, GuiObjectID>(WithoutDot(G3dPath.TexSrtAnimTextExtension),			GuiObjectID.TextureSrtAnimation),
                new KeyValuePair<string, GuiObjectID>(WithoutDot(G3dPath.TexSrtAnimBinaryExtension),		GuiObjectID.TextureSrtAnimation),
                new KeyValuePair<string, GuiObjectID>(WithoutDot(G3dPath.MatVisibilityAnimTextExtension),	GuiObjectID.MaterialVisibilityAnimation),
                new KeyValuePair<string, GuiObjectID>(WithoutDot(G3dPath.MatVisibilityAnimBinaryExtension),	GuiObjectID.MaterialVisibilityAnimation),
                new KeyValuePair<string, GuiObjectID>(WithoutDot(G3dPath.BoneVisibilityAnimTextExtension),	GuiObjectID.BoneVisibilityAnimation),
                new KeyValuePair<string, GuiObjectID>(WithoutDot(G3dPath.BoneVisibilityAnimBinaryExtension),GuiObjectID.BoneVisibilityAnimation),
                new KeyValuePair<string, GuiObjectID>(WithoutDot(G3dPath.TexPatternAnimTextExtension),		GuiObjectID.TexturePatternAnimation),
                new KeyValuePair<string, GuiObjectID>(WithoutDot(G3dPath.TexPatternAnimBinaryExtension),	GuiObjectID.TexturePatternAnimation),
                new KeyValuePair<string, GuiObjectID>(WithoutDot(G3dPath.SceneAnimTextExtension),			GuiObjectID.SceneAnimation),
                new KeyValuePair<string, GuiObjectID>(WithoutDot(G3dPath.SceneAnimBinaryExtension),			GuiObjectID.SceneAnimation),
//				new KeyValuePair<string, GuiObjectID>(WithoutDot(G3dPath.ShaderDefinitionTextExtension),	GuiObjectID.ShaderDefinition),
                new KeyValuePair<string, GuiObjectID>(WithoutDot(G3dPath.ShaderDefinitionBinaryExtension),	GuiObjectID.ShaderDefinition),
                new KeyValuePair<string, GuiObjectID>(WithoutDot(G3dPath.MaterialBinaryExtension),          GuiObjectID.SeparateMaterial),
                new KeyValuePair<string, GuiObjectID>("f3pj",                                               GuiObjectID.Project),
            };

            if ((ConfigData.ApplicationConfig.Setting.Debug != null) &&
                ConfigData.ApplicationConfig.Setting.Debug.SupportFsda)
            {
                extIdPairs.Add(new KeyValuePair<string, GuiObjectID>(WithoutDot(G3dPath.ShaderDefinitionTextExtension),	GuiObjectID.ShaderDefinition));
            }

            // 参照元を先に閉じます。
            GuiObjectID[] closeOrder = new GuiObjectID[]
            {
                GuiObjectID.Project,
                GuiObjectID.Model,
                GuiObjectID.SeparateMaterial,
                GuiObjectID.SkeletalAnimation,
                GuiObjectID.ShapeAnimation,
                GuiObjectID.MaterialAnimation,
                GuiObjectID.ShaderParameterAnimation,
                GuiObjectID.ColorAnimation,
                GuiObjectID.TextureSrtAnimation,
                GuiObjectID.MaterialVisibilityAnimation,
                GuiObjectID.BoneVisibilityAnimation,
                GuiObjectID.TexturePatternAnimation,
                GuiObjectID.SceneAnimation,
                GuiObjectID.Texture,
                GuiObjectID.ShaderDefinition            };

            ModelAnimationIDs = new GuiObjectID[]
            {
                GuiObjectID.SkeletalAnimation,
                GuiObjectID.ShapeAnimation,
                GuiObjectID.MaterialAnimation,
                GuiObjectID.ShaderParameterAnimation,
                GuiObjectID.ColorAnimation,
                GuiObjectID.TextureSrtAnimation,
                GuiObjectID.MaterialVisibilityAnimation,
                GuiObjectID.BoneVisibilityAnimation,
                GuiObjectID.TexturePatternAnimation,
                //GuiObjectID.SceneAnimation
            };

            CloseOrder = new Dictionary<GuiObjectID,int>();
            int order = 0;
            foreach (var id in closeOrder)
            {
                CloseOrder.Add(id, order++);
            }

            extToID = new Dictionary<string,GuiObjectID>();
            idToExt = new Dictionary<GuiObjectID,List<string>>();

            foreach (var keyValue in extIdPairs)
            {
                extToID.Add(keyValue.Key, keyValue.Value);
                if (!idToExt.ContainsKey(keyValue.Value))
                {
                    idToExt[keyValue.Value] = new List<string>();
                }

                idToExt[keyValue.Value].Add(keyValue.Key);
            }
        }

        public static GuiObjectID? ExtToId(string ext)
        {
            if (ext.StartsWith("."))
            {
                ext = ext.Substring(1);
            }
            ext = ext.ToLower();
            if (!extToID.ContainsKey(ext))
            {
                return null;
            }

            return extToID[ext];
        }

        public static IEnumerable<string> IdToExt(GuiObjectID id)
        {
            Debug.Assert(idToExt.ContainsKey(id));
            return idToExt[id];
        }

        public static string FileFilter(GuiObjectID id)
        {
            string filter = res.Strings.ResourceManager.GetString("FileFilter_" + id.ToString());

            string exts = string.Empty; // フィルター用
            string exts2 = string.Empty; // 表示用
            var extList = IdToExt(id).ToList();
            // tga対応
            if (id == GuiObjectID.Texture)
            {
                extList.Add("tga");
            }

            foreach (var ext in extList)
            {
                if (exts == string.Empty)
                {
                    exts = "*." + ext;
                }
                else
                {
                    exts += ";*." + ext;
                }

                if (exts2 == string.Empty)
                {
                    exts2 = "*." + ext;
                }
                else
                {
                    exts2 += ";*." + ext;
                }

            }

            filter += " (" + exts2 + ")|" + exts;

            return filter;
        }

        public static string FileFilterProject()
        {
            string exts = "*.f3pj";
            return res.Strings.FileFilter_Project + " (" + exts + ")|" + exts;
        }

        public static string FileFilterAll()
        {
            StringBuilder builder = new StringBuilder(); // フィルター用
            StringBuilder builder2 = new StringBuilder(); // 表示用
            foreach (var ext in extIdPairs)
            {
                builder.Append(builder.Length == 0 ? "*." : ";*.");
                builder.Append(ext.Key);

                if (ext.Key == "ftxb")
                {
                    // tga 対応
                    builder.Append(builder.Length == 0 ? "*." : ";*.");
                    builder.Append("tga");
                }
            }

            string exts = builder.ToString();
            string exts2 = builder.ToString();
            return res.Strings.FileFilter_All + " (" + exts2 + ")|" + exts;
        }

        public static string FileFilter(string fileType, IEnumerable<GuiObjectID> ids)
        {
            StringBuilder builder = new StringBuilder();
            foreach (var ext in ids.SelectMany(x => IdToExt(x)))
            {
                builder.Append(builder.Length == 0 ? "*." : ";*.");
                builder.Append(ext);
            }

            string exts = builder.ToString();
            return fileType + " (" + exts + ")|" + exts;
        }

        public static string FileFilterAnimation()
        {
            return FileFilter(res.Strings.FileFilter_Animation, ModelAnimationIDs.Concat(Enumerable.Repeat(GuiObjectID.SceneAnimation, 1)));
        }

        public static string FileFilterModelAnimation()
        {
            return FileFilter(res.Strings.FileFilter_Animation, ModelAnimationIDs);
        }

        public enum UnsupportedType
        {
            ShaderDefinitionTextExtension,
            //
            Undefined	= -1
        }

        public static bool IsSupportExtension(string ext, out UnsupportedType unsupportedType)
        {
            ext = WithoutDot(ext).ToLower();

            if ((ConfigData.ApplicationConfig.Setting.Debug == null) ||
                (ConfigData.ApplicationConfig.Setting.Debug.SupportFsda == false))
            {
                // http://www-sdd.zelda.nintendo.co.jp/project/nintendoware3/kagemai/html/user.cgi?project=nw3_3de&action=view_report&id=1380
                // fsda は対象外とする
                // 特別扱いする
                if (ext == WithoutDot(G3dPath.ShaderDefinitionTextExtension))
                {
                    unsupportedType = UnsupportedType.ShaderDefinitionTextExtension;
                    return false;
                }
            }

            //
            unsupportedType = UnsupportedType.Undefined;
            return extToID.ContainsKey(ext);
        }

        public static bool IsTgaExtension(string ext)
        {
            ext = WithoutDot(ext);
            return (ext.ToLower() == "tga");
        }


        public static bool CanOpenSameNameFile(string ext)
        {
            ext = WithoutDot(ext);
            var id = ExtToId(ext);
            if (!id.HasValue)
            {
                if (IsTgaExtension(ext))
                {
                    id = GuiObjectID.Texture;
                }
                else
                {
                    return false;
                }
            }

            return CanOpenSameNameFile(id.Value);
        }

        public static bool CanOpenSameNameFile(GuiObjectID id)
        {
            return id == GuiObjectID.Texture || ModelAnimationIDs.Any(x => x == id);
        }

        private static string WithoutDot(string src)
        {
            // 先頭の. を消去
            if (src.StartsWith("."))
            {
                src = src.Substring(1);
            }

            return src;
        }
    }
}
