﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using App.Command;
using App.ConfigData;
using App.Controls;
using App.Data;
using App.PropertyEdit.ShaderParamControls;
using App.Utility;
using App.res;
using App.ObjectView.List;
using ConfigCommon;
using nw.g3d.nw4f_3dif;

namespace App.PropertyEdit
{
    using System.ComponentModel;
    using System.Windows.Forms;

    using AppConfig;

    using nw.g3d.iflib.nw3de;

    using Viewer;

    /// <summary>
    /// モデルプロパティパネルクラス。
    /// </summary>
    public sealed class MaterialPropertyPanel : ObjectPropertyPanel
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public MaterialPropertyPanel()
        {
            InitializeComponent();
        }

        public override Utility.HelpUtility.PageKey HelpKey
        {
            get
            {
                return Utility.HelpUtility.PageKey.p_material_property_window;
            }
        }

        private PropertyCategoryNode ctgRoot = null;
        private PropertyCategoryNode ctgGeneral = null;
        private PropertyCategoryNode ctgPreview = null;
        private PropertyCategoryNode ctgShader = null;
        private PropertyCategoryNode ctgOriginal = null;
        private PropertyCategoryNode ctgRenderState = null;
        private bool IsCtgRenderStateVisible = false;
        private PropertyCategoryNode ctgSampler = null;
        private PropertyCategoryNode ctgParents = null;
        private PropertyCategoryNode ctgFileInfo = null;
        private PropertyCategoryNode ctgSearchPath = null;
        private PropertyCategoryNode ctgUserData = null;
        private UIContextMenuStrip cmsGroup;
        private UIToolStripMenuItem cmiGroupCopy;
        private UIToolStripMenuItem cmiGroupPaste;
        private UIToolStripMenuItem cmiGroupDefault;
        private UIContextMenuStrip cmsPage;
        private UIToolStripMenuItem cmiPageCopy;
        private UIToolStripMenuItem cmiPagePaste;
        private UIToolStripMenuItem cmiPageDefault;
        private UIToolStripSeparator uiToolStripSeparator1;
        private UIToolStripMenuItem cmiGroupShowMaterialList;
        private UIToolStripMenuItem cmiGroupHideMaterialList;
        private UIToolStripSeparator uiToolStripSeparator2;
        private UIToolStripMenuItem cmiPageShowMaterialList;
        private UIToolStripMenuItem cmiPageHideMaterialList;
        private UIContextMenuStrip cmsCategory;
        private UIToolStripMenuItem cmiCategoryShowMaterialList;
        private UIToolStripMenuItem cmiCategoryHideMaterialList;
        private UIToolStripSeparator uiToolStripSeparator3;
        public ToolStripMenuItem cmiGroupMaterialReferenceBehaviorValue;
        private ToolStripMenuItem cmiGroupMaterialReferenceBehaviorValue_Override;
        private ToolStripMenuItem cmiGroupMaterialReferenceBehaviorValue_Default;
        private ToolStripMenuItem cmiGroupMaterialReferenceBehaviorValue_Refer;
        private UIToolStripSeparator uiToolStripSeparator4;
        public ToolStripMenuItem cmiPageMaterialReferenceBehaviorValue;
        private ToolStripMenuItem cmiPageMaterialReferenceBehaviorValue_Override;
        private ToolStripMenuItem cmiPageMaterialReferenceBehaviorValue_Default;
        private ToolStripMenuItem cmiPageMaterialReferenceBehaviorValue_Refer;
        public ToolStripMenuItem cmiPageMaterialReferenceBehaviorRestriction;
        private ToolStripMenuItem cmiPageMaterialReferenceBehaviorRestriction_ForceRefer;
        private ToolStripMenuItem cmiPageMaterialReferenceBehaviorRestriction_ForceOverride;
        private ToolStripMenuItem cmiPageMaterialReferenceBehaviorRestriction_None;
        public ToolStripMenuItem cmiGroupMaterialReferenceBehaviorRestriction;
        private ToolStripMenuItem cmiGroupMaterialReferenceBehaviorRestriction_ForceRefer;
        private ToolStripMenuItem cmiGroupMaterialReferenceBehaviorRestriction_ForceOverride;
        private ToolStripMenuItem cmiGroupMaterialReferenceBehaviorRestriction_None;
        public readonly List<PropertyCategoryNode> ShaderParamPageNodes = new List<PropertyCategoryNode>();

        public PropertyCategoryNode FindShaderParamPageNode(Func<ShaderPageNodeTag, bool> predicate)
        {
            return ShaderParamPageNodes.Where(x => x.Tag != null).FirstOrDefault(x => predicate(x.Tag as ShaderPageNodeTag));
        }

        public class ShaderPageNodeTag
        {
            public string pageName;
            public string groupName;
            public bool isRoot;
            public bool IsSame(ShaderPageNodeTag tag)
            {
                return tag != null &&
                    tag.pageName == pageName &&
                    tag.groupName == groupName &&
                    tag.isRoot == isRoot;
            }
        }

        private void InitializeCategoryView()
        {
            CategoryView.ContextMenuPopup += CategoryView_ContextMenuPopup;

            // カテゴリ登録
            ctgRoot = RegisterCategory(0, Strings.ObjectPropertyPanel_Material_Root, PropertyPageID.MaterialRoot, MaterialRootPage.CreateInstance);
            ctgGeneral = RegisterCategory(1, Strings.ObjectPropertyPanel_Material_General, PropertyPageID.MaterialGeneral, MaterialGeneralPage.CreateInstance,
                updateModifiedFunc: () => MaterialGeneralPage.IsModified(ActiveTarget));
            ctgPreview = RegisterCategory(4, Strings.ObjectPropertyPanel_Material_Preview, PropertyPageID.MaterialPreview, MaterialPreviewPage.CreateInstance,
                updateModifiedFunc: () => MaterialPreviewPage.IsModified(ActiveTarget));
            ctgSampler = RegisterCategory(1, Strings.ObjectPropertyPanel_Material_Sampler, PropertyPageID.MaterialSampler, MaterialSamplerPage.CreateInstance,
                updateModifiedFunc: () => MaterialSamplerPage.IsModified(ActiveTarget));
            ctgOriginal = RegisterCategory(1, Strings.ObjectPropertyPanel_Material_OriginalInfo, PropertyPageID.MaterialOriginalInfo, MaterialOriginalInfoPage.CreateInstance);
            ctgShader = RegisterCategory(1, Strings.ObjectPropertyPanel_Material_Shader, PropertyPageID.MaterialShader, MaterialShaderPage.CreateInstance,
                updateModifiedFunc: () => MaterialShaderPage.IsModified(ActiveTarget));
            ctgParents = RegisterCategory(1, Strings.MaterialPropertyPanel_Material_Parents, PropertyPageID.MaterialBases, MaterialParentsPage.CreateInstance,
                updateModifiedFunc: () => MaterialParentsPage.IsModified(ActiveTarget),
                isValidPageFunc: () => MaterialParentsPage.IsValid(ActiveTarget));

            ctgUserData = RegisterCategory(1, Strings.ObjectPropertyPanel_UserData, PropertyPageID.UserData, UserDataPage.CreateInstance,
                updateModifiedFunc: () => UserDataPage.IsModified(ActiveTarget));

            ShaderParamPageNodes.Add(ctgShader);
            ctgShader.PropertyPageCreated += SetShaderParamPageNodePages;

            CategoryView.Nodes.Add(ctgRoot);
            {
                ctgRoot.Nodes.Add(ctgGeneral);
                ctgRoot.Nodes.Add(ctgPreview);

                // ctgRenderState は UpdateCategoryView で更新する
                ctgRoot.Nodes.Add(ctgSampler);
                ctgRoot.Nodes.Add(ctgShader);
                ctgRoot.Nodes.Add(ctgOriginal);
                ctgRoot.Nodes.Add(ctgParents);

                ctgRoot.Nodes.Add(ctgUserData);
            }

            ctgShader.Tag = new ShaderPageNodeTag()
            {
                isRoot = true,
            };
            CategoryView.ExpandAll();
        }

        private void CategoryView_ContextMenuPopup(object sender, ContextMenuPopupEventArgs e)
        {
            var node = CategoryView.SelectedNode as PropertyCategoryNode;
            if (node == null)
            {
                return;
            }

            switch (node.PageID)
            {
                case PropertyPageID.MaterialRoot:
                case PropertyPageID.MaterialPreview:
                case PropertyPageID.MaterialSampler:
                case PropertyPageID.MaterialRenderState:
                case PropertyPageID.MaterialGeneral:
                    cmsCategory.Show(CategoryView.PointToScreen(e.Location));
                    break;
                case PropertyPageID.MaterialShader:
                    {
                        var tag = (ShaderPageNodeTag)node.Tag;

                        if (string.IsNullOrEmpty(tag.groupName))
                        {
                            if (node == ctgShader)
                            {
                                // シェーダノード
                                cmsCategory.Show(CategoryView.PointToScreen(e.Location));
                            }
                            else
                            {
                                // ページノード
                                ShaderParamControlGroup.GroupCopyData data;
                                MaterialShaderPage.pageCopyData.TryGetValue(tag.pageName ?? string.Empty, out data);
                                cmiPagePaste.Enabled = data != null;
                                var groups = ActiveTarget.MaterialShaderAssign?.ShadingModel?.Groups().Where(x => x.page_name == tag.pageName).Select(x => x.name);
                                var behaviorItems = new List<MaterialReferenceBehaviorItem>();
                                if (groups != null)
                                {
                                    foreach (var groupName in groups)
                                    {
                                        behaviorItems.AddRange(ActiveTarget.GetAllReferenceBehaviorsInGroup(groupName, removeNull: false));
                                    }
                                    if (behaviorItems.Contains(null))
                                    {
                                        // null はマテリアル参照設定がデフォルトのパラメータが存在することを表すのでダミーを追加する
                                        behaviorItems.Add(Material.GetDefaultReferenceBehaviorItem("dummy"));
                                        behaviorItems.RemoveAll(x => x == null);
                                    }

                                    var values =
                                        new HashSet<ShaderItemValueState>(
                                            behaviorItems.Select(x => x?.Value ?? ShaderItemValueState.Refer));
                                    var restrictions =
                                        new HashSet<ShaderItemRestrictionState>(
                                            behaviorItems.Select(x => x?.ChildRestriction ?? ShaderItemRestrictionState.None));

                                    SetMaterialReferenceItemStateMenus(
                                        values,
                                        cmiPageMaterialReferenceBehaviorValue_Refer,
                                        cmiPageMaterialReferenceBehaviorValue_Override,
                                        cmiPageMaterialReferenceBehaviorValue_Default);
                                    SetMaterialReferenceItemStateMenus(
                                        restrictions,
                                        cmiPageMaterialReferenceBehaviorRestriction_None,
                                        cmiPageMaterialReferenceBehaviorRestriction_ForceRefer,
                                        cmiPageMaterialReferenceBehaviorRestriction_ForceOverride);
                                }

                                cmsPage.Show(CategoryView.PointToScreen(e.Location));
                            }
                        }
                        else
                        {
                            // グループノード
                            var groupId = tag.groupName;
                            cmiGroupPaste.Enabled = ShaderParamControlGroup.groupCopyData != null &&
                                 ((ShaderParamControlGroup.groupCopyData.group_id == groupId &&
                                   ShaderParamControlGroup.CanItemsPaste(Targets, ShaderParamControlGroup.groupCopyData, groupId, false, null, true)) ||
                                   ShaderParamControlGroup.CanItemsPaste2(Targets, ShaderParamControlGroup.groupCopyData2, groupId));
                            var behaviorItems = ActiveTarget.GetAllReferenceBehaviorsInGroup(groupId, removeNull: false).ToList();
                            if (behaviorItems.Contains(null))
                            {
                                // null はマテリアル参照設定がデフォルトのパラメータが存在することを表すのでダミーを追加する
                                behaviorItems.Add(Material.GetDefaultReferenceBehaviorItem("dummy"));
                                behaviorItems.RemoveAll(x => x == null);
                            }

                            var values = new HashSet<ShaderItemValueState>(behaviorItems.Select(x => x?.Value ?? ShaderItemValueState.Refer));
                            var restrictions =
                                new HashSet<ShaderItemRestrictionState>(
                                    behaviorItems.Select(x => x?.ChildRestriction ?? ShaderItemRestrictionState.None));

                            SetMaterialReferenceItemStateMenus(
                                values,
                                cmiGroupMaterialReferenceBehaviorValue_Refer,
                                cmiGroupMaterialReferenceBehaviorValue_Override,
                                cmiGroupMaterialReferenceBehaviorValue_Default);
                            SetMaterialReferenceItemStateMenus(
                                restrictions,
                                cmiGroupMaterialReferenceBehaviorRestriction_None,
                                cmiGroupMaterialReferenceBehaviorRestriction_ForceRefer,
                                cmiGroupMaterialReferenceBehaviorRestriction_ForceOverride);
                            cmsGroup.Show(CategoryView.PointToScreen(e.Location));
                        }
                    }
                    break;
                default:
                    return;
            }
        }

        private void SetShaderParamPageNodePages(PropertyCategoryNode sender, ObjectPropertyPage page)
        {
            foreach (var node in ShaderParamPageNodes)
            {
                if (node != sender)
                {
                    node.PropertyPage = page;
                }
            }
        }

        public override void SetDefaultCategory()
        {
            // UpdateCategoryView でページが選ばれるはずなのでここには来ない？
            Debug.Assert(false);
            CategoryView.SelectedNode = ctgGeneral;
        }

        protected override void InitializeFormInternal()
        {
            InitializeCategoryView();
            UpdateCategoryView();
            base.InitializeFormInternal();
            SetCategoryMinimumSize();

            // 個別のページでイベントを張ったほうがシンプルかもしれない。
            Action selectedChangedHandler = () =>
            {
                if (Targets != null && Targets.Active != null && ((Targets.Active.ObjectID == GuiObjectID.Material) || (Targets.Active.ObjectID == GuiObjectID.SeparateMaterial)))
                {
                    UpdateCategoryView();
                }
            };
            DocumentAddOrRemovedEventHandler addedOrRemoved = (o, a, r, s, reload) =>
                {
                    if (Targets != null && Targets.Active != null &&
                    ((Targets.Active.ObjectID == GuiObjectID.Material) || (Targets.Active.ObjectID == GuiObjectID.SeparateMaterial)) &&
                    Owner != null && Owner.Visible && (DocumentManager.Models.Contains(Targets.Active.OwnerDocument) || DocumentManager.SeparateMaterials.Contains(Targets.Active.OwnerDocument)))
                    {
                        UpdateCategoryView();
                    }
                };

            Action specChangedEventHandler = () =>
                {
                    if (Targets != null && Targets.Active != null &&
                    ((Targets.Active.ObjectID == GuiObjectID.Material) || (Targets.Active.ObjectID == GuiObjectID.SeparateMaterial)) &&
                    Owner != null && Owner.Visible && (DocumentManager.Models.Contains(Targets.Active.OwnerDocument) || DocumentManager.SeparateMaterials.Contains(Targets.Active.OwnerDocument)))
                    {
                        UpdateCategoryView();
                    }
                };

            App.AppContext.DocumentAddedOrRemoved += addedOrRemoved;
            App.AppContext.SpecChanged += specChangedEventHandler;
            Owner.TargetChanged += selectedChangedHandler;

            // 削除（Disposedで行う）
            Disposed += delegate
            {
                App.AppContext.DocumentAddedOrRemoved -= addedOrRemoved;
                App.AppContext.SpecChanged -= specChangedEventHandler;
            };
        }

        public static pageType defaultPage = new pageType()
        {
            name = string.Empty,
            ui_label = new ui_labelType() { value = res.Strings.MaterialPropertyPanel_Default },
            ui_order = new ui_orderType() { value = int.MinValue },
        };

        public void UpdateCategoryView()
        {
            if (ctgShader.PropertyPage != null)
            {
                ((IShaderPage)ctgShader.PropertyPage).UpdateControls = true;
            }

            // 選択されるノード
            PropertyCategoryNode lastSelectedNode = (PropertyCategoryNode)CategoryView.SelectedNode;
            ShaderPageNodeTag lastSelectedTag = null;
            PropertyCategoryNode selectedNode = null;
            if (lastSelectedNode != null && lastSelectedNode.PageID == PropertyPageID.MaterialShader)
            {
                lastSelectedTag = (ShaderPageNodeTag)lastSelectedNode.Tag;

                if (lastSelectedTag == null || lastSelectedTag.isRoot)
                {
                    selectedNode = ctgShader;
                }
            }
            else
            {
                selectedNode = lastSelectedNode;
            }

            if (ApplicationConfig.DefaultValue.RenderStateInfoVisible && App.AppContext.SelectedPlatformPreset.UseNw)
            {
                ctgRenderState = RegisterCategory(1, Strings.ObjectPropertyPanel_Material_RenderState, PropertyPageID.MaterialRenderState, MaterialRenderStatePage.CreateInstance,
                    updateModifiedFunc: () => MaterialRenderStatePage.IsModified(ActiveTarget));
            }
            var customUIGroups = ActiveTarget.GetAllAncestorMaterials().Where(x => x != null).Select(x => x.CustomUI.groups).ToArray();

            List<PropertyCategoryNode> nodes = new List<PropertyCategoryNode>();
            Dictionary<string, List<PropertyCategoryNode>> pageToGroups = new Dictionary<string, List<PropertyCategoryNode>>();
            Predicate<string> IsLabelMatch = MaterialShaderPage.IsMatch;
            bool pageMode = false;
            if (Targets != null && Targets.Active != null && ((Targets.Active.ObjectID == GuiObjectID.Material) || (Targets.Active.ObjectID == GuiObjectID.SeparateMaterial)))
            {
                var shadingModel = ActiveTarget.MaterialShaderAssign.ShadingModel;
                var table = Definition.GetTable(ActiveTarget.MaterialShaderAssign.Definition);
                var nameToGroup = shadingModel != null ? shadingModel.Groups().ToDictionary(x => x.name): new Dictionary<string, groupType>();
                var groupToRoot = new Dictionary<string, string>();
                var groups = ActiveTarget.MaterialShaderAssign.GetGroups(true).ToArray();

                var info = new VisibleInfo(ActiveTarget);

                var customUI = ActiveTarget.CustomUI;
                var attribGroups = (from attrib in shadingModel.Attributes()
                                    where IsLabelMatch(LabelHelper.GetLabelForFilter(attrib.id, attrib.Label(), customUI.attrib_vars, customUI, table))
                                    group attrib by RootGroupName(attrib.Group(), nameToGroup, groupToRoot))
                                    .ToDictionary(x => x.Key, x => x.Select(y => y.id).ToArray());
                var optionGroups = (from option in shadingModel.Options()
                                    where IsLabelMatch(LabelHelper.GetLabelForFilter(option.id, option.Label(), customUI.option_vars, customUI, table))
                                    group option by RootGroupName(option.Group(), nameToGroup, groupToRoot))
                                   .ToDictionary(x => x.Key, x => x.Select(y => y.id).ToArray());
                var samplerGroups = (from sampler in shadingModel.Samplers()
                                     where IsLabelMatch(LabelHelper.GetLabelForFilter(sampler.id, sampler.Label(), customUI.sampler_vars, customUI, table))
                                     group sampler by RootGroupName(sampler.Group(), nameToGroup, groupToRoot))
                                     .ToDictionary(x => x.Key, x => x.Select(y => y.id).ToArray());
                var uniformGroups = (from uniform in shadingModel.MaterialUniforms()
                                     where IsLabelMatch(LabelHelper.GetLabelForFilter(uniform.id, uniform.Label(), customUI.uniform_vars, customUI, table))
                                     group uniform by RootGroupName(uniform.Group(), nameToGroup, groupToRoot))
                                     .ToDictionary(x => x.Key, x => x.Select(y => y.id).ToArray());
                var renderInfoGroups = (from renderInfo in shadingModel.RenderInfoSlots()
                                        where IsLabelMatch(LabelHelper.GetLabelForFilter(renderInfo.name, renderInfo.Label(), customUI.render_info_slots, customUI, table))
                                        group renderInfo by RootGroupName(renderInfo.Group(), nameToGroup, groupToRoot))
                                        .ToDictionary(x => x.Key, x => x.Select(y => y.name).ToArray());
                var groupNameGroups = (from g in groups
                                       group g by RootGroupName(g.groupName, nameToGroup, groupToRoot))
                                       .ToDictionary(x => x.Key, x => x.Select(y => y.groupName).ToArray());

                var groupWithItems = new HashSet<string>(attribGroups.Keys
                    .Concat(optionGroups.Keys)
                    .Concat(samplerGroups.Keys)
                    .Concat(uniformGroups.Keys)
                    .Concat(renderInfoGroups.Keys));

                bool showConditionVariable = ctgShader.PropertyPage != null ?
                                    ((MaterialShaderPage)ctgShader.PropertyPage).ShowConditionVariable :
                                    ConfigData.ApplicationConfig.Setting.PropertyEdit.ShowConditionVariable;

                Predicate<string> isRootVisible;
                if (showConditionVariable)
                {
                    var visibleRoots = new HashSet<string>(groups.Where(x =>
                        (info.IsVisible(x.groupName) || info.ContainsCondition(x.groupName))
                        && groupWithItems.Contains(x.groupName))
                        .Select(x => RootGroupName(x.groupName, nameToGroup, groupToRoot)));

                    isRootVisible = x => visibleRoots.Contains(x);
                }
                else
                {
                    var children = (from g in groups
                                    let parent = g.@group != null ? g.@group.Group() : ""
                                    where !string.IsNullOrEmpty(parent)
                                    group g.groupName by parent).ToDictionary(x => x.Key, x => x.ToArray());

                    isRootVisible = x => IterateTree(x, children, y => info.IsVisible(y) && groupWithItems.Contains(y), y => info.IsVisible(y)).Any();
                }


                // ページノードの作成
                if (shadingModel != null && shadingModel.page_array != null && !MaterialShaderPage.IgnorePage)
                {
                    pageMode = true;

                    var groupToPage = new Dictionary<string, string>();

                    // ページ表示
                    var pages = (from g in groups
                                 where g.@group == null || string.IsNullOrEmpty(g.@group.Group())
                                 let pageName = g.@group != null ? g.@group.page_name: string.Empty
                                 let page = shadingModel.Pages().FirstOrDefault(x => x.name == pageName) ?? defaultPage
                                 orderby page.Order(), page.index
                                 group g by page).ToArray();

                    //var groups = pages.Where(x => x.Key.parentGroup != null)
                    //    .ToDictionary(x => x.Key.parentGroup, y => y.Select(x => x.groupName).ToArray());

                    foreach (var pageGroup in pages)
                    {
                        var attribs = new List<attrib_assignType>();
                        var options = new List<shader_optionType>();
                        var sampler_assigns = new List<sampler_assignType>();
                        var shader_params = new List<shader_paramType>();
                        var renderInfos = new List<RenderInfo>();
                        var groupNames = new List<string>();

                        bool showPage = false;
                        foreach (var group in pageGroup)
                        {
                            if (isRootVisible(group.groupName))
                            {
                                showPage = true;
                            }
                            {
                                string[] attribGroup;
                                if (attribGroups.TryGetValue(group.groupName, out attribGroup))
                                {
                                    attribs.AddRange(ActiveTarget.MaterialShaderAssign.AttribAssigns.Where(x => attribGroup.Contains(x.id)));
                                }
                                string[] optionGroup;
                                if (optionGroups.TryGetValue(group.groupName, out optionGroup))
                                {
                                    options.AddRange(ActiveTarget.MaterialShaderAssign.ShaderOptions.Where(x => optionGroup.Contains(x.id)));
                                }
                                string[] samplerGroup;
                                if (samplerGroups.TryGetValue(group.groupName, out samplerGroup))
                                {
                                    sampler_assigns.AddRange(ActiveTarget.MaterialShaderAssign.SamplerAssigns.Where(x => samplerGroup.Contains(x.id)));
                                }
                                string[] uniformGroup;
                                if (uniformGroups.TryGetValue(group.groupName, out uniformGroup))
                                {
                                    shader_params.AddRange(ActiveTarget.MaterialShaderAssign.ShaderParams.Where(x => uniformGroup.Contains(x.id)));
                                }
                                string[] renderInfoGroup;
                                if (renderInfoGroups.TryGetValue(group.groupName, out renderInfoGroup))
                                {
                                    renderInfos.AddRange(ActiveTarget.MaterialShaderAssign.RenderInfos.Where(x => renderInfoGroup.Contains(x.name)));
                                }
                                string[] groupNameGroup;
                                if (groupNameGroups.TryGetValue(group.groupName, out groupNameGroup))
                                {
                                    groupNames.AddRange(groupNameGroup);
                                }
                            }
                        }

                        if (showPage && (
                            attribs.Any() ||
                            options.Any() ||
                            sampler_assigns.Any() ||
                            shader_params.Any() ||
                            renderInfos.Any()))
                        {
                            var node = RegisterCategory(1, pageGroup.Key.Label() ?? pageGroup.Key.name, PropertyPageID.MaterialShader, MaterialShaderPage.CreateInstance);

                            // スコープを制限
                            var activeTarget = ActiveTarget;
                            node.UpdateModified = () =>
                            {
                                return (!activeTarget.IsShadingModelModified() && (
                                    attribs.Any(x => activeTarget.IsModified(x)) ||
                                    options.Any(x => activeTarget.IsModified(x)) ||
                                    sampler_assigns.Any(x => activeTarget.IsModified(x)) ||
                                    shader_params.Any(x => activeTarget.IsModified(x)) ||
                                    renderInfos.Any(x => activeTarget.IsModified(x)))) ||
                                    groupNames.Any(x => activeTarget.IsGroupCustomLabelModified(x));
                            };

                            node.ShowModifiedMark = node.UpdateModified();

                            node.PropertyPageCreated += SetShaderParamPageNodePages;
                            node.Tag = new ShaderPageNodeTag()
                            {
                                pageName = pageGroup.Key.name,
                            };
                            nodes.Add(node);
                            pageToGroups.Add(pageGroup.Key.name ?? "", new List<PropertyCategoryNode>());
                            if (((ShaderPageNodeTag)node.Tag).IsSame(lastSelectedTag))
                            {
                                selectedNode = node;
                            }
                        }
                    }
                }

                var dummy = new HashSet<string>();

                // グループノードの作成
                foreach (var group in ActiveTarget.MaterialShaderAssign.GetGroups(false).Where(x => x.group == null || string.IsNullOrEmpty(x.group.Group())))
                {
                    if (!isRootVisible(group.groupName))
                    {
                        continue;
                    }

                    var labelHelper = new LabelHelper();
                    var groupLabel = group.group == null ? group.Label: group.group.Label();
                    bool isParentLabel;
                    var customLabel = ShaderParamControlGroup.ResolveCustomLabel(customUIGroups, group.groupName, ActiveTarget.CustomUI.groups, out isParentLabel);
                    labelHelper.SetLabel(group.groupName, groupLabel, customLabel, ActiveTarget.CustomUI, table, x => false, dummy, false, false, isParentLabel);
                    var label = string.IsNullOrEmpty(customLabel)? labelHelper.GetLabelString(false) : customLabel;
                    var node = RegisterCategory(3, label, PropertyPageID.MaterialShader, MaterialShaderPage.CreateInstance);

                    // 変更チェック関数の設定
                    {
                        // スコープを制限
                        var activeTarget = ActiveTarget;
                        string[] attribGroup;
                        attribGroups.TryGetValue(group.groupName, out attribGroup);
                        var attribs = attribGroup == null ? Enumerable.Empty<attrib_assignType>() : ActiveTarget.MaterialShaderAssign.AttribAssigns.Where(x => attribGroup.Contains(x.id)).ToArray();
                        string[] optionGroup;
                        optionGroups.TryGetValue(group.groupName, out optionGroup);
                        var options = optionGroup == null ? Enumerable.Empty<shader_optionType>() : ActiveTarget.MaterialShaderAssign.ShaderOptions.Where(x => optionGroup.Contains(x.id)).ToArray();
                        string[] samplerGroup;
                        samplerGroups.TryGetValue(group.groupName, out samplerGroup);
                        var sampler_assigns = samplerGroup == null ? Enumerable.Empty<sampler_assignType>() : ActiveTarget.MaterialShaderAssign.SamplerAssigns.Where(x => samplerGroup.Contains(x.id)).ToArray();
                        string[] uniformGroup;
                        uniformGroups.TryGetValue(group.groupName, out uniformGroup);
                        var shader_params = uniformGroup == null ? Enumerable.Empty<shader_paramType>() : ActiveTarget.MaterialShaderAssign.ShaderParams.Where(x => uniformGroup.Contains(x.id)).ToArray();
                        string[] renderInfoGroup;
                        renderInfoGroups.TryGetValue(group.groupName, out renderInfoGroup);
                        var renderInfos = renderInfoGroup == null ? Enumerable.Empty<RenderInfo>() : ActiveTarget.MaterialShaderAssign.RenderInfos.Where(x => renderInfoGroup.Contains(x.name)).ToArray();
                        string[] groupNames;
                        groupNameGroups.TryGetValue(group.groupName, out groupNames);
                        Debug.Assert(groupNames != null);
                        node.UpdateModified = () =>
                            {
                                return (!activeTarget.IsShadingModelModified() && (
                            attribs.Any(x => activeTarget.IsModified(x)) ||
                            options.Any(x => activeTarget.IsModified(x)) ||
                            sampler_assigns.Any(x => activeTarget.IsModified(x)) ||
                            shader_params.Any(x => activeTarget.IsModified(x)) ||
                            renderInfos.Any(x => activeTarget.IsModified(x)))) ||
                            groupNames.Any(x => activeTarget.IsGroupCustomLabelModified(x));
                            };
                    }

                    node.ShowModifiedMark = node.UpdateModified();

                    node.PropertyPageCreated += SetShaderParamPageNodePages;

                    string pageName = null;
                    if (group.group != null && group.group.page_name != null)
                    {
                        if (pageToGroups.ContainsKey(group.group.page_name))
                        {
                            pageName = group.group.page_name;
                        }
                    }

                    node.Tag = new ShaderPageNodeTag()
                    {
                        groupName = group.groupName,
                        pageName = pageName,
                    };

                    if (pageMode)
                    {
                        Debug.Assert(pageToGroups.ContainsKey(pageName ?? ""));
                        Debug.Assert(pageToGroups[pageName ?? ""] != null);
                        pageToGroups[pageName ?? ""].Add(node);
                    }
                    else
                    {
                        nodes.Add(node);
                    }

                    if (((ShaderPageNodeTag)node.Tag).IsSame(lastSelectedTag))
                    {
                        selectedNode = node;
                    }
                }
            }

            HashSet<string> expandedPages = new HashSet<string>();
            foreach (var node in ctgShader.Nodes.OfType<PropertyCategoryNode>())
            {
                if (node.IsExpanded && node.Nodes.Count > 0)
                {
                    expandedPages.Add(((ShaderPageNodeTag)node.Tag).pageName ?? "");
                }
            }

            // レンダーステートページの作成
            bool renderStateVisible = ApplicationConfig.DefaultValue.RenderStateInfoVisible && App.AppContext.SelectedPlatformPreset.UseNw;
            if (renderStateVisible && ctgRenderState == null)
            {
                ctgRenderState = RegisterCategory(1, Strings.ObjectPropertyPanel_Material_RenderState, PropertyPageID.MaterialRenderState, MaterialRenderStatePage.CreateInstance,
                    updateModifiedFunc: () => MaterialRenderStatePage.IsModified(ActiveTarget));
            }

            bool expand = ctgShader.IsExpanded || ctgShader.Nodes.Count == 0;
            using (var lw = new LockWindowUpdate(CategoryView))
            {
                // レンダーステートページの更新
                if (renderStateVisible != IsCtgRenderStateVisible)
                {
                    if (renderStateVisible)
                    {
                        // サンプラページの前に挿入
                        var index = ctgRoot.Nodes.IndexOf(ctgSampler);
                        ctgRoot.Nodes.Insert(index, ctgRenderState);

                    }
                    else
                    {
                        ctgRoot.Nodes.Remove(ctgRenderState);
                        if (selectedNode == ctgRenderState)
                        {
                            selectedNode = null;
                        }
                    }

                    IsCtgRenderStateVisible = renderStateVisible;
                }

                ShaderParamPageNodes.Clear();
                ShaderParamPageNodes.Add(ctgShader);
                ShaderParamPageNodes.AddRange(nodes);

                ctgShader.Nodes.Clear();
                ctgShader.Nodes.AddRange(nodes.ToArray());

                if (pageMode)
                {
                    foreach (var page in nodes)
                    {
                        var tag = (ShaderPageNodeTag)page.Tag;
                        var groups = pageToGroups[tag.pageName ?? ""].ToArray();
                        page.Nodes.AddRange(groups);
                        ShaderParamPageNodes.AddRange(groups);
                        if (expandedPages.Contains(tag.pageName ?? ""))
                        {
                            page.Expand();
                        }
                    }
                }

                SetShaderParamPageNodePages(ctgShader, ctgShader.PropertyPage);

                // マテリアル中間ファイル用のページ調整。
                var oCtgFileInfo = ctgFileInfo;
                var oCtgSearchPath = ctgSearchPath;
                if (Targets?.Active.OwnerDocument is SeparateMaterial)
                {
                    SeparateMaterial doc = (SeparateMaterial)Targets.Active.OwnerDocument;
                    ctgFileInfo = RegisterCategory(1, Strings.ObjectPropertyPanel_FileInformation, PropertyPageID.FileInformation, FileInformationPage.CreateInstance,
                        updateModifiedFunc: () => FileInformationPage.IsModified(doc));
                    ctgSearchPath = RegisterCategory(1, Strings.ObjectPropertyPanel_Model_SearchPath, PropertyPageID.ModelSearchPath, SearchPathPage.CreateInstance,
                            updateModifiedFunc: () => SearchPathPage.IsModified(doc));

                    // 選択ページの維持。
                    if ((selectedNode == oCtgFileInfo) && (oCtgFileInfo != null))
                    {
                        selectedNode = ctgFileInfo;
                    }
                    else if ((selectedNode == oCtgSearchPath) && (oCtgSearchPath != null))
                    {
                        selectedNode = ctgSearchPath;
                    }
                }
                else
                {
                    ctgFileInfo = null;
                    ctgSearchPath = null;

                    // 選択ページを維持できない。
                    if (selectedNode == oCtgFileInfo)
                    {
                        selectedNode = null;
                    }
                    else if (selectedNode == oCtgSearchPath)
                    {
                        selectedNode = null;
                    }
                }

                if (selectedNode == null)
                {
                    if (pageMode)
                    {
                        if (lastSelectedTag != null)
                        {
                            selectedNode = nodes.FirstOrDefault(x => ((ShaderPageNodeTag)x.Tag).pageName == lastSelectedTag.pageName);
                        }
                    }

                    if (selectedNode == null)
                    {
                        selectedNode = (ctgShader.FirstNode as PropertyCategoryNode) ?? ctgShader;
                    }
                }

                // 選択ページが確定したので、
                // マテリアル中間ファイル用のページを削除または追加してページを完成させる。
                {
                    // 削除するページが選択されていたら Remove 前に未選択状態にする。
                    // Remove 前にクリアしなければ Remove 時のイベントで不正アクセスが発生する。
                    var targetNode =
                        ((oCtgFileInfo != null) && ((CategoryView.SelectedNode == oCtgFileInfo) || CategoryView.SelectedNodes.Contains(oCtgFileInfo))) ? oCtgFileInfo :
                        ((oCtgSearchPath != null) && ((CategoryView.SelectedNode == oCtgSearchPath) || CategoryView.SelectedNodes.Contains(oCtgSearchPath))) ? oCtgSearchPath :
                        null;
                    if (targetNode != null)
                    {
                        CategoryView.SelectedNodes.Remove(targetNode);
                        CategoryView.SelectedNode = null;
                    }

                    if (oCtgFileInfo != null)
                    {
                        ctgRoot.Nodes.Remove(oCtgFileInfo);
                    }
                    if (oCtgSearchPath != null)
                    {
                        ctgRoot.Nodes.Remove(oCtgSearchPath);
                    }

                    // ユーザーデータの前に挿入
                    var index = ctgRoot.Nodes.IndexOf(ctgUserData);
                    if (ctgSearchPath != null)
                    {
                        ctgRoot.Nodes.Insert(index, ctgSearchPath);
                    }
                    if (ctgFileInfo != null)
                    {
                        ctgRoot.Nodes.Insert(index, ctgFileInfo);
                    }
                }

                // ページの切り替えが起こらないので。
                CategoryView.SelectedNode = null;
                CategoryView.SelectedNode = selectedNode;

                if (expand)
                {
                    ctgShader.Expand();
                }
            }

            SetCategoryMinimumSize();
        }

        private static IEnumerable<string> IterateTree(string root, Dictionary<string, string[]> children, Predicate<string> emitNode, Predicate<string> emitChildren)
        {
            Func<string, IEnumerable<string>> getChildren = x =>
                {
                    string[] items;
                    if (emitNode(x) && children.TryGetValue(x, out items))
                    {
                        return items;
                    }
                    return Enumerable.Empty<string>();
                };
            return TreeUtility.PreOrder(root, getChildren).Where(x => emitNode(x));
        }

        public static string PageName(string groupName, Dictionary<string, groupType> nameToGroup, Dictionary<string, string> groupToPage)
        {
            string page;
            if (groupToPage.TryGetValue(groupName, out page))
            {
                return page;
            }

            page = "";
            groupType group;
            if (nameToGroup.TryGetValue(groupName, out group))
            {
                if (!string.IsNullOrEmpty(group.Group()))
                {
                    page = PageName(group.Group(), nameToGroup, groupToPage);
                }
                else if (!string.IsNullOrEmpty(group.page_name))
                {
                    page = group.page_name;
                }
            }

            groupToPage[groupName] = page;
            return page;
        }

        private static string RootGroupName(string groupName, Dictionary<string, groupType> nameToGroup, Dictionary<string, string> groupToRoot)
        {
            string root;
            if (groupToRoot.TryGetValue(groupName, out root))
            {
                return root;
            }

            root = groupName;
            groupType group;
            if (nameToGroup.TryGetValue(groupName, out group))
            {
                if (!string.IsNullOrEmpty(group.Group()))
                {
                    root = RootGroupName(group.Group(), nameToGroup, groupToRoot);
                }
            }

            groupToRoot[groupName] = root;
            return root;
        }

        /// <summary>
        /// アクティブターゲット。
        /// </summary>
        private Material ActiveTarget
        {
            get
            {
                // Material と MaterialDoc で共通のプロパティページを使用
                return (base.Targets?.Active is SeparateMaterial) ? ((SeparateMaterial)base.Targets.Active).Materials.FirstOrDefault() : (Material)base.Targets?.Active;
            }
        }

        protected override void  Event_Document_PropertyChanged(object sender, IEnumerable<DocumentPropertyChangedArgs> e)
        {
            if (Targets != null &&
                Targets.Active != null &&
                ((Targets.Active.ObjectID == GuiObjectID.Material) || (Targets.Active.ObjectID == GuiObjectID.SeparateMaterial)) &&
                e.Any(x => x is DocumentPropertyChangedFilterStringArgs || (x.Target == ActiveTarget && x is DocumentShaderPropertyChangedArgs)))
            {
                DebugConsole.WriteLine("ActiveMaterialChanged");
                UpdateCategoryView();
            }
            base.Event_Document_PropertyChanged(sender, e);
        }

        private void InitializeComponent()
        {
            System.ComponentModel.ComponentResourceManager resources = new System.ComponentModel.ComponentResourceManager(typeof(MaterialPropertyPanel));
            this.cmsGroup = new App.Controls.UIContextMenuStrip();
            this.cmiGroupCopy = new App.Controls.UIToolStripMenuItem();
            this.cmiGroupPaste = new App.Controls.UIToolStripMenuItem();
            this.cmiGroupDefault = new App.Controls.UIToolStripMenuItem();
            this.uiToolStripSeparator1 = new App.Controls.UIToolStripSeparator();
            this.cmiGroupShowMaterialList = new App.Controls.UIToolStripMenuItem();
            this.cmiGroupHideMaterialList = new App.Controls.UIToolStripMenuItem();
            this.uiToolStripSeparator3 = new App.Controls.UIToolStripSeparator();
            this.cmiGroupMaterialReferenceBehaviorValue = new System.Windows.Forms.ToolStripMenuItem();
            this.cmiGroupMaterialReferenceBehaviorValue_Refer = new System.Windows.Forms.ToolStripMenuItem();
            this.cmiGroupMaterialReferenceBehaviorValue_Override = new System.Windows.Forms.ToolStripMenuItem();
            this.cmiGroupMaterialReferenceBehaviorValue_Default = new System.Windows.Forms.ToolStripMenuItem();
            this.cmiGroupMaterialReferenceBehaviorRestriction = new System.Windows.Forms.ToolStripMenuItem();
            this.cmiGroupMaterialReferenceBehaviorRestriction_None = new System.Windows.Forms.ToolStripMenuItem();
            this.cmiGroupMaterialReferenceBehaviorRestriction_ForceRefer = new System.Windows.Forms.ToolStripMenuItem();
            this.cmiGroupMaterialReferenceBehaviorRestriction_ForceOverride = new System.Windows.Forms.ToolStripMenuItem();
            this.cmsPage = new App.Controls.UIContextMenuStrip();
            this.cmiPageCopy = new App.Controls.UIToolStripMenuItem();
            this.cmiPagePaste = new App.Controls.UIToolStripMenuItem();
            this.cmiPageDefault = new App.Controls.UIToolStripMenuItem();
            this.uiToolStripSeparator2 = new App.Controls.UIToolStripSeparator();
            this.cmiPageShowMaterialList = new App.Controls.UIToolStripMenuItem();
            this.cmiPageHideMaterialList = new App.Controls.UIToolStripMenuItem();
            this.uiToolStripSeparator4 = new App.Controls.UIToolStripSeparator();
            this.cmiPageMaterialReferenceBehaviorValue = new System.Windows.Forms.ToolStripMenuItem();
            this.cmiPageMaterialReferenceBehaviorValue_Refer = new System.Windows.Forms.ToolStripMenuItem();
            this.cmiPageMaterialReferenceBehaviorValue_Override = new System.Windows.Forms.ToolStripMenuItem();
            this.cmiPageMaterialReferenceBehaviorValue_Default = new System.Windows.Forms.ToolStripMenuItem();
            this.cmiPageMaterialReferenceBehaviorRestriction = new System.Windows.Forms.ToolStripMenuItem();
            this.cmiPageMaterialReferenceBehaviorRestriction_None = new System.Windows.Forms.ToolStripMenuItem();
            this.cmiPageMaterialReferenceBehaviorRestriction_ForceRefer = new System.Windows.Forms.ToolStripMenuItem();
            this.cmiPageMaterialReferenceBehaviorRestriction_ForceOverride = new System.Windows.Forms.ToolStripMenuItem();
            this.cmsCategory = new App.Controls.UIContextMenuStrip();
            this.cmiCategoryShowMaterialList = new App.Controls.UIToolStripMenuItem();
            this.cmiCategoryHideMaterialList = new App.Controls.UIToolStripMenuItem();
            ((System.ComponentModel.ISupportInitialize)(this.spcLeft)).BeginInit();
            this.spcLeft.SuspendLayout();
            ((System.ComponentModel.ISupportInitialize)(this.spcClient)).BeginInit();
            this.spcClient.Panel1.SuspendLayout();
            this.spcClient.SuspendLayout();
            this.cmsGroup.SuspendLayout();
            this.cmsPage.SuspendLayout();
            this.cmsCategory.SuspendLayout();
            this.SuspendLayout();
            //
            // spcClient
            //
            //
            // cmsGroup
            //
            this.cmsGroup.Items.AddRange(new System.Windows.Forms.ToolStripItem[] {
            this.cmiGroupCopy,
            this.cmiGroupPaste,
            this.cmiGroupDefault,
            this.uiToolStripSeparator1,
            this.cmiGroupShowMaterialList,
            this.cmiGroupHideMaterialList,
            this.uiToolStripSeparator3,
            this.cmiGroupMaterialReferenceBehaviorValue,
            this.cmiGroupMaterialReferenceBehaviorRestriction});
            this.cmsGroup.Name = "uiContextMenuStrip1";
            resources.ApplyResources(this.cmsGroup, "cmsGroup");
            //
            // cmiGroupCopy
            //
            resources.ApplyResources(this.cmiGroupCopy, "cmiGroupCopy");
            this.cmiGroupCopy.Name = "cmiGroupCopy";
            this.cmiGroupCopy.Overflow = System.Windows.Forms.ToolStripItemOverflow.AsNeeded;
            this.cmiGroupCopy.Click += new System.EventHandler(this.cmiCopy_Click);
            //
            // cmiGroupPaste
            //
            resources.ApplyResources(this.cmiGroupPaste, "cmiGroupPaste");
            this.cmiGroupPaste.Name = "cmiGroupPaste";
            this.cmiGroupPaste.Overflow = System.Windows.Forms.ToolStripItemOverflow.AsNeeded;
            this.cmiGroupPaste.Click += new System.EventHandler(this.cmiPaste_Click);
            //
            // cmiGroupDefault
            //
            resources.ApplyResources(this.cmiGroupDefault, "cmiGroupDefault");
            this.cmiGroupDefault.Name = "cmiGroupDefault";
            this.cmiGroupDefault.Overflow = System.Windows.Forms.ToolStripItemOverflow.AsNeeded;
            this.cmiGroupDefault.Click += new System.EventHandler(this.cmiDefault_Click);
            //
            // uiToolStripSeparator1
            //
            this.uiToolStripSeparator1.Name = "uiToolStripSeparator1";
            resources.ApplyResources(this.uiToolStripSeparator1, "uiToolStripSeparator1");
            //
            // cmiGroupShowMaterialList
            //
            resources.ApplyResources(this.cmiGroupShowMaterialList, "cmiGroupShowMaterialList");
            this.cmiGroupShowMaterialList.Name = "cmiGroupShowMaterialList";
            this.cmiGroupShowMaterialList.Overflow = System.Windows.Forms.ToolStripItemOverflow.AsNeeded;
            this.cmiGroupShowMaterialList.Click += new System.EventHandler(this.cmiShowMaterialList_Click);
            //
            // cmiGroupHideMaterialList
            //
            resources.ApplyResources(this.cmiGroupHideMaterialList, "cmiGroupHideMaterialList");
            this.cmiGroupHideMaterialList.Name = "cmiGroupHideMaterialList";
            this.cmiGroupHideMaterialList.Overflow = System.Windows.Forms.ToolStripItemOverflow.AsNeeded;
            this.cmiGroupHideMaterialList.Click += new System.EventHandler(this.cmiHideMaterialList_Click);
            //
            // uiToolStripSeparator3
            //
            this.uiToolStripSeparator3.Name = "uiToolStripSeparator3";
            resources.ApplyResources(this.uiToolStripSeparator3, "uiToolStripSeparator3");
            //
            // cmiGroupMaterialReferenceBehaviorValue
            //
            this.cmiGroupMaterialReferenceBehaviorValue.DropDownItems.AddRange(new System.Windows.Forms.ToolStripItem[] {
            this.cmiGroupMaterialReferenceBehaviorValue_Refer,
            this.cmiGroupMaterialReferenceBehaviorValue_Override,
            this.cmiGroupMaterialReferenceBehaviorValue_Default});
            this.cmiGroupMaterialReferenceBehaviorValue.Name = "cmiGroupMaterialReferenceBehaviorValue";
            resources.ApplyResources(this.cmiGroupMaterialReferenceBehaviorValue, "cmiGroupMaterialReferenceBehaviorValue");
            //
            // cmiGroupMaterialReferenceBehaviorValue_Refer
            //
            this.cmiGroupMaterialReferenceBehaviorValue_Refer.Name = "cmiGroupMaterialReferenceBehaviorValue_Refer";
            resources.ApplyResources(this.cmiGroupMaterialReferenceBehaviorValue_Refer, "cmiGroupMaterialReferenceBehaviorValue_Refer");
            this.cmiGroupMaterialReferenceBehaviorValue_Refer.Click += new System.EventHandler(this.CmiGroupMaterialReferenceBehaviorValueClick);
            //
            // cmiGroupMaterialReferenceBehaviorValue_Override
            //
            this.cmiGroupMaterialReferenceBehaviorValue_Override.Name = "cmiGroupMaterialReferenceBehaviorValue_Override";
            resources.ApplyResources(this.cmiGroupMaterialReferenceBehaviorValue_Override, "cmiGroupMaterialReferenceBehaviorValue_Override");
            this.cmiGroupMaterialReferenceBehaviorValue_Override.Click += new System.EventHandler(this.CmiGroupMaterialReferenceBehaviorValueClick);
            //
            // cmiGroupMaterialReferenceBehaviorValue_Default
            //
            this.cmiGroupMaterialReferenceBehaviorValue_Default.Name = "cmiGroupMaterialReferenceBehaviorValue_Default";
            resources.ApplyResources(this.cmiGroupMaterialReferenceBehaviorValue_Default, "cmiGroupMaterialReferenceBehaviorValue_Default");
            this.cmiGroupMaterialReferenceBehaviorValue_Default.Click += new System.EventHandler(this.CmiGroupMaterialReferenceBehaviorValueClick);
            //
            // cmiGroupMaterialReferenceBehaviorRestriction
            //
            this.cmiGroupMaterialReferenceBehaviorRestriction.DropDownItems.AddRange(new System.Windows.Forms.ToolStripItem[] {
            this.cmiGroupMaterialReferenceBehaviorRestriction_None,
            this.cmiGroupMaterialReferenceBehaviorRestriction_ForceRefer,
            this.cmiGroupMaterialReferenceBehaviorRestriction_ForceOverride});
            this.cmiGroupMaterialReferenceBehaviorRestriction.Name = "cmiGroupMaterialReferenceBehaviorRestriction";
            resources.ApplyResources(this.cmiGroupMaterialReferenceBehaviorRestriction, "cmiGroupMaterialReferenceBehaviorRestriction");
            //
            // cmiGroupMaterialReferenceBehaviorRestriction_None
            //
            this.cmiGroupMaterialReferenceBehaviorRestriction_None.Name = "cmiGroupMaterialReferenceBehaviorRestriction_None";
            resources.ApplyResources(this.cmiGroupMaterialReferenceBehaviorRestriction_None, "cmiGroupMaterialReferenceBehaviorRestriction_None");
            this.cmiGroupMaterialReferenceBehaviorRestriction_None.Click += new System.EventHandler(this.CmiGroupMaterialReferenceBehaviorRestrictionClick);
            //
            // cmiGroupMaterialReferenceBehaviorRestriction_ForceRefer
            //
            this.cmiGroupMaterialReferenceBehaviorRestriction_ForceRefer.Name = "cmiGroupMaterialReferenceBehaviorRestriction_ForceRefer";
            resources.ApplyResources(this.cmiGroupMaterialReferenceBehaviorRestriction_ForceRefer, "cmiGroupMaterialReferenceBehaviorRestriction_ForceRefer");
            this.cmiGroupMaterialReferenceBehaviorRestriction_ForceRefer.Click += new System.EventHandler(this.CmiGroupMaterialReferenceBehaviorRestrictionClick);
            //
            // cmiGroupMaterialReferenceBehaviorRestriction_ForceOverride
            //
            this.cmiGroupMaterialReferenceBehaviorRestriction_ForceOverride.Name = "cmiGroupMaterialReferenceBehaviorRestriction_ForceOverride";
            resources.ApplyResources(this.cmiGroupMaterialReferenceBehaviorRestriction_ForceOverride, "cmiGroupMaterialReferenceBehaviorRestriction_ForceOverride");
            this.cmiGroupMaterialReferenceBehaviorRestriction_ForceOverride.Click += new System.EventHandler(this.CmiGroupMaterialReferenceBehaviorRestrictionClick);
            //
            // cmsPage
            //
            this.cmsPage.Items.AddRange(new System.Windows.Forms.ToolStripItem[] {
            this.cmiPageCopy,
            this.cmiPagePaste,
            this.cmiPageDefault,
            this.uiToolStripSeparator2,
            this.cmiPageShowMaterialList,
            this.cmiPageHideMaterialList,
            this.uiToolStripSeparator4,
            this.cmiPageMaterialReferenceBehaviorValue,
            this.cmiPageMaterialReferenceBehaviorRestriction});
            this.cmsPage.Name = "cmsPage";
            resources.ApplyResources(this.cmsPage, "cmsPage");
            //
            // cmiPageCopy
            //
            resources.ApplyResources(this.cmiPageCopy, "cmiPageCopy");
            this.cmiPageCopy.Name = "cmiPageCopy";
            this.cmiPageCopy.Overflow = System.Windows.Forms.ToolStripItemOverflow.AsNeeded;
            this.cmiPageCopy.Click += new System.EventHandler(this.cmiPageCopy_Click);
            //
            // cmiPagePaste
            //
            resources.ApplyResources(this.cmiPagePaste, "cmiPagePaste");
            this.cmiPagePaste.Name = "cmiPagePaste";
            this.cmiPagePaste.Overflow = System.Windows.Forms.ToolStripItemOverflow.AsNeeded;
            this.cmiPagePaste.Click += new System.EventHandler(this.cmiPagePaste_Click);
            //
            // cmiPageDefault
            //
            resources.ApplyResources(this.cmiPageDefault, "cmiPageDefault");
            this.cmiPageDefault.Name = "cmiPageDefault";
            this.cmiPageDefault.Overflow = System.Windows.Forms.ToolStripItemOverflow.AsNeeded;
            this.cmiPageDefault.Click += new System.EventHandler(this.cmiPageDefault_Click);
            //
            // uiToolStripSeparator2
            //
            this.uiToolStripSeparator2.Name = "uiToolStripSeparator2";
            resources.ApplyResources(this.uiToolStripSeparator2, "uiToolStripSeparator2");
            //
            // cmiPageShowMaterialList
            //
            resources.ApplyResources(this.cmiPageShowMaterialList, "cmiPageShowMaterialList");
            this.cmiPageShowMaterialList.Name = "cmiPageShowMaterialList";
            this.cmiPageShowMaterialList.Overflow = System.Windows.Forms.ToolStripItemOverflow.AsNeeded;
            this.cmiPageShowMaterialList.Click += new System.EventHandler(this.cmiShowMaterialList_Click);
            //
            // cmiPageHideMaterialList
            //
            resources.ApplyResources(this.cmiPageHideMaterialList, "cmiPageHideMaterialList");
            this.cmiPageHideMaterialList.Name = "cmiPageHideMaterialList";
            this.cmiPageHideMaterialList.Overflow = System.Windows.Forms.ToolStripItemOverflow.AsNeeded;
            this.cmiPageHideMaterialList.Click += new System.EventHandler(this.cmiHideMaterialList_Click);
            //
            // uiToolStripSeparator4
            //
            this.uiToolStripSeparator4.Name = "uiToolStripSeparator4";
            resources.ApplyResources(this.uiToolStripSeparator4, "uiToolStripSeparator4");
            //
            // cmiPageMaterialReferenceBehaviorValue
            //
            this.cmiPageMaterialReferenceBehaviorValue.DropDownItems.AddRange(new System.Windows.Forms.ToolStripItem[] {
            this.cmiPageMaterialReferenceBehaviorValue_Refer,
            this.cmiPageMaterialReferenceBehaviorValue_Override,
            this.cmiPageMaterialReferenceBehaviorValue_Default});
            this.cmiPageMaterialReferenceBehaviorValue.Name = "cmiPageMaterialReferenceBehaviorValue";
            resources.ApplyResources(this.cmiPageMaterialReferenceBehaviorValue, "cmiPageMaterialReferenceBehaviorValue");
            //
            // cmiPageMaterialReferenceBehaviorValue_Refer
            //
            this.cmiPageMaterialReferenceBehaviorValue_Refer.Name = "cmiPageMaterialReferenceBehaviorValue_Refer";
            resources.ApplyResources(this.cmiPageMaterialReferenceBehaviorValue_Refer, "cmiPageMaterialReferenceBehaviorValue_Refer");
            this.cmiPageMaterialReferenceBehaviorValue_Refer.Click += new System.EventHandler(this.CmiPageMaterialReferenceBehaviorValueClick);
            //
            // cmiPageMaterialReferenceBehaviorValue_Override
            //
            this.cmiPageMaterialReferenceBehaviorValue_Override.Name = "cmiPageMaterialReferenceBehaviorValue_Override";
            resources.ApplyResources(this.cmiPageMaterialReferenceBehaviorValue_Override, "cmiPageMaterialReferenceBehaviorValue_Override");
            this.cmiPageMaterialReferenceBehaviorValue_Override.Click += new System.EventHandler(this.CmiPageMaterialReferenceBehaviorValueClick);
            //
            // cmiPageMaterialReferenceBehaviorValue_Default
            //
            this.cmiPageMaterialReferenceBehaviorValue_Default.Name = "cmiPageMaterialReferenceBehaviorValue_Default";
            resources.ApplyResources(this.cmiPageMaterialReferenceBehaviorValue_Default, "cmiPageMaterialReferenceBehaviorValue_Default");
            this.cmiPageMaterialReferenceBehaviorValue_Default.Click += new System.EventHandler(this.CmiPageMaterialReferenceBehaviorValueClick);
            //
            // cmiPageMaterialReferenceBehaviorRestriction
            //
            this.cmiPageMaterialReferenceBehaviorRestriction.DropDownItems.AddRange(new System.Windows.Forms.ToolStripItem[] {
            this.cmiPageMaterialReferenceBehaviorRestriction_None,
            this.cmiPageMaterialReferenceBehaviorRestriction_ForceRefer,
            this.cmiPageMaterialReferenceBehaviorRestriction_ForceOverride});
            this.cmiPageMaterialReferenceBehaviorRestriction.Name = "cmiPageMaterialReferenceBehaviorRestriction";
            resources.ApplyResources(this.cmiPageMaterialReferenceBehaviorRestriction, "cmiPageMaterialReferenceBehaviorRestriction");
            //
            // cmiPageMaterialReferenceBehaviorRestriction_None
            //
            this.cmiPageMaterialReferenceBehaviorRestriction_None.Name = "cmiPageMaterialReferenceBehaviorRestriction_None";
            resources.ApplyResources(this.cmiPageMaterialReferenceBehaviorRestriction_None, "cmiPageMaterialReferenceBehaviorRestriction_None");
            this.cmiPageMaterialReferenceBehaviorRestriction_None.Click += new System.EventHandler(this.CmiPageMaterialReferenceBehaviorRestrictionClick);
            //
            // cmiPageMaterialReferenceBehaviorRestriction_ForceRefer
            //
            this.cmiPageMaterialReferenceBehaviorRestriction_ForceRefer.Name = "cmiPageMaterialReferenceBehaviorRestriction_ForceRefer";
            resources.ApplyResources(this.cmiPageMaterialReferenceBehaviorRestriction_ForceRefer, "cmiPageMaterialReferenceBehaviorRestriction_ForceRefer");
            this.cmiPageMaterialReferenceBehaviorRestriction_ForceRefer.Click += new System.EventHandler(this.CmiPageMaterialReferenceBehaviorRestrictionClick);
            //
            // cmiPageMaterialReferenceBehaviorRestriction_ForceOverride
            //
            this.cmiPageMaterialReferenceBehaviorRestriction_ForceOverride.Name = "cmiPageMaterialReferenceBehaviorRestriction_ForceOverride";
            resources.ApplyResources(this.cmiPageMaterialReferenceBehaviorRestriction_ForceOverride, "cmiPageMaterialReferenceBehaviorRestriction_ForceOverride");
            this.cmiPageMaterialReferenceBehaviorRestriction_ForceOverride.Click += new System.EventHandler(this.CmiPageMaterialReferenceBehaviorRestrictionClick);
            //
            // cmsCategory
            //
            this.cmsCategory.Items.AddRange(new System.Windows.Forms.ToolStripItem[] {
            this.cmiCategoryShowMaterialList,
            this.cmiCategoryHideMaterialList});
            this.cmsCategory.Name = "cmsCategory";
            resources.ApplyResources(this.cmsCategory, "cmsCategory");
            //
            // cmiCategoryShowMaterialList
            //
            resources.ApplyResources(this.cmiCategoryShowMaterialList, "cmiCategoryShowMaterialList");
            this.cmiCategoryShowMaterialList.Name = "cmiCategoryShowMaterialList";
            this.cmiCategoryShowMaterialList.Overflow = System.Windows.Forms.ToolStripItemOverflow.AsNeeded;
            this.cmiCategoryShowMaterialList.Click += new System.EventHandler(this.cmiShowMaterialList_Click);
            //
            // cmiCategoryHideMaterialList
            //
            resources.ApplyResources(this.cmiCategoryHideMaterialList, "cmiCategoryHideMaterialList");
            this.cmiCategoryHideMaterialList.Name = "cmiCategoryHideMaterialList";
            this.cmiCategoryHideMaterialList.Overflow = System.Windows.Forms.ToolStripItemOverflow.AsNeeded;
            this.cmiCategoryHideMaterialList.Click += new System.EventHandler(this.cmiHideMaterialList_Click);
            //
            // MaterialPropertyPanel
            //
            this.Name = "MaterialPropertyPanel";
            this.Controls.SetChildIndex(this.spcClient, 0);
            ((System.ComponentModel.ISupportInitialize)(this.spcLeft)).EndInit();
            this.spcLeft.ResumeLayout(false);
            this.spcClient.Panel1.ResumeLayout(false);
            ((System.ComponentModel.ISupportInitialize)(this.spcClient)).EndInit();
            this.spcClient.ResumeLayout(false);
            this.cmsGroup.ResumeLayout(false);
            this.cmsPage.ResumeLayout(false);
            this.cmsCategory.ResumeLayout(false);
            this.ResumeLayout(false);

        }

        private void cmiCopy_Click(object sender, EventArgs e)
        {
            var tag = (ShaderPageNodeTag)((PropertyCategoryNode)CategoryView.SelectedNode).Tag;
            string groupName = tag.groupName;
            if (!string.IsNullOrEmpty(groupName))
            {
                ShaderParamControlGroup.UpdateGroupCopyData(ActiveTarget, groupName);
            }

            ObjectPropertyDialog.ForEachActivePanel(page =>
            {
                if (page is MaterialShaderPage)
                {
                    ((MaterialShaderPage)page).UpdateItemsCopyPasteButton();
                }
            });
        }

        private void cmiPaste_Click(object sender, EventArgs e)
        {
            var tag = (ShaderPageNodeTag)((PropertyCategoryNode)CategoryView.SelectedNode).Tag;
            string groupName = tag.groupName;
            if (groupName == ShaderParamControlGroup.groupCopyData.group_id &&
                ShaderParamControlGroup.CanItemsPaste(Targets, ShaderParamControlGroup.groupCopyData, groupName, false, null, true))
            {
                var command = ShaderParamControlGroup.CreateItemsPasteCommand(
                    Targets,
                    ShaderParamControlGroup.groupCopyData,
                    ShaderParamControlGroup.groupCopyData.group_id,
                    false,
                    null,
                    true,
                    true,
                    checkVisible: true);
                if (command != null)
                {
                    TheApp.CommandManager.Execute(command);
                }
            }
            else
            {
                var command = ShaderParamControlGroup.CreateItemsPasteCommand2(
                    Targets,
                    ShaderParamControlGroup.groupCopyData2,
                    groupName);
                if (command != null)
                {
                    TheApp.CommandManager.Execute(command);
                }
            }
        }

        private void cmiDefault_Click(object sender, EventArgs e)
        {
            var tag = (ShaderPageNodeTag)((PropertyCategoryNode)CategoryView.SelectedNode).Tag;
            string groupName = tag.groupName;
            if (!string.IsNullOrEmpty(groupName))
            {
                var command = ShaderParamControlGroup.CreateGroupSetToDefaultCommand(Targets, groupName, false, null);
                if (command != null)
                {
                    TheApp.CommandManager.Execute(command);
                }
            }
        }

        /// <summary>
        /// サンプラページを選択してサンプラページでサンプラを選択する
        /// </summary>
        public void SelectSampler(samplerType sampler)
        {
            CategoryView.SelectedNode = ctgSampler;
            var samplerPage = (MaterialSamplerPage)ActivePage;
            if (samplerPage != null)
            {
                samplerPage.SelectSampler(sampler);
            }
        }

        public void SelectShader(string groupName, string paramName, string elemName)
        {
            DebugConsole.WriteLine("MaterialPropertyPanel.SelectShader() -- groupName:{0}, paramName:{1}, elemName:{2}", groupName, paramName, elemName);

            CategoryView.SelectedNode = ctgShader;
            var page = (MaterialShaderPage)ActivePage;
            if (page != null)
            {
                var type = !string.IsNullOrEmpty(paramName) ? ParamType.uniform_var:
                    !string.IsNullOrEmpty(groupName) ? ParamType.group:
                    ParamType.page;
                page.SetScroll(groupName, paramName, type);
            }
        }

        private void cmiPageCopy_Click(object sender, EventArgs e)
        {
            var tag = (ShaderPageNodeTag)((PropertyCategoryNode)CategoryView.SelectedNode).Tag;
            string subId = tag.pageName;
            MaterialShaderPage.pageCopyData[subId ?? string.Empty] = ShaderParamControlGroup.CreatePageCopyData(ActiveTarget, true, subId, true, true);

            ObjectPropertyDialog.ForEachActivePanel(page =>
            {
                if (page is MaterialShaderPage)
                {
                    ((MaterialShaderPage)page).UpdateItemsCopyPasteButton();
                }
            });
        }

        private void cmiPagePaste_Click(object sender, EventArgs e)
        {
            //string subId = ((PropertyCategoryNode)CategoryView.SelectedNode).SubId;
            var tag = (ShaderPageNodeTag)((PropertyCategoryNode)CategoryView.SelectedNode).Tag;
            EditCommandSet commandSet = null;
            var pageMode = ActiveTarget.MaterialShaderAssign.PageMode;
            ShaderParamControlGroup.GroupCopyData copyData = null;
            MaterialShaderPage.pageCopyData.TryGetValue(tag.pageName ?? string.Empty, out copyData);
            Debug.Assert(copyData != null);

            commandSet = ShaderParamControlGroup.CreateItemsPasteCommand(Targets, copyData, null, pageMode, pageMode ? tag.pageName : null, true, false, checkVisible:true);

            if (commandSet != null)
            {
                commandSet.SetViewerDrawSuppressBlockDelegate(ViewerDrawSuppressBlock.DiscardAllMessages);
                var commandSet2 = new EditCommandSet();
                commandSet2.Add(commandSet);
                var models = Targets.Objects.OfType<Material>().SelectMany(x => x.Referrers).Distinct().ToArray();
                commandSet2.OnPostEdit += (s, e1) =>
                {
                    foreach (var model in models)
                    {
                        LoadOrReloadModel.Send(model);
                    }
                };

                TheApp.CommandManager.Execute(commandSet2);
            }
        }

        private void cmiPageDefault_Click(object sender, EventArgs e)
        {
            var tag = (ShaderPageNodeTag)((PropertyCategoryNode)CategoryView.SelectedNode).Tag;
            TheApp.CommandManager.Execute(ShaderParamControlGroup.CreateGroupSetToDefaultCommand(Targets, null, true, tag.pageName));
        }

        private void cmiShowMaterialList_Click(object sender, EventArgs e)
        {
            UpdateMaterialListView(true);
        }

        private void cmiHideMaterialList_Click(object sender, EventArgs e)
        {
            UpdateMaterialListView(false);
        }

        private void UpdateMaterialListView(bool value)
        {
            var pageCategoryNode = (PropertyCategoryNode)CategoryView.SelectedNode;
            Predicate<Column> inCategory = x => false;
            switch (pageCategoryNode.PageID)
            {
                case PropertyPageID.MaterialRoot:

                    // on にするのはすべての項目。off にするのは MaterialRoot 以外の項目!!
                    inCategory = x => value || x.Category != PropertyPageID.MaterialRoot.ToString();
                    break;
                case PropertyPageID.MaterialPreview:
                    inCategory = x => x.Category == PropertyPageID.MaterialPreview.ToString();
                    break;
                case PropertyPageID.MaterialGeneral:
                    inCategory = x => x.Category == PropertyPageID.MaterialGeneral.ToString();
                    break;
                case PropertyPageID.MaterialRenderState:
                    inCategory = x => x.Category == PropertyPageID.MaterialRenderState.ToString();
                    break;
                case PropertyPageID.MaterialSampler:
                    inCategory = x => x.Category == PropertyPageID.MaterialSampler.ToString();
                    break;
                case PropertyPageID.MaterialShader:
                    var tag = (ShaderPageNodeTag)pageCategoryNode.Tag;
                    if (tag.isRoot)
                    {
                        inCategory = x => x.Category == PropertyPageID.MaterialShader.ToString();
                        break;
                    }

                    var groups = ShaderParamControlGroup.CreateIsGroupInPage(
                        tag.groupName,
                        ActiveTarget.MaterialShaderAssign.PageMode,
                        ActiveTarget.MaterialShaderAssign.ShadingModel,
                        tag.pageName);

                    var idLabelType = new HashSet<Tuple<string, string, string>>();
                    var shadingModel = ActiveTarget.MaterialShaderAssign.ShadingModel;
                    foreach (var item in shadingModel.Attributes().Where(x => groups(x.Group())))
                    {
                        idLabelType.Add(
                            new Tuple<string, string, string>(
                                item.id,
                                item.Label() ?? item.id,
                                ColumnTypes.attrib_assign.ToString()));
                    }

                    foreach (var item in shadingModel.Options().Where(x => groups(x.Group())))
                    {
                        idLabelType.Add(
                            new Tuple<string, string, string>(
                                item.id,
                                item.Label() ?? item.id,
                                ColumnTypes.shader_option.ToString()));
                    }

                    foreach (var item in shadingModel.Samplers().Where(x => groups(x.Group())))
                    {
                        idLabelType.Add(
                            new Tuple<string, string, string>(
                                item.id,
                                item.Label() ?? item.id,
                                ColumnTypes.sampler_assign.ToString()));
                    }

                    foreach (var item in shadingModel.MaterialUniforms().Where(x => groups(x.Group())))
                    {
                        idLabelType.Add(
                            new Tuple<string, string, string>(
                                item.id,
                                item.Label() ?? item.id,
                                ColumnTypes.shader_param.ToString()));
                    }

                    foreach (var item in shadingModel.RenderInfoSlots().Where(x => groups(x.Group())))
                    {
                        idLabelType.Add(
                            new Tuple<string, string, string>(
                                item.name,
                                item.Label() ?? item.name,
                                ColumnTypes.render_info.ToString()));
                    }

                    inCategory = x => idLabelType.Contains(new Tuple<string, string, string>(x.Id, x.Name, x.Type));
                    break;
                default:
                    Debug.Assert(false);
                    break;
            }

            var config = ApplicationConfig.Setting.ObjectView.GetListView(ViewID.Material);
            foreach (var column in config.Columns.Where(x => inCategory(x)))
            {
                column.Show = value;
            }
            ObjectListView.Initialize(ViewID.Material);
            ObjectListView.GetColumnInfo(ViewID.Material).NotifySettingChanged();
        }

        private string GetSelectedNodeGroupName()
        {
            var node = CategoryView.SelectedNode as PropertyCategoryNode;
            if (node == null || node.PageID != PropertyPageID.MaterialShader)
            {
                return null;
            }

            var tag = (ShaderPageNodeTag)node.Tag;
            return string.IsNullOrEmpty(tag?.groupName) ? null : tag.groupName;
        }
        private string GetSelectedNodePageName()
        {
            var node = CategoryView.SelectedNode as PropertyCategoryNode;
            if (node == null || node.PageID != PropertyPageID.MaterialShader)
            {
                return null;
            }

            var tag = (ShaderPageNodeTag)node.Tag;
            return string.IsNullOrEmpty(tag?.pageName) ? string.Empty : tag.pageName;
        }

        private void CmiGroupMaterialReferenceBehaviorRestrictionClick(object sender, EventArgs e)
        {
            var groupName = GetSelectedNodeGroupName();
            if (string.IsNullOrEmpty(groupName))
            {
                return;
            }

            ShaderItemRestrictionState? state = null;

            if (sender == cmiGroupMaterialReferenceBehaviorRestriction_ForceRefer)
            {
                state = ShaderItemRestrictionState.ForceRefer;
            }
            else if (sender == cmiGroupMaterialReferenceBehaviorRestriction_ForceOverride)
            {
                state = ShaderItemRestrictionState.ForceOverride;
            }
            else if (sender == cmiGroupMaterialReferenceBehaviorRestriction_None)
            {
                state = ShaderItemRestrictionState.None;
            }
            Material.SetReferenceBehaviorToGroup(Targets, groupName, state, null);
        }

        private void CmiGroupMaterialReferenceBehaviorValueClick(object sender, EventArgs e)
        {
            var groupName = GetSelectedNodeGroupName();
            if (string.IsNullOrEmpty(groupName))
            {
                return;
            }

            ShaderItemValueState? state = null;

            if (sender == cmiGroupMaterialReferenceBehaviorValue_Default)
            {
                state = ShaderItemValueState.Default;
            }
            else if (sender == cmiGroupMaterialReferenceBehaviorValue_Override)
            {
                state = ShaderItemValueState.Override;
            }
            else if (sender == cmiGroupMaterialReferenceBehaviorValue_Refer)
            {
                state = ShaderItemValueState.Refer;
            }
            Material.SetReferenceBehaviorToGroup(Targets, groupName, null, state);
        }

        private void CmiPageMaterialReferenceBehaviorRestrictionClick(object sender, EventArgs e)
        {
            var pageName = GetSelectedNodePageName();
            var groups = ActiveTarget.MaterialShaderAssign?.ShadingModel?.Groups().Where(x => x.page_name == pageName).Select(x => x.name);
            if (groups == null || !groups.Any())
            {
                return;
            }

            ShaderItemRestrictionState? state = null;

            if (sender == cmiGroupMaterialReferenceBehaviorRestriction_ForceRefer)
            {
                state = ShaderItemRestrictionState.ForceRefer;
            }
            else if (sender == cmiGroupMaterialReferenceBehaviorRestriction_ForceOverride)
            {
                state = ShaderItemRestrictionState.ForceOverride;
            }
            else if (sender == cmiGroupMaterialReferenceBehaviorRestriction_None)
            {
                state = ShaderItemRestrictionState.None;
            }
            Material.SetReferenceBehaviorToGroups(Targets, groups, state, null);
        }

        private void CmiPageMaterialReferenceBehaviorValueClick(object sender, EventArgs e)
        {
            var pageName = GetSelectedNodePageName();
            var groups = ActiveTarget.MaterialShaderAssign?.ShadingModel?.Groups().Where(x => x.page_name == pageName).Select(x => x.name);
            if (groups == null || !groups.Any())
            {
                return;
            }
            ShaderItemValueState? state = null;

            if (sender == cmiPageMaterialReferenceBehaviorValue_Default)
            {
                state = ShaderItemValueState.Default;
            }
            else if (sender == cmiPageMaterialReferenceBehaviorValue_Override)
            {
                state = ShaderItemValueState.Override;
            }
            else if (sender == cmiPageMaterialReferenceBehaviorValue_Refer)
            {
                state = ShaderItemValueState.Refer;
            }
            Material.SetReferenceBehaviorToGroups(Targets, groups, null, state);
        }

        internal static void SetMaterialReferenceItemStateMenus<T>(
            HashSet<T> editables,
            ToolStripMenuItem firstMenuItem,
            ToolStripMenuItem secondMenuItem,
            ToolStripMenuItem thirdMenuItem)
        {
            bool firstChecked;
            bool secondChecked;
            bool thirdChecked;
            var multiSelect = GetCheckInfo(editables, out firstChecked, out secondChecked, out thirdChecked);
            firstMenuItem.CheckState = GetCheckState(firstChecked, multiSelect);
            secondMenuItem.CheckState = GetCheckState(secondChecked, multiSelect);
            thirdMenuItem.CheckState = GetCheckState(thirdChecked, multiSelect);
        }

        internal static CheckState GetCheckState(bool trueChecked, bool multiSelect)
        {
            return trueChecked ? (multiSelect ? CheckState.Indeterminate : CheckState.Checked) : CheckState.Unchecked;
        }

        internal static bool GetCheckInfo<T>(
            HashSet<T> states,
            out bool firstChecked,
            out bool secondChecked,
            out bool thirdChecked)
        {
            firstChecked = true;
            secondChecked = false;
            thirdChecked = false;
            if (states.Count == 0)
            {
                return false;
            }

            if (states is HashSet<ShaderItemValueState>)
            {
                var valueStates = states as HashSet<ShaderItemValueState>;
                firstChecked = valueStates.Contains(ShaderItemValueState.Refer);
                secondChecked = valueStates.Contains(ShaderItemValueState.Override);
                thirdChecked = valueStates.Contains(ShaderItemValueState.Default);
            }
            else if (states is HashSet<ShaderItemRestrictionState>)
            {
                var restrictionStates = states as HashSet<ShaderItemRestrictionState>;
                firstChecked = restrictionStates.Contains(ShaderItemRestrictionState.None);
                secondChecked = restrictionStates.Contains(ShaderItemRestrictionState.ForceRefer);
                thirdChecked = restrictionStates.Contains(ShaderItemRestrictionState.ForceOverride);
            }
            return states.Count > 1;
        }
    }

    public class DocumentShaderPropertyChangedArgs : DocumentContentsChangedArgs
    {
        public DocumentShaderPropertyChangedArgs(GuiObject target, EditCommand command) : base(target, command) {}
    }

    public class DocumentPropertyChangedShaderArgs : DocumentShaderPropertyChangedArgs
    {
        public DocumentPropertyChangedShaderArgs(GuiObject target, EditCommand command) : base(target, command) {}
    }

    public class DocumentPropertyChangedShaderOptionArgs : DocumentShaderPropertyChangedArgs
    {
        public DocumentPropertyChangedShaderOptionArgs(GuiObject target, EditCommand command) : base(target, command) { }
    }

    public class DocumentPropertyChangedLabelArgs : DocumentShaderPropertyChangedArgs
    {
        public DocumentPropertyChangedLabelArgs(GuiObject target, EditCommand command) : base(target, command) { }
    }

    public class DocumentPropertyChangedFilterStringArgs : DocumentShaderPropertyChangedArgs
    {
        public DocumentPropertyChangedFilterStringArgs(GuiObject target, EditCommand command) : base(target, command) { }
    }

    public interface IShaderPage
    {
        bool UpdateControls { get; set; }
    }
}
