﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Linq;
using System.Windows.Forms;
using App.Data;
using App.res;
using App.Utility;
using ConfigCommon;
using nw.g3d.nw4f_3dif;

namespace App.PropertyEdit
{
    public partial class CurveViewPainter
    {
        [Flags]
        private enum RangeDrawFlag
        {
            BackGround = 1,
            Line = 1 << 1,
        }

        private CurveViewState	State{					get{ return view_.State;								} }

        private int				CurveViewWidth{				get{ return view_.Width;								} }
        private int				CurveViewHeight{				get{ return view_.Height;								} }

        public int				CurveBgLeft{			get{ return CurveEditorConst.RulerSizeV;					} }
        public int				CurveBgTop{				get{ return 0;											} }
        public int				CurveBgRight{			get{ return CurveBgLeft + CurveBgWidth;					} }
        public int				CurveBgBottom{			get{ return CurveBgTop  + CurveBgHeight;				} }
        public int				CurveBgWidth{			get{ return CurveViewWidth  - CurveEditorConst.RulerSizeV;	} }
        public int				CurveBgHeight{			get{ return CurveViewHeight - CurveEditorConst.RulerSizeH;	} }

        public int				HRulerBgLeft{			get{ return CurveEditorConst.RulerSizeV;					} }
        public int				HRulerBgTop{			get{ return CurveViewHeight - CurveEditorConst.RulerSizeH;	} }
        public int				HRulerBgRight{			get{ return HRulerBgLeft + HRulerBgWidth;				} }
        public int				HRulerBgBottom{			get{ return HRulerBgTop  + HRulerBgHeight;				} }
        public int				HRulerBgWidth{			get{ return CurveViewWidth  - CurveEditorConst.RulerSizeV;	} }
        public int				HRulerBgHeight{			get{ return CurveEditorConst.RulerSizeH;					} }

        public int				HRulerColorBarBottom{	get{ return HRulerBgTop + CurveEditorConst.ColorBarSize;	} }

        public int				VRulerBgLeft{			get{ return 0;											} }

        public int				VRulerBgTop{			get{ return 0;											} }
        public int				VRulerBgRight{			get{ return VRulerBgLeft + VRulerBgWidth;				} }
        public int				VRulerBgBottom{			get{ return VRulerBgTop  + VRulerBgHeight;				} }
        public int				VRulerBgWidth{			get{ return CurveEditorConst.RulerSizeV;					} }
        public int				VRulerBgHeight{			get{ return CurveViewHeight - CurveEditorConst.RulerSizeH;	} }

        /// <summary>
        /// カーブ原点のクライアント座標系のX位置
        /// </summary>
        private double			OriginXInCurveView{	get{ return CurveBgLeft + (CurveBgWidth  * 0.5) - State.ScrollX * State.ScaleX - VRulerBgWidth;	} }
        /// <summary>
        /// カーブ原点のクライアント座標系のY位置
        /// </summary>
        private double OriginYInCurveView { get { return CurveBgTop + (CurveBgHeight * 0.5) - State.ScrollY * State.ScaleY; } }

        private bool			IsSelection{			get{ return State.IsSelection;	} }

        public List<IAnimationCurve>	SelectedCurves{	get{ return view_.SelectedCurves; } }
        public List<IAnimationCurve> 	VisibledCurves{	get{ return view_.VisibledCurves; } }

        public int		FrameCount{	get{ return view_.FrameCount;	} }

        private readonly CurveView	view_;

        public CurveViewPainter(CurveView view)
        {
            view_	= view;
            InitBitmaps();

            // 描画前にヒットチェックが行われることがあるのでインスタンスを生成しておく
            OldDrawCurvePositions = new List<OldDrawCurvePosition>();
        }

        public void Paint(PaintEventArgs e, GuiObject target)
        {
            var valueRange = view_.MakeEditValueRange();

            DrawCurveBg(e, target);
            DrawCurveGrid(e);
            DrawCurveAxis(e);

            // 上下範囲の境界線をカーブよりも先に書き、境界線上のカーブの視認性を上げる。
            DrawValueRange(e, target, valueRange, RangeDrawFlag.Line);

            DrawCurveKeySnap(e);
            DrawCurves(e, target);

            if (IsSelection)
            {
                DrawSelection(e);
            }

            // 範囲外の背景描画をカーブの後に行い、範囲外のカーブに背景色をブレンドさせる。
            DrawValueRange(e, target, valueRange, RangeDrawFlag.BackGround);
            DrawClampLine(e, valueRange);

            DrawHRulerBg(e);

            bool barDrawn = false;
            barDrawn |= DrawHRulerColorBar(e, target);
            barDrawn |= DrawHRulerVisibleBar(e, target);
            barDrawn |= DrawHRulerTexturePatternBar(e, target);

            if (barDrawn)
            {
                DrawHBarAxis(e, CurveEditorConst.ColorBarSize);
            }
            DrawHRulerAxis(e);
            DrawHRulerGrid(e);

            if (State.Pausing)
            {
                DrawPausingFramge(e, barDrawn);
            }
            else
            {
                var animation = target as AnimationDocument;
                DrawCurrentFrame(e, barDrawn, animation != null && animation.PreviewingAnimationSets.Any(x => animation.PauseFrames.ContainsKey(x)));
            }

            DrawVRulerBg(e);
            DrawVRulerAxis(e);
            DrawVRulerGrid(e);

            DrawRulerEdgeBg(e);

            DrawWarningBox(e, target);
        }

        // ディスプレイ座標系からカーブ座標系のｘ座標を求める
        public double GetXInCurvePlane(int displayX)
        {
            return (displayX - (CurveBgLeft + (CurveBgWidth  * 0.5) - State.ScrollX * State.ScaleX - VRulerBgWidth)) * State.ScaleX;
        }

        // ディスプレイ座標系からカーブ座標系のｙ座標を求める
        public double GetYInCurvePlane(int displayY)
        {
            return -(displayY - (CurveBgTop + (CurveBgHeight * 0.5) - State.ScrollY * State.ScaleY)) * State.ScaleY;
        }

        /// <summary>
        /// バックグラウンドの描画
        /// </summary>
        /// <param name="e"></param>
        /// <param name="target"></param>
        private void DrawCurveBg(PaintEventArgs e, GuiObject target)
        {
            e.Graphics.SafeFillRectangle(CurveEditorConst.CurveOuterAnimRangeBrush, CurveBgLeft, CurveBgTop, CurveBgWidth, CurveBgHeight);
            e.Graphics.SafeFillRectangle(CurveEditorConst.CurveInnerAnimRangeBrush, OriginXInCurveView, CurveBgTop, FrameCount / State.ScaleX + 0.5, CurveBgHeight);
        }

        /// <summary>
        /// 座標軸の描画
        /// </summary>
        /// <param name="e"></param>
        private void DrawCurveAxis(PaintEventArgs e)
        {
            // X軸
            if ((OriginYInCurveView > CurveBgTop) && (OriginYInCurveView < CurveBgBottom))
            {
                e.Graphics.DrawLine(CurveEditorConst.CurveAxisPen, CurveBgLeft, (int)OriginYInCurveView, CurveBgRight, (int)OriginYInCurveView);
            }

            // Y軸
            if ((OriginXInCurveView > CurveBgLeft) && (OriginXInCurveView < CurveBgRight))
            {
                e.Graphics.DrawLine(CurveEditorConst.CurveAxisPen, (int)OriginXInCurveView, CurveBgTop, (int)OriginXInCurveView, CurveBgBottom);
            }
        }

        /// <summary>
        /// グリッドの描画
        /// </summary>
        /// <param name="e"></param>
        private void DrawCurveGrid(PaintEventArgs e)
        {
            // X軸
            {
                double interval;
                double intervalViewSize;
                MakeGridInterval(State.ScaleY, out interval, out intervalViewSize);

                // 原点付近のグリッド
                {
                    for(var i = 0; i < 4; ++i)
                    {
                        var drawY = OriginYInCurveView;

                        {
                            var interDrawY = drawY + ((i + 1) * intervalViewSize / 5);
                            e.Graphics.SafeDrawLine(CurveEditorConst.CurveGridMinPen, CurveBgLeft, interDrawY, CurveBgRight, interDrawY);
                        }

                        {
                            var interDrawY = drawY - ((i + 1) * intervalViewSize / 5);
                            e.Graphics.SafeDrawLine(CurveEditorConst.CurveGridMinPen, CurveBgLeft, interDrawY, CurveBgRight, interDrawY);
                        }
                    }
                }

                // 原点より下のグリッド
                {
                    double y = OriginYInCurveView + intervalViewSize;
                    double index = -interval;

                    // 画面に近づける
                    if (y < 0)
                    {
                        double distance = Math.Truncate(Math.Abs(y) / intervalViewSize);

                        y     += intervalViewSize * distance;
                        index += interval         * distance;
                    }

                    for(;y < CurveBgBottom;y += intervalViewSize, index -= interval)
                    {
                        if (y > CurveBgTop)
                        {
                            var drawY = (int)y;
                            e.Graphics.DrawLine(CurveEditorConst.CurveGridPen, CurveBgLeft, drawY, CurveBgRight, drawY);
                        }

                        for(int i = 0;i != 4;++ i)
                        {
                            var drawY = (int)y;

                            {
                                int interDrawY = drawY + (int)((i + 1) * intervalViewSize / 5);
                                e.Graphics.DrawLine(CurveEditorConst.CurveGridMinPen, CurveBgLeft, interDrawY, CurveBgRight, interDrawY);
                            }
                        }
                    }
                }

                // 原点より上のグリッド
                {
                    double y = OriginYInCurveView - intervalViewSize;
                    double index = interval;

                    // 画面に近づける
                    if (y > CurveBgHeight)
                    {
                        double distance = Math.Truncate(Math.Abs(y - CurveBgHeight) / intervalViewSize);

                        y     -= intervalViewSize * distance;
                        index -= interval         * distance;
                    }

                    for(;y > CurveBgTop;y -= intervalViewSize, index += interval)
                    {
                        if (y < CurveBgBottom)
                        {
                            var drawY = (int)y;
                            e.Graphics.DrawLine(CurveEditorConst.CurveGridPen, CurveBgLeft, drawY, CurveBgRight, drawY);
                        }

                        for(int i = 0;i != 4;++ i)
                        {
                            var drawY = (int)y;

                            {
                                int interDrawY = drawY - (int)((i + 1) * intervalViewSize / 5);
                                e.Graphics.DrawLine(CurveEditorConst.CurveGridMinPen, CurveBgLeft, interDrawY, CurveBgRight, interDrawY);
                            }
                        }
                    }
                }
            }

            // Y軸
            {
                double interval;
                double intervalViewSize;
                MakeGridInterval(State.ScaleX, out interval, out intervalViewSize);

                // 原点付近のグリッド
                {
                    for(int i = 0;i != 4;++ i)
                    {
                        var drawX = OriginXInCurveView;

                        {
                            var interDrawX = drawX + ((i + 1) * intervalViewSize / 5);
                            e.Graphics.SafeDrawLine(CurveEditorConst.CurveGridMinPen, interDrawX, CurveBgTop, interDrawX, CurveBgBottom);
                        }

                        {
                            var interDrawX = drawX - ((i + 1) * intervalViewSize / 5);
                            e.Graphics.SafeDrawLine(CurveEditorConst.CurveGridMinPen, interDrawX, CurveBgTop, interDrawX, CurveBgBottom);
                        }
                    }
                }

                // 原点より右のグリッド
                {
                    double x = OriginXInCurveView + intervalViewSize;
                    double index = interval;

                    // 画面に近づける
                    if (x < 0)
                    {
                        double distance = Math.Truncate(Math.Abs(x) / intervalViewSize);

                        x     += intervalViewSize * distance;
                        index += interval         * distance;
                    }

                    for(;x < CurveBgRight;x += intervalViewSize, index += interval)
                    {
                        if (x > CurveBgLeft)
                        {
                            var drawX = (int)x;
                            e.Graphics.DrawLine(CurveEditorConst.CurveGridPen, drawX, CurveBgTop, drawX, CurveBgBottom);
                        }

                        for(int i = 0;i != 4;++ i)
                        {
                            var drawX = (int)x;

                            {
                                int interDrawX = drawX + (int)((i + 1) * intervalViewSize / 5);
                                e.Graphics.DrawLine(CurveEditorConst.CurveGridMinPen, interDrawX, CurveBgTop, interDrawX, CurveBgBottom);
                            }
                        }
                    }
                }

                // 原点より左のグリッド
                {
                    double x = OriginXInCurveView - intervalViewSize;
                    double index = -interval;

                    // 画面に近づける
                    if (x > CurveBgWidth)
                    {
                        double distance = Math.Truncate(Math.Abs(x - CurveBgWidth) / intervalViewSize);

                        x     -= intervalViewSize * distance;
                        index -= interval         * distance;
                    }

                    for(;x > CurveBgLeft;x -= intervalViewSize, index -= interval)
                    {
                        if (x < CurveBgRight)
                        {
                            var drawX = (int)x;
                            e.Graphics.DrawLine(CurveEditorConst.CurveGridPen, drawX, CurveBgTop, drawX, CurveBgBottom);
                        }

                        for(int i = 0;i != 4;++ i)
                        {
                            var drawX = (int)x;

                            {
                                int interDrawX = drawX - (int)((i + 1) * intervalViewSize / 5);
                                e.Graphics.DrawLine(CurveEditorConst.CurveGridMinPen, interDrawX, CurveBgTop, interDrawX, CurveBgBottom);
                            }
                        }
                    }
                }
            }
        }

        /// <summary>
        /// グリッドの間隔を計算する
        /// </summary>
        /// <param name="scale"></param>
        /// <param name="interval"></param>
        /// <param name="intervalViewSize"></param>
        /// <param name="decimalPlaces"></param>
        private static void MakeGridInterval(double scale, out double interval, out double intervalViewSize, uint decimalPlaces = 5)
        {
            interval = CurveEditorConst.GridUnitInterval;
            intervalViewSize = interval / scale;

            if (scale >= 1.0f)
            {
                var bound = scale * CurveEditorConst.GridUnitInterval;
                interval = Math.Pow(10, Math.Floor(Math.Log10(bound)));
                if (interval >= bound)
                {
                }
                else if (2 * interval >= bound)
                {
                    interval *= 2;
                }
                else if (5 * interval >= bound)
                {
                    interval *= 5;
                }
                else
                {
                    interval *= 10;
                }
                intervalViewSize = interval / scale;
            }
            else
            {
                var scales = new[] { 0.5, 0.2, 0.1 };
                var lastscale = 1.0;
                var scaled = false;
                for (var i = 0; i < decimalPlaces; i++)
                {
                    foreach (var s in scales)
                    {
                        var ts = s * Math.Pow(0.1, i);
                        if (scale > ts)
                        {
                            interval *= lastscale;
                            intervalViewSize *= lastscale;
                            scaled = true;
                            break;
                        }
                        lastscale = ts;
                    }
                    if (scaled) break;
                }

                if (!scaled)
                {
                    interval *= lastscale;
                    intervalViewSize *= lastscale;
                }
            }
        }

        private void DrawCurveKeySnap(PaintEventArgs e)
        {
            // X軸
            foreach (var value in view_.SnappedValues)
            {
                var y = MakeCurveViewScreenPosY(value);
                if ((y > CurveBgTop) && (y < CurveBgBottom))
                {
                    e.Graphics.DrawLine(CurveEditorConst.KeySnapLinePen, CurveBgLeft, (int)y, CurveBgRight, (int)y);
                }
            }

            // Y軸
            foreach (var frame in view_.SnappedFrames)
            {
                var x = MakeCurveViewScreenPosX(frame);
                if ((x > CurveBgLeft) && (x < CurveBgRight))
                    e.Graphics.DrawLine(CurveEditorConst.KeySnapLinePen, (int)x, CurveBgTop, (int)x, CurveBgBottom);
            }
            view_.SnappedFrames.Clear();
            view_.SnappedValues.Clear();
        }

        /// <summary>
        /// 選択範囲の描画
        /// </summary>
        /// <param name="e"></param>
        private void DrawSelection(PaintEventArgs e)
        {
            e.Graphics.SafeFillRectangle(CurveEditorConst.SelectionBrush,	State.SelectionX, State.SelectionY, State.SelectionW, State.SelectionH);
            e.Graphics.SafeDrawRectangle(CurveEditorConst.SelectionPen,		State.SelectionX, State.SelectionY, State.SelectionW, State.SelectionH);
        }

        /// <summary>
        /// 水平ルーラの背景の描画
        /// </summary>
        /// <param name="e"></param>
        private void DrawHRulerBg(PaintEventArgs e)
        {
            e.Graphics.SafeFillRectangle(CurveEditorConst.RulerInAnimFrameBrush,  HRulerBgLeft, HRulerBgTop, HRulerBgWidth, HRulerBgHeight);
            e.Graphics.SafeFillRectangle(CurveEditorConst.RulerOutAnimFrameBrush, OriginXInCurveView, HRulerBgTop, FrameCount / State.ScaleX + 0.5, HRulerBgHeight);

            // カーブとの境界線
            e.Graphics.DrawLine(CurveEditorConst.CurveBorderPen, HRulerBgLeft, HRulerBgTop, HRulerBgRight, HRulerBgTop);
        }

        /// <summary>
        /// 水平ルーラに座標軸を描画
        /// </summary>
        /// <param name="e"></param>
        private void DrawHRulerAxis(PaintEventArgs e)
        {
            e.Graphics.SafeDrawLine(CurveEditorConst.CurveAxisPen, OriginXInCurveView, HRulerBgTop, OriginXInCurveView, HRulerBgBottom);
        }

        /// <summary>
        /// 水平ルーラ（カラーバー）に座標軸を描画
        /// </summary>
        /// <param name="e"></param>
        private void DrawHBarAxis(PaintEventArgs e, int barSize)
        {
            e.Graphics.SafeDrawLine(CurveEditorConst.CurveAxisPen, OriginXInCurveView, HRulerBgTop, OriginXInCurveView, HRulerBgTop - barSize);
        }

        /// <summary>
        /// 水平ルーラにグリッドを描画
        /// </summary>
        /// <param name="e"></param>
        private void DrawHRulerGrid(PaintEventArgs e)
        {
            // Y軸
            {
                double interval;
                double intervalViewSize;
                MakeGridInterval(State.ScaleX, out interval, out intervalViewSize);

                int drawY = HRulerBgBottom - 16;

                {
                    var drawX = OriginXInCurveView;
                    double index = 0;

                    // 余裕を持ってクランプ
                    if (e.Graphics.ClipBounds.Left - 1000 < drawX && drawX < e.Graphics.ClipBounds.Right)
                    {
                        e.Graphics.DrawString(index.ToString(), TheApp.GuiFont, CurveEditorConst.CurveFontDarkBrush, (int)drawX, drawY);
                    }

                    for (int i = 0; i != 4; ++i)
                    {
                        {
                            var interDrawX = drawX + ((i + 1) * intervalViewSize / 5);
                            e.Graphics.SafeDrawLine(CurveEditorConst.RulerGridPen, interDrawX, HRulerBgTop + 1, interDrawX, HRulerBgTop + 10);
                        }

                        {
                            var interDrawX = drawX - ((i + 1) * intervalViewSize / 5);
                            e.Graphics.SafeDrawLine(CurveEditorConst.RulerGridPen, interDrawX, HRulerBgTop + 1, interDrawX, HRulerBgTop + 10);
                        }
                    }
                }

                {
                    double x = OriginXInCurveView + intervalViewSize;
                    double index = interval;

                    // 画面に近づける
                    if (x < -intervalViewSize)
                    {
                        double distance = Math.Truncate(Math.Abs(x + intervalViewSize) / intervalViewSize);

                        x += intervalViewSize * distance;
                        index += interval * distance;
                    }

                    for (; x < HRulerBgRight; x += intervalViewSize, index += interval)
                    {
                        var drawX = (int)x;

                        var str = index.ToString();//index < 1e6 ? ((int)index).ToString() : index.ToString();
                        e.Graphics.DrawString(str, TheApp.GuiFont, CurveEditorConst.CurveFontDarkBrush, drawX, drawY);

                        if (x > HRulerBgLeft)
                        {
                            e.Graphics.DrawLine(CurveEditorConst.CurveGridPen, drawX, HRulerBgTop + 1, drawX, HRulerBgBottom);
                        }

                        for (int i = 0; i != 4; ++i)
                        {
                            int interDrawX = drawX + (int)((i + 1) * intervalViewSize / 5);

                            e.Graphics.DrawLine(CurveEditorConst.RulerGridPen, interDrawX, HRulerBgTop + 1, interDrawX, HRulerBgTop + 10);
                        }
                    }
                }

                {
                    double x = OriginXInCurveView - intervalViewSize;
                    double index = -interval;

                    // 画面に近づける
                    if (x > (HRulerBgWidth + intervalViewSize))
                    {
                        double distance = Math.Truncate(Math.Abs(x - (HRulerBgWidth + intervalViewSize)) / intervalViewSize);

                        x -= intervalViewSize * distance;
                        index -= interval * distance;
                    }

                    for (; x > 0; x -= intervalViewSize, index -= interval)						// 縦ルーラーにかかっても描く
                    {
                        var drawX = (int)x;

                        e.Graphics.DrawString((index).ToString(), TheApp.GuiFont, CurveEditorConst.CurveFontDarkBrush, drawX, drawY);

                        if (x < HRulerBgRight)
                        {
                            e.Graphics.DrawLine(CurveEditorConst.CurveGridPen, drawX, HRulerBgTop + 1, drawX, HRulerBgBottom);
                        }

                        for (int i = 0; i != 4; ++i)
                        {
                            int interDrawX = drawX - (int)((i + 1) * intervalViewSize / 5);

                            e.Graphics.DrawLine(CurveEditorConst.RulerGridPen, interDrawX, HRulerBgTop + 1, interDrawX, HRulerBgTop + 10);
                        }
                    }
                }
            }
        }

        /// <summary>
        /// 現在のフレームを表すラインを描画
        /// </summary>
        /// <param name="e"></param>
        /// <param name="barDrawn"></param>
        /// <param name="duplicate"></param>
        private void DrawCurrentFrame(PaintEventArgs e, bool barDrawn, bool duplicate)
        {
            var x = (CurveBgWidth * 0.5) - (State.ScrollX * State.ScaleX) + (State.CurrentFrame / State.ScaleX);

            // 線
            e.Graphics.SafeDrawLine(duplicate ? CurveEditorConst.CurveCurrentFramePenNotPreviewed : CurveEditorConst.CurveCurrentFramePen, (int)x, CurveBgTop, (int)x, HRulerBgBottom);

            // 数値
            // 大雑把にクリップ
            if (e.Graphics.ClipBounds.Left - 1000 < x && x < e.Graphics.ClipBounds.Right + 1000)
            {
                int y = HRulerBgTop - 12;
                if (barDrawn)
                {
                    y -= CurveEditorConst.ColorBarSize + 1;
                }
                e.Graphics.DrawString(State.CurrentFrame.ToString(), TheApp.GuiFont, duplicate ? CurveEditorConst.CurveCurrentFrameBrushNotPreviewed: CurveEditorConst.CurveCurrentFrameBrush, (int)x, y);
            }
        }

        /// <summary>
        /// ポーズ中フレームを表すラインを描画
        /// </summary>
        /// <param name="e"></param>
        /// <param name="barDrawn"></param>
        private void DrawPausingFramge(PaintEventArgs e, bool barDrawn)
        {
            var x = (CurveBgWidth * 0.5) - (State.ScrollX * State.ScaleX) + (State.PauseFrame / State.ScaleX);

            // 線
            e.Graphics.SafeDrawLine(CurveEditorConst.CurvePausingFramePen, x, CurveBgTop, x, HRulerBgBottom);

            // 数値
            // 大雑把にクリップ
            if (e.Graphics.ClipBounds.Left - 1000 < x && x < e.Graphics.ClipBounds.Right + 1000)
            {
                int y = HRulerBgTop - 12;
                if (barDrawn)
                {
                    y -= CurveEditorConst.ColorBarSize + 1;
                }
                e.Graphics.DrawString(State.PauseFrame.ToString(), TheApp.GuiFont, CurveEditorConst.CurvePausingFrameBrush, (int)x, y);
            }
        }

        /// <summary>
        /// クランプ範囲を表すラインを描画
        /// </summary>
        /// <param name="e"></param>
        /// <param name="valueRange"></param>
        private void DrawClampLine(PaintEventArgs e, CurveView.ValueRange valueRange)
        {
            if (view_.IsClampValue || view_.IsClampFrame)
            {
                var x0 = MakeCurveViewScreenPosX(0.0f);
                var x1 = MakeCurveViewScreenPosX(FrameCount);
                var y0 = (valueRange.MaxValue != null) ? MakeCurveViewScreenPosY((float)valueRange.MaxValue) : CurveBgTop;
                var y1 = (valueRange.MinValue != null) ? MakeCurveViewScreenPosY((float)valueRange.MinValue) : CurveBgBottom;

                if (view_.IsClampFrame)
                {
                    if ((x0 > CurveBgLeft) && (x0 < CurveBgRight))
                    {
                        e.Graphics.SafeDrawLine(CurveEditorConst.CurveClampLinePen, x0, y0, x0, y1);
                    }

                    if ((x1 > CurveBgLeft) && (x1 < CurveBgRight))
                    {
                        e.Graphics.SafeDrawLine(CurveEditorConst.CurveClampLinePen, x1, y0, x1, y1);
                    }
                }

                if (view_.IsClampValue)
                {
                    if (valueRange.MinValue != null)
                    {
                        if ((y1 < CurveBgBottom) && (y1 > CurveBgTop))
                        {
                            e.Graphics.SafeDrawLine(CurveEditorConst.CurveClampLinePen, x0, y1, x1, y1);
                        }
                    }

                    if (valueRange.MaxValue != null)
                    {
                        if ((y0 < CurveBgBottom) && (y0 > CurveBgTop))
                        {
                            e.Graphics.SafeDrawLine(CurveEditorConst.CurveClampLinePen, x0, y0, x1, y0);
                        }
                    }
                }
            }
        }

        /// <summary>
        /// 値域を表す背景・ラインを描画
        /// </summary>
        /// <param name="e"></param>
        /// <param name="target"></param>
        /// <param name="valueRange"></param>
        /// <param name="flags"></param>
        private void DrawValueRange(PaintEventArgs e, GuiObject target, CurveView.ValueRange valueRange, RangeDrawFlag flags)
        {
            var x0 = CurveBgLeft;
            var x1 = CurveBgRight;

            // BG
            if (flags.HasFlag(RangeDrawFlag.BackGround))
            {
                if (valueRange.MinValue != null)
                {
                    var y0 = MakeCurveViewScreenPosY((float)valueRange.MinValue);
                    var y1 = CurveBgBottom;
                    var w = x1 - x0;
                    var h = y1 - y0;

                    e.Graphics.SafeFillRectangle(CurveEditorConst.CurveValueRangeBrush, x0, y0, w, h);
                }

                if (valueRange.MaxValue != null)
                {
                    var y0 = CurveBgTop;
                    var y1 = MakeCurveViewScreenPosY((float)valueRange.MaxValue);
                    var w = x1 - x0;
                    var h = y1 - y0;

                    e.Graphics.SafeFillRectangle(CurveEditorConst.CurveValueRangeBrush, x0, y0, w, h);
                }
            }

            // 線
            if (flags.HasFlag(RangeDrawFlag.Line))
            {
                if (valueRange.MinValue != null)
                {
                    var y  = MakeCurveViewScreenPosY((float)valueRange.MinValue);

                    e.Graphics.SafeDrawLine(CurveEditorConst.CurveValueRangePen, x0, y, x1, y);
                }

                if (valueRange.MaxValue != null)
                {
                    var y  = MakeCurveViewScreenPosY((float)valueRange.MaxValue);

                    e.Graphics.SafeDrawLine(CurveEditorConst.CurveValueRangePen, x0, y, x1, y);
                }
            }
        }


        /// <summary>
        /// 垂直ルーラの背景の描画
        /// </summary>
        /// <param name="e"></param>
        private void DrawVRulerBg(PaintEventArgs e)
        {
            e.Graphics.FillRectangle(CurveEditorConst.RulerInAnimFrameBrush, VRulerBgLeft, VRulerBgTop, VRulerBgWidth, VRulerBgHeight);

            // カーブとの境界線
            e.Graphics.DrawLine(CurveEditorConst.CurveBorderPen, VRulerBgRight, VRulerBgTop, VRulerBgRight, VRulerBgBottom);
        }

        /// <summary>
        /// 垂直ルーラに座標軸を描画
        /// </summary>
        /// <param name="e"></param>
        private void DrawVRulerAxis(PaintEventArgs e)
        {
            e.Graphics.SafeDrawLine(CurveEditorConst.CurveAxisPen, VRulerBgLeft, OriginYInCurveView, VRulerBgRight, OriginYInCurveView);
        }

        /// <summary>
        /// 垂直ルーラにグリッドを描画
        /// </summary>
        /// <param name="e"></param>
        private void DrawVRulerGrid(PaintEventArgs e)
        {
            // X軸
            {
                double interval;
                double intervalViewSize;
                MakeGridInterval(State.ScaleY, out interval, out intervalViewSize);

                int drawX = VRulerBgLeft;

                {
                    var drawY = OriginYInCurveView;
                    double index = 0;

                    // 余裕を持ってクランプ
                    if (e.Graphics.ClipBounds.Top - 1000 < drawY && drawY < e.Graphics.ClipBounds.Bottom + 1000)
                    {
                        DrawHString(e.Graphics, index.ToString(), drawX, (int)drawY);
                    }

                    for(int i = 0;i != 4;++ i)
                    {
                        {
                            var interDrawY = drawY + ((i + 1) * intervalViewSize / 5);
                            e.Graphics.SafeDrawLine(CurveEditorConst.RulerGridPen, VRulerBgRight - 1, interDrawY, VRulerBgRight - 10, interDrawY);
                        }

                        {
                            var interDrawY = drawY - ((i + 1) * intervalViewSize / 5);
                            e.Graphics.SafeDrawLine(CurveEditorConst.RulerGridPen, VRulerBgRight - 1, interDrawY, VRulerBgRight - 10, interDrawY);
                        }
                    }
                }

                {
                    double y = OriginYInCurveView + intervalViewSize;
                    double index = -interval;

                    // 画面に近づける
                    if (y < -intervalViewSize)
                    {
                        double distance = Math.Truncate(Math.Abs(y + intervalViewSize) / intervalViewSize);

                        y     += intervalViewSize * distance;
                        index -= interval         * distance;
                    }

                    for(;y < (VRulerBgBottom + intervalViewSize);y += intervalViewSize, index -= interval)
                    {
                        var drawY = (int)y;
                        DrawHString(e.Graphics, (index).ToString(), drawX, drawY);

                        if (y > VRulerBgTop)
                        {
                            e.Graphics.DrawLine(CurveEditorConst.CurveGridPen, VRulerBgLeft, drawY, VRulerBgRight - 1, drawY);
                        }

                        for(int i = 0;i != 4;++ i)
                        {
                            int interDrawY = drawY + (int)((i + 1) * intervalViewSize / 5);

                            e.Graphics.DrawLine(CurveEditorConst.RulerGridPen, VRulerBgRight - 1, interDrawY, VRulerBgRight - 10, interDrawY);
                        }
                    }
                }

                {
                    double y = OriginYInCurveView - intervalViewSize;
                    double index = interval;

                    // 画面に近づける
                    if (y > (VRulerBgHeight + intervalViewSize))
                    {
                        double distance = Math.Truncate(Math.Abs(y - (VRulerBgHeight + intervalViewSize)) / intervalViewSize);

                        y     -= intervalViewSize * distance;
                        index += interval         * distance;
                    }

                    for(;y > (VRulerBgTop - intervalViewSize);y -= intervalViewSize, index += interval)
                    {
                        var drawY = (int)y;
                        var str = index < 1e6 ? (index).ToString() : index.ToString();
                        DrawHString(e.Graphics, str, drawX, drawY);

                        if (y < VRulerBgBottom)
                        {
                            e.Graphics.DrawLine(CurveEditorConst.CurveGridPen, VRulerBgLeft, drawY, VRulerBgRight - 1, drawY);
                        }

                        for(int i = 0;i != 4;++ i)
                        {
                            int interDrawY = drawY - (int)((i + 1) * intervalViewSize / 5);

                            e.Graphics.DrawLine(CurveEditorConst.RulerGridPen, VRulerBgRight - 1, interDrawY, VRulerBgRight - 10, interDrawY);
                        }
                    }
                }
            }
        }

        /// <summary>
        /// ルーラ交差点の描画
        /// </summary>
        /// <param name="e"></param>
        private void DrawRulerEdgeBg(PaintEventArgs e)
        {
            e.Graphics.FillRectangle(CurveEditorConst.RulerInAnimFrameBrush, VRulerBgLeft, VRulerBgBottom, VRulerBgWidth, HRulerBgHeight);
            e.Graphics.DrawRectangle(CurveEditorConst.CurveBorderPen, VRulerBgLeft - 1, VRulerBgBottom, VRulerBgWidth + 1, HRulerBgHeight);
        }

        private static readonly Font WarningBoxFont	= new Font("Tahoma", 18, FontStyle.Regular, GraphicsUnit.Point, 0);

        /// <summary>
        /// カーブに対する警告を表示
        /// </summary>
        /// <param name="e"></param>
        /// <param name="target"></param>
        private void DrawWarningBox(PaintEventArgs e, GuiObject target)
        {
            if (target.ObjectID == GuiObjectID.SkeletalAnimation)
            {
                DrawWarningBoxString(e, Strings.NonEditableKind_SkeletalAnimation_NotEditable);
                return;
            }
            if (target.ObjectID == GuiObjectID.ShapeAnimation)
            {
                DrawWarningBoxString(e, Strings.NonEditableKind_ShapeAnimation_NotEditable);
                return;
            }

            // http://www-sdd.zelda.nintendo.co.jp/project/nintendoware3/kagemai/html/user.cgi?project=nw3_3de&action=view_report&id=1278
            //・カーブが複数選択されているときは、その中に一つでも編集可能カーブが
            //  含まれる場合にはメッセージは表示しない。
            if (SelectedCurves.Any(x => x.IsEditable) == false)
            {
                var message = MakeWarningBoxMessageString();
                if (message != null)
                {
                    DrawWarningBoxString(e, message);
                    return;
                }
            }

            if (target.ObjectID != GuiObjectID.ShaderParameterAnimation) return;


            // http://www-sdd.zelda.nintendo.co.jp/project/nintendoware3/kagemai/html/user.cgi?project=nw3_3de&action=view_report&id=1278#5
            //	モデルにバインドされていない時と、シェーダー定義が読み込まれていない時について、
            //	キーが全くついていないときはカーブエディタを開いた段階でメッセージの表示を行うよう

            // キーフレームがあるかどうか？
            var hasKeyFrame = view_.ParentPanel.AllCurves.Any(x => x.KeyFrames.Any());

            if (hasKeyFrame == false)
            {
                // シェーダー定義が読み込まれていない時
                if (DocumentManager.ShaderDefinitions.Any() == false)
                {
                    DrawWarningBoxString(e, UIText.EnumValue(AnimationDocument.NonEditableKind.ShaderParamAnim_NotLoadShaderDef));
                }
                else
                    // モデルにバインドされていない時
                    if (DocumentManager.Models.SelectMany(x => x.AllAnimations).Any(x => x == target) == false)
                    {
                        DrawWarningBoxString(e, UIText.EnumValue(AnimationDocument.NonEditableKind.ShaderParamAnim_NotBindModel));
                    }
            }
        }

        /// <summary>
        /// 警告メッセージの作成
        /// </summary>
        /// <returns></returns>
        private string MakeWarningBoxMessageString()
        {
            var curve = SelectedCurves.FirstOrDefault(x => x.NonEditableKind != AnimationDocument.NonEditableKind.Editable);
            if (curve != null)
            {
                if (curve.NonEditableKind != AnimationDocument.NonEditableKind.Editable)
                {
                    return UIText.EnumValue(curve.NonEditableKind, curve.NonEditableKindDisplayAux);
                }
            }

            return null;
        }

        /// <summary>
        /// 警告メッセージの描画
        /// </summary>
        /// <param name="e"></param>
        /// <param name="message"></param>
        private void DrawWarningBoxString(PaintEventArgs e, string message)
        {
            const int padding = 4;

            var messageSize	= e.Graphics.MeasureString(message, WarningBoxFont, 10000);
            var boxWidth	= messageSize.Width  + padding * 2;
            var boxHeight	= messageSize.Height + padding * 2;
            var boxX = (CurveViewWidth - boxWidth) ;
            var boxY = (CurveViewHeight - HRulerBgHeight - boxHeight) ;

            using(var brush = new SolidBrush(Color.FromArgb(0x60, 0x00, 0x00, 0x00)))
            {
                e.Graphics.FillRectangle(brush, boxX, boxY, boxWidth, boxHeight);
            }

            e.Graphics.DrawString(message, WarningBoxFont, Brushes.White, boxX + padding, boxY + padding);
        }



        /// <summary>
        /// ルーラ用文字描画（注意:DrawVString 対応した関数です。）
        /// </summary>
        /// <param name="graphics"></param>
        /// <param name="s"></param>
        /// <param name="x"></param>
        /// <param name="y"></param>
        private static void DrawHString(Graphics graphics, string s, int x, int y)
        {
            using(var sf = new StringFormat())
            {
                sf.SetMeasurableCharacterRanges(new[]{ new CharacterRange(0, s.Length - 1) });

                var stringRegions = graphics.MeasureCharacterRanges(s, TheApp.GuiFont, graphics.ClipBounds, sf);
                var offset        = CurveEditorConst.RulerSizeV - stringRegions[0].GetBounds(graphics).Width - 10;

                graphics.DrawString(s, TheApp.GuiFont, CurveEditorConst.CurveFontDarkBrush, Math.Max(x + offset, 0), y - 13);
            }
        }

        #region 編集バー関係
        private class EditBarInterval
        {
            public double LeftX{ get; set; }
            public double RightX{ get; set; }
            public double Width { get{ return RightX - LeftX; } }

            public float Value{ get; set; }
        }

        private IEnumerable<EditBarInterval> MakeEditBarIntervals(IAnimationCurve curve, GuiObject target)
        {
            var animTarget = curve.GetAnimTarget(target);

            var intervals = new List<EditBarInterval>();
            {
                var keyFrameCount	= curve.KeyFrames.Count;

                if (keyFrameCount == 1)
                {
                    var keyFrame = curve.KeyFrames.First();

                    intervals.Add(
                        new EditBarInterval
                        {
                            LeftX = HRulerBgLeft,
                            RightX = HRulerBgRight,
                            Value = keyFrame.Value,
                        }
                    );
                }
                else
                if (keyFrameCount >= 2)
                {
                    for(int i = 0;i != keyFrameCount - 1;++ i)
                    {
                        var left  = curve.KeyFrames[i + 0];
                        var right = curve.KeyFrames[i + 1];

                        intervals.Add(
                            new EditBarInterval
                            {
                                LeftX	= MakeCurveViewScreenPosX(left.Frame),
                                RightX	= MakeCurveViewScreenPosX(right.Frame),
                                Value	= left.Value
                            }
                        );
                    }

                    // 最後
                    {
                        var last = curve.KeyFrames.Last();

                        intervals.Add(
                            new EditBarInterval
                            {
                                LeftX	= MakeCurveViewScreenPosX(last.Frame),
                                RightX	= MakeCurveViewScreenPosX(last.Frame),
                                Value	= last.Value
                            }
                        );
                    }

                    // プレ・ポストラップ部分を作る
                    {
                        var preIntervals  = new List<EditBarInterval>();
                        {
                            var width  = intervals.Last().RightX - intervals.First().LeftX;
                            var offset = -width;

                            switch(animTarget.pre_wrap)
                            {
                                case curve_wrapType.clamp:
                                {
                                    preIntervals.Add(
                                        new EditBarInterval
                                        {
                                            LeftX	= HRulerBgLeft,
                                            RightX	= intervals.First().LeftX,
                                            Value	= intervals.First().Value,
                                        }
                                    );

                                    break;
                                }

                                case curve_wrapType.mirror:
                                {
                                    // 同じフレームに複数キーが打たれている場合。
                                    if (width == 0)
                                    {
                                        break;
                                    }
                                    bool isFlip = true;
                                    while(
                                        (preIntervals.Any() == false) ||
                                        (preIntervals.Last().LeftX > HRulerBgLeft))
                                    {
                                        if (isFlip)
                                        {
                                            var temp = new List<EditBarInterval>();
                                            {
                                                foreach(var i in intervals)
                                                {
                                                    temp.Add(
                                                        new EditBarInterval
                                                        {
                                                            LeftX	= offset + i.LeftX,
                                                            RightX	= offset + i.RightX,
                                                            Value	= i.Value,
                                                        }
                                                    );
                                                }
                                            }

                                            var first = temp.First();
                                            foreach(var t in temp)
                                            {
                                                var left  = (width - (t.LeftX  - first.LeftX)) + first.LeftX;
                                                var right = (width - (t.RightX - first.LeftX)) + first.LeftX;

                                                // 反転しているので左右を入れ替える
                                                preIntervals.Add(
                                                    new EditBarInterval
                                                    {
                                                        LeftX	= right,
                                                        RightX	= left,
                                                        Value	= t.Value,
                                                    }
                                                );
                                            }
                                        }
                                        else
                                        {
                                            foreach(var i in intervals)
                                            {
                                                preIntervals.Add(
                                                    new EditBarInterval
                                                    {
                                                        LeftX	= offset + i.LeftX,
                                                        RightX	= offset + i.RightX,
                                                        Value	= i.Value,
                                                    }
                                                );
                                            }
                                        }

                                        offset -= width;
                                        isFlip = ! isFlip;
                                    }

                                    break;
                                }
                                case curve_wrapType.relative_repeat:
                                {
                                    // 同じフレームに複数キーが打たれている場合。
                                    if (width == 0)
                                    {
                                        break;
                                    }

                                    while (
                                        (preIntervals.Any() == false) ||
                                        (preIntervals.Last().LeftX > HRulerBgLeft))
                                    {
                                        var valueOffset =
                                            preIntervals.Any() ? (preIntervals[preIntervals.Count - intervals.Count].Value - intervals.LastOrDefault().Value) :
                                            (intervals.FirstOrDefault().Value - intervals.LastOrDefault().Value);
                                        foreach (var i in intervals)
                                        {
                                            preIntervals.Add(
                                                new EditBarInterval
                                                {
                                                    LeftX = offset + i.LeftX,
                                                    RightX = offset + i.RightX,
                                                    Value = i.Value + valueOffset
                                                }
                                            );
                                        }

                                        offset -= width;
                                    }

                                    // 相対値なので、追加した値が範囲内に収まらないこともあるが、値のクランプは行わない。
                                    // relative_repeat 以外のケースでもクランプしていないため。

                                    break;
                                }
                                case curve_wrapType.repeat:
                                {
                                    // 同じフレームに複数キーが打たれている場合。
                                    if (width == 0)
                                    {
                                        break;
                                    }
                                    while (
                                        (preIntervals.Any() == false) ||
                                        (preIntervals.Last().LeftX > HRulerBgLeft))
                                    {
                                        foreach(var i in intervals)
                                        {
                                            preIntervals.Add(
                                                new EditBarInterval
                                                {
                                                    LeftX	= offset + i.LeftX,
                                                    RightX	= offset + i.RightX,
                                                    Value	= i.Value,
                                                }
                                            );
                                        }

                                        offset -= width;
                                    }

                                    break;
                                }
                            }
                        }

                        var postIntervals = new List<EditBarInterval>();
                        {
                            var width  = intervals.Last().RightX - intervals.First().LeftX;
                            var offset = width;

                            switch(animTarget.post_wrap)
                            {
                                case curve_wrapType.clamp:
                                {
                                    postIntervals.Add(
                                        new EditBarInterval
                                        {
                                            LeftX	= intervals.Last().RightX,
                                            RightX	= HRulerBgRight,
                                            Value	= intervals.Last().Value,
                                        }
                                    );

                                    break;
                                }

                                case curve_wrapType.mirror:
                                {
                                    // 同じフレームに複数キーが打たれている場合。
                                    if (width == 0)
                                    {
                                        break;
                                    }

                                    bool isFlip = true;
                                    while(
                                        (postIntervals.Any() == false) ||
                                        (postIntervals.Last().RightX < HRulerBgRight))
                                    {
                                        if (isFlip)
                                        {
                                            var temp = new List<EditBarInterval>();
                                            {
                                                foreach(var i in intervals)
                                                {
                                                    temp.Add(
                                                        new EditBarInterval()
                                                        {
                                                            LeftX	= offset + i.LeftX,
                                                            RightX	= offset + i.RightX,
                                                            Value	= i.Value,
                                                        }
                                                    );
                                                }
                                            }

                                            var first = temp.First();
                                            foreach(var t in temp)
                                            {
                                                var left  = (width - (t.LeftX  - first.LeftX)) + first.LeftX;
                                                var right = (width - (t.RightX - first.LeftX)) + first.LeftX;

                                                // 反転しているので左右を入れ替える
                                                postIntervals.Add(
                                                    new EditBarInterval
                                                    {
                                                        LeftX	= right,
                                                        RightX	= left,
                                                        Value	= t.Value,
                                                    }
                                                );
                                            }
                                        }
                                        else
                                        {
                                            foreach(var i in intervals)
                                            {
                                                postIntervals.Add(
                                                    new EditBarInterval
                                                    {
                                                        LeftX	= offset + i.LeftX,
                                                        RightX	= offset + i.RightX,
                                                        Value	= i.Value,
                                                    }
                                                );
                                            }
                                        }

                                        offset += width;
                                        isFlip = ! isFlip;
                                    }

                                    break;
                                }
                                case curve_wrapType.relative_repeat:
                                {
                                    // 同じフレームに複数キーが打たれている場合。
                                    if (width == 0)
                                    {
                                        break;
                                    }

                                    while (
                                        (postIntervals.Any() == false) ||
                                        (postIntervals.Last().RightX < HRulerBgRight))
                                    {
                                        var valueOffset =
                                            postIntervals.Any() ? (postIntervals.Last().Value - intervals.FirstOrDefault().Value) :
                                            (intervals.LastOrDefault().Value - intervals.FirstOrDefault().Value);
                                        foreach (var i in intervals)
                                        {
                                            postIntervals.Add(
                                                new EditBarInterval
                                                {
                                                    LeftX = offset + i.LeftX,
                                                    RightX = offset + i.RightX,
                                                    Value = i.Value + valueOffset
                                                }
                                            );
                                        }

                                        offset += width;
                                    }

                                    // 相対値なので、追加した値が範囲内に収まらないこともあるが、値のクランプは行わない。
                                    // relative_repeat 以外のケースでもクランプしていないため。

                                    break;
                                }
                                case curve_wrapType.repeat:
                                {
                                    // 同じフレームに複数キーが打たれている場合。
                                    if (width == 0)
                                    {
                                        break;
                                    }

                                    while (
                                        (postIntervals.Any() == false) ||
                                        (postIntervals.Last().RightX < HRulerBgRight))
                                    {
                                        foreach(var i in intervals)
                                        {
                                            postIntervals.Add(
                                                new EditBarInterval
                                                {
                                                    LeftX	= offset + i.LeftX,
                                                    RightX	= offset + i.RightX,
                                                    Value	= i.Value,
                                                }
                                            );
                                        }

                                        offset += width;
                                    }

                                    break;
                                }
                            }
                        }

                        // 作ったプレ・ポストラップ情報を追加して位置でソートしておく
                        intervals.AddRange(preIntervals);
                        intervals.AddRange(postIntervals);
                        intervals.Sort((x, y) => x.LeftX.CompareTo(y.LeftX));
                    }
                }
            }

            intervals = intervals.Where(x => x.Width > 0.0f).ToList();

            return intervals;
        }

        private void DrawEmptyBar(PaintEventArgs e)
        {
            var rect = new Rectangle(CurveBgLeft, HRulerBgTop + 1 - 1 - CurveEditorConst.ColorBarSize, CurveBgWidth, CurveEditorConst.ColorBarSize);

            using (var brush = new HatchBrush(HatchStyle.DiagonalCross, Color.Gray, Color.FromArgb(0,0,0,0)))
            {
                e.Graphics.FillRectangle(brush, rect);
            }

            e.Graphics.DrawRectangle(Pens.Gray, rect);
        }

        private void DrawHDRBar(PaintEventArgs e)
        {
            var rect = new Rectangle(CurveBgLeft, HRulerBgTop + 1 - 1 - CurveEditorConst.ColorBarSize, CurveBgWidth, CurveEditorConst.ColorBarSize);

            using (var brush = new HatchBrush(HatchStyle.Percent10, Color.Gray, Color.FromArgb(0, 0, 0, 0)))
            {
                e.Graphics.FillRectangle(brush, rect);
            }

            e.Graphics.DrawRectangle(Pens.Gray, rect);
        }
#endregion
    }
}
