﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Windows.Forms;

using App.Data;
using App.Win32;

namespace App.PropertyEdit
{
    public class DragDetailBase
    {
    };

    public interface DragableInterface
    {
        void Dragging(DragDetailBase detail, int moveDiffX, int moveDiffY, int moveClipDiffX, int moveClipDiffY, InputControlBase inputControl);
        void DragEnd(DragDetailBase detail, InputControlBase inputControl);
        void DragCancel(DragDetailBase detail, InputControlBase inputControl);
    }

    public class DragableObject
    {
        public DragableObject(DragableInterface dragable, DragDetailBase detail)
        {
            Interface	= dragable;
            Detail		= detail;
        }

        public DragableInterface	Interface{	get; set;	}
        public DragDetailBase		Detail{		get; set;	}
    }

    public class InputControlBase
    {
        public class DragBeginInfo
        {
            public int			BeginX{			get; set;	}
            public int			BeginY{			get; set;	}
            public MouseButtons	Buttons{		get; set;	}
            public bool			IsPushSpace{	get; set;	}
            public Keys			ModifierKeys{	get; set;	}
        };

        public class DragMoveInfo
        {
            public int	MoveX{		get; set;	}
            public int	MoveY{		get; set;	}
            public int	OldX{		get; set;	}
            public int	OldY{		get; set;	}
            public bool	IsClipX{	get; set;	}
            public bool	IsClipY{	get; set;	}

            public bool	IsMoved{ get { return (MoveX != OldX) || (MoveY != OldY); } }
        };

        public bool				IsDragging{			get; set;			}
        public bool				IsInMouse{			get; private set;	}

        public DragBeginInfo	DragBegin{			get; private set;	}
        public DragMoveInfo		DragMove{			get; private set;	}

        public int				MoveFrame{			get; private set;	}
        public int				DragFrame{			get; private set;	}
        public bool ConstraintEnabled { get; set; }
        public bool IsDragged { get; set; }

        public MouseEventArgs	DownArgs{			get; private set;	}
        public MouseEventArgs	MoveArgs{			get; private set;	}
        public MouseEventArgs	UpArgs{				get; private set;	}
        public EventArgs		EnterArgs{			get; private set;	}
        public EventArgs		LeaveArgs{			get; private set;	}
        public MouseEventArgs	WheelArgs{			get; private set;	}
        public EventArgs		ClickArgs{			get; private set;	}
        public EventArgs		DoubleClickArgs{	get; private set;	}

        public List<DragableObject> DraggableObjects { get; private set; }

        public delegate void MouseDownEventHandler(MouseEventArgs e);
        public delegate void MouseDragEventHandler(MouseEventArgs e, int moveDiffX, int moveDiffY, int moveClipDiffX, int moveClipDiffY);
        public delegate void MouseMoveEventHandler(MouseEventArgs e);
        public delegate void MouseUpEventHandler(MouseEventArgs e);
        public delegate void MouseEnterEventHandler(EventArgs e);
        public delegate void MouseLeaveEventHandler(EventArgs e);
        public delegate void MouseWheelEventHandler(MouseEventArgs e);
        public delegate void MouseClickEventHandler(EventArgs e);
        public delegate void MouseDoubleClickEventHandler(EventArgs e);
        public delegate void LostFocusEventHandler(EventArgs e);

        public event MouseDownEventHandler			MouseDown;
        public event MouseDragEventHandler			MouseDrag;
        public event MouseMoveEventHandler			MouseMove;
        public event MouseUpEventHandler			MouseUp;
        public event MouseEnterEventHandler			MouseEnter;
        public event MouseLeaveEventHandler			MouseLeave;
        public event MouseWheelEventHandler			MouseWheel;
        public event MouseClickEventHandler			MouseClick;
        public event MouseDoubleClickEventHandler	MouseDoubleClick;
        public event LostFocusEventHandler LostFocus;

        public enum MoveDirType
        {
            Horizontality,
            Verticality,
            Central,
        }

        public MoveDirType MoveDir{ get; private set; }

        public InputControlBase(Panel panel)
        {
            DragBegin	= new DragBeginInfo();
            DragMove	= new DragMoveInfo();

            DraggableObjects	= new List<DragableObject>();

            panel.MouseDown		+= OnMouseDownEvent;
            panel.MouseMove		+= OnMouseMoveEvent;
            panel.MouseUp		+= OnMouseUpEvent;
            panel.MouseEnter	+= OnMouseEnterEvent;
            panel.MouseLeave	+= OnMouseLeaveEvent;
            panel.MouseWheel	+= OnMouseWheelEvent;
            panel.Click			+= OnMouseClickEvent;
            panel.DoubleClick	+= OnMouseDoubleClickEvent;
            panel.LostFocus += OnLostFocus;
        }

        public void AddDragObject(DragableInterface di)
        {
            AddDragObject(di, new DragDetailBase());
        }

        public void AddDragObject(DragableInterface di, DragDetailBase detail)
        {
            Debug.Assert(di		!= null);
            Debug.Assert(detail	!= null);

            DraggableObjects.Add(new DragableObject(di, detail));
        }

        protected void OnMouseDownEvent(object sender, MouseEventArgs e)
        {
            if (IsDragging == false)
            {
                ((Panel)sender).Focus();

                DownArgs = e;

                {
                    IsDragging = true;

                    DragBegin.BeginX		= e.X;
                    DragBegin.BeginY		= e.Y;
                    DragBegin.Buttons		= e.Button;
                    DragBegin.IsPushSpace	= (NativeMethods.GetAsyncKeyState(' ') & 0x8000) == 0x8000;
                    DragBegin.ModifierKeys	= Control.ModifierKeys;

                    DragMove.MoveX		= e.X;
                    DragMove.MoveY		= e.Y;
                    DragMove.OldX		= e.X;
                    DragMove.OldY		= e.Y;
                    DragMove.IsClipX	= false;
                    DragMove.IsClipY	= false;

                    MoveFrame = 0;
                    DragFrame = 0;
                    IsDragged = false;
                    ConstraintEnabled = true;

                    MoveDir = MoveDirType.Central;

                    if(MouseDown != null)
                    {
                        MouseDown(e);
                    }
                }
            }
        }

        protected void OnMouseMoveEvent(object sender, MouseEventArgs e)
        {
            MoveArgs = e;

            DragMove.MoveX	= e.X;
            DragMove.MoveY	= e.Y;

            Debug.Assert(e.Button != MouseButtons.None || !IsDragging);

            if(IsDragging)
            {
                int moveDiffX		= DragBegin.BeginX - e.X;
                int moveDiffY		= DragBegin.BeginY - e.Y;
                int moveClipDiffX	= moveDiffX;
                int moveClipDiffY	= moveDiffY;

                if((moveDiffX != 0) || (moveDiffY != 0))
                {
                    ++ DragFrame;
                    IsDragged = true;

                    if (DragFrame == 1 && ConstraintEnabled)
                    {
                        if ((DragBegin.ModifierKeys & Keys.Shift) != 0)
                        {
                            if (Math.Abs(moveDiffX) > Math.Abs(moveDiffY))
                            {
                                DragMove.IsClipX = false;
                                DragMove.IsClipY = true;
                            }
                            else
                            {
                                DragMove.IsClipX = true;
                                DragMove.IsClipY = false;
                            }
                        }
                    }

                    if (DragFrame == 2)
                    {
                        MoveDir = (Math.Abs(moveDiffX) > Math.Abs(moveDiffY)) ? MoveDirType.Horizontality : MoveDirType.Verticality;
                    }

                    if (DragMove.IsClipX)
                    {
                        moveClipDiffX = 0;
                    }

                    if (DragMove.IsClipY)
                    {
                        moveClipDiffY = 0;
                    }
                }

                foreach(var obj in DraggableObjects)
                {
                    obj.Interface.Dragging(obj.Detail, moveDiffX, moveDiffY, moveClipDiffX, moveClipDiffY, this);
                }

                if(MouseDrag != null)
                {
                    MouseDrag(e, moveDiffX, moveDiffY, moveClipDiffX, moveClipDiffY);
                }
            }
            else
            {
                ++ MoveFrame;

                if(MouseMove != null)
                {
                    MouseMove(e);
                }
            }

            DragMove.OldX = e.X;
            DragMove.OldY = e.Y;
        }

        internal IEnumerable<KeyFrame> DraggingKeys
        {
            get
            {
                return DraggableObjects.Select(x => x.Interface)
                        .OfType<CurveViewState.KeyFrameHolder>()
                        .Select(holder => holder.KeyFrame);
            }
        }

        internal IEnumerable<IAnimationCurve> DraggingCurves
        {
            get
            {
                return DraggableObjects.Select(x => x.Interface)
                        .OfType<CurveViewState.KeyFrameHolder>()
                        .Select(holder => holder.Curve);
            }
        }
        internal IEnumerable<KeyFrame> DraggingCurvesKeys
        {
            get
            {
                return DraggableObjects.Select(x => x.Interface)
                        .OfType<CurveViewState.KeyFrameHolder>()
                        .Select(holder => holder.Curve)
                        .SelectMany(x => x.KeyFrames);
            }
        }


        protected void OnMouseUpEvent(object sender, MouseEventArgs e)
        {
            UpArgs = e;

            if(IsDragging)
            {
                IsDragging = false;

                foreach(DragableObject o in DraggableObjects)
                {
                    o.Interface.DragEnd(o.Detail, this);
                }

                if(MouseUp != null)
                {
                    MouseUp(e);
                }

                DraggableObjects.Clear();
            }
        }

        protected void OnLostFocus(object sender, EventArgs e)
        {
            if (IsDragging)
            {
                IsDragging = false;

                foreach (DragableObject o in DraggableObjects)
                {
                    o.Interface.DragCancel(o.Detail, this);
                }

                if (LostFocus != null)
                {
                    LostFocus(e);
                }

                DraggableObjects.Clear();
            }
        }

        protected void OnMouseEnterEvent(object sender, EventArgs e)
        {
            EnterArgs = e;

            if(IsInMouse == false)
            {
                IsInMouse = true;

                if(MouseEnter != null)
                {
                    MouseEnter(e);
                }
            }
        }

        protected void OnMouseLeaveEvent(object sender, EventArgs e)
        {
            LeaveArgs = e;

            if(IsInMouse)
            {
                IsInMouse = false;

                if(MouseLeave != null)
                {
                    MouseLeave(e);
                }
            }
        }

        protected void OnMouseWheelEvent(object sender, MouseEventArgs e)
        {
            // 中ボタンドラック中は処理しない
            if (IsDragging && DragBegin.Buttons == MouseButtons.Middle)
            {
                ;
            }
            else
            {
                WheelArgs = e;

                if(IsInMouse)
                {
                    if(MouseWheel != null)
                    {
                        MouseWheel(e);
                    }
                }
            }
        }

        protected void OnMouseClickEvent(object sender, EventArgs e)
        {
            ClickArgs = e;

            if(IsInMouse)
            {
                if(MouseClick != null)
                {
                    MouseClick(e);
                }
            }
        }

        protected void OnMouseDoubleClickEvent(object sender, EventArgs e)
        {
            DoubleClickArgs = e;

            if(IsInMouse)
            {
                OnMouseUpEvent(sender, DownArgs);

                if(MouseDoubleClick != null)
                {
                    MouseDoubleClick(e);
                }
            }
        }
    }
}
