﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;
using App.Controls;
using App.Data;
using ConfigCommon;

namespace App.PropertyEdit
{
    public partial class CurveView : UIPanel
    {
        public CurveViewState	State{		get; private set;	}
        public CurveViewPainter	Painter{	get; private set;	}

        public EditModeType		EditMode{	get{ return CurveEditorPanel.EditMode;			} }

        public bool		IsUseFrameSnap{			get { return parent_.IsUseFrameSnap;		} }
        public bool		IsUseValueSnap{			get { return parent_.IsUseValueSnap;		} }
        public double	FrameSnapFactor{		get { return parent_.FrameSnapFactor;		} }
        public double	ValueSnapFactor{		get { return parent_.ValueSnapFactor;		} }
        public bool		IsKeyAddMode{			get { return parent_.IsKeyAddMode;			} }

        public bool		IsVisibleFrameValue{	get { return parent_.IsVisibleFrameValue;	} }
        public bool		IsVisibleSlope{			get { return parent_.IsVisibleSlope;		} }
        public bool		IsVisibleMinMax{		get { return parent_.IsVisibleMinMax;		} }
        public bool		IsVisibleCurrentValue{	get { return parent_.IsVisibleCurrentValue;	} }
        public bool IsVisibleCurveName { get { return parent_.IsVisibleCurveName; } }
        public bool		IsClampFrame{			get { return parent_.IsClampFrame;	 		} }
        public bool		IsClampValue{			get { return parent_.IsClampValue;	 		} }
        public bool		IsClampCurrentFrame{	get { return parent_.IsClampCurrentFrame;	} }
        public bool IsAutoSplineSlope { get { return parent_.IsAutoSplineSlope; } }
        public bool IsSnapToKey { get { return parent_.IsSnapToKey; } }

        public bool		IsInMouse{				get { return State.IsInMouse;				} }
        public bool		IsDragging{				get { return State.IsDragging;				} }

        public int		FrameCount{	get{ return parent_.FrameCount;	} }

        public int		CurveBgLeft{	get{ return Painter.CurveBgLeft;	} }
        public int		CurveBgTop{		get{ return Painter.CurveBgTop;		} }
        public int		CurveBgRight{	get{ return Painter.CurveBgRight;	} }
        public int		CurveBgBottom{	get{ return Painter.CurveBgBottom;	} }
        public int		CurveBgWidth{	get{ return Painter.CurveBgWidth;	} }
        public int		CurveBgHeight{	get{ return Painter.CurveBgHeight;	} }

        // コンテキストメニューの表示
        public void ShowContextMenu(ContextMenuPopupEventArgs e)
        {
            if (ContextMenuPopup != null)
            {
                ContextMenuPopup(this, e);
            }
        }

        public event ContextMenuPopupEventHandler ContextMenuPopup = null;

        /// <summary>
        /// ディスプレイ上のX方向の1ピクセルのカーブ平面での量
        /// </summary>
        public double ScaleX{	get { return State.ScaleX;	}	set { State.ScaleX = value;		} }

        /// <summary>
        /// ディスプレイ上のY方向の1ピクセルのカーブ平面での量
        /// </summary>
        public double ScaleY{	get { return State.ScaleY;	}	set { State.ScaleY = value;		} }

        /// <summary>
        /// ディスプレイ座標X軸方向のスクロールをScaleX で割った量。
        /// ディスプレイ座標でもカーブ平面座標でもない。
        /// </summary>
        public double ScrollX{	get { return State.ScrollX;	}	set { State.ScrollX = value;	} }

        /// <summary>
        /// ディスプレイ座標Y軸方向のスクロールをScaleY で割った量。
        /// ディスプレイ座標でもカーブ平面座標でもない。
        /// </summary>
        public double ScrollY{	get { return State.ScrollY;	}	set { State.ScrollY = value;	} }

        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public List<IAnimationCurve> SelectedCurves { get { return parent_.SelectedCurves; } set { parent_.SelectedCurves = value; } }

        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public List<IAnimationCurve> EditableCurves { get { return parent_.EditableCurves; }}

        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public List<IAnimationCurve> VisibledCurves { get { return parent_.VisibledCurves; } }

        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public List<IAnimationCurve> MovableCurves { get { return parent_.MovableCurves; } }

        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public List<IAnimationCurve> SelectedColorCurves
        {
            get
            {
                return
                    SelectedCurves.Where(
                        x =>
                        {
                            return x.IsColorCurve;
                        }
                    ).ToList();
            }
        }

        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public List<IAnimationCurve> SelectedVisibleCurves
        {
            get
            {
                return SelectedCurves.Where(x => (x is MaterialVisibilityAnimationCurveTreeNodeInfo2) || (x is MaterialVisibilityAnimationCurveTreeNodeInfo) || (x is BoneVisibilityAnimationCurveTreeNodeInfo)).ToList();
            }
        }

        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public List<IAnimationCurve> SelectedTexturePatternCurves
        {
            get
            {
                return SelectedCurves.Where(x => x is TexturePatternAnimationCurveTreeNodeInfo2 || x is TexturePatternAnimationCurveTreeNodeInfo).ToList();
            }
        }

        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        [Browsable(false)]
        public IEnumerable<IAnimationCurve> AllCurves { get { return parent_.AllCurves; } }

        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public GuiObjectGroup	TargetGroup {		get { return parent_.TargetGroup; }	set{ parent_.TargetGroup = value; }	}

        public GuiObjectID		TargetGuiObjectID { get { return parent_.TargetGuiObjectID; } }

        // 選択されているカーブでタイプが「色」のものがあるか
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public bool IsSelectedColorCurve
        {
            get
            {
                return
                    SelectedCurves.Any(x => x.IsColorCurve);
            }
        }

        // 選択されているカーブでビジブルものがあるか
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public bool IsSelectedVisibleCurve
        {
            get
            {
//				return SelectedVisibleCurves.Where(x => x.KeyFrames.Any()).Any();
                return SelectedVisibleCurves.Any();
            }
        }

        // 選択されているカーブでビジブルものがあるか
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public bool IsSelectedTexturePatternCurve
        {
            get
            {
//				return SelectedTexturePatternCurves.Where(x => x.KeyFrames.Any()).Any();
                return SelectedTexturePatternCurves.Any();
            }
        }

        private CurveEditorPanel parent_;

        public CurveEditorPanel ParentPanel{ get{ return parent_; } }

        // キースナップ補助線用
        public List<float> SnappedFrames = new List<float>();
        public List<float> SnappedValues = new List<float>();

        public CurveView()
        {
            State   = new CurveViewState(this);
            Painter = new CurveViewPainter(this);

            Resize += (s, e) => Invalidate();
            State.KeyParamChanged += (s, e) => parent_.SetKeyParam();
        }

        public void InitializeForm(CurveEditorPanel parent)
        {
            parent_ = parent;
        }

        protected override void OnPaint(PaintEventArgs e)
        {
            if (TheApp.IsInitializedMainFrame)			// フォームデザイナで走ってしまう。DesignModeでもだめ。
            {
                Painter.Paint(e, TargetGroup.Active);
            }
        }

        protected override void OnKeyUp(KeyEventArgs e)
        {
            // コンテキストメニュー
            if (e.KeyCode == Keys.Apps)
            {
                // コンテキストメニュー
                ShowContextMenu(new ContextMenuPopupEventArgs(PointToClient(Cursor.Position), true));
            }
            base.OnKeyUp(e);
        }

        public bool IsHDR()
        {
            return Painter.IsHDR(TargetGroup.Active);
        }

        public bool CheckEditRestriction()
        {
            return parent_.CheckEditRestriction();
        }

        public double GetXInCurvePlane(int displayX)
        {
            return Painter.GetXInCurvePlane(displayX);
        }

        public double GetYInCurvePlane(int displayY)
        {
            return Painter.GetYInCurvePlane(displayY);
        }

        public void MakeCurveViewScreenPos(float srcX, float srcY, out double dstX, out double dstY)
        {
            Painter.MakeCurveViewScreenPos(srcX, srcY, out dstX, out dstY);
        }

        public void MakeCurveViewScreenHandleOffset(double slope, double handleLineLength, out double handleX, out double handleY)
        {
            Painter.MakeCurveViewScreenHandleOffset(slope, handleLineLength, out handleX, out handleY);
        }

        public double MakeSlope(double x, double y)
        {
            return Painter.MakeSlope(x, y);
        }

        public void SetFitView(float frameMin, float frameMax, float valueMin, float valueMax)
        {
            // スケール
            var viewHeight = CurveBgHeight - CurveEditorConst.CurveViewFitMarginY;
            var viewWidth = CurveBgWidth - CurveEditorConst.CurveViewFitMarginX;
            {
                double frameRange = frameMax - frameMin;
                double valueRange = valueMax - valueMin;
                ScaleX = frameRange / viewWidth;
                if (valueRange == 0.0)
                {
                    ScaleY = 2.0 / viewHeight;
                }
                else
                {
                    ScaleY = valueRange / viewHeight;
                }

                if (valueRange == 0.0)
                {
                    valueMin -= (float)(ScaleY * viewHeight * 0.5);//(float)Math.Max(-CurveEditorUtility.ScaleMin * (CurveBgHeight - CurveEditorConst.CurveViewFitMargin), valueMin);
                }
                else if (ScaleY == CurveEditorUtility.ScaleMax)
                {
                    // クランプされたら
                    valueMin = (float)Math.Max(-CurveEditorUtility.ScaleMax * viewHeight, valueMin);
                }
                else if (ScaleY == CurveEditorUtility.ScaleMinY)
                {
                    // クランプされたら中央に表示
                    valueMin -= (float)((CurveEditorUtility.ScaleMinY * viewHeight * 0.5) - valueRange * 0.5);
                }
            }

            // スクロール
            {
                ScrollX =  (viewWidth * 0.5f + frameMin / ScaleX) / ScaleX - CurveBgLeft / ScaleX;
                ScrollY = -(viewHeight * 0.5f + valueMin / ScaleY - CurveEditorConst.CurveViewFitOffsetY) / ScaleY;
            }
        }

        public void SetFitX(float frameMin, float frameMax)
        {
            // スケール
            {
                double frameRange = frameMax - frameMin;
                ScaleX = frameRange / (CurveBgWidth - CurveEditorConst.CurveViewFitMarginX);
            }

            // スクロール
            {
                ScrollX = ((CurveBgWidth - CurveEditorConst.CurveViewFitMarginX) * 0.5f + frameMin / ScaleX) / ScaleX - CurveBgLeft / ScaleX;
            }
        }

        public void SetFitY(float valueMin, float valueMax)
        {
            // スケール
            {
                double valueRange = valueMax - valueMin;
                ScaleY = valueRange / (CurveBgHeight - CurveEditorConst.CurveViewFitMarginY);

                if (ScaleY == CurveEditorUtility.ScaleMax)
                {
                    // クランプされたら
                    valueMin = (float)Math.Max(-CurveEditorUtility.ScaleMax * (CurveBgHeight - CurveEditorConst.CurveViewFitMarginY), valueMin);
                }
            }

            // スクロール
            {
                ScrollY = -((CurveBgHeight - CurveEditorConst.CurveViewFitMarginY) * 0.5f + valueMin / ScaleY - CurveEditorConst.CurveViewFitOffsetY) / ScaleY;
            }
        }

        public void SetCenterFrame(double centerFrame)
        {
            ScrollX = (centerFrame / ScaleX) / ScaleX - CurveBgLeft / ScaleX;
        }

        public class ValueRange
        {
            public ValueRange()
            {
                MinValue = null;
                MaxValue = null;
            }

            public float?	MinValue{	get; set; }
            public float?	MaxValue{	get; set; }
        }

        // 値の編集範囲を作る
        public ValueRange MakeEditValueRange()
        {
            if (SelectedCurves.Any())
            {
                float? minValue = +float.MaxValue;
                float? maxValue = -float.MaxValue;
                {
                    foreach(var curve in SelectedCurves)
                    {
                        if (minValue != null)
                        {
                            if (curve.MinFitValue == null)
                            {
                                minValue = null;
                            }
                            else
                            {
                                minValue = Math.Min((float)curve.MinFitValue, (float)minValue);
                            }
                        }

                        if (maxValue != null)
                        {
                            if (curve.MaxFitValue == null)
                            {
                                maxValue = null;
                            }
                            else
                            {
                                maxValue = Math.Max((float)curve.MaxFitValue, (float)maxValue);
                            }
                        }
                    }
                }

                return
                    new ValueRange
                    {
                        MinValue = minValue,
                        MaxValue = maxValue
                    };
            }
            return new ValueRange();
        }

        public void AddKey(List<IAnimationCurve> curves, Point pos)
        {
            parent_.AddKey(curves, pos);
        }

        public void AddKey(List<IAnimationCurve> curves, float frame, float value)
        {
            parent_.AddKey(curves, frame, value);
        }

        public void AddColorKey(List<IAnimationCurve> curves, float frame, Color color, float hdrFactor)
        {
            parent_.AddColorKey(curves, frame, color, hdrFactor);
        }

#region 色カーブ関係

        // 複数のRGB(A)の組み合わせがあるかをしらべる
        public bool IsAnyColorSet(List<IAnimationCurve> curves)
        {
            var anySet = new HashSet<string>();
            {
                foreach (var curve in curves)
                {
                    anySet.Add(curve.ParentName + "\\" + curve.Name);
                }
            }

            return anySet.Count >= 2;
        }

        public class ColorSet
        {
            public float						Frame{	get; set; }
            //public float HDRFactor { get; set; }
            //public Color						Color{	get; set; }
            public RgbaColor Color { get; set; }
            public IEnumerable<IAnimationCurve>	Curves{	get; set; }
        }

        // 同じフレームの組みを作る
        public IEnumerable<ColorSet> MakeSameFrameColorSet(
            IEnumerable<IAnimationCurve> curves,
            GuiObject target,
            Dictionary<string, List<IAnimationCurve>> colorCurveGroup)
        {
            var colorSets = new List<ColorSet>();
            {
                if (curves.Any())
                {
                    var front = curves.First();

                    if (colorCurveGroup.ContainsKey(front.ParentName + " " + front.Name))
                    {
                        var rgba = new float[4];

                        var colorSetCurves = colorCurveGroup[front.ParentName + " " + front.Name];
                        {
                            var frontKeyFrames = colorSetCurves.Where(x => x.KeyFrames.Any()).Select(x => x.KeyFrames).FirstOrDefault();
                            if (frontKeyFrames != null)
                            {
                                foreach(var frontKeyFrame in frontKeyFrames)
                                {
                                    rgba[0] = 0.0f;
                                    rgba[1] = 0.0f;
                                    rgba[2] = 0.0f;
                                    rgba[3] = 1.0f;

                                    var enableCurves = new List<IAnimationCurve>();

                                    var enableCurveCount = 0;		// 有効なカーブ数。キーを持たないカーブはカウントに含めない
                                    var sameCount = 0;				// 同じフレームの数。
                                    var hasNotSameAlphaKey = false; // 同じフレームにアルファのキーがない
                                    var rgbCurveSelected = false;
                                    var frontKeyFrameFrame = frontKeyFrame.Frame;
                                    {
                                        foreach(var curveSet in colorSetCurves)
                                        {
                                            var curve      = curveSet;
                                            var animTarget = curveSet.GetAnimTarget(target);

                                            if (curve.KeyFrames.Any(x => x.Frame == frontKeyFrameFrame))
                                            {
                                                ++ sameCount;
                                                if (curve.ColorComponentIndex != 3 && curves.Contains(curve))
                                                {
                                                    rgbCurveSelected = true;
                                                }
                                                rgba[curve.ColorComponentIndex] =
                                                        Math.Max(0.0f, curve.GetValue(frontKeyFrameFrame, animTarget.pre_wrap, animTarget.post_wrap));
                                            }
                                            else if (curve.KeyFrames.Any() && curve.ColorComponentIndex == 3)
                                            {
                                                hasNotSameAlphaKey = true;
                                                rgba[3] = curve.GetValue(
                                                    frontKeyFrameFrame,
                                                    animTarget.pre_wrap,
                                                    animTarget.post_wrap);
                                            }

                                            if (curve.KeyFrames.Any())
                                            {
                                                ++ enableCurveCount;
                                            }
                                            enableCurves.Add(curve);
                                        }
                                    }

                                    // 全て同じフレームあるいはアルファ以外が同じフレームであれば保存する。
                                    if ((sameCount == enableCurveCount)
                                        || (hasNotSameAlphaKey && rgbCurveSelected && (sameCount == enableCurveCount - 1)))
                                    {
                                        var rgbaColor = new RgbaColor(rgba[0], rgba[1], rgba[2], rgba[3]);
                                        colorSets.Add(
                                            new ColorSet
                                            {
                                                Frame  = frontKeyFrameFrame,
                                                Curves = enableCurves,
                                                Color  = rgbaColor,
                                            }
                                        );
                                    }
                                }
                            }
                        }
                    }
                }
            }
            return colorSets;
        }

        /// <summary>
        /// ColorSet を更新
        /// </summary>
        public ColorSet UpdateColorSet(ColorSet old, bool[] hasKey)
        {
            var colorSet = new ColorSet();
            var curves = new List<IAnimationCurve>();
            var rgba = new [] {old.Color.R, old.Color.G, old.Color.B, old.Color.A};
            foreach (var item in old.Curves)
            {
                bool ok = false;
                foreach (var curve in AllCurves)
                {
                    if (curve.IsSameCurve(item) && curve.IsEditable)
                    {
                        var key = curve.KeyFrames.FirstOrDefault(x => x.Frame == old.Frame);

                        // hasKey が null でなければ、hasKey の成分のキーと一致している必要がある
                        if (hasKey != null && hasKey[curve.ColorComponentIndex] != (key != null))
                        {
                            return null;
                        }

                        if (key != null)
                        {
                            rgba[curve.ColorComponentIndex] = key.Value;
                        }

                        ok = true;
                        curves.Add(curve);
                        break;
                    }
                }
                if (!ok)
                {
                    return null;
                }
            }

            colorSet.Frame = old.Frame;
            colorSet.Curves = curves;
            colorSet.Color = new RgbaColor(rgba[0], rgba[1], rgba[2], rgba[3]);
            return colorSet;
        }

        public RgbaColor MakeColorFromColorCurve(
            float frame,
            out float HDRFactor,
            out bool hasA)
        {
            hasA = false;
            var rgba = new float[4]{0.0f, 0.0f, 0.0f, 1.0f};
            {
                var colorSet = Painter.MakeColorCurveGroup(true);
                if(colorSet.Any())
                {
                    // TODO: First でいいのか？
                    foreach (var curve in colorSet.First().Value)
                    {
                        if (curve.ColorComponentIndex == 3)
                        {
                            hasA = true;
                        }

                        var target = curve.GetAnimTarget(TargetGroup.Active);

                        // カーブを評価する
                        // アルファチャンネルでキーが１つもない場合は1.0fとして扱う
                        rgba[curve.ColorComponentIndex] =
                            ((curve.ColorComponentIndex == 3) && (curve.KeyFrames.Any() == false)) ?
                                1.0f :
                                Math.Max(0.0f, curve.GetValue(frame, target.pre_wrap, target.post_wrap));


                    }
                }
            }

            HDRFactor = Math.Max(Math.Max(Math.Max(rgba[0], rgba[1]), rgba[2]), 1);
            var rgbaColor = new RgbaColor(rgba[0], rgba[1], rgba[2], rgba[3]);
            return rgbaColor;
        }

        #endregion

#region ビジブルカーブ関係
        public class VisibleSet
        {
            public float			Frame{		get; set; }
            public bool				IsVisible{	get; set; }
            public IAnimationCurve	Curve{		get; set; }
        }
#endregion

#region テクスチャパターンカーブ関係
        public class TexturePatternSet
        {
            public float			Frame{					get; set; }
            public int				TexturePatternIndex{	get; set; }
            public IAnimationCurve	Curve{					get; set; }
        }
#endregion
    }
}
