﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using App.Command;
using App.Data;
using App.Utility;

namespace App.PropertyEdit
{
    /// <summary>
    /// カーブエディタパネルクラス。
    /// </summary>
    public partial class CurveEditorPanel
    {
        private class Data
        {
            public IAnimationCurve	Curve{			get; set; }
            public KeyFrame			KeyFrame{		get; set; }
            public KeyFrame			BeginKeyFrame{	get; set; }
            public KeyFrame			EndKeyFrame{	get; set; }
        };

        private List<Data> datas;

        private void BeginMoveSelectedKeys()
        {
            datas = new List<Data>();

            foreach(var visibleCurve in MovableCurves)
            {
                visibleCurve.SetCurvesSequentialIndex();

                foreach(var key in visibleCurve.KeyFrames)
                {
                    if (key.Selected)
                    {
                        datas.Add(
                            new Data()
                            {
                                Curve			= visibleCurve,
                                KeyFrame		= key,
                                BeginKeyFrame	= key.Clone(),
                            }
                        );
                    }
                }
            }
        }

        private void EndMoveSelectedKeys()
        {
            Debug.Assert(datas != null);
            if (datas.Any())
            {
                foreach(Data data in datas)
                {
                    // 結果を保存する
                    data.EndKeyFrame	= data.KeyFrame.Clone();

                    // 元に戻す
                    data.KeyFrame.Set(data.BeginKeyFrame);
                }

                // 並びを元に戻す
                MovableCurves.ForEach(x => x.SortByIndex());

                var commandSet = new EditCommandSet();
                {
                    commandSet.SetViewerDrawSuppressBlockDelegate(AnimationCurveEditCommand.AnimationMessageFilter);
                    var keyDataDict = new Dictionary<IAnimationCurve, List<KeyFrame>>();
                    {
                        foreach(var data in datas)
                        {
                            var curve = data.Curve;

                            if (keyDataDict.ContainsKey(curve) == false)
                            {
                                keyDataDict[curve] = ObjectUtility.Clone(curve.KeyFrames);
                            }

                            keyDataDict[curve][data.EndKeyFrame.Index] = data.EndKeyFrame.Clone();
                        }

                        foreach (var value in keyDataDict.Values)
                        {
                            KeyFramesUtility.SortByFrame(value);
                        }
                    }

                    foreach (var keyData in keyDataDict)
                    {
                        var curve = keyData.Key;
                        var keys  = keyData.Value;

                        commandSet.Add(
                            AnimationCurveEditCommand.Create(
                                TargetGroup,
                                TargetGuiObjectID,
                                curve,
                                keys
                            )
                        );
                    }
                }

                if (commandSet.CommandCount > 0)
                {
                    commandSet.OnPostEdit += (s, e) => Viewer.HioUtility.FlushCurves(TargetGroup, TargetGuiObjectID);
                    TheApp.CommandManager.Add(commandSet.Execute());
                }
            }

            datas = null;
        }

        private void MoveSelectedKeys(float moveX, float moveY)
        {
            if (datas.Any())
            {
                foreach(var data in datas)
                {
                    float dX = moveX;
                    float dY = moveY;
                    switch (data.Curve.CurvePrimitiveType)
                    {
                        case PrimitiveTypeKind.Bool:
                        case PrimitiveTypeKind.Int:
                        case PrimitiveTypeKind.Uint:
                            if (dY < 0)
                            {
                                dY = -1;
                            }
                            else if (dY > 0)
                            {
                                dY = 1;
                            }
                            break;
                    }

                    float value = data.KeyFrame.Value + dY;
                    float frame = data.KeyFrame.Frame + dX;
                    if (IsClampValue)
                    {
                        value = data.Curve.ClampValue(value);
                    }
                    else
                    {
                        switch (data.Curve.CurvePrimitiveType)
                        {
                            case PrimitiveTypeKind.Bool:
                                value = (value >= 0.5f) ? 1.0f : 0.0f;
                                break;
                            case PrimitiveTypeKind.Int:
                                value =(float)Math.Round(value, MidpointRounding.AwayFromZero);
                                break;
                            case PrimitiveTypeKind.Uint:
                                value = (float)Math.Max(Math.Round(value, MidpointRounding.AwayFromZero), 0);
                                break;
                        }
                    }
                    if (IsClampFrame)
                    {
                        frame = Math.Min(Math.Max(frame, 0.0f), FrameCount);
                    }
                    data.KeyFrame.Value = value;
                    data.KeyFrame.Frame = frame;
                }

                MovableCurves.ForEach(x => x.SortByFrame());

                CurveView.Invalidate();
            }
        }
    }
}
