﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using App.Data;

namespace App.PropertyEdit
{
    public class MaterialVisibilityAnimationCurveTreeNodeInfo : IAnimationCurve
    {
        public bool IsEditable { get; set; }
        public AnimationDocument.NonEditableKind NonEditableKind { get; set; }
        public object NonEditableKindDisplayAux { get; set; }
        public bool IsColorCurve { get { return false; } }
        public int ColorComponentIndex { get { return -1; } }

        private readonly MaterialVisibilityAnimation	target_;
        private readonly string					materialName_;

        public AnimTarget GetAnimTarget(GuiObject editTarget)
        {
            var animation = (MaterialVisibilityAnimation)editTarget;
            return animation.MaterialVisibilityMatAnims.FirstOrDefault(x => x.mat_name == materialName_);
        }

        public void UpdateIsModified(GuiObject editTarget)
        {
            var animation = (MaterialVisibilityAnimation)editTarget;
            var animTarget = GetAnimTarget(animation);
            var savedTarget = animation.savedMaterialVisiblityMatAnims.FirstOrDefault(x => x.mat_name == materialName_);
            animation.UpdateIsModifiedAnimTarget(
                (MaterialVisibilityAnimation.MaterialVisibilityMatAnim)animTarget,
                savedTarget);
        }

        public void SetAnimTarget(GuiObject editTarget, AnimTarget animTarget)
        {
            var animation = (MaterialVisibilityAnimation)editTarget;
            var index = animation.MaterialVisibilityMatAnims.FindIndex(x => x.mat_name == materialName_);
            if (index == -1)
            {
                Debug.Assert(false);
                throw new NotImplementedException();
            }

            Debug.Assert(animTarget is MaterialVisibilityAnimation.MaterialVisibilityMatAnim);
            animation.MaterialVisibilityMatAnims[index] = (MaterialVisibilityAnimation.MaterialVisibilityMatAnim)animTarget;
        }

        public AnimTarget CreateAnimTarget(GuiObject editTarget)
        {
            Debug.Assert(false);
            throw new NotImplementedException();
        }

        public AnimTarget CreateTemporaryTarget(GuiObject editTarget)
        {
            var source = (MaterialVisibilityAnimation.MaterialVisibilityMatAnim)GetAnimTarget(editTarget);
            return new MaterialVisibilityAnimation.MaterialVisibilityMatAnim()
            {
                mat_name = source.mat_name,
            };
        }

        public float GetDefaultValue(GuiObject editTarget)
        {
            return GetAnimTarget(editTarget).GetBaseValue();
        }

        public void IntermediateFileIndices(MaterialVisibilityAnimation target, out int materialIndex, out int curveIndexInBinary)
        {
            materialIndex = 0;
            curveIndexInBinary = 0;
            foreach (var matAnim in target.MaterialVisibilityMatAnims)
            {
                if (!matAnim.KeyFrames.Any())
                {
                    continue;
                }

                if (matAnim.mat_name == materialName_)
                {
                    return;
                }

                switch (matAnim.ExportType)
                {
                    case CurveExportType.Curve:
                    case CurveExportType.Constant:
                        curveIndexInBinary++;
                        break;
                }

                materialIndex++;

            }
            Debug.Assert(false);
        }

        public MaterialVisibilityAnimationCurveTreeNodeInfo(MaterialVisibilityAnimation target, string materialName)
        {
            target_			= target;
            materialName_	= materialName;
        }

        public List<KeyFrame> KeyFrames
        {
            get
            {
                var paramAnim = target_.GetParamAnimFromId(materialName_);
                return
                    (paramAnim != null) ?
                        paramAnim.KeyFrames :
                        new List<KeyFrame>();
            }
        }

        public InterpolationType CurveInterpolationType
        {
            get
            {
                return InterpolationType.Step;
            }

            set
            {
                ;   // 無視
            }
        }

        public string ParentName{		get { return materialName_;		} }
        public string Name{				get { return "dummy";			} }
        public string FullPath{			get { return string.Format("{0} / {1}", target_.Name, ParentName);	} }
        public int ComponentIndex{		get { return 0;					} }
        public bool IsRotate{			get { return false;				} }
        public float? MinClampValue{	get { return 0.0f;				} }
        public float? MaxClampValue{	get { return 1.0f;				} }
        public float? MinFitValue{		get { return 0.0f;				} }
        public float? MaxFitValue{		get { return 1.0f;				} }
        public PrimitiveTypeKind CurvePrimitiveType { get { return PrimitiveTypeKind.Bool; } }

        public Color CurveColor { get { return CurveView.MakeColorFromString(target_.Name + ParentName); } }

        public bool IsSame(IAnimationCurve curve)
        {
            var visibilityCurve = (MaterialVisibilityAnimationCurveTreeNodeInfo)curve;
            return visibilityCurve.target_ == target_ &&
                visibilityCurve.materialName_ == materialName_;
        }
    }

    public class MaterialVisibilityAnimationCurveTreeNodeInfo2 : IAnimationCurve
    {
        public bool IsEditable { get; set; }
        public AnimationDocument.NonEditableKind NonEditableKind { get; set; }
        public object NonEditableKindDisplayAux { get; set; }
        public bool IsColorCurve { get { return false; } }
        public int ColorComponentIndex { get { return -1; } }

        private readonly MaterialAnimation target_;
        private readonly string materialName_;

        public AnimTarget GetAnimTarget(GuiObject editTarget)
        {
            var animation = (MaterialAnimation)editTarget;
            return animation.PerMaterialAnims[materialName_].MaterialVisibilityMatAnim;
        }

        public void UpdateIsModified(GuiObject editTarget)
        {
            var animation = (MaterialAnimation)editTarget;
            var animTarget = GetAnimTarget(animation);
            var perMatAnim = animation.savedPerMaterialAnims[materialName_];
            var savedTarget = perMatAnim != null ? perMatAnim.MaterialVisibilityMatAnim: null;
            animation.UpdateIsModifiedAnimTarget(
                (MaterialVisibilityAnimation.MaterialVisibilityMatAnim)animTarget,
                savedTarget);
        }

        public void SetAnimTarget(GuiObject editTarget, AnimTarget animTarget)
        {
            var animation = (MaterialAnimation)editTarget;
            var perMaterialAnim = animation.PerMaterialAnims[materialName_];
            if (perMaterialAnim == null)
            {
                Debug.Assert(false);
                throw new NotImplementedException();
            }

            Debug.Assert(animTarget is MaterialVisibilityAnimation.MaterialVisibilityMatAnim);
            perMaterialAnim.MaterialVisibilityMatAnim = (MaterialVisibilityAnimation.MaterialVisibilityMatAnim)animTarget;
        }

        public AnimTarget CreateAnimTarget(GuiObject editTarget)
        {
            Debug.Assert(false);
            throw new NotImplementedException();
        }

        public AnimTarget CreateTemporaryTarget(GuiObject editTarget)
        {
            var source = (MaterialVisibilityAnimation.MaterialVisibilityMatAnim)GetAnimTarget(editTarget);
            return new MaterialVisibilityAnimation.MaterialVisibilityMatAnim()
            {
                mat_name = source.mat_name,
            };
        }

        public float GetDefaultValue(GuiObject editTarget)
        {
            return GetAnimTarget(editTarget).GetBaseValue();
        }

        public void IntermediateFileIndices(MaterialAnimation target, out int materialIndex, out int curveIndexInBinary)
        {
            materialIndex = 0;
            curveIndexInBinary = 0;
            foreach (var perMaterialAnim in target.PerMaterialAnims)
            {
                if (perMaterialAnim.mat_name == materialName_)
                {
                    curveIndexInBinary += ShaderParameterAnimationCurveTreeNodeInfo2.CurvesInBinary(perMaterialAnim).Count()
                        + TexturePatternAnimationCurveTreeNodeInfo2.CurvesInBinary(perMaterialAnim).Count();
                    return;
                }
                else
                {
                    var curveCount = ShaderParameterAnimationCurveTreeNodeInfo2.CurvesInBinary(perMaterialAnim).Count() +
                        TexturePatternAnimationCurveTreeNodeInfo2.CurvesInBinary(perMaterialAnim).Count() +
                        (MaterialVisibilityAnimationCurveTreeNodeInfo2.AnyCurveInBinary(perMaterialAnim) ? 1 : 0);
                    if (curveCount > 0)
                    {
                        curveIndexInBinary += curveCount;
                        materialIndex++;
                    }
                }
            }
        }

        public static bool AnyCurveInBinary(MaterialAnimation.PerMaterialAnim perMaterialAnim)
        {
            var t = perMaterialAnim.MaterialVisibilityMatAnim.ExportType;
            return t == CurveExportType.Curve || t == CurveExportType.Constant;
        }

        public MaterialVisibilityAnimationCurveTreeNodeInfo2(MaterialAnimation target, string materialName)
        {
            target_ = target;
            materialName_ = materialName;
        }

        public List<KeyFrame> KeyFrames
        {
            get
            {
                var paramAnim = target_.GetVisiblityAnimFromId(materialName_);
                return
                    (paramAnim != null) ?
                        paramAnim.KeyFrames :
                        new List<KeyFrame>();
            }
        }

        public InterpolationType CurveInterpolationType
        {
            get
            {
                return InterpolationType.Step;
            }

            set
            {
                ;   // 無視
            }
        }

        public string ParentName { get { return materialName_; } }
        public string Name { get { return "dummy"; } }
        public string FullPath { get; set; }
        public int ComponentIndex { get { return 0; } }
        public bool IsRotate { get { return false; } }
        public float? MinClampValue { get { return 0.0f; } }
        public float? MaxClampValue { get { return 1.0f; } }
        public float? MinFitValue { get { return 0.0f; } }
        public float? MaxFitValue { get { return 1.0f; } }
        public PrimitiveTypeKind CurvePrimitiveType { get { return PrimitiveTypeKind.Bool; } }

        public Color CurveColor { get { return CurveView.MakeColorFromString(target_.Name + ParentName); } }

        public bool IsSame(IAnimationCurve curve)
        {
            var visibilityCurve = curve as MaterialVisibilityAnimationCurveTreeNodeInfo2;
            return visibilityCurve != null &&
                visibilityCurve.target_ == target_ &&
                visibilityCurve.materialName_ == materialName_;
        }
    }
}
