﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;
using App.Controls;
using App.Data;
using App.res;
using App.Utility;
using System.Text.RegularExpressions;
using nw.g3d.iflib.nw3de;
using nw.g3d.nw4f_3dif;

namespace App.PropertyEdit.ShaderParamControls
{
    public partial class ShaderParamControl : UIUserControl, IGroupName
    {
        public event ValueChangedEventHandler ValueChanged	= null;

        public event LabelChangedEventHandler LabelChanged = null;

        public event LinkClickedEventHandler LinkClicked;

        public void OnLinkClicked(LinkClickedEventArgs e)
        {
            if (LinkClicked != null)
            {
                LinkClicked(this, e);
            }
        }

        public LabelHelper labelHelper = new LabelHelper();
        public virtual string ParamName
        {
            get;
            set;
        }

        public string ParamLabel
        {
            get;
            set;
        }

        public virtual void UpdateLabel()
        {
            lblParameter.Text = labelHelper.GetLabelString();
            lblParameter.ForeColor = labelHelper.color ?? SystemColors.ControlText;
            lblParameter.BackColor = labelHelper.backGroundColor ?? SystemColors.Control;
            UpdateDropDownButtonLocation();
            UpdateLinkToolTip(false);
        }

        public virtual void UpdateLinkToolTip(bool force)
        {
            try
            {
                if (btnLink != null && parentHint != null)
                {
                    var toolTip = labelHelper.GetLinkLabelString();
                    var current = parentHint.GetToolTip(btnLink);
                    if (toolTip != current || force)
                    {
                        parentHint.SetToolTip(btnLink, toolTip);
                    }
                }
            }
            catch
            {
                Debug.Assert(false);
            }
        }

        /// <summary>
        /// ツールチップヒントに表示するコメント
        /// </summary>
        public string ParamComment { get; set; }

        public virtual ParamType ParamType { get { return ShaderParamControls.ParamType.uniform_var; } }

        internal static UIContextMenuStrip labelContextMenuStrip;
        public static ToolStripMenuItem cmiCopyID;
        public static ToolStripMenuItem cmiEditLabel;
        public static ToolStripMenuItem cmiMaterialReferenceBehavior;
        public static ToolStripMenuItem cmiMaterialReferenceBehaviorValue;
        public static ToolStripMenuItem cmiMaterialReferenceBehaviorRestriction;
        public static ToolStripMenuItem cmiMaterialReferenceBehaviorValue_Override;
        public static ToolStripMenuItem cmiMaterialReferenceBehaviorValue_Default;
        public static ToolStripMenuItem cmiMaterialReferenceBehaviorValue_Refer;
        public static ToolStripMenuItem cmiMaterialReferenceBehaviorRestriction_None;
        public static ToolStripMenuItem cmiMaterialReferenceBehaviorRestriction_ForceRefer;
        public static ToolStripMenuItem cmiMaterialReferenceBehaviorRestriction_ForceOverride;

        protected static readonly Bitmap MarteralReferenceIcon_Refer         = App.Properties.Resources.PropertyEdit_MaterialReference_Refer;
        protected static readonly Bitmap MarteralReferenceIcon_Default       = App.Properties.Resources.PropertyEdit_MaterialReference_Default;
        protected static readonly Bitmap MarteralReferenceIcon_ForceRefer    = App.Properties.Resources.PropertyEdit_MaterialReference_ForceRefer;
        protected static readonly Bitmap MarteralReferenceIcon_ForceOverride = App.Properties.Resources.PropertyEdit_MaterialReference_ForceOverride;

        private static ToolStripMenuItem[] ReferenceBehaviorValueMenuItems => new[]{
            cmiMaterialReferenceBehaviorValue_Override,
            cmiMaterialReferenceBehaviorValue_Refer,
            cmiMaterialReferenceBehaviorValue_Default
        };

        private static ToolStripMenuItem[] ReferenceBehaviorRestrictionMenuItems => new[]{
            cmiMaterialReferenceBehaviorRestriction_None,
            cmiMaterialReferenceBehaviorRestriction_ForceRefer,
            cmiMaterialReferenceBehaviorRestriction_ForceOverride
        };


        public ShaderParamControl()
        {
#if false
            InitializeComponent();
#else
            AutoScaleMode = System.Windows.Forms.AutoScaleMode.Font;

            if (labelContextMenuStrip == null)
            {
                labelContextMenuStrip = new App.Controls.UIContextMenuStrip
                {
                    Name = "labelContextMenuStrip",
                    Size = new System.Drawing.Size(153, 48)
                };
                labelContextMenuStrip.Opening += labelContextMenuStrip_OpeningBase;
            }

            lblParameter = new UIModifiedMarkAndImageLabel();
            lblParameter.AutoSize = true;
            lblParameter.Location = new Point(4, 4);
            lblParameter.Size = new Size(46, 13);
            lblParameter.TabIndex = 0;
            lblParameter.ContextMenuStrip = labelContextMenuStrip.ProxyContextMenuStrip;
            Controls.Add(lblParameter);
#endif
        }

        void labelContextMenuStrip_OpeningBase(object s, System.ComponentModel.CancelEventArgs e)
        {
            GetSourceControl()?.labelContextMenuStrip_Opening(s, e);
        }

        protected virtual void labelContextMenuStrip_Opening(object sender, System.ComponentModel.CancelEventArgs e)
        {
            SetupMenuItems();
            var control = GetSourceControl();
            Debug.Assert(control != null, "control != null");
            control.SetReferenceBehaviorMenu();

            SetEventHandler(cmiCopyID,      cmiCopyID_Click);
            SetEventHandler(cmiEditLabel,   cmiEditLabel_Click);
            SetEventHandlers(ReferenceBehaviorValueMenuItems,      cmiMaterialReferenceBehaviorValue_Click);
            SetEventHandlers(ReferenceBehaviorRestrictionMenuItems, cmiMaterialReferenceBehaviorRestriction_Click);

            e.Cancel = false;
        }

        private static ShaderParamControl GetSourceControl()
        {
            var control = labelContextMenuStrip.ProxySourceControl;
            while (control != null && !(control is ShaderParamControl))
            {
                control = control.Parent;
            }
            return control as ShaderParamControl;
        }

        private void SetReferenceBehaviorMenu()
        {
            var referenceBehaviorItem = GetMaterialReferenceBehaviorItem();
            var value = referenceBehaviorItem?.Value ?? ShaderItemValueState.Refer;
            var restriction = referenceBehaviorItem?.ChildRestriction ?? ShaderItemRestrictionState.None;
            var isEditable = IsReferenceBehaviorEditable();

            for (int i = 0; i < ReferenceBehaviorValueMenuItems.Length; i++)
            {
                ReferenceBehaviorValueMenuItems[i].Checked = (i == (int)value);
            }
            for (int i = 0; i < ReferenceBehaviorRestrictionMenuItems.Length; i++)
            {
                ReferenceBehaviorRestrictionMenuItems[i].Checked = (i == (int)restriction);
            }

            cmiMaterialReferenceBehaviorValue.Enabled = isEditable;
            cmiMaterialReferenceBehaviorRestriction.Enabled = isEditable;
        }

        internal static Bitmap GetReferenceStateIcon(Material.ValueResolvedState valueResolvedState)
        {
            switch (valueResolvedState)
            {
                case Material.ValueResolvedState.Refer:
                    return MarteralReferenceIcon_Refer;
                case Material.ValueResolvedState.Default:
                    return MarteralReferenceIcon_Default;
                case Material.ValueResolvedState.ForceRefer:
                    return MarteralReferenceIcon_ForceRefer;
                case Material.ValueResolvedState.ForceOverride:
                    return MarteralReferenceIcon_ForceOverride;
                default:
                    return null;
            }
        }

        internal static List<Bitmap> GetReferenceStateIcons(MaterialReferenceBehaviorItem referenceBehaviorItem)
        {
            var icons = new List<Bitmap>();
            if (referenceBehaviorItem == null)
            {
                icons.Add(MarteralReferenceIcon_Refer);
                return icons;
            }

            switch (referenceBehaviorItem.Value)
            {
                case ShaderItemValueState.Refer:
                    icons.Add(MarteralReferenceIcon_Refer);
                    break;
                case ShaderItemValueState.Default:
                    icons.Add(MarteralReferenceIcon_Default);
                    break;
            }

            switch (referenceBehaviorItem.ChildRestriction)
            {
                case ShaderItemRestrictionState.ForceRefer:
                    icons.Add(MarteralReferenceIcon_ForceRefer);
                    break;
                case ShaderItemRestrictionState.ForceOverride:
                    icons.Add(MarteralReferenceIcon_ForceOverride);
                    break;
            }
            return icons;
        }

        internal static List<Bitmap> GetReferenceStateIcons(Material.ValueResolvedState valueResolvedState, MaterialReferenceBehaviorItem referenceBehaviorItem)
        {
            var icons = new List<Bitmap>();
            switch (valueResolvedState)
            {
                case Material.ValueResolvedState.Refer:
                    icons.Add(MarteralReferenceIcon_Refer);
                    break;
                case Material.ValueResolvedState.Default:
                    icons.Add(MarteralReferenceIcon_Default);
                    break;
                case Material.ValueResolvedState.ForceRefer:
                    icons.Add(MarteralReferenceIcon_Refer);
                    break;
                case Material.ValueResolvedState.ForceOverride:
                    break;
            }

            if (referenceBehaviorItem != null)
            {
                switch (referenceBehaviorItem.ChildRestriction)
                {
                    case ShaderItemRestrictionState.ForceRefer:
                        icons.Add(MarteralReferenceIcon_ForceRefer);
                        break;
                    case ShaderItemRestrictionState.ForceOverride:
                        icons.Add(MarteralReferenceIcon_ForceOverride);
                        break;
                }
            }

            return icons;
        }


        internal MaterialReferenceBehaviorItem GetMaterialReferenceBehaviorItem()
        {
            Debug.Assert(material != null);
            MaterialReferenceBehaviorItem shaderItem;
            switch (ParamType)
            {
                case ParamType.attrib_var:
                    shaderItem = material?.GetReferenceBehaviorAttribAssign(ParamName);
                    break;
                case ParamType.option_var:
                    shaderItem = material?.GetReferenceBehaviorShaderOption(ParamName);
                    break;
                case ParamType.sampler_var:
                    shaderItem = material?.GetReferenceBehaviorSamplerAssign(ParamName);
                    break;
                case ParamType.uniform_var:
                    shaderItem = material?.GetReferenceBehaviorShaderParam(ParamName);
                    break;
                case ParamType.render_info_slot:
                    shaderItem = material?.GetReferenceBehaviorRenderInfo(ParamName);
                    break;
                default:
                    throw new ArgumentOutOfRangeException();
            }
            return ObjectUtility.Clone(shaderItem);
        }

        internal void SetMaterialReferenceBehaviorItem(MaterialReferenceBehaviorItem referenceBehaviorItem)
        {
            Debug.Assert(material != null);
            switch (ParamType)
            {
                case ParamType.attrib_var:
                    material.SetReferenceBehaviorAttribAssign(referenceBehaviorItem);
                    break;
                case ParamType.option_var:
                    material.SetReferenceBehaviorShaderOption(referenceBehaviorItem);
                    break;
                case ParamType.sampler_var:
                    material.SetReferenceBehaviorSamplerAssign(referenceBehaviorItem);
                    break;
                case ParamType.uniform_var:
                    material.SetReferenceBehaviorShaderParam(referenceBehaviorItem);
                    break;
                case ParamType.render_info_slot:
                    material.SetReferenceBehaviorRenderInfo(referenceBehaviorItem);
                    break;
                default:
                    throw new ArgumentOutOfRangeException();
            }
        }

        internal bool IsReferenceBehaviorEditable()
        {
            Debug.Assert(material != null);
            var t = GetG3dParamType();
            return material.IsReferenceBehaviorEditable(ParamName, t);
        }
        internal bool IsValueEditable()
        {
            Debug.Assert(material != null);
            var t = GetG3dParamType();

            var paramName = ParamName;

            return material.IsResolvedValueEditable(paramName, t);
        }

        internal Type GetG3dParamType()
        {
            Type t;
            switch (ParamType)
            {
                case ParamType.attrib_var:
                    t = typeof(attrib_assignType);
                    break;
                case ParamType.option_var:
                    t = typeof(shader_optionType);
                    break;
                case ParamType.sampler_var:
                    t = typeof(sampler_assignType);
                    break;
                case ParamType.uniform_var:
                    t = typeof(shader_paramType);
                    break;
                case ParamType.render_info_slot:
                    t = typeof(render_infoType);
                    break;
                default:
                    throw new ArgumentOutOfRangeException();
            }
            return t;
        }

        private static void SetEventHandlers(IEnumerable<ToolStripMenuItem> toolStripMenuItems, EventHandler clickevent)
        {
            foreach (var menuItem in toolStripMenuItems)
            {
                SetEventHandler(menuItem, clickevent);
            }
        }

        private static void SetEventHandler(ToolStripMenuItem item, EventHandler clickevent)
        {
            item.Click -= clickevent;
            item.Click += clickevent;
        }

        // メニューアイテムをスタティックフィールドにコピー

        private static void SetupMenuItems()
        {
            labelContextMenuStrip.Items.Clear();
            // コンテキストメニューの構築
            TheApp.MainFrame.AddShaderParameterContextMenu(labelContextMenuStrip);

            var items = new List<ToolStripMenuItem>();
            Action<ToolStripMenuItem, List<ToolStripMenuItem>> action = null;
            action = (item, list) =>
            {
                list.Add(item);
                foreach (var subitem in item.DropDownItems.OfType<ToolStripMenuItem>())
                {
                    action(subitem, items);
                }
            };

            foreach (var item in labelContextMenuStrip.Items.OfType<ToolStripMenuItem>())
            {
                action(item, items);
            }

            var type = typeof(ShaderParamControl);

            foreach (var item in items)
            {
                var fi = type.GetField(item.Name);
                if (fi != null && fi.GetValue(null) == null)
                {
                    fi.SetValue(null, item);
                }
            }
        }

        #region アニメーションボタン

        /// <summary>
        /// 「カーブエディタを開く」ボタンを追加する成分数を設定する。
        /// InitializeDropDownButton() でボタンが生成され ComponentCurveEditorOpeningButtons に設定される。
        /// </summary>
        protected int NumComponentCurveEditorOpeningButtons
        {
            set { numComponentCurveEditorOpeningButtons = value; }
        }
        private int numComponentCurveEditorOpeningButtons = 0;

        /// <summary>
        /// 成分毎の「カーブエディタを開く」ボタンを取得する。
        /// リストの並びは成分 ID 順。(ゼロ開始＆欠けなし)
        /// </summary>
        protected System.Collections.Generic.IReadOnlyList<DropDownButton> ComponentCurveEditorOpeningButtons
        {
            get { return componentCurveEditorOpeningButtons; }
        }
        private List<DropDownButton> componentCurveEditorOpeningButtons = new List<DropDownButton>();

        protected static readonly System.Drawing.Size ComponentCurveEditorOpeningButtonDefaultSize = new System.Drawing.Size(36, 18);

        private static readonly Image curveImage = App.Properties.Resources.PropertyEdit_Curve;

        private static readonly Image duplicatedCurveImage = App.Properties.Resources.PropertyEdit_DuplicatedCurve;

        private static readonly Image noCurveImage = App.Properties.Resources.PropertyEdit_NoCurve;

        /// <summary>
        /// カーブエディタへのリンクボタンの作成
        /// </summary>
        public void InitializeDropDownButton()
        {
            dropDownButton1 = new DropDownButton();
            dropDownButton1.Image = curveImage;
            dropDownButton1.ImageAlign = System.Drawing.ContentAlignment.MiddleLeft;
            dropDownButton1.Location = new System.Drawing.Point(64, 2);
            dropDownButton1.Name = "dropDownButton1";
            dropDownButton1.Size = new System.Drawing.Size(36, 18);
            dropDownButton1.TabIndex = 0;
            Controls.Add(dropDownButton1);

            componentCurveEditorOpeningButtons.Clear();
            for (int i = 0; i < numComponentCurveEditorOpeningButtons; i++)
            {
                var dropDownButton = new DropDownButton()
                {
                    Image = curveImage,
                    ImageAlign = System.Drawing.ContentAlignment.MiddleLeft,
                    Size = ComponentCurveEditorOpeningButtonDefaultSize,
                    TabStop = false
                };
                componentCurveEditorOpeningButtons.Add(dropDownButton);
            }
        }

        /// <summary>
        /// 表示位置の更新
        /// </summary>
        public virtual void UpdateDropDownButtonLocation()
        {
            if (dropDownButton1 != null)
            {
                dropDownButton1.Left = lblParameter.Right + 5;
            }

            if (labelHelper.linkTarget != null)
            {
                if (btnLink == null)
                {
                    btnLink = LabelHelper.CreateLinkButton();
                    btnLink.Click += (s, e) =>
                    {
                        if (LinkClicked != null)
                        {
                            LinkClicked(this, labelHelper.ClickArgs());
                        }
                    };
                    Controls.Add(btnLink);
                }

                if (dropDownButton1 != null)
                {
                    btnLink.Left = dropDownButton1.Right + 5;
                }
                else
                {
                    btnLink.Left = lblParameter.Right + 5;
                }

                btnLink.Enabled = !labelHelper.LinkError;
            }
        }

        /// <summary>
        /// マテリアル
        /// </summary>
        internal Material material;

        /// <summary>
        /// コンテキストメニューを設定する。
        /// </summary>
        public void SetContextMenu(IHasShaderParameterAnimation[] allAnimations, AnimationSet[] animationSets, Material material, ObjectPropertyDialog owner)
        {
            var animations = FilterAnimation(allAnimations).OfType<AnimationDocument>().ToArray();

            Action<DropDownButton, int?> setup = (dropDownButton, componentIndex) =>
            {
                var hasCurveAnimations = animations.Where(x =>
                {
                    var anim = ((IHasShaderParameterAnimation)x).GetParamAnimFromId(material.Name, ParamName);
                    return anim != null && anim.ParamAnimTargets.Any(y =>
                    {
                        var type = y.ExportType;
                        return (componentIndex.HasValue ? (y.component_index == componentIndex) : true) && (type == CurveExportType.Constant || type == CurveExportType.Curve);
                    });
                }).ToArray();

                var duplicatedAnimations = animationSets.SelectMany(x =>
                {
                    var intersect = hasCurveAnimations.OfType<AnimationDocument>().Where(y => x.Animations.Contains(new AnimationSetItem(y.FileName, y.FileLocation))).ToArray();
                    if (intersect.Length > 1)
                    {
                        return intersect;
                    }

                    return Enumerable.Empty<AnimationDocument>();
                }).Distinct().ToArray();

                // メニューの作成を遅延処理する。
                dropDownButton.BeforeShowMenu = () =>
                {
                    var menuStrip = dropDownButton.DropDownMenu;

                    // アニメーションが一つのとき
                    if (animations.Length == 1)
                    {
                        if (menuStrip != null)
                        {
                            menuStrip.Items.Clear();
                        }

                        ShowAnimationPropertyDialog(animations[0], componentIndex);
                        return;
                    }

                    if (menuStrip != null)
                    {
                        menuStrip.Items.Clear();
                    }
                    else
                    {
                        menuStrip = new UIContextMenuStrip();
                    }

                    // 項目の設定
                    var items = new List<UIToolStripMenuItem>();
                    foreach (var anim in animations.OfType<AnimationDocument>())
                    {
                        var item = new UIToolStripMenuItem() { Text = anim.FileName };
                        item.Tag = anim;
                        item.Click += AnimationMenuClicked;
                        if (duplicatedAnimations.Contains(anim))
                        {
                            item.Image = duplicatedCurveImage;
                        }
                        else if (hasCurveAnimations.Contains(anim))
                        {
                            item.Image = curveImage;
                        }
                        else
                        {
                            item.Image = noCurveImage;
                        }

                        items.Add(item);
                    }
                    menuStrip.Items.AddRange(items.OfType<ToolStripItem>().ToArray());

                    dropDownButton.DropDownMenu = menuStrip;
                };

                dropDownButton.ShowDropDownMark = animations.Length > 1;
                dropDownButton.Enabled = animations.Length > 0 && CanEnabled;
                if (duplicatedAnimations.Any())
                {
                    dropDownButton.Image = duplicatedCurveImage;
                }
                else if (hasCurveAnimations.Any())
                {
                    dropDownButton.Image = curveImage;
                }
                else
                {
                    dropDownButton.Image = noCurveImage;
                }
            };

            setup(dropDownButton1, null);
            for (int i = 0; i < ComponentCurveEditorOpeningButtons.Count; i++)
            {
                setup(ComponentCurveEditorOpeningButtons[i], i);
            }
        }

        public virtual void UpdateEnabled(bool value)
        {
            value = value && CanEnabled && IsValueEditable() && material.ParentMaterialsValidity();
            foreach (Control item in Controls)
            {
                if (item == lblParameter || item == dropDownButton1 || ComponentCurveEditorOpeningButtons.Contains(item) || item is LinkButton)
                {
                    continue;
                }
                item.Enabled = value;
            }
        }

        /// <summary>
        /// ユニフォームの種類に応じてアニメーションをフィルタする。
        /// </summary>
        public virtual IEnumerable<IHasShaderParameterAnimation> FilterAnimation(IEnumerable<IHasShaderParameterAnimation> animations)
        {
            return animations.Where(x =>
            {
                var anim = (IHasShaderParameterAnimation) x;
                return anim != null && anim.IsShaderParameterType();
            });
        }

        /// <summary>
        /// メニュークリックイベントハンドラ
        /// </summary>
        private void AnimationMenuClicked(object sender, EventArgs e)
        {
            var anim = (AnimationDocument) ((ToolStripMenuItem) sender).Tag;
            ShowAnimationPropertyDialog(anim, null);
        }

        /// <summary>
        /// ダイアログを表示
        /// </summary>
        public void ShowAnimationPropertyDialog(AnimationDocument anim, int? componentIndex)
        {
            // 開く
            using (var wait = new WaitCursor())
            {
                // プロパティウィンドウを生成
                var objectGroup = new GuiObjectGroup(anim);

                ObjectPropertyDialog dialog = new ObjectPropertyDialog(null, objectGroup);

                // 現在のウィンドウの上に表示する
                Control parent = dialog;
                while (parent != null)
                {
                    if (parent is ObjectPropertyDialog)
                    {
                        dialog.SetLocationNextTo((ObjectPropertyDialog) parent);
                        break;
                    }

                    parent = parent.Parent;
                }

                // ページの表示設定
                IHasShaderParameterCurvePage panel = null;
                switch (anim.ObjectID)
                {
                    case ConfigCommon.GuiObjectID.ShaderParameterAnimation:
                    {
                        panel = (IHasShaderParameterCurvePage) dialog.GetPropertyPanel(ObjectPropertyDialog.PanelID.ShaderParameterAnimation);
                    }
                        break;
                    case ConfigCommon.GuiObjectID.ColorAnimation:
                    {
                        panel = (IHasShaderParameterCurvePage) dialog.GetPropertyPanel(ObjectPropertyDialog.PanelID.ColorAnimation);
                    }
                        break;
                    case ConfigCommon.GuiObjectID.TextureSrtAnimation:
                    {
                        panel = (IHasShaderParameterCurvePage) dialog.GetPropertyPanel(ObjectPropertyDialog.PanelID.TextureSrtAnimation);
                    }
                        break;
                    case ConfigCommon.GuiObjectID.MaterialAnimation:
                    {
                        panel = (IHasShaderParameterCurvePage) dialog.GetPropertyPanel(ObjectPropertyDialog.PanelID.MaterialAnimation);
                    }
                        break;
                }

                dialog.Show();

                // 先に選択するとツールストリップのレイアウトが崩れるので Show のあと
                {
                    // 選択されているマテリアル。
                    var selectedMaterials = App.AppContext.SelectedTarget.GetObjects(ConfigCommon.GuiObjectID.Material).Select(x => x as Material);

                    // 現時点のカーブエディタでは、ひとつのファイルしか開けないので、
                    // このページのマテリアルのファイルと同じものに限定。
                    var sameFileMaterials = selectedMaterials.Where(x => x.OwnerDocument == material.OwnerDocument);

                    // リスト先頭のマテリアルカーブがツリーノードで選択状態となるので
                    // このページのマテリアルを先頭に配置しておく。
                    var materialNames = sameFileMaterials.Select(x => x.Name).Distinct();
                    materialNames = Enumerable.Repeat(material.Name, 1).Concat(materialNames.Where(x => x != material.Name));

                    panel.SelectShaderParameterCurvePage(materialNames, ParamName, componentIndex);
                }

                // SelectCurvePage の後で固定表示を設定
                dialog.TargetFixed = true;

                // カーブにフィットする
                panel.FitCurvePage();
            }
        }

        #endregion

        protected void InvokeValueChanged(object sender, ValueChangedEventArgs e)
        {
            if (ValueChanged != null)
            {
                ValueChanged(this, e);
            }
        }


        public virtual bool SetLabel(string label, CustomUI customUI, Definition.ShadingModelTable table, Predicate<string> visibleGroups, HashSet<string> visiblePages, bool showId, bool showOriginalLabel, Material.ValueResolvedState? valueResolvedState, bool isParentCustomLabel)
        {
            var update = labelHelper.SetLabel(ParamName, ParamLabel, label, customUI, table, visibleGroups, visiblePages, showId, showOriginalLabel, isParentCustomLabel);
            if (valueResolvedState.HasValue)
            {
                //var icons = GetReferenceStateIcons(GetMaterialReferenceBehaviorItem());
                var icons = GetReferenceStateIcons(valueResolvedState.Value, GetMaterialReferenceBehaviorItem());

                var tailImages = lblParameter.TailImages;
                if ((tailImages?.Count != icons?.Count) ||
                    (tailImages?.All(x => icons?.Contains(x) == true) != true)
                )
                {
                    lblParameter.TailImages = icons;
                    update = true;
                }
            }

            if (update)
            {
                UpdateLabel();
            }
            return update;
        }

        /// <summary>
        /// abstractにしたいがデザイナで開けなくなるのでvirtualにしています。
        /// サイズが変更されたかどうか
        /// </summary>
        public virtual bool SetValue(Material material, string value, CustomUI customUI, Definition.ShadingModelTable table, Predicate<string> visibleGroups, HashSet<string> visiblePages, bool showId, bool showOriginalLabel)
        {
            Debug.Assert(false);
            return false;
        }

        public Rectangle ChildrenBound
        {
            get
            {
                bool first = true;
                Rectangle rect = new Rectangle();
                foreach (Control child in Controls)
                {
                    var bound = new Rectangle(child.Location, child.Size);
                    if (first)
                    {
                        rect = bound;
                        first = false;
                    }
                    else
                    {
                        rect = Rectangle.Union(rect, bound);
                    }
                }
                return rect;
            }
        }

        public const int DefaultWidth = 430;

        public virtual void Align()
        {
            ;
        }

        /// <summary>
        /// ヒントの設定
        /// 詳しい理由は不明だが、親コントロールに追加された後に呼ばないと表示されない。
        /// </summary>
        public virtual void SetHint(HintToolTip hint)
        {
            //DebugConsole.WriteLine("SetHint");
            parentHint = hint;

            if (dropDownButton1 != null)
            {
                try
                {
                    hint.SetToolTip(dropDownButton1, res.Strings.ShaderParamControl_ShowCurveEditor);
                }
                catch
                {
                    Debug.Assert(false);
                }
            }

            foreach (var button in ComponentCurveEditorOpeningButtons)
            {
                try
                {
                    hint.SetToolTip(button, res.Strings.ShaderParamControl_ShowCurveEditor);
                }
                catch
                {
                    Debug.Assert(false);
                }
            }

            if (!string.IsNullOrWhiteSpace(ParamComment))
            {
                try
                {
                    hint.SetToolTip(lblParameter, ParamComment);
                }
                catch
                {
                    Debug.Assert(false);
                }
            }

            UpdateLinkToolTip(true);
        }

        #region ツールチップヒント関係

        protected HintToolTip parentHint;

        private bool showHint = false;

        public static Timer timer;

        static ShaderParamControl()
        {
            timer = new Timer()
            {
                Interval = 500, Enabled = false,
            };
            timer.Tick += (s, e) =>
            {
                timer.Stop();
                if (showToolTip != null)
                {
                    showToolTip();
                }
            };
        }

        public static Control toolTipTarget;

        public static Action showToolTip;

        protected override void OnMouseMove(MouseEventArgs e)
        {
            // ボタンが無効のときにツールチップが表示されないので自前で表示する
            var child = GetChildAtPoint(e.Location);
            if ((dropDownButton1 == child || child is LinkButton) && parentHint != null && !showHint)
            {
                toolTipTarget = child;
                StartToolTip();
            }
            else if (showHint && toolTipTarget != child)
            {
                StopToolTip();
            }

            base.OnMouseMove(e);
        }

        protected override void OnMouseLeave(EventArgs e)
        {
            if (showHint)
            {
                StopToolTip();
            }
            base.OnMouseLeave(e);
        }

        private void StartToolTip()
        {
            showToolTip = (Action) (() =>
            {
                // TODO: プロパティウィンドウにフォーカスがないと表示されない
                // 座標も指定しないと無効なコントロール上で一瞬で消える
                if (toolTipTarget != null)
                {
                    var pt = toolTipTarget.PointToClient(Cursor.Position);
                    var text = parentHint.GetToolTip(toolTipTarget);
                    if (!string.IsNullOrEmpty(text))
                    {
                        parentHint.Show(text, toolTipTarget, pt.X, pt.Y + toolTipTarget.Height);
                    }
                }
            });
            timer.Start();
            showHint = true;
        }

        private void StopToolTip()
        {
            timer.Stop();
            if (toolTipTarget != null)
            {
                parentHint.Hide(toolTipTarget);
            }
            showHint = false;
            toolTipTarget = null;
        }

        #endregion

        public void Fit()
        {
            var rect = ChildrenBound;
            Size = new Size(rect.Right, rect.Bottom);
        }

        public virtual bool IsModified
        {
            set { lblParameter.IsModified = value; }
            get { return lblParameter.IsModified; }
        }

        /// <summary>
        /// グループが有効のときに編集可能になるかどうか
        /// </summary>
        public bool CanEnabled = true;

        private static void cmiCopyID_Click(object sender, EventArgs e)
        {
            var control = GetSourceControl();
            Debug.Assert(control != null, "control != null");
            var copyString = control?.ParamName ?? "";
            DebugConsole.WriteLine("Copy ID:[{0}]", copyString);
            App.Utility.ClipboardUtility.SetDataObject(copyString);
        }

        private static void cmiEditLabel_Click(object sender, EventArgs e)
        {
            var control = GetSourceControl();
            Debug.Assert(control != null, "control != null");
            bool showName = control.ParamType == ShaderParamControls.ParamType.render_info_slot;
            using (var dialog = new EditLabelDialog(control.ParamName, control.ParamLabel, control.labelHelper.CustumLabelForEdit, showName))
            {
                var result = dialog.ShowDialog();
                if (result == DialogResult.OK)
                {
                    Debug.Assert(control.LabelChanged != null);
                    if (control.LabelChanged != null)
                    {
                        control.LabelChanged(control, new LabelChangedEventArgs()
                        {
                            ParamName = control.ParamName, Lebel = dialog.Value, ParamType = control.ParamType,
                        });
                    }
                }
            }
        }

        internal static void UpdateControls(ShaderParamControl control)
        {
            UpdateControlGroup(control?.Parent as ShaderParamControlGroup);
        }

        internal static void UpdateControlGroup(ShaderParamControlGroup controlGroup)
        {
            if (controlGroup?.ParentMaterialShaderPage != null)
            {
                var materialShaderPage = controlGroup.ParentMaterialShaderPage;
                materialShaderPage.UpdateControls = true;
                ((MaterialPropertyPanel)materialShaderPage.Owner).UpdateCategoryView();
                materialShaderPage.UpdateForm(false, false);
            }
        }

        private static void cmiMaterialReferenceBehaviorValue_Click(object sender, EventArgs e)
        {
            var control = GetSourceControl();
            Debug.Assert(control != null, "control != null");

            var index = CheckSelected(ReferenceBehaviorValueMenuItems, sender);
            var referenceBehaviorItem = control.GetMaterialReferenceBehaviorItem() ??
                                    new MaterialReferenceBehaviorItem()
                                    {
                                        Id = control.ParamName,
                                        ChildRestriction = ShaderItemRestrictionState.None,
                                    };
            referenceBehaviorItem.Value = (ShaderItemValueState)index;
            control.SetMaterialReferenceBehaviorItem(referenceBehaviorItem);
            App.AppContext.OnMaterialReferenceBehaviorSettingChanged();
            UpdateControls(control);
        }

        private static void cmiMaterialReferenceBehaviorRestriction_Click(object sender, EventArgs e)
        {
            var control = GetSourceControl();
            Debug.Assert(control != null, "control != null");

            var index = CheckSelected(ReferenceBehaviorRestrictionMenuItems, sender);
            var referenceBehaviorItem = control.GetMaterialReferenceBehaviorItem() ??
                                    new MaterialReferenceBehaviorItem()
                                    {
                                        Id = control.ParamName,
                                        Value = ShaderItemValueState.Refer
                                    };
            referenceBehaviorItem.ChildRestriction = (ShaderItemRestrictionState)index;
            control.SetMaterialReferenceBehaviorItem(referenceBehaviorItem);
            App.AppContext.OnMaterialReferenceBehaviorSettingChanged();
            UpdateControls(control);
        }

        private static int CheckSelected(ToolStripMenuItem[] menuItems, object sender)
        {
            var index = 0;
            for (var i = 0; i < menuItems.Length; i++)
            {
                var menuItem = menuItems[i];
                if (menuItem == sender)
                {
                    menuItem.Checked = true;
                    index = i;
                }
                else
                {
                    menuItem.Checked = false;
                }
            }
            return index;
        }

        /// <summary>
        /// 破棄またはキャッシュされる直前
        /// </summary>
        public virtual void OnBeforeDisposedOrCached()
        {
        }

        public string GroupName { get; set; }
    }

    public interface IGroupName
    {
        string GroupName { get; }
    }
}
