﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;
using App.Command;
using App.Controls;
using App.Data;
using App.Properties;
using App.Utility;
using App.res;
using App.ConfigData;
using App.ObjectView.List;
using ConfigCommon;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;

namespace App.PropertyEdit.ShaderParamControls
{
    using App.FileView;

    using nw.g3d.iflib.nw3de;

    [ToolboxItem(true)]
    public partial class ShaderParamControlGroup : UIUserControl
    {
        public string CategoryName { get; set; }
        public event ValueChangedEventHandler ValueChanged	= null;
        public event LabelChangedEventHandler LabelChanged = null;
        public event ColorControlChangedEventHandler ColorControlChanged = null;
        public event LinkClickedEventHandler LinkClicked = null;
        public event EventHandler EnterGroup = null;
        public event EventHandler EditGroupLabel = null;
        public MaterialShaderPage ParentMaterialShaderPage { get; set; }


        //public event Action FixParameters = null;
        private readonly Dictionary<ShaderParamControl, object> CacheKeys = new Dictionary<ShaderParamControl,object>();
        private readonly Dictionary<sampler_varType, ShaderParamControl> samplerControlCache_ = new Dictionary<sampler_varType, ShaderParamControl>(new ShaderTypeUtility.SamplerVarComparer());
        private readonly Dictionary<ShaderTypeUtility.UniformUI, ShaderParamControl> uniformControlCache_ = new Dictionary<ShaderTypeUtility.UniformUI, ShaderParamControl>(new ShaderTypeUtility.UniformUIComparer());
        private readonly Dictionary<option_varType, ShaderParamControl> optionControlCache_ = new Dictionary<option_varType, ShaderParamControl>(new ShaderTypeUtility.OptionVarComparer());
        private readonly Dictionary<attrib_varType, ShaderParamControl> attribControlCache_ = new Dictionary<attrib_varType, ShaderParamControl>(new ShaderTypeUtility.AttribComparer());
        private readonly Dictionary<render_info_slotType, EditRenderInfoBase> renderInfoCache_ = new Dictionary<render_info_slotType, EditRenderInfoBase>(new ShaderTypeUtility.RenderInfoSlotComparer());
        private readonly Dictionary<string, ShaderParamControl> samplerControls = new Dictionary<string, ShaderParamControl>();
        private readonly Dictionary<string, ShaderParamControl> uniformControls = new Dictionary<string, ShaderParamControl>();
        private readonly Dictionary<string, ShaderParamControl> optionControls = new Dictionary<string, ShaderParamControl>();
        private readonly Dictionary<string, ShaderParamControl> attribControls = new Dictionary<string, ShaderParamControl>();
        private readonly Dictionary<string, ShaderParamControl> renderInfoControls = new Dictionary<string, ShaderParamControl>();
        private Dictionary<string, ShaderGroupBox> visibleGroupBoxes = new Dictionary<string, ShaderGroupBox>();
        private readonly Dictionary<ShaderGroupBox, Control[]> groupBoxControls = new Dictionary<ShaderGroupBox, Control[]>();

        private readonly Dictionary<string, ShaderGroupBox> attribToGroup = new Dictionary<string, ShaderGroupBox>();
        private readonly Dictionary<string, ShaderGroupBox> samplerToGroup = new Dictionary<string, ShaderGroupBox>();
        private readonly Dictionary<string, ShaderGroupBox> uniformToGroup = new Dictionary<string, ShaderGroupBox>();
        private readonly Dictionary<string, ShaderGroupBox> optionToGroup = new Dictionary<string, ShaderGroupBox>();
        private readonly Dictionary<string, ShaderGroupBox> renderInfoToGroup = new Dictionary<string, ShaderGroupBox>();
        private Predicate<string> isGroupVisible;

        private CommentLabel commentLabel = null;

        public IEnumerable<ShaderParamControl> ShaderParamControls
        {
            get
            {
                return
                    samplerControls.Values.
                    Concat(uniformControls.Values).
                    Concat(optionControls.Values).
                    Concat(attribControls.Values).
                    Concat(renderInfoControls.Values);
            }
        }

        // アクティブターゲットの取得
        public Func<Material> GetActiveTarget;

        // ターゲットの所得
        public Func<GuiObjectGroup> GetTargets;

        public HintToolTip hint;
        public ShaderParamControlGroup()
        {
            InitializeComponent();

            cmiCopy.Text  = Strings.ShaderParamControl_GroupCopy;
            cmiPaste.Text = Strings.ShaderParamControl_GroupPaste;
            cmiDefault.Text = Strings.ShaderParamControl_GroupDefault;
            cmiShowMaterialList.Text = Strings.ShaderParamControl_GroupShowMaterialList;
            cmiHideMaterialList.Text = Strings.ShaderParamControl_GroupHideMaterialList;
            cmiCustomLabel.Text = Strings.ShaderParamControl_CustomLabel;
            // マテリアル参照メニューのテキストもコピー
            var treeViewMenu = TheApp.MainFrame?.TreeViewMenu;
            if (!DesignMode && treeViewMenu != null)
            {
                cmiMaterialReferenceBehaviorValue.Text = treeViewMenu.cmiMaterialReferenceBehaviorValue.Text;
                cmiMaterialReferenceBehaviorValue_Default.Text = treeViewMenu.cmiMaterialReferenceBehaviorValue_Default.Text;
                cmiMaterialReferenceBehaviorValue_Override.Text = treeViewMenu.cmiMaterialReferenceBehaviorValue_Override.Text;
                cmiMaterialReferenceBehaviorValue_Refer.Text = treeViewMenu.cmiMaterialReferenceBehaviorValue_Refer.Text;
                cmiMaterialReferenceBehaviorRestriction.Text = treeViewMenu.cmiMaterialReferenceBehaviorRestriction.Text;
                cmiMaterialReferenceBehaviorRestriction_None.Text = treeViewMenu.cmiMaterialReferenceBehaviorRestriction_None.Text;
                cmiMaterialReferenceBehaviorRestriction_ForceRefer.Text = treeViewMenu.cmiMaterialReferenceBehaviorRestriction_ForceRefer.Text;
                cmiMaterialReferenceBehaviorRestriction_ForceOverride.Text = treeViewMenu.cmiMaterialReferenceBehaviorRestriction_ForceOverride.Text;
            }
            Disposed += ShaderParamControlGroup_Disposed;
        }

        private void ShaderParamControlGroup_Disposed(object sender, EventArgs e)
        {
            foreach (var key in CacheKeys.Keys.Except(ShaderParamControls))
            {
                key.Dispose();
            }

            if (hint != null)
            {
                hint.Dispose();
            }
        }

        [Browsable(true)]
        public bool MultiColumn { get; set; }

        [Browsable(true)]
        public bool MultiColumnInGroup { get; set; }
        private Control[] orderedControls = new Control[0];

        private bool ShowParamId = false;

        private bool lastFixedSlider = false;

        private Label LackResource = null;

        /// <summary>
        /// シェーダーグループボックスに表示する通知アイコン。
        /// UIGroupBox.ShowImage() と UIGroupBox.Hide() の image には同じインスタンスを使う必要がある。
        /// App.Properties.Resources で取得する画像インスタンスは呼び出し毎に変わるので、このクラスはそれを回避するためのもの。
        /// </summary>
        class NotificationIcon
        {
            /// <summary>
            /// condition 式の評価失敗
            /// </summary>
            public static readonly Bitmap ConditionEvalFailure = Resources.Merge_MergeTargetSelectWarning;
        }

        class CommentLabel : UILabel
        {
            public CommentLabel()
            {
                AutoSize = false;
                BackColor = SystemColors.GradientInactiveCaption;
                Anchor = AnchorStyles.Top | AnchorStyles.Left | AnchorStyles.Right;
            }

            public void UpdateSize(Size proposedSize)
            {
                using (var g = CreateGraphics())
                {
                    var txtSize = TextRenderer.MeasureText(g, Text, Font, proposedSize, TextFormatFlags.WordBreak);

                    var oAnchor = Anchor;
                    Anchor &= ~(AnchorStyles.Right | AnchorStyles.Bottom);
                    Size = new Size(txtSize.Width, txtSize.Height);
                    Anchor = oAnchor;
                }
            }
        }

        /// <summary>
        /// コントロールの再配置
        /// リソース不足の場合は false
        /// </summary>
        public bool UpdateControls(
            Material material,
            shading_modelType shadingModel,
            bool showConditionVariable,
            bool pageMode,
            pageType page,
            bool showParamId)
        {
            ShowParamId = showParamId;
            var visibleInfo = new VisibleInfo(material);
            var oldControls = ShaderParamControls.ToArray();
            samplerControls.Clear();
            uniformControls.Clear();
            optionControls.Clear();
            attribControls.Clear();
            renderInfoControls.Clear();

            // リソースの解放
            if (lastFixedSlider != ConfigData.ApplicationConfig.UserSetting.UI.FixedSlider)
            {
                lastFixedSlider = ConfigData.ApplicationConfig.UserSetting.UI.FixedSlider;
                ClearCacheAll();
            }

            if (hint != null)
            {
                hint.RemoveAll();
                // 以前のコメントが表示されないようにする
                hint.Dispose();
            }

            if (!CheckResources())
            {
                Controls.Clear();
                foreach (var groupBox in visibleGroupBoxes.Values)
                {
                    groupBox.Controls.Clear();
                    groupBox.Dispose();
                }
                visibleGroupBoxes.Clear();
                foreach (var control in oldControls)
                {
                    DisposeOrCacheControl(control);
                }
                oldControls = new ShaderParamControl[0];
                ClearCacheAll();

                if (!CheckResources())
                {
                    if (LackResource == null)
                    {
                        LackResource = new Label() { AutoSize = true, Text = res.Strings.ObjectPropertyPanel_LackResources };

                        // とりあえずハードコード
                        LackResource.MinimumSize = new Size(0, 60);
                    }
                    Controls.Add(LackResource);
                    orderedControls = Controls.OfType<Control>().ToArray();
                    return false;
                }
            }

            if (LackResource != null)
            {
                Controls.Remove(LackResource);
                LackResource.Dispose();
                LackResource = null;
            }

            SuspendLayout();

            hint = new HintToolTip();

            //this.Enabled = isConsistent;
            // タブインデックスが等しければ、親コントロールにおけるインデックス順になるはず。
            // XPだとその通りの動作にならないのでIndex を使ている。子control を使いまわしているのが原因?
            int tabIndex = 0;

            var oldCommentLabel = commentLabel;
            if (page != null && !string.IsNullOrEmpty(page.Comment()))
            {
                var pageComment = page.Comment();

                // ページコメントのラベルを新規作成もしくは作成済みのラベルの文字列を置き換える。
                if (commentLabel == null)
                {
                    // ラベルを新規作成。
                    commentLabel = new CommentLabel()
                    {
                        Text = pageComment
                    };
                }
                else if (commentLabel.Text != pageComment)
                {
                    // ラベルの文字列を更新。
                    commentLabel.Text = pageComment;
                    oldCommentLabel = null;
                }

                commentLabel.TabIndex = tabIndex++;
            }
            else if (commentLabel != null)
            {
                commentLabel = null;
            }

            var groupTypes = material.MaterialShaderAssign.ShadingModel != null ?
                material.MaterialShaderAssign.ShadingModel.Groups().ToDictionary(x => x.name) :
                new Dictionary<string, groupType>();

            Func<string, string> toParent = x => {
                groupType g;
                if (groupTypes.TryGetValue(x, out g))
                {
                    return g.Group();
                }

                return string.Empty;
            };

            var table = new Dictionary<string, bool>();
            Predicate<string> isAllAncesterEnabled = x => IsAllAncesterTrue(x, toParent, visibleInfo.IsVisible, table);

            if (showConditionVariable)
            {
                if (material.MaterialShaderAssign.ShadingModel != null)
                {
                    var groupsWithConditionVariable = new HashSet<string>();
                    var groupsWithConditionOptionVariable = new HashSet<string>(
                        material.MaterialShaderAssign.ShadingModel.Options().Where(x => visibleInfo.IsConditionVariable(x)).Select(x => x.Group())
                        );
                    var groupsWithConditionUniformVariable = new HashSet<string>(
                        material.MaterialShaderAssign.ShadingModel.Uniforms().Where(x => visibleInfo.IsConditionVariable(x)).Select(x => x.Group())
                        );
                    var groupsWithConditionRenderInfoVariable = new HashSet<string>(
                        material.MaterialShaderAssign.ShadingModel.RenderInfoSlots().Where(x => visibleInfo.IsConditionVariable(x)).Select(x => x.Group())
                        );
                    Action<HashSet<string>> addVariables = x =>
                    {
                        foreach (var group in x)
                        {
                            if (!groupsWithConditionVariable.Contains(group))
                            {
                                groupsWithConditionVariable.Add(group);
                            }
                        }
                    };
                    addVariables(groupsWithConditionOptionVariable);
                    addVariables(groupsWithConditionUniformVariable);
                    addVariables(groupsWithConditionRenderInfoVariable);

                    var toChildren = TreeUtility.Revert(
                        x => Enumerable.Repeat(toParent(x), 1),
                        material.MaterialShaderAssign.ShadingModel.Groups().Select(x => x.name));

                    var hasConditionVariable = TreeUtility.PreOrderDP(groupsWithConditionVariable.Contains, toChildren, x => x.Any(y => y));

                    isGroupVisible = x => isAllAncesterEnabled(x) || hasConditionVariable(x);
                }
            }
            else
            {
                isGroupVisible = isAllAncesterEnabled;
            }
            var allControls = GetControls(material, shadingModel, visibleInfo, showConditionVariable, pageMode, page, isAllAncesterEnabled, toParent).ToArray();

            //var groups = allControls.GroupBy(x => x.group).ToDictionary(x => x.Key);

            var groups = (from param in allControls
                          orderby param.order
                          group param by param.@group into g
                          select g).ToDictionary(x => x.Key);



            //var items = controls.Where(x => x.group == "")
            //	.OrderBy(x => x.order).Select(x => x.control);
            var oldVisibleGroupBoxes = visibleGroupBoxes;
            visibleGroupBoxes = new Dictionary<string, ShaderGroupBox>();
            int height = 0;
            using (var graphics = CreateGraphics())
            {
                Func<string, Control[], ShaderGroupBox> updateShaderGroupBox = (name, items) =>
                {
                    ShaderGroupBox groupBox;
                    if (!oldVisibleGroupBoxes.TryGetValue(name, out groupBox))
                    {
                        groupBox = new ShaderGroupBox();
                        if (EnterGroup != null)
                        {
                            groupBox.Enter += EnterGroup;
                        }
                        groupBox.ContextMenuStrip = groupContextMenuStrip.ProxyContextMenuStrip;
                        groupBox.LinkClicked += LinkClicked;
                        groupBox.ParamName = name;
                        groupBox.Anchor = AnchorStyles.Top | AnchorStyles.Left;
                    }

                    visibleGroupBoxes[name] = groupBox;

                    groupBox.SuspendLayout();
                    groupBox.Tag = name;
                    if (visibleInfo.IsVisible(name))
                    {
                        if (isAllAncesterEnabled(name))
                        {
                            groupBox.State = ShaderGroupBox.GroupState.Enabled;
                        }
                        else
                        {
                            groupBox.State = ShaderGroupBox.GroupState.ParentDisabled;
                        }
                    }
                    else
                    {
                        groupBox.State = ShaderGroupBox.GroupState.Disabled;
                    }
                    groupBox.TabIndex = tabIndex++;

                    groupType groupType;
                    groupTypes.TryGetValue(name, out groupType);
                    if (groupType != null)
                    {
                        groupBox.ParamLabel = groupType.Label();
                        groupBox.GroupName = groupType.Group();
                    }
                    else
                    {
                        groupBox.ParamLabel = null;
                        groupBox.GroupName = string.Empty;
                    }

                    groupBox.SetHint(hint, groupType != null ? groupType.Comment() : null);

                    Debug.Assert(items.All(x => x != null));
                    var visibleItems = new HashSet<Control>(items);
                    foreach (var control in items.OfType<ShaderParamControl>())
                    {
                        control.SetHint(hint);
                        control.Align();
                        control.Fit();
                        control.TabIndex = tabIndex++;
                    }

                    groupBoxControls[groupBox] = items;

                    for (int i = groupBox.Controls.Count - 1; i >= 0; i--)
                    {
                        Control control = groupBox.Controls[i];
                        if (visibleItems.Contains(control))
                        {
                            visibleItems.Remove(control);
                        }
                        else if (control is ShaderParamControl || control is ShaderGroupBox)
                        {
                            groupBox.Controls.RemoveAt(i);
                        }
                    }

                    // 最小サイズを決める
                    var textSize = graphics.MeasureString(groupBox.Text, groupBox.Font);
                    groupBox.MinimumSize = new Size((int)textSize.Width + 20, (int)textSize.Height);
                    if (visibleItems.Count > 0)
                    {
                        groupBox.Controls.AddRange(visibleItems.ToArray());
                    }
                    groupBox.ResumeLayout();
                    return groupBox;
                };

                IGrouping<string, Parameter> root;
                if (groups.TryGetValue(string.Empty, out root))
                {
                    //var controls = root.SelectMany(x => GroupToControl(x, groups, oldVisibleGroupBoxes, updateShaderGroupBox)).ToArray();

                    // コメントラベルはリストの先頭にあることが前提。詳細は UpdateLayout() の hasCommentLabel を参照。
                    var controls = new List<Control>(System.Linq.Enumerable.Repeat(commentLabel, (commentLabel != null) ? 1 : 0));
                    var orderedRoots = from p in root
                                       let isParam = !(p is GroupParamter)
                                       let groupName = !isParam ? ((GroupParamter)p).Data.name : ""
                                       let gt = groupName != "" ? groupTypes[groupName] : null
                                       let order = gt != null ? gt.Order() : 0
                                       let index = gt != null ? gt.index : int.MinValue
                                       orderby isParam, order, index, groupName
                                       select p;

                    foreach (var item in orderedRoots)
                    {
                        if (item is GroupParamter)
                        {
                            var group = (GroupParamter)item;
                            IGrouping<string, Parameter> grouping;
                            if (groups.TryGetValue(group.Data.name, out grouping))
                            {
                                controls.AddRange(GroupToControl(grouping, groups, oldVisibleGroupBoxes, groupTypes, updateShaderGroupBox));
                            }
                        }
                        else
                        {
                            item.control.SetHint(hint);
                            item.control.Align();
                            item.control.Fit();
                            item.control.TabIndex = tabIndex++;
                            controls.Add(item.control);
                        }
                    }

                    orderedControls = controls.ToArray();
                    int lowerBound;
                    UpdateLayout(orderedControls, Size.Width, 2, 2, out lowerBound, out height, MultiColumn, false);
                    var visibleItems = new HashSet<Control>(controls);

                    for (int i = Controls.Count - 1; i >= 0; i--)
                    {
                        Control control = Controls[i];
                        if (visibleItems.Contains(control))
                        {
                            visibleItems.Remove(control);
                        }
                        else if (control is ShaderParamControl || control is ShaderGroupBox || (control == oldCommentLabel))
                        {
                            Controls.RemoveAt(i);
                        }
                    }

                    if (visibleItems.Count > 0)
                    {
                        Controls.AddRange(visibleItems.ToArray());
                    }
                }
                else
                {
                    Controls.Clear();
                    orderedControls = new Control[0];
                }
            }



            previousWidth = Width;
            Height = height;

            ResumeLayout();

            // 古いコントロールに対する処理
            foreach (var groupBox in oldVisibleGroupBoxes.Values.Except(visibleGroupBoxes.Values))
            {
                groupBox.Controls.Clear();
                groupBox.Dispose();
            }

            foreach (var control in oldControls.Except(ShaderParamControls))
            {
                DisposeOrCacheControl(control);
            }

            ClearCache(allControls.Select(x => x.control));

            // グループボックス
            SetGroupDictionary(shadingModel);

            // 警告マーク
            foreach(var groupBox in visibleGroupBoxes)
            {
                string condition;
                if (visibleInfo.IsConditionEvalSuccess(groupBox.Key, out condition) == false)
                {
                    groupBox.Value.ShowImage(NotificationIcon.ConditionEvalFailure, Strings.ShaderParamControl_FailedConditionEval + "\n--\n" + condition);
                }
                else
                {
                    groupBox.Value.HideImage(NotificationIcon.ConditionEvalFailure);
                }
            }

            return true;
        }

        private IEnumerable<Control> GroupToControl(
            IGrouping<string, Parameter> grouping,
            Dictionary<string, IGrouping<string, Parameter>> dict,
            Dictionary<string, ShaderGroupBox> oldVisibleGroupBoxes,
            Dictionary<string, groupType> nameToGroup,
            Func<string, Control[], ShaderGroupBox> updateGroupBox)
        {
            var name = grouping.Key;

            var controls = grouping.SelectMany(x =>
                {
                    var groupParameter = x as GroupParamter;
                    IGrouping<string, Parameter> child;
                    if (groupParameter != null)
                    {
                        if (dict.TryGetValue(groupParameter.Data.name, out child))
                        {
                            return GroupToControl(child, dict, oldVisibleGroupBoxes, nameToGroup, updateGroupBox);
                        }
                        else
                        {
                            return Enumerable.Empty<Control>();
                        }
                    }
                    else
                    {
                        return Enumerable.Repeat(x.control, 1);
                    }
                }).ToArray();

            if (!controls.Any())
            {
                return Enumerable.Empty<Control>();
            }

            groupType group;
            if (nameToGroup.TryGetValue(name, out group) && (group.ui_label == null && !string.IsNullOrEmpty(group.Group())))
            {
                return controls;
            }

            return Enumerable.Repeat(updateGroupBox(name, controls), 1);
        }

#if false
        public void SetShowParamId(bool showParamId)
        {
            if (showParamId != ShowParamId)
            {
                ShowParamId = showParamId;
                foreach (var control in ShaderParamControls)
                {
                    control.SetShowId(showParamId);
                    control.Align();
                    control.Fit();
                }

                // グループボックスの更新
                var material = ParentMaterialShaderPage.ActiveTarget;
                var groupTypes = material.MaterialShaderAssign.ShadingModel != null ?
                    material.MaterialShaderAssign.ShadingModel.Groups().ToDictionary(x => x.name):
                    new Dictionary<string, groupType>();
                foreach (var groupBox in groupBoxs)
                {
                    var control = groupBox.Value;
                    var groupName = groupBox.Key;
                    groupType group;
                    groupTypes.TryGetValue(groupName, out group);
                    SetGroupBoxLabel(material, group, groupName, control);
                }

                UpdateLayout(true);
            }
        }
#endif
        public void SetGroupDictionary(shading_modelType shadingModel)
        {
            attribToGroup.Clear();
            foreach (var attrib in shadingModel.Attributes())
            {
                ShaderGroupBox groupBox;
                if (visibleGroupBoxes.TryGetValue(attrib.Group(), out groupBox))
                {
                    attribToGroup.Add(attrib.id, groupBox);
                }
            }

            samplerToGroup.Clear();
            foreach (var sampler in shadingModel.Samplers())
            {
                ShaderGroupBox groupBox;
                if (visibleGroupBoxes.TryGetValue(sampler.Group(), out groupBox))
                {
                    samplerToGroup.Add(sampler.id, groupBox);
                }
            }

            uniformToGroup.Clear();
            foreach (var uniform in shadingModel.MaterialUniforms())
            {
                ShaderGroupBox groupBox;
                if (visibleGroupBoxes.TryGetValue(uniform.Group(), out groupBox))
                {
                    uniformToGroup.Add(uniform.id, groupBox);
                }
            }

            optionToGroup.Clear();
            foreach (var option in shadingModel.Options())
            {
                ShaderGroupBox groupBox;
                if (visibleGroupBoxes.TryGetValue(option.Group(), out groupBox))
                {
                    optionToGroup.Add(option.id, groupBox);
                }
            }

            renderInfoToGroup.Clear();
            foreach (var renderInfo in shadingModel.RenderInfoSlots())
            {
                ShaderGroupBox groupBox;
                if (visibleGroupBoxes.TryGetValue(renderInfo.Group(), out groupBox))
                {
                    renderInfoToGroup.Add(renderInfo.name, groupBox);
                }
            }
        }

        public int GetGroupY(string group)
        {
            ShaderGroupBox groupBox = null;
            if (visibleGroupBoxes.TryGetValue(group, out groupBox))
            {
                if (groupBox.Visible)
                {
                    int y = 0;
                    while (groupBox != null)
                    {
                        y += groupBox.Location.Y;
                        groupBox = groupBox.Parent as ShaderGroupBox;
                    }

                    return y;
                }
            }

            return -1;
        }

        public int GetUniformY(string group, ParamType type, string uniform)
        {
            var groupY = GetGroupY(group);

            if (groupY != -1)
            {
                Dictionary<string, ShaderParamControl> dict = null;
                switch (type)
                {
                    case ParamType.uniform_var:
                        dict = uniformControls;
                        break;
                    case ParamType.sampler_var:
                        dict = samplerControls;
                        break;
                    case ParamType.option_var:
                        dict = optionControls;
                        break;
                    case ParamType.attrib_var:
                        dict = attribControls;
                        break;
                    case ParamType.render_info_slot:
                        dict = renderInfoControls;
                        break;
                }
                ShaderParamControl control;
                if (dict != null && dict.TryGetValue(uniform, out control))
                {
                    if (control.Visible)
                    {
                        return groupY + control.Location.Y;
                    }
                }
            }

            return -1;
        }

        public int GetUniformY(string uniform)
        {
            ShaderParamControl control;
            if (uniformControls.TryGetValue(uniform, out control))
            {
                if (control.Visible)
                {
                    return control.Location.Y;
                }
            }

            return -1;
        }

        int previousWidth;
        protected override void OnSizeChanged(EventArgs e)
        {
            if (orderedControls.Length > 0 && previousWidth != Width)
            {
                UpdateLayout(false);
            }
        }

        public void UpdateLayout(bool fit)
        {
            DebugConsole.WriteLine("UpdateLayout");
            Win32.NativeMethods.LockWindowUpdate(Handle);
            SuspendLayout();
            int lowerBound;
            int height;
            UpdateLayout(orderedControls, Width, 2, 2, out lowerBound, out height, MultiColumn, fit);
            previousWidth = Width;
            Height = height;
            ResumeLayout();
            Win32.NativeMethods.LockWindowUpdate(IntPtr.Zero);
            Refresh();
        }

        private void UpdateLayout(Control[] controls, int width, int X, int Y, out int lowerBound, out int height, bool multiColumn, bool fit)
        {
            const int vMargin = 4;
            const int vGroupMargin = 2;
            const int hMargin = 4;
            foreach (var group in controls.OfType<ShaderGroupBox>())
            {
                int groupLowerBound;
                int groupHeight;
                UpdateLayout(groupBoxControls[group],
                    width - 12 - 2 * hMargin,
                    6,
                    21,
                    out groupLowerBound,
                    out groupHeight,
                    MultiColumnInGroup, fit);

                // グループを横にならべないようにするために幅を広げる
                group.Width = Math.Max(groupLowerBound + 6, width - 6 -  hMargin);

                group.Height = groupHeight + 6;
            }

            // テキストブロックも横並びにさせないのでグループ幅まで広げる
            foreach (Control control in controls)
            {
                if (control is EditRenderInfo)
                {
                    if (((EditRenderInfo)control).IsTextBlock)
                    {
                        control.Width = Math.Max(0, width - (2 + 2));
                        control.Height = 21;
                    }
                }
            }

            bool hasCommentLabel = (controls.Length > 0) && (controls[0] == commentLabel);

            // i を行頭にしたときの次の行の行頭
            int[] newRows = Enumerable.Range(1, controls.Length).ToArray();
            if (multiColumn)
            {
                // 動的計画法

                // i から始めたときの高さ
                int[] heights = new int[controls.Length + 1];
                // i から始めたときの幅
                int[] widths = new int[controls.Length + 1];
                // i から始めたときの各列の最大の高さの差の和
                int[] diffs = new int[controls.Length + 1];

                int last = hasCommentLabel ? 1:0;
                for (int i = controls.Length - 1; i >= last; i--)
                {
                    if (controls[i] is EditRenderInfo && ((EditRenderInfo)controls[i]).IsTextBlock)
                    {
                        // テキストブロックはサイズを設定済みなので調整不要
                    }
                    else
                    if (fit && controls[i] is ShaderParamControl)
                    {
                        ((ShaderParamControl)controls[i]).Fit();
                    }

                    int w = controls[i].Width + hMargin;
                    int h = controls[i].Height;
                    if (controls[i] is ShaderGroupBox)
                    {
                        h += vGroupMargin;
                    }

                    int minHeight = h + heights[i + 1];
                    int minWidth = Math.Max(w, widths[i + 1]);
                    int column = i;
                    int j;
                    int lowest = controls[i].Height;
                    int minDiff = diffs[i + 1];
                    for (j = i + 1; j < controls.Length; j++)
                    {
                        w += controls[j].Width + hMargin;
                        if (w > width || ((IGroupName)controls[i]).GroupName != ((IGroupName)controls[j]).GroupName)
                        {
                            break;
                        }
                        lowest = Math.Min(controls[j].Height, lowest);
                        h = Math.Max(controls[j].Height, h);
                        int mw = Math.Max(w, widths[j + 1]);
                        int diff = h - lowest;
                        if (h + heights[j + 1] < minHeight ||
                            (h + heights[j + 1] == minHeight &&
                            (mw < minWidth ||
                            (mw == minWidth &&
                            diff + diffs[j + 1] < minDiff
                            ))))
                        {
                            minHeight = h + heights[j + 1];
                            minWidth = mw;
                            minDiff = diff + diffs[j + 1];
                            column = j;
                        }
                    }
                    heights[i] = minHeight + vMargin;
                    newRows[i] = column + 1;
                    widths[i] = minWidth;
                    diffs[i] = minDiff;
                }
            }

            int y = Y;
            lowerBound = 0;
            for (int i = 0; i < controls.Length; i = newRows[i])
            {
                int x = X;
                int h = 0;
                int j = i;

                // ラベルだけ別処理
                if (i == 0 && hasCommentLabel)
                {
                    // ラベルのサイズを変更する。
                    var proposedSize = new Size(Math.Max(0, width - (2 + 2)), int.MaxValue);
                    commentLabel.UpdateSize(proposedSize);

                    h = controls[0].Height + 2;
                    controls[0].Location = new Point(x+2, y+2);
                    j++;
                }

                for (; j < newRows[i]; j++)
                {
                    h = Math.Max(controls[j].Height + vMargin + ((controls[j] is ShaderGroupBox || controls[j] is Label) ? vGroupMargin : 0), h);
                    if (controls[j] is ShaderGroupBox)
                    {
                        controls[j].Location = new Point(x + hMargin, y + (vMargin + vGroupMargin) / 2);
                    }
                    else
                    {
                        controls[j].Location = new Point(x + hMargin, y + vMargin / 2);
                    }
                    x += controls[j].Width + hMargin;
                }

                y += h;
                lowerBound = Math.Max(x, lowerBound);
            }
            height = y;
        }

        private void DisposeOrCacheControl(ShaderParamControl control)
        {
            control.OnBeforeDisposedOrCached();
            if (!CacheKeys.ContainsKey(control))
            {
                control.Dispose();
            }
        }

        private void ClearCacheAll()
        {
            foreach (var item in CacheKeys.ToArray())
            {
                CacheKeys.Remove(item.Key);
                if (item.Value is sampler_varType)
                {
                    samplerControlCache_.Remove((sampler_varType)item.Value);
                }
                else if (item.Value is ShaderTypeUtility.UniformUI)
                {
                    uniformControlCache_.Remove((App.Utility.ShaderTypeUtility.UniformUI)item.Value);
                }
                else if (item.Value is option_varType)
                {
                    optionControlCache_.Remove((option_varType)item.Value);
                }
                else if (item.Value is attrib_varType)
                {
                    attribControlCache_.Remove((attrib_varType)item.Value);
                }
                else if (item.Value is render_info_slotType)
                {
                    renderInfoCache_.Remove((render_info_slotType)item.Value);
                }
                else
                {
                    Debug.Assert(false);
                }

                item.Key.Dispose();
            }
        }

        private void ClearCache(IEnumerable<ShaderParamControl> controls)
        {
            if (CacheKeys.Count < 100)
            {
                return;
            }

            var remain = new HashSet<ShaderParamControl>(controls);
            int i = 0;
            foreach (var item in CacheKeys.ToArray())
            {
                if (remain.Contains(item.Key))
                {
                    continue;
                }
                // 約半分削除
                if (i % 2 == 0)
                {
                    CacheKeys.Remove(item.Key);
                    if (item.Value is sampler_varType)
                    {
                        samplerControlCache_.Remove((sampler_varType)item.Value);
                    }
                    else if (item.Value is ShaderTypeUtility.UniformUI)
                    {
                        uniformControlCache_.Remove((ShaderTypeUtility.UniformUI)item.Value);
                    }
                    else if (item.Value is option_varType)
                    {
                        optionControlCache_.Remove((option_varType)item.Value);
                    }
                    else if (item.Value is attrib_varType)
                    {
                        attribControlCache_.Remove((attrib_varType)item.Value);
                    }
                    else if (item.Value is render_info_slotType)
                    {
                        renderInfoCache_.Remove((render_info_slotType)item.Value);
                    }
                    else
                    {
                        Debug.Assert(false);
                    }

                    item.Key.Dispose();
                }
                i++;
            }
        }

        public static Predicate<string> CreateIsGroupInPage(string group_id, bool pageMode, shading_modelType shadingModel, string page)
        {
            var nameToGroup = shadingModel != null ? shadingModel.Groups().ToDictionary(x => x.name): new Dictionary<string, groupType>();

            Predicate<string> condition;
            if (group_id != null)
            {
                condition = x => x == group_id;
            }
            else if (pageMode)
            {
                if (shadingModel == null || shadingModel.page_array == null)
                {
                    return x => false;
                }

                if (!string.IsNullOrEmpty(page))
                {
                    condition = x =>
                    {
                        groupType group;
                        return nameToGroup.TryGetValue(x, out group) &&
                            string.IsNullOrEmpty(group.Group()) &&
                            group.page_name == page;
                    };
                }
                else
                {
                    var pages = new HashSet<string>(shadingModel.Pages().Select(x => x.name));
                    condition = x =>
                    {
                        groupType group;
                        return !nameToGroup.TryGetValue(x, out group) ||
                            (string.IsNullOrEmpty(group.Group()) &&
                             !pages.Contains(group.page_name));
                    };
                }
            }
            else
            {
                return x => true;
            }

            Func<string, Tuple<bool, string>> next = x =>
            {
                groupType group;
                if (nameToGroup.TryGetValue(x, out group) && !string.IsNullOrEmpty(group.Group()))
                {
                    return new Tuple<bool,string>(true, group.Group());
                }

                return new Tuple<bool, string>(false, null);
            };

            var results = new Dictionary<string, bool>();

            return x => AnyItem(x, condition, next, results);
        }

        /// <summary>
        /// current から next でたどれる要素で condition が true になるものがあるかの判定
        /// メモ化再帰を行う
        /// </summary>
        private static bool AnyItem(string current, Predicate<string> condition, Func<string, Tuple<bool, string>> next, Dictionary<string, bool> results)
        {
            bool result;
            if (results.TryGetValue(current, out result))
            {
                return result;
            }

            if (condition(current))
            {
                return results[current] = true;
            }

            var tuple = next(current);
            if (tuple.Item1)
            {
                return results[current] = AnyItem(tuple.Item2, condition, next, results);
            }

            return results[current] = false;
        }

        /// <summary>
        /// コントロールを割り当てる
        /// </summary>
        /// <param name="material">マテリアル</param>
        /// <param name="shadingModel">割り当てられたシェーダープログラム, シェーダーが不明の場合は null</param>
        /// <param name="visibleInfo"></param>
        /// <param name="showConditionVariable"></param>
        /// <param name="pageMode"></param>
        /// <param name="page"></param>
        /// <returns></returns>
        private IEnumerable<Parameter> GetControls(
            Material material,
            shading_modelType shadingModel,
            VisibleInfo visibleInfo,
            bool showConditionVariable,
            bool pageMode,
            pageType page,
            Predicate<string> isAllAncesterEnabled,
            Func<string, string> toParent)
        {
            Predicate<string> isGroupInPage = CreateIsGroupInPage(null, pageMode, shadingModel, page != null ? page.name: "");

            Predicate<string> showGroupInPage;

            Predicate<string> IsLabelMatch = MaterialShaderPage.IsMatch;

            var customUI = material.CustomUI;
            var table = Definition.GetTable(material.MaterialShaderAssign.Definition);
            if (showConditionVariable)
            {
                showGroupInPage = isGroupInPage;
            }
            else
            {
                showGroupInPage = x => isGroupInPage(x) && (string.IsNullOrEmpty(x) || isAllAncesterEnabled(x));
            }


            int index = 0;

            var hideForceReferParameter = !ParentMaterialShaderPage.chkShowForceReferParameter.Checked;

            if (shadingModel != null)
            {
                foreach (var group in shadingModel.Groups())
                {
                    if (showGroupInPage(group.name))
                    {
                        yield return new GroupParamter(group, index);
                        index++;
                    }
                }
            }

            foreach (var attrib in material.MaterialShaderAssign.AttribAssigns)
            {
                if (hideForceReferParameter && !IsParamVisible(material, attrib.id, typeof(attrib_assignType)))
                {
                    continue;
                }

                attrib_varType attrib_var = null;
                if (shadingModel != null)
                {
                    attrib_var = shadingModel.Attributes().FirstOrDefault(x => x.id == attrib.id);
                }


                if (attrib_var != null)
                {
                    if (!showGroupInPage(attrib_var.Group()))
                    {
                        continue;
                    }

                    if (!visibleInfo.IsVisible(attrib_var))
                    {
                        continue;
                    }

                    if (!isAllAncesterEnabled(attrib_var.Group()))
                    {
                        continue;
                    }

                    if (!IsLabelMatch(LabelHelper.GetLabelForFilter(attrib_var.id, attrib_var.Label(), customUI.attrib_vars, customUI, table)))
                    {
                        continue;
                    }

                    yield return new AttribParameter(attrib_var, GetControl(attrib_var), index);
                    index++;
                }
                else
                {
                    if (!IsLabelMatch(attrib.id))
                    {
                        continue;
                    }

                    yield return new AttribParameter(null, GetControl(attrib), index);
                    index++;
                }
            }

            foreach (var option in material.MaterialShaderAssign.ShaderOptions)
            {
                if (hideForceReferParameter && !IsParamVisible(material, option.id, typeof(shader_optionType)))
                {
                    continue;
                }

                option_varType option_var = null;
                if (shadingModel != null)
                {
                    option_var = shadingModel.Options().FirstOrDefault(
                        x => x.id == option.id &&
                        x.type != option_var_typeType.dynamic);
                }

                if (option_var != null)
                {
                    if (!showGroupInPage(option_var.Group()))
                    {
                        continue;
                    }

                    if (!visibleInfo.IsVisible(option_var, showConditionVariable))
                    {
                        continue;
                    }

                    if (!((showConditionVariable && visibleInfo.IsConditionVariable(option_var)) ||
                         isAllAncesterEnabled(option_var.Group())))
                    {
                        continue;
                    }

                    if (!IsLabelMatch(LabelHelper.GetLabelForFilter(option_var.id, option_var.Label(), customUI.option_vars, customUI, table)))
                    {
                        continue;
                    }

                    yield return new OptionParameter(option_var, GetControl(option_var), index);
                    index++;
                }
                else if (page == null)
                {
                    if (!IsLabelMatch(option.id))
                    {
                        continue;
                    }

                    yield return new OptionParameter(null, GetControl(option), index);
                    index++;
                }
            }

            foreach (var sampler in material.MaterialShaderAssign.SamplerAssigns)
            {
                if (hideForceReferParameter && !IsParamVisible(material, sampler.id, typeof(sampler_assignType)))
                {
                    continue;
                }
                sampler_varType sampler_var = null;
                if (shadingModel != null)
                {
                    sampler_var = shadingModel.Samplers().FirstOrDefault(
                       x => x.id == sampler.id);
                }


                if (sampler_var != null)
                {
                    if (!showGroupInPage(sampler_var.Group()))
                    {
                        continue;
                    }

                    if (!visibleInfo.IsVisible(sampler_var))
                    {
                        continue;
                    }

                    if (!isAllAncesterEnabled(sampler_var.Group()))
                    {
                        continue;
                    }

                    if (!IsLabelMatch(LabelHelper.GetLabelForFilter(sampler_var.id, sampler_var.Label(), customUI.sampler_vars, customUI, table)))
                    {
                        continue;
                    }

                    yield return new SamplerParameter(sampler_var, GetControl(sampler_var), index);
                    index++;
                }
                else if (page == null)
                {
                    if (!IsLabelMatch(sampler.id))
                    {
                        continue;
                    }

                    yield return new SamplerParameter(null, GetControl(sampler), index);
                    index++;
                }
            }

            foreach (var param in material.MaterialShaderAssign.ShaderParams)
            {
                if (hideForceReferParameter && !IsParamVisible(material, param.id, typeof(shader_paramType)))
                {
                    continue;
                }
                uniform_varType uniform_var = null;
                if (shadingModel != null)
                {
                    uniform_var = shadingModel.MaterialUniforms().FirstOrDefault(
                        x => x.id == param.id && x.type == param.type);
                }

                if (uniform_var != null)
                {
                    if (!showGroupInPage(uniform_var.Group()))
                    {
                        continue;
                    }

                    if (!visibleInfo.IsParamVisible(uniform_var, showConditionVariable))
                    {
                        continue;
                    }

                    if (!((showConditionVariable && visibleInfo.IsConditionVariable(uniform_var)) ||
                        isAllAncesterEnabled(uniform_var.Group())))
                    {
                        continue;
                    }

                    if (!IsLabelMatch(LabelHelper.GetLabelForFilter(uniform_var.id, uniform_var.Label(), customUI.uniform_vars, customUI, table)))
                    {
                        continue;
                    }

                    yield return new UniformParameter(uniform_var, GetControl(uniform_var, material.CustomUI), index);
                    index++;
                }
                else if (page == null)
                {
                    if (!IsLabelMatch(param.id))
                    {
                        continue;
                    }

                    yield return new UniformParameter(null, GetControl(param), index);
                    index++;
                    continue;
                }
            }

            foreach (var info in material.MaterialShaderAssign.RenderInfos)
            {
                if (hideForceReferParameter && !IsParamVisible(material, info.name, typeof(render_infoType)))
                {
                    continue;
                }

                render_info_slotType slot = null;
                if (shadingModel != null)
                {
                    slot = shadingModel.RenderInfoSlots().FirstOrDefault(x => x.name == info.name);
                }

                if (slot != null)
                {
                    if (!showGroupInPage(slot.Group()))
                    {
                        continue;
                    }

                    if (!visibleInfo.IsVisible(slot, showConditionVariable))
                    {
                        continue;
                    }

                    if (!((showConditionVariable && visibleInfo.IsConditionVariable(slot)) ||
                        isAllAncesterEnabled(slot.Group())))
                    {
                        continue;
                    }

                    if (!IsLabelMatch(LabelHelper.GetLabelForFilter(slot.name, slot.Label(), customUI.render_info_slots, customUI, table)))
                    {
                        continue;
                    }

                    yield return new RenderInfoParameter(slot, GetControl(slot), index);
                    index++;
                }
                else if (page == null)
                {
                    if (!IsLabelMatch(info.name))
                    {
                        continue;
                    }

                    yield return new RenderInfoParameter(null, GetControl(info), index);
                    index++;
                }
            }

            if (shadingModel != null && shadingModel.textblock_array != null)
            {
                foreach (var textblock in shadingModel.textblock_array.textblock)
                {
                    if (textblock.ui_group != null)
                    {
                        render_info_slotType slot = new render_info_slotType();
                        slot.name = textblock.id;
                        slot.count = 1;
                        slot.ui_group = textblock.ui_group;
                        slot.ui_order = textblock.ui_order;
                        slot.type = render_info_slot_typeType.@string;

                        // テキストブロックに選択肢は無いのでコンボボックスは表示しない
                        ShaderParamControl control = GetControl(slot);
                        if (control is EditRenderInfo)
                        {
                            ((EditRenderInfo)control).VisibleComboBox(false);
                            ((EditRenderInfo)control).IsTextBlock = true;
                        }
                        control.SetLabel(textblock.text, null, null, null, null, false, false, null, false);

                        yield return new RenderInfoParameter(slot, control, index);
                        index++;
                    }
                }
            }
        }

        private static bool IsParamVisible(Material material, string paramName, Type type)
        {
            return material.GetResolvedChildRestrictionState(paramName, type) != ShaderItemRestrictionState.ForceRefer;
        }

        public static bool IsAllAncesterTrue(string group, Func<string, string> toParent, Predicate<string> condition, Dictionary<string, bool> table)
        {
            bool visible;
            if (table.TryGetValue(group, out visible))
            {
                return visible;
            }

            if (condition(group))
            {
                var parent = toParent(group);

                if (parent == group)
                {
                    visible = true;
                }
                else
                if (!string.IsNullOrEmpty(parent))
                {
                    visible = IsAllAncesterTrue(parent, toParent, condition, table);
                }
                else
                {
                    visible = true;
                }
            }
            else
            {
                visible = false;
            }

            table[group] = visible;
            return visible;
        }

        abstract class Parameter
        {
            public Parameter(ShaderParamControl control, int index)
            {
                this.control = control;
            }

            public abstract int order { get; }
            public abstract string group { get; }
            public ShaderParamControl control { get; private set; }
        }

        class OptionParameter : Parameter
        {
            public OptionParameter(option_varType item, ShaderParamControl control, int index) : base(control, index)
            {
                data = item;
            }
            private readonly option_varType data;
            public override string  group { get { return data != null? data.Group(): string.Empty; } }
            public override int  order { get { return data != null? data.Order(): 0; } }
        }

        class SamplerParameter : Parameter
        {
            public SamplerParameter(sampler_varType item, ShaderParamControl control, int index)
                : base(control, index)
            {
                data = item;
            }
            private readonly sampler_varType data;
            public override string group { get { return data != null ? data.Group() : string.Empty; } }
            public override int order { get { return data != null? data.Order(): 0; } }
        }

        class UniformParameter : Parameter
        {
            public UniformParameter(uniform_varType item, ShaderParamControl control, int index)
                : base(control, index)
            {
                data = item;
            }
            private readonly uniform_varType data;
            public override string group { get { return data != null ? data.Group(): string.Empty; } }
            public override int order { get { return data != null ? data.Order(): 0; } }
        }

        class AttribParameter : Parameter
        {
            public AttribParameter(attrib_varType item, ShaderParamControl control, int index)
                : base(control, index)
            {
                data = item;
            }
            private readonly attrib_varType data;
            public override string group { get { return data != null ? data.Group() : string.Empty; } }
            public override int order { get { return data != null? data.Order(): 0; } }
        }

        class RenderInfoParameter : Parameter
        {
            public RenderInfoParameter(render_info_slotType item, ShaderParamControl control, int index)
                : base (control, index)
            {
                data = item;
            }
            private readonly render_info_slotType data;
            public override string group
            {
                get { return data != null ? data.Group() : string.Empty; }
            }
            public override int order
            {
                get { return data != null ? data.Order() : 0; }
            }
        }

        class GroupParamter : Parameter
        {
            public GroupParamter(groupType item, int index)
                : base (null, index)
            {
                data = item;
            }

            private readonly groupType data;
            public override string group
            {
                get { return data != null ? data.Group() : string.Empty; }
            }
            public override int order
            {
                get { return data != null ? data.Order() : 0; }
            }
            public groupType Data
            {
                get { return data; }
            }
        }

        public void UpdateEnabled(ShaderParamControl control, bool value)
        {
            control.UpdateEnabled(value);
        }

        /// <summary>
        /// コントロールを更新する
        /// サイズが変更されたかどうか？
        /// </summary>
        public bool UpdateValues(Material material, MaterialPropertyPanel materialPropertyPanel, bool showId, bool showOriginalLabel)
        {
            Debug.Assert(UIControlEventSuppressBlock.Enabled);
            var visibleGroups = isGroupVisible;
            var visiblePages = new HashSet<string>(materialPropertyPanel.ShaderParamPageNodes.Select(x => ((MaterialPropertyPanel.ShaderPageNodeTag)x.Tag).pageName));
            var modifiedGroup = new HashSet<ShaderGroupBox>();

            var definition = material.MaterialShaderAssign.Definition;
            bool enable = definition != null;
            Definition.ShadingModelTable table = Definition.GetTable(definition);
            bool sizeChanged = false;
            bool shadingModelModified = material.IsShadingModelModified();
            Material.ValueResolvedState valueResolvedState;

            Dictionary<string, string>[] customUIOptions;
            Dictionary<string, string>[] customUISamplers;
            Dictionary<string, string>[] customUIUniforms;
            Dictionary<string, string>[] customUIAttribs;
            Dictionary<string, string>[] customUIRenderInfos;
            Dictionary<string, string>[] customUIGroups;
            PrepareResolveCustomLabel(material, out customUIOptions, out customUISamplers, out customUIUniforms, out customUIAttribs, out customUIRenderInfos, out customUIGroups);

            foreach (var option in material.MaterialShaderAssign.ShaderOptions)
            {
                ShaderParamControl control;
                if (optionControls.TryGetValue(option.id, out control))
                {
                    control.material = material;
                    var isModified = !shadingModelModified && material.IsModified(option);
                    control.IsModified = isModified;
                    bool isParentLabel;
                    var label = ResolveCustomLabel(customUIOptions, option.id, material.CustomUI.option_vars, out isParentLabel);
                    var value = material.GetResolvedOptionValue(option, out valueResolvedState);
                    sizeChanged |= control.SetLabel(label, material.CustomUI, table, visibleGroups, visiblePages, showId, showOriginalLabel, valueResolvedState, isParentLabel);
                    sizeChanged |= control.SetValue(material, value, material.CustomUI, table, visibleGroups, visiblePages, showId, showOriginalLabel);
                    UpdateEnabled(control, enable);

                    ShaderGroupBox groupBox;
                    if (isModified && optionToGroup.TryGetValue(option.id, out groupBox))
                    {
                        modifiedGroup.Add(groupBox);
                    }
                }
            }

            foreach (var samplerAssign in material.MaterialShaderAssign.SamplerAssigns)
            {
                ShaderParamControl control;
                if (samplerControls.TryGetValue(samplerAssign.id, out control))
                {
                    control.material = material;
                    var isModified = !shadingModelModified && material.IsModified(samplerAssign);
                    control.IsModified = isModified;
                    bool isParentLabel;
                    var label = ResolveCustomLabel(customUISamplers, samplerAssign.id, material.CustomUI.sampler_vars, out isParentLabel);
                    var samplerName = material.GetResolvedSamplerAssignValue(samplerAssign, out valueResolvedState);
                    sizeChanged |= control.SetLabel(label, material.CustomUI, table, visibleGroups, visiblePages, showId, showOriginalLabel, valueResolvedState, isParentLabel);
                    sizeChanged |= control.SetValue(material, samplerName, material.CustomUI, table, visibleGroups, visiblePages, showId, showOriginalLabel);
                    UpdateEnabled(control, enable);
                    ShaderGroupBox groupBox;
                    if (isModified && samplerToGroup.TryGetValue(samplerAssign.id, out groupBox))
                    {
                        modifiedGroup.Add(groupBox);
                    }
                }
            }

            foreach (var shaderParam in material.MaterialShaderAssign.ShaderParams)
            {
                ShaderParamControl control;
                if (uniformControls.TryGetValue(shaderParam.id, out control))
                {
                    control.material = material;
                    var isModified = !shadingModelModified && material.IsModified(shaderParam);
                    control.IsModified = isModified;
                    bool isParentLabel;
                    var label = ResolveCustomLabel(customUIUniforms, shaderParam.id, material.CustomUI.uniform_vars, out isParentLabel);
                    var value = material.GetResolvedShaderParamValue(shaderParam, out valueResolvedState);
                    sizeChanged |= control.SetLabel(label, material.CustomUI, table, visibleGroups, visiblePages, showId, showOriginalLabel, valueResolvedState, isParentLabel);
                    sizeChanged |= control.SetValue(material, value, material.CustomUI, table, visibleGroups, visiblePages, showId, showOriginalLabel);
                    UpdateEnabled(control, enable && IsShaderParamEditable(shaderParam));
                    ShaderGroupBox groupBox;
                    if (isModified && uniformToGroup.TryGetValue(shaderParam.id, out groupBox))
                    {
                        modifiedGroup.Add(groupBox);
                    }
                }
            }

            foreach (var attribAssign in material.MaterialShaderAssign.AttribAssigns)
            {
                ShaderParamControl control;
                if (attribControls.TryGetValue(attribAssign.id, out control))
                {
                    control.material = material;
                    var isModified = !shadingModelModified && material.IsModified(attribAssign);
                    control.IsModified = isModified;
                    bool isParentLabel;
                    var label = ResolveCustomLabel(customUIAttribs, attribAssign.id, material.CustomUI.attrib_vars, out isParentLabel);
                    var attribName = material.GetResolvedAttribAssignValue(attribAssign, out valueResolvedState);
                    sizeChanged |= control.SetLabel(label, material.CustomUI, table, visibleGroups, visiblePages, showId, showOriginalLabel, valueResolvedState, isParentLabel);
                    sizeChanged |= control.SetValue(material, attribName, material.CustomUI, table, visibleGroups, visiblePages, showId, showOriginalLabel);
                    UpdateEnabled(control, enable);
                    ShaderGroupBox groupBox;
                    if (isModified && attribToGroup.TryGetValue(attribAssign.id, out groupBox))
                    {
                        modifiedGroup.Add(groupBox);
                    }
                }
            }

            foreach (var renderInfo in material.MaterialShaderAssign.RenderInfos)
            {
                ShaderParamControl tmp;
                if (renderInfoControls.TryGetValue(renderInfo.name, out tmp))
                {
                    var control = (EditRenderInfoBase)tmp;
                    control.material = material;
                    var isModified = !shadingModelModified && material.IsModified(renderInfo);
                    control.IsModified = !shadingModelModified && material.IsModified(renderInfo);
                    bool isParentLabel;
                    var label = ResolveCustomLabel(customUIRenderInfos, renderInfo.name, material.CustomUI.render_info_slots, out isParentLabel);
                    var values = material.GetResolvedRenderInfoValues(renderInfo, out valueResolvedState);
                    sizeChanged |= control.SetLabel(label, material.CustomUI, table, visibleGroups, visiblePages, showId, showOriginalLabel, valueResolvedState, isParentLabel);
                    sizeChanged |= control.SetValues(material, values, renderInfo.candidateModified, material.CustomUI, table, visibleGroups, visiblePages, showId, showOriginalLabel, valueResolvedState);

                    // テキストチェックボックスの値はセーブデータに入るので、初期値と同じくテキストに変換しておく
                    if (control is EditRenderInfoCheckBox)
                    {
                        EditRenderInfoCheckBox checkBoxControl = (EditRenderInfoCheckBox)control;
                        if (checkBoxControl.ValueText != null && renderInfo.values.Count > 0)
                        {
                            renderInfo.values[0] = checkBoxControl.ValueText;
                        }

                        // 保持しているコントロールに編集マークを設定
                        checkBoxControl.SetModified(isModified);
                    }

                    // EditRenderInfo の SetValues で Enabled を設定しているので、UpdateEnabled は呼ばない。
                    // ただしマテリアル参照で値の設定が Refer, Defaultのときや、祖先マテリアルの子マテリアルの制限がForceReferになっている時は UpdateEnabled で Enabled を false にする
                    if (!material.IsResolvedValueEditable(renderInfo.name, typeof(render_infoType)))
                    {
                        UpdateEnabled(control, false);
                    }
                    ShaderGroupBox groupBox;
                    if (isModified && renderInfoToGroup.TryGetValue(renderInfo.name, out groupBox))
                    {
                        modifiedGroup.Add(groupBox);
                    }
                }
            }

            // グループボックスの更新

            var groupTypes = material.MaterialShaderAssign.ShadingModel?.Groups().ToDictionary(x => x.name) ?? new Dictionary<string, groupType>();
            foreach (var group in visibleGroupBoxes)
            {
                var control = group.Value;
                var groupName = group.Key;
                groupType groupType;
                groupTypes.TryGetValue(groupName, out groupType);
                bool isParentLabel;
                var label = ResolveCustomLabel(customUIGroups, groupName, material.CustomUI.groups, out isParentLabel);
                sizeChanged |= control.SetLabel(label, material.CustomUI, table, visibleGroups, visiblePages, showId, showOriginalLabel, isParentLabel);
                group.Value.IsModified = modifiedGroup.Contains(control) || material.IsGroupCustomLabelModified(groupName);
            }

            return sizeChanged;
        }

        internal static string GetGroupName(shading_modelType shadingModel, string id)
        {
            if (shadingModel != null)
            {
                // Options
                var option = shadingModel.Options().FirstOrDefault(x => x.id == id && x.ui_group != null);
                if (option != null)
                {
                    return option.ui_group.group_name;
                }

                // Samplers
                var sampler = shadingModel.Samplers().FirstOrDefault(x => x.id == id && x.ui_group != null);
                if (sampler != null)
                {
                    return sampler.ui_group.group_name;
                }

                // Uniforms
                var uniform = shadingModel.Uniforms().FirstOrDefault(x => x.id == id && x.ui_group != null);
                if (uniform != null)
                {
                    return uniform.ui_group.group_name;
                }

                // RenderInfoSlots
                var renderInfoSlots = shadingModel.RenderInfoSlots().FirstOrDefault(x => x.name == id && x.ui_group != null);
                if (renderInfoSlots != null)
                {
                    return renderInfoSlots.ui_group.group_name;
                }
            }

            return null;
        }

        internal static string GetGroupName(Material material, string id)
        {
            var shadingModel = material.MaterialShaderAssign.ShadingModel;
            string groupName = GetGroupName(shadingModel, id);
            if (groupName != null)
            {
                var group = shadingModel.Groups().Where(x => x.name == groupName).ToArray();
                if (group != null && group.Length > 0)
                {
                    return group.First().ui_label.value + " (" + groupName + ")";
                }
            }

            return string.Empty;
        }

        internal static void PrepareResolveCustomLabel(Material material, out Dictionary<string, string>[] customUIOptions, out Dictionary<string, string>[] customUISamplers,
            out Dictionary<string, string>[] customUIUniforms, out Dictionary<string, string>[] customUIAttribs, out Dictionary<string, string>[] customUIRenderInfos, out Dictionary<string, string>[] customUIGroups)
        {
            var customUIs = material.GetAllAncestorMaterials().Where(x => x != null).Select(x => x.CustomUI).ToArray();
            customUIOptions = customUIs.Select(x => x.option_vars).ToArray();
            customUISamplers = customUIs.Select(x => x.sampler_vars).ToArray();
            customUIUniforms = customUIs.Select(x => x.uniform_vars).ToArray();
            customUIAttribs = customUIs.Select(x => x.attrib_vars).ToArray();
            customUIRenderInfos = customUIs.Select(x => x.render_info_slots).ToArray();
            customUIGroups = customUIs.Select(x => x.groups).ToArray();
        }

        internal static string ResolveCustomLabel(Dictionary<string, string>[] customUis, string id, Dictionary<string, string> materialVars, out bool isParentLabel)
        {
            string label;
            Debug.Assert(materialVars != null);
            isParentLabel = false;
            materialVars.TryGetValue(id, out label);
            if (string.IsNullOrEmpty(label))
            {
                var dict = customUis?.FirstOrDefault(x => x.ContainsKey(id));
                if (dict != null)
                {
                    if (dict.TryGetValue(id, out label))
                    {
                        isParentLabel = !string.IsNullOrEmpty(label);
                    }
                }
            }
            return label;
        }

        private bool SetGroupBoxLabel(Material material, groupType groupType, string groupName, ShaderGroupBox control)
        {
            string customLabel;
            material.CustomUI.groups.TryGetValue(groupName, out customLabel);
            var label = GetGroupLabelString(ShowParamId, groupName, groupType, customLabel);
            if (control.Text != label)
            {
                control.Text = label;
                return true;
            }

            return false;
        }

        /// <summary>
        /// グループボックスのラベルを取得
        /// </summary>
        public static string GetGroupLabelString(bool showId, string groupName, groupType group, string customLabel)
        {
            string uiLabel = null;
            if (group != null && group.ui_label != null)
            {
                uiLabel = group.ui_label.value;
            }

            if (!string.IsNullOrEmpty(customLabel))
            {
                if (showId)
                {
                    if (string.IsNullOrEmpty(uiLabel) || groupName == ShaderTypeUtility.AttribAssignGroupName())
                    {
                        return string.Format("{0} ({1})", customLabel, groupName);
                    }
                    else
                    {
                        return string.Format("{0} ({1}/{2})", customLabel, uiLabel, groupName);
                    }
                }
                else
                {
                    return customLabel;
                }
            }
            else if (!string.IsNullOrEmpty(uiLabel))
            {
                if (showId && groupName != ShaderTypeUtility.AttribAssignGroupName())
                {
                    return string.Format("{0} ({1})", uiLabel, groupName);
                }
                else
                {
                    return uiLabel;
                }
            }
            else
            {
                return groupName;
            }
        }

        /// <summary>
        /// カーブエディタへのリンクを更新
        /// </summary>
        public void UpdateAnimationMenu(Material material, ObjectPropertyDialog owner)
        {
            var anims = material.Referrers.SelectMany(x => x.AllAnimations).OfType<IHasShaderParameterAnimation>().ToArray();
            var animationSets = material.Referrers.SelectMany(x => x.AnimationSetsWithDefault).ToArray();
            foreach (var shaderParam in material.MaterialShaderAssign.ShaderParams)
            {
                ShaderParamControl control;
                if (uniformControls.TryGetValue(shaderParam.id, out control))
                {
                    control.SetContextMenu(anims, animationSets, material, owner);
                }
            }
        }

        private ShaderParamControl GetControl(uniform_varType param, CustomUI customUI)
        {
            ShaderParamControl control = null;
            {
                var key = new ShaderTypeUtility.UniformUI() { uniform = param, colorControl = customUI.colorControls.Contains(param.id) };
                if (uniformControlCache_.TryGetValue(key, out control) == false)
                {
                    control = CreateControl(param, customUI);
                    uniformControlCache_[key] = control;
                    CacheKeys.Add(control, key);
                }
                uniformControls[param.id] = control;
            }

            return control;
        }

        private ShaderParamControl GetControl(shader_paramType param)
        {
            var control = CreateControl(param);

            uniformControls[param.id] = control;
            return control;
        }

        private ShaderParamControl GetControl(sampler_varType param)
        {
            ShaderParamControl control = null;
            {
                if (samplerControlCache_.TryGetValue(param, out control) == false)
                {
                    control = CreateControl(param);
                    samplerControlCache_[param] = control;
                    CacheKeys.Add(control, param);
                }
                samplerControls[param.id] = control;
            }

            return control;
        }

        private ShaderParamControl GetControl(sampler_assignType param)
        {
            EditSamplerAssign control = new EditSamplerAssign();
            {
                control.ParamName = param.id;
                control.Anchor = AnchorStyles.Left | AnchorStyles.Top;
                control.ValueChanged += (s, e) =>
                {
                    if (ValueChanged != null)
                    {
                        ValueChanged(this, e);
                    }
                };
                control.LabelChanged += (s, e) =>
                {
                    if (LabelChanged != null)
                    {
                        LabelChanged(this, e);
                    }
                };
                control.LinkClicked += (s, e) =>
                {
                    if (LinkClicked != null)
                    {
                        LinkClicked(this, e);
                    }
                };
                control.CanEnabled = false;
            }
            samplerControls[param.id] = control;

            return control;
        }

        private ShaderParamControl GetControl(option_varType option)
        {
            ShaderParamControl control = null;
            {
                if (optionControlCache_.TryGetValue(option, out control) == false)
                {
                    control = CreateControl(option);
                    optionControlCache_[option] = control;
                    CacheKeys.Add(control, option);
                }
                optionControls[option.id] = control;
            }

            return control;
        }

        private ShaderParamControl GetControl(shader_optionType shaderOption)
        {
            var control = new EditReadOnly(shaderOption.id, ParamType.option_var);
            control.CanEnabled = false;
            control.LabelChanged += (s, e) =>
            {
                if (LabelChanged != null)
                {
                    LabelChanged(this, e);
                }
            };

            optionControls[shaderOption.id] = control;
            return control;
        }

        private ShaderParamControl GetControl(attrib_varType attrib)
        {
            ShaderParamControl control = null;
            if (attribControlCache_.TryGetValue(attrib, out control) == false)
            {
                control = CreateControl(attrib);
                attribControlCache_[attrib] = control;
                CacheKeys.Add(control, attrib);
            }
            attribControls[attrib.id] = control;
            return control;
        }


        private ShaderParamControl GetControl(attrib_assignType attribAssign)
        {
            var control = new EditReadOnly(attribAssign.id, ParamType.attrib_var);
            control.CanEnabled = false;
            control.LabelChanged += (s, e) =>
            {
                if (LabelChanged != null)
                {
                    LabelChanged(this, e);
                }
            };
            attribControls[attribAssign.id] = control;
            return control;
        }

        private ShaderParamControl GetControl(render_info_slotType slot)
        {
            EditRenderInfoBase control = null;
            if (renderInfoCache_.TryGetValue(slot, out control) == false)
            {
                // type="string" item="check" の場合はチェックボックスを作成する
                if (slot.type == render_info_slot_typeType.@string)
                {
                    if (ui_item_valueType.check == slot.Item())
                    {
                        control = new EditRenderInfoCheckBox(slot.count, 1, slot);
                    }
                }

                if (control == null)
                {
                    if (slot.Item() == ui_item_valueType.color &&
                        slot.type == render_info_slot_typeType.@float &&
                        !slot.optional &&
                        (slot.count == 3 || slot.count == 4))
                    {
                        control = new EditRenderInfoColor(slot);
                    }
                    else
                    {
                        switch (slot.type)
                        {
                            case render_info_slot_typeType.@int:
                                {
                                    int min;
                                    int max;
                                    if (!string.IsNullOrEmpty(slot.choice) && !IfShaderAssignUtility.TryParseIntRange(slot.choice, out min, out max))
                                    {
                                        control = new EditRenderInfo(slot.count, 1, slot);
                                    }
                                    else if (ConfigData.ApplicationConfig.UserSetting.UI.FixedSlider)
                                    {
                                        control = new EditRenderInfoIntSliders(slot.count, slot);
                                    }
                                    else
                                    {
                                        control = new EditRenderInfoInt(slot.count, 1, slot);
                                    }
                                    break;
                                }
                            case render_info_slot_typeType.@float:
                                {
                                    float min;
                                    float max;
                                    if (!string.IsNullOrEmpty(slot.choice) && !IfShaderAssignUtility.TryParseFloatRange(slot.choice, out min, out max))
                                    {
                                        control = new EditRenderInfo(slot.count, 1, slot);
                                    }
                                    else if (ConfigData.ApplicationConfig.UserSetting.UI.FixedSlider)
                                    {
                                        control = new EditRenderInfoFloatSliders(slot.count, slot);
                                    }
                                    else
                                    {
                                        control = new EditRenderInfoFloat(slot.count, 1, slot);
                                    }
                                    break;
                                }
                            case render_info_slot_typeType.@string:
                                control = new EditRenderInfo(slot.count, 1, slot);
                                break;
                            default:
                                Debug.Assert(false);
                                break;
                        }
                    }
                }

                control.ParamName = slot.name;
                control.Anchor = AnchorStyles.Left | AnchorStyles.Top;

                if (slot.Label() != null)
                {
                    control.ParamLabel = slot.Label();
                }

                if (slot.Comment() != null)
                {
                    control.ParamComment = slot.Comment();
                }

                control.ValueChanged += (s, e) =>
                    {
                        if (ValueChanged != null)
                        {
                            ValueChanged(s, e);
                        }
                    };
                control.LabelChanged += (s, e) =>
                {
                    if (LabelChanged != null)
                    {
                        LabelChanged(this, e);
                    }
                };
                control.LinkClicked += (s, e) =>
                {
                    if (LinkClicked != null)
                    {
                        LinkClicked(this, e);
                    }
                };
                control.GroupName = slot.Group();
                renderInfoCache_[slot] = control;
                CacheKeys.Add(control, slot);
            }
            renderInfoControls[slot.name] = control;
            return control;
        }

        private ShaderParamControl GetControl(RenderInfo renderInfo)
        {
            EditRenderInfoBase control = null;
            switch (renderInfo.type)
            {
                case render_info_typeType.@int:
                    if (ConfigData.ApplicationConfig.UserSetting.UI.FixedSlider)
                    {
                        control = new EditRenderInfoIntSliders(Math.Max(renderInfo.values.Count, 1), null);
                    }
                    else
                    {
                        control = new EditRenderInfoInt(Math.Max(renderInfo.values.Count, 1), 1, null);
                    }
                    break;
                case render_info_typeType.@float:
                    if (ConfigData.ApplicationConfig.UserSetting.UI.FixedSlider)
                    {
                        control = new EditRenderInfoFloatSliders(Math.Max(renderInfo.values.Count, 1), null);
                    }
                    else
                    {
                        control = new EditRenderInfoFloat(Math.Max(renderInfo.values.Count, 1), 1, null);
                    }
                    break;
                case render_info_typeType.@string:
                    control = new EditRenderInfo(Math.Max(renderInfo.values.Count, 1), 1, null);
                    break;
                default:
                    Debug.Assert(false);
                    break;
            }
            control.ParamName = renderInfo.name;
            control.Anchor = AnchorStyles.Left | AnchorStyles.Top;
            control.CanEnabled = false;
            control.LabelChanged += (s, e) =>
            {
                if (LabelChanged != null)
                {
                    LabelChanged(this, e);
                }
            };
            renderInfoControls[renderInfo.name] = control;
            return control;
        }

        private ShaderParamControl CreateControl(attrib_varType param)
        {
            ShaderParamControl control;
            if (param.Editable())
            {
                control = new EditAttribute(param);
                {
                    control.ParamName = param.id;
                }
            }
            else
            {
                // TODO: option_var でいいか検証
                control = new EditReadOnly(param.id, ParamType.attrib_var);
                control.ParamLabel = param.Label();
                control.ParamComment = param.Comment();
            }

            if (param.Label() != null)
            {
                control.ParamLabel = param.Label();
            }

            if (param.Comment() != null)
            {
                control.ParamComment = param.Comment();
            }

            control.Anchor = AnchorStyles.Left | AnchorStyles.Top;

            control.ValueChanged += (s, e) =>
            {
                if (ValueChanged != null)
                {
                    ValueChanged(this, e);
                }
            };

            control.LabelChanged += (s, e) =>
            {
                if (LabelChanged != null)
                {
                    LabelChanged(this, e);
                }
            };

            control.LinkClicked += (s, e) =>
            {
                if (LinkClicked != null)
                {
                    LinkClicked(this, e);
                }
            };

            control.GroupName = param.Group();

            return control;
        }

        private ShaderParamControl CreateControl(sampler_varType param)
        {
            var control = new EditSamplerAssign();
            {
                control.ParamName = param.id;
                control.ParamHint = param.hint;

                if (param.Label() != null)
                {
                    control.ParamLabel = param.Label();
                }

                if (param.Comment() != null)
                {
                    control.ParamComment = param.Comment();
                }

                control.Anchor = AnchorStyles.Left | AnchorStyles.Top;

                control.ValueChanged += (s, e) =>
                {
                    if (ValueChanged != null)
                    {
                        ValueChanged(this, e);
                    }
                };

                control.LabelChanged += (s, e) =>
                {
                    if (LabelChanged != null)
                    {
                        LabelChanged(this, e);
                    }
                };

                control.LinkClicked += (s, e) =>
                {
                    if (LinkClicked != null)
                    {
                        LinkClicked(this, e);
                    }
                };

                control.DragOver += EditSamplerAssignDragOver;
                control.DragDrop += EditSamplerAssignDragDrop;
                control.AllowDrop = true;
            }

            control.GroupName = param.Group();

            return control;
        }

        private void EditSamplerAssignDragOver(object sender, DragEventArgs e)
        {
            e.Effect = DragDropEffects.None;
            var materials = GetTargets().GetObjects(GuiObjectID.Material).OfType<Material>().ToArray();
            MaterialSamplerPage.CheckTextureDragOver(materials, e);
        }

        private void EditSamplerAssignDragDrop(object sender, DragEventArgs e)
        {
            base.OnDragDrop(e);

            var sampleAssign = sender as EditSamplerAssign;
            if (sampleAssign == null)
            {
                return;
            }
            if (!e.Data.GetDataPresent(DataFormats.FileDrop) && !e.Data.GetDataPresent(typeof(FileTreeView.ViewItem)))
            {
                return;
            }

            var materials = ParentMaterialShaderPage.Targets.Objects.OfType<Material>().ToList();
            if (!materials.Any())
            {
                return;
            }

            var commandSets = new EditCommandSet();

            // プリセットからヒントに応じたサンプラ名を決定
            string targetSamplerName = null;
            if (!string.IsNullOrEmpty(sampleAssign.ParamHint))
            {
                var presetSampler = ApplicationConfig.Preset.SamplerNamePresets.FirstOrDefault(
                    x => x.Hint == sampleAssign.ParamHint);
                if (presetSampler != null)
                {
                    // 同じサンプラ名が既に存在した場合は上書きしないでsamplerXXにする
                    var samplers = materials.SelectMany(material => material.ResolvedSamplers).Select(x => x.name).ToArray();
                    if (samplers.All(x => x != presetSampler.Name))
                    {
                        targetSamplerName = presetSampler.Name;
                    }
                    else
                    {
                        targetSamplerName = StringUtility.UniqueName(
                            "sampler",
                            samplers,
                            Enumerable.Range(0, int.MaxValue).Select(x => x.ToString()));
                    }
                }
            }

            var textures = MaterialSamplerPage.DropedTextures(e, commandSets);
            MaterialSamplerPage.SetSampler(materials, textures, commandSets, targetSamplerName);

            var addedSampler = ParentMaterialShaderPage.ActiveTarget.ResolvedSamplers.LastOrDefault();
            if (addedSampler != null)
            {
                var target = ApplicableSamplerObjects(ParentMaterialShaderPage.ActiveTarget, ParentMaterialShaderPage.Targets, sampleAssign.ParamName);
                commandSets.Add(CreateEditCommand_material_shader_assign_sampler_name(target, sampleAssign.ParamName, addedSampler.name, false).Execute());
            }

            var models = materials.SelectMany(x => x.Referrers).ToArray();
            EventHandler reload = (s, a) =>
            {
                foreach (var model in models)
                {
                    Viewer.LoadOrReloadModel.Send(model);
                }
            };

            reload(null, EventArgs.Empty);

            commandSets.OnPostEdit += reload;

            commandSets.Reverse();
            TheApp.CommandManager.Add(commandSets);
        }

        private ShaderParamControl CreateControl(uniform_varType param, CustomUI customUI)
        {
            ShaderParamControl control = CreateControl(param.type, param.Item(), param.ui_min, param.ui_max, param.ui_default_min, param.ui_default_max,
                customUI.colorControls.Contains(param.id));

            if (control != null)
            {
                control.InitializeDropDownButton();
                control.ParamName = param.id;
                control.ParamLabel = param.Label();
                control.ParamComment = param.Comment();
            }

            control.GroupName = param.Group();
            return control;
        }

        private ShaderParamControl CreateControl(shader_paramType param)
        {
            ShaderParamControl control = CreateControl(param.type, ui_item_valueType.auto, null, null, null, null, false);

            if (control != null)
            {
                control.InitializeDropDownButton();
                control.ParamName = param.id;
                control.CanEnabled = false;
            }

            return control;
        }


        private ShaderParamControl CreateControl(shader_param_typeType type, ui_item_valueType item, ui_minType ui_min, ui_maxType ui_max, ui_default_minType default_min, ui_default_maxType default_max, bool customColor)
        {
            //ShaderParamControl control = null;
            int column = 1;
            int row = 1;
            shader_param_typeType baseType = type;
            bool checkBox = item == ui_item_valueType.check;
            switch (type)
            {
                case shader_param_typeType.@bool: column = 1; baseType = shader_param_typeType.@bool; break;
                case shader_param_typeType.bool2: column = 2; baseType = shader_param_typeType.@bool; break;
                case shader_param_typeType.bool3: column = 3; baseType = shader_param_typeType.@bool; break;
                case shader_param_typeType.bool4: column = 4; baseType = shader_param_typeType.@bool; break;
                case shader_param_typeType.@int: column = 1; baseType = shader_param_typeType.@int; break;
                case shader_param_typeType.int2: column = 2; baseType = shader_param_typeType.@int; break;
                case shader_param_typeType.int3: column = 3; baseType = shader_param_typeType.@int; break;
                case shader_param_typeType.int4: column = 4; baseType = shader_param_typeType.@int; break;
                case shader_param_typeType.@uint: column = 1; baseType = shader_param_typeType.@uint; break;
                case shader_param_typeType.uint2: column = 2; baseType = shader_param_typeType.@uint; break;
                case shader_param_typeType.uint3: column = 3; baseType = shader_param_typeType.@uint; break;
                case shader_param_typeType.uint4: column = 4; baseType = shader_param_typeType.@uint; break;
                case shader_param_typeType.@float: column = 1; baseType = shader_param_typeType.@float; break;
                case shader_param_typeType.float2: column = 2; baseType = shader_param_typeType.@float; break;
                case shader_param_typeType.float3: column = 3; baseType = shader_param_typeType.@float; break;
                case shader_param_typeType.float4: column = 4; baseType = shader_param_typeType.@float; break;
                case shader_param_typeType.float2x2: row = 2; column = 2; baseType = shader_param_typeType.@float; break;
                case shader_param_typeType.float2x3: row = 2; column = 3; baseType = shader_param_typeType.@float; break;
                case shader_param_typeType.float2x4: row = 2; column = 4; baseType = shader_param_typeType.@float; break;
                case shader_param_typeType.float3x2: row = 3; column = 2; baseType = shader_param_typeType.@float; break;
                case shader_param_typeType.float3x3: row = 3; column = 3; baseType = shader_param_typeType.@float; break;
                case shader_param_typeType.float3x4: row = 3; column = 4; baseType = shader_param_typeType.@float; break;
                case shader_param_typeType.float4x2: row = 4; column = 2; baseType = shader_param_typeType.@float; break;
                case shader_param_typeType.float4x3: row = 4; column = 3; baseType = shader_param_typeType.@float; break;
                case shader_param_typeType.float4x4: row = 4; column = 4; baseType = shader_param_typeType.@float; break;
                default:
                    checkBox = false;
                    break;
            }

            ShaderParamControl control = null;
            if (checkBox)
            {
                control = new EditBoolMatrix(row, column, type, ui_min == null ? 0 : ui_min.value, ui_max == null ? 1 : ui_max.value);
            }
            else
            {
                bool clamp = ConfigData.ApplicationConfig.DefaultValue.UIRange.Clamp;
                switch (baseType)
                {
                    case shader_param_typeType.srt2d:
                        if (ConfigData.ApplicationConfig.UserSetting.UI.FixedSlider)
                        {
                            control = new TexCoordSliders(type);
                        }
                        else
                        {
                            control = new TexCoord(type);
                        }
                        break;
                    case shader_param_typeType.texsrt:
                        if (ConfigData.ApplicationConfig.UserSetting.UI.FixedSlider)
                        {
                            control = new EditTexSrtSliders(type);
                        }
                        else
                        {
                            control = new EditTexsrt(type);
                        }
                        break;
                    case shader_param_typeType.srt3d:
                        control = new EditTexCoord3(type);
                        break;
                    case shader_param_typeType.@bool:
                        control = new EditBoolMatrix(row, column, type);
                        break;
                    case shader_param_typeType.@int:
                        {
                            int min = -10;
                            int max = 10;
                            int inputMin = int.MinValue;
                            int inputMax = int.MaxValue;

                            if (clamp)
                            {
                                if (ui_min != null)
                                {
                                    inputMin = (int)ui_min.value;
                                    if (default_min == null)
                                    {
                                        min = inputMin;
                                    }
                                }

                                if (default_min != null)
                                {
                                    min = (int)default_min.value;
                                }

                                if (ui_max != null)
                                {
                                    inputMax = (int)ui_max.value;
                                    if (default_max == null)
                                    {
                                        max = inputMax;
                                    }
                                }

                                if (default_max != null)
                                {
                                    max = (int)default_max.value;
                                }
                            }
                            else
                            {
                                if (ui_min != null)
                                {
                                    min = (int)ui_min.value;
                                }

                                if (ui_max != null)
                                {
                                    max = (int)ui_max.value;
                                }
                            }

                            if (row == 1 && ConfigData.ApplicationConfig.UserSetting.UI.FixedSlider)
                            {
                                EditIntSliders intControl = new EditIntSliders(column, type);
                                intControl.SetMinimumValue(min, inputMin);
                                intControl.SetMaximumValue(max, inputMax);
                                intControl.UpdateIncrement();
                                control = intControl;
                                intControl.SetDefaultMinValue();
                            }
                            else
                            {
                                EditIntMatrix intControl = new EditIntMatrix(row, column, type);
                                intControl.SetMinimumValue(min, inputMin);
                                intControl.SetMaximumValue(max, inputMax);

                                intControl.UpdateIncrement();
                                control = intControl;
                            }
                        }
                        break;
                    case shader_param_typeType.@uint:
                        {
                            uint min = 0;
                            uint max = 10;
                            uint inputMin = 0;
                            uint inputMax = uint.MaxValue;

                            try
                            {
                                if (clamp)
                                {
                                    if (ui_min != null)
                                    {
                                        inputMin = Convert.ToUInt32(ui_min.value);
                                        if (default_min == null)
                                        {
                                            min = inputMin;
                                        }
                                    }

                                    if (default_min != null)
                                    {
                                        min = Convert.ToUInt32(default_min.value);
                                    }

                                    if (ui_max != null)
                                    {
                                        inputMax = Convert.ToUInt32(ui_max.value);
                                        if (default_max == null)
                                        {
                                            max = inputMax;
                                        }
                                    }

                                    if (default_max != null)
                                    {
                                        max = Convert.ToUInt32(default_max.value);
                                    }
                                }
                                else
                                {
                                    if (ui_min != null)
                                    {
                                        min = Convert.ToUInt32(ui_min.value);
                                    }

                                    if (ui_max != null)
                                    {
                                        max = Convert.ToUInt32(ui_max.value);
                                    }
                                }
                            }
                            catch
                            {
                            }

                            if (ConfigData.ApplicationConfig.UserSetting.UI.FixedSlider)
                            {
                                EditUintSliders uintControl = new EditUintSliders(column, type);
                                uintControl.SetMinimumValue(min, inputMin);
                                uintControl.SetMaximumValue(max, inputMax);
                                uintControl.UpdateIncrement();
                                control = uintControl;
                                uintControl.SetDefaultMinValue();

                            }
                            else
                            {
                                EditUintMatrix uintControl = new EditUintMatrix(row, column, type);
                                uintControl.SetMinimumValue(min, inputMin);
                                uintControl.SetMaximumValue(max, inputMax);
                                uintControl.UpdateIncrement();
                                control = uintControl;
                            }
                        }
                        break;
                    case shader_param_typeType.@float:
                        {
                            // カラー
                            if (type == shader_param_typeType.float3)
                            {
                                if (item == ui_item_valueType.color || customColor)
                                {
                                    control = new ColorFloat
                                    {
                                        Value = new RgbaColor(1.0f, 1.0f, 1.0f, 1.0f),
                                        EditMode = ColorEditMode.RGB,
                                        //										MinimumValue = (ui_min != null) ? ui_min.value : (float?)null,
                                        MinimumValue = 0.0f,
                                        MaximumValue = (ui_max != null) ? ui_max.value :
                                            //(clamp && default_max != null)? 1000 * default_max.value:
                                                       (float?)null,
                                    };
                                    if (clamp && default_max != null)
                                    {
                                        ((ColorFloat)control).SetDefaultMax(default_max.value);
                                    }
                                    if (customColor)
                                    {
                                        ((ColorFloat)control).ColorControlChanged += ColorControlChanged;
                                    }
                                    break;
                                }
                            }
                            else if (type == shader_param_typeType.float4)
                            {
                                if (item == ui_item_valueType.color || customColor)
                                {
                                    control = new ColorFloat
                                    {
                                        Value = new RgbaColor(1, 1, 1, 1),
                                        EditMode = ColorEditMode.RGBA,
                                        //										MinimumValue = (ui_min != null) ? ui_min.value : (float?)null,
                                        MinimumValue = 0.0f,
                                        MaximumValue = (ui_max != null) ? ui_max.value :
                                            //(clamp && default_max != null)? 1000 * default_max.value:
                                                       (float?)null,
                                    };
                                    if (clamp && default_max != null)
                                    {
                                        ((ColorFloat)control).SetDefaultMax(default_max.value);
                                    }
                                    if (customColor)
                                    {
                                        ((ColorFloat)control).ColorControlChanged += ColorControlChanged;
                                    }
                                    break;
                                }
                            }

                            float min = -1;
                            float max = 1;
                            float inputMin = float.MinValue;
                            float inputMax = float.MaxValue;

                            if (clamp)
                            {
                                if (ui_min != null)
                                {
                                    inputMin = ui_min.value;
                                    if (default_min == null)
                                    {
                                        min = inputMin;
                                    }
                                }

                                if (default_min != null)
                                {
                                    min = default_min.value;
                                }

                                if (ui_max != null)
                                {
                                    inputMax = ui_max.value;
                                    if (default_max == null)
                                    {
                                        max = inputMax;
                                    }
                                }

                                if (default_max != null)
                                {
                                    max = default_max.value;
                                }
                            }
                            else
                            {
                                if (ui_min != null)
                                {
                                    min = ui_min.value;
                                }

                                if (ui_max != null)
                                {
                                    max = ui_max.value;
                                }
                            }

                            if (row == 1 && ConfigData.ApplicationConfig.UserSetting.UI.FixedSlider)
                            {
                                EditFloatSliders floatControl = new EditFloatSliders(column, type);
                                floatControl.SetMinimumValue(min, inputMin);
                                floatControl.SetMaximumValue(max, inputMax);
                                floatControl.UpdateIncrement();
                                control = floatControl;
                                floatControl.SetDefaultMinValue();

                                if (type == shader_param_typeType.float3 || type == shader_param_typeType.float4)
                                {
                                    floatControl.ColorControlChanged += ColorControlChanged;
                                }
                            }
                            else
                            {
                                EditFloatMatrix floatControl = new EditFloatMatrix(row, column, type);
                                floatControl.SetMinimumValue(min, inputMin);
                                floatControl.SetMaximumValue(max, inputMax);
                                floatControl.UpdateIncrement();
                                control = floatControl;

                                if (type == shader_param_typeType.float3 || type == shader_param_typeType.float4)
                                {
                                    floatControl.ColorControlChanged += ColorControlChanged;
                                }
                            }

                        }
                        break;
                }
            }
            Debug.Assert(control != null);

            if (control != null)
            {
                control.Anchor = AnchorStyles.Left | AnchorStyles.Top;

                control.ValueChanged += (s, e) =>
                {
                    if (ValueChanged != null)
                    {
                        ValueChanged(this, e);
                    }
                };

                control.LabelChanged += (s, e) =>
                {
                    if (LabelChanged != null)
                    {
                        LabelChanged(this, e);
                    }
                };

                control.LinkClicked += (s, e) =>
                {
                    if (LinkClicked != null)
                    {
                        LinkClicked(this, e);
                    }
                };
            }

            return control;
        }

        private ShaderParamControl CreateControl(option_varType option)
        {
            ShaderParamControl control = null;
            if (option.ui_item?.value != ui_item_valueType.combiner)
            {
                var type = ShaderOptionUtility.Type(option.choice);
                switch (type)
                {
                    case ShaderOptionUtility.OptionType.Bool:
                        control = new EditBoolMatrix(option);
                        break;
                    case ShaderOptionUtility.OptionType.Int:
                        {
                            int min;
                            int max;
                            IfShaderAssignUtility.TryParseIntRange(option.choice, out min, out max);
                            if (min == max)
                            {
                                control = new EditReadOnly(option, HorizontalAlignment.Right);
                            }
                            else if (ConfigData.ApplicationConfig.UserSetting.UI.FixedSlider)
                            {
                                control = new EditIntSliders(option);
                            }
                            else
                            {
                                control = new EditIntMatrix(option);
                            }
                        }
                        break;
                    case ShaderOptionUtility.OptionType.Enum:
                        {
                            if (IfShaderOptionUtility.Enum(option.choice).Length == 1)
                            {
                                control = new EditReadOnly(option, HorizontalAlignment.Left);
                            }
                            else
                            {
                                control = new EditEnum(option);
                            }
                        }
                        break;
                }
            }
            else
            {
                control = new EditCombiner(option);
            }

            if (control != null)
            {
                if (option.Label() != null)
                {
                    control.ParamLabel = option.Label();
                }

                if (option.Comment() != null)
                {
                    control.ParamComment = option.Comment();
                }

                control.Anchor = AnchorStyles.Left | AnchorStyles.Top;

                control.ValueChanged += (s, e) =>
                {
                    if (ValueChanged != null)
                    {
                        ValueChanged(this, e);
                    }
                };

                control.LabelChanged += (s, e) =>
                {
                    if (LabelChanged != null)
                    {
                        LabelChanged(this, e);
                    }
                };

                control.LinkClicked += (s, e) =>
                {
                    if (LinkClicked != null)
                    {
                        LinkClicked(this, e);
                    }
                };

                control.GroupName = option.Group();
            }
            else
            {
                Debug.Assert(false);
            }

            return control;
        }

        public void scgGroup_ValueChanged(Material activeTarget, GuiObjectGroup target, SamplerValueChangedEventArgs e)
        {
            target = ApplicableSamplerObjects(activeTarget, target, e.ParamName);
            TheApp.CommandManager.Execute(CreateEditCommand_material_shader_assign_sampler_name(target, e.ParamName, (string)e.ParamValue, true));
            App.AppContext.OnMaterialReferenceBehaviorSettingChanged();
        }

        public void scgGroup_ValueChanged(Material activeTarget, GuiObjectGroup targets, ShaderParamValueChangedEventArgs e)
        {
            if (e.IsSequentialChanging)
            {
                // HIO
                if(Viewer.Manager.Instance.IsConnected)
                {
                    object param =
                        (e.ParamElemType == ShaderTypeUtility.ParamPrimitiveTypeKind.@bool) ? (object)e.BoolValueArray :
                        (e.ParamElemType == ShaderTypeUtility.ParamPrimitiveTypeKind.@int) ? (object)e.IntValueArray :
                        (e.ParamElemType == ShaderTypeUtility.ParamPrimitiveTypeKind.@uint) ? (object)e.UintValueArray :
                        (e.ParamElemType == ShaderTypeUtility.ParamPrimitiveTypeKind.@float) ? (object)e.FloatValueArray :
                                                                        null;

                    Debug.Assert(param != null);
                    var descendantMaterials = new List<GuiObject>();
                    foreach (var material in targets.GetObjects(GuiObjectID.Material).OfType<Material>())
                    {
                        descendantMaterials.Add(material);
                        descendantMaterials.AddRange(material.GetAllDescendantShaderParamMaterials(e.ParamName));
                    }

                    Viewer.MaterialShaderParameter.Send(
                        new GuiObjectGroup(descendantMaterials.Distinct().Where(x => x.OwnerDocument is Model)),
                        new Viewer.MaterialShaderParameter.Data()
                        {
                            ParamName = e.ParamName,
                            ShaderParamType = e.ShaderParamType,
                            Params    = ObjectUtility.Clone(param)
                        },
                        0xFFFFFFFF,
                        e.ElementBits
                    );
                }
            }
            else
            {
                targets = ApplicableParamObjects(activeTarget, targets, e.ParamName);

                var commandSet = new EditCommandSet();

                if ((targets.Objects.Count > 0) && Viewer.Manager.Instance.IsConnected)
                {
                    shader_paramType param = GetShaderParamFromId(targets.Objects[0] as Material, e.ParamName);

                    switch (e.ParamElemType)
                    {
                        case ShaderTypeUtility.ParamPrimitiveTypeKind.@bool:
                        {
                            bool[] targetParamArray = G3dDataParser.ParseInt2BoolArray(param.Value);
                            bool[] srcParamArray = e.BoolValueArray;

                            for (int i = 0; i != srcParamArray.Length; ++i)
                            {
                                if ((e.ElementBits & (1 << i)) != 0)
                                {
                                    targetParamArray[i] = srcParamArray[i];
                                }
                            }

                            commandSet.Add(CreateEditCommand_shader_assign_shader_param(targets, e.ParamName, IfUtility.MakeArrayString(targetParamArray), e.ElementBits));

                            break;
                        }

                        case ShaderTypeUtility.ParamPrimitiveTypeKind.@int:
                        {
                            int[] targetParamArray = G3dDataParser.ParseIntArray(param.Value);
                            int[] srcParamArray = e.IntValueArray;

                            for (int i = 0; i != srcParamArray.Length; ++i)
                            {
                                if ((e.ElementBits & (1 << i)) != 0)
                                {
                                    targetParamArray[i] = srcParamArray[i];
                                }
                            }

                            commandSet.Add(CreateEditCommand_shader_assign_shader_param(targets, e.ParamName, IfUtility.MakeArrayString(targetParamArray), e.ElementBits));

                            break;
                        }

                        case ShaderTypeUtility.ParamPrimitiveTypeKind.@uint:
                        {
                            uint[] targetParamArray = G3dDataParser.ParseUIntArray(param.Value);
                            uint[] srcParamArray = e.UintValueArray;

                            for (int i = 0; i != srcParamArray.Length; ++i)
                            {
                                if ((e.ElementBits & (1 << i)) != 0)
                                {
                                    targetParamArray[i] = srcParamArray[i];
                                }
                            }

                            commandSet.Add(CreateEditCommand_shader_assign_shader_param(targets, e.ParamName, IfUtility.MakeArrayString(targetParamArray), e.ElementBits));

                            break;
                        }

                        case ShaderTypeUtility.ParamPrimitiveTypeKind.@float:
                        {
                            float[] targetParamArray = G3dDataParser.ParseFloatArray(param.Value);
                            float[] srcParamArray = e.FloatValueArray;

                            for (int i = 0; i != srcParamArray.Length; ++i)
                            {
                                if ((e.ElementBits & (1 << i)) != 0)
                                {
                                    targetParamArray[i] = srcParamArray[i];
                                }
                            }

                            commandSet.Add(CreateEditCommand_shader_assign_shader_param(targets, e.ParamName, IfUtility.MakeArrayString(targetParamArray), e.ElementBits));

                            break;
                        }
                    }
                }
                else
                {
                    foreach (Material target in targets.Objects)
                    {
                        shader_paramType param = GetShaderParamFromId(target, e.ParamName);
                        switch (e.ParamElemType)
                        {
                            case ShaderTypeUtility.ParamPrimitiveTypeKind.@bool:
                                {
                                    bool[] targetParamArray = G3dDataParser.ParseInt2BoolArray(param.Value);
                                    bool[] srcParamArray = e.BoolValueArray;

                                    for (int i = 0; i != srcParamArray.Length; ++i)
                                    {
                                        if ((e.ElementBits & (1 << i)) != 0)
                                        {
                                            targetParamArray[i] = srcParamArray[i];
                                        }
                                    }

                                    commandSet.Add(CreateEditCommand_shader_assign_shader_param(new GuiObjectGroup(target), e.ParamName, IfUtility.MakeArrayString(targetParamArray), e.ElementBits));

                                    break;
                                }

                            case ShaderTypeUtility.ParamPrimitiveTypeKind.@int:
                                {
                                    int[] targetParamArray = G3dDataParser.ParseIntArray(param.Value);
                                    int[] srcParamArray = e.IntValueArray;

                                    for (int i = 0; i != srcParamArray.Length; ++i)
                                    {
                                        if ((e.ElementBits & (1 << i)) != 0)
                                        {
                                            targetParamArray[i] = srcParamArray[i];
                                        }
                                    }

                                    commandSet.Add(CreateEditCommand_shader_assign_shader_param(new GuiObjectGroup(target), e.ParamName, IfUtility.MakeArrayString(targetParamArray), e.ElementBits));

                                    break;
                                }

                            case ShaderTypeUtility.ParamPrimitiveTypeKind.@uint:
                                {
                                    uint[] targetParamArray = G3dDataParser.ParseUIntArray(param.Value);
                                    uint[] srcParamArray = e.UintValueArray;

                                    for (int i = 0; i != srcParamArray.Length; ++i)
                                    {
                                        if ((e.ElementBits & (1 << i)) != 0)
                                        {
                                            targetParamArray[i] = srcParamArray[i];
                                        }
                                    }

                                    commandSet.Add(CreateEditCommand_shader_assign_shader_param(new GuiObjectGroup(target), e.ParamName, IfUtility.MakeArrayString(targetParamArray), e.ElementBits));

                                    break;
                                }

                            case ShaderTypeUtility.ParamPrimitiveTypeKind.@float:
                                {
                                    float[] targetParamArray = G3dDataParser.ParseFloatArray(param.Value);
                                    float[] srcParamArray = e.FloatValueArray;

                                    for (int i = 0; i != srcParamArray.Length; ++i)
                                    {
                                        if ((e.ElementBits & (1 << i)) != 0)
                                        {
                                            targetParamArray[i] = srcParamArray[i];
                                        }
                                    }

                                    commandSet.Add(CreateEditCommand_shader_assign_shader_param(new GuiObjectGroup(target), e.ParamName, IfUtility.MakeArrayString(targetParamArray), e.ElementBits));

                                    break;
                                }
                        }
                    }
                }
                TheApp.CommandManager.Execute(commandSet);
                App.AppContext.OnMaterialReferenceBehaviorSettingChanged();
            }
        }

        public void scgGroup_ValueChanged(Material activeTarget, GuiObjectGroup targets, OptionValueChangedEventArgs e)
        {
            if (e.IsSequentialChanging)
            {
                if (targets.Objects.Count > 0 && Viewer.Manager.Instance.IsConnected)
                {
                    targets = ApplicableOptionObjects(activeTarget, targets, e.ParamName);
                    var descendantMaterials = new List<GuiObject>();
                    foreach (var material in targets.GetObjects(GuiObjectID.Material).OfType<Material>())
                    {
                        descendantMaterials.Add(material);
                        descendantMaterials.AddRange(material.GetAllDescendantShaderOptionMaterials(e.ParamName));
                    }

                    if (descendantMaterials.Any())
                    {
                        Material target = targets.Objects[0] as Material;
                        shader_optionType option = GetShaderOptionFromId(target, e.ParamName);

                        // ビューア転送
                        Viewer.ViewerUtility.SendMaterialShaderOption(new GuiObjectGroup(descendantMaterials.Distinct()), e, option);
                    }
                }
            }
            else
            {
                targets = ApplicableOptionObjects(activeTarget, targets, e.ParamName);
                if (e is CombinerOptionValueChangedEventArgs)
                {
                    TheApp.CommandManager.Execute(CreateEditCommand_combiner_option(targets, e.ParamName, (string)e.ParamValue, true));
                }
                else
                {
                    TheApp.CommandManager.Execute(CreateEditCommand_shader_option(targets, e.ParamName, (string)e.ParamValue, true));
                }
                App.AppContext.OnMaterialReferenceBehaviorSettingChanged();
            }
        }

        public void scgGroup_ValueChanged(Material activeTarget, GuiObjectGroup targets, AttributeValueChangedEventArgs e)
        {
            targets = ApplicableAttribObjects(activeTarget, targets, e.ParamName, (string)e.ParamValue);
            var commandSet = new EditCommandSet();
            {
                using (var block = new Viewer.ViewerDrawSuppressBlock())
                {
                    commandSet.Add(CreateEditCommand_attrib_assign(targets, e.ParamName, (string)e.ParamValue, true).Execute());
                    commandSet.Add(ShaderAssignUtility.CreateVtxBufferEditCommand(targets).Execute());
                }
            }
            commandSet.Reverse();
            TheApp.CommandManager.Add(commandSet);
            App.AppContext.OnMaterialReferenceBehaviorSettingChanged();
        }

        public void scgGroup_ValueChanged(Material activeTarget, GuiObjectGroup targets, RenderInfoValueChangedEventArgs e)
        {
            targets = ApplicableRenderInfoObjects(activeTarget, targets, e.ParamName);
            if (e.IsSequentialChanging)
            {
                // HIOに送る
                if (Viewer.Manager.Instance.IsConnected)
                {
                    var renderSlots = activeTarget.MaterialShaderAssign.Definition.Data.RenderInfoSlots();
                    var values = (List<string>)e.ParamValue;
                    var descendantMaterials = new List<Material>();
                    foreach (var material in targets.GetObjects(GuiObjectID.Material).OfType<Material>())
                    {
                        descendantMaterials.Add(material);
                        descendantMaterials.AddRange(material.GetAllDescendantRenderInfoMaterials(e.ParamName));
                    }

                    Viewer.TransactionMessage.Send(true, overInterval: true, waitRuntimeState: true);
                    foreach (var target in descendantMaterials.Distinct().Where(x => x.OwnerDocument is Model))
                    {
                        Viewer.SetRenderInfo.Send(target, e.ParamName, e.type, values, false);
                    }
                    Viewer.TransactionMessage.Send(false, overInterval: true, waitRuntimeState: true);
                }
            }
            else
            {
                TheApp.CommandManager.Execute(CreateEditCommand_RenderInfo(targets, e.ParamName, (List<string>)e.ParamValue, false));
                App.AppContext.OnMaterialReferenceBehaviorSettingChanged();
            }
        }

        #region コマンド
        public static GroupEditCommand CreateEditCommand_RenderInfo(GuiObjectGroup targets, string name, List<string> values, bool isDefaultEmpty, bool sendViewer = true)
        {
            return new GeneralGroupReferenceEditCommand<Tuple<List<string>, bool, bool>>(
                targets,
                GuiObjectID.Material,
                Enumerable.Range(0, targets.Objects.Count)
                          .Select(x => new Tuple<List<string>, bool, bool>(ObjectUtility.Clone(values), false, isDefaultEmpty)),
                delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        Debug.Assert(target is Material);
                        var material = (Material) target;
                        var param = material.MaterialShaderAssign.RenderInfos.First(x => x.name == name);

                        swap = new Tuple<List<string>, bool, bool>(param.internalValues, param.candidateModified, param.IsDefaultEmpty);
                        var tuple = (Tuple<List<string>, bool, bool>) data;
                        param.internalValues = tuple.Item1;
                        param.candidateModified = tuple.Item2;
                        param.IsDefaultEmpty = tuple.Item3;

                        // HIOに送る
                        if (sendViewer)
                        {
                            List<string> paramValues = new List<string>();

                            // 選択肢から文字列を送信するパターン
                            if (param.type == render_info_typeType.@string)
                            {
                                var renderSlots = material.MaterialShaderAssign.Definition.Data.RenderInfoSlots();
                                render_info_slotType slot = renderSlots.FirstOrDefault(x => x.name == param.name);
                                if (slot != null && slot.ui_item != null && slot.ui_item.value == ui_item_valueType.check)
                                {
                                    if(param.values.Count > 0)
                                    {
                                        string[] choice = slot.choice.Split(',');
                                        int choiceIndex = Array.IndexOf(choice, param.values[0]);
                                        if (choiceIndex == -1)
                                        {
                                            // ランタイムに送る場合は数値を文字列に変換して送る
                                            if (int.TryParse(param.values[0], out choiceIndex))
                                            {
                                                if (choiceIndex < choice.Length)
                                                {
                                                    paramValues.Add(choice[choiceIndex]);
                                                }
                                            }
                                        }
                                    }
                                }
                            }

                            // 上記の特殊ケースに該当しない場合はそのまま送信する
                            if (paramValues.Count == 0)
                            {
                                paramValues = param.values;
                            }

                            var descendantMaterials = new List<Material>();
                            foreach (var targetMaterial in targets.GetObjects(GuiObjectID.Material).OfType<Material>())
                            {
                                descendantMaterials.Add(targetMaterial);
                                descendantMaterials.AddRange(targetMaterial.GetAllDescendantRenderInfoMaterials(name));
                            }
                            foreach (var descendantMaterial in descendantMaterials.Distinct().Where(x => x.OwnerDocument is Model))
                            {
                                Viewer.SetRenderInfo.Send(descendantMaterial, name, param.type, paramValues, true);
                            }
                        }
                    }
                ,
                preEditDelegate: (a, o) =>
                    {
                        if (sendViewer)
                        {
                            // メッセージをまとめるタイミングをまたいでもいいようにする
                            Viewer.TransactionMessage.Send(true, overInterval: true);
                        }
                    },
                postEditDelegate: (a, o) =>
                    {
                        if (sendViewer)
                        {
                            Viewer.QueryRenderInfoQueue.Send();
                            Viewer.TransactionMessage.Send(false, overInterval: true);
                        }
                    },
                createEventArgsDelegate: (x, y) => new DocumentPropertyChangedShaderArgs(x, y)
            );
        }

        public static GroupEditCommand CreateEditCommand_shader_assign_shader_param(GuiObjectGroup targets, string paramName, string paramValue, uint elementBits, bool sendViewer = true)
        {
            return
                 new GeneralGroupReferenceEditCommand<string>(
                    targets,
                    GuiObjectID.Material,
                    Enumerable.Repeat(paramValue, targets.Objects.Count),
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        Debug.Assert(target is Material);
                        Material material = target as Material;

                        shader_paramType param = GetShaderParamFromId(material, paramName);

                        swap = param.Value;

                        switch (ShaderTypeUtility.ParamPrimitiveTypeFromType(param.type))
                        {
                            case ShaderTypeUtility.ParamPrimitiveTypeKind.@bool:
                            {
                                var srcArray = G3dDataParser.ParseInt2BoolArray((string)data);
                                var dstArray = G3dDataParser.ParseInt2BoolArray(param.Value);

                                for (int i = 0; i != srcArray.Length; ++i)
                                {
                                    if ((elementBits & (1 << i)) != 0)
                                    {
                                        dstArray[i] = srcArray[i];
                                    }
                                }

                                param.Value = IfUtility.MakeArrayString(dstArray);

                                break;
                            }

                            case ShaderTypeUtility.ParamPrimitiveTypeKind.@int:
                            {
                                var srcArray = G3dDataParser.ParseIntArray((string)data);
                                var dstArray = G3dDataParser.ParseIntArray(param.Value);

                                for (int i = 0; i != srcArray.Length; ++i)
                                {
                                    if ((elementBits & (1 << i)) != 0)
                                    {
                                        dstArray[i] = srcArray[i];
                                    }
                                }

                                param.Value = IfUtility.MakeArrayString(dstArray);

                                break;
                            }

                            case ShaderTypeUtility.ParamPrimitiveTypeKind.@uint:
                            {
                                var srcArray = G3dDataParser.ParseUIntArray((string)data);
                                var dstArray = G3dDataParser.ParseUIntArray(param.Value);

                                for (int i = 0; i != srcArray.Length; ++i)
                                {
                                    if ((elementBits & (1 << i)) != 0)
                                    {
                                        dstArray[i] = srcArray[i];
                                    }
                                }

                                param.Value = IfUtility.MakeArrayString(dstArray);

                                break;
                            }

                            case ShaderTypeUtility.ParamPrimitiveTypeKind.@float:
                            {
                                var srcArray = G3dDataParser.ParseFloatArray((string)data);
                                var dstArray = G3dDataParser.ParseFloatArray(param.Value);

                                for (int i = 0; i != srcArray.Length; ++i)
                                {
                                    if ((elementBits & (1 << i)) != 0)
                                    {
                                        dstArray[i] = srcArray[i];
                                    }
                                }

                                param.Value = IfUtility.MakeArrayString(dstArray);

                                break;
                            }
                        }
                    },
                    postEditDelegate: delegate(System.Collections.ArrayList editTargets, object[] data)
                    {
                        if (sendViewer)
                        {
                            // ビューア転送
                            if (editTargets.Count > 0)
                            {
                                var param = GetShaderParamFromId((Material)editTargets[0], paramName);

                                var materials = editTargets.OfType<Material>().ToList();
                                var descendantMaterials = new List<GuiObject>(materials);
                                materials.ForEach(x => descendantMaterials.AddRange(x.GetAllDescendantMaterials(param)));
                                editTargets = new ArrayList(descendantMaterials.Distinct().Where(x => x.OwnerDocument is Model).ToArray());

                                // HIO
                                {
                                    var editData = new Viewer.MaterialShaderParameter.Data()
                                    {
                                        ParamName = paramName,
                                        ShaderParamType = param.type,
                                    };

                                    switch (ShaderTypeUtility.ParamPrimitiveTypeFromType(param.type))
                                    {
                                        case ShaderTypeUtility.ParamPrimitiveTypeKind.@bool: editData.Params = G3dDataParser.ParseInt2BoolArray((string)data[0]); break;
                                        case ShaderTypeUtility.ParamPrimitiveTypeKind.@int: editData.Params = G3dDataParser.ParseIntArray((string)data[0]); break;
                                        case ShaderTypeUtility.ParamPrimitiveTypeKind.@uint: editData.Params = G3dDataParser.ParseUIntArray((string)data[0]); break;
                                        case ShaderTypeUtility.ParamPrimitiveTypeKind.@float: editData.Params = G3dDataParser.ParseFloatArray((string)data[0]); break;
                                    }
                                    Debug.Assert(editData.Params != null);

                                    Viewer.MaterialShaderParameter.Send(
                                        editTargets,
                                        Enumerable.Repeat(editData, data.Length).ToArray(),
                                        0xFFFFFFFF,
                                        elementBits
                                    );
                                }
                            }
                        }
                    },
                    createEventArgsDelegate: (x, y) => new DocumentPropertyChangedShaderArgs(x, y)
                );
        }

        public static GroupEditCommand CreateEditCommand_shader_option(GuiObjectGroup targets, string paramName, string paramValue, bool sendViewer)
        {
            return new GeneralGroupReferenceEditCommand<string>(
                targets,
                GuiObjectID.Material,
                Enumerable.Repeat(paramValue, targets.Objects.Count),
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    Material material = target as Material;
                    shader_optionType option = GetShaderOptionFromId(material, paramName);
                    swap = option.value;
                    option.value = (string)data;
                },
                postEditDelegate: delegate(System.Collections.ArrayList editTargets, object[] data)
                {
                    // ビューア転送
                    if (sendViewer && editTargets.Count > 0)
                    {
                        var option = GetShaderOptionFromId((Material)editTargets[0], paramName);

                        var materials = editTargets.OfType<Material>().ToList();
                        var descendantMaterials = new List<GuiObject>(materials);
                        materials.ForEach(x => descendantMaterials.AddRange(x.GetAllDescendantMaterials(option)));
                        editTargets = new ArrayList(descendantMaterials.Distinct().ToArray());

                        // オプションの変更時にはシェーダーのバイナライズが行われるが、失敗済みであれば再バイナライズが行われない。
                        // コンバイナー指定が変更された場合には常に再バイナライズを行うようために失敗済みフラグを無視する設定を行うが、
                        // ここで変更するオプションにはコンバイナーオプションは含まれないため、常に無視しなくてよい。
                        // コンバイナーオプション変更は CreateEditCommand_combiner_option() を参照。
                        var ignoreLastErrorOfShaderBinarization = false;

                        // ビューアへ送信
                        Viewer.ViewerUtility.SendMaterialShaderOption(editTargets, data, option);

                        foreach(Model m in editTargets.OfType<Material>().SelectMany(x => x.Referrers).Distinct())
                        {
                            Viewer.LoadOrReloadModel.Send(m, ignoreLastErrorOfShaderBinarization);
                        }
                    }
                },
                createEventArgsDelegate: (x, y) => new DocumentPropertyChangedShaderOptionArgs(x, y)
            );
        }

        public static GroupEditCommand CreateEditCommand_combiner_option(GuiObjectGroup targets, string paramName, string paramValue, bool sendViewer)
        {
            return new GeneralGroupReferenceEditCommand<string>(
                targets,
                GuiObjectID.Material,
                Enumerable.Repeat(paramValue, targets.Objects.Count),
                delegate (ref GuiObject target, ref object data, ref object swap)
                {
                    Material material = target as Material;
                    var combinerOptions = material.CombinerOptionInfo?.CombinerOptions?.ToDictionary(x => x.Id, x => x);
                    if (combinerOptions == null)
                    {
                        combinerOptions = new Dictionary<string, CombinerOption>();
                    }

                    shader_optionType option = GetShaderOptionFromId(material, paramName);
                    CombinerOption oCombinerOption;
                    swap = (combinerOptions.TryGetValue(paramName, out oCombinerOption) && !string.IsNullOrEmpty(oCombinerOption.FileName)) ? oCombinerOption.FileName : CombinerShaderConverterManager.ReservedValue.Unset;
                    option.value = (string)data;

                    CombinerOption combinerOption;
                    if (!combinerOptions.TryGetValue(paramName, out combinerOption))
                    {
                        combinerOption = new CombinerOption()
                        {
                            Id = paramName
                        };
                        if (material.CombinerOptionInfo == null)
                        {
                            material.CombinerOptionInfo = new nw3de_CombinerOptionInfo();
                        }
                        if (material.CombinerOptionInfo.CombinerOptions == null)
                        {
                            material.CombinerOptionInfo.CombinerOptions = new List<CombinerOption>();
                        }
                        material.CombinerOptionInfo.CombinerOptions.Add(combinerOption);
                    }
                    combinerOption.FileName = (string)data;
                    if (CombinerShaderConverterManager.ReservedValue.IsNullOrEmptyOrUnset(option.value))
                    {
                        // ファイル未指定の場合はファイル名を null にしておく。
                        combinerOption.FileName = null;
                    }
                    else
                    {
                        // オプション値をファイル名から生成したハッシュ値に変更する。
                        option.value = CombinerShaderConverterManager.ComputeHashUInt32(option.value).ToString();
                    }
                },
                postEditDelegate: delegate (System.Collections.ArrayList editTargets, object[] data)
                {
                    // ビューア転送
                    if (sendViewer && editTargets.Count > 0)
                    {
                        var option = GetShaderOptionFromId((Material)editTargets[0], paramName);

                        var materials = editTargets.OfType<Material>().ToList();
                        var descendantMaterials = new List<GuiObject>(materials);
                        materials.ForEach(x => descendantMaterials.AddRange(x.GetAllDescendantMaterials(option)));
                        editTargets = new ArrayList(descendantMaterials.Distinct().ToArray());

                        // オプションの変更時にはシェーダーのバイナライズが行われるが、失敗済みであれば再バイナライズが行われない。
                        // しかし、コンバイナー指定が変更された場合には常に再バイナライズを行うようにしたいので、失敗済みフラグを無視する設定を行う。
                        // ここで変更するオプションはコンバイナーオプションのみのため、常に無視する。
                        // コンバイナーオプション以外のオプション変更は CreateEditCommand_shader_option() を参照。
                        var ignoreLastErrorOfShaderBinarization = true;

                        // ビューアへ送信
                        // data にはコンバイナーファイル名が入っているので、ハッシュ値に変換するしたものを送る。
                        var hash = data.Cast<string>().Select(x =>
                            CombinerShaderConverterManager.ReservedValue.IsNullOrEmptyOrUnset(x) ?
                            CombinerShaderConverterManager.ReservedValue.Unset :
                            CombinerShaderConverterManager.ComputeHashUInt32(x).ToString()).ToArray();
                        Viewer.ViewerUtility.SendMaterialShaderOption(editTargets, hash, option);

                        foreach (Model m in editTargets.OfType<Material>().Select(x => x.Owner).Distinct())
                        {
                            Viewer.LoadOrReloadModel.Send(m, ignoreLastErrorOfShaderBinarization);
                        }
                    }
                },
                createEventArgsDelegate: (x, y) => new DocumentPropertyChangedShaderOptionArgs(x, y)
            );
        }

        public static GroupEditCommand CreateEditCommand_attrib_assign(GuiObjectGroup targets, string paramName, string paramValue, bool sendViewer)
        {
            // targets 毎のプリセット検索を効率的に行うためのコンテナ。
            var presetValues = new HashSet<string>(ApplicationConfig.Preset.VertexAttributeAssignPresets.Select(x => x.Name));

            // targets 毎に paramName を適用できるか確認。
            // 同様の条件確認が ShaderParamControlGroup.ApplicableAttribObjects() にもあるので、条件を変更した場合はそちらも変更すること。
            var values = new List<string>();
            foreach (var material in targets.GetObjects(GuiObjectID.Material).OfType<Material>())
            {
                var shadingModel = material.MaterialShaderAssign.ShadingModel;
                var attrib_var = shadingModel.attrib_var_array.GetItems().First(x => x.id == paramName);
                bool accepted = false;
                if (material.Referrers.Any())
                {
                    var validAttribs = IfAttributeAssignUtility.AssignableAttribs(material.Referrers.First().Data, material.Name, attrib_var);
                    accepted = validAttribs.Select(x => x.Key).Any(x => x == paramValue);
                }
                else if (material.OwnerDocument is SeparateMaterial)
                {
                    accepted = presetValues.Contains(paramValue);
                }
                values.Add(accepted ? paramValue : string.Empty);
            }

            return new GeneralGroupReferenceEditCommand<string>(
                targets,
                GuiObjectID.Material,
                values,
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    Material material = target as Material;
                    attrib_assignType attrib = GetAttribAssignFromId(material, paramName);
                    swap = attrib.attrib_name;
                    attrib.attrib_name = (string)data;
                },
                postEditDelegate: delegate(System.Collections.ArrayList editTargets, object[] data)
                {
                    // ビューア転送
                    if (sendViewer && editTargets.Count > 0)
                    {
                        System.Collections.ArrayList sendTargets = new System.Collections.ArrayList();
                        List<Viewer.SetMaterialAttribAssign.PacketData> sendData = new List<Viewer.SetMaterialAttribAssign.PacketData>();
                        int count = 0;

                        var materials = editTargets.OfType<Material>().ToList();
                        var descendantMaterials = new List<GuiObject>(materials);
                        materials.ForEach(x => descendantMaterials.AddRange(x.GetAllDescendantAttribAssignMaterials(paramName)));
                        editTargets = new ArrayList(descendantMaterials.Distinct().ToArray());

                        foreach (var value in data)
                        {
                            var material = (Material)editTargets[count];
                            var attribAssign = material.MaterialShaderAssign.AttribAssigns.FirstOrDefault(x => x.id == paramName);
                            Debug.Assert(attribAssign != null);
                            sendTargets.Add(material);
                            Viewer.SetMaterialAttribAssign.PacketData param = new Viewer.SetMaterialAttribAssign.PacketData(attribAssign.id, (string)value);
                            sendData.Add(param);
                            count++;
                        }

                        if (sendTargets.Count > 0)
                        {
                            // メッセージを送る。
                            // アトリビュートアサイン変更時にマテリアルのUpdateが必要なので、DisableDrawing、OnUpdateBegin等のメッセージで挟みます。
                            using (var vdsb = new Viewer.ViewerDrawSuppressBlock())
                            {
                                Viewer.SetMaterialAttribAssign.Send(sendTargets, sendData.ToArray(), 0xFFFFFFFF, 0xFFFFFFFF);
                            }
                        }

                        foreach(Model m in editTargets.OfType<Material>().SelectMany(x => x.Referrers).Distinct())
                        {
                            Viewer.LoadOrReloadModel.Send(m);
                        }
                    }
                }
            );
        }

        public static GuiObjectGroup ApplicableParamObjects(Material active, GuiObjectGroup group, string id)
        {
            var applicableGroup = new GuiObjectGroup();
            // シェーダー定義はなくてもいい
            var shader_param = active.MaterialShaderAssign.ShaderParams.First(x => x.id == id);

            foreach (Material item in group.GetObjects(GuiObjectID.Material))
            {
                if (item.MaterialShaderAssign.ShaderParams.Any(x => x.id == id && x.type == shader_param.type) &&
                    item.MaterialShaderAssign.ShadingModel.MaterialUniforms().Any(x => x.id == id && x.IsParamVisible())
                    )
                {
                    applicableGroup.Add(item);
                }
            }

            return applicableGroup;
        }

        public static GuiObjectGroup ApplicableOptionObjects(Material active, GuiObjectGroup group, string id)
        {
            var applicableGroup = new GuiObjectGroup();
            var activeProgram = active.MaterialShaderAssign.ShadingModel;
            if (activeProgram == null)
            {
                return applicableGroup;
            }

            option_varType activeParam = activeProgram.Options().FirstOrDefault(x => x.id == id);
            if (activeParam == null || activeParam.type == option_var_typeType.dynamic)
            {
                return applicableGroup;
            }

            foreach (Material material in group.GetObjects(GuiObjectID.Material))
            {
                var program = material.MaterialShaderAssign.ShadingModel;

                if (program == null)
                {
                    continue;
                }

                var option = program.Options().FirstOrDefault(y => y.id == id);
                if (option != null && option.OptionVisible() && ShaderTypeUtility.OptionVarComparer.OptionEquals(activeParam, option))
                {
                    applicableGroup.Add(material);
                }
            }

            return applicableGroup;
        }

        public static GuiObjectGroup ApplicableAttribObjects(Material active, GuiObjectGroup group, string id, string attrib_name)
        {
            var applicableGroup = new GuiObjectGroup();
            var activeProgram = active.MaterialShaderAssign.ShadingModel;
            if (activeProgram == null)
            {
                return applicableGroup;
            }

            attrib_varType activeParam = activeProgram.Attributes().FirstOrDefault(x => x.id == id);
            if (activeParam == null || !activeParam.Editable())
            {
                return applicableGroup;
            }

            foreach (Material material in group.GetObjects(GuiObjectID.Material))
            {
                var program = material.MaterialShaderAssign.ShadingModel;

                if (program == null)
                {
                    continue;
                }

                // ここでの条件確認と同様の処理が ShaderParamControlGroup.CreateEditCommand_attrib_assign() にもある。
                var param = program.Attributes().FirstOrDefault(y => y.id == id);
                if (param != null &&
                    object.Equals(activeParam, param) && // Object.Equals が呼ばれている!
                    (string.IsNullOrEmpty(attrib_name) ||
                     material.Referrers.Any(x => IfAttributeAssignUtility.AssignableAttribs(x.Data, material.Name, param).Any(y => y.Key == attrib_name)) ||
                     (!material.Referrers.Any() && (material.OwnerDocument is SeparateMaterial) && ApplicationConfig.Preset.VertexAttributeAssignPresets.Any(x => x.Name == attrib_name))))
                {
                    applicableGroup.Add(material);
                }
            }

            return applicableGroup;
        }

        public static GuiObjectGroup ApplicableRenderInfoObjects(Material active, GuiObjectGroup group, string id)
        {
            var applicableGroup = new GuiObjectGroup();
            var activeProgram = active.MaterialShaderAssign.ShadingModel;
            if (activeProgram == null)
            {
                return applicableGroup;
            }

            render_info_slotType slot = activeProgram.RenderInfoSlots().FirstOrDefault(x=> x.name == id);
            if ((slot == null) || !slot.RenderInfoVisible())
            {
                return applicableGroup;
            }

            var activePack = active.RenderInfoPackFromHio;
            IntRenderInfoCandidate activeInt = null;
            FloatRenderInfoCandidate activeFloat = null;
            StringRenderInfoCandidate activeString = null;
            if (Viewer.Manager.Instance.IsConnected && activePack != null)
            {
                switch (slot.type)
                {
                    case render_info_slot_typeType.@float:
                        activeFloat = activePack.FloatItems.FirstOrDefault(x => x.name == slot.name);
                        break;
                    case render_info_slot_typeType.@int:
                        activeInt = activePack.IntItems.FirstOrDefault(x => x.name == slot.name);
                        break;
                    case render_info_slot_typeType.@string:
                        activeString = activePack.StringItems.FirstOrDefault(x => x.name == slot.name);
                        break;
                }
            }

            foreach (Material material in group.GetObjects(GuiObjectID.Material))
            {
                var program = material.MaterialShaderAssign.ShadingModel;

                if (program == null)
                {
                    continue;
                }

                var param = program.RenderInfoSlots().FirstOrDefault(y => y.name == id);
                if (param != null && param.RenderInfoVisible() && ShaderTypeUtility.RenderInfoSlotComparer.RenderInfoSlotEquals(slot, param))
                {
                    var pack = material.RenderInfoPackFromHio;
                    bool add = false;
                    IntRenderInfoCandidate intCandidate = null;
                    FloatRenderInfoCandidate floatCandidate = null;
                    StringRenderInfoCandidate stringCandidate = null;
                    if (Viewer.Manager.Instance.IsConnected && pack != null)
                    {
                        switch (param.type)
                        {
                            case render_info_slot_typeType.@float:
                                floatCandidate = pack.FloatItems.FirstOrDefault(x => x.name == slot.name);
                                break;
                            case render_info_slot_typeType.@int:
                                intCandidate = pack.IntItems.FirstOrDefault(x => x.name == slot.name);
                                break;
                            case render_info_slot_typeType.@string:
                                stringCandidate = pack.StringItems.FirstOrDefault(x => x.name == slot.name);
                                break;
                        }
                    }

                    switch (param.type)
                    {
                        case render_info_slot_typeType.@float:
                            {
                                var candidate = floatCandidate;
                                if (candidate == activeFloat)
                                {
                                    add = true;
                                }
                                else if (candidate != null && activeFloat != null &&
                                    candidate.hasMinMax == activeFloat.hasMinMax &&
                                    (!candidate.hasMinMax ||
                                    (candidate.min == activeFloat.min && candidate.max == activeFloat.max)))
                                {
                                    add = true;
                                }
                            }
                            break;
                        case render_info_slot_typeType.@int:
                            {
                                var candidate = intCandidate;
                                if (candidate == activeInt)
                                {
                                    add = true;
                                }
                                else if (candidate != null && activeInt != null &&
                                    candidate.hasMinMax == activeInt.hasMinMax &&
                                    (!candidate.hasMinMax ||
                                    (candidate.min == activeInt.min && candidate.max == activeInt.max)))
                                {
                                    add = true;
                                }

                            }
                            break;
                        case render_info_slot_typeType.@string:
                            {
                                var candidate = stringCandidate;
                                if (candidate == activeString)
                                {
                                    add = true;
                                }
                                else if (candidate != null && activeString != null &&
                                    ((candidate.Choices == null && activeString.Choices == null) ||
                                    (candidate != null && activeString != null && candidate.Choices.SequenceEqual(activeString.Choices))))
                                {
                                    add = true;
                                }
                            }
                            break;
                    }

                    if (add == true)
                    {
                        applicableGroup.Add(material);
                    }
                }
            }

            return applicableGroup;
        }

        public static GroupEditCommand CreateEditCommand_shader_assign_shader_param_array_Clear(GuiObjectGroup targets)
        {
            return
                 new GeneralGroupReferenceEditCommand<List<shader_paramType>>(
                    targets,
                    GuiObjectID.Material,
                    targets.Objects.Select(x => (new List<shader_paramType>())),
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        Debug.Assert(target is Material);
                        var material = target as Material;

                        swap = material.MaterialShaderAssign.ShaderParams;
                        material.MaterialShaderAssign.ShaderParams = (List<shader_paramType>)data;
                    }

                    );
        }
        public static GroupEditCommand CreateEditCommand_shader_assign_shader_option_array_Clear(GuiObjectGroup targets)
        {
            return
                 new GeneralGroupReferenceEditCommand<List<shader_optionType>>(
                    targets,
                    GuiObjectID.Material,
                    targets.Objects.Select(x => (new List<shader_optionType>())),
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        Debug.Assert(target is Material);
                        var material = target as Material;

                        swap = material.MaterialShaderAssign.ShaderOptions;
                        material.MaterialShaderAssign.ShaderOptions = (List<shader_optionType>)data;
                    }
                );
        }

        public static GroupEditCommand CreateEditCommand_shader_assign_sampler_assign_array_initialize(GuiObjectGroup targets, string[] id)
        {
            return
                 new GeneralGroupReferenceEditCommand<List<sampler_assignType>>(
                    targets,
                    GuiObjectID.Material,
                    targets.Objects.Select(x => id.Select(y => new sampler_assignType() { id = y }).ToList()), // TODO: 必要なら index_hint 対応
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        Debug.Assert(target is Material);
                        var material = target as Material;

                        swap = material.MaterialShaderAssign.SamplerAssigns;
                        material.MaterialShaderAssign.SamplerAssigns = (List<sampler_assignType>)data;
                    }
                );
        }

        public static GroupEditCommand CreateEditCommand_material_shader_assign_sampler_name(GuiObjectGroup targets, string id, string samplerName, bool sendViewer)
        {
            var values = targets.GetObjects(GuiObjectID.Material).OfType<Material>().Select(x => x.sampler_array.sampler.Any(y => y.name == samplerName) ? samplerName : "");
            return new GeneralGroupReferenceEditCommand<string>(
                targets,
                GuiObjectID.Material,
                values,
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    var material = (Material)target;
                    var samplerAssign = GetSamplerAssignFromId(material, id);
                    Debug.Assert(samplerAssign != null);
                    swap = samplerAssign.sampler_name;
                    samplerAssign.sampler_name = (string)data;
                },
                postEditDelegate: delegate(System.Collections.ArrayList editTargets, object[] data)
                {
                    // ビューア転送
                    if (sendViewer && editTargets.Count > 0)
                    {
                        var sendTargets = new System.Collections.ArrayList();
                        var sampler = GetSamplerAssignFromId((Material)editTargets[0], id);

                        var materials = editTargets.OfType<Material>().ToList();
                        var descendantMaterials = new List<GuiObject>(materials);
                        materials.ForEach(x => descendantMaterials.AddRange(x.GetAllDescendantMaterials(sampler)));
                        editTargets = new ArrayList(descendantMaterials.Distinct().ToArray());

                        foreach (Material material in editTargets)
                        {
                            if (string.IsNullOrEmpty(samplerName) || material.sampler_array?.sampler?.Any(x => x.name == samplerName) == true)
                            {
                                sendTargets.Add(material);
                            }
                        }

                        if (sendTargets.Count > 0)
                        {
                            // メッセージを送る。
                            // サンプラアサイン変更時にマテリアルのUpdateが必要なので、DisableDrawing、OnUpdateBegin等のメッセージで挟みます。
                            using (var vdsb = new Viewer.ViewerDrawSuppressBlock())
                            {
                                foreach (var model in sendTargets.OfType<Material>().SelectMany(x => x.Referrers).Distinct())
                                {
                                    Viewer.LoadOrReloadModel.Send(model);
                                }
                            }
                        }
                    }
                }
                );
        }

        #endregion

        public static GuiObjectGroup ApplicableSamplerObjects(Material active, GuiObjectGroup group, string id)
        {
            var applicableGroup = new GuiObjectGroup();

            foreach (Material material in group.GetObjects(GuiObjectID.Material))
            {
                if (!material.MaterialShaderAssign.ShadingModel.Samplers().Any(x => x.id == id && x.SamplerVisible()))
                {
                    continue;
                }

                if (material.MaterialShaderAssign.SamplerAssigns.Any(x => x.id == id))
                {
                    applicableGroup.Add(material);
                }
            }

            return applicableGroup;
        }

        private static shader_optionType GetShaderOption(Material target, option_varType option_var)
        {
            shader_optionType option = target.MaterialShaderAssign.ShaderOptions.FirstOrDefault(x => x.id == option_var.id);

            // 見つからなかったので新規につくって追加する
            if (option == null)
            {
                option = new shader_optionType();
                {
                    option.id = option_var.id;
                    option.value = option_var.@default;
                };

                target.MaterialShaderAssign.ShaderOptions.Add(option);
            }

            return option;
        }

        private static shader_optionType GetShaderOptionFromId(Material target, string id)
        {
            return target.MaterialShaderAssign.ShaderOptions.First(x => x.id == id);
        }

        private static shader_paramType GetShaderParamFromId(Material target, string id)
        {
            return target.MaterialShaderAssign.ShaderParams.First(x => x.id == id);
        }

        private static sampler_assignType GetSamplerAssignFromId(Material material, string id)
        {
            return material.MaterialShaderAssign.SamplerAssigns.First(x => x.id == id);
        }

        private static samplerType GetSamplerFromName(Material material, string name, out int samplerIndex)
        {
            samplerIndex = 0;

            foreach (var sampler in material.ResolvedSamplers)
            {
                if (sampler.name == name)
                {
                    return sampler;
                }

                ++samplerIndex;
            }

            samplerIndex = -1;
            return null;
        }

        private static attrib_assignType GetAttribAssignFromId(Material material, string id)
        {
            return material.MaterialShaderAssign.AttribAssigns.First(x => x.id == id);
        }

        #region グループのコピペ
        /// <summary>
        /// グループのコピペ用データ
        /// </summary>
        public class GroupCopyData
        {

            public string group_id { get; set; }

            // コピーデータ
            public List<Tuple<attrib_varType, attrib_assignType>> AttribAssigns { get; set; }
            public List<Tuple<option_varType, shader_optionType>> ShaderOptions { get; set; }
            public List<Tuple<sampler_varType, sampler_assignType>> SamplerAssigns { get; set; }
            public List<Tuple<uniform_varType, shader_paramType>> ShaderParams { get; set; }
            public List<Tuple<render_info_slotType, RenderInfo>> RenderInfos { get; set; }

            public GroupCopyData()
            {
                AttribAssigns = new List<Tuple<attrib_varType, attrib_assignType>>();
                ShaderOptions = new List<Tuple<option_varType, shader_optionType>>();
                SamplerAssigns = new List<Tuple<sampler_varType, sampler_assignType>>();
                ShaderParams = new List<Tuple<uniform_varType, shader_paramType>>();
                RenderInfos = new List<Tuple<render_info_slotType, RenderInfo>>();
            }
        }

        public static GroupCopyData CreatePageCopyData(Material material, bool pageMode, string page, bool checkEditable, bool checkVisible)
        {
            var shading_model = material.MaterialShaderAssign.ShadingModel;
            var groups = CreateIsGroupInPage(null, pageMode, shading_model, page);
            var data = CreateItemsCopyData(material.MaterialShaderAssign, shading_model, groups, checkEditable, checkVisible);

            return data;
        }

        public static GroupCopyData CreatePageCopyData(MaterialShaderAssign shaderAssign, shading_modelType currentProgram, bool pageMode, string page, bool checkEditable, bool checkVisible)
        {
            var groups = CreateIsGroupInPage(null, pageMode, currentProgram, page);
            return CreateItemsCopyData(shaderAssign, currentProgram, groups, checkEditable, checkVisible);
        }

        public static GroupCopyData CreateGroupCopyData(Material material, string group_id, bool checkEditable, bool checkVisible)
        {
            return CreateGroupCopyData(material.MaterialShaderAssign, material.MaterialShaderAssign.ShadingModel, group_id, checkEditable, checkVisible);
        }

        public static GroupCopyData CreateGroupCopyData(MaterialShaderAssign shaderAssign, shading_modelType currentProgram, string group_id, bool checkEditable, bool checkVisible)
        {
            var groups = CreateIsGroupInPage(group_id, false, currentProgram, null);
            var data = CreateItemsCopyData(shaderAssign, currentProgram, groups, checkEditable, checkVisible);
            data.group_id = group_id;
            return data;
        }

        private static GroupCopyData CreateItemsCopyData(MaterialShaderAssign shaderAssign, shading_modelType currentProgram, Predicate<string> group_id, bool checkEditable, bool checkVisible)
        {
            var data = new GroupCopyData
            {
            };

            if (currentProgram == null)
            {
                return data;
            }

            // 頂点属性
            data.AttribAssigns = (from attrib_var in currentProgram.Attributes()
                                  where (!checkEditable || attrib_var.Editable()) && group_id(attrib_var.Group())
                                  let attrib = shaderAssign.AttribAssigns.FirstOrDefault(x => x.id == attrib_var.id)
                                  where attrib != null
                                  select new Tuple<attrib_varType, attrib_assignType>(attrib_var, ObjectUtility.Clone(attrib))
                                  ).ToList();

            // オプション
            data.ShaderOptions = (from option_var in currentProgram.Options()
                                  where (!checkVisible || option_var.OptionVisible()) && option_var.type != option_var_typeType.dynamic && group_id(option_var.Group())
                                  let option = shaderAssign.ShaderOptions.FirstOrDefault(x => x.id == option_var.id)
                                  where option != null
                                  select new Tuple<option_varType, shader_optionType>(option_var, ObjectUtility.Clone(option))).ToList();

            // サンプラ割り当て
            data.SamplerAssigns = (from sampler_var in currentProgram.Samplers()
                                   where (!checkVisible || sampler_var.SamplerVisible()) && group_id(sampler_var.Group())
                                   let sampler = shaderAssign.SamplerAssigns.FirstOrDefault(x => x.id == sampler_var.id)
                                   where sampler != null
                                   select new Tuple<sampler_varType, sampler_assignType>(sampler_var, ObjectUtility.Clone(sampler))
                                   ).ToList();

            // パラメーター
            data.ShaderParams = (from uniform_var in currentProgram.MaterialUniforms()
                                 where (!checkVisible || uniform_var.IsParamVisible()) && group_id(uniform_var.Group())
                                 let param = shaderAssign.ShaderParams.FirstOrDefault(x => x.id == uniform_var.id)
                                 where param != null
                                 select new Tuple<uniform_varType, shader_paramType>(uniform_var, ObjectUtility.Clone(param))).ToList();

            // 描画情報
            data.RenderInfos = (from slot in currentProgram.RenderInfoSlots()
                                where (!checkVisible || slot.RenderInfoVisible()) && group_id(slot.Group())
                                let info = shaderAssign.RenderInfos.FirstOrDefault(x => x.name == slot.name)
                                where info != null
                                select new Tuple<render_info_slotType, RenderInfo>(slot, ObjectUtility.Clone(info))).ToList();

            return data;
        }

        public static EditCommandSet CreateItemsPasteCommand(
            GuiObjectGroup targets,
            GroupCopyData data,
            //Predicate<string> checkGroupId,
            string group_id,
            bool pageMode,
            string page,
            bool checkEditable,
            bool ReloadModel,
            bool isDefaultShaderAssign = false,
            bool sendViewer = true,
            bool checkVisible = true)
        {
            // リロードする場合はコマンドセットでラップする
            if (ReloadModel)
            {
                EditCommandSet pasteCommands = CreateItemsPasteCommand(targets, data, group_id, pageMode, page, checkEditable, false, isDefaultShaderAssign, sendViewer, checkVisible);
                if (pasteCommands == null)
                {
                    return null;
                }
                pasteCommands.SetViewerDrawSuppressBlockDelegate(Viewer.ViewerDrawSuppressBlock.DiscardAllMessages);
                EditCommandSet commandSet2 = new EditCommandSet();
                var models = targets.Objects.OfType<Material>().SelectMany(x => x.Referrers).Distinct().ToArray();
                commandSet2.OnPostEdit += (s, e) =>
                    {
                        if (sendViewer)
                        {
                            foreach (var model in models)
                            {
                                Viewer.LoadOrReloadModel.Send(model);
                            }
                        }
                    };
                commandSet2.Add(pasteCommands);
                return commandSet2;
            }

            EditCommandSet commandSet = new EditCommandSet();
            var materials = (from material in targets.Objects.OfType<Material>()
                             let shadingModel = material.MaterialShaderAssign.ShadingModel
                             where shadingModel != null
                             let inGroup = CreateIsGroupInPage(group_id, pageMode, shadingModel, page)
                             select new { material, inGroup }).ToArray();

            bool attribModified = false;
            foreach (var attrib in data.AttribAssigns.Where(x => x.Item2 != null))
            {
                GuiObjectGroup applicableTargets = new GuiObjectGroup(
                    materials.Where(x => AttribCanPaste(attrib.Item1, x.material, attrib.Item2.attrib_name, x.inGroup, checkEditable)).Select(x => x.material));
                if (applicableTargets.Objects.Count > 0)
                {
                    commandSet.Add(CreateEditCommand_attrib_assign(applicableTargets, attrib.Item1.id, attrib.Item2.attrib_name, false));
                    attribModified = true;
                }
            }

            if (attribModified)
            {
                commandSet.Add(new LazyCommand(() => ShaderAssignUtility.CreateVtxBufferEditCommand(targets)));
            }

            foreach (var option in data.ShaderOptions.Where(x => x.Item2 != null))
            {
                GuiObjectGroup applicableTargets = new GuiObjectGroup(materials.Where(x => OptionCanPaste(option.Item1, x.material, x.inGroup, checkEditable)).Select(x => x.material));
                if (applicableTargets.Objects.Count > 0)
                {
                    commandSet.Add(CreateEditCommand_shader_option(applicableTargets, option.Item1.id, option.Item2.value, sendViewer));
                }
            }

            foreach (var sampler in data.SamplerAssigns.Where(x => x.Item2 != null))
            {
                GuiObjectGroup applicableTargets = new GuiObjectGroup(
                    materials.Where(x => SamplerAssignCanPaste(sampler.Item1, x.material, x.inGroup, checkVisible)).Select(x => x.material));
                if (applicableTargets.Objects.Count > 0)
                {
                    commandSet.Add(CreateEditCommand_material_shader_assign_sampler_name(applicableTargets, sampler.Item1.id, sampler.Item2.sampler_name, sendViewer));
                }
            }

            foreach (var param in data.ShaderParams.Where(x => x.Item2 != null))
            {
                GuiObjectGroup applicableTargets = new GuiObjectGroup(
                    materials.Where(x => ParamCanPaste(param.Item1, x.material, x.inGroup, checkVisible)).Select(x => x.material));
                if (applicableTargets.Objects.Count > 0)
                {
                    commandSet.Add(CreateEditCommand_shader_assign_shader_param(applicableTargets, param.Item1.id, param.Item2.Value, 0xFFFFFFFF));
                }
            }

            foreach (var param in data.RenderInfos.Where(x => x.Item2 != null))
            {
                GuiObjectGroup applicableTargets = new GuiObjectGroup(
                    materials.Where(x => RenderInfoCanPaste(param.Item1, x.material, x.inGroup, checkVisible)).Select(x => x.material));
                if (applicableTargets.Objects.Count > 0)
                {
                    commandSet.Add(CreateEditCommand_RenderInfo(applicableTargets, param.Item1.name, param.Item2.values, isDefaultShaderAssign));
                }
            }

            if (commandSet.CommandCount == 0)
            {
                return null;
            }

            return commandSet;
        }

        public static EditCommandSet CreateItemsPasteCommand2(GuiObjectGroup targets, CopyPasteNode source, string group_id)
        {
            var commandSet2 = new EditCommandSet();
            var commandSet = new EditCommandSet();
            commandSet.SetViewerDrawSuppressBlockDelegate(Viewer.ViewerDrawSuppressBlock.DiscardAllMessages);
            var models = new List<Model>();
            foreach (Material material in targets.GetObjects(GuiObjectID.Material))
            {
                var pasteRoot = CreateGroupCopyNodes(group_id, material);//GetCopyPasteNodes(material, material.MaterialShaderAssign.ShadingModel);
                if (source.CanPaste(pasteRoot))
                {
                    models.AddRange(material.Referrers);
                    var targetGroup = new GuiObjectGroup(material);
                    var sources = TreeUtility.PreOrder(source, x => (x is GroupNode) ? ((GroupNode)x).children : Enumerable.Empty<CopyPasteNode>());
                    var pasteNodes = TreeUtility.PreOrder(pasteRoot, x => (x is GroupNode) ? ((GroupNode)x).children : Enumerable.Empty<CopyPasteNode>());
                    bool attribModified = false;
                    foreach (var item in sources.Zip(pasteNodes, (x, y) => new { source = x, target = y }))
                    {
                        var attribNode = item.target as AttribNode;
                        if (attribNode != null)
                        {
                            commandSet.Add(CreateEditCommand_attrib_assign(targetGroup, attribNode.var.id, ((AttribNode)item.source).value.attrib_name, false));
                            attribModified = true;
                            continue;
                        }

                        var optionNode = item.target as OptionNode;
                        if (optionNode != null)
                        {
                            commandSet.Add(CreateEditCommand_shader_option(targetGroup, optionNode.var.id, ((OptionNode)item.source).value.value, false));
                            continue;
                        }

                        var samplerNode = item.target as SamplerNode;
                        if (samplerNode != null)
                        {
                            commandSet.Add(CreateEditCommand_material_shader_assign_sampler_name(targetGroup, samplerNode.var.id, ((SamplerNode)item.source).value.sampler_name, false));
                        }

                        var uniformNode = item.target as UniformNode;
                        if (uniformNode != null)
                        {
                            commandSet.Add(CreateEditCommand_shader_assign_shader_param(targetGroup, uniformNode.var.id, ((UniformNode)item.source).value.Value, unchecked((uint)-1), false));
                        }

                        var renderInfoNode = item.target as RenderInfoNode;
                        if (renderInfoNode != null)
                        {
                            commandSet.Add(CreateEditCommand_RenderInfo(targetGroup, renderInfoNode.var.name, ((RenderInfoNode)item.source).value.values, false));
                        }
                    }

                    if (attribModified)
                    {
                        commandSet.Add(new LazyCommand(() => ShaderAssignUtility.CreateVtxBufferEditCommand(targetGroup)));
                    }
                }
            }

            if (commandSet.CommandCount > 0)
            {
                models = models.Distinct().ToList();
                commandSet2.Add(commandSet);
                commandSet2.OnPostEdit += (s, e) =>
                    {
                        foreach (var model in models)
                        {
                            Viewer.LoadOrReloadModel.Send(model);
                        }
                    };
                return commandSet2;
            }
            else
            {
                return null;
            }
        }

        public static EditCommandSet CreateGroupSetToDefaultCommand(GuiObjectGroup targets, string groupId, bool pageMode, string pageId)
        {
            var materials = targets.GetObjects(GuiObjectID.Material).OfType<Material>().Where(x => x.MaterialShaderAssign.ShadingModel != null);
            var shaderAssigns = (from material in materials
                                 select new Tuple<Material, MaterialShaderAssign>(material,
                                     MaterialShaderPage.GetDefaultShaderAssign(material, material.MaterialShaderAssign.ShaderDefinitionFileName, material.MaterialShaderAssign.ShaderName))).ToArray();

            var commandSet = new EditCommandSet();
            foreach (var item in shaderAssigns)
            {
                GroupCopyData data;
                if (item.Item1.MaterialShaderAssign.PageMode && !MaterialShaderPage.IgnorePage)
                {
                    data = CreatePageCopyData(item.Item2, item.Item1.MaterialShaderAssign.ShadingModel, pageMode, pageId, false, false);
                }
                else
                {
                    data = CreateGroupCopyData(item.Item2, item.Item1.MaterialShaderAssign.ShadingModel, groupId, false, false);
                }

                var command = CreateItemsPasteCommand(new GuiObjectGroup(item.Item1), data, groupId, pageMode, pageId, false, true, true, checkVisible:false);
                if (command != null)
                {
                    commandSet.Add(command);
                }
            }

            if (commandSet.CommandCount != 0)
            {
                return commandSet;
            }
            else
            {
                return null;
            }
        }

        private static bool AttribCanPaste(attrib_varType source, Material material, string attrib_name, Predicate<string> checkGroupId, bool checkEditable)
        {
            var target_var = material.MaterialShaderAssign.ShadingModel.Attributes().FirstOrDefault(x =>
                x.id == source.id &&
                (!checkEditable || x.Editable()));

            return material.MaterialShaderAssign.ShadingModel != null &&
                target_var != null &&
                material.MaterialShaderAssign.AttribAssigns.Any(x => x.id == source.id && checkGroupId(target_var.Group())) &&
                (string.IsNullOrEmpty(attrib_name) ||
                 material.Referrers.Any(x => IfAttributeAssignUtility.AssignableAttribs(x.Data, material.Name, target_var).Any(y => y.Key == attrib_name)));
        }

        private static bool OptionCanPaste(option_varType source, Material material, Predicate<string> checkGroupId, bool checkEditable)
        {
            return material.MaterialShaderAssign.ShadingModel != null &&
                material.MaterialShaderAssign.ShadingModel.Options().Any(x => x.id == source.id && x.type == source.type && (!checkEditable || x.OptionVisible()) && IfShaderOptionUtility.ChoiceEquivalent(x.choice, source.choice) && checkGroupId(x.Group())) &&
                material.MaterialShaderAssign.ShaderOptions.Any(x => x.id == source.id);
        }

        private static bool SamplerAssignCanPaste(sampler_varType source, Material material, Predicate<string> checkGroupId, bool checkEditable)
        {
            return material.MaterialShaderAssign.ShadingModel != null &&
                material.MaterialShaderAssign.ShadingModel.Samplers().Any(x => x.id == source.id && x.type == source.type && (!checkEditable || x.SamplerVisible()) && checkGroupId(x.Group())) &&
                material.MaterialShaderAssign.SamplerAssigns.Any(x => x.id == source.id);
        }

        private static bool ParamCanPaste(uniform_varType source, Material material, Predicate<string> checkGroupId, bool checkEditable)
        {
            return material.MaterialShaderAssign.ShadingModel != null &&
                material.MaterialShaderAssign.ShadingModel.MaterialUniforms().Any(x =>
                    x.id == source.id && x.type == source.type && (!checkEditable || x.IsParamVisible()) && checkGroupId(x.Group())) &&
                material.MaterialShaderAssign.ShaderParams.Any(x => x.id == source.id && x.type == source.type && IsShaderParamEditable(x));
        }

        private static bool RenderInfoCanPaste(render_info_slotType source, Material material, Predicate<string> checkGroupId, bool checkEditable)
        {
            return material.MaterialShaderAssign.ShadingModel != null &&
                material.MaterialShaderAssign.ShadingModel.RenderInfoSlots().Any(
                    x => x.name == source.name && x.type == source.type && x.optional == source.optional && (!checkEditable || x.RenderInfoVisible()) && checkGroupId(x.Group())) &&
                material.MaterialShaderAssign.RenderInfos.Any(x => x.name == source.name && x.type == source.Type());
        }

        private static bool IsShaderParamEditable(shader_paramType shaderParam)
        {
            return string.IsNullOrEmpty(shaderParam.original_hint)
                || !ApplicationConfig.Preset.DisableEditTextureSrtWithOriginal
                || (shaderParam.type != shader_param_typeType.texsrt);
        }

        /// <summary>
        /// ペースト可能か?
        /// </summary>
        public static bool CanItemsPaste(GuiObjectGroup targets, GroupCopyData data, string groupId, bool pageMode, string page, bool checkEditable)
        {
            var materials = (from material in targets.Objects.OfType<Material>()
                             let shadingModel = material.MaterialShaderAssign.ShadingModel
                   		     where shadingModel != null
                             let inGroup = CreateIsGroupInPage(groupId, pageMode, shadingModel, page)
                       	     select new { material, inGroup }).ToArray();

            if (materials.Length == 0)
            {
                return false;
            }

            foreach (var attrib in data.AttribAssigns)
            {
                if (materials.Any(x => AttribCanPaste(attrib.Item1, x.material, attrib.Item2.attrib_name, x.inGroup, checkEditable)))
                {
                    return true;
                }
            }

            foreach (var option in data.ShaderOptions)
            {
                if (materials.Any(x => OptionCanPaste(option.Item1, x.material, x.inGroup, checkEditable)))
                {
                    return true;
                }
            }

            foreach (var sampler in data.SamplerAssigns)
            {
                if (materials.Any(x => SamplerAssignCanPaste(sampler.Item1, x.material, x.inGroup, checkEditable)))
                {
                    return true;
                }
            }

            foreach (var param in data.ShaderParams)
            {
                if (materials.Any(x => ParamCanPaste(param.Item1, x.material, x.inGroup, checkEditable)))
                {
                    return true;
                }
            }

            foreach (var info in data.RenderInfos)
            {
                if (materials.Any(x => RenderInfoCanPaste(info.Item1, x.material, x.inGroup, checkEditable)))
                {
                    return true;
                }
            }

            return false;
        }

        public static bool CanItemsPaste2(GuiObjectGroup targets, CopyPasteNode data, string groupId)
        {
            if (data == null)
            {
                return false;
            }

            var materials = from material in targets.Objects.OfType<Material>()
                            let shadingModel = material.MaterialShaderAssign.ShadingModel
                            where shadingModel != null
                            select material;

            foreach (var material in materials)
            {
                var targetData = CreateGroupCopyNodes(groupId, material);

                if (targetData != null && targetData.CanPaste(data))
                {
                    return true;
                }
            }

            return false;
        }

        public static GroupCopyData groupCopyData;
        public static CopyPasteNode groupCopyData2;

        public static void UpdateGroupCopyData(Material material, string groupId)
        {
            groupCopyData = CreateGroupCopyData(material, groupId, true, true);
            groupCopyData2 = ObjectUtility.Clone(CreateGroupCopyNodes(groupId, material));
        }

        private void cmiCopy_Click(object sender, EventArgs e)
        {
            UpdateGroupCopyData(GetActiveTarget(), (string)lastGroupBox.Tag);
        }

        private void cmiPaste_Click(object sender, EventArgs e)
        {
            var groupBox = (ShaderGroupBox)groupContextMenuStrip.ProxySourceControl;
            if (groupCopyData.group_id == (string)groupBox.Tag && CanItemsPaste(GetTargets(), groupCopyData, groupCopyData.group_id, false, null, true))
            {
                var command = CreateItemsPasteCommand(GetTargets(), groupCopyData, groupCopyData.group_id, false, null, true, true, checkVisible: true);
                if (command != null)
                {
                    TheApp.CommandManager.Execute(command);
                }
            }
            else
            {
                var command = CreateItemsPasteCommand2(GetTargets(), groupCopyData2, (string)groupBox.Tag);
                if (command != null)
                {
                    TheApp.CommandManager.Execute(command);
                }
            }
        }

        private void cmiDefault_Click(object sender, EventArgs e)
        {
            var command = CreateGroupSetToDefaultCommand(GetTargets(), (string)lastGroupBox.Tag, false, null);
            if (command != null)
            {
                TheApp.CommandManager.Execute(command);
            }
        }

        public ShaderGroupBox lastGroupBox;

        private void ContextMenuStrip_Opening(object sender, CancelEventArgs e)
        {
            var groupBox = (ShaderGroupBox)groupContextMenuStrip.ProxySourceControl;
            int y = groupBox.PointToClient(Cursor.Position).Y;
            if (y < 0 || groupBox.TitleBarHeight < y)
            {
                e.Cancel = true;
            }
            else
            {
                lastGroupBox = groupBox;
                var material = GetActiveTarget();
                cmiCopy.Enabled = material != null && material.MaterialShaderAssign.ShadingModel != null;
                var groupId = (string)groupBox.Tag;
                cmiPaste.Enabled = groupCopyData != null &&
                    ((groupCopyData.group_id == groupId &&
                      CanItemsPaste(GetTargets(), groupCopyData, groupId, false, null, true)) ||
                      CanItemsPaste2(GetTargets(), groupCopyData2, groupId));
                if (material == null)
                {
                    return;
                }

                var behaviorItems = material.GetAllReferenceBehaviorsInGroup(groupId, removeNull: false).ToList();
                if (behaviorItems.Contains(null))
                {
                    // null はマテリアル参照設定がデフォルトのパラメータが存在することを表すので追加する
                    behaviorItems.Add(Material.GetDefaultReferenceBehaviorItem("dummy"));
                    behaviorItems.RemoveAll(x => x == null);
                }

                var values = new HashSet<ShaderItemValueState>(behaviorItems.Select(x => x?.Value ?? ShaderItemValueState.Refer));
                var restrictions =
                    new HashSet<ShaderItemRestrictionState>(behaviorItems.Select(x => x?.ChildRestriction ?? ShaderItemRestrictionState.None));

                MaterialPropertyPanel.SetMaterialReferenceItemStateMenus(
                    values,
                    cmiMaterialReferenceBehaviorValue_Refer,
                    cmiMaterialReferenceBehaviorValue_Override,
                    cmiMaterialReferenceBehaviorValue_Default);
                MaterialPropertyPanel.SetMaterialReferenceItemStateMenus(
                    restrictions,
                    cmiMaterialReferenceBehaviorRestriction_None,
                    cmiMaterialReferenceBehaviorRestriction_ForceRefer,
                    cmiMaterialReferenceBehaviorRestriction_ForceOverride);
            }
        }

        private IEnumerable<string> RenderInfoInGroup(string groupName)
        {
            return renderInfoToGroup.Where(d => (string)d.Value.Tag == groupName).Select(x => x.Key);
        }

        private IEnumerable<string> OptionInGroup(string groupName)
        {
            return optionToGroup.Where(d => (string)d.Value.Tag == groupName).Select(x => x.Key);
        }

        private IEnumerable<string> UniformInGroup(string groupName)
        {
            return uniformToGroup.Where(d => (string)d.Value.Tag == groupName).Select(x => x.Key);
        }

        private IEnumerable<string> SamplerInGroup(string groupName)
        {
            return samplerToGroup.Where(d => (string)d.Value.Tag == groupName).Select(x => x.Key);
        }

        private IEnumerable<string> AttribInGroup(string groupName)
        {
            return attribToGroup.Where(d => (string)d.Value.Tag == groupName).Select(x => x.Key);
        }

        #endregion

        #region コピペ用

        private static CopyPasteNode CreateGroupCopyNodes(string group_id, Material material)
        {
            if (material.MaterialShaderAssign.ShadingModel == null)
            {
                return null;
            }

            var leafs = GetCopyPasteNodes(material, material.MaterialShaderAssign.ShadingModel).ToArray();
            var groupNodes = (from @group in material.MaterialShaderAssign.ShadingModel.Groups()
                              select new GroupNode { @group = @group, name = @group.name }).ToArray();

            var nameToGroupNodes = groupNodes.ToDictionary(x => x.name);

            var orderedNodes = from node in groupNodes.Concat(leafs)
                               orderby node.order // あとは groupNodes 内、leafs 内の順番で決まるので index 等は見る必要はない。
                               // 定義されていないグループは見ないので名前は見る必要ない
                               select node;

            foreach (var node in orderedNodes)
            {
                GroupNode groupNode;
                if (!nameToGroupNodes.TryGetValue(node.parentGroupName, out groupNode))
                {
                    groupNode = new GroupNode() { name = node.parentGroupName };
                    nameToGroupNodes[groupNode.name] = groupNode;
                }

                groupNode.children.Add(node);
            }

            GroupNode copyPasteNode;
            nameToGroupNodes.TryGetValue(group_id, out copyPasteNode);

            return copyPasteNode;
        }

        private static IEnumerable<CopyPasteNode> GetCopyPasteNodes(Material material, shading_modelType shadingModel)
        {
            var attribNodes = from value in material.MaterialShaderAssign.AttribAssigns
                              let var = shadingModel.Attributes().FirstOrDefault(x => x.id == value.id && x.Editable())
                              where var != null
                              select new AttribNode() { var = var, value = value };

            var optionNodes = from value in material.MaterialShaderAssign.ShaderOptions
                              let var = shadingModel.Options().FirstOrDefault(x => x.id == value.id && x.OptionVisible() && x.type != option_var_typeType.dynamic)
                              where var != null
                              select new OptionNode() { var = var, value = value };

            var samplerNodes = from value in material.MaterialShaderAssign.SamplerAssigns
                               let var = shadingModel.Samplers().FirstOrDefault(x => x.id == value.id && x.SamplerVisible())
                               where var != null
                               select new SamplerNode() { var = var, value = value };

            var uniformNodes = from value in material.MaterialShaderAssign.ShaderParams
                               let var = shadingModel.MaterialUniforms().FirstOrDefault(x => x.id == value.id && x.IsParamVisible())
                               where var != null
                               select new UniformNode() { var = var, value = value };

            var renderInfoNodes = from value in material.MaterialShaderAssign.RenderInfos
                                  let var = shadingModel.RenderInfoSlots().FirstOrDefault(x => x.name == value.name && x.RenderInfoVisible())
                                  select new RenderInfoNode() { var = var, value = value };


            return attribNodes
                .Concat<CopyPasteNode>(optionNodes)
                .Concat(samplerNodes)
                .Concat(uniformNodes)
                .Concat(renderInfoNodes);
        }

        [Serializable]
        public abstract class CopyPasteNode
        {
            abstract public int order { get; }
            abstract public string parentGroupName { get; }
            abstract public bool CanPaste(CopyPasteNode node);
        }

        [Serializable]
        public class GroupNode : CopyPasteNode
        {
            public groupType group;
            public string name;
            public List<CopyPasteNode> children = new List<CopyPasteNode>();
            public override int order
            {
                get { return group != null ? group.Order() : 0; }
            }

            public override string parentGroupName
            {
                get { return group != null ? group.Group() : string.Empty; }
            }

            public static int num = 0;
            public override bool CanPaste(CopyPasteNode node)
            {
                var target = node as GroupNode;
                if (target == null)
                {
                    return false;
                }

                if (num > 50)
                {

                }
                num++;
                return children.Count == target.children.Count &&
                    children.Zip(target.children, (x, y) => x.CanPaste(y)).All(x => x);
            }
        }

        [Serializable]
        public class AttribNode : CopyPasteNode
        {
            public attrib_varType var;
            public attrib_assignType value;
            public override int order
            {
                get { return var.Order(); }
            }

            public override string parentGroupName
            {
                get { return var.Group(); }
            }

            public override bool CanPaste(CopyPasteNode node)
            {
                return node is AttribNode;
            }
        }

        [Serializable]
        public class OptionNode : CopyPasteNode
        {
            public option_varType var;
            public shader_optionType value;

            public override int order
            {
                get { return var.Order(); }
            }

            public override string parentGroupName
            {
                get { return var.Group(); }
            }

            public override bool CanPaste(CopyPasteNode node)
            {
                var target = node as OptionNode;
                return target != null && var.choice == target.var.choice;
            }
        }

        [Serializable]
        public class SamplerNode : CopyPasteNode
        {
            public sampler_varType var;
            public sampler_assignType value;

            public override int order
            {
                get { return var.Order(); }
            }

            public override string parentGroupName
            {
                get { return var.Group(); }
            }

            public override bool CanPaste(CopyPasteNode node)
            {
                var target = node as SamplerNode;
                return target != null && var.type == target.var.type;
            }
        }

        [Serializable]
        public class UniformNode : CopyPasteNode
        {
            public uniform_varType var;
            public shader_paramType value;

            public override int order
            {
                get { return var.Order(); }
            }

            public override string parentGroupName
            {
                get { return var.Group(); }
            }

            public override bool CanPaste(CopyPasteNode node)
            {
                var target = node as UniformNode;
                return target != null &&
                    var.type == target.var.type &&
                    (var.Min() ?? Enumerable.Empty<float>()).SequenceEqual(target.var.Min() ?? Enumerable.Empty<float>()) &&
                    (var.Max() ?? Enumerable.Empty<float>()).SequenceEqual(target.var.Max() ?? Enumerable.Empty<float>());
            }
        }

        [Serializable]
        public class RenderInfoNode : CopyPasteNode
        {
            public render_info_slotType var;
            public RenderInfo value;

            public override int order
            {
                get { return var.Order(); }
            }

            public override string parentGroupName
            {
                get { return var.Group(); }
            }

            public override bool CanPaste(CopyPasteNode node)
            {
                var target = node as RenderInfoNode;
                return target != null &&
                    var.type == target.var.type &&
                    var.optional == target.var.optional &&
                    var.count == target.var.count &&
                    var.choice == target.var.choice;
            }
        }
        #endregion

        private bool CheckResources()
        {
            IntPtr processHandle = App.Win32.NativeMethods.GetCurrentProcess();
            uint gdiObjects = App.Win32.NativeMethods.GetGuiResources(processHandle, (uint)App.Win32.NativeMethods.ResourceType.Gdi);
            if (gdiObjects > 7500)
            {
                return false;
            }
            uint userObjects = App.Win32.NativeMethods.GetGuiResources(processHandle, (uint)App.Win32.NativeMethods.ResourceType.User);
            if (userObjects > 7500)
            {
                return false;
            }

            return true;
        }

        private void cmiShowMaterialList_Click(object sender, EventArgs e)
        {
            UpdateMaterialList(true);
        }

        private void cmiHideMaterialList_Click(object sender, EventArgs e)
        {
            UpdateMaterialList(false);
        }

        private void UpdateMaterialList(bool show)
        {
            var group = (string)lastGroupBox.Tag;
            var idLabelType = new HashSet<Tuple<string, string, string>>();
            var shadingModel = GetActiveTarget().MaterialShaderAssign.ShadingModel;
            var groups = CreateIsGroupInPage(group, shadingModel.Pages().Any(), shadingModel, null);
            foreach (var item in shadingModel.Attributes().Where(x => groups(x.Group())))
            {
                idLabelType.Add(new Tuple<string, string, string>(item.id, item.Label() ?? item.id, ColumnTypes.attrib_assign.ToString()));
            }

            foreach (var item in shadingModel.Options().Where(x => groups(x.Group())))
            {
                idLabelType.Add(new Tuple<string, string, string>(item.id, item.Label() ?? item.id, ColumnTypes.shader_option.ToString()));
            }

            foreach (var item in shadingModel.Samplers().Where(x => groups(x.Group())))
            {
                idLabelType.Add(new Tuple<string, string, string>(item.id, item.Label() ?? item.id, ColumnTypes.sampler_assign.ToString()));
            }

            foreach (var item in shadingModel.MaterialUniforms().Where(x => groups(x.Group())))
            {
                idLabelType.Add(new Tuple<string, string, string>(item.id, item.Label() ?? item.id, ColumnTypes.shader_param.ToString()));
            }

            foreach (var item in shadingModel.RenderInfoSlots().Where(x => groups(x.Group())))
            {
                idLabelType.Add(new Tuple<string, string, string>(item.name, item.Label() ?? item.name, ColumnTypes.render_info.ToString()));
            }

            var config = ConfigData.ApplicationConfig.Setting.ObjectView.GetListView(ObjectView.List.ViewID.Material);
            foreach (var column in config.Columns.Where(x => idLabelType.Contains(new Tuple<string, string, string>(x.Id, x.Name, x.Type))))
            {
                column.Show = show;
            }

            App.ObjectView.List.ObjectListView.Initialize(ObjectView.List.ViewID.Material);
            App.ObjectView.List.ObjectListView.GetColumnInfo(ObjectView.List.ViewID.Material).NotifySettingChanged();
        }

        private void cmiCustomLabel_Click(object sender, EventArgs e)
        {
            if (EditGroupLabel != null)
            {
                EditGroupLabel(sender, e);
            }
        }

        private void CmiMaterialReferenceBehaviorValueClick(object sender, EventArgs e)
        {
            ShaderItemValueState? state = null;
            if (sender == cmiMaterialReferenceBehaviorValue_Default)
            {
                state = ShaderItemValueState.Default;
            }
            else if (sender == cmiMaterialReferenceBehaviorValue_Override)
            {
                state = ShaderItemValueState.Override;
            }
            else if (sender == cmiMaterialReferenceBehaviorValue_Refer)
            {
                state = ShaderItemValueState.Refer;
            }
            Material.SetReferenceBehaviorToGroup(GetTargets(), (string)lastGroupBox.Tag, null, state);
        }

        private void CmiMaterialReferenceBehaviorRestrictionClick(object sender, EventArgs e)
        {
            ShaderItemRestrictionState? state = null;
            if (sender == cmiMaterialReferenceBehaviorRestriction_None)
            {
                state = ShaderItemRestrictionState.None;
            }
            else if (sender == cmiMaterialReferenceBehaviorRestriction_ForceRefer)
            {
                state = ShaderItemRestrictionState.ForceRefer;
            }
            else if (sender == cmiMaterialReferenceBehaviorRestriction_ForceOverride)
            {
                state = ShaderItemRestrictionState.ForceOverride;
            }
            Material.SetReferenceBehaviorToGroup(GetTargets(), (string)lastGroupBox.Tag, state, null);
        }

    }
}
