﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;
using App.Controls;
using App.Data;
using App.Utility;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;

namespace App.PropertyEdit.ShaderParamControls
{
    public partial class EditRenderInfoCheckBox : EditRenderInfoBase
    {
        public bool[] Value
        {
            get
            {
                return boolEditPanels.Take(ItemCount).Select(x => x.Checked).ToArray();
            }
            set
            {
                ItemCount = 0;
                for (int i = 0; i < Math.Min(value.Length, Row * Column); i++)
                {
                    boolEditPanels[i].Checked = value[i];
                    ItemCount++;
                }
            }
        }

        int ItemCount = 0;
        readonly Controls.UIModifiedMarkAndImageCheckBox[] boolEditPanels;

        public int Row { get; private set; }
        public int Column { get; private set; }
        public const int MarginX = 4;
        public const int MarginY = 20;
        public int subControlHeight = 24;
        public int marginBetweenColumn = 6;
        public render_info_slotType type;
        public bool option = false;
        public EditRenderInfoCheckBox(int row, int column, render_info_slotType type)
        {
            //checkOff = min;
            //checkOn = max;
            //eps = Math.Min(Math.Abs(checkOff - checkOn) / 2, eps);
            InitializeComponent();
            Row = row;
            Column = column;
            this.type = type;
            boolEditPanels = new UIModifiedMarkAndImageCheckBox[Row * Column];
            int tabIndex = 0;
            for (int i = 0; i < Row; i++)
            {
                for (int j = 0; j < Column; j++)
                {
                    // Row mager
                    int k = Column * i + j;
                    var control = new UIModifiedMarkAndImageCheckBox();
                    control.Tag = 1u << k;
                    control.CheckedChanged += cbxValue_CheckedChanged;
                    control.TabIndex = tabIndex;
                    tabIndex++;
                    Controls.Add(control);
                    boolEditPanels[k] = control;
                }
            }
            if (type != null && type.optional)
            {
                btnIncrement.TabIndex = tabIndex;
                tabIndex++;
                btnDecrement.TabIndex = tabIndex;
            }
            else
            {
                btnIncrement.Visible = false;
                btnDecrement.Visible = false;
            }
        }

        public override void Align()
        {
            if (boolEditPanels == null)
            {
                return;
            }

            if (boolEditPanels.Length == 1 && CanEnabled)
            {
                var control = boolEditPanels[0];
                control.AutoSize = true;
                lblParameter.Visible = false;
                if (lblParameter.ContextMenuStrip != null) // ラベルが消されるので、コンテクストメニューを付け替える
                {
                    control.ContextMenuStrip = lblParameter.ContextMenuStrip;
                }
                control.Location = lblParameter.Location;
                UpdateLabel();
            }
            else
            {
                int width = DefaultWidth - MarginX + marginBetweenColumn;
                for (int i = 0; i < Row; i++)
                {
                    for (int j = 0; j < Column; j++)
                    {
                        Control control = boolEditPanels[Column * i + j];
                        control.Location = new Point(MarginX + (width * j) / 4, MarginY + subControlHeight * i);
                    }
                }
            }

            if (type != null && type.optional)
            {
                btnIncrement.Location = new Point(boolEditPanels[Row - 1].Right + 4, boolEditPanels.Length == 1 ? boolEditPanels[0].Top : MarginY);
                btnDecrement.Location = new Point(btnIncrement.Right + 4, btnIncrement.Top);
            }
            else
            {
                // Align の都合上左上に寄せておく
                btnIncrement.Location = new Point(0, 0);
                btnDecrement.Location = new Point(0, 0);
            }
        }

        public override void UpdateLabel()
        {
            if (boolEditPanels != null && !lblParameter.Visible && boolEditPanels.Length == 1 && CanEnabled)
            {
                var control = boolEditPanels[0];
                var label = labelHelper.GetLabelString(false);
                control.Text = label;
                control.ForeColor = labelHelper.color ?? SystemColors.ControlText;
                control.TailImages = lblParameter.TailImages;
                UpdateDropDownButtonLocation();
                UpdateLinkToolTip(false);
            }
            else
            {
                base.UpdateLabel();
            }
        }

        // テキストチェックボックスの値に割り当てられる文字列
        public string ValueText { get; set; }

        // 持っているチェックボックスに変更マークを設定
        public void SetModified(bool isModified)
        {
            foreach (var control in boolEditPanels)
            {
                control.IsModified = isModified;
            }
        }

        int intOn = 1;
        int intOff = 0;
        float floatOn = 1;
        float floatOff = 0;
        public override bool SetValues(Material material, List<string> values, bool candidateModified, CustomUI customUI, Definition.ShadingModelTable table, Predicate<string> visibleGroups, HashSet<string> visiblePages, bool showId, bool showOriginalLabel, Material.ValueResolvedState valueResolvedState)
        {
            ValueText = null;
            bool hasCandidate = false;
            bool validDefault = false;
            bool packFromHio = false;
            switch (type.type)
            {
                case render_info_slot_typeType.@int:
                    {
                        int[] defaults = new int[0];
                        if (Viewer.Manager.Instance.IsConnected)
                        {
                            var candidate = material.RenderInfoPackFromHio.IntItems.FirstOrDefault(x => x.name == ParamName);
                            if (candidate != null)
                            {
                                if (candidate.hasMinMax)
                                {
                                    intOn = candidate.max;
                                    intOff = candidate.min;
                                    hasCandidate = true;
                                }
                                defaults = candidate.defaults.ToArray();
                                packFromHio = true;
                            }
                        }

                        if (type != null && !packFromHio)
                        {
                            if (IfShaderAssignUtility.TryParseIntRange(type.choice, out intOff, out intOn))
                            {
                                hasCandidate = true;
                            }
                            defaults = G3dDataParser.Tokenize(type.Default()).Select(x => { int result; int.TryParse(x, out result); return result; }).ToArray();
                        }

                        if (boolEditPanels.Length == 1)
                        {
                            boolEditPanels[0].BackColor = GetBgColor(candidateModified, packFromHio, hasCandidate);
                        }
                        else
                        {
                            lblParameter.BackColor = GetBgColor(candidateModified, packFromHio, hasCandidate);
                        }


                        if (hasCandidate)
                        {
                            hasCandidate = true;
                            Value = values.Select(x => { int result; int.TryParse(x, out result); return result; })
                                .Select(x => intOn == x).ToArray();
                            if (type != null)
                            {
                                for (int i = ItemCount; i < Math.Min(boolEditPanels.Length, defaults.Length); i++)
                                {
                                    boolEditPanels[i].Checked = intOn == defaults[i];
                                }
                            }

                            validDefault = defaults.All(x => intOff <= x && x <= intOn);
                        }
                    }
                    break;
                case render_info_slot_typeType.@float:
                    {
                        float[] defaults = new float[0];
                        if (Viewer.Manager.Instance.IsConnected)
                        {
                            var candidate = material.RenderInfoPackFromHio.FloatItems.FirstOrDefault(x => x.name == ParamName);
                            if (candidate != null)
                            {
                                if (candidate.hasMinMax)
                                {
                                    floatOn = candidate.max;
                                    floatOff = candidate.min;
                                    hasCandidate = true;
                                }
                                defaults = candidate.defaults.ToArray();
                                packFromHio = true;
                            }
                        }

                        if (type != null && !packFromHio)
                        {
                            if (IfShaderAssignUtility.TryParseFloatRange(type.choice, out floatOff, out floatOn))
                            {
                                hasCandidate = true;
                            }

                            defaults = G3dDataParser.Tokenize(type.Default()).Select(x => { float result; float.TryParse(x, out result); return result; }).ToArray();
                        }

                        if (boolEditPanels.Length == 1)
                        {
                            boolEditPanels[0].BackColor = GetBgColor(candidateModified, packFromHio, hasCandidate);
                        }
                        else
                        {
                            lblParameter.BackColor = GetBgColor(candidateModified, packFromHio, hasCandidate);
                        }

                        if (hasCandidate)
                        {
                            float eps = 1.0e-6f;
                            eps = Math.Min(Math.Abs(floatOn - floatOff) / 2, eps);
                            Value = values.Select(x => { float result; float.TryParse(x, out result); return result; })
                                .Select(x => Math.Abs(x - floatOff) >= eps).ToArray();
                            if (type != null)
                            {
                                for (int i = ItemCount; i < Math.Min(boolEditPanels.Length, defaults.Length); i++)
                                {
                                    boolEditPanels[i].Checked = Math.Abs(defaults[i] - floatOff) >= eps;
                                }
                            }

                            validDefault = defaults.All(x => intOff <= x && x <= intOn);
                        }
                    }
                    break;
                case render_info_slot_typeType.@string:
                    {
                        // 文字列からチェックボックスを作成した場合、候補の0番を無効、それ以降を有効として扱う
                        if (type.choice != null && values.Count > 0)
                        {
                            // 文字列なので値に範囲制限は無し
                            ItemCount = 1;
                            hasCandidate = true;
                            validDefault = true;

                            string[] choice = type.choice.Split(',');
                            if (choice != null)
                            {
                                int result;
                                if (int.TryParse(values[0], out result))
                                {
                                    // UIから変更した場合は数値で0・1で通知される
                                    boolEditPanels[0].Checked = result != 0;
                                    ValueText = choice[result];
                                }
                                else
                                {
                                    // テキストで格納されている場合
                                    boolEditPanels[0].Checked = values[0] != choice[0];
                                    ValueText = values[0];
                                }
                            }
                        }
                    }
                    break;
                default:
                    Debug.Assert(false);
                    throw new NotImplementedException();
            }

            for (int i = 0; i < boolEditPanels.Length; i++)
            {
                var control = boolEditPanels[i];
                control.Enabled = i < ItemCount && hasCandidate && validDefault;
            }

            btnIncrement.Enabled = ItemCount < boolEditPanels.Length && hasCandidate && type != null && validDefault;
            btnDecrement.Enabled = 0 < ItemCount && hasCandidate && type != null && validDefault;

            return false;
        }

        private void cbxValue_CheckedChanged(object sender, EventArgs e)
        {
            var args = new RenderInfoValueChangedEventArgs()
            {
                ParamName = ParamName,
                type = type.Type(),
            };

            switch (type.type)
            {
                case render_info_slot_typeType.@int:
                    args.ParamValue = Value.Select(x => x ? intOn.ToString() : intOff.ToString()).ToList();
                    break;
                case render_info_slot_typeType.@float:
                    args.ParamValue = Value.Select(x => x ? floatOn.ToString() : floatOff.ToString()).ToList();
                    break;
                case render_info_slot_typeType.@string:
                    {
                        if (type.ui_item != null && type.ui_item.value == ui_item_valueType.check)
                        {
                            // 文字列からチェックボックスを作成した場合もUIには数値で設定する
                            List<string> paramValue = new List<string>();
                            for (int i = 0; i < Value.Length; ++i)
                            {
                                paramValue.Add(Value[i] ? "1" : "0");
                            }
                            args.ParamValue = paramValue;
                        }
                    }
                    break;
                default:
                    Debug.Assert(false);
                    throw new NotImplementedException();
            }

            InvokeValueChanged(this, args);
        }

        public override void SetHint(HintToolTip hint)
        {
            if (boolEditPanels != null && boolEditPanels.Length == 1)
            {
                parentHint = hint;

                if (dropDownButton1 != null)
                {
                    hint.SetToolTip(dropDownButton1, res.Strings.ShaderParamControl_ShowCurveEditor);
                }

                if (!string.IsNullOrWhiteSpace(ParamComment))
                {
                    hint.SetToolTip(boolEditPanels[0], ParamComment);
                }
            }
            else
            {
                base.SetHint(hint);
            }
        }

        private void btnIncrement_Click(object sender, EventArgs e)
        {
            ItemCount++;
            Debug.Assert(ItemCount <= boolEditPanels.Length);
            var args = new RenderInfoValueChangedEventArgs()
            {
                ParamName = ParamName,
                SequentialValueChangedEventArgs = null,
                type = render_info_typeType.@float,
            };
            switch (type.type)
            {
                case render_info_slot_typeType.@int:
                    args.ParamValue = Value.Select(x => x ? intOn.ToString() : intOff.ToString()).ToList();
                    break;
                case render_info_slot_typeType.@float:
                    args.ParamValue = Value.Select(x => x ? floatOn.ToString() : floatOff.ToString()).ToList();
                    break;
                default:
                    Debug.Assert(false);
                    throw new NotImplementedException();
            }
            InvokeValueChanged(this, args);
        }

        private void btnDecrement_Click(object sender, EventArgs e)
        {
            ItemCount--;
            Debug.Assert(ItemCount >= 0);
            var args = new RenderInfoValueChangedEventArgs()
            {
                ParamName = ParamName,
                SequentialValueChangedEventArgs = null,
                type = render_info_typeType.@float,
            };
            switch (type.type)
            {
                case render_info_slot_typeType.@int:
                    args.ParamValue = Value.Select(x => x ? intOn.ToString() : intOff.ToString()).ToList();
                    break;
                case render_info_slot_typeType.@float:
                    args.ParamValue = Value.Select(x => x ? floatOn.ToString() : floatOff.ToString()).ToList();
                    break;
                default:
                    Debug.Assert(false);
                    throw new NotImplementedException();
            }
            InvokeValueChanged(this, args);
        }

        public override void UpdateDropDownButtonLocation()
        {
            if (boolEditPanels == null || boolEditPanels.Length != 1)
            {
                base.UpdateDropDownButtonLocation();
                return;
            }

            if (type != null && type.optional)
            {
                btnIncrement.Location = new Point(boolEditPanels[Row - 1].Right + 4, boolEditPanels.Length == 1 ? boolEditPanels[0].Top : MarginY);
                btnDecrement.Location = new Point(btnIncrement.Right + 4, btnIncrement.Top);
            }

            if (labelHelper.linkTarget != null)
            {
                if (btnLink == null)
                {
                    btnLink = LabelHelper.CreateLinkButton();
                    btnLink.Click += (s, e) =>
                    {
                        OnLinkClicked(labelHelper.ClickArgs());
                    };
                    btnLink.Top = boolEditPanels[0].Top;
                    Controls.Add(btnLink);
                }

                if (type != null && type.optional)
                {
                    btnLink.Left = btnDecrement.Right + 5;
                }
                else
                {
                    btnLink.Left = boolEditPanels[0].Right + 5;
                }

                btnLink.Enabled = !labelHelper.LinkError;
            }
        }

        protected override void OnMouseUp(MouseEventArgs e)
        {
            if (e.Button == MouseButtons.Right)
            {
                var child = GetChildAtPoint(e.Location);
                if (child?.ContextMenuStrip != null && !child.Enabled)
                    child.ContextMenuStrip.Show(this, e.Location);
            }
            base.OnMouseUp(e);
        }
    }
}
