﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;
using App.Controls;
using App.Data;
using App.res;
using nw.g3d.nw4f_3dif;

namespace App.PropertyEdit.ShaderParamControls
{
    public partial class EditFloatMatrix : ShaderParamControl
    {
        // カラーの表示切替
        public event ColorControlChangedEventHandler ColorControlChanged = null;

        protected override void labelContextMenuStrip_Opening(object sender, System.ComponentModel.CancelEventArgs e)
        {
            // コンテキストメニューの構築
            base.labelContextMenuStrip_Opening(sender, e);
            if (ColorControlChanged != null)
            {
                var cmiColorControl = new System.Windows.Forms.ToolStripMenuItem();
                cmiColorControl.Size = new System.Drawing.Size(152, 22);
                cmiColorControl.Text = Strings.ShaderParamControl_ShowAsColor;
                cmiColorControl.Click += (s, args) =>
                {
                    // フォーカスを外す
                    ActiveControl = null;

                    ColorControlChanged(s, new ColorControlChangedEventArgs()
                    {
                        ParamName = ParamName,
                        ShowColor = true,
                    });
                };
                labelContextMenuStrip.Items.Add(cmiColorControl);
            }
        }

        public float[] Value
        {
            get
            {
                return floatEditPanels.Select(x => x.Value).ToArray();
            }
            set
            {
                for (int i= 0; i< Row*Column; i++)
                {
                    floatEditPanels[i].Value = value[i];
                }
            }
        }

        public void SetMinimumValue(float minimum, float inputMin)
        {
            for (int i = 0; i < Row * Column; i++)
            {
                floatEditPanels[i].MinBound = inputMin;
                floatEditPanels[i].Minimum = floatEditPanels[i].MinDefault = minimum;
            }
        }

        public void SetMaximumValue(float maximum, float inputMax)
        {
            for (int i = 0; i < Row * Column; i++)
            {
                floatEditPanels[i].MaxBound = inputMax;
                floatEditPanels[i].Maximum = floatEditPanels[i].MaxDefault = maximum;
            }
        }

        public void UpdateIncrement()
        {
            double Increment = Math.Min(1, Math.Pow(10, Math.Round(Math.Log10(Math.Max(0.001, (double)floatEditPanels[0].Maximum - (double)floatEditPanels[0].Minimum))) - 2));
            double LargeIncrement = Increment * 10;
            for (int i = 0; i < Row * Column; i++)
            {
                floatEditPanels[i].Increment = Increment;
                floatEditPanels[i].LargeIncrement = LargeIncrement;
            }
        }

        readonly Controls.FloatEditPanel[] floatEditPanels;
        readonly UICheckBox linkValueCheck;
        int row = 0;
        int column = 0;

        public int Row
        {
            get
            {
                return row;
            }
            private set
            {
                row = value;
                NumComponentCurveEditorOpeningButtons = row * Column;
            }
        }
        public int Column
        {
            get
            {
                return column;
            }
            private set
            {
                column = value;
                NumComponentCurveEditorOpeningButtons = Row * column;
            }
        }
        public const int MarginX = 4;
        public const int MarginY = 20;
        public int subControlHeight = 24;
        public int marginBetweenColumn = 6;
        public shader_param_typeType type;

        public static HashSet<string> CheckedNames = new HashSet<string>();
        public override string ParamName
        {
            get
            {
                return base.ParamName;
            }
            set
            {
                base.ParamName = value;
                if (linkValueCheck != null)
                {
                    linkValueCheck.Checked = CheckedNames.Contains(value);
                }
            }
        }

        private void chkLinkValueCheck_CheckedChanged(object sender, EventArgs e)
        {
            if (linkValueCheck.Checked)
            {
                CheckedNames.Add(ParamName);
            }
            else
            {
                CheckedNames.Remove(ParamName);
            }
        }

        public override ParamType ParamType
        {
            get { return ParamType.uniform_var; }
        }

        private bool fixedSlider = false;
        public EditFloatMatrix(int row, int column, shader_param_typeType type)
        {
            InitializeComponent();
            fixedSlider = ConfigData.ApplicationConfig.Setting.PropertyEdit.FixedSlider && (Row == 1 || Column == 1);

            if (fixedSlider)
            {
                if (Column > 1)
                {
                    Row = Column;
                }
            }

            Row = row;
            Column = column;
            this.type = type;
            floatEditPanels = new FloatEditPanel[Row * Column];
            for (int i = 0; i < Row; i++)
            {
                for (int j = 0; j < Column; j++)
                {
                    // Row mager
                    int k = Column * i + j;
                    var control = new FloatEditPanel();
                    control.Tag = 1u << k;
                    control.SliderRangeAdjustable = true;
                    control.Minimum = -100;
                    control.Maximum = 100;
                    control.Value = i * 10 + j;
                    control.SequentialValueChanged += edpValue_SequentialValueChanged;
                    Controls.Add(control);
                    floatEditPanels[k] = control;
                }
                if (Row == 1 && Column != 1) // add link checkbox
                {
                    var chk = new UICheckBox();
                    chk.Text = Strings.ShaderParamControl_LinkScale;
                    chk.CheckedChanged += chkLinkValueCheck_CheckedChanged;
                    Controls.Add(chk);
                    linkValueCheck = chk;
                }
            }
        }

        public override void Align()
        {
            if (floatEditPanels == null)
            {
                return;
            }

            var numFloatEditPanels = floatEditPanels.Length;
            bool showButtons = (numFloatEditPanels > 1) && (numFloatEditPanels == ComponentCurveEditorOpeningButtons.Count);
            var extraWidth = showButtons ? ((ComponentCurveEditorOpeningButtonDefaultSize.Width + 2) * 4) : 0;

            {
                int width = DefaultWidth + extraWidth - MarginX + marginBetweenColumn;
                for (int i = 0; i < Row; i++)
                {
                    for (int j = 0; j < Column; j++)
                    {
                        Control control = floatEditPanels[Column * i + j];
                        control.Size = new Size((width - extraWidth) / 4 - marginBetweenColumn, subControlHeight);
                        control.Location = new Point(MarginX + (width * j) / 4, MarginY + subControlHeight * i);
                        //DebugConsole.WriteLineIf(i == 0 && j==0, "Align" + control.Size.ToString());
                        //DebugConsole.Write(control.Location.ToString());
                    }
                    //DebugConsole.WriteLine("");
                    if (linkValueCheck != null && Row == 1 && Column != 1) // link value checkbox
                    {
                        linkValueCheck.Location = new Point(MarginX + (width * Column) / 4, MarginY);
                    }
                }
            }

            if (showButtons)
            {
                for (int i = 0; i < numFloatEditPanels; i++)
                {
                    var floatEditPanel = floatEditPanels[i];
                    ComponentCurveEditorOpeningButtons[i].Location = new Point(floatEditPanel.Right + 2, floatEditPanel.Top + Math.Max(0, (floatEditPanel.Height - ComponentCurveEditorOpeningButtonDefaultSize.Height) / 2));
                }

                Controls.AddRange(ComponentCurveEditorOpeningButtons.Where(x => !Controls.Contains(x)).ToArray());
            }
            else
            {
                foreach (var button in ComponentCurveEditorOpeningButtons)
                {
                    Controls.Remove(button);
                }
            }
        }

        public override bool SetValue(Material material, string value, CustomUI customUI, Definition.ShadingModelTable table, Predicate<string> visibleGroups, HashSet<string> visiblePages, bool showId, bool showOriginalLabel)
        {
            float[] paramArray = G3dDataParser.ParseFloatArray(value);
            if (paramArray == null || paramArray.Length != Row * Column)
            {
                Enabled = false;
                return false;
            }

            Value = paramArray;
            return false;
        }

        private void edpValue_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            var args = new ShaderParamValueChangedEventArgs(type);
            uint elemBits = 0;

            // 連動がオンのときはUIを同じ値で更新しまとめて一つのイベントにする
            if (linkValueCheck != null && linkValueCheck.Checked)
            {
                var modifiedPanel = sender as FloatEditPanel;
                Debug.Assert(modifiedPanel != null);

                using (var block = new UIControlEventSuppressBlock())
                {
                    foreach (var panel in floatEditPanels)
                    {
                        if (modifiedPanel != panel)
                        {
                            panel.Value = modifiedPanel.Value;
                        }
                        elemBits |= (uint) panel.Tag;
                    }
                }
            }
            else
            {
                elemBits = (uint)(((Control)sender).Tag);
            }

            args.ParamName = ParamName;
            args.ParamValue = Value;
            args.ElementBits = elemBits;
            args.SequentialValueChangedEventArgs = e;

            InvokeValueChanged(this, args);
        }
    }
}
