﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;
using App.Command;
using App.Controls;
using App.Data;
using App.PropertyEdit.ShaderParamControls;
using App.Utility;
using ConfigCommon;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;
using System.Text.RegularExpressions;

namespace App.PropertyEdit
{
    public partial class MaterialShaderPage : MaterialPropertyPage, IShaderPage
    {
        private class Finder
        {
            private class BackAndForeColors
            {
                public BackAndForeColors(Color backColor, Color foreColor)
                {
                    BackColor = backColor;
                    ForeColor = foreColor;
                }

                public readonly Color BackColor;
                public readonly Color ForeColor;
            }

            private class Item
            {
                public Item(Control control, Point location, string text, Action<BackAndForeColors> update, BackAndForeColors defaultColor, BackAndForeColors highlightColor)
                {
                    Control = control;
                    Location = location;
                    Text = text;
                    Update = update;
                    DefaultColor = defaultColor;
                    HighlightColor = highlightColor;
                }
                public readonly Control Control;
                public readonly Point Location;
                public readonly string Text;
                public readonly Action<BackAndForeColors> Update;
                public readonly BackAndForeColors DefaultColor;
                public readonly BackAndForeColors HighlightColor;
            }

            private readonly UIScrollPanel pnlScroll;
            private readonly ShaderParamControlGroup scgGroup;
            private string text = string.Empty;
            private readonly List<Item> items = new List<Item>();
            private int lastScrollItemIndex = -1;

            public Finder(MaterialShaderPage page)
            {
                pnlScroll = page.pnlScroll;
                scgGroup = page.scgGroup;
            }

            public void ShowHighlights()
            {
                foreach (var item in items)
                {
                    item.Update(item.HighlightColor);
                }
            }

            public void ClearHighlights()
            {
                foreach (var item in items)
                {
                    item.Update(item.DefaultColor);
                }
            }

            private void Search(string str)
            {
                Search(str, scgGroup.Controls, new Point(0, 0));
            }

            private void Search(string str, ControlCollection ctrls, Point offset)
            {
                foreach (var ctrl in ctrls.OfType<Control>())
                {
                    if (!ctrl.Visible)
                    {
                        // 非表示のものは、検索対象外。(子コントロールも)
                        continue;
                    }

                    var location = new Point(ctrl.Location.X + offset.X, ctrl.Location.Y + offset.Y);

                    if ((ctrl as ShaderGroupBox) != null)
                    {
                        var shaderGroupBox = (ShaderGroupBox)ctrl;
                        if ((shaderGroupBox.Text != null) && (shaderGroupBox.Text.IndexOf(str, StringComparison.CurrentCultureIgnoreCase) >= 0))
                        {
                            items.Add(
                                new Item(
                                    shaderGroupBox,
                                    location,
                                    shaderGroupBox.Text,
                                    new Action<BackAndForeColors>(x =>
                                    {
                                        shaderGroupBox.TitleBarColor = x.BackColor;
                                        shaderGroupBox.TitleTextColor = x.ForeColor;
                                    }),
                                    new BackAndForeColors(shaderGroupBox.TitleBarColor, shaderGroupBox.TitleTextColor),
                                    new BackAndForeColors(SystemColors.Highlight, SystemColors.HighlightText)));
                        }
                    }
                    else if ((ctrl as Label) != null)
                    {
                        var label = (Label)ctrl;
                        if ((label.Text != null) && (label.Text.IndexOf(str, StringComparison.CurrentCultureIgnoreCase) >= 0))
                        {
                            items.Add(
                                new Item(
                                    label,
                                    location,
                                    label.Text,
                                    new Action<BackAndForeColors>(x =>
                                    {
                                        label.BackColor = x.BackColor;
                                        label.ForeColor = x.ForeColor;
                                    }),
                                    new BackAndForeColors(label.BackColor, label.ForeColor),
                                    new BackAndForeColors(SystemColors.Highlight, SystemColors.HighlightText)));
                        }
                    }
                    else if ((ctrl as CheckBox) != null)
                    {
                        var checkBox = (CheckBox)ctrl;
                        if ((checkBox.Text != null) && (checkBox.Text.IndexOf(str, StringComparison.CurrentCultureIgnoreCase) >= 0))
                        {
                            items.Add(
                                new Item(
                                    checkBox,
                                    location,
                                    checkBox.Text,
                                    new Action<BackAndForeColors>(x =>
                                    {
                                        checkBox.BackColor = x.BackColor;
                                        checkBox.ForeColor = x.ForeColor;
                                    }),
                                    new BackAndForeColors(checkBox.BackColor, checkBox.ForeColor),
                                    new BackAndForeColors(SystemColors.Highlight, SystemColors.HighlightText)));
                        }
                    }

                    Search(str, ctrl.Controls, location);
                }
            }

            public string Text
            {
                get
                {
                    return text;
                }
                set
                {
                    ClearHighlights();
                    lastScrollItemIndex = -1;

                    if (!string.IsNullOrEmpty(value) && !string.IsNullOrEmpty(Text) && (value.Length > Text.Length) && value.StartsWith(Text))
                    {
                        // 絞り込む。
                        var list = items.ToArray();
                        foreach (var i in Enumerable.Range(0, list.Length).Reverse())
                        {
                            var item = list[i];
                            if (item.Text.IndexOf(value, StringComparison.CurrentCultureIgnoreCase) < 0)
                            {
                                items.RemoveAt(i);
                            }
                        }
                    }
                    else
                    {
                        // 再検索。
                        items.Clear();
                        if (!string.IsNullOrEmpty(value))
                        {
                            Search(value);
                        }

                        // 位置でソート。
                        items.Sort(
                            (x, y) =>
                            {
                                var i = x.Location.Y - y.Location.Y;
                                return (i != 0) ? i : (x.Location.X - y.Location.X);
                            });
                    }

                    ShowHighlights();

                    text = !string.IsNullOrEmpty(value) ? value : string.Empty;
                }
            }

            public void ScrollToNextControl()
            {
                if (items.Any())
                {
                    if ((lastScrollItemIndex >= 0) && (lastScrollItemIndex < items.Count))
                    {
                        var lastScrollItem = items[lastScrollItemIndex];
                        lastScrollItem.Update(lastScrollItem.HighlightColor);
                    }

                    var index = (lastScrollItemIndex >= 0) ? ((lastScrollItemIndex + 1) % items.Count) : 0;
                    var item = items[index];
                    pnlScroll.AutoScrollPosition = new Point(item.Location.X, item.Location.Y);
                    item.Update(new BackAndForeColors(
                        Color.FromArgb(
                            item.HighlightColor.BackColor.A,
                            (byte)(item.HighlightColor.BackColor.R * 0.75f),
                            (byte)(item.HighlightColor.BackColor.G * 0.75f),
                            (byte)(item.HighlightColor.BackColor.B * 0.75f)),
                            item.HighlightColor.ForeColor));
                    lastScrollItemIndex = index;
                }
            }

            public void ScrollToPrevControl()
            {
                if (items.Any())
                {
                    if ((lastScrollItemIndex >= 0) && (lastScrollItemIndex < items.Count))
                    {
                        var lastScrollItem = items[lastScrollItemIndex];
                        lastScrollItem.Update(lastScrollItem.HighlightColor);
                    }

                    var index = (lastScrollItemIndex > 0) ? ((lastScrollItemIndex - 1) % items.Count) : (items.Count - 1);
                    var item = items[index];
                    pnlScroll.AutoScrollPosition = new Point(item.Location.X, item.Location.Y);
                    item.Update(new BackAndForeColors(
                        Color.FromArgb(
                            item.HighlightColor.BackColor.A,
                            (byte)(item.HighlightColor.BackColor.R * 0.75f),
                            (byte)(item.HighlightColor.BackColor.G * 0.75f),
                            (byte)(item.HighlightColor.BackColor.B * 0.75f)),
                            item.HighlightColor.ForeColor));
                    lastScrollItemIndex = index;
                }
            }
        }

        public string CategoryName { get; set; }
        public bool UpdateControls { get; set; }
        public bool ShowConditionVariable { get; private set; }
        public bool ShowForceReferParameter { get; private set; }
        public bool ShowParamId { get; private set; }
        public bool ShowOriginalLabel { get; private set; }
        public bool SucceedVariable { get; private set; }
        public MaterialShaderPage() :
            base(PropertyPageID.MaterialShader)
        {
            InitializeComponent();

            scgGroup.ParentMaterialShaderPage = this;
            scgGroup.EnterGroup += StopScrollTemporarily;
            App.AppContext.MaterialReferenceBehaviorSettingChanged += OnMaterialReferenceBehaviorSettingChanged;
            Disposed += delegate
            {
                App.AppContext.MaterialReferenceBehaviorSettingChanged -= OnMaterialReferenceBehaviorSettingChanged;
            };

            UpdateControls = true;

            // 内部でスクロールを管理するので親はスクロールさせない
            HideParentScrollBar = true;

            findBar = new FindBar()
            {
                Visible = false,
                WatermarkText = App.res.Strings.ShaderParameterFindBar_WatermarkText
            };
            findBarVisibleProperty = typeof(FindBar).GetProperty("Visible", System.Reflection.BindingFlags.Public | System.Reflection.BindingFlags.Instance);
            System.Diagnostics.Debug.Assert(findBarVisibleProperty != null);

            Controls.Add(findBar);

            finder = new Finder(this);

            findBar.VisibleChanged += (ss, ee) =>
            {
                if (findBar.Visible)
                {
                    finder.Text = findBar.Text;
                }
                else
                {
                    finder.Text = string.Empty;
                }
            };
            findBar.TextChanged += (ss, ee) =>
            {
                finder.Text = findBar.Text;
            };
            findBar.EnterKeyDown += (ss, ee) =>
            {
                if (!ModifierKeys.HasFlag(Keys.Shift))
                {
                    finder.ScrollToNextControl();
                }
                else
                {
                    finder.ScrollToPrevControl();
                }
            };
            findBar.CloseButtonClick += (ss, ee) =>
            {
                findBar.Visible = false;
            };

            ClientSizeChanged += (ss, ee) =>
            {
                findBar.Location = new Point(ClientRectangle.Right - (findBar.Width + findBar.Margin.Right), ClientRectangle.Top + findBar.Margin.Top);
            };
        }

        private void OnMaterialReferenceBehaviorSettingChanged()
        {
            if (Visible && Targets?.Active != null && Targets.Active.ObjectID == GuiObjectID.Material && Owner != null && Owner.Visible && DocumentManager.Models.Contains(Targets.Active.OwnerDocument))
            {
                UpdateControls = true;
                ((MaterialPropertyPanel) Owner).UpdateCategoryView();
                UpdateForm(false, false);
            }
        }

        public override Utility.HelpUtility.PageKey HelpKey
        {
            get
            {
                return Utility.HelpUtility.PageKey.p_material_property_window_shader_page;
            }
        }

        public static ObjectPropertyPage CreateInstance(object arg)
        {
            DebugConsole.WriteLine("CreateInstance");
            return new MaterialShaderPage();
        }

        protected override void InitializeFormInternal()
        {
            ShowConditionVariable = ConfigData.ApplicationConfig.Setting.PropertyEdit.ShowConditionVariable;
            ShowForceReferParameter = ConfigData.ApplicationConfig.Setting.PropertyEdit.ShowForceReferParameter;
            ShowParamId = ConfigData.ApplicationConfig.Setting.PropertyEdit.ShowShaderParamId;
            ShowOriginalLabel = ConfigData.ApplicationConfig.Setting.PropertyEdit.ShowOriginalLabelId;
            SucceedVariable = ConfigData.ApplicationConfig.Setting.PropertyEdit.SucceedVariable;
            scgGroup.GetActiveTarget = () => ActiveTarget;
            scgGroup.GetTargets = () => Targets;
        }

        public static bool IsModified(Material material)
        {
            return material != null &&
                (material.IsShadingModelModified() ||
                material.IsShaderOptionsModified() ||
                material.IsSamplerAssignsModified() ||
                material.IsShaderParamsModified() ||
                material.IsAttribAssignModified() ||
                material.IsRenderInfosModified() ||
                material.IsVtxBufferArrayModified() || // ツリービューだけ表示される
                material.IsGroupCustomLabelModified());
        }

        public override bool ChangeDockStyle
        {
            get
            {
                return false;
            }
        }

        public override Size DefaultPageSize
        {
            get
            {
                return new Size(
                    Math.Max(base.DefaultPageSize.Width, ConfigData.ApplicationConfig.Setting.PropertyEdit.ShaderPageSize.Width),
                    Math.Max(base.DefaultPageSize.Height, ConfigData.ApplicationConfig.Setting.PropertyEdit.ShaderPageSize.Height));
            }
        }

        FindBar findBar = null;
        System.Reflection.PropertyInfo findBarVisibleProperty = null;
        Finder finder = null;

        protected override void OnLoad(EventArgs e)
        {
            base.OnLoad(e);

            var root = Parent;
            if (root != null)
            {
                while (root.Parent != null)
                {
                    root = root.Parent;
                }
            }

            var dialog = root as ObjectPropertyDialog;
            if (dialog != null)
            {
                dialog.KeyPreview = true;
                dialog.KeyDown += (ss, ee) =>
                {
                    switch (ee.KeyCode)
                    {
                        case Keys.F:
                            if (ee.Modifiers.HasFlag(Keys.Control))
                            {
                                findBar.BringToFront();
                                findBar.Location = new Point(ClientRectangle.Right - (findBar.Width + findBar.Margin.Right), ClientRectangle.Top + findBar.Margin.Top);
                                findBar.Visible = true;
                                findBar.Focus();
                                ee.Handled = true;
                            }
                            break;
                        case Keys.Escape:
                            if (findBar.IsEditing && findBar.Visible)
                            {
                                findBar.Visible = false;
                                ee.Handled = true;
                            }
                            break;
                    }
                };
                dialog.FormClosingByEscapeKey += (ss, ee) =>
                {
                    // 検索窓入力中合は、エスケープキーでのフォームクローズをキャンセルする。
                    // キャンセルによって ObjectPropertyDialog.KeyDown イベントが呼ばれる。
                    ee.Cancel = findBar.IsEditing && findBar.Visible;
                };
            }
        }

        protected override void OnEnter(EventArgs e)
        {
            StopScrollTemporarily(this, e);
        }

        public void StopScrollTemporarily(object sender, EventArgs e)
        {
            // Handle 作成前に BeginInvoke すると例外が飛ぶ
            // Handle 作成前ならScroll が動く問題は起きないはず。
            if (!IsHandleCreated)
            {
                return;
            }

            if (pnlScroll.StopScroll)
            {
                return;
            }

            // フォーカスの当たったコントロールに対して勝手にスクロールしないようにする。
            pnlScroll.StopScroll = true;

            // スクロールされるタイミングがよく分からないので BeginInvoke で待って有効にする。
            Action action = () => {
                pnlScroll.StopScroll = false;
            };

            BeginInvoke(action, null);
        }

        protected override void OnSizeChanged(EventArgs e)
        {
            if ((Owner != null) && (Owner.ActivePage == this) && ((Targets.Active is Material) || (Targets.Active is SeparateMaterial)))
            {
                if (!ObjectPropertyDialog.InternallyChangingSize.IsChanging && Owner.Owner.WindowState == FormWindowState.Normal)
                {
                    ConfigData.ApplicationConfig.Setting.PropertyEdit.ShaderPageSize.Width = Width;
                    ConfigData.ApplicationConfig.Setting.PropertyEdit.ShaderPageSize.Height = Height;
                }

                StopScrollTemporarily(null, null);

                // scgGroup に直接スクロールを表示するとコントロールの位置がずれるので pnlScroll を挟む
                // 順番を気にしないと一瞬横方向のスクロールが表示される
                if (scgGroup.Width > Width - 30)
                {
                    scgGroup.Width = Width - 30;
                    pnlScroll.Width = Width - 6;
                }
                else if (scgGroup.Width < Width - 30)
                {
                    pnlScroll.Width = Width - 6;
                    scgGroup.Width = Width - 30;
                }
                pnlScroll.Height = Height - pnlScroll.Location.Y - 3;
            }
            base.OnSizeChanged(e);
        }

        public override void BeforePageDeactivated()
        {
            findBar.Visible = false;

            base.BeforePageDeactivated();
        }

        // マテリアル参照が解決されているか？
        private bool IsReferenceResolved()
        {
            // マテリアル参照は使用されていない
            if (ActiveTarget.MaterialReference?.ParentMaterials == null || !ActiveTarget.MaterialReference.ParentMaterials.Any())
            {
                return false;
            }

            foreach (var parentMaterial in ActiveTarget.MaterialReference.ParentMaterials)
            {
                // 親マテリアルのマテリアルファイルが読み込まれているか？
                foreach (var material in DocumentManager.Materials.Where(x => x.OwnerDocument is SeparateMaterial))
                {
                    if (parentMaterial.MaterialName == material.Name && parentMaterial.Path == material.OwnerDocument.FileName)
                    {
                        return true;
                    }
                }

                // 親マテリアルのモデルが読み込まれているか？
                Model parentModel = DocumentManager.Models.FirstOrDefault(x => x.FileName == parentMaterial.Path);
                if (parentModel == null)
                {
                    return false;
                }

                // 対象のモデルに親マテリアルは存在するか？
                if (parentModel.Materials.FirstOrDefault(x => x.Name == parentMaterial.MaterialName) == null)
                {
                    return false;
                }
            }

            return true;
        }

        private bool lastFixedSliders = false;

        protected override void UpdateFormInternal(UpdateFormInfo updateFormInfo)
        {
            var currentShaderDefinition = ActiveTarget.MaterialShaderAssign.ShaderDefinition;
            var currentShadingModel = ActiveTarget.MaterialShaderAssign.ShadingModel;

            // マテリアル参照が解決済みの場合、シェーディングモデルは変更できない
            bool changeable = !IsReferenceResolved();
            cmbShaderArchive.Enabled = changeable;
            cmbShadingModel.Enabled = changeable;

            if (lastFixedSliders != ConfigData.ApplicationConfig.UserSetting.UI.FixedSlider)
            {
                lastFixedSliders = ConfigData.ApplicationConfig.UserSetting.UI.FixedSlider;
                UpdateControls = true;
            }

            // 必要なら描画情報を問い合わせ
            if (currentShaderDefinition != null)
            {
                Viewer.QueryRenderInfoMessage.Send(Targets);
            }

            lblShaderDefinition.IsModified = ActiveTarget.IsShaderArchiveModified();

            // コンボボックスの更新
            UpdateComboboxes();

            // ボタン類の更新
            chkHideConditionParam.Enabled = currentShadingModel != null;
            chkHideConditionParam.Checked = !ShowConditionVariable;

            chkShowParamId.Enabled = currentShadingModel != null;
            chkShowParamId.Checked = ShowParamId;

            chkOriginalLabel.Enabled = currentShadingModel != null;
            chkOriginalLabel.Checked = ShowOriginalLabel;

            cbxShadingModel.Enabled = currentShadingModel != null;
            cbxShadingModel.Checked = SucceedVariable;

            btnDefault.Enabled = currentShadingModel != null;

            chkShowForceReferParameter.Checked = ShowForceReferParameter;

            UpdateItemsCopyPasteButton();

            // コントロールの更新
            if (UpdateControls)
            {
                bool pageMode = false;
                pageType page = null;
                if (currentShadingModel != null &&
                    currentShadingModel.page_array != null &&
                    !MaterialShaderPage.IgnorePage)
                {
                    pageMode = true;
                    page = currentShadingModel.Pages().FirstOrDefault(x => x.name == pageId);
                }
                UpdateControls = !UpdateScgGroup(pageMode, page);
            }

            // コントロールの値を更新
            if (scgGroup.UpdateValues(ActiveTarget, (MaterialPropertyPanel)Owner, ShowParamId, ShowOriginalLabel) || UpdateControls)
            {
                // サイズが変更された場合はレイアウトを更新
                scgGroup.UpdateLayout(true);
            }

            // コントロールのアニメーションボタンを更新
            scgGroup.UpdateAnimationMenu(ActiveTarget, Owner.Owner);

            // 検索窓を非表示にする。
            // findBar と finder は表示切替時のイベントで連動しているためイベント呼び出しが必要だが、
            // UpdateFormInternal() は UIControlEventSuppressBlock 内なので
            // findBar.Visible = false ではイベントを呼び出せない。
            // AfterPageActiveted() でもページ表示中の更新に対応できないため。
            // UIControlEventSuppressBlock を回避する InvokeMethodWithForceEventRaising() で処理する。
            findBar.InvokeMethodWithForceEventRaising(findBarVisibleProperty.GetSetMethod(), new object[] { false });
        }

        // コンボボックスの更新
        private void UpdateComboboxes()
        {
            var currentShaderDefinitionName = ActiveTarget.MaterialShaderAssign.ShaderDefinitionFileName;
            var currentShadingModel = ActiveTarget.MaterialShaderAssign.ShaderName;
            var currentShaderDefinition = ActiveTarget.MaterialShaderAssign.ShaderDefinition;

            cmbShadingModel.BeginUpdate();
            cmbShaderArchive.Items.Clear();

            // プログラムグループ
            int count = 0;
            var shaderDefinitionItems = new HashSet<string>();

            var definitions = DocumentManager.ShaderDefinitions;
            var skippedShaderDefinitions = new HashSet<string>();
            foreach (var shaderDefinition in definitions)
            {
                if (shaderDefinition.Data.shading_model_array != null)

                {
                    if (shaderDefinition.Data.shading_model_array.GetItems().Any(x => x.IsMaterialShader()))
                    {
                        cmbShaderArchive.AddItem(shaderDefinition.Name, shaderDefinition.Name);
                        shaderDefinitionItems.Add(shaderDefinition.Name);
                        if (shaderDefinition.Name == currentShaderDefinitionName)
                        {
                            cmbShaderArchive.SelectedIndex = count;
                        }
                        count++;
                    }
                    else
                    {
                        skippedShaderDefinitions.Add(shaderDefinition.Name);
                    }
                }
            }

            // 参照切れ
            var poolDefinition = from assign in ActiveTarget.MaterialShaderAssignPool.Keys
                                 select assign.Key;
            foreach (var shaderDefinition in poolDefinition.Distinct().OrderBy(x => x))
            {
                if (!shaderDefinitionItems.Contains(shaderDefinition) && !skippedShaderDefinitions.Contains(shaderDefinition))
                {
                    cmbShaderArchive.AddItem(shaderDefinition == string.Empty ? res.Strings.ShaderParamControl_NotAssigned : shaderDefinition, shaderDefinition);
                    shaderDefinitionItems.Add(shaderDefinition);
                    if (shaderDefinition == currentShaderDefinitionName)
                    {
                        cmbShaderArchive.SelectedIndex = count;
                    }
                    count++;
                }
            }

            if (!shaderDefinitionItems.Contains(currentShaderDefinitionName))
            {
                cmbShaderArchive.AddItem(currentShaderDefinitionName == string.Empty ? res.Strings.ShaderParamControl_NotAssigned : currentShaderDefinitionName, currentShaderDefinitionName);
                shaderDefinitionItems.Add(currentShaderDefinitionName);
                cmbShaderArchive.SelectedIndex = count;
            }

            // 未指定
            if (!shaderDefinitionItems.Contains(string.Empty))
            {
                cmbShaderArchive.AddItem(res.Strings.ShaderParamControl_NotAssigned, string.Empty);
            }

            var programs = new HashSet<string>();

            lblShadingModel.IsModified = ActiveTarget.IsShaderNameModified() || ActiveTarget.IsShaderRevisionModified();
            // プログラム
            cmbShadingModel.Items.Clear();
            count = 0;
            var skippedShadingModels = new HashSet<string>();

            if (currentShaderDefinition != null && currentShaderDefinition.Data.shading_model_array != null)
            {
                foreach (var shading_model in currentShaderDefinition.Data.shading_model_array.shading_model)
                {
                    if (shading_model.IsMaterialShader())
                    {
                        cmbShadingModel.AddItem(string.Format("{0} (Rev. {1})", shading_model.name, shading_model.revision), shading_model.name);
                        programs.Add(shading_model.name);
                        if (shading_model.name == currentShadingModel)
                        {
                            cmbShadingModel.SelectedIndex = count;
                        }
                        count++;
                    }
                    else
                    {
                        skippedShaderDefinitions.Add(shading_model.name);
                    }
                }
            }

            // 参照切れ
            var poolProgram = from assign in ActiveTarget.MaterialShaderAssignPool.Keys
                              where assign.Key == currentShaderDefinitionName
                              select assign.Value;

            foreach (var program in poolProgram)
            {
                if (!programs.Contains(program) && skippedShadingModels.Contains(program))
                {
                    string text;
                    if (program == string.Empty)
                    {
                        text = res.Strings.ShaderParamControl_NotAssigned;
                    }
                    else
                    {
                        var definition = ActiveTarget.MaterialShaderAssignPool[currentShaderDefinitionName, program];
                        if (definition != null)
                        {
                            text = string.Format("{0} (Rev. {1})", program, definition.Revision);
                        }
                        else
                        {
                            text = program;
                        }
                    }
                    cmbShadingModel.AddItem(text, program);
                    programs.Add(program);
                    if (program == currentShadingModel)
                    {
                        cmbShadingModel.SelectedIndex = count;
                    }
                    count++;
                }
            }

            if (!programs.Contains(currentShadingModel))
            {
                string text;
                if (currentShadingModel == string.Empty)
                {
                    text = res.Strings.ShaderParamControl_NotAssigned;
                }
                else
                {
                    text = string.Format("{0} (Rev. {1})", currentShadingModel, ActiveTarget.MaterialShaderAssign.Revision);
                }

                cmbShadingModel.AddItem(text, currentShadingModel);
                cmbShadingModel.SelectedIndex = count;
            }
            cmbShadingModel.EndUpdate();
        }

        // 項目のコピーペーストの更新
        public void UpdateItemsCopyPasteButton()
        {
            btnCopy.Enabled = ActiveTarget.MaterialShaderAssign.ShadingModel != null;

            bool pageMode = ActiveTarget.MaterialShaderAssign != null &&
                ActiveTarget.MaterialShaderAssign.ShadingModel != null &&
                ActiveTarget.MaterialShaderAssign.ShadingModel.page_array != null &&
                !MaterialShaderPage.IgnorePage;

            if (pageMode)
            {
                ShaderParamControlGroup.GroupCopyData copyData;
                if (pageCopyData.TryGetValue(pageId ?? "", out copyData))
                {
                    btnPaste.Enabled = ShaderParamControlGroup.CanItemsPaste(Targets, copyData, null, true, pageId ?? "", true);
                }
                else
                {
                    btnPaste.Enabled = false;
                }
            }
            else
            {
                btnPaste.Enabled = ActiveTarget.MaterialShaderAssign.ShadingModel != null &&
                    ItemsCopyData != null &&
                    ShaderParamControlGroup.CanItemsPaste(Targets, ItemsCopyData, null, false, null, true);
            }
        }

        // ページごとのスクロール位置
        private readonly Dictionary<string, int> pageScrollPosition = new Dictionary<string, int>();

        // 切り替え直前のスクロール位置
        private int nullSubPagelastY;
        private string pageId;
        private string groupId;
        private string lastPageId;
        private string lastGroupId;
        private Material lastActiveTarget;
        public override PropertyCategoryNode CategoryNode
        {
            get
            {
                return base.CategoryNode;
            }
            set
            {
                base.CategoryNode = value;

                var tag = (MaterialPropertyPanel.ShaderPageNodeTag)value.Tag;

                var pageMode = ActiveTarget != null && ActiveTarget.MaterialShaderAssign.PageMode && !MaterialShaderPage.IgnorePage;
                lastPageId = pageId;
                lastGroupId = groupId;
                pageId = tag.pageName;
                groupId = tag.groupName;

                if (pageMode)
                {
                    if (pageId != lastPageId)
                    {
                        var currentShadingModel = ActiveTarget.MaterialShaderAssign.ShadingModel;

                        // スクロール位置保存
                        pageScrollPosition[lastPageId ?? ""] = pnlScroll.AutoScrollPosition.Y;

                        // ページ切り替え
                        pageId = tag.pageName;
                        {
                            pageType page = currentShadingModel.Pages().FirstOrDefault(x => x.name == pageId);
                            UpdateScgGroup(true, page);

                            scgGroup.UpdateLayout(true);
                        }

                        // 前回のスクロールに設定
                        if (string.IsNullOrEmpty(tag.groupName))
                        {
                            int y;
                            pageScrollPosition.TryGetValue(pageId ?? "", out y);
                            pnlScroll.AutoScrollPosition = new Point(-pnlScroll.AutoScrollPosition.X, -y);
                        }
                    }
                }
                else
                {
                    if (tag.groupName == null)
                    {
                        // 現在のスクロール位置を記録する
                        nullSubPagelastY = pnlScroll.AutoScrollPosition.Y;
                        DebugConsole.WriteLine("#### subId == null " + nullSubPagelastY);
                    }
                }
            }
        }

        public override void AfterPageActiveted(ObjectPropertyPage oldPage)
        {

            bool activeChanged = oldPage != this;
            if (lastActiveTarget == ActiveTarget && groupId == lastGroupId && pageId == lastPageId && !activeChanged)
            {
                return;
            }

            lastActiveTarget = ActiveTarget;
            bool pageMode = ActiveTarget != null && ActiveTarget.MaterialShaderAssign.PageMode && !MaterialShaderPage.IgnorePage;
            if (groupId != null)
            {
                // グループのスクロール位置を設定
                int y = scgGroup.GetGroupY(groupId);

                if (y >= 0)
                {
                    y += scgGroup.Location.Y - pnlScroll.AutoScrollPosition.Y - 5;
                    pnlScroll.AutoScrollPosition = new Point(-pnlScroll.AutoScrollPosition.X, y);
                }
                DebugConsole.WriteLine("#### Scroll subId != null");
            }
            else if (!pageMode)
            {
                // 以前のスクロール位置を設定
                pnlScroll.AutoScrollPosition = new Point(-pnlScroll.AutoScrollPosition.X, -nullSubPagelastY);
                DebugConsole.WriteLine("#### Scroll subId == null");
            }
        }

        // 名前からスクロールを設定する
        public void SetScroll(string groupName, string paramName, ParamType type, string pageName = null)
        {
            // ページ情報があるかどうか？
            var hasPage =
                ActiveTarget != null &&
                ActiveTarget.MaterialShaderAssign.ShadingModel != null &&
                ActiveTarget.MaterialShaderAssign.ShadingModel.page_array != null &&
                !MaterialShaderPage.IgnorePage;

            if (type == ParamType.page)
            {
                if (ActiveTarget.MaterialShaderAssign.ShadingModel != null)
                {
                    var page = ActiveTarget.MaterialShaderAssign.ShadingModel.Pages().FirstOrDefault(x => x.name == pageName);
                    if (page != null)
                    {
                        // ページ切り替え
                        UpdateScgGroup(true, page);
                        scgGroup.UpdateLayout(true);
                    }
                }
            }
            // ページ情報がある場合は、まず指定グループ名、パラメーター名をもつページを探し、
            // そのページをに切り替える
            else if (hasPage)
            {
                // グループを見つける
                if (ActiveTarget.MaterialShaderAssign.ShadingModel != null)
                {
                    var nameToGroup = ActiveTarget.MaterialShaderAssign.ShadingModel.Groups().ToDictionary(x => x.name);
                var dummy = new Dictionary<string, string>();
                var name = MaterialPropertyPanel.PageName(groupName, nameToGroup, dummy);
                var page = ActiveTarget.MaterialShaderAssign.ShadingModel.Pages().FirstOrDefault(x => x.name == name);
                if (page != null)
                {
                    // ページ切り替え
                    UpdateScgGroup(true, page);
                    scgGroup.UpdateLayout(true);
                }
                }
            }

            // scgGroup.UpdateLayout 後に Visible が確定されるまで待つ
            OneShotIdleProcess.Execute(
                () =>
                {
                    if (this.IsDisposed)
                    {
                        return;
                    }

                    // カテゴリビューのカーソルを設定する
                    {
                        Debug.Assert(Owner is MaterialPropertyPanel);
                        var materialPropertyPanel = Owner as MaterialPropertyPanel;
                        if (groupName != null)
                        {
                            Dictionary<string, groupType> nameToGroup = ActiveTarget.MaterialShaderAssign.ShadingModel != null ?
                                ActiveTarget.MaterialShaderAssign.ShadingModel.Groups().ToDictionary(x => x.name):
                                new Dictionary<string, groupType>();
                            Func<string, IEnumerable<string>> parent = x =>
                            {
                                groupType group;
                                if (nameToGroup.TryGetValue(x, out group) && !string.IsNullOrEmpty(group.Group()))
                                {
                                    return Enumerable.Repeat(group.Group(), 1);
                                }

                                return Enumerable.Empty<string>();
                            };

                            var root = TreeUtility.PostOrder(groupName, parent).First();
                            var node = materialPropertyPanel.FindShaderParamPageNode(x => x.groupName == root);
                            if (node != null)
                            {
                                CategoryNode.TreeView.SelectedNode = node;
                            }

                            // 表示対象のグループを探す
                            groupName = TreeUtility.PreOrder(groupName, parent).FirstOrDefault(x => !parent(x).Any() ||
                                string.IsNullOrEmpty(parent(x).First()) ||
                                (nameToGroup.ContainsKey(x) && !string.IsNullOrEmpty(nameToGroup[x].Label())));
                        }
                        else if (pageName != null)
                        {
                            var node = materialPropertyPanel.FindShaderParamPageNode(x => x.pageName == pageName && x.groupName == null);
                            if (node != null)
                            {
                                CategoryNode.TreeView.SelectedNode = node;
                            }
                        }
                    }

                    // グループがなく、パラメーターだけの指定
                    if ((groupName == null) && (paramName != null))
                    {
                        // パラメーターのスクロール位置を設定
                        int y = scgGroup.GetUniformY(paramName);
                        if (y >= 0)
                        {
                            y += scgGroup.Location.Y - pnlScroll.AutoScrollPosition.Y - 5;
                            pnlScroll.AutoScrollPosition = new Point(-pnlScroll.AutoScrollPosition.X, y);
                        }
                    }
                    else
                    // グループの指定
                    if (paramName == null)
                    {
                        // グループのスクロール位置を設定
                        int y = (groupName == null) ? -1 : scgGroup.GetGroupY(groupName);
                        if (y >= 0)
                        {
                            y += scgGroup.Location.Y - pnlScroll.AutoScrollPosition.Y - 5;
                            pnlScroll.AutoScrollPosition = new Point(-pnlScroll.AutoScrollPosition.X, y);
                        }
                    }
                    // パラメーターの指定
                    else
                    if (paramName != null)
                    {
                        // パラメーターのスクロール位置を設定
                        int y = (groupName == null) ? -1 : scgGroup.GetUniformY(groupName, type, paramName);
                        if (y >= 0)
                        {
                            y += scgGroup.Location.Y - pnlScroll.AutoScrollPosition.Y - 5;
                            pnlScroll.AutoScrollPosition = new Point(-pnlScroll.AutoScrollPosition.X, y);
                        }
                        // パラメーターが見つからないのでグループのみで試みる
                        else
                        {
                            // グループのスクロール位置を設定
                            int gy = (groupName == null) ? -1 : scgGroup.GetGroupY(groupName);
                            if (gy >= 0)
                            {
                                gy += scgGroup.Location.Y - pnlScroll.AutoScrollPosition.Y - 5;
                                pnlScroll.AutoScrollPosition = new Point(-pnlScroll.AutoScrollPosition.X, gy);
                            }
                        }
                    }
                }
            );
        }

        /// <summary>
        /// スクロールを維持したままコントロールを更新
        /// リソース不足の場合は false
        /// </summary>
        protected bool UpdateScgGroup(bool pageMode, pageType page)
        {
            DebugConsole.WriteLine("UpdateScgGroup");
            var currentProgram = ActiveTarget.MaterialShaderAssign.ShadingModel;
            // こちらだとスライダーバーの表示がおかしくなる
            //Win32.User32.SendMessage(Handle, Win32.WM.WM_SETREDRAW, 0, 0);
            Win32.NativeMethods.LockWindowUpdate(pnlScroll.Handle);

            var result = scgGroup.UpdateControls(ActiveTarget, currentProgram, ShowConditionVariable, pageMode, page, ShowParamId);
            //Win32.User32.SendMessage(Handle, Win32.WM.WM_SETREDRAW, 1, 0);
            Win32.NativeMethods.LockWindowUpdate(IntPtr.Zero);

            return result;
        }

        private void cmbShadingModel_CustomDrawItem(object sender, CustomDrawListControlItemEventArgs e)
        {
            // ダミーは灰色にする
            if ((e.Item != null) && (e.Item.Data == null))
            {
                e.ForeColor = SystemColors.GrayText;
            }
        }

        private void cmbShaderArchive_SelectedIndexChanged(object sender, EventArgs e)
        {
            string program = null;
            string shaderArchive = (string)cmbShaderArchive.SelectedItemData;
            if (ActiveTarget.MaterialShaderAssign.ShaderDefinitionFileName == shaderArchive)
            {
                program = (string)cmbShadingModel.SelectedItemData;
            }

            if (program == null && ActiveTarget.LastProgram.ContainsKey(shaderArchive))
            {
                program = ActiveTarget.LastProgram[shaderArchive];
            }

            if (program == null)
            {
                var shaderDefinition = DocumentManager.ShaderDefinitions.FirstOrDefault(x => x.Name == shaderArchive);
                if (shaderDefinition != null &&
                    shaderDefinition.Data.shading_model_array != null &&
                    shaderDefinition.Data.shading_model_array.shading_model != null &&
                    shaderDefinition.Data.shading_model_array.shading_model.Length > 0)
                {
                    program = shaderDefinition.Data.shading_model_array.shading_model[0].name;
                }
                else
                {
                    program = "";
                }
            }

            ChangeDefinition(shaderArchive, program);
        }

        private void cmbShadingModel_SelectedIndexChanged(object sender, EventArgs e)
        {
            string shaderArchive = (string)cmbShaderArchive.SelectedItemData;
            ChangeDefinition(shaderArchive, (string)cmbShadingModel.SelectedItemData);
        }

        // 割り当てるシェーダープログラムの切り替え
        private void ChangeDefinition(string shader_archive, string programName)
        {
            EditCommandSet commandSet2 = new EditCommandSet();
            EditCommandSet commandSet = new EditCommandSet();
            using (var propertyblock = new App.AppContext.PropertyChangedSuppressBlock())
            {
                var models = Targets.GetObjects(GuiObjectID.Material).OfType<Material>().SelectMany(x => x.Referrers).Distinct().ToArray();
                HashSet<Model> notLoaded = new HashSet<Model>(models.Where(x => !x.HioLoaded));

                using (var viewerDrawSuppressBlock = new Viewer.ViewerDrawSuppressBlock(Viewer.ViewerDrawSuppressBlock.DiscardAllMessages))
                {
                    // プール
                    commandSet.Add(CreateEditCommand_ShaderPool(Targets).Execute());

                    var shaderAssigns = (from material in Targets.GetObjects(GuiObjectID.Material).OfType<Material>()
                                         select new Tuple<Material, MaterialShaderAssign>(material, GetShaderAssign(material, shader_archive, programName))).ToArray();
                    commandSet.Add(ShaderAssignUtility.CreateEditCommand_ShaderAssign(Targets, shaderAssigns.Select(x => x.Item2), false).Execute());

                    if (cbxShadingModel.Checked)
                    {
                        var command = CreateShaderAssignEditCommand(Targets, false, true, false); // 見えないパラメーターも引き継ぐ
                        if (command != null)
                        {
                            commandSet.Add(command.Execute());
                        }
                    }

                    commandSet.Add(ShaderAssignUtility.CreateVtxBufferEditCommand(Targets).Execute());
                    var inconsistentMaterials = Targets.GetObjects(GuiObjectID.Material).OfType<Material>().Where(x => !ShaderAssignUtility.IsConsistentWithDefinition(x)).ToArray();
                    if (inconsistentMaterials.Any())
                    {
                        commandSet.Add(ShaderAssignUtility.ExecuteFixParameters(inconsistentMaterials, reload: true));
                    }

                    // バインドを更新
                    commandSet.Add(DocumentManager.CreateAnimationUpdateBindCommand(
                        Targets.Objects.OfType<Material>().SelectMany(x => x.Referrers).Distinct().SelectMany(x => x.AllAnimations).Distinct()
                        ).Execute());

                    // コマンドセットにモデル再転送のデリゲートを設定する。
                    commandSet.SetViewerDrawSuppressBlockDelegate(Viewer.ViewerDrawSuppressBlock.DiscardAllMessages);
                }
                commandSet.Reverse();

                commandSet2.Add(commandSet);
                EventHandler postEdit = (s, e) =>
                    {
                        foreach (var model in models)
                        {
                            Viewer.LoadOrReloadModel.Send(model);
                            if (notLoaded.Contains(model) || string.IsNullOrEmpty(programName))
                            {
                                // プレビュー情報を送る
                                model.SendEditBoneBind();
                                model.SendEditModelLayout(false, sendChildren: true);

                                // アニメーションをバインドする
                                Viewer.ViewerUtility.SendAnimationSet(model);
                            }
                        }
                    };
                postEdit(null, null);
                commandSet2.OnPostEdit += postEdit;
                TheApp.CommandManager.Add(commandSet2);
            }
        }

        private void SetDefaultShaderAssign()
        {
            EditCommandSet commandSet2 = new EditCommandSet();
            EditCommandSet commandSet = new EditCommandSet();
            using (var propertyblock = new App.AppContext.PropertyChangedSuppressBlock())
            {
                var materials = Targets.GetObjects(GuiObjectID.Material).OfType<Material>().Where(x => x.MaterialShaderAssign.ShadingModel != null);
                var models = materials.SelectMany(x => x.Referrers).Distinct().ToArray();
                using (var viewerDrawSuppressBlock = new Viewer.ViewerDrawSuppressBlock(Viewer.ViewerDrawSuppressBlock.DiscardAllMessages))
                {
                    // プール
                    var targets = new GuiObjectGroup(materials);
                    var shaderAssigns = (from material in materials
                                         select new Tuple<Material, MaterialShaderAssign>(material,
                                             GetDefaultShaderAssign(material, material.MaterialShaderAssign.ShaderDefinitionFileName, material.MaterialShaderAssign.ShaderName))).ToArray();
                    commandSet.Add(ShaderAssignUtility.CreateEditCommand_ShaderAssign(targets, shaderAssigns.Select(x => x.Item2), false).Execute());

                    commandSet.Add(ShaderAssignUtility.CreateVtxBufferEditCommand(targets).Execute());
                    var inconsistentMaterials = targets.Objects.OfType<Material>().Where(x => !ShaderAssignUtility.IsConsistentWithDefinition(x)).ToArray();
                    if (inconsistentMaterials.Any())
                    {
                        commandSet.Add(ShaderAssignUtility.ExecuteFixParameters(inconsistentMaterials, reload: true));
                    }

                    // バインドを更新
                    commandSet.Add(DocumentManager.CreateAnimationUpdateBindCommand(
                        materials.SelectMany(x => x.Referrers).Distinct().SelectMany(x => x.AllAnimations).Distinct()
                        ).Execute());

                    // コマンドセットにモデル再転送のデリゲートを設定する。
                    commandSet.SetViewerDrawSuppressBlockDelegate(Viewer.ViewerDrawSuppressBlock.DiscardAllMessages);
                }
                commandSet.Reverse();

                commandSet2.Add(commandSet);
                EventHandler postEdit = (s, e) =>
                    {
                        foreach (var model in models)
                        {
                            Viewer.LoadOrReloadModel.Send(model);
                        }
                    };
                postEdit(null, null);
                commandSet2.OnPostEdit += postEdit;
                TheApp.CommandManager.Add(commandSet2);
            }
        }

        private EditCommand CreateShaderAssignEditCommand(GuiObjectGroup targets, bool sendViewer, bool checkEditable, bool checkVisible)
        {
            var materials = targets.Objects.OfType<Material>();

            var commandSet = new EditCommandSet();
            commandSet.SetViewerDrawSuppressBlockDelegate(Viewer.ViewerDrawSuppressBlock.DiscardAllMessages);
            foreach (var material in materials)
            {
                var copyData = CreateCopyData(material.MaterialShaderAssignPool, ShaderParamControlGroup.CreatePageCopyData(material, false, null, checkEditable, checkVisible));
                var command = ShaderParamControlGroup.CreateItemsPasteCommand(new GuiObjectGroup(material), copyData, null, false, null, checkEditable, false, sendViewer:sendViewer, checkVisible:checkVisible);
                if (command != null)
                {
                    commandSet.Add(command);
                }
            }

            if (commandSet.CommandCount > 0)
            {
                var commandSet2 = new EditCommandSet();
                var models = materials.SelectMany(x => x.Referrers).Distinct().ToArray();
                commandSet2.OnPostEdit += (s, e) =>
                    {
                        if (sendViewer)
                        {
                            foreach (var model in models)
                            {
                                Viewer.LoadOrReloadModel.Send(model);
                            }
                        }
                    };
                commandSet2.Add(commandSet);
                return commandSet2;
            }
            else
            {
                return null;
            }
        }

        private static MaterialShaderAssign GetShaderAssign(Material material, string shaderArchive, string programName)
        {
            if (material.MaterialShaderAssign.ShaderDefinitionFileName == shaderArchive &&
                material.MaterialShaderAssign.ShaderName == programName)
            {
                return material.MaterialShaderAssign;
            }

            var assign = material.MaterialShaderAssignPool[shaderArchive, programName];
            if (assign != null)
            {
                return assign;
            }

            return GetDefaultShaderAssign(material, shaderArchive, programName);
        }

        public static MaterialShaderAssign GetDefaultShaderAssign(Material material, string shaderArchive, string programName)
        {
            var shaderDefinition = DocumentManager.ShaderDefinitions.FirstOrDefault(x => x.Name == shaderArchive);
            var definition = shaderDefinition != null ? shaderDefinition.Definitions.FirstOrDefault(x => x.Name == programName): null;

            // 新規作成
            if (definition == null)
            {
                return new MaterialShaderAssign(shaderArchive, programName);
            }

            Debug.Assert(definition.Data.IsMaterialShader());

            // model が null でも IfShaderAssignUtility.UpdateShaderAssign() での更新が必要。
            var model = (material.OwnerDocument as Model)?.Data;
            material.Data.shader_assign = null;
            IfShaderAssignUtility.UpdateShaderAssign(
                model,
                material.Data,
                shaderArchive,
                definition.Data,
                initialize: true,
                allowEmptyString: true);
            var shader_assign = material.Data.shader_assign;
            Debug.Assert(shader_assign != null);
            return new MaterialShaderAssign(shader_assign, true);
        }


        /// <summary>
        /// CreateEditCommand_ShaderPool 用のクラス
        /// </summary>
        public class ShaderPoolData
        {
            public MaterialShaderAssign assign;
            public string shaderArchive;
            public string programName;
            public string lastProgram;
            public ShaderParamControlGroup.GroupCopyData itemsData;
        }


        public static GroupEditCommand CreateEditCommand_ShaderPool(GuiObjectGroup targets)
        {
            var shaderAssigns = from material in targets.GetObjects(GuiObjectID.Material).OfType<Material>()
                                let shaderAssign = material.MaterialShaderAssign
                                select new ShaderPoolData {
                                    assign = shaderAssign,
                                    shaderArchive = shaderAssign.ShaderDefinitionFileName,
                                    programName = shaderAssign.ShaderName,
                                    lastProgram = shaderAssign.ShaderName,
                                    itemsData = ShaderParamControlGroup.CreatePageCopyData(material, false, null, false, false),
                                };
            return new GeneralGroupReferenceEditCommand<ShaderPoolData>(
                targets,
                GuiObjectID.Material,
                shaderAssigns,
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    Debug.Assert(target is Material);
                    Material material = target as Material;
                    ShaderPoolData pooldata = (ShaderPoolData)data;

                    // プールの状態を取得
                    swap = new ShaderPoolData()
                    {
                        assign = material.MaterialShaderAssignPool[pooldata.shaderArchive, pooldata.programName],
                        programName = pooldata.programName,
                        shaderArchive = pooldata.shaderArchive,
                        lastProgram = material.LastProgram.ContainsKey(pooldata.shaderArchive) ? material.LastProgram[pooldata.shaderArchive] : null,
                        itemsData = CreateCopyData(material.MaterialShaderAssignPool, pooldata.itemsData),
                    };

                    // プールの更新
                    material.MaterialShaderAssignPool[pooldata.shaderArchive, pooldata.programName] = pooldata.assign;

                    foreach (var pair in pooldata.itemsData.AttribAssigns)
                    {
                        material.MaterialShaderAssignPool.AttribAssigns[pair.Item1] = pair.Item2;
                    }

                    foreach (var pair in pooldata.itemsData.ShaderOptions)
                    {
                        material.MaterialShaderAssignPool.ShaderOptions[pair.Item1] = pair.Item2;
                    }

                    foreach (var pair in pooldata.itemsData.SamplerAssigns)
                    {
                        material.MaterialShaderAssignPool.SamplerAssigns[pair.Item1] = pair.Item2;
                    }

                    foreach (var pair in pooldata.itemsData.ShaderParams)
                    {
                        material.MaterialShaderAssignPool.ShaderParams[pair.Item1] = pair.Item2;
                    }

                    foreach (var pair in pooldata.itemsData.RenderInfos)
                    {
                        material.MaterialShaderAssignPool.RenderInfos[pair.Item1] = pair.Item2;
                    }

                    if (pooldata.lastProgram == null)
                    {
                        material.LastProgram.Remove(pooldata.shaderArchive);
                    }
                    else
                    {
                        material.LastProgram[pooldata.shaderArchive] = pooldata.lastProgram;
                    }
                }
            );
        }

        public static ShaderParamControlGroup.GroupCopyData CreateCopyData(MaterialShaderAssignPool pool, ShaderParamControlGroup.GroupCopyData newData)
        {
            return new ShaderParamControlGroup.GroupCopyData
            {
                AttribAssigns = newData.AttribAssigns.Select(x =>
                    new Tuple<attrib_varType, attrib_assignType>(x.Item1, pool.AttribAssigns.ContainsKey(x.Item1) ? pool.AttribAssigns[x.Item1] : null)).ToList(),
                ShaderOptions = newData.ShaderOptions.Select(x =>
                    new Tuple<option_varType, shader_optionType>(x.Item1, pool.ShaderOptions.ContainsKey(x.Item1) ? pool.ShaderOptions[x.Item1] : null)).ToList(),
                SamplerAssigns = newData.SamplerAssigns.Select(x =>
                    new Tuple<sampler_varType, sampler_assignType>(x.Item1, pool.SamplerAssigns.ContainsKey(x.Item1) ? pool.SamplerAssigns[x.Item1] : null)).ToList(),
                ShaderParams = newData.ShaderParams.Select(x =>
                    new Tuple<uniform_varType, shader_paramType>(x.Item1, pool.ShaderParams.ContainsKey(x.Item1) ? pool.ShaderParams[x.Item1] : null)).ToList(),
                RenderInfos = newData.RenderInfos.Select(x =>
                    new Tuple<render_info_slotType, RenderInfo>(x.Item1, pool.RenderInfos.ContainsKey(x.Item1) ? pool.RenderInfos[x.Item1] : null)).ToList(),
            };
        }

        #region コピー＆ペースト
        public class CopyData
        {
            public string shader_archive { get; set; }
            public string shader_assign_name{ get; set; }
            public ShaderParamControlGroup.GroupCopyData itemsCopyData { get; set; }
            public CustomUI CustomUI { get; set; }
        }

        /// <summary>
        /// コピーが可能か。
        /// </summary>
        public override bool CanCopy()
        {
            return CanCopy(ActiveTarget);
        }

        public static bool CanCopy(Material target)
        {
            return target != null && target.MaterialShaderAssign.ShadingModel != null && target.MaterialShaderAssign.ShadingModel.IsMaterialShader();
        }

        /// <summary>
        /// コピー。
        /// </summary>
        public override object Copy(ref object copyObjectInfo)
        {
            return Copy(ActiveTarget);
        }

        /// <summary>
        /// コピー。
        /// </summary>
        public static object Copy(Material target)
        {
            var data = new CopyData();
            data.shader_archive = target.MaterialShaderAssign.ShaderDefinitionFileName;
            data.shader_assign_name = target.MaterialShaderAssign.ShaderName;
            data.itemsCopyData = ShaderParamControlGroup.CreatePageCopyData(target, false, null, true, true);
            data.CustomUI = target.CustomUI.Clone();
            return data;
        }

        public override bool CanPaste(object copiedObjectInfo, object copiedObject)
        {
            if (!base.CanPaste(copiedObjectInfo, copiedObject))
            {
                return false;
            }

            return CanPaste(copiedObject);
        }

        public static bool CanPaste(object copiedObject)
        {
            var data = (CopyData)copiedObject;
            var definition = DocumentManager.ShaderDefinitions.FirstOrDefault(x => x.Name == data.shader_archive);
            if (definition != null)
            {
                var shadingModel = definition.Data.shading_model_array.GetItems().FirstOrDefault(x => x.name == data.shader_assign_name);
                if (shadingModel != null && !shadingModel.IsMaterialShader())
                {
                    return false;
                }
            }
            else
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// ペースト。
        /// </summary>
        public override void Paste(object pasteObject)
        {
            TheApp.CommandManager.Add(Paste(Targets, pasteObject));
        }

        private static List<T> FilterInvisibles<T>(
            Material material,
            CopyData copyData,
            Func<Material, IEnumerable<T>> elements,
            Func<CopyData, IEnumerable<T>> sources,
            Predicate<T> invisible,
            Func<T, T, bool> equal)
        {
            if (material.MaterialShaderAssign.ShaderDefinitionFileName == copyData.shader_archive &&
                material.MaterialShaderAssign.ShaderName == copyData.shader_assign_name)
            {
                return ObjectUtility.Clone(sources(copyData)
                    .Select(
                    y => invisible(y) ? elements(material).FirstOrDefault(z => equal(z, y)) : y // 見えない場合は元のを使う
                    ).Where(y => y != null).ToList());
            }
            else
            {
                return ObjectUtility.Clone(sources(copyData).Where(y => !invisible(y)).ToList()); // 見えない場合は削除する
            }
        }

#if true
        /// <summary>
        /// ペースト。
        /// </summary>
        public static ICommand Paste(GuiObjectGroup targets, object pasteObject, bool sendViewer = true)
        {
            EditCommandSet commandSet2 = new EditCommandSet();
            EditCommandSet commandSet = new EditCommandSet();
            using (var block = new App.AppContext.PropertyChangedSuppressBlock())
            {
                var targetMaterials = targets.GetObjects(GuiObjectID.Material).OfType<Material>().ToArray();
                var models = targetMaterials.SelectMany(x => x.Referrers).Distinct().ToArray();
                HashSet<Model> notLoaded = new HashSet<Model>(models.Where(x => !x.HioLoaded));
                var copyData = (CopyData)pasteObject;
                using (var viewerDrawSuppressBlock = new Viewer.ViewerDrawSuppressBlock(Viewer.ViewerDrawSuppressBlock.DiscardAllMessages))
                {
                    var shaderDefinition = DocumentManager.ShaderDefinitions.FirstOrDefault(x => x.Name == copyData.shader_archive);
                    var shadingModel = shaderDefinition != null ? shaderDefinition.Data.shading_model_array.shading_model.FirstOrDefault(x => x.name == copyData.shader_assign_name) : null;
                    /*
                    // 編集後のパラメーター
                    var invisibleParams = new HashSet<string>(shadingModel.MaterialUniforms().Where(x => !x.IsParamVisible()).Select(x => x.id));
                    var shaderParams = targetMaterials.Select(
                        x => FilterInvisibles(
                            x,
                            copyData,
                            y => y.MaterialShaderAssign.ShaderParams,
                            y => y.ShaderParams,
                            y => invisibleParams.Contains(y.id),
                            (y, z) => y.id == z.id)).ToArray();

                    var invisibleSamplerAssigns = new HashSet<string>(shadingModel.Samplers().Where(x => !x.SamplerVisible()).Select(x => x.id));
                    var samplerAssigns = targetMaterials.Select(
                        x => FilterInvisibles(
                            x,
                            copyData,
                            y => y.MaterialShaderAssign.SamplerAssigns,
                            y => y.SamplerAssigns,
                            y => invisibleSamplerAssigns.Contains(y.id),
                            (y, z) => y.id == z.id)).ToArray();

                    var invisibleAttribAssigns = new HashSet<string>(shadingModel.Attributes().Where(x => !x.Editable()).Select(x => x.id));
                    var attribAssigns = targetMaterials.Select(
                        x => FilterInvisibles(
                            x,
                            copyData,
                            y => y.MaterialShaderAssign.AttribAssigns,
                            y => y.AttribAssigns,
                            y => invisibleAttribAssigns.Contains(y.id),
                            (y, z) => y.id == z.id)).ToArray();
                    */

                    // プール
                    commandSet.Add(CreateEditCommand_ShaderPool(targets).Execute());

                    var shaderAssigns = (from material in targets.GetObjects(GuiObjectID.Material).OfType<Material>()
                                         select GetShaderAssign(material, copyData.shader_archive, copyData.shader_assign_name)).ToArray();
                    commandSet.Add(ShaderAssignUtility.CreateEditCommand_ShaderAssign(targets, shaderAssigns, false).Execute());


                    commandSet.Add(ShaderParamControlGroup.CreateItemsPasteCommand(targets, copyData.itemsCopyData, null, false, null, true, false, sendViewer: false, checkVisible: true).Execute());

                    // バッファを更新
                    commandSet.Add(ShaderAssignUtility.CreateVtxBufferEditCommand(targets).Execute());

                    var inconsistentMaterials = targets.GetObjects(GuiObjectID.Material).OfType<Material>().Where(x => !ShaderAssignUtility.IsConsistentWithDefinition(x)).ToArray();
                    if (inconsistentMaterials.Any())
                    {
                        commandSet.Add(ShaderAssignUtility.ExecuteFixParameters(inconsistentMaterials, reload: false));
                    }

                    // バインドを更新
                    commandSet.Add(DocumentManager.CreateAnimationUpdateBindCommand(
                        targets.Objects.OfType<Material>().SelectMany(x => x.Referrers).Distinct().SelectMany(x => x.AllAnimations).Distinct()
                        ).Execute());


                    commandSet.Add(CreateEditCommand_CustomUI(targets, copyData.CustomUI).Execute());

                    // コマンドセットにモデル再転送のデリゲートを設定する。
                    commandSet.SetViewerDrawSuppressBlockDelegate(Viewer.ViewerDrawSuppressBlock.DiscardAllMessages);
                }
                commandSet.Reverse();
                commandSet2.Add(commandSet);
                if (sendViewer)
                {
                    EventHandler onPostEdit = (s, e) =>
                    {
                        foreach (var model in models)
                        {
                            Viewer.LoadOrReloadModel.Send(model);

                            if (notLoaded.Contains(model) || string.IsNullOrEmpty(copyData.shader_assign_name))
                            {
                                // プレビュー情報を送る
                                //if (model.PreviewInfo.IsEdited)
                                {
                                    model.SendEditBoneBind();
                                    model.SendEditModelLayout(false, sendChildren: true);
                                }

                                // アニメーションをバインドする
                                Viewer.ViewerUtility.SendAnimationSet(model);
                            }
                        }
                    };
                    onPostEdit(null, null);
                    commandSet2.OnPostEdit += onPostEdit;
                }
                return commandSet2;
            }
        }
#else
        /// <summary>
        /// ペースト。
        /// </summary>
        public static ICommand Paste(GuiObjectGroup targets, object pasteObject, bool sendViewer = true)
        {
            EditCommandSet commandSet2 = new EditCommandSet();
            EditCommandSet commandSet = new EditCommandSet();
            using (var block = new App.AppContext.PropertyChangedSuppressBlock())
            {
                var targetMaterials = targets.GetObjects(GuiObjectID.Material).OfType<Material>().ToArray();
                var models = targetMaterials.Select(x => x.Owner).Distinct().ToArray();
                HashSet<Model> notLoaded = new HashSet<Model>(models.Where(x => !x.HioLoaded));
                var copyData = (CopyData)pasteObject;
                using (var viewerDrawSuppressBlock = new Viewer.ViewerDrawSuppressBlock(Viewer.ViewerDrawSuppressBlock.DiscardAllMessages))
                {
                    var shaderDefinition = DocumentManager.ShaderDefinitions.FirstOrDefault(x => x.Name == copyData.shader_archive);
                    var shadingModel = shaderDefinition != null ? shaderDefinition.Data.shading_model_array.shading_model.FirstOrDefault(x => x.name == copyData.shader_assign_name) : null;

                    // 編集後のパラメーター
                    var invisibleParams = new HashSet<string>(shadingModel.MaterialUniforms().Where(x => !x.IsParamVisible()).Select(x => x.id));
                    var shaderParams = targetMaterials.Select(
                        x => FilterInvisibles(
                            x,
                            copyData,
                            y => y.MaterialShaderAssign.ShaderParams,
                            y => y.ShaderParams,
                            y => invisibleParams.Contains(y.id),
                            (y, z) => y.id == z.id)).ToArray();

                    var invisibleSamplerAssigns = new HashSet<string>(shadingModel.Samplers().Where(x => !x.SamplerVisible()).Select(x => x.id));
                    var samplerAssigns = targetMaterials.Select(
                        x => FilterInvisibles(
                            x,
                            copyData,
                            y => y.MaterialShaderAssign.SamplerAssigns,
                            y => y.SamplerAssigns,
                            y => invisibleSamplerAssigns.Contains(y.id),
                            (y, z) => y.id == z.id)).ToArray();

                    var invisibleAttribAssigns = new HashSet<string>(shadingModel.Attributes().Where(x => !x.Editable()).Select(x => x.id));
                    var attribAssigns = targetMaterials.Select(
                        x => FilterInvisibles(
                            x,
                            copyData,
                            y => y.MaterialShaderAssign.AttribAssigns,
                            y => y.AttribAssigns,
                            y => invisibleAttribAssigns.Contains(y.id),
                            (y, z) => y.id == z.id)).ToArray();

                    // プール
                    commandSet.Add(CreateEditCommand_ShaderPool(targets).Execute());

                    var shaderAssigns = (from material in targets.GetObjects(GuiObjectID.Material).OfType<Material>()
                                         select GetShaderAssign(material, copyData.shader_archive, copyData.shader_assign_name)).ToArray();
                    commandSet.Add(ShaderAssignUtility.CreateEditCommand_ShaderAssign(targets, shaderAssigns, false).Execute());

                    var shaderOptions = ObjectUtility.MultipleClone<List<shader_optionType>>(copyData.ShaderOptions.ToList(), targetMaterials.Length);
                    commandSet.Add(ShaderAssignUtility.CreateShaderOptionsEditCommand(targets, shaderOptions).Execute());

                    commandSet.Add(ShaderAssignUtility.CreateSamplerAssignsEditCommand(targets, samplerAssigns, reload:false).Execute());

                    commandSet.Add(ShaderAssignUtility.CreateShaderParamsEditCommand(targets, shaderParams).Execute());

                    commandSet.Add(ShaderAssignUtility.CreateAttribAssignsEditCommand(targets, attribAssigns, reload:false).Execute());

                    var renderInfos = ObjectUtility.MultipleClone<List<RenderInfo>>(copyData.RenderInfos.ToList(), targetMaterials.Length);
                    commandSet.Add(ShaderAssignUtility.CreateRenderInfoEditCommand(targets, renderInfos, reload:false).Execute());

                    // バッファを更新
                    commandSet.Add(ShaderAssignUtility.CreateVtxBufferEditCommand(targets).Execute());

                    var inconsistentMaterials = targets.GetObjects(GuiObjectID.Material).OfType<Material>().Where(x => !ShaderAssignUtility.IsConsistentWithDefinition(x)).ToArray();
                    if (inconsistentMaterials.Any())
                    {
                        commandSet.Add(ShaderAssignUtility.ExecuteFixParameters(inconsistentMaterials, reload:false));
                    }

                    // バインドを更新
                    commandSet.Add(DocumentManager.CreateAnimationUpdateBindCommand(
                        targets.Objects.OfType<Material>().Select(x => x.Owner).Distinct().SelectMany(x => x.AllAnimations).Distinct()
                        ).Execute());


                    commandSet.Add(CreateEditCommand_CustomUI(targets, copyData.CustomUI).Execute());

                    // コマンドセットにモデル再転送のデリゲートを設定する。
                    commandSet.SetViewerDrawSuppressBlockDelegate(Viewer.ViewerDrawSuppressBlock.DiscardAllMessages);
                }
                commandSet.Reverse();
                commandSet2.Add(commandSet);
                if (sendViewer)
                {
                    EventHandler onPostEdit = (s, e) =>
                    {
                        foreach (var model in models)
                        {
                            Viewer.LoadOrReloadModel.Send(model);

                            if (notLoaded.Contains(model) || string.IsNullOrEmpty(copyData.shader_assign_name))
                            {
                                // プレビュー情報を送る
                                //if (model.PreviewInfo.IsEdited)
                                {
                                    model.SendEditBoneBind();
                                    model.SendEditModelLayout(false);
                                }

                                // アニメーションをバインドする
                                Viewer.ViewerUtility.SendAnimationSet(model);
                            }
                        }
                    };
                    onPostEdit(null, null);
                    commandSet2.OnPostEdit += onPostEdit;
                }
                return commandSet2;
            }
        }
#endif
        #endregion


        public static GroupEditCommand CreateEditCommand_CustomUI(
            GuiObjectGroup targets,
            CustomUI customUI)
        {
            return new GeneralGroupReferenceEditCommand<CustomUI>(
                targets,
                GuiObjectID.Material,
                targets.Objects.OfType<Material>().Select(x => customUI.Clone()),
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    var material = (Material)target;
                    swap = material.CustomUI;
                    material.CustomUI = (CustomUI)data;
                });
        }

        private void scgGroup_ValueChanged(object sender, ValueChangedEventArgs e)
        {
            if (e is SamplerValueChangedEventArgs)
            {
                scgGroup.scgGroup_ValueChanged(ActiveTarget, Targets, e as SamplerValueChangedEventArgs);
            }
            else if (e is ShaderParamValueChangedEventArgs)
            {
                scgGroup.scgGroup_ValueChanged(ActiveTarget, Targets, e as ShaderParamValueChangedEventArgs);
            }
            else if (e is OptionValueChangedEventArgs)
            {
                scgGroup.scgGroup_ValueChanged(ActiveTarget, Targets, e as OptionValueChangedEventArgs);
            }
            else if (e is AttributeValueChangedEventArgs)
            {
                scgGroup.scgGroup_ValueChanged(ActiveTarget, Targets, e as AttributeValueChangedEventArgs);
            }
            else if (e is RenderInfoValueChangedEventArgs)
            {
                scgGroup.scgGroup_ValueChanged(ActiveTarget, Targets, e as RenderInfoValueChangedEventArgs);
            }
            else
            {
                Debug.Assert(false);
            }
        }

        public void LabelEditCommand(object sender, LabelChangedEventArgs e)
        {
            var paramName = e.ParamName;
            var label = e.Lebel;
            EditCommand command = null;
            switch (e.ParamType)
            {
                case ParamType.option_var:
                    command = CreateEditCommand_CustomOptionLabel(paramName, label);
                    break;
                case ParamType.attrib_var:
                    command = CreateEditCommand_CustomAttribLabel(paramName, label);
                    break;
                case ParamType.sampler_var:
                    command = CreateEditCommand_CustomSamplerLabel(paramName, label);
                    break;
                case ParamType.uniform_var:
                    command = CreateEditCommand_CustomUniformLabel(paramName, label);
                    break;
                case ParamType.render_info_slot:
                    command = CreateEditCommand_CustomRenderInfoLabel(paramName, label);
                    break;
                case ParamType.group:
                    Debug.Assert(false);
                    //command = CreateEditCommand_CustomGroupLabel(paramName, label);
                    break;
            }

            if (command != null)
            {
                TheApp.CommandManager.Execute(command);
            }
        }

        public EditCommand CreateEditCommand_CustomOptionLabel(string id, string label)
        {
            var targets = new GuiObjectGroup(Targets.GetObjects(GuiObjectID.Material).OfType<Material>()
                .Where(x => x.MaterialShaderAssign.ShaderOptions.Any(y => y.id == id)));
            return CreateEditCommand_CustomLabel(targets, id, label, x => x.CustomUI.option_vars, (target, command) => new DocumentPropertyChangedLabelArgs(target, command));
        }

        public EditCommand CreateEditCommand_CustomAttribLabel(string id, string label)
        {
            var targets = new GuiObjectGroup(Targets.GetObjects(GuiObjectID.Material).OfType<Material>()
                .Where(x => x.MaterialShaderAssign.AttribAssigns.Any(y => y.id == id)));
            return CreateEditCommand_CustomLabel(targets, id, label, x => x.CustomUI.attrib_vars, (target, command) => new DocumentPropertyChangedLabelArgs(target, command));
        }

        public EditCommand CreateEditCommand_CustomSamplerLabel(string id, string label)
        {
            var targets = new GuiObjectGroup(Targets.GetObjects(GuiObjectID.Material).OfType<Material>()
                .Where(x => x.MaterialShaderAssign.SamplerAssigns.Any(y => y.id == id)));
            return CreateEditCommand_CustomLabel(targets, id, label, x => x.CustomUI.sampler_vars, (target, command) => new DocumentPropertyChangedLabelArgs(target, command));
        }

        public EditCommand CreateEditCommand_CustomUniformLabel(string id, string label)
        {
            var targets = new GuiObjectGroup(Targets.GetObjects(GuiObjectID.Material).OfType<Material>()
                .Where(x => x.MaterialShaderAssign.ShaderParams.Any(y => y.id == id)));
            return CreateEditCommand_CustomLabel(targets, id, label, x => x.CustomUI.uniform_vars, (target, command) => new DocumentPropertyChangedLabelArgs(target, command));
        }

        public EditCommand CreateEditCommand_CustomRenderInfoLabel(string id, string label)
        {
            var targets = new GuiObjectGroup(Targets.GetObjects(GuiObjectID.Material).OfType<Material>()
                .Where(x => x.MaterialShaderAssign.RenderInfos.Any(y => y.name == id)));
            return CreateEditCommand_CustomLabel(targets, id, label, x => x.CustomUI.render_info_slots, (target, command)=>new DocumentPropertyChangedLabelArgs(target, command));
        }

        public EditCommand CreateEditCommand_CustomGroupLabel(string id, string label)
        {
            var targets = new GuiObjectGroup(Targets.GetObjects(GuiObjectID.Material).OfType<Material>()
                .Where(x => x.MaterialShaderAssign.GetGroups(false).Any(y => y.groupName == id)));
            return CreateEditCommand_CustomLabel(targets, id, label, x => x.CustomUI.groups, (target, command)=>new DocumentPropertyChangedLabelArgs(target, command));
        }

        public static GroupEditCommand CreateEditCommand_CustomLabel(
            GuiObjectGroup targets,
            string id,
            string label,
            Func<Material, Dictionary<string, string>> selectDictionary,
            GeneralGroupReferenceEditCommand<string>.CreateEventArgsDelegate createEventArgs = null)
        {
            if (label == string.Empty)
            {
                label = null;
            }

            return new GeneralGroupReferenceEditCommand<string>(
                targets,
                GuiObjectID.Material,
                targets.Objects.OfType<Material>().Select(x => label),
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    var material = (Material)target;
                    var dict = selectDictionary(material);
                    string old;
                    dict.TryGetValue(id, out old);
                    swap = old;
                    var value = (string)data;
                    if (value != null)
                    {
                        dict[id] = value;
                    }
                    else
                    {
                        dict.Remove(id);
                    }
                },
                createEventArgsDelegate: createEventArgs,
                postEditDelegate: delegate
                {
                    App.AppContext.OnMaterialReferenceBehaviorSettingChanged();
                });
        }

        private void chkHideConditionParam_CheckedChanged(object sender, EventArgs e)
        {
            ShowConditionVariable = !chkHideConditionParam.Checked;
            ConfigData.ApplicationConfig.Setting.PropertyEdit.ShowConditionVariable = ShowConditionVariable;
            //UpdateScgGroup();
            UpdateControls = true;
            ((MaterialPropertyPanel)Owner).UpdateCategoryView();
            UpdateForm(false, false);
        }

        private void chkShowParamId_CheckedChanged(object sender, EventArgs e)
        {
            ShowParamId = chkShowParamId.Checked;
            ConfigData.ApplicationConfig.Setting.PropertyEdit.ShowShaderParamId = ShowParamId;
            BeginInvoke((Action)(() =>
                {
                    using (new UIControlEventSuppressBlock())
                    {
                        // コントロールの値を更新
                        if (scgGroup.UpdateValues(ActiveTarget, (MaterialPropertyPanel)Owner, ShowParamId, ShowOriginalLabel))
                        {
                            // サイズが変更された場合はレイアウトを更新
                            scgGroup.UpdateLayout(true);

                            // 検索窓を非表示にする。
                            // findBar と finder は表示切替時のイベントで連動しているためイベント呼び出しが必要だが、
                            // UIControlEventSuppressBlock 内なので findBar.Visible = false ではイベントを呼び出せない。
                            // UIControlEventSuppressBlock を回避する InvokeMethodWithForceEventRaising() で処理する。
                            findBar.InvokeMethodWithForceEventRaising(findBarVisibleProperty.GetSetMethod(), new object[] { false });
                        }
                    }
                }));
        }


        private void chkOriginalLabel_CheckedChanged(object sender, EventArgs e)
        {
            ShowOriginalLabel = chkOriginalLabel.Checked;
            ConfigData.ApplicationConfig.Setting.PropertyEdit.ShowOriginalLabelId = ShowOriginalLabel;
            BeginInvoke((Action)(() =>
                {
                    using (new UIControlEventSuppressBlock())
                    {
                        // コントロールの値を更新
                        if (scgGroup.UpdateValues(ActiveTarget, (MaterialPropertyPanel)Owner, ShowParamId, ShowOriginalLabel))
                        {
                            // サイズが変更された場合はレイアウトを更新
                            scgGroup.UpdateLayout(true);

                            // 検索窓を非表示にする。
                            // findBar と finder は表示切替時のイベントで連動しているためイベント呼び出しが必要だが、
                            // UIControlEventSuppressBlock 内なので findBar.Visible = false ではイベントを呼び出せない。
                            // UIControlEventSuppressBlock を回避する InvokeMethodWithForceEventRaising() で処理する。
                            findBar.InvokeMethodWithForceEventRaising(findBarVisibleProperty.GetSetMethod(), new object[] { false });
                        }
                    }
                }));
        }

        private void cbxShadingModel_CheckedChanged(object sender, EventArgs e)
        {
            SucceedVariable = cbxShadingModel.Checked;
            ConfigData.ApplicationConfig.Setting.PropertyEdit.SucceedVariable = SucceedVariable;
        }

        private void chkShowForceReferParameter_CheckedChanged(object sender, EventArgs e)
        {
            ShowForceReferParameter = chkShowForceReferParameter.Checked;
            ConfigData.ApplicationConfig.Setting.PropertyEdit.ShowForceReferParameter = ShowForceReferParameter;
            UpdateControls = true;
            ((MaterialPropertyPanel)Owner)?.UpdateCategoryView();
            UpdateForm(false, false);
        }

        static ShaderParamControlGroup.GroupCopyData ItemsCopyData = null;
        static public Dictionary<string, ShaderParamControlGroup.GroupCopyData> pageCopyData = new Dictionary<string, ShaderParamControlGroup.GroupCopyData>();

        private void btnCopy_Click(object sender, EventArgs e)
        {
            if (ActiveTarget.MaterialShaderAssign.PageMode && !MaterialShaderPage.IgnorePage)
            {
                pageCopyData[pageId ?? ""] = ShaderParamControlGroup.CreatePageCopyData(ActiveTarget, true, pageId, true, true);
            }
            else
            {
                ItemsCopyData = ShaderParamControlGroup.CreatePageCopyData(ActiveTarget, false, null, true, true);
            }

            ObjectPropertyDialog.ForEachActivePanel(page =>
            {
                if (page is MaterialShaderPage)
                {
                    ((MaterialShaderPage)page).UpdateItemsCopyPasteButton();
                }
            });
        }

        private void btnPaste_Click(object sender, EventArgs e)
        {
            EditCommandSet commandSet = null;
            var pageMode = ActiveTarget.MaterialShaderAssign.PageMode && !MaterialShaderPage.IgnorePage;
            ShaderParamControlGroup.GroupCopyData copyData = null;
            if (pageMode)
            {
                pageCopyData.TryGetValue(pageId ?? "", out copyData);
            }
            else
            {
                copyData = ItemsCopyData;
            }

            commandSet = ShaderParamControlGroup.CreateItemsPasteCommand(Targets, copyData, null, pageMode, (pageMode ? pageId: null), true, false, sendViewer:false, checkVisible:true);

            if (commandSet != null)
            {
                commandSet.SetViewerDrawSuppressBlockDelegate(Viewer.ViewerDrawSuppressBlock.DiscardAllMessages);
                var commandSet2 = new EditCommandSet();
                commandSet2.Add(commandSet);
                var models = Targets.Objects.OfType<Material>().SelectMany(x => x.Referrers).Distinct().ToArray();
                commandSet2.OnPostEdit += (s, e1) =>
                    {
                        foreach (var model in models)
                        {
                            Viewer.LoadOrReloadModel.Send(model);
                        }
                    };

                TheApp.CommandManager.Execute(commandSet2);
            }
        }

        private void btnDefault_Click(object sender, EventArgs e)
        {
            if (ActiveTarget.MaterialShaderAssign.PageMode && !MaterialShaderPage.IgnorePage)
            {
                var command = ShaderParamControlGroup.CreateGroupSetToDefaultCommand(Targets, null, true, pageId);
                if (command != null)
                {
                    TheApp.CommandManager.Execute(command);
                }
            }
            else
            {
                SetDefaultShaderAssign();
            }
        }

        private void scgGroup_EditGroupLabel(object sender, EventArgs e)
        {
            var groupName = (string)scgGroup.lastGroupBox.Tag;
            var group = ActiveTarget.MaterialShaderAssign.ShadingModel.Groups()
                .FirstOrDefault(x => x.name == groupName);
            string customLabel;
            ActiveTarget.CustomUI.groups.TryGetValue(groupName, out customLabel);
            using (var dialog = new EditLabelDialog(groupName, group != null ? group.Label() : null, customLabel, true))
            {
                var result = dialog.ShowDialog();
                if (result == DialogResult.OK)
                {
                    TheApp.CommandManager.Execute(CreateEditCommand_CustomGroupLabel(groupName, dialog.Value));
                }
            }
        }

        private void scgGroup_LinkClicked(object sender, ShaderParamControls.LinkClickedEventArgs e)
        {
            switch (e.ParamType)
            {
                case ParamType.page:
                    SetScroll(null, null, e.ParamType, e.Target);
                    break;
                case ParamType.group:
                    SetScroll(e.Target, null, e.ParamType, null);
                    break;
                default:
                    SetScroll(e.Group, e.Target, e.ParamType);
                    break;
            }
        }

        #region Filter
        public static Regex IncludeLabelFilter = new Regex("");
        public static Regex ExcludeLabelFilter = null;
        public static string includeString = "";
        public static string excludeString = "";
        public static bool IgnorePage = false;

        // 結果のキャッシュ
        public static Dictionary<string, bool> isMatch = new Dictionary<string, bool>();

        private static string ConvertToRegex(string text)
        {
            var terms = from splitted in (text ?? string.Empty).Split(new [] {','}, StringSplitOptions.RemoveEmptyEntries)
                        let trimmed = splitted.TrimStart(null) // 先頭の空白を削除
                        where trimmed.Length > 0
                        let escaped = Regex.Escape(trimmed)
                        let regex = escaped.Replace(@"\*", ".*").Replace(@"\?", ".")
                        select "("+regex+")";
            return string.Join("|", terms.ToArray());
        }

        public static void SetFilter(string include, string exclude, bool ignorePage)
        {
            if (include == includeString && exclude == excludeString && ignorePage == IgnorePage)
            {
                return;
            }

            IgnorePage = ignorePage;

            // キャッシュをクリア
            isMatch.Clear();
            includeString = include;
            excludeString = exclude;

            IncludeLabelFilter = new Regex(ConvertToRegex(include));
            var excludeRegex = ConvertToRegex(exclude);
            if (excludeRegex.Length > 0)
            {
                ExcludeLabelFilter = new Regex(excludeRegex);
            }
            else
            {
                ExcludeLabelFilter = null;
            }
        }

        public static bool IsMatch(string name)
        {
            bool result;
            if (isMatch.TryGetValue(name, out result))
            {
                return result;
            }

            return isMatch[name] = IsMatchFilter(name);
        }

        public static bool IsMatchFilter(string name)
        {
            return IncludeLabelFilter.IsMatch(name) && (ExcludeLabelFilter == null || !ExcludeLabelFilter.IsMatch(name));
        }

        #endregion

        private void scgGroup_ColorControlChanged(object sender, ColorControlChangedEventArgs e)
        {
            TheApp.CommandManager.Execute(CreateEditCommand_ColorControl(
                Targets,
                e.ParamName,
                e.ShowColor));
        }

        private static GroupEditCommand CreateEditCommand_ColorControl(
            GuiObjectGroup targets,
            string id,
            bool show)
        {
            return new GeneralGroupValueEditCommand<bool>(
                targets,
                GuiObjectID.Material,
                show,
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    var material = (Material)target;
                    bool old = material.CustomUI.colorControls.Contains(id);
                    swap = old;
                    var value = (bool)data;
                    if (value)
                    {
                        material.CustomUI.colorControls.Add(id);
                    }
                    else
                    {
                        material.CustomUI.colorControls.Remove(id);
                    }
                },
                createEventArgsDelegate: (target, command)=>{
                    return new DocumentShaderPropertyChangedArgs(target, command);
                });
        }
    }
}
