﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using App.Command;
using App.Controls;
using App.Data;
using App.PropertyEdit;
using App.res;
using App.Utility;
using ConfigCommon;
using nw.g3d.iflib.nw3de;

namespace App.PropertyEdit
{
    using App.FileView;
    using App.Properties;

    using nw.g3d.nw4f_3dif;

    public partial class MaterialParentsPage : MaterialPropertyPage
    {
        private static readonly string[] ReferenceBehaviorTypes = {
            res.Strings.MaterialParentsPage_ShaderParameter,
            res.Strings.MaterialParentsPage_RenderInfo,
            res.Strings.MaterialParentsPage_SamplerAssign,
            res.Strings.MaterialParentsPage_Option,
            res.Strings.MaterialParentsPage_AttribAssign,
        };

        private static readonly string[] ShaderItemValueStateText =
        {
            Strings.MaterialParentsPage_ShaderItemValueStateText_Override,
            Strings.MaterialParentsPage_ShaderItemValueStateText_Refer,
            Strings.MaterialParentsPage_ShaderItemValueStateText_Default,
        };

        private static readonly string[] ShaderItemRestrictionStateText =
        {
            Resources.MaterialParentsPage_ShaderItemStateRestrictionText_None,
            Resources.MaterialParentsPage_ShaderItemStateRestrictionText_ForceRefer,
            Resources.MaterialParentsPage_ShaderItemStateRestrictionText_ForceOverride,
        };

        public MaterialParentsPage() :
            base(PropertyPageID.MaterialBases)
        {
            InitializeComponent();
            Debug.Assert(ShaderItemValueStateText.Length    == Enum.GetValues(typeof(ShaderItemValueState)).Length);
            Debug.Assert(ShaderItemRestrictionStateText.Length == Enum.GetValues(typeof(ShaderItemRestrictionState)).Length);
        }

        public static ObjectPropertyPage CreateInstance(object arg)
        {
            return new MaterialParentsPage();
        }

        public static bool IsModified(Material material)
        {
            return material != null &&
                (material.IsParentMaterialModified() || material.IsReferenceBehaviorsModified());
        }

        public static bool IsValid(Material material)
        {
            return material != null && material.ParentMaterialsValidity();
        }

        protected override void InitializeFormInternal()
        {
            //base.InitializeFormInternal();
        }

        internal IntermediateFileDocument ActiveDocument
        {
            get { return Targets.Active as IntermediateFileDocument; }
        }

        protected override void UpdateFormInternal(UpdateFormInfo updateFormInfo)
        {
            //親マテリアル
            uiModifiedMarkLabelParentMaterials.IsModified = ActiveTarget.IsParentMaterialModified();
            var selectIndex = -1;
            if (lvwParentMaterials.SelectedIndices.Count > 0)
            {
                selectIndex = lvwParentMaterials.SelectedIndices[0];
            }

            lvwParentMaterials.BeginUpdate();
            {
                var parentMaterials = ActiveTarget.GetParentMaterials();
                var parentMaterialsInfo = ActiveTarget.MaterialReference?.ParentMaterials;
                if (parentMaterialsInfo != null)
                {
                    var errorFont = new Font(DefaultFont, FontStyle.Italic);
                    lvwParentMaterials.SetItemCount(parentMaterialsInfo.Count);
                    for (var i = 0; i < parentMaterialsInfo.Count; i++)
                    {
                        var item = lvwParentMaterials.Items[i];
                        var materialName = parentMaterialsInfo[i].MaterialName;
                        var modelPath = parentMaterialsInfo[i].Path;
                        var modelFileName = Path.GetFileName(modelPath);
                        var parentMaterial =
                            !string.IsNullOrEmpty(modelPath) ? parentMaterials.FirstOrDefault(x => (x != null) && (x.OwnerDocument?.FileName == modelFileName) && (x.Name == materialName)) :
                            parentMaterials.FirstOrDefault(x => (x?.OwnerDocument == ActiveTarget.OwnerDocument) && (x?.Name == materialName));
                        item.ForeColor = parentMaterial != null ? ForeColor : Color.Red;
                        item.Font = parentMaterial != null ? DefaultFont : errorFont;
                        // SIGLO-73403 での暫定対応
                        // ラベルがあれば「ラベル(マテリアル名)」、なければ「コメント(マテリアル名)」、コメントもなければ「マテリアル名」
                        item.SubItems[clhMaterialName.Index].Text =
                            !string.IsNullOrEmpty(parentMaterial?.Label) ? $"{parentMaterial.Label} ({materialName})" :
                            !string.IsNullOrEmpty(parentMaterial?.Comment) ? $"{parentMaterial.Comment} ({materialName})" :
                            materialName;
                        // SIGLO-73403 での暫定対応
                        // ラベルがあれば「ラベル(モデル名)」、なければ「コメント(モデル名)」、コメントもなければ「モデル名」
                        item.SubItems[clhModelPath.Index].Text =
                            string.IsNullOrEmpty(modelPath) ? string.Empty :
                            !string.IsNullOrEmpty(parentMaterial?.OwnerDocument?.Label) ? $"{parentMaterial.OwnerDocument?.Label} ({modelPath})" :
                            !string.IsNullOrEmpty(parentMaterial?.OwnerDocument?.Comment) ? $"{parentMaterial.OwnerDocument?.Comment} ({modelPath})" :
                            modelPath;
                    }
                }
                else
                {
                    lvwParentMaterials.SetItemCount(0);
                }
            }
            lvwParentMaterials.EndUpdate();

            if (lvwParentMaterials.Items.Count > 0)
            {
                if ((selectIndex != -1) && (selectIndex < lvwParentMaterials.Items.Count))
                {
                    lvwParentMaterials.Items[selectIndex].Selected = true;
                    lvwParentMaterials.Items[selectIndex].Focused = true;
                }
                else
                {
                    btnParentMaterialsAdd.Select();
                }
            }
            else
            {
                lvwParentMaterials_SelectionChanged(null, null);
            }

            UpdateParentMaterialsState();

            // マテリアル参照
            uiModifiedMarkLabelMaterialReferenceBehaviors.IsModified = ActiveTarget.IsReferenceBehaviorsModified();
            selectIndex = -1;
            if (lvwMaterialReferenceBehaviors.SelectedIndices.Count > 0)
            {
                selectIndex = lvwMaterialReferenceBehaviors.SelectedIndices[0];
            }
            var materialReference = ActiveTarget.MaterialReference;
            lvwMaterialReferenceBehaviors.BeginUpdate();
            {
                if (materialReference == null)
                {
                    lvwMaterialReferenceBehaviors.SetItemCount(0);
                }
                else
                {
                    List<MaterialReferenceBehaviorItem>[] behaviorItemsList =
                    {
                        materialReference.ShaderParamBehaviors,
                        materialReference.RenderInfoBehaviors,
                        materialReference.SamplerAssignBehaviors,
                        materialReference.ShaderOptionBehaviors,
                        materialReference.AttribAssignBehaviors,
                    };

                    Debug.Assert(behaviorItemsList.Length == ReferenceBehaviorTypes.Length);

                    var totalCount = behaviorItemsList.Sum(x => x?.Count ?? 0);
                    totalCount += materialReference.SamplerBehaviors.SamplerBehaviorItems.Sum(x => x?.SamplerParamBehaviors?.Count ?? 0);
                    totalCount += materialReference.SamplerBehaviors.SamplerBehaviorItems.Count(x => x.IsRequiredForChild);
                    lvwMaterialReferenceBehaviors.SetItemCount(totalCount);
                    if (totalCount != 0)
                    {
                        Dictionary<string, string>[] customUIOptions;
                        Dictionary<string, string>[] customUISamplers;
                        Dictionary<string, string>[] customUIUniforms;
                        Dictionary<string, string>[] customUIAttribs;
                        Dictionary<string, string>[] customUIRenderInfos;
                        Dictionary<string, string>[] customUIGroups;
                        ShaderParamControls.ShaderParamControlGroup.PrepareResolveCustomLabel(ActiveTarget, out customUIOptions, out customUISamplers, out customUIUniforms, out customUIAttribs, out customUIRenderInfos, out customUIGroups);

                        var line = 0;
                        var items = lvwMaterialReferenceBehaviors.Items;
                        if (materialReference.ShaderParamBehaviors != null)
                        {
                            foreach (var shaderItem in materialReference.ShaderParamBehaviors)
                            {
                                var group = ShaderParamControls.ShaderParamControlGroup.GetGroupName(ActiveTarget, shaderItem.Id);

                                SetSubItem(
                                    items[line],
                                GetShaderParamLabel(customUIUniforms, shaderItem.Id),
                                    Strings.MaterialParentsPage_ShaderParameter,
                                    shaderItem,
                                    group);
                                line++;
                            }
                        }
                        if (materialReference.RenderInfoBehaviors != null)
                        {
                            foreach (var shaderItem in materialReference.RenderInfoBehaviors)
                            {
                                var group = ShaderParamControls.ShaderParamControlGroup.GetGroupName(ActiveTarget, shaderItem.Id);

                                SetSubItem(
                                    items[line],
                                GetRenderInfoLabel(customUIRenderInfos, shaderItem.Id),
                                    Strings.MaterialParentsPage_RenderInfo,
                                    shaderItem,
                                    group);
                                line++;
                            }
                        }
                        if (materialReference.SamplerAssignBehaviors != null)
                        {
                            foreach (var shaderItem in materialReference.SamplerAssignBehaviors)
                            {
                                var group = ShaderParamControls.ShaderParamControlGroup.GetGroupName(ActiveTarget, shaderItem.Id);

                                SetSubItem(
                                    items[line],
                                GetSamplerAssignLabel(customUISamplers, shaderItem.Id),
                                    Strings.MaterialParentsPage_SamplerAssign,
                                    shaderItem,
                                    group);
                                line++;
                            }
                        }
                        if (materialReference.ShaderOptionBehaviors != null)
                        {
                            foreach (var shaderItem in materialReference.ShaderOptionBehaviors)
                            {
                                var group = ShaderParamControls.ShaderParamControlGroup.GetGroupName(ActiveTarget, shaderItem.Id);

                                SetSubItem(
                                    items[line],
                                GetShaderOptionLabel(customUIOptions, shaderItem.Id),
                                    Strings.MaterialParentsPage_Option,
                                    shaderItem,
                                    group);
                                line++;
                            }
                        }
                        if (materialReference.AttribAssignBehaviors != null)
                        {
                            foreach (var shaderItem in materialReference.AttribAssignBehaviors)
                            {
                                var group = ShaderParamControls.ShaderParamControlGroup.GetGroupName(ActiveTarget, shaderItem.Id);

                                SetSubItem(
                                    items[line],
                                GetAttribAssigLabel(customUIAttribs, shaderItem.Id),
                                    Strings.MaterialParentsPage_AttribAssign,
                                    shaderItem,
                                    group);
                                line++;
                            }
                        }
                        if (materialReference.SamplerBehaviors?.SamplerBehaviorItems != null)
                        {
                            foreach (var samplerBehaviorItem in materialReference.SamplerBehaviors.SamplerBehaviorItems)
                            {
                                if (samplerBehaviorItem.IsRequiredForChild)
                                {
                                    items[line].SubItems[clhType.Index].Text =
                                        string.Format(Strings.MaterialParentsPage_Sampler, samplerBehaviorItem.Id);
                                    items[line].SubItems[clhId.Index].Text = string.Empty;
                                    items[line].SubItems[clhValue.Index].Text = string.Empty;
                                    items[line].SubItems[clhRestriction.Index].Text =
                                        Strings.MaterialParentsPage_UpdateFormInternal_IsRequiredForChild;
                                    line++;
                                }

                                foreach (var paramBehavior in samplerBehaviorItem.SamplerParamBehaviors)
                                {
                                    var group = ShaderParamControls.ShaderParamControlGroup.GetGroupName(ActiveTarget, samplerBehaviorItem.Id);

                                    SetSubItem(
                                        items[line],
                                    paramBehavior.Id,
                                        string.Format(Strings.MaterialParentsPage_Sampler, samplerBehaviorItem.Id),
                                        paramBehavior,
                                        group);
                                    line++;
                                }
                            }
                        }
                    }
                }
            }
            lvwMaterialReferenceBehaviors.AutoResizeColumnsByHeaderAndContents();
            lvwMaterialReferenceBehaviors.EndUpdate();
        }

        private string GetShaderParamLabel(Dictionary<string, string>[] customUIUniforms, string id)
        {
            var label = ActiveTarget?.MaterialShaderAssign?.ShadingModel?.MaterialUniforms().FirstOrDefault(x => x.id == id)?.Label();
            bool isParentLabel;
            var custlabel = ShaderParamControls.ShaderParamControlGroup.ResolveCustomLabel(customUIUniforms, id, ActiveTarget?.CustomUI.uniform_vars, out isParentLabel);
            label = string.IsNullOrEmpty(custlabel) ? label : custlabel;
            return string.IsNullOrEmpty(label) ? id : $"{label} ({id})";
        }

        private string GetRenderInfoLabel(Dictionary<string, string>[] customUIRenderInfos, string name)
        {
            var label = ActiveTarget?.MaterialShaderAssign?.ShadingModel?.RenderInfoSlots().FirstOrDefault(x => x.name == name)?.Label();
            bool isParentLabel;
            var custlabel = ShaderParamControls.ShaderParamControlGroup.ResolveCustomLabel(customUIRenderInfos, name, ActiveTarget?.CustomUI.render_info_slots, out isParentLabel);
            label = string.IsNullOrEmpty(custlabel) ? label : custlabel;
            return string.IsNullOrEmpty(label) ? name : $"{label} ({name})";
        }

        private string GetSamplerAssignLabel(Dictionary<string, string>[] customUISamplers, string id)
        {
            var label = ActiveTarget?.MaterialShaderAssign?.ShadingModel?.Samplers().FirstOrDefault(x => x.id == id)?.Label();
            bool isParentLabel;
            var custlabel = ShaderParamControls.ShaderParamControlGroup.ResolveCustomLabel(customUISamplers, id, ActiveTarget?.CustomUI.sampler_vars, out isParentLabel);
            label = string.IsNullOrEmpty(custlabel) ? label : custlabel;
            return string.IsNullOrEmpty(label) ? id : $"{label} ({id})";
        }

        private string GetShaderOptionLabel(Dictionary<string, string>[] customUIOptions, string id)
        {
            var label = ActiveTarget?.MaterialShaderAssign?.ShadingModel?.Options().FirstOrDefault(x => x.id == id)?.Label();
            bool isParentLabel;
            var custlabel = ShaderParamControls.ShaderParamControlGroup.ResolveCustomLabel(customUIOptions, id, ActiveTarget?.CustomUI.option_vars, out isParentLabel);
            label = string.IsNullOrEmpty(custlabel) ? label : custlabel;
            return string.IsNullOrEmpty(label) ? id : $"{label} ({id})";
        }

        private string GetAttribAssigLabel(Dictionary<string, string>[] customUIAttribs, string id)
        {
            var label = ActiveTarget?.MaterialShaderAssign?.ShadingModel?.Attributes().FirstOrDefault(x => x.id == id)?.Label();
            bool isParentLabel;
            var custlabel = ShaderParamControls.ShaderParamControlGroup.ResolveCustomLabel(customUIAttribs, id, ActiveTarget?.CustomUI.attrib_vars, out isParentLabel);
            label = string.IsNullOrEmpty(custlabel) ? label : custlabel;
            return string.IsNullOrEmpty(label) ? id : $"{label} ({id})";
        }

        private void SetSubItem(ListViewItem item, string id, string referenceBehaviorType, MaterialReferenceBehaviorItem shaderItem, string group)
        {
            item.SubItems[clhType.Index].Text = referenceBehaviorType;
            item.SubItems[clhId.Index].Text = id;
            item.SubItems[clhValue.Index].Text = ShaderItemValueStateText[(int)shaderItem.Value];
            item.SubItems[clhRestriction.Index].Text = ShaderItemRestrictionStateText[(int)shaderItem.ChildRestriction];
            item.SubItems[clhGroup.Index].Text = group;
        }

        #region コピー＆ペースト
        /// <summary>
        /// コピーが可能か。
        /// </summary>
        public override bool CanCopy()
        {
            return true;
        }

        private class CopyData
        {
            public nw3de_MaterialReference MaterialReference { get; set; }
        }

        public override object Copy(ref object copyObjectInfo)
        {
            return Copy(ActiveTarget);
        }

        public static object Copy(GuiObject target)
        {
            var material = (Material) target;

            return new CopyData()
            {
                MaterialReference = ObjectUtility.Clone(material.MaterialReference)
            };
        }

        public override bool CanPaste(object copiedObjectInfo, object copiedObject)
        {
            return base.CanPaste(copiedObjectInfo, copiedObject);
        }

        public override void Paste(object pasteObject)
        {
            TheApp.CommandManager.Add(Paste(Targets, pasteObject));
        }

        public static ICommand Paste(GuiObjectGroup targets, object pasteObject)
        {
            var commandSet = new EditCommandSet();
            {
                var copyData = (CopyData) pasteObject;
                commandSet.Add(Material.CreateEditCommand_MaterialReference(targets, copyData.MaterialReference));
            }

            return commandSet.Execute();
        }

        #endregion

        #region イベント

        private void lvwParentMaterials_Click(object sender, EventArgs e)
        {
            UpdateParentMaterialsState();
        }

        private void lvwParentMaterials_DoubleClick(object sender, EventArgs e)
        {
            UpdateParentMaterialsState();
        }

        private void lvwParentMaterials_SelectionChanged(object sender, EventArgs e)
        {
            UpdateParentMaterialsState();
        }

        private void UpdateParentMaterialsState()
        {
            var selectedIndex = lvwParentMaterials.SelectedIndex;
            var isSelected = selectedIndex != -1;

            //btnParentMaterialsAdd.Enabled = !string.IsNullOrEmpty(ActiveTarget.Owner.FilePath);
            btnParentMaterialsEdit.Enabled = isSelected && ActiveTarget.ParentMaterialsValidity();
            btnParentMaterialsDelete.Enabled = isSelected;
            btnParentMaterialsUp.Enabled = isSelected && (selectedIndex > 0);
            btnParentMaterialsDown.Enabled = isSelected && (selectedIndex >= 0) && (selectedIndex <= lvwParentMaterials.Items.Count - 2);
        }

        private void btnParentMaterialsUp_Click(object sender, EventArgs e)
        {
            if (lvwParentMaterials.SelectedItems.Count == 0)
            {
                return;
            }
            Debug.Assert(lvwParentMaterials.SelectedItems.Count == 1);

            // リストビューの変更
            ListViewItem item = lvwParentMaterials.SelectedItems[0];
            int index = lvwParentMaterials.Items.IndexOf(item);
            if (index == 0)
            {
                return;
            }

            // コマンド発行
            var info = GetMaterialReference();
            var tmp = info.ParentMaterials[index - 1];
            info.ParentMaterials[index - 1] = info.ParentMaterials[index];
            info.ParentMaterials[index] = tmp;
            TheApp.CommandManager.Execute(Material.CreateEditCommand_MaterialReference(Targets, info));
            lvwParentMaterials.Items[index - 1].Selected = true;
            btnParentMaterialsUp.Focus();
        }

        private void btnParentMaterialsDown_Click(object sender, EventArgs e)
        {
            if (lvwParentMaterials.SelectedItems.Count == 0)
            {
                return;
            }
            Debug.Assert(lvwParentMaterials.SelectedItems.Count == 1);

            // リストビューの変更
            ListViewItem item = lvwParentMaterials.SelectedItems[0];
            int index = lvwParentMaterials.Items.IndexOf(item);
            if ((index + 1) == lvwParentMaterials.Items.Count)
            {
                return;
            }

            // コマンド発行
            var info = GetMaterialReference();
            var tmp = info.ParentMaterials[index + 1];
            info.ParentMaterials[index + 1] = info.ParentMaterials[index];
            info.ParentMaterials[index] = tmp;
            TheApp.CommandManager.Execute(Material.CreateEditCommand_MaterialReference(Targets, info));
            lvwParentMaterials.Items[index + 1].Selected = true;
            btnParentMaterialsDown.Focus();
        }

        private void btnParentMaterialsAdd_Click(object sender, EventArgs e)
        {
            if (string.IsNullOrEmpty(ActiveTarget.OwnerDocument.FilePath))
            {
                UIMessageBox.Warning(Resources.MaterialParentsPage_CannotAddParentBeforeFilePathFix);
                return;
            }

            using (var dialog = new MaterialParentDialog(ActiveTarget))
            {
                dialog.Text = Resources.MaterialParentsPage_btnParentMaterialsAdd_Click_AddParentMaterial;
                var result = dialog.ShowDialog(Owner.Owner);
                var materialInfo = dialog.ParentMaterialInfo;
                if (result == DialogResult.OK && materialInfo != null && materialInfo.HasMaterial())
                {
                    var commandSet = new EditCommandSet();
                    string modelPath;

                    if (!GetModelPath(materialInfo, commandSet, out modelPath))
                    {
                        return;
                    }

                    int index;
                    if (lvwParentMaterials.SelectedIndices.Count == 0)
                    {
                        index = lvwParentMaterials.Items.Count;
                    }
                    else
                    {
                        Debug.Assert(lvwParentMaterials.SelectedIndices.Count == 1);
                        index = lvwParentMaterials.SelectedIndices[0];
                    }
                    ++index;
                    index = Math.Min(index, lvwParentMaterials.Items.Count);

                    // コマンド発行
                    var info = GetMaterialReference();
                    info.ParentMaterials.Insert(
                        index,
                        new ParentMaterial() { MaterialName = materialInfo.MaterialName, Path = modelPath });
                    commandSet.Add(Material.CreateEditCommand_MaterialReference(Targets, info).Execute());
                    commandSet.Reverse();
                    TheApp.CommandManager.Add(commandSet);
                    lvwParentMaterials.Items[index].Selected = true;
                }
            }
        }

        private void btnParentMaterialsEdit_Click(object sender, EventArgs e)
        {
            if (lvwParentMaterials.SelectedItems.Count == 0)
            {
                return;
            }
            Debug.Assert(lvwParentMaterials.SelectedItems.Count == 1);

            // リストビューの変更
            ListViewItem item = lvwParentMaterials.SelectedItems[0];
            int index = lvwParentMaterials.Items.IndexOf(item);
            if (index < 0)
            {
                return;
            }

            // 選択中の親マテリアル
            if (!ActiveTarget.ParentMaterialsValidity())
            {
                return;
            }

            using (var dialog = new MaterialParentDialog(ActiveTarget, ActiveTarget.ParentMaterials[index]))
            {
                dialog.Text = Resources.MaterialParentsPage_btnParentMaterialsEdit_Click_EditParentMaterial;
                var result = dialog.ShowDialog(Owner.Owner);
                var materialInfo = dialog.ParentMaterialInfo;
                if (result == DialogResult.OK && materialInfo != null && materialInfo.HasMaterial())
                {
                    var commandSet = new EditCommandSet();
                    string modelPath;

                    if (!GetModelPath(materialInfo, commandSet, out modelPath))
                    {
                        return;
                    }

                    // コマンド発行
                    var info = GetMaterialReference();
                    info.ParentMaterials[index] = new ParentMaterial() {MaterialName = materialInfo.MaterialName, Path = modelPath };
                    commandSet.Add(Material.CreateEditCommand_MaterialReference(Targets, info).Execute());
                    commandSet.Reverse();
                    TheApp.CommandManager.Add(commandSet);
                    lvwParentMaterials.Items[index].Selected = true;
                }
            }
        }

        private bool GetModelPath(MaterialParentDialog.MaterialInfo materialInfo, EditCommandSet commandSet, out string modelPath)
        {
            modelPath = string.Empty;
            if (!materialInfo.IsLoaded)
            {
                if (materialInfo.IsSeparateMaterial)
                {
                    if (DocumentManager.SeparateMaterials.All(
                        x => !materialInfo.FullPath.Equals(x.FilePath, StringComparison.OrdinalIgnoreCase)))
                    {
                        //load
                        DocumentPropertyChangedEventArgs args;
                        using (var vdsb = new Viewer.ViewerDrawSuppressBlock())
                        using (var reloadModelSuppressBlock = new Viewer.HioUtility.ReloadModelSuppressBlock())
                        using (var propertyChangedSuppressBlock = new App.AppContext.PropertyChangedSuppressBlock())
                        {
                            DocumentManager.LoadFromFile(
                                new[] { new DocumentManager.PathWithName(materialInfo.FullPath) },
                                commandSet: commandSet);
                            var material = DocumentManager.SeparateMaterials.FirstOrDefault(
                                x => x.FilePath.Equals(materialInfo.FullPath, StringComparison.OrdinalIgnoreCase));
                            if (commandSet.CommandCount == 0 ||
                                material == null)
                            {
                                return false;
                            }
                            args = new DocumentPropertyChangedEventArgs(new DocumentPropertyChangedShowInObjViewArgs(material));
                        }
                        AppContext.ExecutePropertyChangedEvent(this, args.GetArgs());
                    }
                }
                else
                {
                    if (DocumentManager.Models.All(
                        x => !materialInfo.FullPath.Equals(x.FilePath, StringComparison.OrdinalIgnoreCase)))
                    {
                        //load
                        DocumentPropertyChangedEventArgs args;
                        using (var vdsb = new Viewer.ViewerDrawSuppressBlock())
                        using (var reloadModelSuppressBlock = new Viewer.HioUtility.ReloadModelSuppressBlock())
                        using (var propertyChangedSuppressBlock = new App.AppContext.PropertyChangedSuppressBlock())
                        {
                            DocumentManager.LoadFromFile(
                                new[] { new DocumentManager.PathWithName(materialInfo.FullPath) },
                                commandSet: commandSet);
                            var model = DocumentManager.Models.FirstOrDefault(
                                x => x.FilePath.Equals(materialInfo.FullPath, StringComparison.OrdinalIgnoreCase));
                            if (commandSet.CommandCount == 0 ||
                                model == null)
                            {
                                return false;
                            }
                            model.PreviewInfo.Visible = false;
                            model.PreviewInfo.ShowInObjView = false;
                            args = new DocumentPropertyChangedEventArgs(new DocumentPropertyChangedShowInObjViewArgs(model));
                        }
                        AppContext.ExecutePropertyChangedEvent(this, args.GetArgs());
                    }
                }
            }

            if (!ActiveTarget.OwnerDocument.FilePath.Equals(materialInfo.FullPath, StringComparison.OrdinalIgnoreCase))
            {
                if (!string.IsNullOrEmpty(materialInfo.RelativePath))
                {
                    modelPath = materialInfo.RelativePath;
                }
                else if (string.IsNullOrEmpty(ActiveTarget.OwnerDocument.FileLocation))
                {
                    modelPath = materialInfo.FullPath;
                }
                else
                {
                    modelPath = PathUtility.MakeRelativePath(ActiveTarget.OwnerDocument.FileLocation + "/", materialInfo.FullPath);
                }
            }
            return true;
        }

        private void btnParentMaterialsDelete_Click(object sender, EventArgs e)
        {
            if (lvwParentMaterials.SelectedItems.Count == 0)
            {
                return;
            }
            Debug.Assert(lvwParentMaterials.SelectedItems.Count == 1);

            // リストビューの変更
            ListViewItem item = lvwParentMaterials.SelectedItems[0];
            int index = lvwParentMaterials.Items.IndexOf(item);
            if (index < 0)
            {
                return;
            }

            // コマンド発行
            var info = GetMaterialReference();
            info.ParentMaterials.RemoveAt(index);
            TheApp.CommandManager.Execute(Material.CreateEditCommand_MaterialReference(Targets, info));
            btnParentMaterialsDelete.Focus();
        }

        private nw3de_MaterialReference GetMaterialReference()
        {
            var materialReference = ActiveTarget.MaterialReference == null ? new nw3de_MaterialReference() : ObjectUtility.Clone(ActiveTarget.MaterialReference);
            if (materialReference.ParentMaterials == null)
            {
                materialReference.ParentMaterials = new List<ParentMaterial>();
            }
            return materialReference;
        }

        #endregion
    }
}
