﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Windows.Forms;
using App.ConfigData;
using App.Controls;
using App.ObjectView.List;
using App.ObjectView.Schematic;
using App.Utility;
using App.res;

namespace App.ObjectView
{
    [ToolboxItem(true)]
    public partial class ObjectViewClient : UIUserControl
    {
        // レイアウトパネルテーブル
        private readonly Dictionary<AppConfig.ObjectViewLayout, LayoutPanel> _layoutPanels = new Dictionary<AppConfig.ObjectViewLayout, LayoutPanel>();
        // ビューコントロールテーブル
        private readonly Dictionary<AppConfig.ObjectViewMode, IObjectView> _viewControls = new Dictionary<AppConfig.ObjectViewMode, IObjectView>();

        public ObjectViewClient()
        {
            using (var watch = new DebugStopWatch("ObjectViewClient.ObjectViewClient() : A"))
            {
                InitializeComponent();

                ctlTitleBar.Text = Strings.ObjectViewClient_TitleBar;
            }

            // テーブル初期化
            using (var watch = new DebugStopWatch("ObjectViewClient.ObjectViewClient() : B"))
            {
                this.InitializeLayoutPanels();
            }

            // コンフィグ書き込み
            using (var watch = new DebugStopWatch("ObjectViewClient.ObjectViewClient() : C"))
            {
                HandleDestroyed += (s, e) =>
                {
                    ApplicationConfig.Setting.ObjectView.Layout = _viewLayout;
                };
            }

            // レイアウト初期化
            using (var watch = new DebugStopWatch("ObjectViewClient.ObjectViewClient() : D"))
            {
                ViewLayout = _viewLayout;
            }

            // イベント登録
            using (var watch = new DebugStopWatch("ObjectViewClient.ObjectViewClient() : E"))
            {
                App.AppContext.SelectedTargetChanged += GuiObjectManager_SelectedChanged;
            }

            // マテリアルリストを取得
            using (var watch = new DebugStopWatch("ObjectViewClient.ObjectViewClient() : F"))
            {
                GetViewControl(AppConfig.ObjectViewMode.MaterialList);
            }
        }

        public void InitializeLayoutPanels()
        {
            this._layoutPanels.Clear();
            this._viewControls.Clear();

            foreach (AppConfig.ObjectViewLayout viewLayout in Enum.GetValues(typeof(AppConfig.ObjectViewLayout)))
            {
                this._layoutPanels.Add(viewLayout, null);
            }
            foreach (AppConfig.ObjectViewMode viewMode in Enum.GetValues(typeof(AppConfig.ObjectViewMode)))
            {
                this._viewControls.Add(viewMode, null);
            }

            this._viewLayout = ApplicationConfig.Setting.ObjectView.Layout;
        }

        private AppConfig.ObjectViewLayout _viewLayout = AppConfig.ObjectViewLayout.Single;
        public AppConfig.ObjectViewLayout ViewLayout
        {
            get { return _viewLayout; }
            set
            {
                var prevPanel = _layoutPanels[_viewLayout];
                var nextPanel = _layoutPanels[value];

                if (nextPanel == null || prevPanel != nextPanel)
                {
                    using(var lockWindowUpdate = new LockWindowUpdate(this))
                    {
                        _viewLayout = value;

                        // レイアウトパネル作成
                        if (nextPanel == null)
                        {
                            nextPanel = GetLayoutPanel(value);
                        }

                        // ビューパネル設定
                        foreach (ObjectViewPanel viewPanel in nextPanel.ViewPanels)
                        {
                            viewPanel.View = GetViewControl(viewPanel.ViewMode);
                        }

                        nextPanel.SetFilterFromConfig();

                        // 変更後パネルを表示
                        nextPanel.Visible = true;
                        nextPanel.ViewPanels[0].View.FocusClient();

                        // 変更前パネルを非表示
                        if (prevPanel != null)
                        {
                            prevPanel.Visible = false;
                        }
                    }
                }
            }
        }

        /// <summary>
        /// 多画面レイアウトの分割比率をリセット。
        /// </summary>
        public void ResetMultiLayoutSplitRatio()
        {
            _layoutPanels[_viewLayout].ResetSplitRatio();
        }

        /// <summary>
        /// レイアウトパネルを取得。
        /// </summary>
        private LayoutPanel GetLayoutPanel(AppConfig.ObjectViewLayout layout)
        {
            // 未登録なら作成
            if (_layoutPanels[layout] == null)
            {
                using (var wait = new WaitCursor())
                {
                    LayoutPanel panel = null;
                    switch (layout)
                    {
                        case AppConfig.ObjectViewLayout.Single:
                            panel = new SingleLayoutPanel();
                            break;
                        case AppConfig.ObjectViewLayout.Multi2Stacked:
                            panel = new Multi2StackedLayoutPanel();
                            break;
                        case AppConfig.ObjectViewLayout.Multi2SideBySide:
                            panel = new Multi2SideBySideLayoutPanel();
                            break;
                        case AppConfig.ObjectViewLayout.Multi3SplitBottom:
                            panel = new Multi3SplitBottomLayoutPanel();
                            break;
                        case AppConfig.ObjectViewLayout.Multi3SplitTop:
                            panel = new Multi3SplitTopLayoutPanel();
                            break;
                        case AppConfig.ObjectViewLayout.Multi3SplitRight:
                            panel = new Multi3SplitRightLayoutPanel();
                            break;
                        case AppConfig.ObjectViewLayout.Multi3SplitLeft:
                            panel = new Multi3SplitLeftLayoutPanel();
                            break;
                        case AppConfig.ObjectViewLayout.Multi4:
                            panel = new Multi4LayoutPanel();
                            break;
                        default:
                            Debug.Assert(false);
                            break;
                    }

                    panel.Parent = pnlViewArea;
                    panel.Dock = DockStyle.Fill;
                    panel.ViewModeChanged += layoutPanel_ViewModeChanged;
                    _layoutPanels[layout] = panel;
                }
            }
            return _layoutPanels[layout];
        }

        /// <summary>
        /// ビューコントロールを取得。
        /// </summary>
        public IObjectView GetViewControl(AppConfig.ObjectViewMode mode)
        {
            // 未登録なら作成
            if (_viewControls[mode] == null)
            {
                using (var wait = new WaitCursor())
                {
                    IObjectView view = null;
                    switch (mode)
                    {
                        case AppConfig.ObjectViewMode.SchematicView:	view = new ObjectSchematicViewPanel();	break;
                        case AppConfig.ObjectViewMode.ModelList:		view = new ModelListView();				break;
                        case AppConfig.ObjectViewMode.MaterialList:		view = new MaterialListView();			break;
                        case AppConfig.ObjectViewMode.BoneList:			view = new BoneListView();				break;
                        case AppConfig.ObjectViewMode.ShapeList:		view = new ShapeListView();				break;
                        case AppConfig.ObjectViewMode.TextureList:		view = new TextureListView(); 			break;
                        default:										Debug.Assert(false);					break;
                    }

                    view.Control.Dock = DockStyle.Fill;
                    _viewControls[mode] = view;
                }
            }
            return _viewControls[mode];
        }

        #region イベントハンドラ
        //---------------------------------------------------------------------
        // レイアウトパネル
        private void layoutPanel_ViewModeChanged(object sender, ObjectViewModeChangedEventArgs e)
        {
            LayoutPanel layoutPanel = (LayoutPanel)sender;

            // 同じビューモードの他のパネルを探す
            ObjectViewPanel sameModeViewPanel = null;
            foreach (ObjectViewPanel viewPanel in layoutPanel.ViewPanels)
            {
                if (viewPanel != e.ViewPanel && viewPanel.ViewMode == e.ViewMode)
                {
                    sameModeViewPanel = viewPanel;
                    break;
                }
            }

            // 同じビューモードのパネルを変更前ビューモードにする
            if (sameModeViewPanel != null)
            {
                AppConfig.ObjectViewMode viewMode = e.ViewPanel.ViewMode;
                sameModeViewPanel.View = GetViewControl(viewMode);
            }

            // ビューモード変更
            e.ViewPanel.View = GetViewControl(e.ViewMode);
        }

        //---------------------------------------------------------------------
        // オブジェクト選択変更
        private void GuiObjectManager_SelectedChanged(object sender, EventArgs e)
        {
            ctlObjectBox.Objects = App.AppContext.SelectedTarget;
        }
        #endregion

        //---------------------------------------------------------------------
        // レイアウトパネルクラス
        //---------------------------------------------------------------------
        #region LayoutPanel
        /// <summary>
        /// レイアウトパネルクラス。
        /// </summary>
        private abstract class LayoutPanel : UIPanel
        {
            // ビューパネル配列
            private readonly ObjectViewPanel[] _viewPanels;

            /// <summary>ビューモード変更イベント。</summary>
            public event ObjectViewModeChangedEventHandler ViewModeChanged = null;

            /// <summary>
            /// コンストラクタ。
            /// </summary>
            protected LayoutPanel(int viewCount)
            {
                _viewPanels = new ObjectViewPanel[viewCount];
            }

            /// <summary>
            /// ビューパネル配列。
            /// </summary>
            public ObjectViewPanel[] ViewPanels
            {
                get { return _viewPanels; }
            }

            /// <summary>
            /// ビューレイアウト。
            /// </summary>
            public abstract AppConfig.ObjectViewLayout ViewLayout { get; }

            /// <summary>
            /// 分割比率をリセット。
            /// </summary>
            public abstract void ResetSplitRatio();

            /// <summary>
            /// スプリットコンテナを作成。
            /// </summary>
            protected LayoutSplitContainer CreateSplitContainer(Orientation orientation)
            {
                LayoutSplitContainer splitContainer = new LayoutSplitContainer(orientation);
                splitContainer.MouseDoubleClick += splitContainer_MouseDoubleClick;
                return splitContainer;
            }

            /// <summary>
            /// ビューパネルを作成。
            /// </summary>
            protected void CreateViewPanel(int layoutIndex, AppConfig.ObjectViewMode viewMode, Control parentControl)
            {
                Debug.Assert(layoutIndex < _viewPanels.Length);
                Debug.Assert(_viewPanels[layoutIndex] == null);

                // 作成して配列に登録
                ObjectViewPanel panel = new ObjectViewPanel(layoutIndex, viewMode);
                panel.Dock = DockStyle.Fill;
                panel.Parent = parentControl;
                panel.ViewModeChanged += viewPanel_ViewModeChanged;
                _viewPanels[layoutIndex] = panel;
                panel.FilterChanged += SaveFilterToConfig;
            }

            public void SaveFilterToConfig()
            {
                foreach (var panel in ViewPanels)
                {
                    var materialListView = panel.View as MaterialListView;
                    if (materialListView != null)
                    {
                        ApplicationConfig.Setting.ObjectView.MaterialListView.Filter = materialListView.FilterCategoryString;
                    }
                }
            }

            public void SetFilterFromConfig()
            {
                foreach (var panel in ViewPanels)
                {
                    var materialListView = panel.View as MaterialListView;
                    if (materialListView != null)
                    {
                        materialListView.FilterCategoryString = ApplicationConfig.Setting.ObjectView.MaterialListView.Filter;
                        panel.UpdateFilterComboBox();
                    }
                }
            }

            #region イベントハンドラ
            //-----------------------------------------------------------------
            // スプリットコンテナ
            private void splitContainer_MouseDoubleClick(object sender, MouseEventArgs e)
            {
                // 分割比率をリセット
                ResetSplitRatio();
            }

            //-----------------------------------------------------------------
            // ビューパネル
            private void viewPanel_ViewModeChanged(object sender, ObjectViewModeChangedEventArgs e)
            {
                if (ViewModeChanged != null)
                {
                    ViewModeChanged(this, e);
                }
            }
            #endregion
        }
        #endregion

        #region SingleLayoutPanel
        /// <summary>
        /// １画面レイアウトパネルクラス。
        /// </summary>
        private sealed class SingleLayoutPanel : LayoutPanel
        {
            /// <summary>
            /// コンストラクタ。
            /// </summary>
            public SingleLayoutPanel()
                : base(1)
            {
                SuspendLayout();
                {
                    CreateViewPanel(0, ApplicationConfig.Setting.ObjectView.Single.Mode0, this);
                }
                ResumeLayout();

                // コンフィグ書き込み
                HandleDestroyed += (s, e) =>
                {
                    ApplicationConfig.Setting.ObjectView.Single.Mode0 = ViewPanels[0].ViewMode;
                };
            }

            /// <summary>
            /// オーバーライド。
            /// </summary>
            public override AppConfig.ObjectViewLayout ViewLayout
            {
                get { return AppConfig.ObjectViewLayout.Single; }
            }

            /// <summary>
            /// オーバーライド。
            /// </summary>
            public override void ResetSplitRatio()
            {
                // 何もしない
            }
        }
        #endregion

        #region Multi2StackedLayoutPanel
        /// <summary>
        /// ２分割（上下）レイアウトパネルクラス。
        /// </summary>
        private sealed class Multi2StackedLayoutPanel : LayoutPanel
        {
            // クライアントコンテナ
            private readonly LayoutSplitContainer _spcClient;

            /// <summary>
            /// コンストラクタ。
            /// </summary>
            public Multi2StackedLayoutPanel()
                : base(2)
            {
                SuspendLayout();
                {
                    _spcClient = CreateSplitContainer(Orientation.Horizontal);
                    _spcClient.Parent = this;
                    _spcClient.SplitRatio = ApplicationConfig.Setting.ObjectView.Multi2Stacked.Rate0;

                    CreateViewPanel(0, ApplicationConfig.Setting.ObjectView.Multi2Stacked.Mode0, _spcClient.Panel1);
                    CreateViewPanel(1, ApplicationConfig.Setting.ObjectView.Multi2Stacked.Mode1, _spcClient.Panel2);
                }
                ResumeLayout();

                // コンフィグ書き込み
                HandleDestroyed += (s, e) =>
                {
                    ApplicationConfig.Setting.ObjectView.Multi2Stacked.Mode0 = ViewPanels[0].ViewMode;
                    ApplicationConfig.Setting.ObjectView.Multi2Stacked.Mode1 = ViewPanels[1].ViewMode;
                    ApplicationConfig.Setting.ObjectView.Multi2Stacked.Rate0 = _spcClient.SplitRatio;
                };
            }

            /// <summary>
            /// オーバーライド。
            /// </summary>
            public override AppConfig.ObjectViewLayout ViewLayout
            {
                get { return AppConfig.ObjectViewLayout.Multi2Stacked; }
            }

            /// <summary>
            /// オーバーライド。
            /// </summary>
            public override void ResetSplitRatio()
            {
                _spcClient.SplitRatio = 0.5f;
            }
        }
        #endregion

        #region Multi2SideBySideLayoutPanel
        /// <summary>
        /// ２分割（左右）レイアウトパネルクラス。
        /// </summary>
        private sealed class Multi2SideBySideLayoutPanel : LayoutPanel
        {
            // クライアントコンテナ
            private readonly LayoutSplitContainer _spcClient;

            /// <summary>
            /// コンストラクタ。
            /// </summary>
            public Multi2SideBySideLayoutPanel()
                : base(2)
            {
                SuspendLayout();
                {
                    _spcClient = CreateSplitContainer(Orientation.Vertical);
                    _spcClient.Parent = this;
                    _spcClient.SplitRatio = ApplicationConfig.Setting.ObjectView.Multi2SideBySide.Rate0;

                    CreateViewPanel(0, ApplicationConfig.Setting.ObjectView.Multi2SideBySide.Mode0, _spcClient.Panel1);
                    CreateViewPanel(1, ApplicationConfig.Setting.ObjectView.Multi2SideBySide.Mode1, _spcClient.Panel2);
                }
                ResumeLayout();

                // コンフィグ書き込み
                HandleDestroyed += (s, e) =>
                {
                    ApplicationConfig.Setting.ObjectView.Multi2SideBySide.Mode0 = ViewPanels[0].ViewMode;
                    ApplicationConfig.Setting.ObjectView.Multi2SideBySide.Mode1 = ViewPanels[1].ViewMode;
                    ApplicationConfig.Setting.ObjectView.Multi2SideBySide.Rate0 = _spcClient.SplitRatio;
                };
            }

            /// <summary>
            /// オーバーライド。
            /// </summary>
            public override AppConfig.ObjectViewLayout ViewLayout
            {
                get { return AppConfig.ObjectViewLayout.Multi2SideBySide; }
            }

            /// <summary>
            /// オーバーライド。
            /// </summary>
            public override void ResetSplitRatio()
            {
                _spcClient.SplitRatio = 0.5f;
            }
        }
        #endregion

        #region Multi3SplitBottomLayoutPanel
        /// <summary>
        /// ３分割（下を分割）レイアウトパネルクラス。
        /// </summary>
        private sealed class Multi3SplitBottomLayoutPanel : LayoutPanel
        {
            // クライアントコンテナ
            private readonly LayoutSplitContainer _spcClient;
            // 下領域コンテナ
            private readonly LayoutSplitContainer _spcBottom;

            /// <summary>
            /// コンストラクタ。
            /// </summary>
            public Multi3SplitBottomLayoutPanel()
                : base(3)
            {
                SuspendLayout();
                {
                    _spcClient = CreateSplitContainer(Orientation.Horizontal);
                    _spcClient.Parent = this;
                    _spcClient.SplitRatio = ApplicationConfig.Setting.ObjectView.Multi3SplitBottom.Rate0;

                    _spcBottom = CreateSplitContainer(Orientation.Vertical);
                    _spcBottom.Parent = _spcClient.Panel2;
                    _spcBottom.SplitRatio = ApplicationConfig.Setting.ObjectView.Multi3SplitBottom.Rate1;

                    CreateViewPanel(0, ApplicationConfig.Setting.ObjectView.Multi3SplitBottom.Mode0, _spcClient.Panel1);
                    CreateViewPanel(1, ApplicationConfig.Setting.ObjectView.Multi3SplitBottom.Mode1, _spcBottom.Panel1);
                    CreateViewPanel(2, ApplicationConfig.Setting.ObjectView.Multi3SplitBottom.Mode2, _spcBottom.Panel2);
                }
                ResumeLayout();

                // コンフィグ書き込み
                HandleDestroyed += (s, e) =>
                {
                    ApplicationConfig.Setting.ObjectView.Multi3SplitBottom.Mode0 = ViewPanels[0].ViewMode;
                    ApplicationConfig.Setting.ObjectView.Multi3SplitBottom.Mode1 = ViewPanels[1].ViewMode;
                    ApplicationConfig.Setting.ObjectView.Multi3SplitBottom.Mode2 = ViewPanels[2].ViewMode;
                    ApplicationConfig.Setting.ObjectView.Multi3SplitBottom.Rate0 = _spcClient.SplitRatio;
                    ApplicationConfig.Setting.ObjectView.Multi3SplitBottom.Rate1 = _spcBottom.SplitRatio;
                };
            }

            /// <summary>
            /// オーバーライド。
            /// </summary>
            public override AppConfig.ObjectViewLayout ViewLayout
            {
                get { return AppConfig.ObjectViewLayout.Multi3SplitBottom; }
            }

            /// <summary>
            /// オーバーライド。
            /// </summary>
            public override void ResetSplitRatio()
            {
                _spcClient.SplitRatio = 0.5f;
                _spcBottom.SplitRatio = 0.5f;
            }
        }
        #endregion

        #region Multi3SplitTopLayoutPanel
        /// <summary>
        /// ３分割（上を分割）レイアウトパネルクラス。
        /// </summary>
        private sealed class Multi3SplitTopLayoutPanel : LayoutPanel
        {
            // クライアントコンテナ
            private readonly LayoutSplitContainer _spcClient;
            // 上領域コンテナ
            private readonly LayoutSplitContainer _spcTop;

            /// <summary>
            /// コンストラクタ。
            /// </summary>
            public Multi3SplitTopLayoutPanel()
                : base(3)
            {
                SuspendLayout();
                {
                    _spcClient = CreateSplitContainer(Orientation.Horizontal);
                    _spcClient.Parent = this;
                    _spcClient.SplitRatio = ApplicationConfig.Setting.ObjectView.Multi3SplitTop.Rate0;

                    _spcTop = CreateSplitContainer(Orientation.Vertical);
                    _spcTop.Parent = _spcClient.Panel1;
                    _spcTop.SplitRatio = ApplicationConfig.Setting.ObjectView.Multi3SplitTop.Rate1;

                    CreateViewPanel(0, ApplicationConfig.Setting.ObjectView.Multi3SplitTop.Mode0, _spcTop.Panel1);
                    CreateViewPanel(1, ApplicationConfig.Setting.ObjectView.Multi3SplitTop.Mode1, _spcTop.Panel2);
                    CreateViewPanel(2, ApplicationConfig.Setting.ObjectView.Multi3SplitTop.Mode2, _spcClient.Panel2);
                }
                ResumeLayout();

                // コンフィグ書き込み
                HandleDestroyed += (s, e) =>
                {
                    ApplicationConfig.Setting.ObjectView.Multi3SplitTop.Mode0 = ViewPanels[0].ViewMode;
                    ApplicationConfig.Setting.ObjectView.Multi3SplitTop.Mode1 = ViewPanels[1].ViewMode;
                    ApplicationConfig.Setting.ObjectView.Multi3SplitTop.Mode2 = ViewPanels[2].ViewMode;
                    ApplicationConfig.Setting.ObjectView.Multi3SplitTop.Rate0 = _spcClient.SplitRatio;
                    ApplicationConfig.Setting.ObjectView.Multi3SplitTop.Rate1 = _spcTop.SplitRatio;
                };
            }

            /// <summary>
            /// オーバーライド。
            /// </summary>
            public override AppConfig.ObjectViewLayout ViewLayout
            {
                get { return AppConfig.ObjectViewLayout.Multi3SplitTop; }
            }

            /// <summary>
            /// オーバーライド。
            /// </summary>
            public override void ResetSplitRatio()
            {
                _spcClient.SplitRatio = 0.5f;
                _spcTop.SplitRatio = 0.5f;
            }
        }
        #endregion

        #region Multi3SplitRightLayoutPanel
        /// <summary>
        /// ３分割（右を分割）レイアウトパネルクラス。
        /// </summary>
        private sealed class Multi3SplitRightLayoutPanel : LayoutPanel
        {
            // クライアントコンテナ
            private readonly LayoutSplitContainer _spcClient;
            // 右領域コンテナ
            private readonly LayoutSplitContainer _spcRight;

            /// <summary>
            /// コンストラクタ。
            /// </summary>
            public Multi3SplitRightLayoutPanel()
                : base(3)
            {
                SuspendLayout();
                {
                    _spcClient = CreateSplitContainer(Orientation.Vertical);
                    _spcClient.Parent = this;
                    _spcClient.SplitRatio = ApplicationConfig.Setting.ObjectView.Multi3SplitRight.Rate0;

                    _spcRight = CreateSplitContainer(Orientation.Horizontal);
                    _spcRight.Parent = _spcClient.Panel2;
                    _spcRight.SplitRatio = ApplicationConfig.Setting.ObjectView.Multi3SplitRight.Rate1;

                    CreateViewPanel(0, ApplicationConfig.Setting.ObjectView.Multi3SplitRight.Mode0, _spcClient.Panel1);
                    CreateViewPanel(1, ApplicationConfig.Setting.ObjectView.Multi3SplitRight.Mode1, _spcRight.Panel1);
                    CreateViewPanel(2, ApplicationConfig.Setting.ObjectView.Multi3SplitRight.Mode2, _spcRight.Panel2);
                }
                ResumeLayout();

                // コンフィグ書き込み
                HandleDestroyed += (s, e) =>
                {
                    ApplicationConfig.Setting.ObjectView.Multi3SplitRight.Mode0 = ViewPanels[0].ViewMode;
                    ApplicationConfig.Setting.ObjectView.Multi3SplitRight.Mode1 = ViewPanels[1].ViewMode;
                    ApplicationConfig.Setting.ObjectView.Multi3SplitRight.Mode2 = ViewPanels[2].ViewMode;
                    ApplicationConfig.Setting.ObjectView.Multi3SplitRight.Rate0 = _spcClient.SplitRatio;
                    ApplicationConfig.Setting.ObjectView.Multi3SplitRight.Rate1 = _spcRight.SplitRatio;
                };
            }

            /// <summary>
            /// オーバーライド。
            /// </summary>
            public override AppConfig.ObjectViewLayout ViewLayout
            {
                get { return AppConfig.ObjectViewLayout.Multi3SplitRight; }
            }

            /// <summary>
            /// オーバーライド。
            /// </summary>
            public override void ResetSplitRatio()
            {
                _spcClient.SplitRatio = 0.5f;
                _spcRight.SplitRatio = 0.5f;
            }
        }
        #endregion

        #region Multi3SplitLeftLayoutPanel
        /// <summary>
        /// ３分割（左を分割）レイアウトパネルクラス。
        /// </summary>
        private sealed class Multi3SplitLeftLayoutPanel : LayoutPanel
        {
            // クライアントコンテナ
            private readonly LayoutSplitContainer _spcClient;
            // 左領域コンテナ
            private readonly LayoutSplitContainer _spcLeft;

            /// <summary>
            /// コンストラクタ。
            /// </summary>
            public Multi3SplitLeftLayoutPanel()
                : base(3)
            {
                SuspendLayout();
                {
                    _spcClient = CreateSplitContainer(Orientation.Vertical);
                    _spcClient.Parent = this;
                    _spcClient.SplitRatio = ApplicationConfig.Setting.ObjectView.Multi3SplitLeft.Rate0;

                    _spcLeft = CreateSplitContainer(Orientation.Horizontal);
                    _spcLeft.Parent = _spcClient.Panel1;
                    _spcLeft.SplitRatio = ApplicationConfig.Setting.ObjectView.Multi3SplitLeft.Rate1;

                    CreateViewPanel(0, ApplicationConfig.Setting.ObjectView.Multi3SplitLeft.Mode0, _spcLeft.Panel1);
                    CreateViewPanel(1, ApplicationConfig.Setting.ObjectView.Multi3SplitLeft.Mode1, _spcLeft.Panel2);
                    CreateViewPanel(2, ApplicationConfig.Setting.ObjectView.Multi3SplitLeft.Mode2, _spcClient.Panel2);
                }
                ResumeLayout();

                // コンフィグ書き込み
                HandleDestroyed += (s, e) =>
                {
                    ApplicationConfig.Setting.ObjectView.Multi3SplitLeft.Mode0 = ViewPanels[0].ViewMode;
                    ApplicationConfig.Setting.ObjectView.Multi3SplitLeft.Mode1 = ViewPanels[1].ViewMode;
                    ApplicationConfig.Setting.ObjectView.Multi3SplitLeft.Mode2 = ViewPanels[2].ViewMode;
                    ApplicationConfig.Setting.ObjectView.Multi3SplitLeft.Rate0 = _spcClient.SplitRatio;
                    ApplicationConfig.Setting.ObjectView.Multi3SplitLeft.Rate1 = _spcLeft.SplitRatio;
                };
            }

            /// <summary>
            /// オーバーライド。
            /// </summary>
            public override AppConfig.ObjectViewLayout ViewLayout
            {
                get { return AppConfig.ObjectViewLayout.Multi3SplitLeft; }
            }

            /// <summary>
            /// オーバーライド。
            /// </summary>
            public override void ResetSplitRatio()
            {
                _spcClient.SplitRatio = 0.5f;
                _spcLeft.SplitRatio = 0.5f;
            }
        }
        #endregion

        #region Multi4LayoutPanel
        /// <summary>
        /// ４分割レイアウトパネルクラス。
        /// </summary>
        private sealed class Multi4LayoutPanel : LayoutPanel
        {
            // クライアントコンテナ
            private readonly LayoutSplitContainer _spcClient;
            // 上領域コンテナ
            private readonly LayoutSplitContainer _spcTop;
            // 下領域コンテナ
            private readonly LayoutSplitContainer _spcBottom;

            /// <summary>
            /// コンストラクタ。
            /// </summary>
            public Multi4LayoutPanel()
                : base(4)
            {
                SuspendLayout();
                {
                    _spcClient = CreateSplitContainer(Orientation.Horizontal);
                    _spcClient.Parent = this;
                    _spcClient.SplitRatio = ApplicationConfig.Setting.ObjectView.Multi4.Rate0;

                    _spcTop = CreateSplitContainer(Orientation.Vertical);
                    _spcTop.Parent = _spcClient.Panel1;
                    _spcTop.SplitRatio = ApplicationConfig.Setting.ObjectView.Multi4.Rate1;

                    _spcBottom = CreateSplitContainer(Orientation.Vertical);
                    _spcBottom.Parent = _spcClient.Panel2;
                    _spcBottom.SplitRatio = ApplicationConfig.Setting.ObjectView.Multi4.Rate2;

                    CreateViewPanel(0, ApplicationConfig.Setting.ObjectView.Multi4.Mode0, _spcTop.Panel1);
                    CreateViewPanel(1, ApplicationConfig.Setting.ObjectView.Multi4.Mode1, _spcTop.Panel2);
                    CreateViewPanel(2, ApplicationConfig.Setting.ObjectView.Multi4.Mode2, _spcBottom.Panel1);
                    CreateViewPanel(3, ApplicationConfig.Setting.ObjectView.Multi4.Mode3, _spcBottom.Panel2);
                }
                ResumeLayout();

                // コンフィグ書き込み
                HandleDestroyed += (s, e) =>
                {
                    ApplicationConfig.Setting.ObjectView.Multi4.Mode0 = ViewPanels[0].ViewMode;
                    ApplicationConfig.Setting.ObjectView.Multi4.Mode1 = ViewPanels[1].ViewMode;
                    ApplicationConfig.Setting.ObjectView.Multi4.Mode2 = ViewPanels[2].ViewMode;
                    ApplicationConfig.Setting.ObjectView.Multi4.Mode3 = ViewPanels[3].ViewMode;
                    ApplicationConfig.Setting.ObjectView.Multi4.Rate0 = _spcClient.SplitRatio;
                    ApplicationConfig.Setting.ObjectView.Multi4.Rate1 = _spcTop.SplitRatio;
                    ApplicationConfig.Setting.ObjectView.Multi4.Rate2 = _spcBottom.SplitRatio;
                };
            }

            /// <summary>
            /// オーバーライド。
            /// </summary>
            public override AppConfig.ObjectViewLayout ViewLayout
            {
                get { return AppConfig.ObjectViewLayout.Multi4; }
            }

            /// <summary>
            /// オーバーライド。
            /// </summary>
            public override void ResetSplitRatio()
            {
                _spcClient.SplitRatio = 0.5f;
                _spcTop.SplitRatio = 0.5f;
                _spcBottom.SplitRatio = 0.5f;
            }
        }
        #endregion

        #region LayoutSplitContainer
        /// <summary>
        /// レイアウト用スプリットコンテナクラス。
        /// </summary>
        private sealed class LayoutSplitContainer : UISplitContainer
        {
            /// <summary>
            /// コンストラクタ。
            /// </summary>
            public LayoutSplitContainer(Orientation orientation)
            {
                // 初期設定
                Orientation = orientation;
                Dock = DockStyle.Fill;

                // 分割比率設定時に誤差を生じさせない用規定値を大きくする
                Width = 480;
                Height = 480;
            }
        }
        #endregion
    }

    //----------------------------------------------------------------------------
    // イベントデータ
    #region ObjectViewModeChangedEvent
    /// <summary>
    /// オブジェクトビューモード変更イベントハンドラデリゲート。
    /// </summary>
    public delegate void ObjectViewModeChangedEventHandler(object sender, ObjectViewModeChangedEventArgs e);

    /// <summary>
    /// オブジェクトビューモード変更イベントデータクラス。
    /// </summary>
    public sealed class ObjectViewModeChangedEventArgs : EventArgs
    {
        // レイアウトインデクス
        private readonly int _layoutIndex;
        // ビューモード
        private readonly AppConfig.ObjectViewMode _viewMode;
        // ビューパネル
        private readonly ObjectViewPanel _viewPanel;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ObjectViewModeChangedEventArgs(int layoutIndex, AppConfig.ObjectViewMode viewMode, ObjectViewPanel viewPanel)
        {
            _layoutIndex = layoutIndex;
            _viewMode = viewMode;
            _viewPanel = viewPanel;
        }

        /// <summary>
        /// レイアウトインデクス。
        /// </summary>
        public int LayoutIndex
        {
            get { return _layoutIndex; }
        }

        /// <summary>
        /// ビューモード。
        /// </summary>
        public AppConfig.ObjectViewMode ViewMode
        {
            get { return _viewMode; }
        }

        /// <summary>
        /// ビューパネル。
        /// </summary>
        public ObjectViewPanel ViewPanel
        {
            get { return _viewPanel; }
        }
    }
    #endregion

    //----------------------------------------------------------------------------
    // インタフェース
    #region IObjectView
    /// <summary>
    /// オブジェクトビューインタフェース。
    /// </summary>
    public interface IObjectView
    {
        /// <summary>
        /// ビューモード。
        /// </summary>
        AppConfig.ObjectViewMode ViewMode { get; }

        /// <summary>
        /// コントロール。
        /// </summary>
        Control Control { get; }

        /// <summary>
        /// クライアント領域にフォーカスを設定する。
        /// </summary>
        void FocusClient();

        /// <summary>
        /// すべて選択可能か
        /// </summary>
        bool EnableSelectAll { get; }

        /// <summary>
        /// すべて選択
        /// </summary>
        void SelectAll();

        /// <summary>
        /// 選択切り替え可能か
        /// </summary>
        bool EnableToggleSelection { get; }

        /// <summary>
        /// 選択切り替え
        /// </summary>
        void ToggleSelection();
    }
    #endregion
}
