﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Windows.Forms;
using System.Linq;
using System.Text;
using App.Controls;
using App.Data;
using App.Utility;
using App.res;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;
using ConfigCommon;

namespace App.ObjectView.List
{
    /// <summary>
    /// モデルリストビュークラス。
    /// </summary>
    [ToolboxItem(false)]
    public sealed class TextureListView : StandardObjectListView
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public TextureListView()
            : base(ViewID.Texture, AppConfig.ObjectViewMode.TextureList)
        {
            // 行の高さを指定します。
            SmallImageList = new ImageList() { ImageSize = new Size(34, 34) };
        }

        /// <summary>
        /// 列項目初期化。
        /// </summary>
        public static void InitializeColumn()
        {
            BeginRegisterColumn(ViewID.Texture, "TextureListView");
            CurrentPropertyPageID = PropertyEdit.PropertyPageID.TextureRoot;

            // 名前
            RegisterTextureColumn(
                Strings.ListView_Texture_Column_Name,
                x => x,
                x =>
                {
                    var name = x.Name;
                    var indexText = DocumentManager.GetSameNameIndexText(x, false);
                    if (indexText.Length > 0)
                    {
                        name += string.Format("  ({0} {1})", x.FileName, indexText);
                    }
                    else if (DocumentManager.TextureNameCount[name] > 1)
                    {
                        name += string.Format("  ({0})", x.FileName);
                    }
                    return name;
                },
                HorizontalAlignment.Left,
                ColumnDefaultWidthL,
                (x, y) => {
                    var a = (Texture)x;
                    var b = (Texture)y;
                    var result = string.Compare(a.Name, b.Name);
                    if (result != 0)
                    {
                        return result;
                    }

                    result = string.Compare(a.FileName, b.FileName);
                    if (result != 0)
                    {
                        return result;
                    }

                    return DocumentManager.IndexInSameNames[a].CompareTo(DocumentManager.IndexInSameNames[b]);
                }
            );
            RegisterImageColumn(
                Strings.ListView_Texture_Column_Image,
                ColumnDefaultWidthS);
            RegisterTextureColumn(
                Strings.ListView_Texture_Column_Hint,
                x => x.Data.texture_info.hint,
                null,
                HorizontalAlignment.Left,
                ColumnDefaultWidthM);
            // コメント
            RegisterCommentColorColumn(ColumnID.CommentColor, "CommentColor", res.Strings.ListView_Object_Column_CommentColor, HorizontalAlignment.Left, 16, true);
            RegisterColumn(ColumnID.CommentText, "CommentText", res.Strings.ListView_Object_Column_CommentText, HorizontalAlignment.Left, ColumnDefaultWidthS, true,
                data => ((Texture)data).Comment ?? string.Empty, ColumnID.CommentText.ToString());
            RegisterColumn(ColumnID.CommentLabel, "CommentLabel", res.Strings.ListView_Object_Column_CommentLabel, HorizontalAlignment.Left, ColumnDefaultWidthS, true,
                data => ((Texture)data).Label ?? string.Empty, ColumnID.CommentLabel.ToString());
            RegisterTextureColumn(
                Strings.ListView_Texture_Column_Linear,
                x => x.Data.texture_info.linear,
                x => {
                    var builder = new StringBuilder();
                    foreach(var i in x)
                    {
                        if (builder.Length > 0)
                        {
                            builder.Append(", ");
                        }
                        builder.Append(UIText.FlagYesNo(i));
                    }
                    return builder.ToString();
                },
                HorizontalAlignment.Left,
                ColumnDefaultWidthL);
            RegisterTextureColumn(
                Strings.ListView_Texture_Column_Dimension,
                x => x.Data.texture_info.dimension,
                UIText.EnumValue,
                HorizontalAlignment.Left,
                ColumnDefaultWidthM);
            RegisterTextureColumn(
                Strings.ListView_Texture_Column_Format,
                x => x.Data.texture_info.quantize_type,
                UIText.EnumValue,
                HorizontalAlignment.Left,
                ColumnDefaultWidthS);
            RegisterTextureColumn(
                Strings.ListView_Texture_Column_Width,
                x => x.Data.texture_info.width,
                null,
                HorizontalAlignment.Right,
                ColumnDefaultWidthS);
            RegisterTextureColumn(
                Strings.ListView_Texture_Column_Height,
                x => x.Data.texture_info.height,
                null,
                HorizontalAlignment.Right,
                ColumnDefaultWidthS);
            RegisterTextureColumn(
                Strings.ListView_Texture_Column_Depth,
                x => x.Data.texture_info.depth,
                null,
                HorizontalAlignment.Right,
                ColumnDefaultWidthS);
            RegisterTextureColumn(
                Strings.ListView_Texture_Column_VRAMSize,
                x => x.Data.texture_info.size,
                DataSize.XBytesText,
                HorizontalAlignment.Left,
                ColumnDefaultWidthS);
            RegisterTextureColumn(
                Strings.ListView_Texture_Column_MipLevel,
                x => x.Data.texture_info.mip_level,
                UIText.MipLevel,
                HorizontalAlignment.Right,
                ColumnDefaultWidthS);
            RegisterTextureColumn(
                Strings.ListView_Texture_Column_MipMapMinWidth,
                x => Math.Max(x.Data.texture_info.width >> (x.Data.texture_info.mip_level - 1), 1),
                null,
                HorizontalAlignment.Right,
                ColumnDefaultWidthM);
            RegisterTextureColumn(
                Strings.ListView_Texture_Column_MipMapMinHeight,
                x => Math.Max(x.Data.texture_info.height >> (x.Data.texture_info.mip_level - 1), 1),
                null,
                HorizontalAlignment.Right,
                ColumnDefaultWidthM);
            RegisterTextureColumn(
                Strings.ListView_Texture_Column_DccPreset,
                x => x.Data.texture_info.dcc_preset,
                null,
                HorizontalAlignment.Left,
                ColumnDefaultWidthL);
            RegisterTextureColumn(
                Strings.ListView_Texture_Column_MipGenFilter,
                x => x.Data.texture_info.mip_gen_filter,
                null,
                HorizontalAlignment.Left,
                ColumnDefaultWidthL);
            RegisterTextureColumn(
                Strings.ListView_Texture_Column_ComponentSelector,
                x => ToString(x.Data.texture_info.comp_sel),
                null,
                HorizontalAlignment.Left,
                ColumnDefaultWidthM);
            RegisterTextureColumn(
                Strings.ListView_Texture_Column_WeightedCompress,
                x => UIText.FlagYesNo(x.Data.texture_info.weighted_compress),
                null,
                HorizontalAlignment.Left,
                ColumnDefaultWidthM);
            RegisterTextureColumn(
                Strings.ListView_Texture_Column_InitialSwizzle,
                x => x.Data.texture_info.initial_swizzle,
                null,
                HorizontalAlignment.Right,
                ColumnDefaultWidthS);
            RegisterTextureColumn(
                Strings.ListView_Texture_Column_OriginalImageHash,
                x => x.Data.texture_info.original_image_hash,
                null,
                HorizontalAlignment.Left,
                ColumnDefaultWidthS);
            EndRegisterColumn();
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void CreateItem()
        {
            foreach (Texture texture in DocumentManager.OrderedTextures)
            {
                if (IsMatchedToSearchText(texture.Name))
                {
                    RegisterItem(texture, GuiObjectID.Texture, List.ViewID.Texture);
                }
            }
        }

        /// <summary>
        /// マテリアル用のRegisterColumn
        /// </summary>
        private static void RegisterTextureColumn<T>(
            string name,
            Func<Texture, T> memberSelect,
            Func<T, string> specificText,
            HorizontalAlignment TextAlign,
            int width,
            Comparison<object> comparer = null)
        {
            var item = new ColumnItem()
            {
                Text = name,
                TextAlign = TextAlign,
                Width = width,
                GetParamObject = x => memberSelect((Texture)x),
                CustomComparer = comparer,
            };
            if (specificText != null)
            {
                item.CustomDraw = (v, e, x) => e.SpecificText = specificText((T)x);
            }

            RegisterColumn(item);
        }

        /// <summary>
        /// マテリアル用のRegisterColumn
        /// </summary>
        private static void RegisterImageColumn(
            string name,
            int width)
        {
            var item = new ColumnItem()
            {
                Text = name,
                TextAlign = HorizontalAlignment.Left,
                Width = width,
                GetParamObject = x => x,
            };
            item.CustomDraw = (v, e, x) =>
                {
                    Texture texture = (Texture)x;
                    Rectangle rcImage = e.Bounds;
                    rcImage.X += 2;
                    rcImage.Y += 2;
                    rcImage.Width = Texture.ThumbnailSize;
                    rcImage.Height = Texture.ThumbnailSize;

                    texture.DrawColorThumbnail(e.Graphics, rcImage, true, true);

                    Rectangle rcFrame = rcImage;
                    rcFrame.X -= 1;
                    rcFrame.Y -= 1;

                    GraphicsUtility.DrawRectangle(e.Graphics, Pens.Black, rcFrame);

                    // アルファイメージ
                    rcImage.X += (rcImage.Width + 1);
                    if (texture.HasAlpha)
                    {
                        texture.DrawAlphaThumbnail(e.Graphics, rcImage, true, true);

                        rcFrame = rcImage;
                        rcFrame.X -= 1;
                        rcFrame.Y -= 1;

                        GraphicsUtility.DrawRectangle(e.Graphics, Pens.Black, rcFrame);
                    }

                    rcImage.X += rcImage.Width + 1;
                    e.Handled = true;
                };

            RegisterColumn(item);
        }

        static private readonly Dictionary<texture_info_comp_selValue, string> texture_info_comp_selValueDict_ = new Dictionary<texture_info_comp_selValue, string>()
        {
            {texture_info_comp_selValue.r,		"R"},
            {texture_info_comp_selValue.g,		"G"},
            {texture_info_comp_selValue.b,		"B"},
            {texture_info_comp_selValue.a,		"A"},
            {texture_info_comp_selValue.Item0,	"0"},
            {texture_info_comp_selValue.Item1,	"1"},
        };

        private static string ToString(texture_info_comp_selValue[] compSel)
        {
            var s = string.Empty;
            {
                foreach(var c in compSel)
                {
                    s += texture_info_comp_selValueDict_[c] + ", ";
                }
            }
            return s.Substring(0, s.Length - 2);
        }
        #region ColumnID
        /// <summary>
        /// 列項目ＩＤ。
        /// </summary>
        public enum ColumnID
        {
            /// <summary>コメント。</summary>
            Comment = 0,
            /// <summary>コメントカラー。</summary>
            CommentColor,
            /// <summary>コメントテキスト。</summary>
            CommentText,
            /// <summary>コメントラベル。</summary>
            CommentLabel,
        }
        #endregion
    }
}

