﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Windows.Forms;
using System.Linq;
using App.Controls;
using App.Data;
using App.res;
using App.Utility;
using ConfigCommon;
using nw.g3d.nw4f_3dif;

namespace App.ObjectView.List
{
    /// <summary>
    /// モデルリストビュークラス。
    /// </summary>
    [ToolboxItem(false)]
    public sealed class ShapeListView : StandardObjectListView
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ShapeListView()
            : base(ViewID.Shape, AppConfig.ObjectViewMode.ShapeList)
        {
        }

        protected override void Document_AddedOrRemoved(object sender, System.Collections.Generic.IEnumerable<Document> opened, System.Collections.Generic.IEnumerable<Document> closed, System.Collections.Generic.Dictionary<GuiObject, GuiObject> swaped, System.Collections.Generic.IEnumerable<Document> reloaded)
        {
            IEnumerable<Document> documents = opened.Concat(closed).Concat(reloaded);
            if (documents.Any(x => x.ObjectID == GuiObjectID.Model))
            {
                // 列の更新
                SetupColumn();
                NotifyColumnSettingChanged();
            }
            else
            {
                base.Document_AddedOrRemoved(sender, opened, closed, swaped, reloaded);
            }
        }

        /// <summary>
        /// 列項目初期化。
        /// </summary>
        public static void InitializeColumn()
        {
            SetupColumn();
        }

        public static void SetupColumn()
        {
            BeginRegisterColumn(ViewID.Shape, "ShapeListView");
            {
                CurrentPropertyPageID = PropertyEdit.PropertyPageID.ShapeRoot;

                // ColumnIDで宣言している順番で登録します。
                // シェイプインデックス
                RegisterColumn(ColumnID.ShapeIndex, "ShapeIndex", res.Strings.ListView_Shape_Column_ShapeIndex, HorizontalAlignment.Right, ColumnDefaultWidthNumber, true,
                    data => (data as Shape).Data.index);
                // シェイプ名
                RegisterColumn(ColumnID.Name, "Name", res.Strings.ListView_Shape_Column_Name, HorizontalAlignment.Left, ColumnDefaultWidthS, true,
                    data => (data as Shape).Data.name);
                // オーナーモデル
                RegisterColumn(ColumnID.Owner, "Owner", res.Strings.ListView_Shape_Column_Owner, HorizontalAlignment.Left, ColumnDefaultWidthL, true,
                    data => (data as Shape).Owner.Name);
                // マテリアル
                RegisterColumn(ColumnID.Material, "Material", Strings.ListView_Shape_Column_Material, HorizontalAlignment.Left, ColumnDefaultWidthL, true,
                    data => (data as Shape).Data.shape_info.mat_name);
                // ボーン
                RegisterColumn(ColumnID.BoneIndex, "BoneName", res.Strings.ListView_Shape_Column_BoneName, HorizontalAlignment.Left, ColumnDefaultWidthL, true,
                    data => (data as Shape).Data.shape_info.bone_name);
                // 頂点インデックス
                RegisterColumn(ColumnID.VertexName, "VertexIndex", res.Strings.ListView_Shape_Column_VertexIndex, HorizontalAlignment.Right, ColumnDefaultWidthL, true,
                    data => (data as Shape).Data.shape_info.vertex_index);
                // オリジナルシェイプハッシュ
                RegisterColumn(ColumnID.OriginalShapeHash, "OriginalShapeHash", res.Strings.ListView_Shape_Column_OriginalShapeHash, HorizontalAlignment.Left, ColumnDefaultWidthL, true,
                    data => (data as Shape).Data.shape_info.original_shape_hash);
                // オリジナルシェイプ数
                RegisterColumn(ColumnID.OriginalShapeCount, "OriginalShapeCount", res.Strings.ListView_Shape_Column_OriginalShapeCount, HorizontalAlignment.Right, ColumnDefaultWidthL, true,
                    data => (data as Shape).Data.shape_info.original_shape_count);
                // 近似頂点削除モード
                RegisterColumn(ColumnID.DivideSubmeshMode, "DeleteNearVertexMode", res.Strings.ListView_Shape_Column_DeleteNearVertexMode, HorizontalAlignment.Left, ColumnDefaultWidthL, true,
                    data => (data as Shape).Data.shape_info.delete_near_vertex_mode);
                // サブメッシュ分割モード
                RegisterColumn(ColumnID.DivideSubmeshMode, "DivideSubmeshMode", res.Strings.ListView_Shape_Column_DivideSubmeshMode, HorizontalAlignment.Left, ColumnDefaultWidthL, true,
                    data => (data as Shape).Data.shape_info.divide_submesh_mode);
                // プリミティブ最適化モード
                RegisterColumn(ColumnID.OptimizePrimitiveMode, "OptimizePrimitiveMode", res.Strings.ListView_Shape_Column_OptimizePrimitiveMode, HorizontalAlignment.Left, ColumnDefaultWidthL, true,
                    data => (data as Shape).Data.shape_info.optimize_primitive_mode);
                // 一つの頂点に影響を与えるボーン数
                RegisterColumn(ColumnID.VertexSkinningCount, "VertexSkinningCount", res.Strings.ListView_Shape_Column_VertexSkinningCount, HorizontalAlignment.Right, ColumnDefaultWidthL, true,
                    data => (data as Shape).Data.shape_info.vertex_skinning_count);
                // オリジナルボーン
                RegisterColumn(ColumnID.OriginalBoneName, "OriginalBoneName", res.Strings.ListView_Shape_Column_OriginalBoneName, HorizontalAlignment.Left, ColumnDefaultWidthL, true,
                    data => (data as Shape).Data.shape_info.original_bone_name);
                // オリジナルマテリアル
                RegisterColumn(ColumnID.OriginalMaterialName, "OriginalMaterialName", res.Strings.ListView_Shape_Column_OriginalMaterialName, HorizontalAlignment.Left, ColumnDefaultWidthL, true,
                    data => (data as Shape).Data.shape_info.original_material_name);

                int meshCount = 1;
                if (DocumentManager.Shapes.Any())
                {
                    meshCount = Math.Max(1, DocumentManager.Shapes.Max(x => x.Data.mesh_array != null ? x.Data.mesh_array.mesh.Length : 0));
                }

                for (int i = 0; i < meshCount; i++)
                {
                    int ii = i;
                    Predicate<object> isValidMesh = x =>
                        {
                            return ((Shape)x).Data.mesh_array.mesh.Length > ii;
                        };
                    Func<object, meshType> mesh = x => ((Shape)x).Data.mesh_array.mesh[ii];
                    Func<string, int> width = x => Math.Min(ColumnDefaultWidthL, GraphicsUtility.MeasureString(x) + 10);

                    // メッシュ(頂点インデックス)のプリミティブ
                    string text = string.Format(res.Strings.ListView_Shape_Column_MeshMode, i);
                    RegisterColumn(
                        ColumnID.MeshMode,
                        "MeshMode",
                        text,
                        HorizontalAlignment.Left,
                        width(text),
                        true,
                        data => UIText.EnumValue(mesh(data).mode),
                        isValid:isValidMesh);
                    // メッシュ(頂点インデックス)の型
                    text = string.Format(res.Strings.ListView_Shape_Column_MeshQuantizeType, i);
                    RegisterColumn(
                        ColumnID.MeshQuantizeType,
                        "MeshQuantizeType",
                        text,
                        HorizontalAlignment.Left,
                        width(text),
                        true,
                        data => UIText.EnumValue(mesh(data).quantize_type),
                        isValid: isValidMesh);
                    // メッシュ(頂点インデックス)数
                    text = string.Format(res.Strings.ListView_Shape_Column_MeshCount, i);
                    RegisterColumn(
                        ColumnID.MeshCount,
                        "MeshCount",
                        text,
                        HorizontalAlignment.Right,
                        width(text),
                        true,
                        data => mesh(data).count,
                        isValid: isValidMesh);

                    // 要らない？
                    // メッシュ(頂点インデックス)の実データを格納する <stream> のインデックス
                    //RegisterColumn(ColumnID.MeshStreamIndex, "MeshStreamIndex", res.Strings.ListView_Shape_Column_MeshStreamIndex, HorizontalAlignment.Right, ColumnDefaultWidthL, true,
                    //	data => (data as Shape).Data.mesh_array.mesh[i].stream_index);
                }
            }
            EndRegisterColumn();
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void CreateItem()
        {
            foreach (var shape in DocumentManager.OrderedModels.Where(x => x.IsShowInObjView).SelectMany(x => x.Shapes))
            {
                if (IsMatchedToSearchText(shape.Name))
                {
                    RegisterItem(shape, GuiObjectID.Shape, ViewID.Shape);
                }
            }
        }

        #region ColumnID
        /// <summary>
        /// 列項目ＩＤ。
        /// </summary>
        public enum ColumnID
        {
            /// <summary>インデックス。</summary>
            ShapeIndex,
            /// <summary>名前。</summary>
            Name,
            /// <summary>オーナーモデル。</summary>
            Owner,
            /// <summary>マテリアル。</summary>
            Material,
            /// <summary>ボーンインデックス。</summary>
            BoneIndex,
            /// <summary>頂点名。</summary>
            VertexName,
            /// <summary>オリジナルシェイプハッシュ。</summary>
            OriginalShapeHash,
            /// <summary>オリジナルシェイプ数。</summary>
            OriginalShapeCount,
            /// <summary>近似頂点削除モード。</summary>
            DeleteNearVertexMode,
            /// <summary>サブメッシュ分割モード。</summary>
            DivideSubmeshMode,
            /// <summary>プリミティブ最適化モード。</summary>
            OptimizePrimitiveMode,
            /// <summary>一つの頂点に影響を与えるボーン数。</summary>
            VertexSkinningCount,
            /// <summary>メッシュ(頂点インデックス)のプリミティブ。</summary>
            MeshMode,
            /// <summary>メッシュ(頂点インデックス)の型。</summary>
            MeshQuantizeType,
            /// <summary>メッシュ(頂点インデックス)数。</summary>
            MeshCount,
            /// <summary>メッシュ(頂点インデックス)の実データを格納する stream のインデックス。</summary>
            MeshStreamIndex,
            /// <summary>オリジナルボーン名</summary>
            OriginalBoneName,
            /// <summary>オリジナルマテリアル名</summary>
            OriginalMaterialName,
        }
        #endregion
    }
}
