﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Windows.Forms;
using App.Controls;
using App.Data;
using App.Utility;
using ConfigCommon;

using AppConfig;

namespace App.ObjectView.List
{
    using System.Linq;

    using App.ConfigData;

    using MainFrame = App.MainFrame;
    using ObjectView = AppConfig.ObjectView;

    /// <summary>
    /// 列項目設定ダイアログクラス。
    /// </summary>
    public sealed partial class ColumnSettingDialog : OkCancelDialog
    {
        private readonly List<ViewID> viewIDTable = new List<ViewID>();
        private AppConfig.ObjectView objectViewSettings = new ObjectView(false);

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ColumnSettingDialog()
        {
            InitializeComponent();

            DialogUtility.SetHelp(this, HelpUtility.PageKey.p_display_menu_a_view_list_dialog);

            tabSetting.ImageList = new ImageList();
            // タブページ作成
            foreach (ViewID viewID in (ViewID[])Enum.GetValues(typeof(ViewID)))
            {
                //if (viewID.ToString().Contains("Deprecated") == false)
                tabSetting.TabPages.Add(new SettingTabPage(viewID, tabSetting.ImageList));
                viewIDTable.Add(viewID);
            }
            if (viewIDTable.Contains(InitialPageViewID))
            {
                tabSetting.SelectedIndex = viewIDTable.IndexOf(InitialPageViewID);
            }

            // キャンセルした時にもとに戻す情報を保存
            objectViewSettings.BoneListView = ApplicationConfig.Setting.ObjectView.BoneListView.Clone();
            objectViewSettings.MaterialListView = ApplicationConfig.Setting.ObjectView.MaterialListView.Clone();
            objectViewSettings.ModelListView = ApplicationConfig.Setting.ObjectView.ModelListView.Clone();
            objectViewSettings.ShapeListView = ApplicationConfig.Setting.ObjectView.ShapeListView.Clone();
            objectViewSettings.TextureListView = ApplicationConfig.Setting.ObjectView.TextureListView.Clone();
        }

        /// <summary>
        /// 初期ページ番号。
        /// </summary>
        public static ViewID InitialPageViewID { get; set; }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnFormClosed(FormClosedEventArgs e)
        {
            // 初期ページ番号保存
            if (0 < tabSetting.SelectedIndex && tabSetting.SelectedIndex < viewIDTable.Count)
            {
                InitialPageViewID = viewIDTable[tabSetting.SelectedIndex];
            }
            base.OnFormClosed(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool OnResultOk()
        {
            // 無効な設定がされていないかをチェック
            foreach (SettingTabPage page in tabSetting.TabPages)
            {
                if (page.SettingPanel.IsInvalidateState)
                {
                    UIMessageBox.Warning(string.Format(App.res.Strings.ListView_ColumnSetting_NoSelection, page.Text));
                    return false;
                }
            }

            // 確定処理
            foreach (SettingTabPage page in tabSetting.TabPages)
            {
                page.SettingPanel.OnResultOk();
            }

            return true;
        }

        protected override bool OnResultCancel()
        {
            // キャンセルした時にはもとに戻す
            ApplicationConfig.Setting.ObjectView.BoneListView = objectViewSettings.BoneListView;
            ApplicationConfig.Setting.ObjectView.MaterialListView = objectViewSettings.MaterialListView;
            ApplicationConfig.Setting.ObjectView.ModelListView = objectViewSettings.ModelListView;
            ApplicationConfig.Setting.ObjectView.ShapeListView = objectViewSettings.ShapeListView;
            ApplicationConfig.Setting.ObjectView.TextureListView = objectViewSettings.TextureListView;
            BoneListView.InitializeColumn();
            MaterialListView.InitializeColumn();
            ModelListView.InitializeColumn();
            ShapeListView.InitializeColumn();
            TextureListView.InitializeColumn();

            foreach (var id in Enum.GetValues(typeof(ViewID)))
            {
                var columnInfo = ObjectListView.GetColumnInfo((ViewID)id);
                if (columnInfo != null)
                {
                    columnInfo.NotifySettingChanged();
                }
            }

            return base.OnResultCancel();
        }

        //---------------------------------------------------------------------
        // 内部クラス
        //---------------------------------------------------------------------
        #region SettingTabPage
        /// <summary>
        /// 設定タブページクラス。
        /// </summary>
        private sealed class SettingTabPage : TabPage
        {
            // 設定パネル
            private readonly ColumnSettingPanel _settingPanel = null;

            /// <summary>
            /// コンストラクタ。
            /// </summary>
            public SettingTabPage(ViewID viewID, ImageList imageList)
            {
                // 設定パネル作成
                _settingPanel = new ColumnSettingPanel(viewID);
                _settingPanel.Parent = this;
                _settingPanel.Dock   = DockStyle.Fill;

                // テキストとアイコン
                GuiObjectID objectID = (GuiObjectID)0;
                switch (viewID)
                {
                    case ViewID.Model:			objectID = GuiObjectID.Model;			break;
                    case ViewID.Texture: objectID = GuiObjectID.Texture; break;
                    case ViewID.Shape: objectID = GuiObjectID.Shape; break;
                    case ViewID.Bone: objectID = GuiObjectID.Bone; break;
                    case ViewID.Material: objectID = GuiObjectID.Material; break;
                    default: Debug.Assert(false); break;
                }
                Text       = UIText.EnumValue(objectID);
                ImageIndex = imageList.Images.Count;
                imageList.Images.Add(GuiObject.GetObjectIcon(objectID));
            }

            /// <summary>
            /// 設定パネル。
            /// </summary>
            public ColumnSettingPanel SettingPanel
            {
                get { return _settingPanel; }
            }
        }
        #endregion

        private void btnColumnInfoSave_Click(object sender, EventArgs e)
        {
            // ダイアログ状態をconfigに反映
            var objectView = new AppConfig.ObjectView();

            foreach (SettingTabPage page in tabSetting.TabPages)
            {
                var columns = new List<AppConfig.Column>();
                foreach (var item in page.SettingPanel.ListViewItems)
                {
                    if (!item.Deleted)
                    {
                        var column = ((AppConfig.Column)item.Tag).Clone();
                        column.Show = item.Checked;
                        columns.Add(column);
                    }
                }
                switch (page.SettingPanel.ViewID)
                {
                    case ViewID.Bone:
                        objectView.BoneListView.Columns = columns;
                        break;
                    case ViewID.Material:
                        objectView.MaterialListView.Columns = columns;
                        break;
                    case ViewID.Model:
                        objectView.ModelListView.Columns = columns;
                        break;
                    case ViewID.Shape:
                        objectView.ShapeListView.Columns = columns;
                        break;
                    case ViewID.Texture:
                        objectView.TextureListView.Columns = columns;
                        break;
                }
            }

            MainFrame.SaveColumnInfos(objectView);
        }

        private void btnColumnInfoLoad_Click(object sender, EventArgs e)
        {

            var objectView = MainFrame.LoadColumnInfos();
            if (objectView == null)
            {
                return;
            }
            ApplicationConfig.Setting.ObjectView.BoneListView = objectView.BoneListView;
            ApplicationConfig.Setting.ObjectView.MaterialListView = objectView.MaterialListView;
            ApplicationConfig.Setting.ObjectView.ModelListView = objectView.ModelListView;
            ApplicationConfig.Setting.ObjectView.ShapeListView = objectView.ShapeListView;
            ApplicationConfig.Setting.ObjectView.TextureListView = objectView.TextureListView;
            BoneListView.InitializeColumn();
            MaterialListView.InitializeColumn();
            ModelListView.InitializeColumn();
            ShapeListView.InitializeColumn();
            TextureListView.InitializeColumn();

            foreach (SettingTabPage page in tabSetting.TabPages)
            {
                page.SettingPanel.SetModified();
                page.SettingPanel.InitializeListView();
                page.SettingPanel.UpdateItemListView();
            }
        }
    }
}
