﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Runtime.InteropServices;
using System.Text;
using System.Threading;
using System.Windows.Forms;
using App.Command;
using App.ConfigData;
using App.Controls;
using App.Data;
using App.ObjectView;
using App.ObjectView.Schematic;
using App.PropertyEdit;
using App.Utility;
using App.res;
using ConfigCommon;
using nw.g3d.nw4f_3dif;
using System.IO;
using System.Xml;
using System.Xml.Serialization;
using App.ObjectView.List;
using AppConfig;

namespace App
{
    using App.FileView;

    public partial class MainFrame : UIForm
    {
        /// <summary>
        /// ステータスメッセージ。
        /// </summary>
        public string StatusMessage
        {
            set
            {
                statusMessage = value;
                if (!showingPriorStatusMessage)
                {
                    tsiStatus.Text = value;
                }
            }
        }

        public bool UiControlEnabled
        {
            set
            {
                // 別スレッドからの呼び出し用
                if (Thread.CurrentThread != TheApp.MainThread && TheApp.MainFrame != null && TheApp.MainFrame.IsHandleCreated)
                {
                    //var resetEvent = new ManualResetEventSlim();
                    {
                        TheApp.MainFrame.BeginInvoke(
                            new MethodInvoker(
                                () =>
                                {
                                    if (pnlControlBase.Enabled != value)
                                    {
                                        pnlControlBase.Enabled = value;
                                    }
                                    //resetEvent.Set();
                                }
                            )
                        );
                    }
                    //resetEvent.Wait();
                }
                else
                {
                    if (pnlControlBase.Enabled != value)
                    {
                        pnlControlBase.Enabled = value;
                    }
                }
            }
        }

        /// <summary>
        /// 優先ステータスメッセージ
        /// </summary>
        public string PriorStatusMessage
        {
            set
            {
                if (value == string.Empty)
                {
                    showingPriorStatusMessage = false;
                    tsiStatus.Text = statusMessage;
                }
                else
                {
                    showingPriorStatusMessage = true;
                    tsiStatus.Text = value;
                }
            }
        }

        public UIListView LogList{ get{ return lvwLogList; } }

        //private bool IsUiViewerTargetCafe { get { return (ConfigCommon.HioTarget)tsiViewerTarget.Tag == ConfigCommon.HioTarget.CatDev; } }

        private string statusMessage = string.Empty;
        private bool showingPriorStatusMessage = false;

        private readonly ObjectViewClient objectViewClient_ = null;
        public ObjectViewClient ObjectViewClient{ get{ return objectViewClient_; } }

        public NintendoWare.G3d.Edit.PlayPolicy	PlayPolicy  { get { return plcPlay.PlayPolicy;    } }
        public double FrameScale { get { return plcPlay.FrameScale; } }

        public void SetFrameScale(float value)
        {
            plcPlay.SetFrameScaleExternally(value);
        }

        public double							CurrentFrame { get { return plcPlay.CurrentFrame; } }
        public float FrameCount { get { return plcPlay.FrameCount; } }

        public void UpdateCurrentFrame()	{ plcPlay.UpdateCurrentFrame();	}
        public void SendCurrentFrame()		{ plcPlay.SendCurrentFrame();	}
        public void PlayAnimation()         { plcPlay.PlayClick();          }
        public void PauseAnimation()		{ plcPlay.Pause();				}

        public double PreviewCurrentFrame
        {
            get { return plcPlay.CurrentFrame;  }
            set { plcPlay.CurrentFrame = value; }
        }

        public bool IsEdittingCurrentFrame
        {
            get { return plcPlay.IsTimeSliderDragging;  }
            set { plcPlay.IsTimeSliderDragging = value; }
        }

        private MainFrameContextMenuItems treeViewMenu_ = null;

        internal MainFrameContextMenuItems TreeViewMenu
        {
            get
            {
                if (treeViewMenu_ == null)
                {
                    treeViewMenu_ = new MainFrameContextMenuItems();
                }
                return treeViewMenu_;
            }
        }

        /// <summary>
        /// フォームを閉じるときにドキュメントの保存確認を行うかどうか
        /// </summary>
        private bool documentSavingConfirmation_ = true;
        public bool DocumentSavingConfirmation
        {
            get
            {
                return documentSavingConfirmation_;
            }
            set
            {
                documentSavingConfirmation_ = value;
            }
        }

        public MainFrame()
        {
            using (var watch = new DebugStopWatch("MainFrame.MainFrame() : A"))
            {
                InitializeComponent();
            }

            using (var watch = new DebugStopWatch("MainFrame.MainFrame() : A1"))
            {
                objectViewClient_ = new ObjectViewClient(){ Dock = System.Windows.Forms.DockStyle.Fill };
            }

            ActiveControl = FileViewPanel;

            using (var watch = new DebugStopWatch("MainFrame.MainFrame() : B"))
            {
                spcClient.Panel2.Controls.Add(objectViewClient_);
            }

            // コンフィグから設定
            using (var watch = new DebugStopWatch("MainFrame.MainFrame() : C"))
            {
                SetFromConfig();
            }

            tsiViewerTarget.Tag			= ApplicationConfig.Setting.Preview.Target;
            mniViewerTargetPc.Tag		= ConfigCommon.HioTarget.Pc;
            mniViewerTargetCatDev.Tag	= ConfigCommon.HioTarget.Device;
            tsiViewerTarget.Items.Add(res.Strings.Viewer_Target_PC);
            tsiViewerTarget.Items.Add(res.Strings.Viewer_Target_Device);

            // シェーダー自動生成の設定
            cbxAutomaticShaderRegeneration.SelectedIndex = (int)DocumentManager.AutomaticShaderRegenerationID.MaterialSetting;

            // メニューコマンド初期化
            {
                // 初期化までメニューを無効化する。
                // 初期化時に無効化直前の値に復元する。
                // 初期化までの間に Enabled を変更しても復元によって上書きされるので注意。
                var targets = Menus.Select(x => Tuple.Create(x, x.Enabled)).ToArray();
                foreach (var target in targets)
                {
                    target.Item1.Enabled = false;
                }

                // 初期化までの間に Enabled を意図的に変更する可能性もあるので、変更を監視する。
                EventHandler enabledChanged = (ss, ee) =>
                {
                    var toolStrip = (ToolStrip)ss;
                    if (toolStrip.Enabled)
                    {
                        // 初期化まで有効化できない。
                        Debug.Assert(false);

                        // 無効に戻す。
                        toolStrip.Enabled = false;
                    }
                };
                foreach (var target in targets)
                {
                    target.Item1.EnabledChanged += enabledChanged;
                }

                // 初期化処理を登録する。
                OneShotIdleProcess.Execute(() =>
                {
                    // 監視のために登録したイベントを取り除く。
                    foreach (var target in targets)
                    {
                        target.Item1.EnabledChanged -= enabledChanged;
                    }

                    // Enabled を復元。
                    foreach (var target in targets)
                    {
                        target.Item1.Enabled = target.Item2;
                    }

                    // メニューコマンド初期化。
                    using (var watch = new DebugStopWatch("MainFrame.MainFrame() : D"))
                    {
                        InitializeMenuCommand();
                    }
                });
            }

            // ログリスト
            using (var watch = new DebugStopWatch("MainFrame.InitializeLogList() : D2"))
            {
                InitializeLogList();
            }

            // ヘルプ設定
            using (var watch = new DebugStopWatch("MainFrame.MainFrame() : E"))
            {
                DialogUtility.SetHelp(this, HelpUtility.PageKey.p_3deditor);
                DialogUtility.SetHelp(stsMain, HelpUtility.PageKey.p_main_menu);
                DialogUtility.SetHelp(tlpToolBar, HelpUtility.PageKey.p_toolbar);
                DialogUtility.SetHelp(tspFilter, HelpUtility.PageKey.p_toolbar);
                DialogUtility.SetHelp(sspMain, HelpUtility.PageKey.p_main_window);
                DialogUtility.SetHelp(spcClient.Panel2,	HelpUtility.PageKey.p_main_window);
                DialogUtility.SetHelp(plcPlay, HelpUtility.PageKey.p_main_window);
                DialogUtility.SetHelp(FileViewPanel, HelpUtility.PageKey.p_file_view);
            }

            using (var watch = new DebugStopWatch("MainFrame.MainFrame() : F"))
            {
                UpdateIcon();
                UpdateText();

                tsiHioConnection_SwitchConnection.Text = Viewer.Manager.Instance.IsConnected ? res.Strings.Viewer_PreviewConnectting : res.Strings.Viewer_PreviewDisconnectting;
            }

            using (var watch = new DebugStopWatch("MainFrame.MainFrame() : G"))
            {
                App.AppContext.PropertyChanged += (s, e) => UpdateText();

                // プロジェクトが読み込まれたらプレビュー情報を送る
                App.AppContext.DocumentAddedOrRemoved += (sender, added, removed, swaped, reloaded) =>
                {
                    if (added.Any(x => x.ObjectID == GuiObjectID.Project))
                    {
                        // afterModelLoad は true で OK
                        SendPreviewInfo(true);
                    }
                    else
                    {
                        //追加されたモデルとその子にプレビュー情報を送る
                        var addedModels = Model.SortByPreviewDepth(added.OfType<Model>(), true).ToList();
                        if (addedModels.Any())
                        {
                            foreach (var model in addedModels)
                            {
                                model.SendEditBoneBind();
                                model.SendEditModelLayout(true);
                            }
                        }

                    }
                };
            }
        }

        private bool updateIconFlag = false;

        public void UpdateIcon()
        {
            updateIconFlag = true;

            var platformPreset = AppContext.SelectedPlatformPreset;

            bool hasPc = platformPreset.DeviceOptions.Any(x => x.DeviceType == "Pc");
            mniViewerTargetPc.Enabled = hasPc;

            bool hasDevice = platformPreset.DeviceOptions.Any(x => x.DeviceType != "Pc");
            mniViewerTargetCatDev.Enabled = hasDevice;

            // プラットフォームに合わせてコンボボックスの選択肢を更新
            if (hasDevice)
            {
                tsiViewerTarget.Items.Clear();
                tsiViewerTarget.Items.Add(res.Strings.Viewer_Target_PC);
                tsiViewerTarget.Items.Add(res.Strings.Viewer_Target_Device);
            }
            else
            {
                tsiViewerTarget.Items.Clear();
                tsiViewerTarget.Items.Add(res.Strings.Viewer_Target_PC);
            }

            // コンボボックスの表示・メニュー側のチェックを更新
            if (ConfigData.ApplicationConfig.Setting.Preview.Target != HioTarget.Pc)
            {
                mniViewerTargetPc.Checked = false;
                mniViewerTargetCatDev.Checked = true;
                tsiViewerTarget.SelectedItem = res.Strings.Viewer_Target_Device;
            }
            else
            {
                mniViewerTargetPc.Checked = true;
                mniViewerTargetCatDev.Checked = false;
                tsiViewerTarget.SelectedItem = res.Strings.Viewer_Target_PC;
            }

            updateIconFlag = false;
        }

        private void UpdateText()
        {
            StringBuilder builder = new StringBuilder();
            builder.Append("3DEditor - ");
            builder.Append(DocumentManager.ProjectDocument.Name);
            if (!string.IsNullOrEmpty(DocumentManager.ProjectDocument.FileLocation))
            {
                builder.Append(" - ");
                builder.Append(DocumentManager.ProjectDocument.FilePath);
            }
            Text = builder.ToString();
        }

        private bool SettingFromConfig;
        private void SetFromConfig()
        {
            SettingFromConfig = true;
            DesktopLocation					= new Point(ApplicationConfig.Setting.MainFrame.X, ApplicationConfig.Setting.MainFrame.Y);
            Size							= new Size(ApplicationConfig.Setting.MainFrame.Width, ApplicationConfig.Setting.MainFrame.Height);
            WindowState						= ApplicationConfig.Setting.MainFrame.Maximized ? FormWindowState.Maximized : FormWindowState.Normal;

            // 最初にアクティブになった時にサイズを調整する。
            EventHandler activated = null;
            activated = (s, e) =>
            {
                tspFilter.Visible = ApplicationConfig.Setting.MainFrame.ToolBar;
                sspMain.Visible = ApplicationConfig.Setting.MainFrame.StatusBar;
                ctlLogFolder.Visible = ApplicationConfig.Setting.MainFrame.LogBar;
                spcLog.SplitterDistance = ApplicationConfig.Setting.MainFrame.LogListHeight;
                spcClient.SplitterDistance = ApplicationConfig.Setting.MainFrame.FileViewWidth;
                FileViewPanel.SplitterDistance = ApplicationConfig.Setting.MainFrame.FileViewHeight;
                spcLog.Panel2Collapsed = ApplicationConfig.Setting.MainFrame.LogListCollapsed;
                Activated -= activated;
            };
            Activated += activated;

            DocumentManager.OptimizeShader = ApplicationConfig.Setting.Preview.OptimizeShader;

            foreach (var path in ApplicationConfig.Setting.MainFrame.MostRecentlyUsedFiles.Reverse())
            {
                AddMostRecentlyUsedFile(path, true, ApplicationConfig.UserSetting.IO.MaximumRecentlyUsedFileCount);
            }

            foreach (var path in ApplicationConfig.Setting.MainFrame.MostRecentlyUsedProjects.Reverse())
            {
                AddMostRecentlyUsedFile(path, false, ApplicationConfig.UserSetting.IO.MaximumRecentlyUsedProjectCount);
            }

            OneShotIdleProcess.Execute(() =>
            {
                G3dHioLibProxy.Hio.IsAutoConnection = ApplicationConfig.Setting.Preview.IsAutoConnection;
            });

            // 画面外ならリセット
            if (!ObjectPropertyDialog.InScreen(DesktopLocation, Size))
            {
                Location = new Point(0, 0);
            }

            if (ApplicationConfig.Preset.ShaderParameterFilters.Any())
            {
                int index = 0;

                // 幅を計算
                int width = tcbShaderParameterFilter.Width;
                foreach (var filter in ApplicationConfig.Preset.ShaderParameterFilters)
                {
                    tcbShaderParameterFilter.Items.Add(filter.Name);

                    // AAA で少し余裕を持たせる。
                    width = Math.Max(TextRenderer.MeasureText(filter.Name + "AAA", tcbShaderParameterFilter.Font).Width, width);
                    if (filter.Name == ApplicationConfig.Setting.MainFrame.ShaderParameterFilter)
                    {
                        index = tcbShaderParameterFilter.Items.Count - 1;
                    }
                }

                tcbShaderParameterFilter.Width = width;
                tcbShaderParameterFilter.SelectedIndex = index;
                tcbShaderParameterFilter.Visible = true;
            }

            if (!ApplicationConfig.PlatformPresets.Any(x => x.Name == ApplicationConfig.Setting.MainFrame.SelectedSpec))
            {
                ApplicationConfig.Setting.MainFrame.SelectedSpec = ApplicationConfig.PlatformPresets.First().Name;
            }

            foreach (var spec in ApplicationConfig.PlatformPresets)
            {
                cbxPlatform.Items.Add(spec.Name);
            }

            SelectPlatform(ApplicationConfig.Setting.MainFrame.SelectedSpec);

            // プラットフォーム設定にしたがって必要なら接続先を変更
            if (ApplicationConfig.Setting.Preview.Target == HioTarget.Pc)
            {
                if (!App.AppContext.SelectedPlatformPreset.DeviceOptions.Any(x => x.DeviceType == "Pc"))
                {
                    ApplicationConfig.Setting.Preview.Target = HioTarget.Device;
                }
            }
            else
            {
                if (!App.AppContext.SelectedPlatformPreset.DeviceOptions.Any(x => x.DeviceType != "Pc"))
                {
                    ApplicationConfig.Setting.Preview.Target = HioTarget.Pc;
                }
            }

            Viewer.Connecter.Platform = App.AppContext.SelectedPlatformPreset;

            SettingFromConfig = false;
        }

        public void SelectPlatform(string platform)
        {
            cbxPlatform.SelectedItem = platform;
        }

        private void SaveConfig()
        {
            // 位置情報は this.Location で取得すると最大化されている場合に
            // 通常時のサイズが取れないので、GetWindowPlacement() で取得する
            Win32.WINDOWPLACEMENT wp = new Win32.WINDOWPLACEMENT();
            wp.length = Marshal.SizeOf(wp);
            Win32.NativeMethods.GetWindowPlacement(Handle, ref wp);

            ApplicationConfig.Setting.MainFrame.X					= wp.rcNormalPosition.left;
            ApplicationConfig.Setting.MainFrame.Y					= wp.rcNormalPosition.top;
            ApplicationConfig.Setting.MainFrame.Width				= wp.rcNormalPosition.right - wp.rcNormalPosition.left;
            ApplicationConfig.Setting.MainFrame.Height				= wp.rcNormalPosition.bottom - wp.rcNormalPosition.top;
            ApplicationConfig.Setting.MainFrame.Maximized			= WindowState == FormWindowState.Maximized;
            ApplicationConfig.Setting.MainFrame.ToolBar				= tspFilter.Visible;
            ApplicationConfig.Setting.MainFrame.StatusBar			= sspMain.Visible;
            ApplicationConfig.Setting.MainFrame.LogBar				= ctlLogFolder.Visible;
            ApplicationConfig.Setting.MainFrame.FileViewWidth		= spcClient.SplitterDistance;
            ApplicationConfig.Setting.MainFrame.FileViewHeight		= FileViewPanel.SplitterDistance;
            ApplicationConfig.Setting.MainFrame.LogListHeight		= spcLog.SplitterDistance;
            ApplicationConfig.Setting.MainFrame.LogListCollapsed	= spcLog.Panel2Collapsed;

            recentlyUsedFiles.Capacity = ApplicationConfig.UserSetting.IO.MaximumRecentlyUsedFileCount;
            ApplicationConfig.Setting.MainFrame.MostRecentlyUsedFiles = recentlyUsedFiles.ToArray();
            recentlyUsedFiles.Capacity = ApplicationConfig.UserSetting.IO.MaximumRecentlyUsedProjectCount;
            ApplicationConfig.Setting.MainFrame.MostRecentlyUsedProjects = recentlyUsedProjects.ToArray();

            ApplicationConfig.Setting.Preview.IsAutoConnection = G3dHioLibProxy.Hio.IsAutoConnection;
            ApplicationConfig.Setting.Preview.OptimizeShader = DocumentManager.OptimizeShader;
            ApplicationConfig.Setting.MainFrame.ShaderParameterFilter = tcbShaderParameterFilter.SelectedItem as string;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnClosing(CancelEventArgs e)
        {
            base.OnClosing(e);

            if (DocumentSavingConfirmation)
            {
                if (DocumentManager.CheckAndSaveModified(DocumentManager.Documents) == false)
                {
                    e.Cancel = true;
                }
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnClosed(EventArgs e)
        {
            SaveConfig();

            // 残っているドキュメントすべてをクローズ後コマンドに渡す
            DocumentManager.ExecutePostCloseCommand(DocumentManager.Documents.ToList());

            base.OnClosed(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnDragEnter(DragEventArgs e)
        {
            base.OnDragEnter(e);

            var effect = DragDropEffects.None;

            // ファイルであり、モーダルでないかファイルダイアログが開いている時はドラッグＯＫ
            if (e.Data.GetDataPresent(DataFormats.FileDrop) && pnlControlBase.Enabled)
            {
                if (!UIForm.IsModalState(this) || DialogUtility.IsOpened)
                {
                    effect = DragDropEffects.Copy;
                }
            }

            e.Effect = effect;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnDragDrop(DragEventArgs e)
        {
            // メッセージボックスを出すことを考慮してアクティブ化しておく
            Activate();

            // ドラッグドロップ処理
            string[] paths = (string[])e.Data.GetData(DataFormats.FileDrop);
            if (paths != null)
            {
                // 描画更新
                Update();

                // ファイルを開く
                if (DialogUtility.IsOpened)
                {
                    DocumentManager.LoadFromFileOrDirectory(paths);
                }
                else
                {
                    OneShotIdleProcess.Execute(() => DocumentManager.LoadFromFileOrDirectory(paths));
                }
            }

            base.OnDragDrop(e);
        }


        public event Action OnActivatedActions;
        protected override void WndProc(ref Message m)
        {
            if (Bridge.BridgeManager.WndProc(ref m))
            {
                return;
            }

            if (m.Msg == Win32.WM.WM_ACTIVATEAPP)
            {
                if (m.WParam != (IntPtr)0)
                {
                    if (OnActivatedActions != null)
                    {
                        EventHandler idle = null;
                        idle = (s, e) =>
                        {
                            Application.Idle -= idle;
                            OnActivatedActions();
                            OnActivatedActions = null;
                        };
                        Application.Idle += idle;
                    }
                }
            }

            base.WndProc(ref m);
        }

        /// <summary>
        /// コマンドキー処理。
        /// </summary>
        public bool ProcessCommandKey(ref Message msg, Keys keyData)
        {
            // HWnd を偽装してメインフレーム外からショートカットを処理する
            // 処理後には忘れずに HWnd を戻しておく事！
            IntPtr hWnd = msg.HWnd;
            msg.HWnd = Handle;
            bool result = ProcessCmdKey(ref msg, keyData);
            msg.HWnd = hWnd;

            return result;
        }

        protected bool IsRenaming()
        {
            return App.AppContext.FileTreeView.LabelEdit;
        }

        protected override bool ProcessCmdKey(ref Message msg, Keys keyData)
        {
            // マウスをキャプチャ中は処理しない
            if (ControlUtility.GetCapturingControl() != null)
            {
                return false;
            }

            if (base.ProcessCmdKey(ref msg, keyData))
            {
                return true;
            }

            bool isEnableShortcut = true;

            var activeForm				= Form.ActiveForm;
            var focusedActiveControl	= (activeForm != null) ? ControlUtility.GetFocusedActiveControl(activeForm) : null;

            // UIToolStripComboBox 内の実体クラスをいじれないので直接見る
            var comoBoxCanKeyEdit = false;
            {
                var comboBox = focusedActiveControl as ComboBox;
                if (comboBox != null)
                {
                    comoBoxCanKeyEdit = comboBox.DropDownStyle != ComboBoxStyle.DropDownList;
                }
            }

            if (
                (activeForm != null) &&
                (
                    comoBoxCanKeyEdit ||
                    (focusedActiveControl is App.Controls.KeyEditableControl) && ((focusedActiveControl as  App.Controls.KeyEditableControl).CanKeyEdit)
                )
            )
            {
                isEnableShortcut = false;
            }

            if(isEnableShortcut)
            {
                // FileTreeViewでアイテムがリネームされている時は、キー入力を奪わないようにする。
                // モーダルダイアログ表示中はキー入力を奪わないようにする
                if (!IsRenaming() && CanFocus)
                {
                    // Altキーを含むショートカットは別処理にしておく
                    if (msg.Msg == Win32.WM.WM_SYSKEYDOWN)
                    {
                        switch (keyData)
                        {
                            case Keys.OemPeriod | Keys.Alt:
                                if (plcPlay.FrameControlEnabled)
                                {
                                    plcPlay.Forward();
                                    return true;
                                }
                                break;
                            case Keys.Oemcomma | Keys.Alt:
                                if (plcPlay.FrameControlEnabled)
                                {
                                    plcPlay.Back();
                                    return true;
                                }
                                break;
                            default:
                                if (UserCommandUtil.ProcessShortCutKey(keyData)) return true;
                                break;
                        }
                    }
                    else if (msg.Msg == Win32.WM.WM_KEYDOWN)
                    {
                        switch(keyData)
                        {
                            case Keys.V:
                                if (plcPlay.FrameControlEnabled)
                                {
                                    if (plcPlay.IsPlaying)
                                    {
                                        plcPlay.Pause();
                                    }
                                    else
                                    {
                                        plcPlay.PlayClick();
                                    }
                                    return true;
                                }
                                break;
                            case Keys.V | Keys.Shift:
                                if (plcPlay.FrameControlEnabled)
                                {
                                    plcPlay.Stop();
                                    return true;
                                }
                                break;
                            case Keys.Z:
                                {
                                    // ZキーのみでもUndoを行う。
                                    if (TheApp.CommandManager.CanUndo())
                                    {
                                        TheApp.CommandManager.Undo();
                                    }
                                    return true;
                                }
                            case (Keys.Control | Keys.Shift | Keys.W):
                                {
                                    // すべてを閉じるのショートカットが変更されたが、古いショートカットも有効にしておく
                                    var args = new MenuCommandArgs(false, null);
                                    Command_FileCloseAll(args);
                                    return true;
                                }
                            default:
                                if (UserCommandUtil.ProcessShortCutKey(keyData)) return true;
                                break;
                        }
                    }
                }
            }

            return false;
        }

        /// <summary>
        /// オブジェクトメニュー表示。
        /// </summary>
        public void ShowObjectMenu(Point point, IObjectView view)
        {
            var menu = menuCommandHost_.CreateGeneralContextMenu();
            menu.IsObjectViewContextMenu = true;
            SetEditMenu(menu, view);

            if (view is ObjectSchematicView)
            {
                menu.Items.Add(TreeViewMenu.cmiSchematic);
                TreeViewMenu.cmiCopyNames.Tag = view;
                menu.Items.Add(TreeViewMenu.cmiCopyNames);
            }
            else if (view is MaterialListView || view is ModelListView || view is BoneListView || view is ShapeListView || view is TextureListView)
            {
                TreeViewMenu.cmiCopyNames.Tag = view;
                menu.Items.Add(TreeViewMenu.cmiCopyNames);
            }

            if (App.AppContext.SelectedTarget.Active != null)
            {
                switch(App.AppContext.SelectedTarget.Active.ObjectID)
                {
                    case GuiObjectID.Model:
                    {
                        menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());
                        CreateModelMenu(App.AppContext.SelectedTarget.Active as Model, target: menu);
                        break;
                    }

                    case GuiObjectID.Material:
                    {
                        CreateObjectUnderModelMenu(menu);
                        break;
                    }

                    case GuiObjectID.Bone:
                    {
                        CreateObjectUnderModelMenu(menu);
                        break;
                    }

                    case GuiObjectID.Shape:
                    {
                        CreateObjectUnderModelMenu(menu);
                        break;
                    }

                    case GuiObjectID.Texture:
                    {
                        menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());
                        CreateTextureMenu(menu);
                        break;
                    }

                    default:
                    {
                        // アニメーションなど
                        break;
                    }
                }
                menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());
                SetViewPropertyMenuItem(menu);
            }
            menu.Show(point);
        }

        public void ShowFileTreeViewContextMenu(UIContextMenuStrip target, Point point)
        {
            target.IsObjectViewContextMenu = false;
            target.Show(point);
        }

        private void SetViewPropertyMenuItem(UIContextMenuStrip target)
        {
            target.Items.Add(TreeViewMenu.cmiViewProperty);
        }

        private void SetEditMenu(UIContextMenuStrip menu, IObjectView objectView)
        {
            TreeViewMenu.cmiSelectAll.Tag = objectView;
            TreeViewMenu.cmiToggleSelection.Tag = objectView;
            menu.Items.Add(TreeViewMenu.cmiEdit);
        }

        private void mniEditUndo_MenuCommandHandler(MenuCommandArgs args)
        {
            List<Document> documents =
                args.RequireExecute ? DocumentManager.Documents.ToList() : null;

            TheApp.CommandManager.Command_EditUndo(args);

            if (args.RequireExecute)
            {
                // コマンドによりファイルが更新されている場合があるので、再読み込みが必要か確認する
                CompareFileTimestamp();
                // ビューから消えたファイルにクローズ後コマンドを行う
                PostCloseDifferenceDocument(documents);
            }
        }

        private void mniEditRedo_MenuCommandHandler(MenuCommandArgs args)
        {
            List<Document> documents =
                args.RequireExecute ? DocumentManager.Documents.ToList() : null;

            TheApp.CommandManager.Command_EditRedo(args);

            if (args.RequireExecute)
            {
                // コマンドによりファイルが更新されている場合があるので、再読み込みが必要か確認する
                CompareFileTimestamp();
                // ビューから消えたファイルにクローズ後コマンドを行う
                PostCloseDifferenceDocument(documents);
            }
        }

        private void CompareFileTimestamp()
        {
            Func<Document, bool> needsReload = (document) =>
            {
                return (document != null) && document.NeedsReload;
            };

            // 現在の全ドキュメントからリロードが必要な必要なものを集める。
            // 元々はファイルの変更日時の同値上書きによってファイル変更イベントを強制的に発生させる間接的な再読み込みが行われていたが、
            // 下記理由により直接再読み込みを行うように変更。
            // - アプリ都合による外部データの変更は避けるべき
            // - 読み取り専用属性が有効であれば更新日時の上書きは不可
            var updatedDocuments = new List<Document>();
            foreach (var document in DocumentManager.Documents)
            {
                if (needsReload(document))
                {
                    updatedDocuments.Add(document);
                }
            }

            // 更新されたファイルを再読み込みする。
            // ただし、ファイル監視イベントによる遅延再読み込みは意図的だったかもしれないので BeginInvoke() を使う。
            Action<Document> reload = (document) =>
            {
                // BeginInvoke() で処理されるので、更新の要否を都度確認する。
                if (!DocumentManager.Documents.Contains(document))
                {
                    document = DocumentManager.Documents.FirstOrDefault(x => x.FilePath == document.FilePath);
                }
                if (needsReload(document))
                {
                    document.LazyReload();
                }
            };
            foreach (var document in updatedDocuments)
            {
                BeginInvoke(reload, document);
            }
        }

        // 現在との差分から消えたドキュメントにクロース後コマンドを行う
        private void PostCloseDifferenceDocument(List<Document> documents)
        {
            List<string> files = new List<string>();
            foreach (var document in documents)
            {
                if (!string.IsNullOrEmpty(document.FilePath))
                {
                    if (!DocumentManager.Documents.Any(x => x.FilePath == document.FilePath))
                    {
                        files.Add(document.FilePath);
                    }
                }
            }

            if (files.Any())
            {
                PrePostIO.ExecutePostClose(files.ToArray());
            }
        }

        /// <summary>
        /// コマンドハンドラ。
        /// </summary>
        private void Command_ViewLayout(MenuCommandArgs args, AppConfig.ObjectViewLayout viewLayout)
        {
            // 更新時
            if (args.RequireUpdate)
            {
                args.CommandUI.Checked = (objectViewClient_.ViewLayout == viewLayout);
            }
            // 実行時
            else
            {
                objectViewClient_.ViewLayout = viewLayout;
            }
        }

        private void mniViewLayoutSingle_MenuCommandHandler(MenuCommandArgs args)
        {
            Command_ViewLayout(args, AppConfig.ObjectViewLayout.Single);
        }

        private void mniViewLayoutMulti2Stacked_MenuCommandHandler(MenuCommandArgs args)
        {
            Command_ViewLayout(args, AppConfig.ObjectViewLayout.Multi2Stacked);
        }

        private void mniViewLayoutMulti2SideBySide_MenuCommandHandler(MenuCommandArgs args)
        {
            Command_ViewLayout(args, AppConfig.ObjectViewLayout.Multi2SideBySide);
        }

        private void mniViewLayoutMulti3SplitBottom_MenuCommandHandler(MenuCommandArgs args)
        {
            Command_ViewLayout(args, AppConfig.ObjectViewLayout.Multi3SplitBottom);
        }

        private void mniViewLayoutMulti3SplitTop_MenuCommandHandler(MenuCommandArgs args)
        {
            Command_ViewLayout(args, AppConfig.ObjectViewLayout.Multi3SplitTop);
        }

        private void mniViewLayoutMulti3SplitRight_MenuCommandHandler(MenuCommandArgs args)
        {
            Command_ViewLayout(args, AppConfig.ObjectViewLayout.Multi3SplitRight);
        }

        private void mniViewLayoutMulti3SplitLeft_MenuCommandHandler(MenuCommandArgs args)
        {
            Command_ViewLayout(args, AppConfig.ObjectViewLayout.Multi3SplitLeft);
        }

        private void mniViewLayoutMulti4_MenuCommandHandler(MenuCommandArgs args)
        {
            Command_ViewLayout(args, AppConfig.ObjectViewLayout.Multi4);
        }

        private void mniViewLayoutResetSplitRatio_MenuCommandHandler(MenuCommandArgs args)
        {
            // 更新時
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = (objectViewClient_.ViewLayout != AppConfig.ObjectViewLayout.Single);
            }
            // 実行時
            else
            {
                objectViewClient_.ResetMultiLayoutSplitRatio();
            }
        }

        public void FileOpen(MenuCommandArgs args, GuiObjectID id)
        {
            if (args.RequireExecute)
            {
                string[] fileNames;
                if (DialogUtility.ExecuteOpenFileDialog(out fileNames, id))
                {
                    DocumentManager.LoadFromFile(fileNames);
                }
            }
        }

        public void FilesClose(MenuCommandArgs args, IEnumerable<Document> documents)
        {
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = documents.Any();
                return;
            }

            var selected = App.AppContext.SelectedFileViewObjectOwner;
            var animationSet = selected is Model ? ((Model)selected).DefaultAnimationSet: selected as AnimationSet;
            DocumentManager.RemoveDocuments(documents, animationSet, true, false);
        }

        #region コマンドハンドラ
        #region

        #endregion
        #region FileOpen
        /// <summary>
        /// コマンドハンドラ。
        /// </summary>
        private void Command_FileOpen(MenuCommandArgs args)
        {
            if (args.RequireExecute)
            {
                string[] fileNames;
                if (DialogUtility.ExecuteOpenFileDialog(out fileNames))
                {
                    DocumentManager.LoadFromFile(fileNames);
                }
            }
        }
        #endregion

        #region ProjctOpen
        /// <summary>
        /// コマンドハンドラ。
        /// </summary>
        private void Command_ProjectOpen(MenuCommandArgs args)
        {
            if (args.RequireExecute)
            {
                string[] fileNames;
                if (DialogUtility.ExecuteOpenProjectDialog(out fileNames))
                {
                    DocumentManager.LoadFromFile(fileNames);
                }
            }
        }
        #endregion

        #region FileCloseAll
        /// <summary>
        /// コマンドハンドラ。
        /// </summary>
        private void Command_FileCloseAll(MenuCommandArgs args)
        {
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = DocumentManager.Documents.Any(x => !x.OpenedFromStartUp);
            }

            if (args.RequireExecute)
            {
                // ShaderDefinitionファイルが編集されている場合、
                // 閉じるかどうか？の確認を行う。
                var shaderDefinitions = DocumentManager.ShaderDefinitions;
                var modifiedShaderDefinition = shaderDefinitions.Where(x => x.IsModifiedObject && !x.OpenedFromStartUp);
                if (modifiedShaderDefinition.Any())
                {
                    string fileNames = string.Empty;
                    foreach (var shaderDefinition in modifiedShaderDefinition)
                    {
                        fileNames += "'" + shaderDefinition.FileName + "' ";
                    }
                    string msg = string.Format("{0}\n{1}", fileNames, Strings.IO_CloseModifiedShaderDifinitionFiles);
                    using(UIMessageBox msgBox = new UIMessageBox(msg, UIMessageBoxButtons.OKCancel, MessageBoxIcon.Information, MessageBoxDefaultButton.Button1, false))
                    {
                        if (msgBox.ShowDialog() != System.Windows.Forms.DialogResult.OK)
                        {
                            // 閉じないを選択した場合は、後続の処理も行わない。
                            return;
                        }
                    }
                }

                var commandSet = new EditCommandSet();
                // 既存ファイルの変更をチェックします。
                if (DocumentManager.CheckAndSaveModified(DocumentManager.Documents.Where(x => !x.OpenedFromStartUp)))
                {
                    DocumentManager.CloseAll(commandSet, false, true);
                }
                if (commandSet.CommandCount > 0)
                {
                    commandSet.Reverse();
                    TheApp.CommandManager.Add(commandSet);
                }
            }
        }
        #endregion

        #region FileExit
        /// <summary>
        /// コマンドハンドラ。
        /// </summary>
        private void Command_FileExit(MenuCommandArgs args)
        {
            if (args.RequireExecute)
            {
                Close();
            }
        }
        #endregion

        #region FileSave
        /// <summary>
        /// コマンドハンドラ。
        /// </summary>
        private void Command_FileSave(MenuCommandArgs args)
        {
            if (args.RequireUpdate)
            {
                // ドキュメントがロードされていれば有効
                args.CommandUI.Enabled = DocumentManager.Documents.Any();
                return;
            }

            // 実行
            var selectedObject = App.AppContext.CurrentSelectedObject(args);
            var document = selectedObject as Document;
            var documents = new List<Document>();

            if (document == null)
            {
                // シーン系アニメーションなら親を保存
                if (selectedObject != null)
                {
                    switch (selectedObject.ObjectID)
                    {
                        case GuiObjectID.CameraAnimation:
                        case GuiObjectID.LightAnimation:
                        case GuiObjectID.FogAnimation:
                            document = selectedObject.OwnerDocument;
                            break;
                    }
                }

                if (document == null)
                {
                    UIMessageBox.Warning(res.Strings.IO_SaveTargetNotFound);
                    return;
                }
            }

            if (args.CommandUI.IsObjectViewContextMenu)
            {
                documents.Add(document);
            }
            else
            {
                var selectedDocuments = App.AppContext.SelectedFileViewObjects?.OfType<Document>();
                if (selectedDocuments != null && selectedDocuments.Any())
                {
                    documents.AddRange(selectedDocuments);
                }
            }

            if (!documents.Any())
            {
                UIMessageBox.Warning(res.Strings.IO_SaveTargetNotFound);
                return;
            }


            if (document is ShaderDefinition)
            {
                UIMessageBox.Warning(res.Strings.IO_InvalidFileType, document.GetType().Name);
                return;
            }
            else if (document is Texture && documents.OfType<Texture>().Any(x => x.IsTemporary))
            {
                UIMessageBox.Warning(res.Strings.Texture_FileNotOutputErrorMessage, string.Join(", ", documents.OfType<Texture>().Where(x => x.IsTemporary).Select(x => x.Name)));
                return;
            }
            else if (document is ProjectDocument && (document as ProjectDocument).IsFileOutputable == false)
            {
                UIMessageBox.Warning(res.Strings.ProjectDocument_FileNotOutputErrorMessage, document.GetType().Name);
                return;
            }

            var saver = new DocumentSaver();

            var saved = saver.SaveDocuments(documents);

            if (saved && ApplicationConfig.UserSetting.EditHistory.ClearAfterFileSaved)
            {
                TheApp.CommandManager.Clear();
            }
        }
        #endregion

        #region FileSaveAs
        /// <summary>
        /// コマンドハンドラ。
        /// </summary>
        private void Command_FileSaveAs(MenuCommandArgs args)
        {
            if (args.RequireUpdate)
            {
                // ドキュメントがロードされていれば有効
                args.CommandUI.Enabled = DocumentManager.Documents.Any();
                return;
            }

            // 実行
            GuiObject selectedObject = App.AppContext.CurrentSelectedObject(args);
            Document document = selectedObject as Document;

            if (document == null)
            {
                // シーン系アニメーションなら親を保存
                if (selectedObject != null)
                {
                    switch (selectedObject.ObjectID)
                    {
                        case GuiObjectID.CameraAnimation:
                        case GuiObjectID.LightAnimation:
                        case GuiObjectID.FogAnimation:
                            document = selectedObject.OwnerDocument;
                            break;
                    }
                }

                if (document == null)
                {
                    UIMessageBox.Warning(res.Strings.IO_SaveTargetNotFound);
                    return;
                }
            }

            if (document is ShaderDefinition)
            {
                UIMessageBox.Warning(res.Strings.IO_InvalidFileType, document.GetType().Name);
                return;
            }
            else if (document is Texture && (document as Texture).IsTemporary)
            {
                UIMessageBox.Warning(res.Strings.Texture_FileNotOutputErrorMessage, document.GetType().Name);
                return;
            }
            else if (document is ProjectDocument && (document as ProjectDocument).IsFileOutputable == false)
            {
                UIMessageBox.Warning(res.Strings.ProjectDocument_FileNotOutputErrorMessage, document.GetType().Name);
                return;
            }

            // アニメーションの場合、一旦、プレビュー対象のアニメーションセットを取得する。
            List<Model> modelList = new List<Model>();
            if (document is AnimationDocument)
            {
                Model model = App.AppContext.SelectedFileViewObjectOwner as Model;
                if (model == null)
                {
                    modelList.AddRange(DocumentManager.Models.Where(x => x.AllAnimations.Contains(document)));
                }
                else
                {
                    modelList.Add(model);
                }
            }

            EditCommandSet commandSet = new EditCommandSet();
            DocumentSaver saver = new DocumentSaver();
            List<Document> savedDocuments = new List<Document>();

            // 要注意
            bool saved = saver.SaveDocumentAs(document);

            if (saved && ApplicationConfig.UserSetting.EditHistory.ClearAfterFileSaved)
            {
                TheApp.CommandManager.Clear();
            }


        }
        #endregion

        #region FileSaveAllTo
        /// <summary>
        /// コマンドハンドラ。
        /// </summary>
        internal void Command_FileSaveAllTo(MenuCommandArgs args)
        {
            var tgaFiles = new List<string>();
            var saveDocuments = new List<Document>();
            GetSaveDocuments(saveDocuments, tgaFiles);

            if (args.RequireUpdate)
            {
                // ドキュメントがロードされていれば有効
                args.CommandUI.Enabled = saveDocuments.Any();
                return;
            }
            if (tgaFiles.Any())
            {
                UIMessageBox.Warning(Strings.MainFrame_Command_TgaIsNotSaveTarget, string.Join(", ", tgaFiles));
            }
            (new DocumentSaver()).SaveWithChildren(saveDocuments);
        }

        internal static void GetSaveDocuments(List<Document> saveDocuments, List<string> tgaFiles)
        {
            {
                foreach (var document in DocumentManager.Documents)
                {
                    // 保存確認対象でないプロジェクトを弾く
                    if (document is ProjectDocument)
                    {
                        if ((document as ProjectDocument).DefaultProject == false)
                        {
                            saveDocuments.Add(document);
                        }
                    }
                    else if (document is ShaderDefinition)
                    {
                        // 保存対象でない
                        ;
                    }
                    else if (document is Texture && (document as Texture).IsTemporary)
                    {
                        // 保存対象でない
                        tgaFiles.Add(document.GetType().Name);
                    }
                    else
                    {
                        // 保存対象を追加する
                        saveDocuments.Add(document);
                    }
                }
            }
        }

        #endregion

        #region FileSaveAll
        /// <summary>
        /// コマンドハンドラ。
        /// </summary>
        private void Command_FileSaveAll(MenuCommandArgs args)
        {
            var saveDocuments = DocumentManager.GetAllSaveTargetDocuments();

            if (args.RequireUpdate)
            {
                // ドキュメントがロードされていれば有効
                args.CommandUI.Enabled = saveDocuments.Any();
                return;
            }

            var saver = new DocumentSaver();
            saver.SaveDocuments(saveDocuments);
        }
        #endregion

        #region HelpAbout

        #endregion

        private void mniViewToolBar_MenuCommandHandler(MenuCommandArgs args)
        {
            // 更新時
            if (args.RequireUpdate)
            {
                args.CommandUI.Checked = tspFilter.Visible;
            }
            // 実行時
            else
            {
                tspFilter.Visible = !tspFilter.Visible;
            }
        }


        private void mniViewStatusBar_MenuCommandHandler(MenuCommandArgs args)
        {
            // 更新時
            if (args.RequireUpdate)
            {
                args.CommandUI.Checked = sspMain.Visible;
            }
            // 実行時
            else
            {
                sspMain.Visible = !sspMain.Visible;
            }
        }

        private void mniViewLogBar_MenuCommandHandler(MenuCommandArgs args)
        {
            // 更新時
            if (args.RequireUpdate)
            {
                args.CommandUI.Checked = ctlLogFolder.Visible;
            }
            // 実行時
            else
            {
                ctlLogFolder.Visible = ! ctlLogFolder.Visible;
                spcLog.Panel2Collapsed = ! ctlLogFolder.Visible;
            }
        }

        #endregion

        private void mniHelpAbout_MenuCommandHandler(MenuCommandArgs args)
        {
            // 更新時
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = true;
            }
            // 実行時
            else
            {
                using (AboutDialog dialog = new AboutDialog())
                {
                    dialog.ShowDialog(this);
                }
            }
        }

        private void mniHelpHelp_MenuCommandHandler(MenuCommandArgs args)
        {
            // 更新時
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = true;
            }
            // 実行時
            else
            {
                DialogUtility.EmulatePushF1();
            }
        }

        private void mniCreateProject_MenuCommandHandler(MenuCommandArgs args)
        {
            // 更新時
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = true;
            }
            // 実行時
            else
            {
                var commandSet = new EditCommandSet();
                // 既存ファイルの変更をチェックします。
                if (DocumentManager.CheckAndSaveModified(DocumentManager.Documents))
                {
                    using (var dialog = new ProjectCreateDialog())
                    {
                        if (dialog.ShowDialog() == DialogResult.OK)
                        {
                            // 既存ファイルを閉じます。
                            if (dialog.CloseFiles)
                            {
                                DocumentManager.CloseAll(commandSet, true, true);
                            }

                            using (var block = new App.AppContext.PropertyChangedSuppressBlock())
                            {
                                //commandSet.Add(DocumentManager.CreatePathChangeCommand(DocumentManager.ProjectDocument, "", dialog.ProjectName, ProjectDocument.DefaultDotExt, null).Execute());
                                commandSet.Add(DocumentManager.CreateProjectChangeCommand(new ProjectDocument() { Name = dialog.ProjectName }, false).Execute());
                            }
                        }
                    }
                }

                if (commandSet.CommandCount > 0)
                {
                    commandSet.Reverse();
                    TheApp.CommandManager.Add(commandSet);
                }
            }
        }

        private void mniFileRename_MenuCommandHandler(MenuCommandArgs args)
        {
            var guiObject = App.AppContext.CurrentSelectedObject(args) as GuiObject;

            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = (guiObject != null) && guiObject.Renamable;		// 名前変更可なドキュメント

                return;
            }

            App.AppContext.BeginRename(args);
        }

        private void mniClearUndoStack_Click(MenuCommandArgs args)
        {
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = TheApp.CommandManager.CanUndo() || TheApp.CommandManager.CanRedo();
                return;
            }

            var gcTotalBefore = GC.GetTotalMemory(false);
            var worksetBefore = System.Environment.WorkingSet;

            TheApp.CommandManager.Clear();
            GC.Collect();

            var gcTotalAfter = GC.GetTotalMemory(false);
            var worksetAfter = System.Environment.WorkingSet;

            var mes = String.Format("ClearUndoStack before:{0:n0}bytes, after:{1:n0}bytes", gcTotalBefore, gcTotalAfter);
            DebugConsole.WriteLine(mes);

#if DEBUG
            mes += String.Format("\nWorkset before:{0:n0}bytes, after:{1:n0}bytes", worksetBefore, worksetAfter);
            MessageLog.Write(MessageLog.LogType.Information, mes);
#endif
        }

        private void mniViewListColumnSetting_MenuCommandHandler(MenuCommandArgs args)
        {
            // 更新時
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = true;
            }
            // 実行時
            else
            {
                using(var dialog = new App.ObjectView.List.ColumnSettingDialog())
                {
                    dialog.ShowDialog(this);
                }
            }
        }

        private void mniToolTeamConfig_MenuCommandHandler(MenuCommandArgs args)
        {
            // 更新時
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = true;
            }
            // 実行時
            else
            {
                using(var dialog = new App.Controls.TeamConfigSettingDialog())
                {
                    dialog.ShowDialog(this);
                }
            }
        }

        private bool IsReloadbumpShaderSource(bool bumpShaderSource)
        {
            // 接続状態で変更された
            if (G3dHioLibProxy.Hio.IsConnected)
            {
                // 出力のON・OFFが変更されている
                if (bumpShaderSource != ApplicationConfig.UserSetting.IO.DumpShaderSource)
                {
                    // 出力が有効に変更された
                    if (ApplicationConfig.UserSetting.IO.DumpShaderSource)
                    {
                        return true;
                    }
                }
            }

            return false;
        }

        private void ReloadbumpShaderSource()
        {
            foreach (var model in DocumentManager.Models)
            {
                model.ForcedOptimized = true;
                Viewer.LoadOrReloadModel.Send(model);
            }
        }

        private void mniToolUserConfig_MenuCommandHandler(MenuCommandArgs args)
        {
            // 更新時
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = true;
            }
            // 実行時
            else
            {
                using(var dialog = new App.Controls.UserConfigSettingDialog())
                {
                    bool bumpShaderSource = ApplicationConfig.UserSetting.IO.DumpShaderSource;

                    if (dialog.ShowDialog(this) == DialogResult.OK)
                    {
                        // 再読み込み設定が変更されたときは、ファイルのウォッチを設定し直す。
                        var reloadChanged = !ApplicationConfig.UserSetting.ReloadOptions.Equals(dialog.UserSetting.ReloadOptions);

                        var runtimeDebugLogChanged = !ApplicationConfig.UserSetting.ExternalProgram.EnableRuntimeDebugLog.Equals(
                            dialog.UserSetting.ExternalProgram.EnableRuntimeDebugLog);
                        ApplicationConfig.UserSetting = dialog.UserSetting;
                        ApplicationConfig.SaveUserSetting();

                        if (IsReloadbumpShaderSource(bumpShaderSource))
                        {
                            // シェーダ最適化ログの出力設定が変更された場合はシェーダのリロードを行う
                            ReloadbumpShaderSource();
                        }

                        if (reloadChanged)
                        {
                            foreach (var document in DocumentManager.Documents.Where(x => x.ObjectID != GuiObjectID.Project))
                            {
                                document.DisableWatcher();
                                document.EnableWatcher();
                            }
                        }

                        if (runtimeDebugLogChanged)
                        {
                            Viewer.EnableRuntimeDebugMessage.Send(ApplicationConfig.UserSetting.ExternalProgram.EnableRuntimeDebugLog);
                        }

                        App.AppContext.NotifyPropertyChanged(null, new DocumentContentsChangedArgs(DummyObject.TheDummyObject, null));
                    }
                }
            }
        }

        private void mniFileMergeModel_MenuCommandHandler(MenuCommandArgs args)
        {
            var model = App.AppContext.CurrentSelectedObject(args) as Model;
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = model != null;
                return;
            }

            if (model != null)
            {
                (new ModelMerger()).MergeFile(model, MergerUtility.MergeMode.Manual);
            }
        }

        private void mniFileMergeTexture_MenuCommandHandler(MenuCommandArgs args)
        {
            var texture = App.AppContext.CurrentSelectedObject(args) as Texture;
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = texture != null;
                return;
            }

            if (texture != null)
            {
                (new TextureMerger()).MergeFile(texture, MergerUtility.MergeMode.Manual);
            }
        }

        private void mniFileMergeAnimation_MenuCommandHandler(MenuCommandArgs args)
        {
            var anim = App.AppContext.CurrentSelectedObject(args) as AnimationDocument;
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = (anim != null) && anim.CanMerge;
                return;
            }

            if (anim != null)
            {
                (new AnimationMerger()).MergeFile(anim, MergerUtility.MergeMode.Manual);
            }
        }

        private static readonly Bitmap bmpPreviewDisconnect = App.Properties.Resources.TB_PreviewDisconnect;
        private static readonly Bitmap bmpPreviewConnect = App.Properties.Resources.TB_PreviewConnect;
        private void mniHioConnection_SwitchConnection_MenuCommandHandler(MenuCommandArgs args)
        {
            if (args.RequireUpdate)
            {
                if (Viewer.Manager.Instance.IsConnected)
                {
                    tsiHioConnection_SwitchConnection.ToolTipText = Viewer.Manager.IsDeviceTargeted ? res.Strings.Viewer_HioConnection_DisconnectCafe : res.Strings.Viewer_HioConnection_DisconnectPc;
                }
                else
                {
                    tsiHioConnection_SwitchConnection.ToolTipText = ApplicationConfig.Setting.Preview.Target == HioTarget.Device ? res.Strings.Viewer_HioConnection_ConnectCafe : res.Strings.Viewer_HioConnection_ConnectPc;
                }

                // argsからたどっていけないので直接コントロールのプロパティを操作する
                {
                    tsiViewerTarget.Enabled = Viewer.Manager.Instance.IsConnected == false;
                    mniViewerTarget.Enabled = tsiViewerTarget.Enabled;

                    tsiHioConnection_SwitchConnection.Image = Viewer.Manager.Instance.IsConnected ? bmpPreviewConnect : bmpPreviewDisconnect;
                    mniHioConnection_SwitchConnection.Image = Viewer.Manager.Instance.IsConnected ? bmpPreviewDisconnect : bmpPreviewConnect;

                    tsiHioConnection_SwitchConnection.Text = Viewer.Manager.Instance.IsConnected ? res.Strings.Viewer_PreviewConnectting : res.Strings.Viewer_PreviewDisconnectting;
                    mniHioConnection_SwitchConnection.Text = Viewer.Manager.Instance.IsConnected ? res.Strings.Viewer_PreviewDisconnect : res.Strings.Viewer_PreviewConnect;
                }
            }
            else
            {
                // 手動で切断された場合は自動接続を無効にする
                if (Viewer.Manager.Instance.IsConnected)
                {
                    G3dHioLibProxy.Hio.IsAutoConnection = false;
                    Thread.MemoryBarrier();
                }

                ConnectToHio(Viewer.Manager.Instance.IsConnected == false);
            }
        }

        private void UpdateHio()
        {
            if (Viewer.Manager.Instance.IsConnected)
            {
                Viewer.ViewerUtility.AllFowarding();

                // モデルのプレビュ情報を送信する
                SendPreviewInfo(true);
            }
            else
            {
                // 接続解除時、一旦選択クリアのメッセージを転送します。
                Viewer.ViewerUtility.SendSelectedObjects(new GuiObjectGroup());
            }
        }

        // モデルのプレビュ情報を送信する
        private void SendPreviewInfo(bool afterModelLoad)
        {
            foreach(var model in DocumentManager.BindSortedModels)
            {
                // この順番でないと正しく動作しない接続先がある
                model.SendEditBoneBind();

                // identity でも送る
                model.SendEditModelLayout(afterModelLoad);
            }
        }

        public event Action HioConnected;

        public void ConnectToHio(bool isConnect, Action onConnectionFailed = null)
        {
            plcPlay.Pause();

            bool connected = false;
            if (isConnect)
            {
                var selectedSpec = ApplicationConfig.PlatformPresets[cbxPlatform.SelectedIndex];
                Viewer.Manager.Instance.Connect(selectedSpec, ApplicationConfig.Setting.Preview.Target);

                if (Viewer.Manager.Instance.IsConnected == false)
                {
                    if (onConnectionFailed != null)
                    {
                        onConnectionFailed();
                    }

                    App.AppContext.NotificationHandler.MessageBoxError(res.Strings.Viewer_NotConnect);
                }
                else if (HioConnected != null)
                {
                    connected = true;
                }
            }
            else
            {
                Viewer.Manager.Instance.Close();
            }

            UpdateHio();

            App.AppContext.ExecutePropertyChangedEvent(this, (new DocumentPropertyChangedEventArgs()).GetArgs());

            if (connected && HioConnected != null)
            {
                HioConnected();
            }
        }

        private void mniRecentlyUsedFile_MenuCommandHandler(MenuCommandArgs args)
        {
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled =
                    DocumentManager.Documents.All(x => string.Compare(x.FilePath, (string)args.CommandUI.TargetItem.Tag, true) != 0);
                return;
            }

            if (args.RequireExecute)
            {
                DocumentManager.LoadFromFile(Enumerable.Repeat((string)args.CommandUI.TargetItem.Tag, 1));
            }
        }

        private void changeTarget()
        {
            // 接続先を保存する
            var targetForPlatform =
                ApplicationConfig.Setting.Preview.TargetForPlatforms.FirstOrDefault(x => x.Platform == ApplicationConfig.Setting.MainFrame.SelectedSpec);
            string target;
            if (ApplicationConfig.Setting.Preview.Target == HioTarget.Pc)
            {
                target = "Pc";
            }
            else
            {
                target = App.AppContext.SelectedPlatformPreset.DeviceOptions.First(x => x.DeviceType != "Pc").DeviceType;
            }

            if (targetForPlatform == null)
            {

                ApplicationConfig.Setting.Preview.TargetForPlatforms.Add(
                    new Preview.TargetForPlatform()
                    {
                        Platform = ApplicationConfig.Setting.MainFrame.SelectedSpec,
                        Target = target,
                    });
            }
            else
            {
                targetForPlatform.Target = target;
            }

            G3dHioLibProxy.ChangeCommDevice(ApplicationConfig.Setting.Preview.Target, App.AppContext.SelectedPlatformPreset);

            UpdateIcon();
            tlpToolBar.Update();

            // ビューを更新する
            App.AppContext.NotifyPropertyChanged(null, new DocumentContentsChangedArgs(DummyObject.TheDummyObject, null));
        }

        private void mniViewerTarget_Click(object sender, EventArgs e)
        {
            Debug.Assert(sender is App.Controls.UIToolStripMenuItem);
            var item = sender as App.Controls.UIToolStripMenuItem;

            Debug.Assert(item.Tag is ConfigCommon.HioTarget);

            ApplicationConfig.Setting.Preview.Target =
                (ConfigCommon.HioTarget)item.Tag;

            changeTarget();
        }

        private void mniClearUndoStackOnSave_MenuCommandHandler(MenuCommandArgs args)
        {
            // 更新時
            if (args.RequireUpdate)
            {
                args.CommandUI.Checked = ApplicationConfig.UserSetting.EditHistory.ClearAfterFileSaved;
            }
            // 実行時
            else
            {
                ApplicationConfig.UserSetting.EditHistory.ClearAfterFileSaved = !ApplicationConfig.UserSetting.EditHistory.ClearAfterFileSaved;
            }
        }

        private void mniViewerAutoConnection_MenuCommandHandler(MenuCommandArgs args)
        {
            // 更新時
            if (args.RequireUpdate)
            {
                args.CommandUI.Checked = G3dHioLibProxy.Hio.IsAutoConnection;
            }
            // 実行時
            else
            {
                G3dHioLibProxy.Hio.IsAutoConnection = ! G3dHioLibProxy.Hio.IsAutoConnection;
            }
        }

        private void mniFileCloseSelection_MenuCommandHandler(MenuCommandArgs args)
        {
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = DocumentManager.DocumentsWithoutProject.Any();
                return;
            }

            using (var dialog = new DocumentsCloseSelectionDialog(DocumentManager.DocumentsWithoutProject, ApplicationConfig.Setting.IO.CloseReference))
            {
                if (dialog.ShowDialog(this) == DialogResult.OK)
                {
                    ApplicationConfig.Setting.IO.CloseReference = dialog.CloseReference;
                    DocumentManager.RemoveDocuments(dialog.CloseDocuments, null, dialog.CloseReference, false);
                }
            }
        }

        public void cmiCreateAnimation_MenuCommandHandler(MenuCommandArgs args)
        {
            var selected = App.AppContext.CurrentSelectedObject(args);
            var owner = selected is Model ? ((Model)selected).DefaultAnimationSet : selected as AnimationSet;

            // 更新時
            if (args.RequireUpdate)
            {
//				args.CommandUI.Enabled = owner != null;
            }
            // 実行時
            else
            {
#if false
                if (owner == null)
                {
                    return;
                }
#endif
                var tag = (AnimationCreateTag)args.CommandUI.TargetItem.Tag;
                var targetId = tag.id;

                string	animationName	= null;
                int		frameCount		= 0;
                bool	isLoop			= false;
                {
                    switch (targetId)
                    {
                        case GuiObjectID.SceneAnimation:
                        {
                            if (owner != null && !owner.IsSceneAnimationSet)
                            {
                                owner = null;
                                selected = null;
                            }
                            using (var dialog = new SceneAnimationCreateDialog(targetId))
                            {
                                if (dialog.ShowDialog(this) == DialogResult.OK)
                                {
                                    animationName = dialog.AnimationName;
                                }
                            }
                            break;
                        }

                        default:
                        {
                            if (owner != null && owner.IsSceneAnimationSet)
                            {
                                owner = null;
                                selected = null;
                            }
                            var selectedOwner = App.AppContext.SelectedFileViewObjectOwner;
                            var postFix = string.Empty;
                            var postFixTypeString = string.Empty;
                            if (tag is MaterialAnimationCreateTag)
                            {
                                var subType = ((MaterialAnimationCreateTag)tag).subType;
                                postFix = MaterialAnimation.PostFix(subType);
                                postFixTypeString = MaterialAnimation.SubTypeString(subType);
                            }

                            using (var dialog = new AnimationCreateDialog(owner, selectedOwner ?? selected, targetId, postFix, postFixTypeString))
                            {
                                if (dialog.ShowDialog(this) == DialogResult.OK)
                                {
                                    animationName = dialog.AnimationName;
                                    frameCount = dialog.FrameCount;
                                    isLoop = dialog.IsLoop;
                                }
                            }
                            break;
                        }
                    }
                }

                if (animationName != null)
                {
                    AnimationDocument document = null;
                    {
                        switch (targetId)
                        {
                            case GuiObjectID.MaterialAnimation:
                            {
                                var data = new material_animType()
                                {
                                    material_anim_info = new material_anim_infoType()
                                    {
                                        frame_count = frameCount,
                                        loop = isLoop,
                                        frame_resolution = 1,
                                        bake_tolerance_color = 0.001f,
                                        bake_tolerance_tex_scale = 0.01f,
                                        bake_tolerance_tex_rotate = 0.1f,
                                        bake_tolerance_tex_translate = 0.01f,
                                        quantize_tolerance_tex_scale = 0.01f,
                                        quantize_tolerance_tex_rotate = 0.2f,
                                        quantize_tolerance_tex_translate = 0.01f,
                                        dcc_preset = "",
                                        dcc_start_frame = 0,
                                        dcc_end_frame = frameCount,
                                        dcc_fps = (float)ApplicationConfig.Preview.Fps,
                                    }
                                };

                                var anim = new MaterialAnimation(data, new List<G3dStream>())
                                {
                                    Name = animationName,
                                    FileDotExt = G3dPath.MaterialAnimBinaryExtension,
                                };
                                document = anim;
                                break;
                            }
                            case GuiObjectID.ShaderParameterAnimation:
                            {

                                var data = new shader_param_animType()
                                {
                                    shader_param_anim_info = new shader_param_anim_infoType()
                                    {
                                        frame_count = frameCount,
                                        loop = isLoop,
                                        frame_resolution = 1,
                                        bake_tolerance_color				= 0.001f,
                                        bake_tolerance_tex_scale			= 0.01f,
                                        bake_tolerance_tex_rotate			= 0.1f,
                                        bake_tolerance_tex_translate		= 0.01f,
                                        quantize_tolerance_tex_scale		= 0.01f,
                                        quantize_tolerance_tex_rotate		= 0.2f,
                                        quantize_tolerance_tex_translate	= 0.01f,
                                        dcc_preset="",
                                        dcc_start_frame = 0,
                                        dcc_end_frame = frameCount,
                                        dcc_fps = (float)ApplicationConfig.Preview.Fps,
                                    }
                                };

                                var anim = new ShaderParameterAnimation(data, new List<G3dStream>())
                                {
                                    Name = animationName,
                                    FileDotExt = G3dPath.ShaderParamAnimBinaryExtension,
                                };
                                document = anim;
                                break;
                            }

                            case GuiObjectID.ColorAnimation:
                            {
                                var data = new shader_param_animType()
                                {
                                    shader_param_anim_info = new shader_param_anim_infoType()
                                    {
                                        frame_count = frameCount,
                                        loop = isLoop,
                                        frame_resolution = 1,
                                        //
                                        bake_tolerance_color = 0.001f,
                                        bake_tolerance_tex_scale = 0.01f,
                                        bake_tolerance_tex_rotate = 0.1f,
                                        bake_tolerance_tex_translate = 0.01f,
                                        quantize_tolerance_tex_scale = 0.01f,
                                        quantize_tolerance_tex_rotate = 0.2f,
                                        quantize_tolerance_tex_translate = 0.01f,
                                        dcc_preset = "",
                                        dcc_start_frame = 0,
                                        dcc_end_frame = frameCount,
                                        dcc_fps = (float)ApplicationConfig.Preview.Fps,
                                    }
                                };

                                var anim = new ColorAnimation(data, new List<G3dStream>())
                                {
                                    Name = animationName,
                                    FileDotExt = G3dPath.ColorAnimBinaryExtension,
                                };

                                document = anim;
                                break;
                            }

                            case GuiObjectID.TextureSrtAnimation:
                            {
                                var data = new shader_param_animType()
                                {
                                    shader_param_anim_info = new shader_param_anim_infoType()
                                    {
                                        frame_count = frameCount,
                                        loop = isLoop,
                                        frame_resolution = 1,
                                        //
                                        bake_tolerance_color = 0.001f,
                                        bake_tolerance_tex_scale = 0.01f,
                                        bake_tolerance_tex_rotate = 0.1f,
                                        bake_tolerance_tex_translate = 0.01f,
                                        quantize_tolerance_tex_scale = 0.01f,
                                        quantize_tolerance_tex_rotate = 0.2f,
                                        quantize_tolerance_tex_translate = 0.01f,
                                        dcc_preset = "",
                                        dcc_start_frame = 0,
                                        dcc_end_frame = frameCount,
                                        dcc_fps = (float)ApplicationConfig.Preview.Fps,
                                    }
                                };

                                var anim = new TextureSrtAnimation(data, new List<G3dStream>())
                                {
                                    Name = animationName,
                                    FileDotExt = G3dPath.TexSrtAnimBinaryExtension,
                                };

                                document = anim;
                                break;
                            }

                            case GuiObjectID.BoneVisibilityAnimation:
                            {
                                var data = new bone_visibility_animType()
                                {
                                    bone_visibility_anim_info = new bone_visibility_anim_infoType()
                                        {
                                            frame_count = frameCount,
                                            loop = isLoop,
                                            frame_resolution = 1,
                                            dcc_preset = "",
                                            dcc_start_frame = 0,
                                            dcc_end_frame = frameCount,
                                            dcc_fps = (float)ApplicationConfig.Preview.Fps,
                                        }
                                };

                                var anim = new BoneVisibilityAnimation(data, new List<G3dStream>())
                                {
                                    Name = animationName,
                                    FileDotExt = G3dPath.BoneVisibilityAnimBinaryExtension,
                                };


                                document = anim;
                                break;
                            }

                            case GuiObjectID.MaterialVisibilityAnimation:
                            {
                                var data = new mat_visibility_animType()
                                {
                                    mat_visibility_anim_info = new mat_visibility_anim_infoType()
                                    {
                                        frame_count = frameCount,
                                        loop = isLoop,
                                        frame_resolution = 1,
                                        dcc_preset = "",
                                        dcc_start_frame = 0,
                                        dcc_end_frame = frameCount,
                                        dcc_fps = (float)ApplicationConfig.Preview.Fps,
                                    }
                                };

                                var anim = new MaterialVisibilityAnimation(data, new List<G3dStream>())
                                {
                                    Name = animationName,
                                    FileDotExt = G3dPath.MatVisibilityAnimBinaryExtension,
                                };

                                document = anim;
                                break;
                            }

                            case GuiObjectID.TexturePatternAnimation:
                            {
                                var data = new tex_pattern_animType()
                                {
                                    tex_pattern_anim_info = new tex_pattern_anim_infoType()
                                    {
                                        frame_count = frameCount,
                                        loop = isLoop,
                                        frame_resolution = 1,
                                        dcc_preset = "",
                                        dcc_start_frame = 0,
                                        dcc_end_frame = frameCount,
                                        dcc_fps = (float)ApplicationConfig.Preview.Fps,
                                    }
                                };

                                var anim = new TexturePatternAnimation(data, new List<G3dStream>())
                                {
                                    Name = animationName,
                                    FileDotExt = G3dPath.TexPatternAnimBinaryExtension,
                                };

                                document = anim;
                                break;
                            }

                            case GuiObjectID.SceneAnimation:
                            {
                                var data = new scene_animType()
                                {
                                    scene_anim_info = new scene_anim_infoType()
                                    {
                                        frame_resolution = 1,
                                        dcc_magnify = 1,
                                        dcc_preset = "",
                                        dcc_start_frame = 0,
                                        dcc_end_frame = 60,
                                        dcc_fps = (float)ApplicationConfig.Preview.Fps,
                                        bake_all = false,
                                        //
                                        bake_tolerance_rotate = 0.1f,
                                        bake_tolerance_translate = 0.01f,
                                        bake_tolerance_color = 0.001f,
                                        quantize_tolerance_rotate = 0.2f,
                                        quantize_tolerance_translate = 0.01f,
                                    }
                                };

                                var anim = new SceneAnimation(data, new List<G3dStream>())
                                {
                                    Name = animationName,
                                    FileDotExt = G3dPath.SceneAnimBinaryExtension,
                                };

                                document = anim;
                                break;
                            }

                            case GuiObjectID.ShapeAnimation:
                            {
                                var data = new shape_animType()
                                {
                                    shape_anim_info = new shape_anim_infoType()
                                    {
                                        frame_count = frameCount,
                                        frame_resolution = 1,
                                        dcc_preset = "",
                                        dcc_start_frame = 0,
                                        dcc_end_frame = frameCount,
                                        dcc_fps = (float)ApplicationConfig.Preview.Fps,
                                        bake_all = false,
                                        loop = isLoop,
                                    }
                                };

                                var anim = new ShapeAnimation(data, new List<G3dStream>())
                                {
                                    Name = animationName,
                                    FileDotExt = G3dPath.ShapeAnimBinaryExtension,
                                };

                                document = anim;
                                break;
                            }
                        }
                    }

                    // 作ったドキュメントをマネージャーに登録します
                    Debug.Assert(document != null);
                    document.UpdateSavedData();
                    EditCommandSet commandSet = new EditCommandSet();
                    using (var block = new App.AppContext.PropertyChangedSuppressBlock())
                    {
                        using (var vdsb = new Viewer.ViewerDrawSuppressBlock())
                        {
                            commandSet.Add(DocumentManager.CreateAddOrRemoveDocumentCommand(Enumerable.Repeat(document, 1), true).Execute());

                            if (owner != null)
                            {
                                commandSet.Add(DocumentManager.CreateAnimationsEditCommand(owner, owner.Animations.Concat(
                                    Enumerable.Repeat(new AnimationSetItem(document.FileName, document.FileLocation), 1)).ToArray(), false).Execute());
                                commandSet.Add(DocumentManager.CreateAnimationUpdateBindCommand(Enumerable.Repeat(document, 1)).Execute());
                            }
                        }
                    }
                    commandSet.Reverse();
                    TheApp.CommandManager.Add(commandSet);
                }
            }
        }

        public void Command_FileClose(MenuCommandArgs args)
        {
            var document = App.AppContext.CurrentSelectedObject(args) as Document;
            if (args.RequireUpdate)
            {
                // ドキュメントがロードされていれば有効
                args.CommandUI.Enabled = document != null;
                return;
            }

            List<Document> documents = null;
            if (App.AppContext.FileTreeView.Focused)
            {
                documents = App.AppContext.SelectedFileViewObjects?.OfType<Document>().ToList();
            }
            else
            {
                // オブジェクトビュー側で削除された場合、ファイルビュー側の選択は無視する
                GuiObject[] objects = App.AppContext.selectedTarget_.Objects.ToArray();
                if (objects.Count() > 0)
                {
                    documents = new List<Document>();
                    documents.Add(objects[0] as Document);
                }
            }

            if (documents != null && document != null && documents.Any())
            {
                // 複数選択は同じ種類でしかされない
                Debug.Assert(
                    documents.All(x => x.ObjectID == document.ObjectID) || documents.All(x => x is AnimationDocument),
                    "Selected documents isn't same type.");

                if (document.ObjectID == GuiObjectID.Project)
                {
                    using (var block = new App.AppContext.PropertyChangedSuppressBlock())
                    {
                        using (var vdsb = new Viewer.ViewerDrawSuppressBlock())
                        {
                            var commandSet = new EditCommandSet();
                            // 既存ファイルの変更をチェックします。
                            if (DocumentManager.CheckAndSaveModified(DocumentManager.Documents))
                            {
                                DocumentManager.CloseAll(commandSet, true, true);
                            }
                            if (commandSet.CommandCount > 0)
                            {
                                commandSet.Reverse();
                                TheApp.CommandManager.Add(commandSet);
                            }
                        }
                    }
                }
                else if (document.ObjectID == GuiObjectID.ShaderDefinition)
                {
                    // shaderDefinitionの場合、保存はしないが、閉じるかどうか？の確認はする。
                    bool removeFlg = false;
                    if (documents.Any(x => x.IsModifiedObject))
                    {
                        var msgBox = new UIMessageBox(Strings.IO_CloseModifiedShaderDifinitionFile,
                                                                UIMessageBoxButtons.YesNo,
                                                                MessageBoxIcon.Information,
                                                                MessageBoxDefaultButton.Button1,
                                                                false);
                        if (msgBox.ShowDialog() == System.Windows.Forms.DialogResult.Yes)
                        {
                            removeFlg = true;
                        }
                    }
                    else
                    {
                        removeFlg = true;
                    }

                    if (removeFlg)
                    {
                        DocumentManager.RemoveDocuments(documents, App.AppContext.SelectedFileViewObjectOwner as AnimationSet, true, false);
                    }
                }
                else if (document is AnimationDocument)
                {
                    // 他のアニメーションセット内で参照されていたら、閉じるかどうか？の確認はする。
                    var removeFlg = true;

                    var animationSets = DocumentManager.Models.SelectMany(x => x.AnimationSetsWithDefault);

                    if (documents.Any(
                        doc =>
                        animationSets.Count(
                            animSet =>
                            animSet.Animations.Any(
                                anim => anim.Name == doc.FileName && anim.Directory == doc.fileLocation)) >= 2))
                    {
                        if (UIMessageBox.OkCancel(Strings.IO_CloseAnyReferenceAnimationFile) == false)
                        {
                            removeFlg = false;
                        }
                    }

                    if (removeFlg)
                    {
                        DocumentManager.RemoveDocuments(documents, App.AppContext.SelectedFileViewObjectOwner as AnimationSet, true, true);
                    }
                }
                else
                {
                    DocumentManager.RemoveDocuments(documents, App.AppContext.SelectedFileViewObjectOwner as AnimationSet, true, false);
                }
            }
        }

        public void mniViewProperty_MenuCommandHandler(MenuCommandArgs args)
        {
            // 更新時
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = true;
            }
            // 実行時
            else
            {
                if (args.CommandUI.IsObjectViewContextMenu)
                {
                    ObjectPropertyDialog.ShowPropertyDialog();
                }
                else
                {
                    // 新規作成して表示
                    using (var wait = new WaitCursor())
                    {
                        var guiObjectGroup = App.AppContext.SelectedFileViewObjects != null ? new GuiObjectGroup(AppContext.SelectedFileViewObjects) : null;
                        var dialog = new ObjectPropertyDialog(null, guiObjectGroup);
                        dialog.Show();
                    }
                }

            }
        }

        /// <summary>
        /// モーダルダイアログが表示されなくなったタイミングで methodInvoker を実行する
        /// </summary>
        public void InvokeAfterCanFocus(MethodInvoker methodInvoker, bool notWaitRuntime = false)
        {
            BeginInvoke(
                new MethodInvoker(
                () =>
                {
                    EventHandler waitCanFocus = null;
                    waitCanFocus = (s, e) =>
                        {
                            if ((TheApp.MainFrame.CanFocus || (notWaitRuntime && Viewer.LoadProgressDialog.IsDialogVisible))
                            && !ControlUtility.IsAnyCapturing()
                            && !App.AppContext.UpdateFromOutSideBlockCounter.CheckBlock)
                            {
                                Application.Idle -= waitCanFocus;
                                methodInvoker();
                            }
                        };
                    Application.Idle += waitCanFocus;
                }));
        }

        private static readonly Bitmap bmpOptimizeShaderOff = App.Properties.Resources.TB_OptimizeShaderOff;
        private static readonly Bitmap bmpOptimizeShaderOn = App.Properties.Resources.TB_OptimizeShaderOn;

        private void mniOptimizeShader_MenuCommandHandler(MenuCommandArgs args)
        {
            // 更新時
            if (args.RequireUpdate)
            {
                args.CommandUI.Checked = DocumentManager.OptimizeShader;
                args.CommandUI.Enabled = true;
                tsiOptimizeShader.Text = Strings.Menu_OptimizeShader;
                tsiOptimizeShader.Image = DocumentManager.OptimizeShader ? bmpOptimizeShaderOn : bmpOptimizeShaderOff;
                mniOptimizeShader.Image = tsiOptimizeShader.Image;
            }
            // 実行時
            else
            {
                DocumentManager.OptimizeShader = !DocumentManager.OptimizeShader;
                foreach (var model in DocumentManager.Models)
                {
                    Viewer.LoadOptimizedShaderArchive.Send(model);
                }

                App.AppContext.NotifyPropertyChanged(null, new DocumentContentsChangedArgs(DummyObject.TheDummyObject, null));
            }
        }

        private void mniAutoAnimationBindMode_MenuCommandHandler(MenuCommandArgs args)
        {
            // 更新時
            if (args.RequireUpdate)
            {
                args.CommandUI.Checked = ApplicationConfig.Setting.MainFrame.IsAutoAnimationBindMode;
            }
            // 実行時
            else
            {
                ApplicationConfig.Setting.MainFrame.IsAutoAnimationBindMode = ! ApplicationConfig.Setting.MainFrame.IsAutoAnimationBindMode;
            }
        }

        public Action LoadProgreessTimerAction;
        private void LoadProgressTimer_Tick(object sender, EventArgs e)
        {
            if (TheApp.MainFrame.CanFocus && !ControlUtility.IsAnyCapturing())
            {
                LoadProgressTimer.Stop();
                if (LoadProgreessTimerAction != null)
                {
                    LoadProgreessTimerAction();
                }
            }
        }

        private void lvwLogList_Layout(object sender, LayoutEventArgs e)
        {
            lvwLogList.BeginInvokeOrExecute(new Action(() =>
            {
                clhLogMessage.Width = Math.Max(lvwLogList.ClientSize.Width - 4, 0);
            }));
        }

        private void lvwLogList_KeyDown(object sender, KeyEventArgs e)
        {
            // このウィンドウのみのショートカットキー
            if (e.Control)
            {
                // 全選択
                if (e.KeyCode == Keys.A)
                {
                    MessageLog.SelectAll();
                }
                // コピー
                else if (e.KeyCode == Keys.C)
                {
                    MessageLog.CopySelectedMessage();
                }
            }
            else
            {
                // クリア
                if (e.KeyCode == Keys.Delete)
                {
                    MessageLog.Clear();
                }
            }
        }

        private void tsmLogSelectAll_Click(object sender, EventArgs e)
        {
            MessageLog.SelectAll();
        }

        private void tsmLogCopy_Click(object sender, EventArgs e)
        {
            MessageLog.CopySelectedMessage();
        }

        private void tsmLogClear_Click(object sender, EventArgs e)
        {
            MessageLog.Clear();
        }

        private void cmsLog_Opening(object sender, CancelEventArgs e)
        {
            tsmLogSelectAll.Enabled = lvwLogList.Items.Count > 0;
            tsmLogCopy.Enabled = lvwLogList.SelectedItems.Count > 0;
            tsmLogClear.Enabled = lvwLogList.Items.Count > 0;
        }

        private void lvwLogList_Resize(object sender, EventArgs e)
        {
            lvwLogList.Invalidate();
        }

        // ダイアログの枠のサイズ
        public int DialogFrameWidth;
        public int DialogFrameHeight;

        private void MainFrame_Shown(object sender, EventArgs e)
        {
            if (TheApp.OnMainFrameShown != null)
            {
                TheApp.OnMainFrameShown(this, EventArgs.Empty);
            }

            DialogFrameWidth = Size.Width - ClientSize.Width;
            DialogFrameHeight = Size.Height - ClientSize.Height;
        }

        private void mniForceOptimizeShader_MenuCommandHandler(MenuCommandArgs args)
        {
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = Viewer.Manager.Instance.IsConnected && DocumentManager.OptimizeShader;
                return;
            }

            foreach (var model in DocumentManager.Models)
            {
                Viewer.LoadOptimizedShaderArchive.Send(model);
            }
        }


        /// <summary>
        /// ユーザーコマンドの実行(一時的に無効にする
        /// </summary>
        private void mniUserCommand_MenuCommandHandler(MenuCommandArgs args)
        {
            if (args.RequireUpdate)
            {
                if (ApplicationConfig.FileIo.UserCommands.Count == 0)
                {
                    args.CommandUI.TargetItem.Visible = false;
                }

                // メニューを作成
                var item = args.CommandUI.TargetItem as UIToolStripMenuItem;
                item.DropDownItems.Clear();
                var targets = new HashSet<string>(DocumentManager.Documents.Where(x =>
                    x is IntermediateFileDocument &&
                    !ObjectIDUtility.IsTgaExtension(x.fileDotExt)).Select(x => x.FileExt.ToLower()));
                //if (targets.Any())
                {
                    AddUserCommandMenu(
                        item,
                        null,
                        null,
                        x => targets.Contains(x.StartsWith(".") ? x.Substring(1).ToLower(): x.ToLower()),
                        mniSelectAndUserCommandClick,
                        false,
                        targets.Any());
                }

                args.CommandUI.Enabled = item.DropDownItems.Count > 0;
            }
        }

        private void mniSelectAndUserCommandClick(object sender, EventArgs e)
        {
            var command = (UIToolStripMenuItem)sender;
            var userCommandTag = (UserCommandTag)command.Tag;
            Document[] documents;
            var filterExts = userCommandTag.UserCommand.FilterExts.Select(x => x.StartsWith(".") ? x.Substring(1).ToLower() : x.ToLower()).ToArray();
            var intermediateFiles = DocumentManager.Documents
                .Where(x =>
                    x is IntermediateFileDocument
                    && !ObjectIDUtility.IsTgaExtension(x.fileDotExt)
                    && (userCommandTag.UserCommand.Temporary || !string.IsNullOrEmpty(x.FilePath)) //Temporaryがoffの時はパスが必要
                );
            if (!userCommandTag.UserCommand.FilterExts.Any())
            {
                documents = intermediateFiles.ToArray();
            }
            else
            {
                var hashSet = new HashSet<string>(filterExts);
                documents = intermediateFiles.Where(x => hashSet.Contains(x.FileExt.ToLower())).ToArray();
            }

            using (var dialog = new UserCommandDialog(documents, filterExts))
            {
                dialog.Text = userCommandTag.UserCommand.Name;
                if (dialog.ShowDialog(TheApp.MainFrame) == System.Windows.Forms.DialogResult.OK)
                {
                    UserCommandUtil.ExecuteUserCommand(
                        dialog.TargetDocuments.OfType<IntermediateFileDocument>().ToArray(),
                        userCommandTag.Path,
                        userCommandTag.UserCommand);
                }
            }
        }

        private void mniMostRecentlyUsedFiles_MenuCommandHandler(MenuCommandArgs args)
        {
            if (args.RequireUpdate)
            {
                AddMostRecentlyUsedFile(true);
            }
        }

        private void mniMostRecentlyUsedProjects_MenuCommandHandler(MenuCommandArgs args)
        {
            if (args.RequireUpdate)
            {
                AddMostRecentlyUsedFile(false);
            }
        }

        private void mniSaveColumnInfos_MenuCommandHandler(MenuCommandArgs args)
        {
            if (args.RequireExecute)
            {
                SaveColumnInfos();
            }
        }

        private void mniLoadColumnInfos_MenuCommandHandler(MenuCommandArgs args)
        {
            if (args.RequireExecute)
            {
                var objectView = LoadColumnInfos();
                if (objectView == null)
                {
                    return;
                }

                ApplicationConfig.Setting.ObjectView.BoneListView = objectView.BoneListView;
                ApplicationConfig.Setting.ObjectView.MaterialListView = objectView.MaterialListView;
                ApplicationConfig.Setting.ObjectView.ModelListView = objectView.ModelListView;
                ApplicationConfig.Setting.ObjectView.ShapeListView = objectView.ShapeListView;
                ApplicationConfig.Setting.ObjectView.TextureListView = objectView.TextureListView;

                BoneListView.InitializeColumn();
                MaterialListView.InitializeColumn();
                ModelListView.InitializeColumn();
                ShapeListView.InitializeColumn();
                TextureListView.InitializeColumn();

                foreach (var id in Enum.GetValues(typeof(ViewID)))
                {
                    var columnInfo = ObjectListView.GetColumnInfo((ViewID)id);
                    if (columnInfo != null)
                    {
                        columnInfo.NotifySettingChanged();
                    }
                }
            }
        }

        public static void SaveColumnInfos(AppConfig.ObjectView objectView = null)
        {

            var dialog = new SaveFileDialog { Filter = Strings.MainFrame_SaveColumnInfos_Column_config_files_Filter, };

            if (dialog.ShowDialog() != DialogResult.OK)
            {
                return;
            }

            try
            {
                var xmlWriterSettings = new XmlWriterSettings()
                                            {
                                                Encoding = new UTF8Encoding(true), // BOM付き
                                                Indent = true,
                                                IndentChars = "\t",
                                                CloseOutput = false,
                                            };

                using (var stream = new FileStream(dialog.FileName, FileMode.Create))
                using (var xmlWriter = XmlTextWriter.Create(stream, xmlWriterSettings))
                {
                    var ns = new XmlSerializerNamespaces();
                    ns.Add(string.Empty, string.Empty);

                    var setting = new Setting(initialize: false) { ObjectView = new AppConfig.ObjectView(initialize: false) };
                    objectView = objectView ?? ApplicationConfig.Setting.ObjectView;

                    setting.ObjectView.BoneListView = objectView.BoneListView;
                    setting.ObjectView.MaterialListView = objectView.MaterialListView;
                    setting.ObjectView.ModelListView = objectView.ModelListView;
                    setting.ObjectView.ShapeListView = objectView.ShapeListView;
                    setting.ObjectView.TextureListView = objectView.TextureListView;

                    (new XmlSerializer(typeof(AppConfig.Setting), "")).Serialize(
                        xmlWriter,
                        setting,
                        ns);
                }
            }
            catch (Exception exception)
            {
                UIMessageBox.Error(App.res.Strings.Config_SaveFailed, exception.Message);
            }
        }

        public static AppConfig.ObjectView LoadColumnInfos()
        {
            var dialog = new OpenFileDialog
                             {
                                 Filter = Strings.MainFrame_SaveColumnInfos_Column_config_files_Filter,
                                 Multiselect = false,
                             };

            if (dialog.ShowDialog() != DialogResult.OK)
            {
                return null;
            }

            try
            {
                if (File.Exists(dialog.FileName))
                {
                    using (var fileStream = new FileStream(dialog.FileName, FileMode.Open, FileAccess.Read))
                    {
                        var setting =
                            (AppConfig.Setting)(new XmlSerializer(typeof(AppConfig.Setting))).Deserialize(fileStream);
                        return setting.ObjectView;
                    }
                }
            }
            catch (Exception exception)
            {
                UIMessageBox.Error(Strings.IO_Load_Failed + "\n{0}\n{1}", dialog.FileName, exception.Message);
            }
            return null;
        }

        private void tcbShaderParameterFilter_SelectedIndexChanged(object sender, EventArgs e)
        {
            Debug.Assert(tcbShaderParameterFilter.SelectedIndex != -1);
            var filter = ApplicationConfig.Preset.ShaderParameterFilters[tcbShaderParameterFilter.SelectedIndex];
            MaterialShaderPage.SetFilter(filter.Include, filter.Exclude, filter.IgnorePage);
            if (!SettingFromConfig)
            {
                App.AppContext.NotifyPropertyChanged(null, new DocumentPropertyChangedFilterStringArgs(DummyObject.TheDummyObject, null));
            }
        }

        private void mniResizeOnCategoryChange_MenuCommandHandler(MenuCommandArgs args)
        {
            // 更新時
            if (args.RequireUpdate)
            {
                args.CommandUI.Checked = ConfigData.ApplicationConfig.Setting.PropertyEdit.OptimizePageSize;
            }
            // 実行時
            else
            {
                ConfigData.ApplicationConfig.Setting.PropertyEdit.OptimizePageSize ^= true;
            }
        }

        private void cbxPlatform_SelectedIndexChanged(object sender, EventArgs e)
        {
            App.AppContext.SelectedPlatformPreset = ApplicationConfig.PlatformPresets[cbxPlatform.SelectedIndex];
            using (var wait = new WaitCursor())
            {
                bool reconnect = false;

                // 切り替え時は一旦自動接続を無効にする
                bool autoConnect = G3dHioLibProxy.Hio.IsAutoConnection;
                if (autoConnect)
                {
                    G3dHioLibProxy.Hio.IsAutoConnection = false;
                }
                Thread.MemoryBarrier();

                // 切り替え時は一旦切断する
                if (Viewer.Manager.Instance.IsConnected)
                {
                    reconnect = true;
                    ConnectToHio(false);
                }

                // 接続先を選択する
                ConfigData.ApplicationConfig.Setting.MainFrame.SelectedSpec = (string)cbxPlatform.SelectedItem;
                var targetForPlatform = ApplicationConfig.Setting.Preview.TargetForPlatforms.FirstOrDefault(x => x.Platform == ApplicationConfig.Setting.MainFrame.SelectedSpec);
                var platformPreset = ApplicationConfig.PlatformPresets.FirstOrDefault(x => x.Name == ApplicationConfig.Setting.MainFrame.SelectedSpec);
                if (targetForPlatform == null)
                {
                    targetForPlatform = new Preview.TargetForPlatform()
                    {
                        Platform = ApplicationConfig.Setting.MainFrame.SelectedSpec,
                        Target = platformPreset.DeviceOptions.First().DeviceType,
                    };
                }
                else if (!platformPreset.DeviceOptions.Any(x => targetForPlatform.Target == x.DeviceType))
                {
                    targetForPlatform.Target = platformPreset.DeviceOptions.First().DeviceType;
                }

                if (targetForPlatform.Target == "Pc")
                {
                    ApplicationConfig.Setting.Preview.Target = HioTarget.Pc;
                }
                else
                {
                    ApplicationConfig.Setting.Preview.Target = HioTarget.Device;
                }

                UpdateIcon();
                tlpToolBar.Update();

                Thread.MemoryBarrier();
                G3dHioLibProxy.ChangeCommDevice(ApplicationConfig.Setting.Preview.Target, App.AppContext.SelectedPlatformPreset);

                if (reconnect)
                {
                    ConnectToHio(true);
                }

                if (autoConnect)
                {
                    G3dHioLibProxy.Hio.IsAutoConnection = autoConnect;
                }

                // Spec変更を通知する
                App.AppContext.OnSpecChanged();
                // ビューを更新する
                App.AppContext.NotifyPropertyChanged(null, new DocumentContentsChangedArgs(DummyObject.TheDummyObject, null));
            }
        }

        private void mniShowDebugConsole_MenuCommandHandler(MenuCommandArgs args)
        {
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = !DebugConsole.IsInitialized;
            }
            else if (args.RequireExecute)
            {
                DebugConsole.Initialize();
            }
        }

        private void mniHelpNews_MenuCommandHandler(MenuCommandArgs args)
        {
            // 更新時
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = true;
            }
            // 実行時
            else
            {
                HelpUtility.Open(HelpUtility.PageKey.p_help_news);
            }
        }

        private void mniHelpTroubleshooting_MenuCommandHandler(MenuCommandArgs args)
        {
            // 更新時
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = true;
            }
            // 実行時
            else
            {
                HelpUtility.Open(HelpUtility.PageKey.p_appendices_troubleshooting);
            }
        }

        private void tsiViewerTarget_Click(object sender, EventArgs e)
        {
        }

        private void tsiViewerTarget_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (updateIconFlag)
            {
                return;
            }

            if (tsiViewerTarget.SelectedItem as string == res.Strings.Viewer_Target_PC)
            {
                ApplicationConfig.Setting.Preview.Target = HioTarget.Pc;
                changeTarget();
            }
            else
            {
                ApplicationConfig.Setting.Preview.Target = HioTarget.Device;
                changeTarget();
            }
        }

        private void cbxAutomaticShaderRegeneration_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (cbxAutomaticShaderRegeneration.SelectedIndex != -1)
            {
                // シェーダー自動生成の設定
                DocumentManager.AutomaticShaderRegeneration =
                    (DocumentManager.AutomaticShaderRegenerationID)cbxAutomaticShaderRegeneration.SelectedIndex;
            }
        }
    }
}
