﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Runtime.InteropServices;
using System.Text;
using System.Threading;
using App.Controls;
using App.Utility;
using Opal.Security.Cryptography;

namespace App
{
    static class CombinerShaderConverterManager
    {
        /// <summary>
        /// 予約値
        /// </summary>
        public static class ReservedValue
        {
            /// <summary>
            /// ファイル未選択を意味する予約値
            /// </summary>
            public const string Unset = "0";

            /// <summary>
            /// 引数文字列が null や空文字や Unset かどうか
            /// </summary>
            public static bool IsNullOrEmptyOrUnset(string str)
            {
                return string.IsNullOrEmpty(str) || string.Equals(Unset, str, StringComparison.Ordinal);
            }
        }

        public const string CombinerShaderFileExtension = ".fcmb";

        private static string ExeFilePath
        {
            get
            {
            	// 安定するまでパッチ提供が多くなりそうなので、3dTools と分離したフォルダを参照できるようにしておく
                string path = Path.GetFullPath(string.Format(
                    @"{0}\..\3dCombinerShaderConverter\{1}",
                    Environment.GetEnvironmentVariable("NW4F_3DEDITOR_ROOT"),
                    "3dCombinerShaderConverter.exe"));
                if (System.IO.File.Exists(path))
                {
                    return path;
                }

                return Path.GetFullPath(string.Format(
                    @"{0}\..\3dTools\{1}",
                    Environment.GetEnvironmentVariable("NW4F_3DEDITOR_ROOT"),
                    "3dCombinerShaderConverter.exe"));
            }
        }

        public static uint ComputeHashUInt32(string str)
        {
            return (new CRC32()).ComputeHashUInt32(str);
        }

        public static bool ConvertShader(string inputFilePath, IEnumerable<string> combinerShaderFilePaths, bool showLog)
        {
            return ConvertShader(inputFilePath, string.Empty, combinerShaderFilePaths, showLog);
        }

        public static bool ConvertShader(string inputFilePath, string outputFilePath, IEnumerable<string> combinerShaderFilePaths, bool showLog)
        {
            bool ret = true;
            using (var process = new Process())
            {
                var outputBuilder = new StringBuilder();
                var errorBuilder = new StringBuilder();

                string cmd = string.Empty;
                try
                {
                    var correctedCombinerShaderFilePaths = combinerShaderFilePaths.Where(x => !string.IsNullOrEmpty(x)).ToArray();

                    var exePath = ExeFilePath;
                    var args = new List<string>();
                    args.Add(string.Format("\"{0}\"", inputFilePath));
                    if (!string.IsNullOrEmpty(outputFilePath))
                    {
                        args.Add(string.Format("--output \"{0}\"", outputFilePath));
                    }
                    if (correctedCombinerShaderFilePaths.Any())
                    {
                        args.Add(string.Format("--combiner-shader-path \"{0}\"", string.Join(";", correctedCombinerShaderFilePaths)));
                    }

                    process.StartInfo.FileName = exePath;
                    process.StartInfo.Arguments = string.Join(" ", args);
                    process.StartInfo.CreateNoWindow = true;
                    process.StartInfo.UseShellExecute = false;
                    process.StartInfo.RedirectStandardError = true;
                    process.StartInfo.StandardErrorEncoding = Encoding.Default;
                    process.StartInfo.RedirectStandardOutput = true;
                    process.StartInfo.StandardOutputEncoding = Encoding.Default;
                    process.OutputDataReceived += (s, a) =>
                    {
                        if (!string.IsNullOrEmpty(a.Data))
                        {
                            DebugConsole.WriteLine(string.Format("CombinerShaderConverterLog: {0}", a.Data));
                            outputBuilder.AppendLine(a.Data);
                        }
                    };
                    process.ErrorDataReceived += (s, a) =>
                    {
                        if (!string.IsNullOrEmpty(a.Data))
                        {
                            DebugConsole.WriteLine(string.Format("CombinerShaderConverterError: {0}", a.Data));
                            errorBuilder.AppendLine(a.Data);
                        }
                    };

                    cmd = string.Format("{0} {1}", process.StartInfo.FileName, process.StartInfo.Arguments);
                    DebugConsole.WriteLine(string.Format("ShaderConvert {0}", cmd));
                    process.Start();
                    process.BeginOutputReadLine();
                    process.BeginErrorReadLine();
                    process.WaitForExit();

                    // 以下の場合はエラーにする。
                    // - 終了コードがゼロ以外
                    if (process.ExitCode != 0)
                    {
                        ret = false;
                    }
                }
                catch
                {
                    ret = false;
                }

                // エラー表示
                if (showLog && !ret)
                {
                    if (process.ExitCode != 0)
                    {
                        var error = errorBuilder.ToString();
                        var output = outputBuilder.ToString();

                        var messages = new StringBuilder();
                        messages.AppendLine(res.Strings.ShaderConvertError);
                        messages.AppendLine();
                        messages.AppendLine("From ShaderConverter -------------------------------");
                        messages.AppendLine(output);
                        messages.AppendLine(error);

                        var log = messages.ToString();

                        // ログ出力
                        ErrorLog.WriteLog(log, ErrorLog.LogKind.ShaderConvert);

                        // エラーダイアログの表示
                        ShdrcvtrManager.ShowShaderConvertErrorMsg(new List<string> { cmd, log }, process.ExitCode == 0);
                    }
                }
            }
            return ret;
        }
    }
}
