﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using App.PropertyEdit;
using App.Utility;
using ConfigCommon;
using nw.g3d.nw4f_3dif;

namespace App.Data
{
    public sealed class Shape : GuiObject
    {
        public shapeType Data{ get; private set; }
        public Model Owner { get; private set; }
        public override Document OwnerDocument
        {
            get { return Owner; }
        }

        // 作業用カラーとコメントの編集は行わない。
        public override bool Editable
        {
            get
            {
                return false;
            }
        }

        // 処理頂点数算出関数。
        // 算出可能かどうかは、この辞書に登録されているかどうかで判定する。
        private static readonly Dictionary<string, Action<Shape>> ProcessVertexCountFunc = new Dictionary<string, Action<Shape>>()
        {
            {
                "Nx",
                (shape) =>
                {
                    int meshIndex = 0;
                    foreach (var mesh in shape.Data.mesh_array.mesh)
                    {
                        var indices = shape.Owner.BinaryStreams[mesh.stream_index].IntData;
                        var processCount = nw.g3d.iflib.IfModelUpdateUtility.GetProcessVertexCount(mesh, indices, nw.g3d.iflib.IfModelUpdateUtility.TargetPlatform.Nx);
                        shape.processVertexCountInMesh[meshIndex] = processCount;
                        meshIndex++;
                        shape.processVertexCount += processCount;
                    }
                }
            },
            {
                "Cafe",
                (shape) =>
                {
                    int meshIndex = 0;
                    foreach (var mesh in shape.Data.mesh_array.mesh)
                    {
                        var indices = shape.Owner.BinaryStreams[mesh.stream_index].IntData;
                        var processCount = nw.g3d.iflib.IfModelUpdateUtility.GetProcessVertexCount(mesh, indices, nw.g3d.iflib.IfModelUpdateUtility.TargetPlatform.Cafe);
                        shape.processVertexCountInMesh[meshIndex] = processCount;
                        meshIndex++;
                        shape.processVertexCount += processCount;
                    }
                }
            }
        };

        private string LastCountedProcessVertexPlatform;

        public Shape(shapeType shape, Model owner, int index)
            : base(GuiObjectID.Shape, index)
        {
            Data = shape;

            // オーナー(モデル)を設定
            Owner = owner;
            // 名前を設定
            Name = shape.name;

            LastCountedProcessVertexPlatform = null;
            UpdateProcessVertexCount(App.AppContext.SelectedPlatformPreset);

            UpdateVertexCountInMesh();
        }

        public void UpdateData()
        {
        }

        // インデックス数
        public int IndexCount
        {
            get
            {
                // TODO:3.0
                return Data.mesh_array.mesh.Sum(x => x.submesh_array.submesh.Sum(y => y.count));//Data.mesh.submesh_array.submesh.Sum(x => x.count);
            }
        }

        public int IndexCountInMesh(int index)
        {
            var mesh = Data.mesh_array.mesh[index];
            return mesh.submesh_array.submesh.Sum(y => y.count);
        }

        // 頂点数
        public int VertexCount
        {
            get
            {
                if ((Owner != null) &&
                    (Owner.Vertices != null) &&
                    (Owner.Vertices[Data.shape_info.vertex_index] != null) &&
                    (Owner.Vertices[Data.shape_info.vertex_index].vtx_attrib_array != null) &&
                    (Owner.Vertices[Data.shape_info.vertex_index].vtx_attrib_array.vtx_attrib != null))
                {
                    return Owner.Vertices[Data.shape_info.vertex_index].vtx_attrib_array.vtx_attrib.Select(x => x.count).FirstOrDefault();
                }
                else
                {
                    return 0;
                }
            }
        }

        // メッシュごとの頂点数
        public int[] VertexCountInMesh
        {
            get;
            private set;
        }

        public void UpdateVertexCountInMesh()
        {
            VertexCountInMesh = new int[Data.mesh_array.mesh.Length];
            int meshIndex = 0;
            foreach (var mesh in Data.mesh_array.mesh)
            {
                var stream = Owner.BinaryStreams[mesh.stream_index];
                Debug.Assert(stream.type == stream_typeType.@int);
                var indices = new HashSet<int>();
                foreach (var submesh in mesh.submesh_array.submesh)
                {
                    for (int i = 0; i < submesh.count; i++)
                    {
                        indices.Add(stream.IntData[i + submesh.offset]);
                    }
                }
                VertexCountInMesh[meshIndex] = indices.Count;
                meshIndex++;
            }
        }

        // ポリゴン数
        public int PolygonCount
        {
            get
            {
                var query =
                    from mesh in Data.mesh_array.mesh
                    from submesh in mesh.submesh_array.submesh
                    select
                        (mesh.mode == mesh_modeType.triangles) ?
                            submesh.count / 3 :
                            submesh.count - 2;

                return query.Sum();
            }
        }

        // メッシュごとのポリゴン数
        public int PolygonCountInMesh(int index)
        {
            var mesh = Data.mesh_array.mesh[index];
            return mesh.submesh_array.submesh.Sum(x => (mesh.mode == mesh_modeType.triangles) ? x.count / 3 : x.count - 2);
        }

        public static int PolygonCountInMesh(meshType mesh)
        {
            return mesh.submesh_array.submesh.Sum(x => (mesh.mode == mesh_modeType.triangles) ? x.count / 3 : x.count - 2);
        }

        /// <summary>
        /// 選択中のプラットフォームで処理頂点数が算出可能かどうか。
        /// </summary>
        public static bool IsProcessVertexCountComputable()
        {
            return IsProcessVertexCountComputable(App.AppContext.SelectedPlatformPreset);
        }

        /// <summary>
        /// 引数のプラットフォームで処理頂点数が算出可能かどうか。
        /// </summary>
        /// <param name="platform">判定するプラットフォーム</param>
        public static bool IsProcessVertexCountComputable(TeamConfig.PlatformPreset platform)
        {
            return ProcessVertexCountFunc.ContainsKey(platform.Name);
        }

        /// <summary>
        /// 処理頂点数。
        /// 処理頂点数はプラットフォーム毎に異なる。
        /// プラットフォーム選択はツール上で任意のタイミングで変更でき、
        /// 選択処理もコマンドに積まれないため、ここでは常に選択プラットフォームで更新した値を返す。
        /// </summary>
        public int ProcessVertexCount
        {
            get
            {
                // 処理頂点数を更新する。
                // 直前に算出した値はキャッシュしているので、
                // 同じプラットフォームであれば更新処理はスキップされる。
                UpdateProcessVertexCount(App.AppContext.SelectedPlatformPreset);
                return processVertexCount;
            }
        }
        private int processVertexCount;

        /// <summary>
        /// メッシュごとの処理頂点数
        /// 処理頂点数はプラットフォーム毎に異なる。
        /// プラットフォーム選択はツール上で任意のタイミングで変更でき、
        /// 選択処理もコマンドに積まれないため、ここでは常に選択プラットフォームで更新した値を返す。
        /// </summary>
        public System.Collections.ObjectModel.ReadOnlyCollection<int> ProcessVertexCountInMesh
        {
            get
            {
                // 処理頂点数を更新する。
                // 直前に算出した値はキャッシュしているので、
                // 同じプラットフォームであれば更新処理はスキップされる。
                UpdateProcessVertexCount(App.AppContext.SelectedPlatformPreset);
                return Array.AsReadOnly(processVertexCountInMesh);
            }
        }
        private int[] processVertexCountInMesh;

        public void UpdateProcessVertexCount(TeamConfig.PlatformPreset platform)
        {
            UpdateProcessVertexCount(platform, false);
        }

        public void UpdateProcessVertexCount(TeamConfig.PlatformPreset platform, bool forceUpdate)
        {
            bool needsUpdate =
                forceUpdate ||
                string.IsNullOrEmpty(LastCountedProcessVertexPlatform) ||
                LastCountedProcessVertexPlatform != platform.Name;
            if (!needsUpdate)
            {
                // 更新不要。
                return;
            }

            if (ProcessVertexCountFunc.ContainsKey(platform.Name))
            {
                processVertexCount = 0;
                processVertexCountInMesh = new int[Data.mesh_array.mesh.Length];
                ProcessVertexCountFunc[platform.Name](this);
            }
            else
            {
                processVertexCount = 0;
                processVertexCountInMesh = Enumerable.Repeat(0, Data.mesh_array.mesh.Length).ToArray();
            }
            LastCountedProcessVertexPlatform = platform.Name;
        }

        private class VertexCache
        {
            public int ProcessCount{ get; private set; }
            private const int cacheSize = 14;
            private LinkedList<int> cache_ = null;

            public void Reset()
            {
                cache_ = new LinkedList<int>();

                for (int i = 0;i != cacheSize;++ i)
                {
                    cache_.AddLast(-1);
                }
            }

            public void Process(int index)
            {
                // キャッシュにある
                if (cache_.Find(index) != null)
                {
                    // なにもしない
                    ;
                }
                // キャッシュにない
                else
                {
                    // カウントする
                    ++ ProcessCount;

                    // 末端に登録する
                    cache_.AddFirst(index);

                    // キャッシュからあふれるものを削除する
                    cache_.RemoveLast();
                }

                Debug.Assert(cache_.Count == cacheSize);
            }
        }

        #region savedData
        public shapeType savedData;
        public vertexType savedVertex;
        public int savedProcessVertexCountInMesh0;
        public int savedVertexCountInMesh0;
        public int savedPolygonCount;
        public int savedIndexCount;
        public Dictionary<int, G3dStream> savedStreams;
        public bool IsStreamModified;

        public override void UpdateSavedData()
        {
            base.UpdateSavedData();
            savedData = ObjectUtility.Clone(Data);

            savedProcessVertexCountInMesh0 = ProcessVertexCountInMesh[0];
            savedVertexCountInMesh0 = VertexCountInMesh[0];
            savedPolygonCount = PolygonCount;
            savedIndexCount = IndexCount;

            savedVertex = ObjectUtility.Clone(Owner.Vertices[Data.shape_info.vertex_index]);

            savedStreams = new Dictionary<int, G3dStream>();
            foreach (var attrib in savedVertex.vtx_attrib_array.vtx_attrib)
            {
                savedStreams[attrib.stream_index] = Owner.BinaryStreams[attrib.stream_index];
            }

            IsStreamModified = false;
        }

        public void CopySavedData(Shape source)
        {
            CopyGuiObjectSavedData(source);
            savedData = source.savedData;

            savedProcessVertexCountInMesh0 = source.savedProcessVertexCountInMesh0;
            savedVertexCountInMesh0 = source.savedVertexCountInMesh0;
            savedPolygonCount = source.savedPolygonCount;
            savedIndexCount = source.savedIndexCount;

            savedVertex = source.savedVertex;

            savedStreams = source.savedStreams;

            IsStreamModified = CheckIsStreamModified();
        }

        private bool CheckIsStreamModified()
        {
            foreach (var savedAttrib in savedVertex.vtx_attrib_array.vtx_attrib)
            {
                var attrib = Owner.Vertices[Data.shape_info.vertex_index].vtx_attrib_array.vtx_attrib.FirstOrDefault(x => x.name == savedAttrib.name);
                if (attrib == null)
                {
                    return true;
                }

                var savedStream = savedStreams[savedAttrib.stream_index];
                var stream = Owner.BinaryStreams[attrib.stream_index];
                if (savedStream.type != stream.type)
                {
                    return true;
                }

                switch (savedStream.type)
                {
                    case stream_typeType.@byte:
                        if (!savedStream.ByteData.SequenceEqual(stream.ByteData))
                        {
                            return true;
                        }
                        break;
                    case stream_typeType.@int:
                        if (!savedStream.IntData.SequenceEqual(stream.IntData))
                        {
                            return true;
                        }
                        break;
                    case stream_typeType.@float:
                        if (!savedStream.FloatData.SequenceEqual(stream.FloatData))
                        {
                            return true;
                        }
                        break;
                    default:
                        Debug.Assert(false);
                        break;
                }
            }

            return false;
        }

        public bool IsValueChanged<T>(Func<shapeType, T> select) where T: struct
        {
            return !select(savedData).Equals(select(Data));
        }

        public bool IsValueChanged(Func<shapeType, string> select)
        {
            return !select(savedData).Equals(select(Data));
        }

        public bool IsSubmeshChanged()
        {
            return savedData.mesh_array.mesh.Length != Data.mesh_array.mesh.Length ||
                savedData.mesh_array.mesh.Zip(Data.mesh_array.mesh, (x, y) => IsSubmeshChanged(x, y)).Any(x => x);
        }

        public bool IsSubmeshChanged(meshType saved, meshType data)
        {
            return saved.submesh_array.submesh.Length != data.submesh_array.submesh.Length ||
                saved.submesh_array.submesh.Zip(data.submesh_array.submesh, (x, y) => x.offset == y.offset && x.count == y.count).Any(x => !x);
        }

        public override bool EqualsToSavedData()
        {
            if (!base.EqualsToSavedData())
            {
                return false;
            }

            return !ShapeGeneralPage.IsModified(this) &&
                !ShapeMeshPage.IsModified(this) &&
                !ShapeVertexPage.IsModified(this);
        }
        #endregion
    }
}
