﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading;
using System.Windows.Forms;
using App.PropertyEdit;
using App.Utility;
using App.res;
using ConfigCommon;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;

namespace App.Data
{
    public sealed class ShaderDefinition :
        IntermediateFileDocument,
        NintendoWare.G3d.Edit.IEditShaderArchiveTarget
    {
        #region ランタイム連携
        /// <summary>
        /// IEditShaderArchiveTarget
        /// </summary>
        public UInt32 ShaderArchiveKey { get; set; }

        /// <summary>
        /// IEditShaderArchiveTarget
        /// </summary>
        public override void ResetStatus()
        {
            base.ResetStatus();
            ShaderArchiveKey = 0;
            updatedAttachedShadingModel.Clear();
        }

        public bool LastIsAttachShaderArchiveBinary = false;
        #endregion

        public shader_definitionType data_;
        public shader_definitionType Data {
            get
            {
                return data_;
            }
            private set
            {
                data_ = value;
                noBranchData = ObjectUtility.Clone(data_);
                foreach (var shading_model in noBranchData.shading_model_array.GetItems())
                {
                    foreach (var option in shading_model.Options())
                    {
                        option.branch = false;
                    }
                }
            }
        }

        // fsv 用のデータ
        private shader_definitionType noBranchData;
        public shader_definitionType NoBranchData
        {
            get
            {
                return noBranchData;
            }
        }

        public List<Definition> Definitions { get; private set; }

        public ShaderDefinition(shader_definitionType shaderDefinition, List<G3dStream> streamArray)
            : base(GuiObjectID.ShaderDefinition, streamArray)
        {
            Data = shaderDefinition;
            ToolData.Load(Data.tool_data);
            Definitions = CreateDefinitions(Data, this);
            DisplayIncludePaths = CreateDisplayIncludePaths(Data);

            UpdateSavedData();

            // SwapAdd 側は実はいらないけどあっても害はないはず
            DocumentSwapAdd += x => EnableSourceWatcher();
            DocumentSwapRemove += DisableSourceWatcher;

            DocumentAdd += EnableSourceWatcher;
            DocumentRemove += DisableSourceWatcher;
        }

        public IEnumerable<int> ShadingModelIndices(string sourcefileName)
        {
            for (int i = 0; i < Definitions.Count; i++)
            {
                string[] srcs;
                if (!SourceFileNames.TryGetValue(Definitions[i].Name, out srcs) ||
                    srcs.Contains(sourcefileName))
                {
                    yield return i;
                }
            }
        }

        // 編集回数の更新
        public void SetModifiedShadingModel(ShaderDefinition old, string filePath)
        {
            shadingModelModifiedCounts = old.shadingModelModifiedCounts.ToDictionary(x => x.Key, x => x.Value);
            ModifiedNotMaterialShadingModelIndices = new HashSet<int>(old.ModifiedNotMaterialShadingModelIndices);
            LastIsAttachShaderArchiveBinary = old.LastIsAttachShaderArchiveBinary;
            foreach (var i in ShadingModelIndices(filePath))
            {
                var name = Definitions[i].Name;
                int count;
                shadingModelModifiedCounts.TryGetValue(name, out count);
                shadingModelModifiedCounts[name] = count + 1;
                //if (!Definitions[i].Data.material_shader)
                {
                    ModifiedNotMaterialShadingModelIndices.Add(i);
                }
            }
        }

        /// <summary>
        /// シェーダー定義が編集された回数(メインスレッド用)
        /// </summary>
        public Dictionary<string, int> shadingModelModifiedCounts = new Dictionary<string, int>();

        public HashSet<int> ModifiedNotMaterialShadingModelIndices = new HashSet<int>();

        public HashSet<string> updatedAttachedShadingModel = new HashSet<string>();
        #region シェーダーバイナリ
        /// <summary>
        /// バイナリの取得
        /// </summary>
        public string GetShaderBinaryPath(string shadingModel, int modifiedCounts, bool forPC, bool ignoreLastErrorOfShaderBinarization, TeamConfig.PlatformPreset platform, Viewer.LoadProgressDialog.DialogBlock block)
        {
            // modifiedCounts が 0 であれば、この位置 (関数の先頭) で常に null を返す実装だったが、
            // コンバイナーシェーダーファイルを参照している場合は modifiedCounts が 0 であってもファイルを取り込む必要がある。
            // したがって、ここで行っていた modifiedCounts の評価はコンバイナーシェーダーファイルの有無取得後に移動。

            // 1 シェーダー定義 1 バイナリファイルで実装
            lock (bfshaPathLock)
            {
                if (FailedToBinarize && !ignoreLastErrorOfShaderBinarization)
                {
                    return null;
                }

                var key = new Tuple<string, string, string>(shadingModel, platform.Name, forPC ? "PC": "Device");

                // シェーダー定義を参照している全表示モデルからコンバイナーファイルを取得する。
                var targetCombiners = DocumentManager.Models
                    .Where(x => x.IsVisible)
                    .Select(x => Tuple.Create(x, x.Materials.SelectMany(y => y.GetCombiners().Where(z => (z.Item1.MaterialShaderAssign.ShaderDefinition == this) && !CombinerShaderConverterManager.ReservedValue.IsNullOrEmptyOrUnset(z.Item2.value))).ToArray())).ToArray();
                var targetModels = targetCombiners.Select(x => x.Item1).Distinct();
                var combinerShaderFileInfos = targetCombiners.SelectMany(x => x.Item2.Select(y => y.Item1.CreateCombinerShaderFilePath(y.Item2))).Distinct().Select(x => Tuple.Create(x, !string.IsNullOrEmpty(x) ? (new System.IO.FileInfo(x)).LastWriteTime : default(DateTime))).ToArray();

                string binaryPath;
                if ((modifiedCounts == 0) && !combinerShaderFileInfos.Any())
                {
                    // 未編集状態でコンバイナーシェーダーファイルも持っていない場合に null を返す。
                    // この関数における modifiedCounts の参照はここのみ。
                    return null;
                }
                else if (bfshaPaths.TryGetValue(key, out binaryPath))
                {
                    // バイナリファイルの更新が必要か調べる。
                    // 不要な更新を防ぐために modifiedCounts を判定に使用してはならない。
                    // http://spdlybra.nintendo.co.jp/jira/browse/SIGLO-80352
                    Dictionary<string, DateTime> lastWriteTimes;
                    var needsUpdate =
                        !combinerShaderFileLastWriteTimes.TryGetValue(binaryPath, out lastWriteTimes) ||
                        combinerShaderFileInfos.Any(x => { DateTime lastWriteTime; return !lastWriteTimes.TryGetValue(x.Item1, out lastWriteTime) || (DateTime.Compare(x.Item2, lastWriteTime) != 0); });
                    if (!needsUpdate)
                    {
                        // 更新不要。

                        // バイナライズ成功時のフラグを設定。
                        // Undo/Redo によって失敗から成功に変わることもあるのでフラグ更新が必要。
                        SucceededInBinarize();

                        return binaryPath;
                    }
                }

                // メッセージを変更するだけ
                // この関数では戻さない
                block.Message = string.Format(App.res.Strings.Viewer_ShaderDefinitionConverting, Name);

                List<Tuple<byte[], string>> inputFiles = new List<Tuple<byte[], string>>();

                string tempDir = TemporaryFileUtility.MakeTempDirectory();

                var shader_definition = ObjectUtility.Clone(data_);

                // 制限
                shader_definition.shading_model_array = data_.shading_model_array.GetItems()
                    .Where(x => x.name == shadingModel).ToG3dArrayElement<shading_modelType, shading_model_arrayType>();

                // fsd
                nw4f_3difType fsd = new nw4f_3difType()
                {
                    Item = shader_definition,
                };

                // バイナリ中間ファイルの場合、nw4f_3difからストリームを削除する。
                if (G3dStreamUtility.HasStreamArray(fsd))
                {
                    // ここには来ないはず?
                    Debug.Assert(false);
                    fsd.RootElement.stream_array = null;
                }

                byte[] fileImage;
                lock (Viewer.Connecter.AbortLock)
                {
                    fileImage = IfBinaryFormatter.FormatStream(fsd, BinaryStreams, null);
                }

                string fsdFilePath = Path.Combine(tempDir, Name + ".fsdb");
                bool combinerFailure = false;
                if (combinerShaderFileInfos.Any())
                {
                    // fileImage を ecmb を埋め込んだものに変換する。
                    // 1. fileImage をファイルに保存
                    // 2. 保存したファイルに ecmb ファイルを埋め込む
                    // 3. 埋め込んだファイルイメージを取得する
                    try
                    {
                        File.WriteAllBytes(fsdFilePath, fileImage);
                        if (!CombinerShaderConverterManager.ConvertShader(fsdFilePath, combinerShaderFileInfos.Select(x => x.Item1), true))
                        {
                            throw new Exception();
                        }
                        fileImage = File.ReadAllBytes(fsdFilePath);
                    }
                    catch
                    {
                        combinerFailure = true;
                    }
                }
                inputFiles.Add(new Tuple<byte[], string>(fileImage, fsdFilePath));

                binaryPath = Path.Combine(tempDir, Name + ".bfsha");

                if (!combinerFailure &&
                    ShdrcvtrManager.ConvertShaderToBinaryForViewer(
                    inputFiles,
                    binaryPath,
                    forPC ? ShdrcvtrManager.TargetType.PC : ShdrcvtrManager.TargetType.Cafe,
                    platform,
                    G3dHioLibProxy.Hio.GetRuntimeAddressType() == NintendoWare.G3d.Edit.PlatformAddressType.Adress32,
                    null))
                {
                    bfshaPaths[key] = binaryPath;
                    combinerShaderFileLastWriteTimes[binaryPath] = combinerShaderFileInfos.ToDictionary(x => x.Item1, x => x.Item2);

                    // ファイルの監視対象を更新。
                    {
                        // ディレクトリ監視を停止。
                        // ディレクトリ変更時にファイルウォッチャーを作成することがあるので
                        // ファイル監視の停止よりも先にディレクトリ監視を停止する必要がある。
                        foreach (var watcher in combinerShaderDirectoryWatchers.Values)
                        {
                            watcher?.Dispose();
                        }
                        combinerShaderDirectoryWatchers.Clear();

                        // ファイルの監視を停止。
                        foreach (var watcher in combinerShaderFileWatchers.Values.SelectMany(x => x.Values))
                        {
                            watcher?.Dispose();
                        }
                        combinerShaderFileWatchers.Clear();

                        foreach (var combiners in targetCombiners.Select(x => x.Item2))
                        {
                            // ディレクトリウォッチャーを作成。
                            foreach (var combiner in combiners)
                            {
                                var material = combiner.Item1;

                                FileSystemWatcher ecmbDirWatcher;
                                if (!combinerShaderDirectoryWatchers.TryGetValue(material.OwnerDocument.FileLocation, out ecmbDirWatcher))
                                {
                                    ecmbDirWatcher = new FileSystemWatcher(material.OwnerDocument.FileLocation, Material.CombinerShaderDirectoryName);
                                    ecmbDirWatcher.NotifyFilter = System.IO.NotifyFilters.DirectoryName | NotifyFilters.LastWrite;
                                    ecmbDirWatcher.IncludeSubdirectories = false;
                                    ecmbDirWatcher.SynchronizingObject = TheApp.MainFrame;
                                    Action dirChanged = () =>
                                    {
                                        Dictionary<string, FileSystemWatcher> fileWatchers;
                                        if (combinerShaderFileWatchers.TryGetValue(ecmbDirWatcher, out fileWatchers))
                                        {
                                            bool updated = false;

                                            // コンバイナーシェーダーディレクトリ下の未作成ファイルウォッチャーの作成を試みる。
                                            foreach (var item in fileWatchers.Where(x => x.Value == null).ToArray())
                                            {
                                                FileSystemWatcher ecmbFileWatcher = null;
                                                try
                                                {
                                                    ecmbFileWatcher = new FileSystemWatcher(Path.Combine(ecmbDirWatcher.Path, ecmbDirWatcher.Filter), System.IO.Path.GetFileName(item.Key));
                                                    ecmbFileWatcher.NotifyFilter = System.IO.NotifyFilters.FileName;
                                                    ecmbFileWatcher.IncludeSubdirectories = false;
                                                    ecmbFileWatcher.SynchronizingObject = TheApp.MainFrame;
                                                    Action fileChanged = () =>
                                                    {
                                                        foreach (var model in targetModels)
                                                        {
                                                            Viewer.LoadOrReloadModel.Send(model, true);
                                                        }
                                                    };
                                                    ecmbFileWatcher.Changed += (ss, ee) => { fileChanged(); };
                                                    ecmbFileWatcher.Created += (ss, ee) => { fileChanged(); };
                                                    ecmbFileWatcher.Deleted += (ss, ee) => { fileChanged(); };
                                                    ecmbFileWatcher.Renamed += (ss, ee) => { fileChanged(); };
                                                    ecmbFileWatcher.EnableRaisingEvents = true;
                                                }
                                                catch
                                                {
                                                    ecmbFileWatcher?.Dispose();
                                                    ecmbFileWatcher = null;
                                                }
                                                fileWatchers[item.Key] = ecmbFileWatcher;

                                                if (ecmbFileWatcher != item.Value)
                                                {
                                                    DebugConsole.WriteLine(string.Format("Reload {0}", item.Key));
                                                    updated |= true;
                                                }
                                            }

                                            if (updated)
                                            {
                                                foreach (var model in targetModels)
                                                {
                                                    Viewer.LoadOrReloadModel.Send(model, true);
                                                }
                                            }
                                        }
                                    };
                                    ecmbDirWatcher.Changed += (ss, ee) => { dirChanged(); };
                                    ecmbDirWatcher.Created += (ss, ee) => { dirChanged(); };
                                    ecmbDirWatcher.Deleted += (ss, ee) => { dirChanged(); };
                                    ecmbDirWatcher.Renamed += (ss, ee) => { dirChanged(); };
                                    ecmbDirWatcher.EnableRaisingEvents = true;
                                    combinerShaderDirectoryWatchers.Add(material.OwnerDocument.FileLocation, ecmbDirWatcher);
                                }

                                // ファイルウォッチャーを作成。
                                Dictionary<string, FileSystemWatcher> ecmbFileWatchers;
                                if (!combinerShaderFileWatchers.TryGetValue(ecmbDirWatcher, out ecmbFileWatchers))
                                {
                                    ecmbFileWatchers = new Dictionary<string, FileSystemWatcher>();
                                    combinerShaderFileWatchers.Add(ecmbDirWatcher, ecmbFileWatchers);
                                }

                                var ecmbFilePath = material.CreateCombinerShaderFilePath(combiner.Item2);
                                if (!string.IsNullOrEmpty(ecmbFilePath) && !ecmbFileWatchers.ContainsKey(ecmbFilePath))
                                {
                                    // コンバイナーシェーダーファイルウォッチャーの作成を試みる。
                                    // 存在しないディレクトリのウォッチャーは作成できないので、
                                    // ここで作成に失敗したものは、ディレクトリウォッチャー側で再作成を試みる。
                                    FileSystemWatcher ecmbFileWatcher = null;
                                    try
                                    {
                                        ecmbFileWatcher = new FileSystemWatcher(Path.Combine(ecmbDirWatcher.Path, ecmbDirWatcher.Filter), System.IO.Path.GetFileName(ecmbFilePath));
                                        ecmbFileWatcher.NotifyFilter = System.IO.NotifyFilters.FileName | NotifyFilters.LastWrite;
                                        ecmbFileWatcher.IncludeSubdirectories = false;
                                        ecmbFileWatcher.SynchronizingObject = TheApp.MainFrame;
                                        Action fileChanged = () =>
                                        {
                                            DebugConsole.WriteLine(string.Format("Reload {0}", ecmbFilePath));
                                            foreach (var model in targetModels)
                                            {
                                                Viewer.LoadOrReloadModel.Send(model, true);
                                            }
                                        };
                                        ecmbFileWatcher.Changed += (ss, ee) => { fileChanged(); };
                                        ecmbFileWatcher.Created += (ss, ee) => { fileChanged(); };
                                        ecmbFileWatcher.Deleted += (ss, ee) => { fileChanged(); };
                                        ecmbFileWatcher.Renamed += (ss, ee) => { fileChanged(); };
                                        ecmbFileWatcher.EnableRaisingEvents = true;
                                    }
                                    catch
                                    {
                                        ecmbFileWatcher?.Dispose();
                                        ecmbFileWatcher = null;
                                    }
                                    ecmbFileWatchers.Add(ecmbFilePath, ecmbFileWatcher);
                                }
                            }
                        }
                    }

                    // バイナライズ成功時のフラグを設定。
                    SucceededInBinarize();

                    return binaryPath;
                }

                FailedToBinarize = true;
                combinerShaderFileLastWriteTimes.Remove(binaryPath);

                // UI 更新用に変更通知
                TheApp.MainFrame.BeginInvokeOrExecute(
                    () => App.AppContext.NotifyPropertyChanged(null, new DocumentContentsChangedArgs(this, null)));

                return null;
            }
        }

        private object bfshaPathLock = new object();

        // シェーディングモデルごとのバイナリパス
        // キー：シェーディングモデル、プラットフォーム、ターゲット
        private Dictionary<Tuple<string, string, string>, string> bfshaPaths = new Dictionary<Tuple<string, string, string>, string>();

        // バイナリ作成時のコンバイナーシェーダーファイル書き込み日時
        // 親キー：バイナリファイルパス
        // 子キー：コンバイナーシェーダーファイルパス
        private Dictionary<string, Dictionary<string, DateTime>> combinerShaderFileLastWriteTimes = new Dictionary<string, Dictionary<string, DateTime>>();

        // コンバイナーシェーダーディレクトリウォッチャー
        // キー：コンバイナーシェーダーディレクトリ
        private Dictionary<string, FileSystemWatcher> combinerShaderDirectoryWatchers = new Dictionary<string, FileSystemWatcher>();

        // コンバイナーシェーダーファイルウォッチャー
        // 親キー：コンバイナーシェーダーディレクトリウォッチャー
        // 子キー：コンバイナーシェーダーディレクトリパス
        private Dictionary<FileSystemWatcher, Dictionary<string, FileSystemWatcher>> combinerShaderFileWatchers = new Dictionary<FileSystemWatcher, Dictionary<string, FileSystemWatcher>>();

        /// <summary>
        /// 変更されたシェーディングモデルを除いてコピーする
        /// </summary>
        public void CopyBinaryPaths(ShaderDefinition shaderDefinition, string changedSourceName)
        {
            var changedShadingModels = new HashSet<string>(
                (from index in shaderDefinition.ShadingModelIndices(changedSourceName)
                 select shaderDefinition.Data.shading_model_array.Items[index].name));
            lock (shaderDefinition.bfshaPathLock)
            {
                bfshaPaths = shaderDefinition.bfshaPaths.Where(x => !changedShadingModels.Contains(x.Key.Item1)).ToDictionary(p => p.Key, p=> p.Value);
            }
        }

        private readonly object FailedToBinarizeLock = new object();

        /// <summary>
        /// 現在の接続先でバイナリ生成に失敗したか？
        /// </summary>
        public bool FailedToBinarize
        {
            get
            {
                lock (FailedToBinarizeLock)
                {
                    var key = new Tuple<string, string>(Viewer.Connecter.Platform.Name, Viewer.Manager.IsDeviceTargeted ? "Device" : "PC");
                    return failedToBinarize.Contains(key);
                }
            }
            set
            {
                Debug.Assert(value == true);
                SetFailedToBinarize(value);
            }
        }

        private HashSet<Tuple<string, string>> failedToBinarize = new HashSet<Tuple<string, string>>();

        /// <summary>
        /// FailedToBinarize の set で false が設定できないようになっているので、それを迂回するための関数。
        /// </summary>
        private void SetFailedToBinarize(bool value)
        {
            lock (FailedToBinarizeLock)
            {
                var key = new Tuple<string, string>(Viewer.Connecter.Platform.Name, Viewer.Manager.IsDeviceTargeted ? "Device" : "PC");

                if (!Viewer.Manager.IsDeviceTargeted)
                {
                    CompileTested = true;
                }

                if (value)
                {
                    if (!failedToBinarize.Contains(key))
                    {
                        failedToBinarize.Add(key);
                    }
                }
                else
                {
                    failedToBinarize.Remove(key);
                }
            }
        }

        /// <summary>
        /// 現在の接続先でバイナリ生成に成功した場合に行う処理
        /// </summary>
        public void SucceededInBinarize()
        {
            SetFailedToBinarize(false);
        }

        private HashSet<Tuple<string, string>> compileTested = new HashSet<Tuple<string, string>>();
        public bool CompileTested
        {
            get
            {
                lock (FailedToBinarizeLock)
                {
                    var key = new Tuple<string, string>(Viewer.Connecter.Platform.Name, Viewer.Manager.IsDeviceTargeted ? "Device" : "PC");
                    return compileTested.Contains(key);
                }
            }
            set
            {
                Debug.Assert(value == true);
                lock (FailedToBinarizeLock)
                {
                    var key = new Tuple<string, string>(Viewer.Connecter.Platform.Name, Viewer.Manager.IsDeviceTargeted ? "Device" : "PC");
                    if (value)
                    {
                        compileTested.Add(key);
                    }
                    else
                    {
                        compileTested.Remove(key);
                    }
                }
            }
        }
        #endregion

        private List<Definition> CreateDefinitions(shader_definitionType data, ShaderDefinition owner)
        {
            return data.shading_model_array.GetItems()
                .Select(x =>
                    new Definition()
                    {
                        Name = x.name,
                        owner = owner,
                        Data = x,
                        shadingModelTable = new Definition.ShadingModelTable()
                        {
                            optionTable = x.Options().ToDictionary(y => y.id),
                            attribTable = x.Attributes().ToDictionary(y => y.id),
                            samplerTable = x.Samplers().ToDictionary(y => y.id),
                            uniformTable = x.MaterialUniforms().ToDictionary(y => y.id),
                            renderInfoTable = x.RenderInfoSlots().ToDictionary(y => y.name),
                            groupTable = x.Groups().ToDictionary(y => y.name),
                            pageTable = x.Pages().ToDictionary(y => y.name),
                        },
                    }
                    ).ToList();
        }

        private Dictionary<string, string> CreateDisplayIncludePaths(shader_definitionType data)
        {
            return data.shader_src_array.GetItems().ToDictionary(source => source.path, source => source.include_path);
        }

        public override bool CheckOwnerDocumentSavedContents()
        {
            return false;
        }

        #region Bridge
        // モデルデータ差し替え用
        public class SwapData
        {
            public ToolData ToolData;
            public List<G3dStream> streams;
            public shader_definitionType data;
            public List<Definition> Definitions;
            public Dictionary<string, string> DisplayIncludePaths;
        }

        public SwapData CreateSwapData(shader_definitionType data, List<G3dStream> streams)
        {
            var toolData = new ToolData();
            toolData.Load(data.tool_data);
            return new SwapData()
            {
                ToolData = toolData,
                streams = streams,
                data = data,
                Definitions = CreateDefinitions(data, this),
                DisplayIncludePaths = CreateDisplayIncludePaths(data),
            };
        }

        public SwapData GetSwapData()
        {
            return new SwapData()
            {
                ToolData = ToolData,
                streams = BinaryStreams,
                data = Data,
                Definitions = Definitions,
                DisplayIncludePaths = DisplayIncludePaths,
            };
        }

        public SwapData Swap(SwapData data, bool copyHistory)
        {
            GlslFilePathList = new List<string>();
            var current = GetSwapData();
            ToolData = data.ToolData;
            BinaryStreams = data.streams;
            Data = data.data;
            Definitions = data.Definitions;
            DisplayIncludePaths = data.DisplayIncludePaths;

            if (copyHistory)
            {
                UpdateModifiedSources();
                UpdateModifiedShadingModels();
            }

            return current;
        }
        #endregion
        /// <summary>
        /// シェーディングモデル毎の参照ソース
        /// </summary>
        public Dictionary<string, string[]> SourceFileNames = new Dictionary<string, string[]>();

        /// <summary>
        /// UI表示用の include_path
        /// </summary>
        public Dictionary<string, string> DisplayIncludePaths = new Dictionary<string, string>();

        private List<string> glslFilePathList_;
        public List<string> GlslFilePathList {
            get
            {
                return glslFilePathList_;
            }
            set
            {
                glslFilePathList_ = value;
                EnableSourceWatcher();
            }

        }

        public override object nw4f_3difItem
        {
            get { return Data; }
        }

        public override nw4f_3difType Create_nw4f_3difType(bool viewer)
        {
            lock (this)
            {
                //Debug.Assert(false);
                //throw new NotImplementedException();


                //			UpdateData();
                //			nw4f_3difType nw4f_3dif = new nw4f_3difType();
                nw4f_3dif_.Item = Data;
                nw4f_3dif_.file_info = null;

                return nw4f_3dif_;
            }
        }

        // 呼び出される?
        public override process_log_arrayType process_log_array
        {
            get {
                Debug.Assert(false);
                return null; // new ？
            }
        }

        public override bool Editable
        {
            get
            {
                return false;
            }
        }

        private List<FileSystemWatcher> watchers_ = new List<FileSystemWatcher>();
        public void EnableSourceWatcher()
        {
            try
            {
                if (GlslFilePathList == null)
                {
                    return;
                }

                if (Data.shader_definition_info == null)
                {
                    return;
                }

                if (Data.shader_src_array == null)
                {
                    return;
                }

                if (watchers_ == null)
                {
                    watchers_ = new List<FileSystemWatcher>();
                }
                DisableSourceWatcher();

                //var paths = Data.shader_src_array.shader_src.Select(x => Path.Combine(FileLocation, x.include_path, x.path));
                foreach (var path in GlslFilePathList)
                {
                    if (File.Exists(path) == false)
                    {
                        return;
                    }

                    var watcher = new FileSystemWatcher()
                    {
                        Path				= Path.GetDirectoryName(path),
                        Filter				= Path.GetFileName(path),
                        EnableRaisingEvents	= true,
                        NotifyFilter		= NotifyFilters.LastWrite | NotifyFilters.CreationTime,
                        SynchronizingObject	= TheApp.MainFrame,
                    };

                    // 変更イベントハンドラ
                    var lastWriteTime = File.GetLastWriteTime(path);
                    watcher.Changed += (s, e) =>
                    {
                        if (!watchers_.Contains(watcher))
                        {
                            // 既に破棄されたWatcherなのに、イベントだけは残っている状態。
                            return;
                        }


                        if (File.Exists(e.FullPath) == false)
                        {
                            return;
                        }

                        // 稀にファイルハンドルの競合が起こるので、他のアプリが手放すのを待つ
                        for (int i = 0; i < 10; i++)
                        {
                            try
                            {
                                using (var stream = File.Open(e.FullPath, FileMode.Open, FileAccess.Read))
                                {
                                    break;
                                }
                            }
                            catch (Exception exception)
                            {
                                Thread.Sleep(100);
                                DebugConsole.WriteLine("wacherChanged " + exception.Message);
                            }
                        }

                        var time     = File.GetLastWriteTime(e.FullPath);
                        var timeSpan = Math.Abs((time - lastWriteTime).TotalMilliseconds);
                        if (timeSpan < 100.0)
                        {
                            return;
                        }
                        lastWriteTime = time;

                        if (RebuildFromFsc(e.FullPath))
                        {
                            DisableSourceWatcher();
DebugConsole.WriteLine("EnableSourceWatcher()  : >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>{0}", e.FullPath);

                            MessageLog.Write(MessageLog.LogType.Information, string.Format(Strings.ShaderDefinition_ReloadOnFileUpdating, e.FullPath, FileName));
                        }
                    };

                    watchers_.Add(watcher);
                }
            }
            catch
            {
                watchers_ = null;
                return;
            }
        }

        public void DisableSourceWatcher()
        {
            foreach (var watcher in watchers_)
            {
                watcher.Dispose();
            }

            watchers_.Clear();
        }

        // 修正があったglslファイルから、それを使用しているShadingModelを探す
        public void SendEditShadingModels(string fileName)
        {
            List<int> indices = new List<int>();
            for (int i = 0; i < Definitions.Count; i++)
            {
                // 編集されかつマテリアルシェーダー以外
                string[] srcs;
                if ((!SourceFileNames.TryGetValue(Definitions[i].Name, out srcs) ||
                     srcs.Contains(fileName))/* &&
                    !Definitions[i].Data.material_shader*/)
                {
                    indices.Add(i);
                }
            }

            if (indices.Any())
            {
                Viewer.GeneralActionMessage.Send(() => Viewer.HioUtility.EditShadingModels(this, indices.ToArray()));
            }
        }

        private bool RebuildFromFsc(string fullPath)
        {
            if (Data.shader_definition_info == null)
            {
                return false;
            }

            {
                string fsdbFilePath;
                // fscファイルを生成し、テンポラリのfsdファイルを作成します。
                byte[] fsdFile;
                fsdbFilePath = ShaderDifinitionUtility.RebuildShaderDefinition(this, out fsdFile, false, App.AppContext.SelectedPlatformPreset);

                if (string.IsNullOrEmpty(fsdbFilePath))
                {
                    return false;
                }

                // fsd再読み込みを行う。
                if (!TheApp.MainFrame.CanFocus || App.AppContext.UpdateFromOutSideBlockCounter.CheckBlock)
                {
                    EventHandler waitCanFocus = null;
                    waitCanFocus = (s, e) =>
                    {
                        // フォーカスを持てるようになるまで待つ
                        if (TheApp.MainFrame.CanFocus && !App.AppContext.UpdateFromOutSideBlockCounter.CheckBlock)
                        {
                            Application.Idle -= waitCanFocus;

                            if (DocumentManager.Documents.Contains(this))
                            {
                                DocumentManager.ReplaceShaderDefinition(this, fsdbFilePath, fullPath, fsdFile, false, true);
                            }
                        }
                    };
                    Application.Idle += waitCanFocus;
                }
                else
                {
                    if (DocumentManager.Documents.Contains(this))
                    {
                        DocumentManager.ReplaceShaderDefinition(this, fsdbFilePath, fullPath, fsdFile, false, true);
                    }
                }
            }

            return true;
        }

        public byte[] CreateDefaultFsvImage(IEnumerable<MaterialShaderAssign> assigns)
        {
            var shaderAssigns = from assign in assigns
                                where !string.IsNullOrEmpty(assign.ShaderName)
                                group assign by assign.ShaderName;

            // fsv
            var target_shaders = new List<target_shaderType>();
            foreach (var shaderAssign in shaderAssigns)
            {
                var target_shader = new target_shaderType()
                {
                    shading_model_name = shaderAssign.Key,
                };
                var shaderPrograms = new List<shader_programType>();
                MaterialShaderAssign materialShaderAssign = shaderAssign.First();
                if (materialShaderAssign.ShadingModel.IsMaterialShader())
                {
                    var shadingModel = materialShaderAssign.Definition;
                    var combinerIds = new HashSet<string>(shadingModel.Data.Options().Where(x => x.ui_item?.value == ui_item_valueType.combiner).Select(x => x.id));

                    var dynamicOptions = materialShaderAssign.ShadingModel.Options().Where(x => x.type == option_var_typeType.dynamic).Select(x => new optionType() { id = x.id, choice = "*" });
                    foreach (var options in shaderAssign.Select(x => x.ShaderOptions).Distinct(new ShaderOptionUtility.OptionsComparer()))
                    {
                        var shader_program = new shader_programType();
                        shader_program.option_array = options.Select(x => new optionType() { id = x.id, choice = x.value }).Concat(dynamicOptions)
                            .ToG3dArrayElement<optionType, option_arrayType>();
                        shaderPrograms.Add(shader_program);
                    }

                    target_shader.shader_program_array = shaderPrograms.ToG3dArrayElement<shader_programType, shader_program_arrayType>();
                    target_shaders.Add(target_shader);
                }
            }

            if (target_shaders.Any())
            {
                var fsv = new nw4f_3difType()
                {
                    Item = new shader_variationType()
                    {
                        shader_variation_info = new shader_variation_infoType()
                        {
                            shader_archive = Name,
                        },
                        target_shader_array = target_shaders.ToG3dArrayElement<target_shaderType, target_shader_arrayType>(),
                    }
                };

                lock (Viewer.Connecter.AbortLock)
                {
                    return IfBinaryFormatter.FormatStream(fsv, new List<G3dStream>(), null);
                }
            }
            else
            {
                return null;
            }
        }

        public byte[] CreateDefaultFsvImage(IEnumerable<materialType> materials)
        {
            var shaderAssigns = from material in materials
                                where material.shader_assign != null
                                group material.shader_assign by material.shader_assign.shading_model;

                        // fsv
            var target_shaders = new List<target_shaderType>();
            foreach (var shaderAssign in shaderAssigns)
            {
                var target_shader = new target_shaderType()
                {
                    shading_model_name = shaderAssign.Key,
                };
                var shadingModel = Data.shading_model_array.GetItems().FirstOrDefault(x => x.name == shaderAssign.Key);
                var shaderPrograms = new List<shader_programType>();
                var materialShaderAssign = shaderAssign.First();
                if (shadingModel != null && shadingModel.IsMaterialShader())
                {
                    var dynamicOptions = shadingModel.Options().Where(x => x.type == option_var_typeType.dynamic).Select(x => new optionType() { id = x.id, choice = "*" });
                    foreach (var options in shaderAssign.Select(x => x.ShaderOptions().ToList()).Distinct(new ShaderOptionUtility.OptionsComparer()))
                    {
                        var shader_program = new shader_programType();
                        shader_program.option_array = options.Select(x => new optionType() { id = x.id, choice = x.value }).Concat(dynamicOptions)
                            .ToG3dArrayElement<optionType, option_arrayType>();
                        shaderPrograms.Add(shader_program);
                    }

                    target_shader.shader_program_array = shaderPrograms.ToG3dArrayElement<shader_programType, shader_program_arrayType>();
                    target_shaders.Add(target_shader);
                }
            }

            if (target_shaders.Any())
            {
                var fsv = new nw4f_3difType()
                {
                    Item = new shader_variationType()
                    {
                        shader_variation_info = new shader_variation_infoType()
                        {
                            shader_archive = Name,
                        },
                        target_shader_array = target_shaders.ToG3dArrayElement<target_shaderType, target_shader_arrayType>(),
                    }
                };

                lock (Viewer.Connecter.AbortLock)
                {
                    return IfBinaryFormatter.FormatStream(fsv, new List<G3dStream>(), null);
                }
            }
            else
            {
                return null;
            }
        }



        #region savedData
        public shader_definitionType savedData;
        public List<G3dStream> savedStream;
        public Dictionary<string, string> savedDisplayIncludePaths;
        public HashSet<string> ModifiedSources;
        public HashSet<string> ModifiedShadingModels;

        public override void UpdateSavedData()
        {
            base.UpdateSavedData();
            savedData = ObjectUtility.Clone(Data);
            savedStream = BinaryStreams.ToList();
            savedDisplayIncludePaths = new Dictionary<string, string>();
            foreach (var pair in DisplayIncludePaths)
            {
                savedDisplayIncludePaths.Add(pair.Key, pair.Value);
            }

            ModifiedSources = new HashSet<string>();
            ModifiedShadingModels = new HashSet<string>();
        }

        public void CopySavedData(ShaderDefinition source)
        {
            CopyDocumentSavedData(source);
            savedData = source.savedData;
            savedStream = source.savedStream;
            savedDisplayIncludePaths = source.savedDisplayIncludePaths;
            UpdateModifiedSources();
            UpdateModifiedShadingModels();
        }

        /// <summary>
        /// ModifiedSources を更新する
        /// </summary>
        public void UpdateModifiedSources()
        {
            ModifiedSources.Clear();
            foreach (var src in Data.shader_src_array.shader_src)
            {
                var saved = savedData.shader_src_array.shader_src.FirstOrDefault(x => x.path == src.path);
                if (saved != null)
                {
                    var currentStream = BinaryStreams[src.stream_index];
                    var savedStr = savedStream[saved.stream_index];
                    if (IsDisplayIncludePathModified(src.path) || currentStream.StringData != savedStr.StringData)
                    {
                        ModifiedSources.Add(src.path);
                    }
                }
                else
                {
                    ModifiedSources.Add(src.path);
                }
            }
        }

        public bool IsDisplayIncludePathModified(string path)
        {
            if (!savedDisplayIncludePaths.ContainsKey(path))
            {
                return false;
            }

            Debug.Assert(DisplayIncludePaths.ContainsKey(path));

            return DisplayIncludePaths[path] != savedDisplayIncludePaths[path];
        }

        public shading_modelType GetSavedShadingModel(string name)
        {
            return savedData.shading_model_array.shading_model.FirstOrDefault(x => x.name == name);
        }

        public void UpdateModifiedShadingModels()
        {
            ModifiedShadingModels.Clear();

            foreach (var c in Data.shading_model_array.shading_model)
            {
                var s = savedData.shading_model_array.shading_model.FirstOrDefault(x => x.name == c.name);
                if (s == null)
                {
                    ModifiedShadingModels.Add(c.name);
                    continue;
                }

                if (c.revision != s.revision ||
                    c.material_shader != s.material_shader ||
                    IsModified(c, s, x => x.macro_array) ||
                    IsModified(c, s, x => x.option_var_array) ||
                    IsModified(c, s, x => x.attrib_var_array) ||
                    IsModified(c, s, x => x.sampler_var_array) ||
                    IsModified(c, s, x => x.block_var_array) ||
                    IsModified(c, s, x => x.render_info_slot_array) ||
                    IsModified(c, s, x => x.interleave_array) ||
                    IsModified(c, s, x => x.group_array) ||
                    IsModified(c, s, x => x.page_array) ||
                    IsModified(c, s, x => x.streamout))
                {
                    ModifiedShadingModels.Add(c.name);
                    continue;
                }

                if (c.vertex_stage == null || s.vertex_stage == null)
                {
                    if (c.vertex_stage != s.vertex_stage)
                    {
                        ModifiedShadingModels.Add(c.name);
                        continue;
                    }
                }
                else if (Data.shader_src_array.shader_src[c.vertex_stage.src_index].path != savedData.shader_src_array.shader_src[s.vertex_stage.src_index].path ||
                        ModifiedSources.Contains(Data.shader_src_array.shader_src[c.vertex_stage.src_index].path))
                {
                    ModifiedShadingModels.Add(c.name);
                    continue;
                }

                if (c.geometry_stage == null || s.geometry_stage == null)
                {
                    if (c.geometry_stage != s.geometry_stage)
                    {
                        ModifiedShadingModels.Add(c.name);
                        continue;
                    }
                }
                else if (Data.shader_src_array.shader_src[c.geometry_stage.src_index].path != savedData.shader_src_array.shader_src[s.geometry_stage.src_index].path ||
                        ModifiedSources.Contains(Data.shader_src_array.shader_src[c.geometry_stage.src_index].path))
                {
                    ModifiedShadingModels.Add(c.name);
                    continue;
                }

                if (c.fragment_stage == null || s.fragment_stage == null)
                {
                    if (c.fragment_stage != s.fragment_stage)
                    {
                        ModifiedShadingModels.Add(c.name);
                        continue;
                    }
                }
                else if (Data.shader_src_array.shader_src[c.fragment_stage.src_index].path != savedData.shader_src_array.shader_src[s.fragment_stage.src_index].path ||
                        ModifiedSources.Contains(Data.shader_src_array.shader_src[c.fragment_stage.src_index].path))
                {
                    ModifiedShadingModels.Add(c.name);
                    continue;
                }

                if (c.compute_stage == null || s.compute_stage == null)
                {
                    if (c.compute_stage != s.compute_stage)
                    {
                        ModifiedShadingModels.Add(c.name);
                        continue;
                    }
                }
                else if (Data.shader_src_array.shader_src[c.compute_stage.src_index].path != savedData.shader_src_array.shader_src[s.compute_stage.src_index].path ||
                    ModifiedSources.Contains(Data.shader_src_array.shader_src[c.compute_stage.src_index].path))
                {
                    ModifiedShadingModels.Add(c.name);
                    continue;
                }
            }
        }

        public bool IsValueChanged<T>(Func<shader_definitionType, T> select) where T : struct
        {
            return !select(Data).Equals(select(savedData));
        }

        public bool IsStringChanged(Func<shader_definitionType, string> select)
        {
            return select(Data) != select(savedData);
        }

        public bool IsValueChanged<T>(shading_modelType c, shading_modelType s, Func<shading_modelType, T> select) where T: struct
        {
            return !select(c).Equals(select(s));
        }

        public bool IsModified<T>(shading_modelType c, shading_modelType s, Func<shading_modelType, T> select)
        {
            return !ObjectUtility.ToBytes(select(c)).SequenceEqual(ObjectUtility.ToBytes(select(s)));
        }

        public static IEnumerable<shader_srcType> ForceIncludeSrcs(shader_definitionType definition)
        {
            return definition.force_include_array.GetItems().Select(x => definition.shader_src_array.shader_src[x.src_index]);
        }

        public bool IsForceIncludeModified(string path)
        {
            return ForceIncludeSrcs(Data).Any(x => x.path == path)
                != ForceIncludeSrcs(savedData).Any(x => x.path == path);
        }

        public bool IsForceIncludeModified()
        {
            var d = ForceIncludeSrcs(Data).ToArray();
            var s = ForceIncludeSrcs(Data).ToArray();
            return d.Length != s.Length || !d.Zip(s, (x,y)=> x.path == y.path).All(x => x);
        }

        public override bool EqualsToSavedData()
        {
            if (!base.EqualsToSavedData())
            {
                return false;
            }

            return !ShaderDefinitionRootPage.IsModified(this) &&
                !ShaderDefinitionSourceCodePage.IsModified(this) &&
                !ModifiedSources.Any() &&
                !ModifiedShadingModels.Any();
        }
        #endregion

        public override void OnDocumentSwapAdd(Document old)
        {
            var oldShaderDefinition = (ShaderDefinition)old;
            DisplayIncludePaths = oldShaderDefinition.DisplayIncludePaths;
            SourceFileNames = oldShaderDefinition.SourceFileNames;
            CopySavedData(oldShaderDefinition);
            base.OnDocumentSwapAdd(old);
        }

        public static event Action<Document, bool> IsAttachedChanged = null;
        public override bool IsAttached
        {
            get
            {
                return base.IsAttached;
            }
            set
            {
                bool changed = base.IsAttached != value;
                base.IsAttached = value;
                if (changed && IsAttachedChanged != null)
                {
                    IsAttachedChanged(this, value);
                }
            }
        }
    }

    public class VisibleInfo
    {
        public VisibleInfo(Material material)
        {
            Debug.Assert(material != null);
            var shadingModel = material.MaterialShaderAssign.Definition;
            var options      = material.MaterialShaderAssign.ShaderOptions;
            var uniforms     = material.MaterialShaderAssign.ShaderParams;
            var renderInfos  = material.MaterialShaderAssign.RenderInfos;

            invisibleGroups = new HashSet<string>();
            conditionOptionVariables = new HashSet<string>();
            conditionUniformVariables = new HashSet<string>();
            conditionRenderInfoVariables = new HashSet<string>();
            conditionContainingGroups = new HashSet<string>();
            conditionGroupEvalResults = new Dictionary<string, App.Utility.ExpressionEvaluator.EvalResult>();
            conditionGroupCondition = new Dictionary<string, string>();

            if (shadingModel != null)
            {
                Dictionary<string, string> optionValues = options.ToDictionary(x => x.id, material.GetResolvedOptionValue);
                Dictionary<string, option_varType> optionVars = shadingModel.Data.Options()
                    .Where(x => x.type != option_var_typeType.dynamic)
                    .ToDictionary(x => x.id, x => x);

                Dictionary<string, string> uniformValues = uniforms.ToDictionary(x => x.id, material.GetResolvedShaderParamValue);
                Dictionary<string, uniform_varType> uniformVars = shadingModel.Data.Uniforms().ToDictionary(x => x.id, x => x);

                Dictionary<string, List<string>> renderInfoValues = renderInfos.ToDictionary(x => x.name, material.GetResolvedRenderInfoValues);
                Dictionary<string, render_info_slotType> renderInfoSlots = shadingModel.Data.RenderInfoSlots().ToDictionary(x => x.name, x => x);

                foreach (var group in shadingModel.Data.Groups())
                {
                    List<string> optionVariables, uniformVariables, renderInfoVariables;
                    App.Utility.ExpressionEvaluator.EvalResult evalResult;

                    var result = Evaluate(
                        group,
                        optionValues, out optionVariables,
                        uniformValues, out uniformVariables,
                        renderInfoValues, renderInfoSlots, out renderInfoVariables,
                        out evalResult);
                    if (result == false)
                    {
                        invisibleGroups.Add(group.name);
                    }

                    foreach (var variable in optionVariables)
                    {
                        option_varType option;
                        optionVars.TryGetValue(variable, out option);

                        // 自分自身の group に対する条件のときのみ conditionOptionVariables に含める
                        if (option != null && option.Group() == group.name)
                        {
                            conditionOptionVariables.Add(variable);
                        }
                    }

                    foreach (var variable in uniformVariables)
                    {
                        uniform_varType uniform;
                        uniformVars.TryGetValue(variable, out uniform);

                        // 自分自身の group に対する条件のときのみ conditionUniformVariables に含める
                        if (uniform != null && uniform.Group() == group.name)
                        {
                            conditionUniformVariables.Add(variable);
                        }
                    }

                    foreach (var variable in renderInfoVariables)
                    {
                        render_info_slotType renderInfo;
                        renderInfoSlots.TryGetValue(variable, out renderInfo);

                        // 自分自身の group に対する条件のときのみ conditionRenderInfoVariables に含める
                        if (renderInfo != null && renderInfo.Group() == group.name)
                        {
                            conditionRenderInfoVariables.Add(variable);
                        }
                    }

                    conditionGroupEvalResults[group.name] = evalResult;
                    conditionGroupCondition[group.name] = group.condition;
                }

                foreach (var variable in conditionOptionVariables)
                {
                    option_varType optionValue;
                    if (optionVars.TryGetValue(variable, out optionValue))
                    {
                        var group = optionValue.Group();
                        if (!string.IsNullOrEmpty(group) && !conditionContainingGroups.Contains(group))
                        {
                            conditionContainingGroups.Add(group);
                        }
                    }
                }

                foreach (var variable in conditionUniformVariables)
                {
                    uniform_varType uniformValue;
                    if (uniformVars.TryGetValue(variable, out uniformValue))
                    {
                        var group = uniformValue.Group();
                        if (!string.IsNullOrEmpty(group) && !conditionContainingGroups.Contains(group))
                        {
                            conditionContainingGroups.Add(group);
                        }
                    }
                }

                foreach (var variable in conditionRenderInfoVariables)
                {
                    render_info_slotType renderInfoValue;
                    if (renderInfoSlots.TryGetValue(variable, out renderInfoValue))
                    {
                        var group = renderInfoValue.Group();
                        if (!string.IsNullOrEmpty(group) && !conditionContainingGroups.Contains(group))
                        {
                            conditionContainingGroups.Add(group);
                        }
                    }
                }
            }
        }

        // 変数名とそれ以外
        public static Regex regex = new Regex(@"[^$]+|\$[0-9A-Za-z_\.]*");

        // 英数字と_からなる文字列で数字でないもの
        public static Regex regexOption = new Regex("[0-9A-Za-z_]*[A-Za-z_][0-9A-Za-z_]*");

        // Condition を評価
        // values は変数の値
        private bool Evaluate(
            groupType group,
            Dictionary<string, string> optionValues, out List<string> optionVariables,
            Dictionary<string, string> uniformValues, out List<string> uniformVariables,
            Dictionary<string, List<string>> renderInfoValues, Dictionary<string, render_info_slotType> renderInfoSlots, out List<string> renderInfoVariables,
            out ExpressionEvaluator.EvalResult evalResult)
        {
            const string optionPrefix = "option.";
            const string uniformPrefix = "uniform.";
            const string renderInfoPrefix = "renderinfo.";

            // Item1 ... 変数 ($) の場合は、その id や name。それ以外は取得した文字列。
            // Item2 ... Item1 が変数の場合は、その種類 (optionPrefix, uniformPrefix, renderInfoPrefix)。それ以外は null。
            var tokens = new List<Tuple<string, string>>();

            optionVariables = new List<string>();
            uniformVariables = new List<string>();
            renderInfoVariables = new List<string>();
            if (!string.IsNullOrWhiteSpace(group.condition))
            {
                var matches = regex.Matches(group.condition);
                foreach (Match match in matches)
                {
                    if (match.Value[0] == '$')
                    {
                        var v = match.Value.Substring(1);

                        if (v.StartsWith(uniformPrefix))
                        {
                            // uniform
                            var id = v.Substring(uniformPrefix.Length);
                            tokens.Add(new Tuple<string, string>(id, uniformPrefix));
                            if (!uniformVariables.Contains(id))
                            {
                                uniformVariables.Add(id);
                            }
                        }
                        else if (v.StartsWith(renderInfoPrefix))
                        {
                            // renderinfo
                            var name = v.Substring(renderInfoPrefix.Length);
                            tokens.Add(new Tuple<string, string>(name, renderInfoPrefix));
                            if (!renderInfoVariables.Contains(name))
                            {
                                renderInfoVariables.Add(name);
                            }
                        }
                        else
                        {
                            // option
                            // 互換性維持のために prefix なしのものも option に含める。
                            var id = v.StartsWith(optionPrefix) ? v.Substring(optionPrefix.Length) : v;
                            tokens.Add(new Tuple<string, string>(id, optionPrefix));
                            if (!optionVariables.Contains(id))
                            {
                                optionVariables.Add(id);
                            }
                        }
                    }
                    else
                    {
                        tokens.Add(new Tuple<string, string>(match.Value, null));
                    }
                }
            }

            if (tokens.Count == 0)
            {
                evalResult = ExpressionEvaluator.EvalResult.Success;
                return true;
            }

            StringBuilder builder = new StringBuilder();

            foreach (var token in tokens)
            {
                if (token.Item2 != null)
                {
                    string prefix = token.Item2;

                    string[] values = null;
                    if (prefix == optionPrefix)
                    {
                        string optionValue;
                        if (optionValues.TryGetValue(token.Item1, out optionValue))
                        {
                            values = new string[] { optionValue };
                        }
                    }
                    else if (prefix == uniformPrefix)
                    {
                        string uniformValue;
                        if (uniformValues.TryGetValue(token.Item1, out uniformValue))
                        {
                            values = uniformValue.Trim().Split(null);
                        }
                    }
                    else if (prefix == renderInfoPrefix)
                    {
                        render_info_slotType slot;
                        if (renderInfoSlots.TryGetValue(token.Item1, out slot))
                        {
                            if (slot.optional)
                            {
                                // Optional 有効時には、常に失敗させる。
                                evalResult = ExpressionEvaluator.EvalResult.IncorrectExpression;
                                return true;
                            }
                        }

                        List<string> renderInfoValue;
                        if (renderInfoValues.TryGetValue(token.Item1, out renderInfoValue))
                        {
                            values = renderInfoValue.ToArray();
                        }
                    }

                    if ((values != null) && (values.Length == 1))
                    {
                        string value = values[0];
                        if (regexOption.Match(value).Length == value.Length)
                        {
                            // " で囲む
                            value = "\"" + value + "\"";
                        }
                        builder.Append(value);
                    }
                    else
                    {
                        evalResult = ExpressionEvaluator.EvalResult.IncorrectExpression;
                        return true;
                    }
                }
                else
                {
                    var replaced = regexOption.Replace(token.Item1, x =>
                    {
                        switch (x.Value)
                        {
                            case "true":
                                return "1";
                            case "false":
                                return "0";
                        }
                        // " で囲む
                        return "\"" + x.Value + "\"";
                    });
                    builder.Append(replaced);
                }
            }

            return ExpressionEvaluator.Eval(builder.ToString(), true, out evalResult);
        }

        private readonly HashSet<string> conditionOptionVariables;
        private readonly HashSet<string> conditionUniformVariables;
        private readonly HashSet<string> conditionRenderInfoVariables;
        private readonly HashSet<string> invisibleGroups;
        private readonly HashSet<string> conditionContainingGroups;
        private readonly Dictionary<string, App.Utility.ExpressionEvaluator.EvalResult> conditionGroupEvalResults;
        private readonly Dictionary<string, string> conditionGroupCondition;			// グループ, コンディション式

        public bool IsVisible(string group)
        {
            return !invisibleGroups.Contains(group);
        }

        public bool ContainsCondition(string group)
        {
            return conditionContainingGroups.Contains(group);
        }

        public bool IsConditionEvalSuccess(string group, out string condition)
        {
            App.Utility.ExpressionEvaluator.EvalResult	evalResult;
            if (conditionGroupEvalResults.TryGetValue(group, out evalResult))
            {
                Debug.Assert(conditionGroupCondition.ContainsKey(group));

                condition = conditionGroupCondition[group];
                return evalResult == ExpressionEvaluator.EvalResult.Success;
            }
            else
            {
                condition = null;
                return true;
            }
        }

        public bool IsVisible(option_varType option, bool showConditionVariables)
        {
            return option.OptionVisible() &&
                    (option.ui_group == null
                       || IsVisible(option.ui_group.group_name)
                       || (showConditionVariables && conditionOptionVariables.Contains(option.id)));
        }

        public bool IsParamVisible(uniform_varType uniform, bool showConditionVariables)
        {
            return uniform.IsParamVisible() &&
                    (uniform.ui_group == null
                       || IsVisible(uniform.ui_group.group_name)
                       || (showConditionVariables && conditionUniformVariables.Contains(uniform.id)));
        }

        public bool IsVisible(render_info_slotType slot, bool showConditionVariables)
        {
            return slot.RenderInfoVisible() &&
                    (slot.Group() == string.Empty
                        || IsVisible(slot.Group())
                        || (showConditionVariables && conditionRenderInfoVariables.Contains(slot.name)));
        }

        public bool IsConditionVariable(option_varType option)
        {
            return conditionOptionVariables.Contains(option.id);
        }

        public bool IsConditionVariable(uniform_varType uniform)
        {
            return conditionUniformVariables.Contains(uniform.id);
        }

        public bool IsConditionVariable(render_info_slotType slot)
        {
            return conditionRenderInfoVariables.Contains(slot.name);
        }

        public bool IsVisible(sampler_varType sampler)
        {
            return sampler.SamplerVisible() && (sampler.ui_group == null || IsVisible(sampler.ui_group.group_name));
        }

        public bool IsVisible(attrib_varType attrib_var)
        {
            return attrib_var.ui_group == null || IsVisible(attrib_var.ui_group.group_name);
        }

        public bool IsParamVisible(uniform_varType uniform)
        {
            return uniform.IsParamVisible() && (uniform.ui_group == null || IsVisible(uniform.ui_group.group_name));
        }

        public bool IsAnimVisible(uniform_varType uniform)
        {
            return uniform.IsAnimVisible() && (uniform.ui_group == null || IsVisible(uniform.ui_group.group_name));
        }

        public bool IsVisible(render_info_slotType slot)
        {
            // 中間ファイルが対応するまでの暫定
            return slot.Group() == string.Empty || IsVisible(slot.Group());
        }
    }

    public class Definition : GuiObject
    {
        public override bool Editable
        {
            get
            {
                return false;
            }
        }
        public Definition() : base(GuiObjectID.Definition) { }
        public override Document OwnerDocument
        {
            get { return owner; }
        }

        public override GuiObjectID ObjectID
        {
            get
            {
                return base.ObjectID;
            }
        }

        public static Definition.ShadingModelTable dummyShadingModelTable = new Definition.ShadingModelTable()
        {
            attribTable = new Dictionary<string, attrib_varType>(),
            groupTable = new Dictionary<string, groupType>(),
            optionTable = new Dictionary<string, option_varType>(),
            pageTable = new Dictionary<string, pageType>(),
            renderInfoTable = new Dictionary<string, render_info_slotType>(),
            samplerTable = new Dictionary<string, sampler_varType>(),
            uniformTable = new Dictionary<string, uniform_varType>(),
        };

        public static Definition.ShadingModelTable GetTable(Definition definition)
        {
            return definition != null ? definition.shadingModelTable : dummyShadingModelTable;
        }

        public class ShadingModelTable
        {
            public Dictionary<string, option_varType> optionTable;
            public Dictionary<string, attrib_varType> attribTable;
            public Dictionary<string, sampler_varType> samplerTable;
            public Dictionary<string, uniform_varType> uniformTable;
            public Dictionary<string, render_info_slotType> renderInfoTable;
            public Dictionary<string, groupType> groupTable;
            public Dictionary<string, pageType> pageTable;
        }

        public ShaderDefinition owner;
        public shading_modelType Data;
        public ShadingModelTable shadingModelTable;
        public List<Group> groups = new List<Group>();
        public string VertexShaderPath
        {
            get
            {
                var target = (ShaderDefinition)(OwnerDocument);
                if(target != null)
                {
                    var vshader = Data.vertex_stage;
                    if (vshader != null)
                    {
                        if (0 <= vshader.src_index && vshader.src_index < target.Data.shader_src_array.shader_src.Length)
                        {
                            var shader_src = target.Data.shader_src_array.shader_src[vshader.src_index];
                            return shader_src.path;
                        }
                    }
                }

                return string.Empty;
            }
        }

        public string FragmentShaderPath
        {
            get
            {
                var target = (ShaderDefinition)(OwnerDocument);
                if(target != null)
                {
                    var fshader = Data.fragment_stage;
                    if (fshader != null)
                    {
                        if (0 <= fshader.src_index && fshader.src_index < target.Data.shader_src_array.shader_src.Length)
                        {
                            var shader_src = target.Data.shader_src_array.shader_src[fshader.src_index];
                            return shader_src.path;
                        }
                    }
                }

                return string.Empty;
            }
        }

        public string GeometryShaderPath
        {
            get
            {
                var target = (ShaderDefinition)(OwnerDocument);
                if(target != null)
                {
                    var gshader = Data.geometry_stage;
                    if (gshader != null)
                    {
                        var shader_src = target.Data.shader_src_array.shader_src[gshader.src_index];
                        return shader_src.path;
                    }
                }

                return string.Empty;
            }
        }
    }
}
