﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using App.PropertyEdit;
using App.Utility;
using nw.g3d.iflib.nw3de;
using nw.g3d.nw4f_3dif;

namespace App.Data
{
    public partial class Material
    {
        private materialType savedData;
        private CustomUI savedCustomUI;
        private vtx_buffer_arrayType[] savedVtxBufferArray;
        private nw3de_MaterialReference savedMaterialReference;
        public override void UpdateSavedData()
        {
            base.UpdateSavedData();
            savedData = ObjectUtility.Clone(Data);

            // 保存時と初期読み込み時で動作を変えている
            // TODO: ちゃんと区別する
            if (savedCustomUI == null)
            {
                savedCustomUI = CustomUI.Clone();
            }
            else
            {
                savedCustomUI = ToolData.convertToCustomUI(CreateNw3de_CustomUI());
            }

            savedVtxBufferArray = ObjectUtility.Clone(GetVertexBufferArrays().ToArray());
            SavedUserDataArray = ObjectUtility.Clone(UserDataArray);
            savedMaterialReference = ObjectUtility.Clone(MaterialReference);
            UserDataArrayChanged = false;
        }

        IEnumerable<vtx_buffer_arrayType> GetVertexBufferArrays()
        {
            if (OwnerDocument is Model)
            {
                var model = (Model)OwnerDocument;
                return
                    from shape in model.Data.shape_array.GetItems().Where(x => x.shape_info.mat_name == Name)
                    let vertex = model.Vertices[shape.shape_info.vertex_index]
                    select vertex.vtx_buffer_array;
            }
            else
            {
                return Enumerable.Empty<vtx_buffer_arrayType>();
            }
        }

        public void CopySavedData(Material source)
        {
            CopyGuiObjectSavedData(source);
            savedData = source.savedData;
            savedCustomUI = source.savedCustomUI.Clone();
            savedVtxBufferArray = source.savedVtxBufferArray;
            SavedUserDataArray = source.SavedUserDataArray;
            UserDataArrayChanged = !source.SavedUserDataArray.IsSame(UserDataArray);
        }

        public override bool EqualsToSavedData()
        {
            if (!base.EqualsToSavedData())
            {
                return false;
            }

            return !MaterialGeneralPage.IsModified(this) &&
                !MaterialSamplerPage.IsModified(this) &&
                !MaterialShaderPage.IsModified(this) &&
                !MaterialRenderStatePage.IsModified(this) &&
                !MaterialParentsPage.IsModified(this) &&
                !UserDataPage.IsModified(this);
        }

        public bool IsVtxBufferArrayModified()
        {
            var vtxBufferArrays = GetVertexBufferArrays().ToArray();
            if (vtxBufferArrays.Length != savedVtxBufferArray.Length)
            {
                return true;
            }

            return !vtxBufferArrays.Zip(savedVtxBufferArray, (x, y)=>{
                var ax = x.GetItems().ToArray();
                var ay = y.GetItems().ToArray();
                return ax.Length == ay.Length && ax.Zip(ay, (z, w) =>
                {
                    var az = z.input_array.GetItems().ToArray();
                    var aw = w.input_array.GetItems().ToArray();
                    return az.Length == aw.Length && az.Zip(aw, (p, q) => p.attrib_index == q.attrib_index).All(s => s);
                }).All(s => s);
            }).All(s => s);
        }


        public bool IsShadingModelModified()
        {
            return IsShaderArchiveModified() ||
                IsShaderNameModified() ||
                IsShaderRevisionModified();
        }

        public bool IsShaderArchiveModified()
        {
            return MaterialShaderAssign.ShaderDefinitionFileName != (savedData.shader_assign != null ? savedData.shader_assign.shader_archive : "");
        }

        public bool IsShaderNameModified()
        {
            return MaterialShaderAssign.ShaderName != (savedData.shader_assign != null ? savedData.shader_assign.shading_model : "");
        }

        public bool IsShaderRevisionModified()
        {
            return MaterialShaderAssign.Revision != (savedData.shader_assign != null ? savedData.shader_assign.revision : 0);
        }

        public bool IsMeshAdjacencyModified()
        {
            return Data.material_info.mesh_adjacency != savedData.material_info.mesh_adjacency;
        }

        static readonly char[] whiteSpaces = " \t\r\n".ToArray();
        private static bool Equals(shader_paramType lhs, shader_paramType rhs)
        {
            return lhs.type == rhs.type &&
                SplitValues(lhs.Value).SequenceEqual(SplitValues(rhs.Value)) &&
                lhs.depend == rhs.depend &&
                lhs.original_hint == rhs.original_hint;
        }

        private static IEnumerable<string> SplitValues(string values)
        {
            return values == null ? Enumerable.Empty<string>(): values.Split(whiteSpaces, StringSplitOptions.RemoveEmptyEntries);
        }

        public bool IsShaderParamsModified()
        {
            return MaterialShaderAssign.ShaderParams.Any(x => IsModified(x)) ||
                MaterialShaderAssign.ShaderParams.Count != savedData.shader_assign.ShaderParams().Count();
        }

        public bool IsModified(shader_paramType param)
        {
            var saved = savedData.shader_assign.ShaderParams().FirstOrDefault(x => x.id == param.id);
            return saved == null
                || !Equals(saved, param)
                || IsCustomLabelModified(param.id, savedCustomUI.uniform_vars, CustomUI.uniform_vars)
                || IsColorControlModified(param.id, savedCustomUI.colorControls, CustomUI.colorControls);
        }

        public bool IsCustomLabelModified(string id, Dictionary<string, string> lhs, Dictionary<string, string> rhs)
        {
            string lLabel;
            string rLabel;
            return lhs.TryGetValue(id, out lLabel) != rhs.TryGetValue(id, out rLabel) ||
                lLabel != rLabel;
        }

        public bool IsColorControlModified(string id, HashSet<string> lhs, HashSet<string> rhs)
        {
            return lhs.Contains(id) != rhs.Contains(id);
        }

        public bool IsShaderOptionsModified()
        {
            return MaterialShaderAssign.ShaderOptions.Any(x => IsModified(x)) ||
                MaterialShaderAssign.ShaderOptions.Count != savedData.shader_assign.ShaderOptions().Count();
        }

        public bool IsModified(shader_optionType param)
        {
            var saved = savedData.shader_assign.ShaderOptions().FirstOrDefault(x => x.id == param.id);
            return saved == null
                || saved.value != param.value
                || IsCustomLabelModified(param.id, savedCustomUI.option_vars, CustomUI.option_vars);
        }

        public bool IsSamplerAssignsModified()
        {
            return MaterialShaderAssign.SamplerAssigns.Any(x => IsModified(x)) ||
                savedData.shader_assign.SamplerAssigns().Any(x => !MaterialShaderAssign.SamplerAssigns.Any(y => x.id == y.id));
        }

        public bool IsModified(sampler_assignType param)
        {
            if (IsCustomLabelModified(param.id, savedCustomUI.sampler_vars, CustomUI.sampler_vars))
            {
                return true;
            }

            var saved = savedData.shader_assign.SamplerAssigns().FirstOrDefault(x => x.id == param.id);
            if (saved == null)
            {
                return !String.IsNullOrEmpty(param.sampler_name);
            }
            else
            {
                return saved.sampler_name != param.sampler_name;
            }
        }

        public bool IsAttribAssignModified()
        {
            return MaterialShaderAssign.AttribAssigns.Any(x => IsModified(x)) ||
                savedData.shader_assign.AttribAssigns().Any(x => !MaterialShaderAssign.AttribAssigns.Any(y => x.id == y.id));
        }

        public bool IsModified(attrib_assignType param)
        {
            if (IsCustomLabelModified(param.id, savedCustomUI.attrib_vars, CustomUI.attrib_vars))
            {
                return true;
            }

            var saved = savedData.shader_assign.AttribAssigns().FirstOrDefault(x => x.id == param.id);
            if (saved == null)
            {
                return !String.IsNullOrEmpty(param.attrib_name);
            }
            else
            {
                return saved.attrib_name != param.attrib_name;
            }
        }

        private static int GetChoiceIndex(string[] choices, IEnumerable<string> values)
        {
            int index = -1;
            if (values.Count() > 0)
            {
                // 数値で格納されている場合
                string value = values.First().Trim();
                if (!int.TryParse(value, out index))
                {
                    // 文字列で格納されている場合
                    index = Array.IndexOf(choices, value);
                }
            }

            return index;
        }

        private static bool Equals(render_infoType lhs, RenderInfo rhs, string[] choices)
        {
            if (lhs.type != rhs.type)
            {
                return false;
            }
            var values0 = SplitValues(lhs.Value);
            var values1 = rhs.values.Where(x => !String.IsNullOrEmpty(x));

            // チェックボックスの選択肢が文字列で存在するタイプ
            if (choices != null)
            {
                int index0 = GetChoiceIndex(choices, values0); // セーブデータの状態
                int index1 = GetChoiceIndex(choices, values1); // 現在のチェック状態
                if (index0 != -1 && index1 != -1)
                {
                    return index0 == index1;
                }
            }

            return values0.SequenceEqual(values1);
        }

        public bool IsRenderInfosModified()
        {
            return MaterialShaderAssign.RenderInfos.Any(x => IsModified(x)) ||
                MaterialShaderAssign.RenderInfos.Count != savedData.shader_assign.RenderInfos().Count();
        }

        public bool IsModified(RenderInfo param)
        {
            string[] choices = null;
            if (param.type == render_info_typeType.@string && param.values.Count != 0)
            {
                if (MaterialShaderAssign.ShadingModel != null &&
                    MaterialShaderAssign.ShadingModel.render_info_slot_array != null &&
                    MaterialShaderAssign.ShadingModel.render_info_slot_array.Items != null)
                {
                    render_info_slotType slot =
                        MaterialShaderAssign.ShadingModel.render_info_slot_array.Items.FirstOrDefault(x => x.name == param.name);
                    if (slot != null)
                    {
                        // 個別に分離して前後の空白は除去する
                        choices = slot.choice.Split(',');
                        for (int i = 0; i < choices.Length; ++i)
                        {
                            choices[i] = choices[i].Trim();
                        }
                    }
                }
            }

            var saved = savedData.shader_assign.RenderInfos().FirstOrDefault(x => x.name == param.name);
            return saved == null
                || !Equals(saved, param, choices)
                || IsCustomLabelModified(param.name, savedCustomUI.render_info_slots, CustomUI.render_info_slots);
        }

        public bool IsValueChanged<T>(Func<materialType, T> select) where T : struct
        {
            return !select(savedData).Equals(select(Data));
        }

        public bool IsSequenceValueChanged<T>(Func<materialType, IEnumerable<T>> select) where T : struct
        {
            return !select(savedData).SequenceEqual(select(Data));
        }

        public samplerType SavedSampler(string name)
        {
            return savedData.Samplers().FirstOrDefault(x => x.name == name);
        }

        public IEnumerable<samplerType> SavedSamplers()
        {
            return savedData.Samplers();
        }

        public bool IsValueChanged<T>(samplerType rhs, samplerType lhs, Func<samplerType, T> select)
        {
            if (rhs == null || lhs == null)
            {
                return false;
            }

            return !select(rhs).Equals(select(lhs));
        }

        public bool IsGroupCustomLabelModified(string group)
        {
            string lhs;
            string rhs;
            return savedCustomUI.groups.TryGetValue(group, out lhs) != CustomUI.groups.TryGetValue(group, out rhs) ||
                lhs != rhs;
        }

        public bool IsGroupCustomLabelModified()
        {
            var current = CreateGroupCustomItems();
            return current.Count != savedCustomUI.groups.Count ||
                current.Any(x =>
                {
                    string label;
                    return !savedCustomUI.groups.TryGetValue(x.id, out label) ||
                        label != x.label;
                });
        }

        public bool IsParentMaterialModified()
        {
            var count = MaterialReference?.ParentMaterials?.Count ?? 0;
            var savedCount = savedMaterialReference?.ParentMaterials?.Count ?? 0;
            if (count != savedCount)
            {
                return true;
            }

            for (var i = 0; i < count; i++)
            {
                var parentMaterial = MaterialReference.ParentMaterials[i];
                var savedParentMaterial = savedMaterialReference.ParentMaterials[i];
                if (parentMaterial.MaterialName != savedParentMaterial.MaterialName ||
                    parentMaterial.Path != savedParentMaterial.Path)
                {
                    return true;
                }
            }
            return false;
        }
        public bool IsReferenceBehaviorsModified()
        {
            if (IsChildSamplerStructureRestrictionsModified())
            {
                return true;
            }

            List<MaterialReferenceBehaviorItem>[] behaviorItemsList =
            {
                MaterialReference?.AttribAssignBehaviors,
                MaterialReference?.RenderInfoBehaviors,
                MaterialReference?.SamplerAssignBehaviors,
                MaterialReference?.ShaderOptionBehaviors,
                MaterialReference?.ShaderParamBehaviors
            };

            List<MaterialReferenceBehaviorItem>[] savedBehaviorItemsList =
            {
                savedMaterialReference?.AttribAssignBehaviors,
                savedMaterialReference?.RenderInfoBehaviors,
                savedMaterialReference?.SamplerAssignBehaviors,
                savedMaterialReference?.ShaderOptionBehaviors,
                savedMaterialReference?.ShaderParamBehaviors
            };

            Debug.Assert(behaviorItemsList.Length == savedBehaviorItemsList.Length);

            var nullcheck = CheckNullBothObjectOrOne(MaterialReference, savedMaterialReference);
            if (nullcheck != null)
            {
                return nullcheck.Value;
            }


            for (var i = 0; i < behaviorItemsList.Length; i++)
            {
                var behaviorItems = behaviorItemsList[i];
                var savedBehaviorItems = savedBehaviorItemsList[i];
                nullcheck = CheckNullBothObjectOrOne(behaviorItems, savedBehaviorItems);

                switch (nullcheck)
                {
                    case true:
                        continue;
                    case false:
                        return true;
                }

                if (behaviorItems.Count != savedBehaviorItems.Count)
                {
                    return true;
                }

                for (var j = 0; j < behaviorItems.Count; j++)
                {
                    var behaviorItem = behaviorItems[j];
                    var savedBehaviorItem = savedBehaviorItems[j];
                    if (behaviorItem.Id != savedBehaviorItem.Id || behaviorItem.Value != savedBehaviorItem.Value || behaviorItem.ChildRestriction != savedBehaviorItem.ChildRestriction)
                    {
                        return true;
                    }
                }
            }

            // サンプラ参照の比較
            {
                var samplerBehaviors = MaterialReference?.SamplerBehaviors;
                var savedSamplerBehaviors = savedMaterialReference?.SamplerBehaviors;
                nullcheck = CheckNullBothObjectOrOne(samplerBehaviors, savedSamplerBehaviors);
                if (nullcheck != null)
                {
                    return nullcheck.Value;
                }

                var samplerBehaviorItems = samplerBehaviors.SamplerBehaviorItems;
                var savedSamplerBehaviorItems = savedSamplerBehaviors.SamplerBehaviorItems;
                nullcheck = CheckNullBothObjectOrOne(samplerBehaviorItems, savedSamplerBehaviorItems);
                if (nullcheck != null)
                {
                    return nullcheck.Value;
                }

                if (samplerBehaviorItems.Count != savedSamplerBehaviorItems.Count)
                {
                    return true;
                }

                for (var i = 0; i < samplerBehaviorItems.Count; i++)
                {
                    var samplerBehaviorItem = samplerBehaviorItems[i];
                    var savedSamplerBehaviorItem = savedSamplerBehaviorItems[i];
                    nullcheck = CheckNullBothObjectOrOne(samplerBehaviorItem, savedSamplerBehaviorItem);

                    switch (nullcheck)
                    {
                        case true:
                            continue;
                        case false:
                            return true;
                    }

                    if (samplerBehaviorItem.Id != savedSamplerBehaviorItem.Id ||
                        samplerBehaviorItem.IsRequiredForChild != savedSamplerBehaviorItem.IsRequiredForChild)
                    {
                        return true;
                    }

                    var samplerParamBehaviors = samplerBehaviorItem.SamplerParamBehaviors;
                    var savedSamplerParamBehaviors = savedSamplerBehaviorItem.SamplerParamBehaviors;

                    nullcheck = CheckNullBothObjectOrOne(samplerParamBehaviors, savedSamplerParamBehaviors);

                    switch (nullcheck)
                    {
                        case true:
                            continue;
                        case false:
                            return true;
                    }

                    if (samplerParamBehaviors.Count != savedSamplerParamBehaviors.Count)
                    {
                        return true;
                    }

                    for (var j = 0; j < samplerParamBehaviors.Count; j++)
                    {
                        var behaviorItem = samplerParamBehaviors[j];
                        var savedBehaviorItem = savedSamplerParamBehaviors[j];
                        if (behaviorItem.Id != savedBehaviorItem.Id ||
                            behaviorItem.Value != savedBehaviorItem.Value ||
                            behaviorItem.ChildRestriction != savedBehaviorItem.ChildRestriction)
                        {
                            return true;
                        }
                    }
                }
            }

            return false;
        }

        public bool IsChildSamplerStructureRestrictionsModified()
        {
            return GetReferenceBehaviorSamplers()?.ChildSamplerStructureRestriction != savedMaterialReference?.SamplerBehaviors?.ChildSamplerStructureRestriction;
        }

        private static bool? CheckNullBothObjectOrOne(object object1, object object2)
        {
            if (object1 == null && object2 == null)
            {
                return false;
            }
            if (object1 == null || object2 == null)
            {
                return true;
            }
            return null;
        }
    }
}
