﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;

namespace App.Data
{
    /// <summary>
    /// ドキュメント構成変更イベントハンドラデリゲート。
    /// </summary>
    public delegate void DocumentComponentChangedEventHandler(object sender, DocumentComponentChangedEventArgs e);

    /// <summary>
    /// ドキュメント構成変更イベントデータクラス。
    /// </summary>
    public sealed class DocumentComponentChangedEventArgs : EventArgs
    {
        // 開いたドキュメントリスト
        private readonly List<Document> _openedDocuments = new List<Document>();
        // 閉じたドキュメントリスト
        private readonly List<Document> _closedDocuments = new List<Document>();

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public DocumentComponentChangedEventArgs() {}

        /// <summary>
        /// 変更があったか。
        /// </summary>
        public bool HasChanged()
        {
            return (HasOpenedDocument() || HasClosedDocument());
        }

        public IEnumerable<Document> ChangedDocuments
        {
            get
            {
                return _openedDocuments.Concat(_closedDocuments);
            }
        }

        #region OpenedDocuments
        /// <summary>
        /// 開いたドキュメントリスト。
        /// </summary>
        public ReadOnlyList<Document> OpenedDocuments
        {
            get { return new ReadOnlyList<Document>(_openedDocuments); }
        }

        /// <summary>
        /// 開いたドキュメントの追加。
        /// </summary>
        public void AddOpenedDocument(Document document)
        {
            _openedDocuments.Add(document);
        }

        /// <summary>
        /// 開いたドキュメントがあるか。
        /// </summary>
        public bool HasOpenedDocument()
        {
            return (_openedDocuments.Count > 0);
        }

        /// <summary>
        /// 開くイベントか。
        /// </summary>
        public bool IsOpenEvent()
        {
            return (HasOpenedDocument() && (!HasClosedDocument()));
        }
        #endregion

        #region ClosedDocuments
        /// <summary>
        /// 閉じたドキュメントリスト。
        /// </summary>
        public ReadOnlyList<Document> ClosedDocuments
        {
            get { return new ReadOnlyList<Document>(_closedDocuments); }
        }

        /// <summary>
        /// 閉じたドキュメントの追加。
        /// </summary>
        public void AddClosedDocument(Document document)
        {
            _closedDocuments.Add(document);
        }

        /// <summary>
        /// 閉じたドキュメントがあるか。
        /// </summary>
        public bool HasClosedDocument()
        {
            return (_closedDocuments.Count > 0);
        }

        /// <summary>
        /// 閉じるイベントか。
        /// </summary>
        public bool IsCloseEvent()
        {
            return (HasClosedDocument() && (!HasOpenedDocument()));
        }
        #endregion
    }
}
