﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Diagnostics;
using System.Linq;
using App.Command;
using App.ConfigData;
using App.PropertyEdit;
using App.Utility;
using ConfigCommon;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;

namespace App.Data
{
    public partial class ShaderParameterAnimation : AnimationDocument, IHasUserData, IHasShaderParameterAnimation
    {
        public shader_param_animType Data{ get; private set; }
        public override object nw4f_3difItem
        {
            get { return Data; }
        }

        public override process_log_arrayType process_log_array
        {
            get { return Data.process_log_array; }
        }

        public override int FrameCount
        {
            get{ return Data.shader_param_anim_info.frame_count; }
            set{ Data.shader_param_anim_info.frame_count = value; }
        }

        public override bool Loop
        {
            get { return Data.shader_param_anim_info.loop; }
            set { Data.shader_param_anim_info.loop = value; }
        }

        public override bool IsFrameCountModified
        {
            get { return savedData.frame_count != Data.shader_param_anim_info.frame_count; }
        }

        public override bool IsLoopModified
        {
            get { return savedData.loop != Data.shader_param_anim_info.loop; }
        }


        // ファイル出力できるか？
        public override bool IsFileOutputable
        {
            get
            {
                return FileNotOutputErrorMessage == null;
            }
        }

        public override string FileNotOutputErrorMessage
        {
            get
            {
                string error = null;
                Action<ShaderParamAnim, ParamAnim, ParamAnimTarget> errorHandler =
                    delegate(ShaderParamAnim shaderParamAnim, ParamAnim paramAnim, ParamAnimTarget paramAnimTarget)
                {
                    error = string.Format(res.Strings.FileNotOutputError_ShaderParamAnimationDocument,
                                        shaderParamAnim.mat_name,
                                        paramAnim.id);
                };
                foreach (var item in QuantizeAnalyse(errorHandler))
                {
                    if (error != null)
                    {
                        return error;
                    }
                }

                return null;
            }
        }

        public override bool HasAnimation
        {
            get
            {
                return ShaderParamAnims != null && ShaderParamAnims
                    .Where(x => x.ParamAnims != null)
                    .Any(x => x.ParamAnims
                        .Where(p => p.ParamAnimTargets != null)
                        .Any(p => p.ParamAnimTargets.Any()));
            }
        }

        public override IEnumerable<Tuple<AnimTarget, IfQuantizationAnalysisResult>> QuantizeAnalyse()
        {
            return QuantizeAnalyse(null);
        }

        private IEnumerable<Tuple<AnimTarget, IfQuantizationAnalysisResult>> QuantizeAnalyse(Action<ShaderParamAnim, ParamAnim, ParamAnimTarget> errorHandler)
        {
            foreach (var shaderParamAnim in ShaderParamAnims)
            {
                foreach (var paramAnim in shaderParamAnim.ParamAnims)
                {
                    foreach (var paramAnimTarget in paramAnim.ParamAnimTargets)
                    {
                        IfAnimCurve.CurveType curveType;
                        {
                            switch (ShaderTypeUtility.ParamPrimitiveTypeFromType(paramAnim.type))
                            {
                                case ShaderTypeUtility.ParamPrimitiveTypeKind.@bool:
                                    curveType = IfAnimCurve.CurveType.Bool;
                                    break;
                                case ShaderTypeUtility.ParamPrimitiveTypeKind.@int:
                                case ShaderTypeUtility.ParamPrimitiveTypeKind.@uint:
                                    curveType = IfAnimCurve.CurveType.Int;
                                    break;
                                case ShaderTypeUtility.ParamPrimitiveTypeKind.@float:
                                    curveType = IfAnimCurve.CurveType.Float;
                                    break;
                                default:
                                    //throw new NotImplementedException();
                                    curveType = IfAnimCurve.CurveType.Float;
                                    Debug.Assert(false);
                                    break;
                            }
                        }

                        var ifAnimCurve = new IfAnimCurve()
                        {
                            curveType = curveType,
                        };

                        G3dStream stream;
                        var anim_target = paramAnimTarget.ConvertTo_param_anim_targetType(out stream);
                        if (anim_target.Curve != null)
                        {
                            var isRotate = false;
                            switch (paramAnim.type)
                            {
                                case shader_param_typeType.srt2d:
                                    isRotate = paramAnimTarget.component_index == 2;
                                    break;
                                case shader_param_typeType.srt3d:
                                    isRotate = 3 <= paramAnimTarget.component_index && paramAnimTarget.component_index <= 5;
                                    break;
                                case shader_param_typeType.texsrt:
                                    isRotate = paramAnimTarget.component_index == 3;
                                    break;
                            }
                            ifAnimCurve.BuildIfAnimCurve(anim_target.Curve, isRotate, stream);

                            IfQuantizationAnalysisResult result = null;
                            try
                            {
                                result = IfShaderParamAnimQuantizationAnalysis.Analyse(
                                    Data.shader_param_anim_info,
                                    paramAnim.type,
                                    paramAnimTarget.component_index,
                                    ifAnimCurve,
                                    true
                                );
                            }
                            catch (Exception)
                            {
                                if (errorHandler != null)
                                {
                                    errorHandler(shaderParamAnim, paramAnim, paramAnimTarget);
                                }
                            }
                            yield return new Tuple<AnimTarget, IfQuantizationAnalysisResult>(paramAnimTarget, result);
                        }
                    }
                }
            }
        }

        public override bool IsEditable
        {
            get
            {
                foreach (var shaderParamMatAnim in ShaderParamAnims)
                {
                    foreach (var tuple in GetParamAnimState(shaderParamMatAnim))
                    {
                        ShaderParameterAnimation.ParamAnim paramAnim = tuple.Item1;
                        AnimationDocument.NonEditableKind nonEditableKind = tuple.Item3;
                        switch (nonEditableKind)
                        {
                            case NonEditableKind.ShaderParamAnim_Invisible:
                                break;
                            case AnimationDocument.NonEditableKind.ShaderParamAnim_InvalidType:
                                if (paramAnim.ParamAnimTargets.All(x => x.ExportType == CurveExportType.Ignored || x.ExportType == CurveExportType.Dependent))
                                {
                                    continue;
                                }
                                return false;
                            case NonEditableKind.Editable:
                                break;
                            case NonEditableKind.ShaderParamAnim_DisableByOptionVar:
                                break;
                            default:
                                return false;
                        }
                    }
                }
                return true;
            }
        }

        public Material[] GetBoundMaterials(string materialName)
        {
            var models = from model in DocumentManager.Models
                         where model.AllAnimations.Contains(this)
                         select model;

            return (from model in models
                         from material in model.Materials
                         where material.Name == materialName
                         select material).ToArray();
        }
        private void GetBoundObjects(
            ShaderParamAnim shaderParamMatAnim,
            out Material[] materials,
            out Dictionary<string, Tuple<uniform_varType, Material>[]> uniformVars,
            out Dictionary<Material, Func<string, string> > visibleInfos)
        {
            materials = GetBoundMaterials(shaderParamMatAnim.mat_name);

            // ユニフォーム
            uniformVars = new Dictionary<string, Tuple<uniform_varType, Material>[]>();
            {
                var groups = from material in materials
                             let shading_model = material.MaterialShaderAssign.ShadingModel
                             where shading_model != null
                             from uniform in shading_model.MaterialUniforms()
                             group new Tuple<uniform_varType, Material>(uniform, material) by uniform.id;

                foreach (var group in groups)
                {
                    uniformVars.Add(group.Key, group.ToArray());
                }
            }

            // VisibleInfo をつくる
            visibleInfos = new Dictionary<Material, Func<string, string>>();
            foreach (var material in materials)
            {
                var groupTypes = material.MaterialShaderAssign.ShadingModel != null ?
                    material.MaterialShaderAssign.ShadingModel.Groups().ToDictionary(x => x.name) :
                    new Dictionary<string, groupType>();
                Func<string, string> toParent = x => {
                    groupType g;
                    if (groupTypes.TryGetValue(x, out g))
                    {
                        return g.Group();
                    }

                    return string.Empty;
                };
                var visibleInfo = new VisibleInfo(material);
                var nameToGroup = material.MaterialShaderAssign.Definition != null ?
                    material.MaterialShaderAssign.Definition.Data.Groups().ToDictionary(x => x.name) :
                    new Dictionary<string, groupType>();

                visibleInfos[material] = TreeUtility.PreOrderDP<string, string>(
                    (Func<string,string>)(x => visibleInfo.IsVisible(x) ? null:x),
                    (Func<string, IEnumerable<string>>)(x => {
                        groupType group;
                        if (nameToGroup.TryGetValue(x, out group) && !string.IsNullOrEmpty(group.Group()))
                        {
                            return Enumerable.Repeat(group.Group(), 1);
                        }
                        return Enumerable.Empty<string>();
                    }),
                    (Func<IEnumerable<string>, string>)(x =>x.FirstOrDefault(y => y != null)));
            }
        }

        public IEnumerable<Tuple<ParamAnim, uniform_varType, NonEditableKind, string>> GetParamAnimState(ShaderParamAnim shaderParamMatAnim)
        {
            // バインドされているマテリアル
            Material[] materials;

            // ユニフォーム
            Dictionary<string, Tuple<uniform_varType, Material>[]> uniformVars;

            // VisibleInfo
            Dictionary<Material, Func<string, string>> visibleInfos;

            GetBoundObjects(shaderParamMatAnim, out materials, out uniformVars, out visibleInfos);

            foreach (var paramAnim in shaderParamMatAnim.ParamAnims)
            {
                Tuple<uniform_varType, Material>[] pairs;
                uniformVars.TryGetValue(paramAnim.id, out pairs);

                // uniform_var を生成
                uniform_varType uniform_var = null;
                Material material = null;

                if (pairs != null && pairs.Any())
                {
                    var pair = pairs.FirstOrDefault(x => x.Item1.type == paramAnim.type);
                    if (pair != null)
                    {
                        uniform_var = pair.Item1;
                        material = pair.Item2;
                    }
                }

                AnimationDocument.NonEditableKind nonEditableKind = AnimationDocument.NonEditableKind.Editable;
                string notEditableGroup = null;

                if (uniform_var == null)
                {
                    // ユニフォームがない場合
                    nonEditableKind = MakeNonEditableKind(paramAnim, materials);
                }
                else
                {
                    // 対象外
                    if (!IsEnableParameter(uniform_var))
                    {
                        // 対象のファイルでない
                        nonEditableKind = NonEditableKind.ShaderParamAnim_InvalidType;
                    }
                    else if (!uniform_var.IsAnimVisible())
                    {
                        // 見えない
                        nonEditableKind = AnimationDocument.NonEditableKind.ShaderParamAnim_Invisible;
                    }
                    else
                    {
                        notEditableGroup = visibleInfos[material](uniform_var.Group());

                        if (notEditableGroup != null)
                        {
                            // オプションが無効
                            nonEditableKind = AnimationDocument.NonEditableKind.ShaderParamAnim_DisableByOptionVar;
                        }
                    }
                }

                yield return new Tuple<ParamAnim, uniform_varType, NonEditableKind, string>(
                    paramAnim, uniform_var, nonEditableKind, notEditableGroup);
            }
        }

        private NonEditableKind MakeNonEditableKind(ParamAnim paramAnim, Material[] allMaterials)
        {
            // モデルにバインドされていない
            if (DocumentManager.Models.SelectMany(x => x.AllAnimations).Any(x => x == this) == false)
            {
                return NonEditableKind.ShaderParamAnim_NotBindModel;
            }

            // モデルにマテリアルが見つからない
            if (allMaterials.Length == 0)
            {
                return NonEditableKind.ShaderParamAnim_NotFoundMaterial;
            }

            // マテリアルにシェーダーが設定されていない
            if (allMaterials.All(x => x.MaterialShaderAssign.ShaderDefinitionFileName == string.Empty))
            {
                return NonEditableKind.ShaderParamAnim_NotAssignShaderDef;
            }

            // シェーダー定義が読み込まれていない
            if (allMaterials.All(x => !DocumentManager.ShaderDefinitions.Any(y => y.Name == x.MaterialShaderAssign.ShaderDefinitionFileName)))
            {
                return NonEditableKind.ShaderParamAnim_NotLoadShaderDef;
            }

            // シェーディングモデルが見つからない
            var shadingModels = allMaterials.Select(x => x.MaterialShaderAssign.ShadingModel).Where(x => x != null).Distinct().ToArray();
            if (shadingModels.Length == 0)
            {
                return NonEditableKind.ShaderParamAnim_NotShadingModel;
            }

            // ユニフォーム変数が見つからない
            uniform_varType[] uniformVars = shadingModels.SelectMany(x => x.MaterialUniforms()).Where(x => x.id == paramAnim.id).ToArray();
            if (uniformVars.Length == 0)
            {
                return NonEditableKind.ShaderParamAnim_NotFoundParam;
            }

            // ユニフォーム変数の型と違う
            if (uniformVars.All(x => x.type != paramAnim.type))
            {
                return NonEditableKind.ShaderParamAnim_TypeConflict;
            }

            Debug.Assert(false);
            return NonEditableKind.Editable;
        }

        // マテリアルに関係するか
        public override bool IsMaterialRelated { get { return true; } }

        public UserDataArray UserDataArray { get; set; }
        public UserDataArray SavedUserDataArray { get; set; }
        public bool UserDataArrayChanged { get; set; }

        private IEnumerable<Material> ParentModelMaterialNames
        {
            get
            {
                return
                    from model in DocumentManager.Models
                    from anim in model.AllAnimations
                    where anim == this
                    from material in model.Materials
                    select material;
            }
        }

        [Serializable]
        public class ShaderParamAnim : IShaderParamMatAnim
        {
            public ShaderParamAnim()
            {
                mat_name	= string.Empty;
                ParamAnims = new ParamAnimList();
            }

            public string			mat_name{	get; set; }			// shader_param_mat_anim.mat_name
            public ParamAnimList ParamAnims { get; set; }
        }

        /// <summary>
        /// 検索を高速化したリスト
        /// </summary>
        [Serializable]
        public class ParamAnimList : ObservableCollection<ParamAnim>
        {

            protected override void OnCollectionChanged(System.Collections.Specialized.NotifyCollectionChangedEventArgs e)
            {
                if (dict != null)
                {
                    dict = null;
                }
            }

            public void AddRange(IEnumerable<ParamAnim> anims)
            {
                foreach (var item in anims)
                {
                    Add(item);
                }
            }

            public int FindIndex(Predicate<ParamAnim> match)
            {
                for (int i = 0; i < Count; i++)
                {
                    if (match(this[i]))
                    {
                        return i;
                    }
                }

                return -1;
            }

            public void RemoveAll(Predicate<ParamAnim> match)
            {
                var array = this.Where(x => !match(x)).ToArray();
                Clear();
                AddRange(array);
            }

            protected override void OnPropertyChanged(PropertyChangedEventArgs e)
            {
            }
            Dictionary<string, ParamAnim> dict;
            public ParamAnim this[string id]
            {
                get
                {
                    if (dict == null)
                    {
                        dict = this.ToDictionary(x => x.id);
                    }
                    ParamAnim value;
                    dict.TryGetValue(id, out value);
                    return value;
                }
            }
        }

        [Serializable]
        public class ParamAnim
        {
            public ParamAnim()
            {
                id					= string.Empty;
                ParamAnimTargets	= new List<ParamAnimTarget>();
                type				= shader_param_typeType.@bool;
                originalHint		= string.Empty;
            }

            public ParamAnim(string id, shader_param_typeType type)
            {
                this.id = id;
                this.type = type;
                var info = IfUtility.GetShaderParamTypeInfo(type);
                ParamAnimTargets = Enumerable.Range(0, info.PrimitiveType.Length).Select(
                    i => new ParamAnimTarget()
                    {
                        component_index = i,
                        CurveInterpolationType = info.PrimitiveType[i] == IfUtility.ShaderParamPrimitiveType.@float ?
                            InterpolationType.Hermite :
                            InterpolationType.Step,
                        IsTexSrt = IsTexSRT,
                    }).ToList();
                originalHint = string.Empty;
            }

            public string					id{					get; set; }			// param_anim.id
            public shader_param_typeType	type {				get; set; }
            public string					originalHint {		get; set; }

            public List<ParamAnimTarget>	ParamAnimTargets{	get; set; }

            [NonSerialized]
            private bool _IsModified;

            public bool IsModified {
                get
                {
                    return _IsModified;
                }
                set
                {
                    _IsModified = value;
                }
            }

            public bool IsTexSRT
            {
                get
                {
                    if ((type == shader_param_typeType.texsrt))
                    {
                        return true;
                    }
                    return false;
                }
            }
        }

        [Serializable]
        public class ParamAnimTarget : AnimTarget
        {
            public int component_index { get; set; }

            public bool IsTexSrt{get; set;}

            override public CurveExportType ExportType
            {
                get
                {
                    if (KeyFrames.Any())
                    {
                        float base_value = 0;
                        if (!KeyFrames.IsConstantCurve(CurveInterpolationType, out base_value))
                        {
                            if (IsTexSrt && component_index == 0)
                            {
                                // カーブだけど、TexSRTの場合は、
                                // 他のTargetにCurveが存在する場合にのみ出力する為
                                return CurveExportType.Dependent;
                            }
                            return CurveExportType.Curve;
                        }

                        if (IsTexSrt && component_index == 0)
                        {
                            // カーブだけど、TexSRTの場合は、
                            // 他のTargetにCurveが存在する場合にのみ出力する為
                            return CurveExportType.Dependent;
                        }
                        return CurveExportType.Constant;
                    }

                    return CurveExportType.Ignored;
                }
            }

            public param_anim_targetType ConvertTo_param_anim_targetType(out G3dStream stream, bool viewer = false)
            {
                IG3dQuantizedCurve curve;
                Convert(out curve, out stream, viewer);
                var target = new param_anim_targetType()
                {
                    base_value = GetBaseValue(),
                    Curve = curve,
                    component_index = component_index,
                };
                return target;
            }
        }

        public ShaderParamAnimList ShaderParamAnims;

        public IEnumerable<IShaderParamMatAnim> IShaderParamMatAnims { get { return ShaderParamAnims; } }

        /// <summary>
        /// 検索を高速化したリスト
        /// </summary>
        [Serializable]
        public class ShaderParamAnimList : ObservableCollection<ShaderParamAnim>
        {
            protected override void OnCollectionChanged(System.Collections.Specialized.NotifyCollectionChangedEventArgs e)
            {
                if (dict != null)
                {
                    dict = null;
                }
            }

            public void AddRange(IEnumerable<ShaderParamAnim> anims)
            {
                foreach (var item in anims)
                {
                    Add(item);
                }
            }

            protected override void OnPropertyChanged(PropertyChangedEventArgs e)
            {
            }

            Dictionary<string, ShaderParamAnim> dict;
            public ShaderParamAnim this[string mat_name]
            {
                get
                {
                    if (dict == null)
                    {
                        dict = this.ToDictionary(x => x.mat_name);
                    }
                    ShaderParamAnim value;
                    dict.TryGetValue(mat_name, out value);
                    return value;
                }
            }
        }

        #region 圧縮率
        // シェーダー定義が読み込まれていないと正しいSRTの判断ができないため、実際に数えてみる
        public bool IsAllFrameCurveConstant            { get { return ShaderParamAnims.SelectMany(x => x.ParamAnims).SelectMany(y => y.ParamAnimTargets).Any(x => x.KeyFrames.Any()) == false; } }
        public bool IsAllTextureScaleCurveConstant     { get { return GetTextureScaleDataSize(true)     == 0; } }
        public bool IsAllTextureRotateCurveConstant    { get { return GetTextureRotateDataSize(true)    == 0; } }
        public bool IsAllTextureTranslateCurveConstant { get { return GetTextureTranslateDataSize(true) == 0; } }

        public int FrameCompressDataSize                  { get { return GetFrameDataSize(false);            } }
        public int FrameUncompressDataSize                { get { return GetFrameDataSize(true);             } }

        public int TextureScaleCompressDataSize       { get { return GetTextureScaleDataSize(false);     } }
        public int TextureScaleUncompressDataSize     { get { return GetTextureScaleDataSize(true);      } }

        public int TextureRotateCompressDataSize      { get { return GetTextureRotateDataSize(false);    } }
        public int TextureRotateUncompressDataSize    { get { return GetTextureRotateDataSize(true);     } }

        public int TextureTranslateCompressDataSize   { get { return GetTextureTranslateDataSize(false); } }
        public int TextureTranslateUncompressDataSize { get { return GetTextureTranslateDataSize(true);  } }

        #endregion

        // 編集コマンド用
        [Serializable]
        public class QuantizeToleranceData
        {
            public float	QuantizeToleranceTextureScale		{ get; set; }
            public float	QuantizeToleranceTextureRotate		{ get; set; }
            public float	QuantizeToleranceTextureTranslate	{ get; set; }
        }

        // カラーアニメーション、テクスチャSRTアニメーション用
        public ShaderParameterAnimation(shader_param_animType animation, List<G3dStream> streams, GuiObjectID guiObjectID) : base(guiObjectID, streams)
        {
            Data = animation;
            Initialize();
        }

        public ShaderParameterAnimation(shader_param_animType animation, List<G3dStream> streams)
            : base(GuiObjectID.ShaderParameterAnimation, streams)
        {
            Data = animation;
            Initialize();
        }

        public void Initialize()
        {
            ToolData.Load(Data.tool_data);
            MakeComment(Data.comment);

            MakeShaderParamAnims();
            // ユーザーデータ作成
            this.MakeUserData(Data.user_data_array, BinaryStreams);
        }

        #region Bridge
        public class ShaderParamSwapData : SwapData
        {
            public ToolData ToolData;
            public List<G3dStream> streams;
            public string Comment;
            public RgbaColor? EditColor;
            public shader_param_animType Data;
            public ShaderParamAnimList ShaderParamAnims;
            public UserDataArray userData;
        }

        public override SwapData CreateSwapData(object data, List<G3dStream> streams)
        {
            var current = GetSwapData();
            Data = (shader_param_animType)data;
            BinaryStreams = streams;
            Initialize();

            return Swap(current, false);
        }

        public override SwapData GetSwapData()
        {
            return new ShaderParamSwapData()
            {
                Data = Data,
                streams = BinaryStreams,
                Comment = Comment,
                EditColor = EditColor,
                ShaderParamAnims = ShaderParamAnims,
                userData = UserDataArray,
            };
        }

        public override SwapData Swap(SwapData swapData, bool copyHistory)
        {
            var current = (ShaderParamSwapData)GetSwapData();
            var data = (ShaderParamSwapData)swapData;
            Data = data.Data;
            BinaryStreams = data.streams;
            Comment = data.Comment;
            EditColor = data.EditColor;
            ShaderParamAnims = data.ShaderParamAnims;
            UserDataArray = data.userData;

            if (copyHistory)
            {
                UserDataArrayChanged = !SavedUserDataArray.IsSame(UserDataArray);
                UpdateIsModifiedAnimTargetAll();
            }

            return current;
        }
        #endregion

        public override nw4f_3difType Create_nw4f_3difType(bool viewer)
        {
            // lock は不要だと思う。
            lock (this)
            {
                UpdateData(viewer);

                //			nw4f_3difType nw4f_3dif = new nw4f_3difType();
                nw4f_3dif_.Item = Data;
                nw4f_3dif_.file_info = null;

                return nw4f_3dif_;
            }
        }

        public void Optimize()
        {
            // ストリームを更新する。
            UpdateData();

            // <hermite_curve>、<linear_curve>、<step_curve> の frame_type と key_type を
            // none に設定しないと、再分析が行われません。
            if (Data.shader_param_mat_anim_array != null)
            {
                foreach (var anim in Data.shader_param_mat_anim_array.shader_param_mat_anim)
                {
                    if (anim.param_anim_array != null)
                    {
                        foreach (var paramAnim in anim.param_anim_array.param_anim)
                        {
                            foreach (var target in paramAnim.param_anim_target)
                            {
                                if (target.Curve != null)
                                {
                                    target.Curve.frame_type = curve_frame_typeType.none;
                                    target.Curve.key_type = curve_key_typeType.none;
                                }
                            }
                        }
                    }
                }
            }

            try
            {
                // アニメーションに合わせたAnimQuantizationAnalysisOptimizerを呼び出す。
                var optimizer = new IfShaderParamAnimQuantizationAnalysisOptimizer();
                if (ApplicationConfig.DefaultValue.DisableAnimationQuantize)
                {
                    var animinfo = Data.shader_param_anim_info;
                    var tolS = animinfo.quantize_tolerance_tex_scale;
                    var tolR = animinfo.quantize_tolerance_tex_rotate;
                    var tolT = animinfo.quantize_tolerance_tex_translate;
                    animinfo.quantize_tolerance_tex_scale = 0.0f;
                    animinfo.quantize_tolerance_tex_rotate = 0.0f;
                    animinfo.quantize_tolerance_tex_translate = 0.0f;

                    optimizer.Optimize(Data, BinaryStreams);

                    animinfo.quantize_tolerance_tex_scale = tolS;
                    animinfo.quantize_tolerance_tex_rotate = tolR;
                    animinfo.quantize_tolerance_tex_translate = tolT;
                }
                else
                {
                    optimizer.Optimize(Data, BinaryStreams);
                }
            }
            catch(Exception e)
            {
                DebugConsole.WriteLine("Exception Catch : IfShaderParamAnimQuantizationAnalysisOptimizer.Optimize() : {0}", e.Message);
            }

            // 最適化後に、データを再作成する。
            MakeShaderParamAnims();
            this.MakeUserData(Data.user_data_array, BinaryStreams);
        }

        private int GetFrameDataSize(bool isBase)
        {
            int size = 0;
            {
                foreach (var shaderParamAnim in ShaderParamAnims)
                {
                    foreach (var paramAnim in shaderParamAnim.ParamAnims)
                    {
                        foreach (var target in paramAnim.ParamAnimTargets)
                        {
                            size += target.KeyFrames.Count * (isBase ? 4 : IfUtility.GetByteSizeFromCurveFrameType(target.QuantizationInfo.frame_type));
                        }
                    }
                }
            }
            return size;
        }

        private int GetTextureScaleDataSize(bool isBase)
        {
            return GetTextureDataSize(isBase)[0];
        }

        private int GetTextureRotateDataSize(bool isBase)
        {
            return GetTextureDataSize(isBase)[1];
        }

        private int GetTextureTranslateDataSize(bool isBase)
        {
            return GetTextureDataSize(isBase)[2];
        }

        // 0:Scale
        // 1:Rotate
        // 2:Translate
        private int[] GetTextureDataSize(bool isBase)
        {
            var size = new int[]{0, 0, 0};
            {
                foreach (var param in ShaderParamAnims)
                {
                    foreach (var paramAnim in param.ParamAnims)
                    {
                        if (IsEnableParameter_TextureSrt(paramAnim.type))
                        {
                            foreach (var target in paramAnim.ParamAnimTargets)
                            {
                                int kind = 0;
                                {
                                    if (paramAnim.type == shader_param_typeType.srt3d)
                                    {
                                        kind = target.component_index / 3;
                                    }
                                    else
                                    {
                                        switch (target.component_index + (paramAnim.type == shader_param_typeType.srt2d? 1: 0))
                                        {
                                            case 0:
                                                continue;
                                            case 1: case 2:
                                                kind = 0;
                                                break;
                                            case 3:
                                                kind = 1;
                                                break;
                                            case 4: case 5:
                                                kind = 2;
                                                break;
                                            default:
                                                Debug.Assert(false);
                                                break;
                                        }
                                    }
                                }

                                size[kind] += target.KeyFrames.Count * (isBase ? 4 : IfUtility.GetByteSizeFromCurveKeyType(target.QuantizationInfo.key_type));
                            }
                        }
                    }
                }
            }
            return size;
        }

        public ParamAnim GetParamAnimFromId(string materialName, string id)
        {

            var shaderParamAnim = ShaderParamAnims[materialName];

            if (shaderParamAnim != null)
            {
                return shaderParamAnim.ParamAnims[id];
            }
            else
            {
                return null;
            }
        }

        // 見つからない場合は-1
        public bool GetParamAnimIndexFromId(string materialName, string id, out int materialIndex, out int paramAnimlIndex)
        {
            materialIndex	= -1;
            paramAnimlIndex	= -1;

            var shaderParamAnim = ShaderParamAnims[materialName];

            if (shaderParamAnim != null)
            {
                materialIndex = ShaderParamAnims.IndexOf(shaderParamAnim);

                var paramAnim = shaderParamAnim.ParamAnims[id];
                if (paramAnim != null)
                {
                    paramAnimlIndex = shaderParamAnim.ParamAnims.IndexOf(paramAnim);
                    return true;
                }
            }

            return false;
        }

        // 有効なパラメーターか？
        public virtual bool IsEnableParameter(uniform_varType param)
        {
            return true;
        }

        protected bool IsEnableParameter_TextureSrt(shader_param_typeType type)
        {
            return
                (type == shader_param_typeType.srt2d) ||
                (type == shader_param_typeType.srt3d) ||
                (type == shader_param_typeType.texsrt);
        }

        protected bool IsEnableParameter_Color(uniform_varType param)
        {
            return (param.ui_item != null) && (param.ui_item.value == ui_item_valueType.color);
        }

        private readonly List<string> textureSRTModeCurveDeleteParamAnimNameList_ = new List<string>();
        public void ShowTextureSRTModeCurveDeleteMessage()
        {
            if (textureSRTModeCurveDeleteParamAnimNameList_.Any())
            {
                string msg = string.Format("File: {0}\n", FileName);
                foreach (var str in textureSRTModeCurveDeleteParamAnimNameList_)
                {
                    msg += string.Format("<param_anim>: {0}\n", str);
                }
                msg += res.Strings.IO_Load_TexSRT_ModeCurveDelete;
                App.Controls.UIMessageBox.Information(msg);
            }
        }

        private void MakeShaderParamAnims()
        {
            textureSRTModeCurveDeleteParamAnimNameList_.Clear();

            ShaderParamAnims = new ShaderParamAnimList();

            if ((Data.shader_param_mat_anim_array == null) ||
                (Data.shader_param_mat_anim_array.shader_param_mat_anim == null))
            {
                return;
            }

            foreach (var shaderParamAnim in Data.shader_param_mat_anim_array.shader_param_mat_anim)
            {

                var newShaderParamAnim = new ShaderParamAnim()
                {
                    mat_name = shaderParamAnim.mat_name
                };

                foreach (var paramAnim in shaderParamAnim.param_anim_array.param_anim)
                {
                    var newParamAnim = CreateParamAnim(paramAnim, BinaryStreams);
                    newShaderParamAnim.ParamAnims.Add(newParamAnim);
                }

                ShaderParamAnims.Add(newShaderParamAnim);
            }
        }

        public ParamAnim CreateParamAnim(param_animType paramAnim, List<G3dStream> streams)
        {
            ParamAnim newParamAnim = new ParamAnim()
            {
                id = paramAnim.id,
                type = paramAnim.type,
                originalHint = paramAnim.original_hint
            };

            // TextureSRTの場合、ターゲットのプロパティに値を設定する。
            bool isTexSRT = newParamAnim.IsTexSRT;

            var targetShaderParamTypeInfo = IfUtility.GetShaderParamTypeInfo(paramAnim.type);
            newParamAnim.ParamAnimTargets = Enumerable.Repeat<ParamAnimTarget>(null, targetShaderParamTypeInfo.PrimitiveType.Length).ToList();
            var paramAnimTargets = from param_anim_target in paramAnim.param_anim_target
                                   let primitiveType = targetShaderParamTypeInfo.PrimitiveType[param_anim_target.component_index]
                                   let defaultInterpolationType = primitiveType == IfUtility.ShaderParamPrimitiveType.@float? InterpolationType.Hermite: InterpolationType.Step
                                   select CreateParamAnimTarget(param_anim_target, BinaryStreams, isTexSRT, defaultInterpolationType);
            foreach (var ParamAnimTarget in paramAnimTargets)
            {
                newParamAnim.ParamAnimTargets[ParamAnimTarget.component_index] = ParamAnimTarget;
            }

            for (int i = 0; i < targetShaderParamTypeInfo.PrimitiveType.Length; i++)
            {
                if (newParamAnim.ParamAnimTargets[i] == null)
                {
                    newParamAnim.ParamAnimTargets[i] = new ShaderParameterAnimation.ParamAnimTarget()
                    {
                        component_index = i,
                        CurveInterpolationType = targetShaderParamTypeInfo.PrimitiveType[i] == IfUtility.ShaderParamPrimitiveType.@float ?
                            InterpolationType.Hermite :
                            InterpolationType.Step,
                        IsTexSrt = isTexSRT,	// TextureSRTかどうか。
                    };

                    if (i == 0 && isTexSRT)
                    {
                        newParamAnim.ParamAnimTargets[i].KeyFrames.Add(
                            new KeyFrame()
                            {
                                Frame = 0.0f,
                                Value = 0
                            }
                        );
                    }
                }
                else
                {
                    if (i == 0 && isTexSRT)
                    {
                        var keyFrames = newParamAnim.ParamAnimTargets[i].KeyFrames;
                        if (keyFrames.Count != 1)
                        {
                            keyFrames.Clear();
                            var param_anim_target = paramAnim.param_anim_target.FirstOrDefault(x => x.component_index == 0);
                            Debug.Assert(param_anim_target != null);
                            keyFrames.Add(new KeyFrame()
                            {
                                Frame = 0,
                                Value = param_anim_target.base_value,
                            });

                            textureSRTModeCurveDeleteParamAnimNameList_.Add(newParamAnim.id);
                        }
                    }
                }
            }

            return newParamAnim;
        }

        public ParamAnimTarget CreateParamAnimTarget(param_anim_targetType paramAnimTarget, List<G3dStream> streams, bool isTexSrt, InterpolationType defaultInterpolationType)
        {
            ParamAnimTarget newParamAnimTarget = new ParamAnimTarget()
            {
                component_index = paramAnimTarget.component_index,
                CurveInterpolationType = (paramAnimTarget.Item is hermite_curveType) ? InterpolationType.Hermite :
                                                              (paramAnimTarget.Item is linear_curveType) ? InterpolationType.Linear :
                                                              (paramAnimTarget.Item is step_curveType) ? InterpolationType.Step :
                                                                                                            defaultInterpolationType,		// null の時
                IsTexSrt = isTexSrt,
            };

            if (paramAnimTarget.Item != null)
            {
                switch (newParamAnimTarget.CurveInterpolationType)
                {
                    case InterpolationType.Hermite:
                        {
                            const int dataElementCount = 4;		// frame, value, in-slope, out-slope

                            var curve = paramAnimTarget.Item as hermite_curveType;
                            Debug.Assert(curve != null);

                            newParamAnimTarget.QuantizationInfo.frame_type = curve.frame_type;
                            newParamAnimTarget.QuantizationInfo.key_type = curve.key_type;
                            newParamAnimTarget.QuantizationInfo.scale = curve.scale;
                            newParamAnimTarget.QuantizationInfo.offset = curve.offset;
                            newParamAnimTarget.pre_wrap = curve.pre_wrap;
                            newParamAnimTarget.post_wrap = curve.post_wrap;
                            newParamAnimTarget.baked = curve.baked;

                            var stream = BinaryStreams[curve.stream_index];
                            var data = stream.FloatData;
                            Debug.Assert(data != null);
                            Debug.Assert(data.Count == curve.count * dataElementCount);

                            for (int i = 0; i != curve.count; ++i)
                            {
                                newParamAnimTarget.KeyFrames.Add(
                                    new KeyFrame()
                                    {
                                        Frame = data[i * dataElementCount + 0],
                                        Value = data[i * dataElementCount + 1],
                                        InSlope = data[i * dataElementCount + 2],
                                        OutSlope = data[i * dataElementCount + 3]
                                    }
                                );
                            }

                            break;
                        }

                    case InterpolationType.Linear:
                    case InterpolationType.Step:
                        {
                            const int dataElementCount = 2;		// frame, value

                            int stream_index = -1;
                            int count = -1;
                            {
                                if (newParamAnimTarget.CurveInterpolationType == InterpolationType.Linear)
                                {
                                    var curve = paramAnimTarget.Item as linear_curveType;
                                    Debug.Assert(curve != null);

                                    stream_index = curve.stream_index;
                                    count = curve.count;

                                    newParamAnimTarget.QuantizationInfo.frame_type = curve.frame_type;
                                    newParamAnimTarget.QuantizationInfo.key_type = curve.key_type;
                                    newParamAnimTarget.QuantizationInfo.scale = curve.scale;
                                    newParamAnimTarget.QuantizationInfo.offset = curve.offset;
                                    newParamAnimTarget.pre_wrap = curve.pre_wrap;
                                    newParamAnimTarget.post_wrap = curve.post_wrap;
                                    newParamAnimTarget.baked = curve.baked;
                                }
                                else
                                    if (newParamAnimTarget.CurveInterpolationType == InterpolationType.Step)
                                    {
                                        var curve = paramAnimTarget.Item as step_curveType;
                                        Debug.Assert(curve != null);

                                        stream_index = curve.stream_index;
                                        count = curve.count;

                                        newParamAnimTarget.QuantizationInfo.frame_type = curve.frame_type;
                                        newParamAnimTarget.QuantizationInfo.key_type = curve.key_type;
                                        newParamAnimTarget.QuantizationInfo.scale = curve.scale;
                                        newParamAnimTarget.QuantizationInfo.offset = curve.offset;
                                        newParamAnimTarget.pre_wrap = curve.pre_wrap;
                                        newParamAnimTarget.post_wrap = curve.post_wrap;
                                        newParamAnimTarget.baked = curve.baked;
                                    }

                                Debug.Assert(stream_index != -1);
                                Debug.Assert(count != -1);
                            }

                            var stream = BinaryStreams[stream_index];
                            var data = stream.FloatData;
                            Debug.Assert(data != null);
                            Debug.Assert(data.Count == count * dataElementCount);

                            for (int i = 0; i != count; ++i)
                            {
                                newParamAnimTarget.KeyFrames.Add(
                                    new KeyFrame()
                                    {
                                        Frame = data[i * dataElementCount + 0],
                                        Value = data[i * dataElementCount + 1]
                                    }
                                );
                            }

                            break;
                        }
                }
            }
            else
            {
                // コンスタントカーブの場合、中間ファイルにはカーブは存在しないので、
                // base_value を使用して、0フレームにキーフレームを生成する。
                newParamAnimTarget.KeyFrames.Add(
                    new KeyFrame()
                    {
                        Frame = 0.0f,
                        Value = paramAnimTarget.base_value
                    }
                );
            }

            return newParamAnimTarget;
        }

        public void UpdateData(bool viewer = false)
        {
            ToolData.Load(Data.tool_data);
            Data.comment = GetComment();

            if (Data.shader_param_mat_anim_array == null)
            {
                Data.shader_param_mat_anim_array = new shader_param_mat_anim_arrayType();
            }

            List<shader_param_mat_animType> ParamMatAnimTypelist = new List<shader_param_mat_animType>();

            // streamを新規に作成します。
            List<G3dStream> tempStreams = new List<G3dStream>();

            //int targetIndex = 0;	// stream_indexとして使用します。
            foreach (var ShaderParamAnim in ShaderParamAnims)
            {
                var matAnimType = new shader_param_mat_animType();

                // 値設定
                matAnimType.mat_name = ShaderParamAnim.mat_name;
                matAnimType.param_anim_array = new param_anim_arrayType();
                matAnimType.param_anim_array.length = ShaderParamAnim.ParamAnims.Count();

                var ParamAnimTypelist = new List<param_animType>();

                foreach (var paramAnim in ShaderParamAnim.ParamAnims)
                {
                    var paramanim = new param_animType();
                    var targetTypeList = new List<param_anim_targetType>();

                    int index = 0;
                    foreach (var target in paramAnim.ParamAnimTargets)
                    {
                        // キーフレームが無いものは削除するように修正します。
                        if (target.KeyFrames.Any() == false)
                        {
                            index++;
                            continue;
                        }

                        G3dStream stream;
                        var targetType = target.ConvertTo_param_anim_targetType(out stream, viewer && (index != 0 || !paramAnim.IsTexSRT));
                        targetTypeList.Add(targetType);

                        if (stream != null)
                        {
                            targetType.Curve.stream_index = tempStreams.Count;
                            tempStreams.Add(stream);
                        }
                        index++;
                    }

                    // TextureSRTの場合、Modeのみのカーブの場合は、保存しない。
                    if ((paramAnim.IsTexSRT))
                    {
                        if (targetTypeList.Count == 1)
                        {
                            var animTarget = targetTypeList[0];
                            if (animTarget.component_index == 0)
                            {
                                continue;
                            }
                        }
                    }

                    // ターゲットを持っていれば出力する
                    if (targetTypeList.Any())
                    {
                        paramanim.param_anim_target = targetTypeList.ToArray();

                        ParamAnimTypelist.Add(paramanim);
                        paramanim.id = paramAnim.id;
                        paramanim.type = paramAnim.type;
                        paramanim.original_hint = paramAnim.originalHint;
                    }
                }
                // リストを配列に変換して設定
                if (ParamAnimTypelist.Any())
                {
                    matAnimType.param_anim_array.param_anim = ParamAnimTypelist.ToArray();
                }
                else
                {
                    matAnimType.param_anim_array.length = 0;
                }

                if (matAnimType.param_anim_array.length == 0)
                {
                    matAnimType.param_anim_array = null;
                }
                else
                {
                    ParamMatAnimTypelist.Add(matAnimType);
                }
            }

            // リストを配列に変換して設定
            Data.shader_param_mat_anim_array.length					= ParamMatAnimTypelist.Count;
            Data.shader_param_mat_anim_array.shader_param_mat_anim	= ParamMatAnimTypelist.ToArray();

            if (Data.shader_param_mat_anim_array.length == 0)
            {
                Data.shader_param_mat_anim_array = null;
            }

            // オリジナルデータから、DCCツールのストリームの追加と参照IDの修正を行う。
            if (Data.original_material_anim_array != null)
            {
                foreach (var original in Data.original_material_anim_array.original_material_anim)
                {
                    if (original.original_color_anim_array != null)
                    {
                        foreach (var color_anim in original.original_color_anim_array.original_color_anim)
                        {
                            if (color_anim.original_color_anim_target != null)
                            {
                                foreach (var target in color_anim.original_color_anim_target)
                                {
                                    if (target.Curve != null)
                                    {
                                        var oldIndex = target.Curve.stream_index;
                                        target.Curve.stream_index = tempStreams.Count;
                                        tempStreams.Add(BinaryStreams[oldIndex]);
                                    }
                                }
                            }
                        }
                    }

                    if (original.original_texsrt_anim_array != null)
                    {
                        foreach (var texsrt_anim in original.original_texsrt_anim_array.original_texsrt_anim)
                        {
                            if (texsrt_anim.original_texsrt_anim_target != null)
                            {
                                foreach (var target in texsrt_anim.original_texsrt_anim_target)
                                {
                                    if (target.Curve != null)
                                    {
                                        var oldIndex = target.Curve.stream_index;
                                        target.Curve.stream_index = tempStreams.Count;
                                        tempStreams.Add(BinaryStreams[oldIndex]);
                                    }
                                }
                            }
                        }
                    }
                }
            }

            // ユーザーデータのシリアライズ
            if (UserDataArray.Data == null ||
                UserDataArray.Data.Count == 0)
            {
                Data.user_data_array = null;
            }
            else
            {
                Data.user_data_array = new user_data_arrayType();
                this.MakeSerializeData(Data.user_data_array, UserDataArray, tempStreams);
            }

            // ストリームデータの設定
            {
                BinaryStreams = tempStreams;

                // ストリームのソート(や削除等)を行う。
                nw.g3d.iflib.StreamUtility.SortStream(Data, BinaryStreams);

                Data.stream_array = null;
            }
        }

        public override EditCommand CreateUpdateBindCommand()
        {
            // ShaderParamAnims の更新


            // バインドされたマテリアルのグループ
            var materials = (from model in DocumentManager.Models
                            where model.AllAnimations.Contains(this)
                            from material in model.Materials
                            select material).ToArray();

            var materialTable = new Dictionary<string, List<Material>>();
            var materialNames = new List<string>();
            foreach (var material in materials)
            {
                List<Material> list;
                if (materialTable.TryGetValue(material.Name, out list))
                {
                    list.Add(material);
                }
                else
                {
                    materialTable.Add(material.Name, new List<Material>() { material });
                    materialNames.Add(material.Name);
                }
            }

            List<ShaderParamAnim> tmpList = new List<ShaderParamAnim>();
            // 既存のShaderParamAnim
            foreach (var shaderParamAnim in ShaderParamAnims)
            {
                if (materialTable.ContainsKey(shaderParamAnim.mat_name) ||
                    shaderParamAnim.ParamAnims.Any(x => x.ParamAnimTargets.Any(y => y.ExportType != CurveExportType.Ignored && y.ExportType != CurveExportType.Dependent)))
                {
                    tmpList.Add(shaderParamAnim);
                }
            }

            // 新たにShaderParamAnim を追加
            foreach (var materialName in materialNames)
            {
                if (!tmpList.Any(x => x.mat_name == materialName))
                {
                    tmpList.Add(new ShaderParamAnim()
                    {
                        mat_name = materialName,
                    });
                }
            }

            var newShaderParamAnims = new ShaderParamAnimList();
            newShaderParamAnims.AddRange(tmpList.OrderBy(x => x.mat_name));
            bool updateShaderParamAnims = false;
            if (!ShaderParamAnims.SequenceEqual(newShaderParamAnims))
            {
                updateShaderParamAnims = true;
            }

            // ParamAnims の更新
            var paramAnims = new List<ParamAnimsSwapData>();
            foreach (var shaderParamAnim in newShaderParamAnims)
            {
                List<Material> materialList;
                var paramNames = new List<string>();
                var shaderParamTable = new Dictionary<string, List<shader_param_typeType>>();
                if (materialTable.TryGetValue(shaderParamAnim.mat_name, out materialList))
                {
                    foreach (var shaderParam in materialList.SelectMany(x => x.MaterialShaderAssign.ShaderParams))
                    {
                        // TODO: カラーアニメーションのフィルタリング
                        if (ObjectID == GuiObjectID.TextureSrtAnimation)
                        {
                            if (!IsEnableParameter_TextureSrt(shaderParam.type))
                            {
                                continue;
                            }
                        }

                        List<shader_param_typeType> types;
                        if (shaderParamTable.TryGetValue(shaderParam.id, out types))
                        {
                            if (types.All(x => x != shaderParam.type))
                            {
                                types.Add(shaderParam.type);
                            }
                        }
                        else
                        {
                            paramNames.Add(shaderParam.id);
                            shaderParamTable.Add(shaderParam.id, new List<shader_param_typeType>() { shaderParam.type });
                        }
                    }
                }

                bool updateParamAnim = false;
                var newParamAnims = new ParamAnimList();

                // シェーダーパラメーターから
                foreach (var paramName in paramNames)
                {
                    List<shader_param_typeType> types;
                    if (shaderParamTable.TryGetValue(paramName, out types))
                    {
                        ParamAnim paramAnim = shaderParamAnim.ParamAnims[paramName];
                        if (paramAnim != null && !types.Contains(paramAnim.type))
                        {
                            if (paramAnim.ParamAnimTargets.All(x => x.ExportType == CurveExportType.Ignored || x.ExportType == CurveExportType.Dependent))
                            {
                                paramAnim = null;
                            }
                        }
                        if (paramAnim == null)
                        {
                            paramAnim = new ParamAnim(paramName, types[0]);
                            if (paramAnim.IsTexSRT)
                            {
                                SetTexSRTModeValue(paramAnim, shaderParamAnim.mat_name);
                                var key = CreateInitialSrtModesKey(shaderParamAnim.mat_name, paramName);
                                initialSrtModes[key] = ObjectUtility.Clone(paramAnim.ParamAnimTargets[0]);
                            }

                            newParamAnims.Add(paramAnim);
                            updateParamAnim = true;
                        }
                        else
                        {
                            newParamAnims.Add(paramAnim);
                        }
                    }
                }

                HashSet<string> ids = new HashSet<string>(newParamAnims.Select(x => x.id));

                // 既存のもののあまり
                foreach (var paramAnim in shaderParamAnim.ParamAnims)
                {
                    if (!ids.Contains(paramAnim.id))
                    {
                        if (paramAnim.ParamAnimTargets.Any(x => x.ExportType == CurveExportType.Constant || x.ExportType == CurveExportType.Curve))
                        {
                            newParamAnims.Add(paramAnim);
                            ids.Add(paramAnim.id);
                        }
                        else
                        {
                            // 削除
                            updateParamAnim = true;
                        }
                    }
                }

                if (updateParamAnim)
                {
                    paramAnims.Add(new ParamAnimsSwapData()
                    {
                        shaderParamAnim = shaderParamAnim,
                        ParamAnims = newParamAnims
                    });
                }
            }

            var comset = new EditCommandSet();

            if (updateShaderParamAnims || paramAnims.Count > 0)
            {
                // 編集フラグを更新しない
                comset.Add(
                    new GeneralGroupReferenceEditCommand<DummyObject>(
                    DummyObject.TheDummyObjectGroup,
                    GuiObjectID.DummyObject,
                    Enumerable.Repeat<DummyObject>(DummyObject.TheDummyObject, 1),
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        if (updateShaderParamAnims)
                        {
                            var temp = ShaderParamAnims;
                            ShaderParamAnims = newShaderParamAnims;
                            newShaderParamAnims = temp;
                        }

                        foreach (var swapData in paramAnims)
                        {
                            var temp = swapData.shaderParamAnim.ParamAnims;
                            swapData.shaderParamAnim.ParamAnims = swapData.ParamAnims;
                            swapData.ParamAnims = temp;
                        }

                        // 変更マークを更新する
                        UpdateIsModifiedAnimTargetAll();

                        // シリアライズされるデータに変更はないので Viewer に再転送する必要はない
                    })
                );
            }

            return comset;
        }

        public ShaderParamAnimList MergeInvisibleShaderParamAnimation(ShaderParamAnimList source)
        {
            var newShaderParamAnimList = source.Select(ObjectUtility.Clone).ToList();

            foreach (var shaderParamMatAnim in ShaderParamAnims)
            {
                var paramAnimState = GetParamAnimState(shaderParamMatAnim).ToArray();
                var invisibles = paramAnimState
                    .Where(x => x.Item3 == NonEditableKind.ShaderParamAnim_Invisible)
                    .ToArray();
                var disables = paramAnimState
                    .Where(x => x.Item3 == NonEditableKind.ShaderParamAnim_DisableByOptionVar)
                    .ToArray();

                if (invisibles.Length > 0 || disables.Length > 0)
                {
                    var targetMatAnim = newShaderParamAnimList.FirstOrDefault(x => x.mat_name == shaderParamMatAnim.mat_name);
                    if (targetMatAnim == null)
                    {
                        targetMatAnim = new ShaderParamAnim
                        {
                            mat_name = shaderParamMatAnim.mat_name,
                        };

                        newShaderParamAnimList.Add(targetMatAnim);
                    }

                    var newParamAnimList = new ParamAnimList();
                    var ids = new HashSet<string>();
                    foreach (var tuple in invisibles)
                    {
                        newParamAnimList.Add(tuple.Item1);
                        ids.Add(tuple.Item1.id);
                    }

                    foreach (var tuple in disables)
                    {
                        newParamAnimList.Add(tuple.Item1);
                        ids.Add(tuple.Item1.id);
                    }

                    foreach (var paramAnim in targetMatAnim.ParamAnims)
                    {
                        if (!ids.Contains(paramAnim.id))
                        {
                            newParamAnimList.Add(paramAnim);
                        }
                    }

                    targetMatAnim.ParamAnims = newParamAnimList;
                }
            }

            var shaderParamAnimList = new ShaderParamAnimList();
            shaderParamAnimList.AddRange(newShaderParamAnimList.OrderBy(x => x.mat_name));
            return shaderParamAnimList;
        }

        public ShaderParamAnimList VisibleShaderParamMatAnim()
        {
            var newShaderParamMatAnim = new ShaderParamAnimList();
            foreach (var shaderParamMatAnim in ShaderParamAnims)
            {
                var visibles = GetParamAnimState(shaderParamMatAnim)
                    .Where(x => x.Item3 != NonEditableKind.ShaderParamAnim_Invisible)
                    .Select(x => x.Item1)
                    .ToArray();
                if (visibles.Any())
                {
                    var shaderParamAnim = new ShaderParamAnim
                    {
                        mat_name = shaderParamMatAnim.mat_name,
                    };
                    shaderParamAnim.ParamAnims.AddRange(ObjectUtility.Clone(visibles));
                    newShaderParamMatAnim.Add(shaderParamAnim);
                }
            }

            return newShaderParamMatAnim;
        }

        private class ParamAnimsSwapData
        {
            public ShaderParamAnim shaderParamAnim;
            public ParamAnimList ParamAnims;
        };


        public IEnumerable<ParamAnim> GetParamAnims(string matName)
        {
            List<ParamAnim> targets = new List<ParamAnim>();

            ShaderParamAnim matAnim = ShaderParamAnims.FirstOrDefault(x => x.mat_name == matName);
            if (matAnim == null)
            {
                return targets;
            }

            if (matAnim.ParamAnims != null)
            {
                return matAnim.ParamAnims;
            }

            return targets;
        }

        public IEnumerable<ParamAnim> GetParamAnims(string matName, string paramAnimId)
        {
            List<ParamAnim> targets = new List<ParamAnim>();

            ShaderParamAnim matAnim = ShaderParamAnims.FirstOrDefault(x => x.mat_name == matName);
            if (matAnim == null)
            {
                return targets;
            }

            IEnumerable<ParamAnim> paramAnims = matAnim.ParamAnims.Where(y => y.id == paramAnimId);
            if (paramAnims == null)
            {
                return targets;
            }

            return paramAnims;
        }

        public void SwapParamAnim(string matName, ParamAnim oldParamAnim, ParamAnim newParamAnim)
        {
            foreach (var matAnim in ShaderParamAnims)
            {
                if (matAnim.mat_name == matName)
                {
                    ParamAnimList paramAnims = new ParamAnimList();
                    foreach (var paramAnim in matAnim.ParamAnims)
                    {
                        paramAnims.Add(paramAnim == oldParamAnim ? newParamAnim : paramAnim);
                    }
                    matAnim.ParamAnims = paramAnims;
                }
            }
        }

        static public void SetTexSRTModeValue(ParamAnim paramAnim, string materialName)
        {
            if (paramAnim.ParamAnimTargets[0].KeyFrames == null)
            {
                paramAnim.ParamAnimTargets[0].KeyFrames = new List<KeyFrame>();
            }
            if (!paramAnim.ParamAnimTargets[0].KeyFrames.Any())
            {
                paramAnim.ParamAnimTargets[0].KeyFrames.Add(new KeyFrame());
            }

            foreach (var material in DocumentManager.Materials)
            {
                if (material.Name == materialName)
                {
                    foreach (var shaderParam in material.MaterialShaderAssign.ShaderParams)
                    {
                        if (paramAnim.id == shaderParam.id)
                        {
                            float[] paramArray = nw.g3d.nw4f_3dif.G3dDataParser.ParseFloatArray(shaderParam.Value);
                            if (paramArray != null)
                            {
                                paramAnim.ParamAnimTargets[0].KeyFrames[0].Value = paramArray[0];
                                paramAnim.ParamAnimTargets[0].IsTexSrt = true;
                                return;
                            }
                        }
                    }
                }
            }
        }

        static public void SetTexSRTModeValue(ParamAnim paramAnim, string materialName, ParamAnimTarget target)
        {
            if (target.KeyFrames == null)
            {
                target.KeyFrames = new List<KeyFrame>();
            }
            if (!target.KeyFrames.Any())
            {
                target.KeyFrames.Add(new KeyFrame());
            }

            foreach (var material in DocumentManager.Materials)
            {
                if (material.Name == materialName)
                {
                    foreach (var shaderParam in material.MaterialShaderAssign.ShaderParams)
                    {
                        if (paramAnim.id == shaderParam.id)
                        {
                            float[] paramArray = nw.g3d.nw4f_3dif.G3dDataParser.ParseFloatArray(shaderParam.Value);
                            if (paramArray != null)
                            {
                                target.KeyFrames[0].Value = paramArray[0];
                                target.IsTexSrt = true;
                                return;
                            }
                        }
                    }
                }
            }
        }

        #region savedData
        private shader_param_anim_infoType savedData;
        public ShaderParamAnimList savedShaderParamAnims { get; private set; }

        /// <summary>
        /// 勝手に割り当てられた SRT モード
        /// </summary>
        public Dictionary<string, ParamAnimTarget> initialSrtModes { get; private set; }

        /// <summary>
        /// initialSrtModes 用のキー生成
        /// </summary>
        public string CreateInitialSrtModesKey(string mat, string param)
        {
            var builder = new System.Text.StringBuilder();
            builder.Append(mat);
            builder.Append("*");
            builder.Append(param);
            return builder.ToString();
        }

        /// <summary>
        /// 空以外の保存されたカーブ
        /// </summary>
        public int SavedCurveCount { get; private set; }

        public override void UpdateSavedData()
        {
            DebugConsole.WriteLine("ShaderParameterAnimation UpdateSavedData");
            base.UpdateSavedData();
            savedData = ObjectUtility.Clone(Data.shader_param_anim_info);

            SavedUserDataArray = ObjectUtility.Clone(UserDataArray);
            UserDataArrayChanged = false;

            SavedCurveCount = 0;
            foreach (var sheaderParamAnim in ShaderParamAnims)
            {
                foreach (var paramAnim in sheaderParamAnim.ParamAnims)
                {
                    foreach (var paramAnimTarget in paramAnim.ParamAnimTargets)
                    {
                        paramAnimTarget.IsModified = false;
                        if (NotEmpty(paramAnimTarget))
                        {
                            paramAnimTarget.IsSaved = true;
                            SavedCurveCount++;
                        }
                        else
                        {
                            paramAnimTarget.IsSaved = false;
                        }
                    }
                }
            }
            savedShaderParamAnims = ObjectUtility.Clone(ShaderParamAnims);
            initialSrtModes = new Dictionary<string, ParamAnimTarget>();
        }

        public void CopySavedData(ShaderParameterAnimation source)
        {
            CopyDocumentSavedData(source);
            savedData = source.savedData;

            SavedUserDataArray = source.SavedUserDataArray;
            UserDataArrayChanged = !SavedUserDataArray.IsSame(UserDataArray);

            SavedCurveCount = source.SavedCurveCount;
            savedShaderParamAnims = source.savedShaderParamAnims;
            initialSrtModes = source.initialSrtModes;

            UpdateIsModifiedAnimTargetAll();
        }

        public override bool EqualsToSavedData()
        {
            if (!base.EqualsToSavedData())
            {
                return false;
            }

            return !ShaderParameterAnimationGeneralPage.IsModified(this) &&
                !IsCurvesModified() &&
                !UserDataPage.IsModified(this);
        }

        public void UpdateIsModifiedAnimTargetAll()
        {
            foreach (var shaderParamAnim in ShaderParamAnims)
            {
                var savedShaderParamAnim = savedShaderParamAnims[shaderParamAnim.mat_name];
                foreach (var paramAnim in shaderParamAnim.ParamAnims)
                {
                    var savedParamAnim = savedShaderParamAnim != null ? savedShaderParamAnim.ParamAnims[paramAnim.id] : null;
                    UpdateIsModifiedParamAnim(paramAnim, savedParamAnim);
                    foreach (var paramAnimTarget in paramAnim.ParamAnimTargets)
                    {
                        var savedAnimTarget = savedParamAnim != null ?
                            savedParamAnim.ParamAnimTargets.FirstOrDefault(x => x.component_index == paramAnimTarget.component_index) :
                            null;

                        if (savedAnimTarget == null && paramAnimTarget.IsTexSrt && paramAnimTarget.component_index == 0)
                        {
                            var key = CreateInitialSrtModesKey(shaderParamAnim.mat_name, paramAnim.id);
                            initialSrtModes.TryGetValue(key, out savedAnimTarget);
                        }

                        UpdateIsModifiedAnimTarget(paramAnimTarget, savedAnimTarget);
                    }

                    UpdateIsModifiedParamAnim(paramAnim, savedParamAnim);
                }
            }
        }

        public void UpdateIsModifiedParamAnim(ParamAnim current, ParamAnim saved)
        {
            current.IsModified = saved != null && (current.type != saved.type || current.originalHint != saved.originalHint);
        }

        /// <summary>
        /// カーブの変更フラグを更新する
        /// </summary>
        public void UpdateIsModifiedAnimTarget(ParamAnimTarget current, ParamAnimTarget saved)
        {
            if (saved == null)
            {
                current.IsModified = NotEmpty(current);
                current.IsSaved = false;
            }
            else
            {
                current.IsModified = !current.IsSame(saved);
                current.IsSaved = saved.IsSaved;
            }
        }

        public bool NotEmpty(ParamAnimTarget current)
        {
            if (current.IsTexSrt && current.component_index == 0)
            {
                if (current.KeyFrames.Any())
                {
                    float baseValue;
                    current.KeyFrames.IsConstantCurve(out baseValue);
                    return baseValue != 0;
                }
                else
                {
                    return false;
                }
            }
            else
            {
                return current.KeyFrames.Count > 0;
            }
        }

        public bool IsValueChanged<T>(Func<shader_param_anim_infoType, T> select) where T : struct
        {
            return !select(savedData).Equals(select(Data.shader_param_anim_info));
        }

        public bool IsStringChanged(Func<shader_param_anim_infoType, string> select)
        {
            return select(savedData) != select(Data.shader_param_anim_info);
        }

        public bool IsCurvesModified()
        {
            int savedCount = 0;
            foreach (var shaderParamAnim in ShaderParamAnims)
            {
                foreach (var paramAnim in shaderParamAnim.ParamAnims)
                {
                    if (paramAnim.IsModified)
                    {
                        return true;
                    }
                    foreach (var paramAnimTarget in paramAnim.ParamAnimTargets)
                    {
                        if (paramAnimTarget.IsModified)
                        {
                            return true;
                        }
                        if (paramAnimTarget.IsSaved)
                        {
                            savedCount++;
                        }
                    }
                }
            }

            return savedCount != SavedCurveCount;
        }

        #endregion

        public override void OnDocumentSwapAdd(Document old)
        {
            CopySavedData((ShaderParameterAnimation)old);
            base.OnDocumentSwapAdd(old);
        }

        public bool IsTextureSrtType()
        {
            return ObjectID == GuiObjectID.ShaderParameterAnimation || ObjectID == GuiObjectID.TextureSrtAnimation;
        }

        public bool IsColorType()
        {
            return ObjectID == GuiObjectID.ShaderParameterAnimation || ObjectID == GuiObjectID.ColorAnimation;
        }

        public bool IsShaderParameterType()
        {
            return ObjectID == GuiObjectID.ShaderParameterAnimation;
        }

        private IEnumerable<AnimTarget> AllAnimTargets()
        {
            return from matAnim in ShaderParamAnims
                   from paramAnim in matAnim.ParamAnims
                   from animTarget in paramAnim.ParamAnimTargets
                   select animTarget;
        }

        public bool HasAnyShaderParamCurves()
        {
            return AllAnimTargets().Any(x => { var type = x.ExportType; return type == CurveExportType.Curve || type == CurveExportType.Constant; });
        }
    }
}
