﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using App.Command;
using ConfigCommon;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;

namespace App.Data
{
    public abstract class AnimationDocument : IntermediateFileDocument, IPause, IQuantizeAnalysis
    {
        // 編集不可種類
        public enum NonEditableKind
        {
            // シェーダーパラメーター
            ShaderParamAnim_DisableByOptionVar,		// オプション変数で無効になっている
            ShaderParamAnim_NotFoundParam,			// パラメーターが見つからない
            ShaderParamAnim_NotFoundMaterial,		// モデルにマテリアルが見つからない
            ShaderParamAnim_NotBindModel,			// モデルにバインドされていない
            ShaderParamAnim_NotAssignShaderDef,		// マテリアルにシェーダーが設定されていない
            ShaderParamAnim_NotLoadShaderDef,		// シェーダー定義ファイルが読み込まれていない
            ShaderParamAnim_NotShadingModel,        // シェーディングモデルが見つかりません
            ShaderParamAnim_TypeConflict,           // カーブの型とパラメーターの型が一致しない
            ShaderParamAnim_InvalidType,           // カーブの型がファイルに一致しない
            ShaderParamAnim_Invisible,             // 非表示

            // ボーンビジビリティ
            BoneVisibility_NotFoundBone,			// ボーンが見つからない
            BoneVisibility_NotBinarize,		    	// バイナリ出力がオフ

            // マテリアルビジビリティ
            MaterialVisibility_NotFoundMaterial,	// マテリアルが見つからない

            // ライト
            Light_NotFoundType,						// 指定タイプのライトが見つからない

            // テクスチャパターン
            TexturePattern_NotFoundSampler,			// サンプラが見つからない

            // スケルタルアニメーション
            SkeletalAnimation_NotEditable,			// スケルタルアニメーションは編集不可

            // シェイプアニメーション
            ShapeAnimation_NotEditable,			    // シェイプアニメーションは編集不可(TeamConfigで有効にできる)

            //
            Editable = -1,							// 編集可能
        }

        protected AnimationDocument(GuiObjectID objectID, List<G3dStream> streams) :
            base(objectID, streams)
        {
        }

        public abstract int  FrameCount { get; set; }
        public abstract bool Loop       { get; set; }

        public abstract bool IsFrameCountModified   { get; }
        public abstract bool IsLoopModified         { get; }


        public virtual bool IsEditable { get { return true; } }	// Todo: abstract にする

        // マテリアルに関係するか
        public virtual bool IsMaterialRelated { get { return false; } }

        // マージできるかどうか
        public virtual bool CanMerge
        {
            get
            {
                return true;
            }
        }

        public abstract bool HasAnimation { get; }

        [Serializable]
        public class QuantizationInfo
        {
            public QuantizationInfo()
            {
                scale = 1.0f;
            }

            public curve_frame_typeType frame_type { get; set; }
            public curve_key_typeType key_type { get; set; }
            public float scale { get; set; }
            public float offset { get; set; }

            public bool IsSame(QuantizationInfo p)
            {
                // Return true if the fields match:
                return p != null &&
                    (frame_type == p.frame_type) &&
                    (key_type == p.key_type) &&
                    (scale == p.scale) &&
                    (offset == p.offset);
            }

            public override bool Equals(object obj)
            {
                // TODO: いずれ削除
                Debug.Assert(false);
                // If parameter is null return false.
                if (obj == null)
                {
                    return false;
                }

                // If parameter cannot be cast to Point return false.
                QuantizationInfo p = obj as QuantizationInfo;
                if ((object)p == null)
                {
                    return false;
                }

                // Return true if the fields match:
                return
                    (frame_type	== p.frame_type) &&
                    (key_type	== p.key_type) &&
                    (scale		== p.scale) &&
                    (offset		== p.offset);
            }

            public override int GetHashCode()
            {
                // TODO: いずれ削除
                Debug.Assert(false);
                return base.GetHashCode();
            }
        }

        // FrameCount 変更イベント。
        // すべてのドキュメントに反応します。
        public static event EventHandler FrameCountChanged = null;
        public static void NotifyFrameCountChanged(object sender, EventArgs args)
        {
            if (FrameCountChanged != null)
            {
                FrameCountChanged(sender, args);
            }
        }

        /// <summary>
        /// バインドの更新に伴うコマンドを作成する
        /// </summary>
        public virtual EditCommand CreateUpdateBindCommand()
        {
            return null;
        }

        public abstract IEnumerable<Tuple<AnimTarget, IfQuantizationAnalysisResult>> QuantizeAnalyse();

        #region PauseData
        public class PauseData
        {
            public HashSet<KeyValuePair<object, string>> InvisibleBinds = new HashSet<KeyValuePair<object, string>>();
            public Dictionary<KeyValuePair<object, string>, double> PauseFrames = new Dictionary<KeyValuePair<object, string>, double>();
            public Dictionary<KeyValuePair<object, string>, double> lastPauseFrames = new Dictionary<KeyValuePair<object, string>, double>();
            public bool Pausing { get; set; }
            public double PauseFrame { get; set; }
            public Dictionary<object, string> PreviewingAnimationSets
                = new Dictionary<object, string>();
        }

        public PauseData Pause = new PauseData();
        public double PauseFrame
        {
            get
            {
                return Pause.PauseFrame;
            }
        }

        public bool Pausing
        {
            get
            {
                return Pause.Pausing;
            }
        }

        public Dictionary<KeyValuePair<object, string>, double> PauseFrames
        {
            get
            {
                return Pause.PauseFrames;
            }
        }

        public Dictionary<object, string> PreviewingAnimationSets
        {
            get
            {
                return Pause.PreviewingAnimationSets;
            }
        }

        public void SetPause(bool pause, double frame)
        {
            if (pause)
            {
                foreach (var key in Pause.PreviewingAnimationSets)
                {
                    Pause.PauseFrames[key] = frame;
                }
            }
            else
            {
                foreach (var key in Pause.PreviewingAnimationSets)
                {
                    Pause.PauseFrames.Remove(key);
                }
            }

            SendViewer();
        }

        public void SetPause(bool pause, double frame, object modelId, string animationSet, bool invisible)
        {
            var key = new KeyValuePair<object, string>(modelId, animationSet);
            if (pause)
            {
                Pause.PauseFrames[key] = frame;
            }
            else
            {
                Pause.PauseFrames.Remove(key);
            }

            if (invisible)
            {
                Pause.InvisibleBinds.Add(key);
            }
            else
            {
                Pause.InvisibleBinds.Remove(key);
            }

            SendViewer();
        }

        public void SetPreviewAnimationSet(object modelId, string animationSet)
        {
            string current;
            if (!Pause.PreviewingAnimationSets.TryGetValue(modelId, out current) ||
                current != animationSet)
            {
                Pause.PreviewingAnimationSets[modelId] = animationSet;
                SendViewer();
            }
        }

        public void ClearPreviewAnimationSet(object modelId)
        {
            if (Pause.PreviewingAnimationSets.Remove(modelId))
            {
                SendViewer();
            }
        }

#if true
        private void SendViewer()
        {
            foreach (var animationSet in Pause.PreviewingAnimationSets)
            {
                if (Pause.InvisibleBinds.Contains(animationSet))
                {
                    continue;
                }

                double frame = 0;
                bool pause = Pause.PauseFrames.TryGetValue(animationSet, out frame);
                var model = DocumentManager.Models.FirstOrDefault(x => x.ModelId == animationSet.Key);
                double lastFrame = 0;
                var lastPause = Pause.lastPauseFrames.TryGetValue(animationSet, out lastFrame);
                if (pause != lastPause || frame != lastFrame)
                {
                    if (!pause)
                    {
                        Pause.lastPauseFrames.Remove(animationSet);
                    }
                    else
                    {
                        Pause.lastPauseFrames[animationSet] = frame;
                    }

                    if (model != null)
                    {
                        Viewer.AnimationPauseMessage.Send(this, pause, (float)frame, model);
                    }
                }
            }
        }
#else
        private void SendViewer()
        {
            bool pause;
            double frame = 0;
            if (Pause.PreviewingAnimationSets.Count == 1)
            {
                var key = Pause.PreviewingAnimationSets.First();
                if (!Pause.PauseFrames.TryGetValue(key, out frame))
                {
                    pause = false;
                }
                else
                {
                    pause = true;
                }
            }
            else
            {
                pause = false;
            }

            bool send = pause != Pause.Pausing || (pause && frame != Pause.PauseFrame);
            Pause.Pausing = pause;
            Pause.PauseFrame = frame;
            if (send)
            {
                Viewer.AnimationPauseMessage.Send(this, Pause.Pausing, (float)Pause.PauseFrame);
            }
        }
#endif
        #endregion

        public abstract class SwapData
        {
        }

        // TODO: abstract にする
        public virtual SwapData CreateSwapData(object nw4f_3difItem, List<G3dStream> streams) { return null; }
        public virtual SwapData GetSwapData() { return null; }
        public virtual SwapData Swap(SwapData data, bool copyHistory) { return null; }

        public override void ResetStatus()
        {
            base.ResetStatus();
        }

        // SetRetargetingHostModel されたモデル
        // 送信スレッド専用
        public Model PreviewingRetargetHost { get; set; }

        public Model GetRetargetHost()
        {
            return DocumentManager.Models.FirstOrDefault(x => x.Name == RetargetingHostModelName);
        }

        virtual public string RetargetingHostModelName
        {
            get
            {
                return null;
            }
            set
            {
                // 何もしない
            }
        }

        public override void UnloadedFromHio()
        {
            PreviewingRetargetHost = null;
            base.UnloadedFromHio();
        }
    }

    interface IQuantizeAnalysis
    {
        IEnumerable<Tuple<AnimTarget, IfQuantizationAnalysisResult>> QuantizeAnalyse();
    }

    /// <summary>
    /// 再生の一時停止
    /// </summary>
    public interface IPause
    {
        bool Pausing { get; }
        double PauseFrame { get; }
        void SetPause(bool pause, double frame);
    }
}
