﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.ComponentModel;
using System.Drawing;
using System.Runtime.InteropServices;
using System.Windows.Forms;

namespace App.Controls
{
    using System.Collections.Generic;
    using System.Linq;

    /// <summary>
    /// ＵＩツリービュークラス。
    /// </summary>
    [ToolboxBitmap(typeof(TreeView))]
    public class UITreeView : TreeView
    {
        // ダブルクリック中フラグ
        private bool _doubleClicking = false;

        // ダブルクリック中のノードの展開の抑止を行わない
        // プログラム内の Expand()
        public bool ForceExpand = false;

        // ダブルクリック展開抑制フラグ
        private bool _suppressDoubleClickExpand = false;

        // 描画バッファ
        private Bitmap   _internalBmp = null;
        private Graphics _internalGfx = null;

        /// <summary>コンテキストメニューポップアップイベント。</summary>
        public event ContextMenuPopupEventHandler ContextMenuPopup = null;

        public int ScrollX
        {
            get
            {
                var scrollInfo = new App.Win32.NativeMethods.SCROLLINFO();
                scrollInfo.cbSize = (uint)Marshal.SizeOf(scrollInfo);
                scrollInfo.fMask = (int)App.Win32.NativeMethods.ScrollInfoMask.SIF_POS;

                App.Win32.NativeMethods.GetScrollInfo(Handle, (int)App.Win32.NativeMethods.ScrollBarDirection.SB_HORZ, ref scrollInfo);

                return scrollInfo.nPos;
            }

            set
            {
                var scrollInfo = new App.Win32.NativeMethods.SCROLLINFO();
                scrollInfo.cbSize = (uint)Marshal.SizeOf(scrollInfo);
                scrollInfo.fMask = (int)App.Win32.NativeMethods.ScrollInfoMask.SIF_POS;

 				scrollInfo.nPos = value;
                App.Win32.NativeMethods.SetScrollInfo(Handle, (int)App.Win32.NativeMethods.ScrollBarDirection.SB_HORZ, ref scrollInfo, 0);
            }
        }

        public int ScrollY
        {
            get
            {
                var scrollInfo = new App.Win32.NativeMethods.SCROLLINFO();
                scrollInfo.cbSize = (uint)Marshal.SizeOf(scrollInfo);
                scrollInfo.fMask = (int)App.Win32.NativeMethods.ScrollInfoMask.SIF_POS;

                App.Win32.NativeMethods.GetScrollInfo(Handle, (int)App.Win32.NativeMethods.ScrollBarDirection.SB_VERT, ref scrollInfo);

                return scrollInfo.nPos;
            }

            set
            {
                var scrollInfo = new App.Win32.NativeMethods.SCROLLINFO();
                scrollInfo.cbSize = (uint)Marshal.SizeOf(scrollInfo);
                scrollInfo.fMask = (int)App.Win32.NativeMethods.ScrollInfoMask.SIF_POS;

 				scrollInfo.nPos = value;
                App.Win32.NativeMethods.SetScrollInfo(Handle, (int)App.Win32.NativeMethods.ScrollBarDirection.SB_VERT, ref scrollInfo, 0);
            }
        }

        public List<TreeNode> SelectedNodes { get; internal set; } = new List<TreeNode>();

        private TreeNode FirstNode { get; set; } = null;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UITreeView()
        {
            base.HideSelection = false;
            LastDragScroll = DateTime.Now;
        }

        /// <summary>
        /// ダブルクリック展開抑制フラグ。
        /// </summary>
        [DefaultValue(false)]
        [Description("項目上でのダブルクリック時に展開を行わないようにします。")]
        public bool SuppressDoubleClickExpand
        {
            get { return _suppressDoubleClickExpand; }
            set { _suppressDoubleClickExpand = value; }
        }

        /// <summary>
        /// コンテキストメニューポップアップハンドラ。
        /// </summary>
        protected virtual void OnContextMenuPopup(ContextMenuPopupEventArgs e)
        {
            if (ContextMenuPopup != null)
            {
                ContextMenuPopup(this, e);
            }
        }

        /// <summary>
        /// 内部コンポーネントを破棄。
        /// </summary>
        private void DisposeComponent()
        {
            if (_internalGfx != null)
            {
                _internalGfx.Dispose();
                _internalGfx = null;
            }
            if (_internalBmp != null)
            {
                _internalBmp.Dispose();
                _internalBmp = null;
            }
        }

        #region オーバーライド
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool CanRaiseEvents
        {
            get
            {
                if (UIControlEventSuppressBlock.Enabled)
                {
                    return false;
                }
                return base.CanRaiseEvents;
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void Dispose(bool disposing)
        {
            if (disposing)
            {
                DisposeComponent();
            }
            base.Dispose(disposing);
        }

        private const int TVM_SETEXTENDEDSTYLE = 0x1100 + 44;
        private const int TVS_EX_DOUBLEBUFFER = 0x0004;

        protected override void OnHandleCreated(EventArgs e)
        {
            base.OnHandleCreated(e);

            // チラツキを抑える
            Win32.NativeMethods.SendMessage(Handle, TVM_SETEXTENDEDSTYLE, new IntPtr(TVS_EX_DOUBLEBUFFER), new IntPtr(TVS_EX_DOUBLEBUFFER));
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void WndProc(ref Message m)
        {
            switch (m.Msg)
            {
                // ダブルクリック終了
                case Win32.WM.WM_LBUTTONUP:
                    {
                        base.WndProc(ref m);
                        _doubleClicking = false;
                    }
                    return;

                // ダブルクリック開始
                case Win32.WM.WM_LBUTTONDBLCLK:
                    {
                        // チェックボックス上のダブルクリックは無視する
                        // http://connect.microsoft.com/VisualStudio/feedback/details/374516/treeview-control-does-not-fire-events-reliably-when-double-clicking-on-checkbox
                        if (IsOnCheckBox(m))
                        {
                            m.Result = IntPtr.Zero;
                        }
                        else
                        {
                            _doubleClicking = true;
                            base.WndProc(ref m);
                        }
                    }
                    return;

                default:
                    break;
            }
            base.WndProc(ref m);
        }

        private bool IsOnCheckBox(Message m)
        {
            var x = m.LParam.ToInt32() & 0xFFFF;
            var y = m.LParam.ToInt32() >> 16;

            return HitTest(x, y).Location == TreeViewHitTestLocations.StateImage;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnBeforeCollapse(TreeViewCancelEventArgs e)
        {
            // ダブルクリック展開抑制
            if (_doubleClicking && _suppressDoubleClickExpand)
            {
                // プラスマイナス領域以外は動作させない
                TreeViewHitTestInfo ht = HitTest(PointToClient(Control.MousePosition));
                if (ht.Location != TreeViewHitTestLocations.PlusMinus)
                {
                    e.Cancel = true;
                }
            }
            base.OnBeforeCollapse(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnBeforeExpand(TreeViewCancelEventArgs e)
        {
            // ダブルクリック展開抑制
            if (_doubleClicking && _suppressDoubleClickExpand && !ForceExpand)
            {
                // プラスマイナス領域以外は動作させない
                TreeViewHitTestInfo ht = HitTest(PointToClient(Control.MousePosition));
                if (ht.Location != TreeViewHitTestLocations.PlusMinus)
                {
                    e.Cancel = true;
                }
            }
            base.OnBeforeExpand(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnResize(EventArgs e)
        {
            // 描画バッファを再作成
            if (_internalBmp        == null       ||
                _internalBmp.Width  != Width ||
                _internalBmp.Height != Height)
            {
                if (Width > 0 && Height > 0)
                {
                    DisposeComponent();
                    _internalBmp = new Bitmap(Width, Height);
                    _internalGfx = Graphics.FromImage(_internalBmp);
                }
            }
            base.OnResize(e);
        }

        // ノードが複数選択出来るかどうか
        protected virtual bool CanMultiSelect(TreeNode node, TreeNode selectedNode = null)
        {
            return true;
        }

        // クリックで複数選択を維持するかどうか(アイコン部分のクリックで維持したいときなどに使用)
        protected virtual bool KeepMultiSelectByClick(TreeNode treenode, MouseEventArgs e)
        {
            return false;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnKeyUp(KeyEventArgs e)
        {
            FirstNode = null;
            // コンテキストメニュー
            if (e.KeyCode == Keys.Apps)
            {
                // コンテキストメニュー
                // ノード位置が取れないのでカーソル位置に表示
                OnContextMenuPopup(new ContextMenuPopupEventArgs(PointToClient(Cursor.Position), true));
            }
            base.OnKeyUp(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnMouseDown(MouseEventArgs e)
        {
            var leftClick = e.Button.HasFlag(MouseButtons.Left);
            var rightClick = e.Button.HasFlag(MouseButtons.Right);
            // 左右ボタン
            if (!leftClick && !rightClick)
            {
                base.OnMouseDown(e);
                return;
            }

            // 選択済みじゃなければ選択
            var node = GetNodeAt(e.X, e.Y);
            if (node == null) // || node == SelectedNode)
            {
                base.OnMouseDown(e);
                return;
            }

            if (!AllowMultiSelect)
            {
                if (node != SelectedNode)
                {
                    SelectedNodes.Clear();
                    SelectedNodes.Add(node);
                    SelectedNode = node;
                }
                base.OnMouseDown(e);
                return;
            }

            // Contrlキーで追加選択
            if (ModifierKeys.HasFlag(Keys.Control))
            {
                // すでに選択済みなら非選択に
                if (SelectedNodes.Contains(node))
                {
                    // 左クリックで、複数選択されているときだけ非選択に出来る（最後の一つは非選択に出来ない）
                    if (leftClick && SelectedNodes.Count >= 2)
                    {
                        SelectedNodes.Remove(node);
                        SelectedNode = SelectedNodes.LastOrDefault();
                    }
                }
                else
                {
                    if (CanMultiSelect(node))
                    {
                        SelectedNodes.Add(node);
                        SelectedNode = node;
                    }
                }
            }
            else if (ModifierKeys.HasFlag(Keys.Shift))
            {
                if (CanMultiSelect(node))
                {
                    // Shiftを押したまま選択し直したとき
                    if (FirstNode == null)
                    {
                        FirstNode = SelectedNode;
                    }
                    SelectedNodes = GetBetweenNodes(FirstNode, node).Distinct().Where(x => CanMultiSelect(x, FirstNode)).ToList();
                    SelectedNode = node;
                }
            }
            else
            {
                // 非選択ノードをクリックで、複数選択解除
                if ((leftClick || !SelectedNodes.Contains(node)) && !KeepMultiSelectByClick(node, e))
                {
                    SelectedNodes.Clear();
                    SelectedNodes.Add(node);
                    SelectedNode = node;
                }
            }
            base.OnMouseDown(e);
            Refresh();
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnMouseUp(MouseEventArgs e)
        {
            // 右ボタン
            if (e.Button == MouseButtons.Right)
            {
                // コンテキストメニュー
                OnContextMenuPopup(new ContextMenuPopupEventArgs(e.Location, false));
            }
            base.OnMouseUp(e);
        }

        // ノードを表示順と同じ並びになるように列挙
        public IEnumerable<TreeNode> GetChildrenNodes(TreeNodeCollection nodeCollection)
        {
            var nodes = new List<TreeNode>();
            foreach (var child in nodeCollection.OfType<TreeNode>())
            {
                nodes.Add(child);
                nodes.AddRange(GetChildrenNodes(child.Nodes));
            }
            return nodes;
        }

        // 2つのツリーノードの間のノードを列挙
        private IEnumerable<TreeNode> GetBetweenNodes(TreeNode startNode, TreeNode endNode, bool includeStart = true, bool includeEnd = true)
        {
            var allNodes = GetChildrenNodes(Nodes);
            var nodes = new List<TreeNode>();

            TreeNode finish = null;
            foreach (var node in allNodes)
            {
                if (finish == null)
                {
                    if (node == startNode)
                    {
                        finish = endNode;
                        if (includeStart)
                        {
                            nodes.Add(node);
                        }
                    }
                    else if (node == endNode)
                    {
                        finish = startNode;
                        if (includeEnd)
                        {
                            nodes.Add(node);
                        }
                    }
                }

                if (finish != null)
                {
                    if (node == finish)
                    {
                        if ((node == endNode && includeEnd) || (node == startNode && includeStart))
                        {
                            nodes.Add(node);
                        }
                        break;
                    }
                    nodes.Add(node);
                }
            }

            return nodes;
        }

        /// <summary>
        /// オーバーライド
        /// </summary>
        protected override void OnBeforeSelect(TreeViewCancelEventArgs e)
        {
            // UITreeView の OnMouseDown 時に Node の選択を直接行っているので、
            // デフォルトのマウス操作によるときは選択しない。
            if (e.Action == TreeViewAction.ByMouse)
            {
                e.Cancel = true;
            }

            base.OnBeforeSelect(e);
        }
        /// <summary>
        /// ドラッグにより最後にスクロールされた時間
        /// </summary>
        private DateTime LastDragScroll;

        /// <summary>
        /// オーバーライド
        /// </summary>
        protected override void OnDragOver(DragEventArgs drgevent)
        {
            // ドラッグ中にスクロールを行う
            var now = DateTime.Now;
            if (Math.Abs((now - LastDragScroll).Milliseconds) > 100)
            {
                Point pt = PointToClient(new Point(drgevent.X, drgevent.Y));
                if (pt.Y + 20 > Height)
                {
                    Win32.NativeMethods.SendMessage(Handle, Win32.WM.WM_VSCROLL, new IntPtr(1), IntPtr.Zero);
                    LastDragScroll = now;
                }
                else if (pt.Y < 20)
                {
                    Win32.NativeMethods.SendMessage(Handle, Win32.WM.WM_VSCROLL, IntPtr.Zero, IntPtr.Zero);
                    LastDragScroll = now;
                }
            }

            base.OnDragOver(drgevent);
        }
        #endregion

        #region デザイナ制御
        /// <summary>
        /// 再定義。
        /// </summary>
        [DefaultValue(false)]
        public new bool HideSelection
        {
            get { return base.HideSelection;  }
            set { base.HideSelection = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new Color LineColor
        {
            // 継承されたフォームで LineColor=Black が強制的に
            // 設定されてしまうのでデザイナ公開しないようにする
            get { return base.LineColor; }
            set { base.LineColor = value; }
        }

        /// <summary>
        /// 複数選択用フラグ
        /// </summary>
        [DefaultValue(false)]
        public bool AllowMultiSelect { get; set; }

        #endregion
    }

    public static class TreeNodeExtension
    {
        // 拡張メソッド
        public static void UpdateText(this TreeNode treeNode, string text)
        {
            if (treeNode.Text != text)
            {
                treeNode.Text = text;
            }
        }
    }
}
