﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using App.Properties;
using App.Utility;
using App.res;
using nw.g3d.iflib;

namespace App.Controls
{
    /// <summary>
    /// マージ対象選択パネルクラス。
    /// </summary>
    public partial class MergeTargetSelectPanel : UIUserControl
    {
        public string[] OrgSrcNames { get; set; }
        public string[] OrgDstNames { get; set; }
        public Image TargetImage { get; set; }

        public bool IsSame
        {
            get
            {
                if (OrgSrcNames.Length == OrgDstNames.Length)
                {
                    for(int i = 0;i != OrgSrcNames.Length;++ i)
                    {
                        if (OrgSrcNames[i] != OrgDstNames[i])
                        {
                            return false;
                        }
                    }

                    return true;
                }
                else
                {
                    return false;
                }
            }
        }

        private string[] srcNames_;
        private Image targetImageWithArrow_;

        private const int imageSize = 16;

        public int NameTablesLength { get { return srcNames_.Where(x => x != null).Count(); } }

        public IEnumerable<IfMergeSrcDstPair> NameTables
        {
            get
            {
                var list = new List<IfMergeSrcDstPair>();
                {
                    for(int i = 0;i != OrgDstNames.Length;++ i)
                    {
                        if (srcNames_[i] != null)
                        {
                            list.Add(
                                new IfMergeSrcDstPair()
                                {
                                    Src = srcNames_[i],
                                    Dst = OrgDstNames[i]
                                }
                            );
                        }
                    }
                }
                return list;
            }
        }

        public event EventHandler NameTablesChange = null;

        private void RunNameTablesChange()
        {
            if (NameTablesChange != null)
            {
                NameTablesChange(this, EventArgs.Empty);
            }
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public MergeTargetSelectPanel()
        {
            InitializeComponent();
        }

        public void Initialize()
        {
            targetImageWithArrow_ = new Bitmap(imageSize + imageSize, imageSize, System.Drawing.Imaging.PixelFormat.Format32bppArgb);
            using (var g = Graphics.FromImage(targetImageWithArrow_))
            {
                g.DrawImage(Resources.Merge_MergeTargetSelectArrow, 0, 0, imageSize, imageSize);
                g.DrawImage(TargetImage, imageSize, 0, imageSize, imageSize);
            }

            srcNames_ = new string[OrgDstNames.Length];
            SetDefault();

            // マージするオブジェクトの選択
            using(var ub = new UpdateBlock(cmbSrcSelection))
            {
                cmbSrcSelection.AddItem(Strings.MergeTargetSelectPanel_Nothing, null);

                foreach(var srcName in OrgSrcNames)
                {
                    cmbSrcSelection.AddItem(srcName, srcName);
                }
            }

            // マージ対象の選択
            using(var ub = new UpdateBlock(lvwTargets))
            {
                // リストアイテムを予め用意する
                lvwTargets.SetItemCount(OrgDstNames.Length);

                int index = 0;
                foreach (var dstName in OrgDstNames)
                {
                    lvwTargets.Items[index].SubItems[0].Text = dstName;
                    lvwTargets.Items[index].SubItems[0].Tag  = dstName;

                    ++ index;
                }
            }

            UpdateForm();
        }

        private void SetAllNoMerge()
        {
            Array.Clear(srcNames_, 0, srcNames_.Length);

            UpdateForm();
            RunNameTablesChange();
        }

        private void SetDefault()
        {
            for(int i = 0;i != srcNames_.Length;++ i)
            {
                int index = Array.IndexOf(OrgSrcNames, OrgDstNames[i]);
                srcNames_[i] = (index >= 0) ? OrgSrcNames[index] : null;
            }

            UpdateForm();
            RunNameTablesChange();
        }

        private void UpdateForm()
        {
            using (var block = new UIControlEventSuppressBlock())
            {
                bool isEnabled = lvwTargets.SelectedIndex != -1;

                lblSrcSelection.Enabled  = isEnabled;
                cmbSrcSelection.Enabled  = isEnabled;
                btnClearSelected.Enabled = isEnabled;
                btnResetSelected.Enabled = isEnabled;

                if (isEnabled)
                {
                    cmbSrcSelection.SelectedItemData = srcNames_[lvwTargets.SelectedIndex];
                }

                lvwTargets.Invalidate();
            }
        }

        private void cmbSrcSelection_CustomDrawItem(object sender, CustomDrawListControlItemEventArgs e)
        {
            if (e.Item != null)
            {
                var name = e.Item.Data as string;
                if (name != null)
                {
                    e.DrawBackground();
                    e.Graphics.DrawImage(TargetImage, 2 + e.Bounds.X, e.Bounds.Y, imageSize, imageSize);

                    int fontHeight = (int)e.Graphics.MeasureString(name, e.Font, e.Bounds.Width, new StringFormat()).Height;

                    int x = imageSize + 4;
                    int y = e.Bounds.Y + (e.Bounds.Height - fontHeight) / 2;

                    e.Graphics.DrawString(name, e.Font, SystemBrushes.ControlText, x, y);
                    e.DrawFocusRectangle();

                    e.DoDefaultProcess = false;
                }
                else
                {
                    e.DoDefaultProcess = true;
                }
            }
        }

        private void lvwTargets_CustomDrawItem(object sender, CustomDrawListViewItemEventArgs e)
        {
            string name = e.Item.Tag as string;

            switch (e.ColumnIndex)
            {
                // マージ先項目
                case 0:
                {
                    e.SpecificText = name;
                    e.SpecificImage = TargetImage;
                    break;
                }

                // マージ元項目
                case 1:
                {
                    var srcName = srcNames_[e.ItemIndex];

                    if (srcName != null)
                    {
                        e.SpecificText = srcNames_[e.ItemIndex];
                        e.SpecificImage = targetImageWithArrow_;
                    }
                    else
                    {
                        e.SpecificText = Strings.MergeTargetSelectPanel_Nothing;
                    }

                    break;
                }
            }
        }

        private void btnClearAll_Click(object sender, EventArgs e)
        {
            SetAllNoMerge();
        }

        private void btnResetAll_Click(object sender, EventArgs e)
        {
            SetDefault();
        }

        private void lvwTargets_SelectionChanged(object sender, EventArgs e)
        {
            UpdateForm();
            RunNameTablesChange();
        }

        private void cmbSrcSelection_SelectedIndexChanged(object sender, EventArgs e)
        {
            srcNames_[lvwTargets.SelectedIndex] = cmbSrcSelection.SelectedItemData as string;
            RunNameTablesChange();

            UpdateForm();
        }

        private void btnClearSelected_Click(object sender, EventArgs e)
        {
            srcNames_[lvwTargets.SelectedIndex] = null;
            RunNameTablesChange();

            UpdateForm();
        }

        private void btnResetSelected_Click(object sender, EventArgs e)
        {
            int i = lvwTargets.SelectedIndex;

            int index = Array.IndexOf(OrgSrcNames, OrgDstNames[i]);
            srcNames_[i] = (index >= 0) ? OrgSrcNames[i] : null;
            RunNameTablesChange();

            UpdateForm();
        }

        private void lvwTargets_SizeChanged(object sender, EventArgs e)
        {
            lvwTargets.BeginInvokeOrExecute(new Action(() =>
                {
                    // 列幅調整
                    int totalWidth = lvwTargets.ClientSize.Width;
                    if (totalWidth > 0)
                    {
                        clhDst.Width = totalWidth / 2;
                        clhSrc.Width = totalWidth - clhDst.Width;
                    }
                }));
        }
    }
}
