﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;

using App.ConfigData;
using App.Data;
using App.res;
using App.Utility;
using ConfigCommon;

namespace App.Controls
{
    /// <summary>
    /// マテリアルテンプレートダイアログ
    /// </summary>
    public sealed partial class MaterialTemplateDialog : OkCancelDialog
    {
        // テンプレートデータ
        //private readonly MtFile _data;
        // 対象マテリアル（読み込み時専用）
        //private readonly List<Material> _target;

        private TreeNode					materialSamplerNode_	= null;
        private MaterialTemplate.ImportType	samplerImport_			= MaterialTemplate.ImportType.Overwrite;
        public MaterialTemplate.ImportType	SamplerImport
        {
            get
            {
                return samplerImport_;
            }

            private set
            {
                samplerImport_ = value;
                UpdateTreeView();
            }
        }

        private TreeNode					materialShaderNode_			= null;

        public MaterialTemplate source;

        private readonly bool IsImporting = false;

        /// <summary>
        /// コンストラクタ（読み込み時：target != null、書き込み時：target == null）。
        /// </summary>
        public MaterialTemplateDialog(MaterialTemplate source, List<Material> targets)
        {
            InitializeComponent();

            if (targets == null)
            {
                DialogUtility.SetHelp(this, HelpUtility.PageKey.p_material_property_window_a_export_dialog);
            }
            else
            {
                DialogUtility.SetHelp(this, HelpUtility.PageKey.p_material_property_window_a_import_dialog);
            }

            // コントロールの初期化
            tsmSamplerImportTypeOverwrite.Tag		= MaterialTemplate.ImportType.Overwrite;
            tsmSamplerImportTypeAdd.Tag				= MaterialTemplate.ImportType.Add;

            this.source = source;
            StartPosition = FormStartPosition.CenterParent;

            //-----------------------------------------------------------------
            // 共通処理
            //Debug.Assert((template != null) && (template.Data != null));

            //_data = template.Data;
            tbxComment.Text = source.Comment;
            var imageList = new ImageList();
            imageList.Images.Add(App.Properties.Resources.DottedLine);
            imageList.Images.Add(GuiObjectID.Material.ToString(), App.Properties.Resources.GuiObject_Material);
            imageList.Images.Add(GuiObjectID.ShaderParameterAnimation.ToString(), App.Properties.Resources.TreeView_Doc_ShaderParameterAnimation);
            imageList.Images.Add(GuiObjectID.ColorAnimation.ToString(), App.Properties.Resources.TreeView_Doc_ColorAnimation);
            imageList.Images.Add(GuiObjectID.TexturePatternAnimation.ToString(), App.Properties.Resources.TreeView_Doc_TexturePatternAnimation);
            imageList.Images.Add(GuiObjectID.TextureSrtAnimation.ToString(), App.Properties.Resources.TreeView_Doc_TextureSrtAnimation);
            imageList.Images.Add(GuiObjectID.MaterialVisibilityAnimation.ToString(), App.Properties.Resources.TreeView_Doc_MaterialVisibilityAnimation);
            tvwTree.Nodes.Add(CreateNode(source.RootItem, imageList));
            tvwTree.ImageList = imageList;
            tvwTree.ExpandAll();
            tvwTree.Select();

            if (targets != null)
            {
                IsImporting = true;
                Text = res.Strings.MaterialTemplate_LoadTitle;
                lblTree.Text = res.Strings.MaterialTemplate_LoadTreeTitle;
                lblTreeDescription.Text = res.Strings.MaterialTemplate_LoadTreeDescription;

                tbxComment.ReadOnly = true;

                uiListView1.SmallImageList = new ImageList()
                {
                    ImageSize = new Size(16, 16),
                    ColorDepth = ColorDepth.Depth8Bit,
                    TransparentColor = Color.Transparent
                };
                uiListView1.SmallImageList.Images.Add(Document.GetObjectIcon(GuiObjectID.Material));
                foreach (var obj in targets)
                {
                    var item = new ListViewItem(new string[] { obj.Name, obj.OwnerDocument.FileName }, 0);
                    item.Tag = obj;
                    item.Checked = true;
                    uiListView1.Items.Add(item);
                }
            }
            // 書き込み時用
            else
            {
                // 専用テキスト
                IsImporting = false;
                Text = res.Strings.MaterialTemplate_SaveTitle;
                lblTree.Text = res.Strings.MaterialTemplate_SaveTreeTitle;
                lblTreeDescription.Text = res.Strings.MaterialTemplate_SaveTreeDescription;
                uiPanel2.Visible = false;
                uiSplitContainer1.Panel1Collapsed = true;
                uiSplitContainer1.BorderStyle = BorderStyle.None;
            }

            UpdateTreeView();
        }

        public TreeNode CreateNode(MaterialTemplate.Item item, ImageList imageList)
        {
            var node = new TreeNode()
            {
                Checked = item.apply,
                Tag = item,
            };

            switch (item.category)
            {
                case MaterialTemplate.Category.File:
                    node.Text = item.name;
                    switch (item.guiObject.ObjectID)
                    {
                        case GuiObjectID.Texture:
                            Texture texture = (Texture)item.guiObject;
                            string key = "Texture_" + item.name;
                            imageList.Images.Add(key, CreateIconImage(texture));
                            node.ImageKey = key;
                            MaterialTemplatePropertyGridItem.CreateTextureItem(item);
                            break;
                        case GuiObjectID.ShaderParameterAnimation:
                        case GuiObjectID.ColorAnimation:
                        case GuiObjectID.TextureSrtAnimation:
                            node.ImageKey = item.guiObject.ObjectID.ToString();
                            MaterialTemplatePropertyGridItem.CreateShaderParameterAnimationItem(item, source.material);
                            break;
                        case GuiObjectID.TexturePatternAnimation:
                            node.ImageKey = item.guiObject.ObjectID.ToString();
                            MaterialTemplatePropertyGridItem.CreateTexturePatternAnimationItem(item, source.material);
                            break;
                        case GuiObjectID.MaterialVisibilityAnimation:
                            node.ImageKey = item.guiObject.ObjectID.ToString();
                            MaterialTemplatePropertyGridItem.CreateMaterialVisibilityAnimationItem(
                                item,
                                source.material);
                            break;
                        case GuiObjectID.MaterialAnimation:
                            node.ImageKey = item.guiObject.ObjectID.ToString();
                            MaterialTemplatePropertyGridItem.CreateMaterialAnimationItem(item, source.material);
                            break;
                    }
                    break;
                case MaterialTemplate.Category.Material:
                    node.Text = item.name;
                    node.ImageKey = item.guiObject.ObjectID.ToString();
                    MaterialTemplatePropertyGridItem.CreateMaterialItem(item);
                    break;
                case MaterialTemplate.Category.Textures:
                    node.Text = res.Strings.DocumentName_Texture;
                    imageList.Images.Add(item.category.ToString(), App.Properties.Resources.TreeView_Folder_Texture);
                    node.ImageKey = item.category.ToString();
                    break;
                case MaterialTemplate.Category.Animations:
                    node.Text = res.Strings.DocumentName_Animation;
                    imageList.Images.Add(item.category.ToString(), App.Properties.Resources.TreeView_Folder_Animation);
                    node.ImageKey = item.category.ToString();
                    break;
                case MaterialTemplate.Category.Curve:
                    node.Text = !string.IsNullOrEmpty(item.label) ? item.label : item.name;
                    break;
                case MaterialTemplate.Category.Root:
                    node.Text = res.Strings.MaterialTemplate;
                    break;
                case MaterialTemplate.Category.MaterialGeneral:
                    node.Text = res.Strings.ObjectPropertyPanel_Material_General;
                    break;
                case MaterialTemplate.Category.MaterialRenderState:
                    node.Text = Strings.ObjectPropertyPanel_Material_RenderState;
                    break;
                case MaterialTemplate.Category.MaterialSampler:
                    node.Text = res.Strings.ObjectPropertyPanel_Material_Sampler;
                    node.ContextMenuStrip = cmsSamplerImportType;
                    materialSamplerNode_ = node;
                    SamplerImport = item.importType;
                    break;
                case MaterialTemplate.Category.Sampler:
                    node.Text = item.name;
                    MaterialTemplatePropertyGridItem.CreateSamplerItem(item);
                    break;
                case MaterialTemplate.Category.MaterialShader:
                    node.Text = res.Strings.ObjectPropertyPanel_Material_Shader;
                    materialShaderNode_ = node;
                    break;
                case MaterialTemplate.Category.ShadingModel:
                    node.Text = res.Strings.MaterialTemplate_ShaderSetting;
                    MaterialTemplatePropertyGridItem.CreateShaderSettingItem(item);
                    break;
                case MaterialTemplate.Category.ShaderAssignGroup:
                    node.Text = ((MaterialShaderAssign.GroupInfo)item.tag).Label;
                    MaterialTemplatePropertyGridItem.CreateShaderAssignGroupItem(item);
                    break;
                case MaterialTemplate.Category.UserData:
                    node.Text = res.Strings.ObjectPropertyPanel_UserData;
                    break;
                case MaterialTemplate.Category.WorkData:
                    node.Text = res.Strings.MaterialTemplate_WorkData;
                    break;
                case MaterialTemplate.Category.CustomLabel:
                    node.Text = res.Strings.MaterialTemplate_CustomLabel;
                    break;
            }

            if (item.category == MaterialTemplate.Category.Root || item.category == MaterialTemplate.Category.File)
            {
                node.Nodes.AddRange(item.children.Where(IsAddItem).Select(x => CreateNode(x, imageList)).ToArray());
            }
            else
            {
                node.Nodes.AddRange(item.children.Where(IsAddItem).Select(x => CreateNode(x, imageList)).OrderBy(x => x.Name).ToArray());
            }

            node.SelectedImageKey = node.ImageKey;
            // プロパティ表示の設定
            //CreatePropertyGridItem(item.PropertyGridItem);

            return node;
        }

        private bool IsAddItem(MaterialTemplate.Item item)
        {
            // TODO:
            return item.category != MaterialTemplate.Category.MaterialRenderState
                   || ( ApplicationConfig.DefaultValue.RenderStateInfoVisible
                        && App.AppContext.SelectedPlatformPreset.UseNw);
        }

        private Bitmap CreateIconImage(Texture texture)
        {
            if (texture.Is1d)
            {
                var image = new Bitmap(16, 16);
                {
                    using (var g = Graphics.FromImage(image))
                    {
                        //g.DrawImage(texture.ColorThumbnail, 0, 0, 16, 256);		// 高さを大きめに取らないと正しく描画してくれない
                        texture.DrawColorThumbnail(g, new Rectangle(0,0,16,256));
                    }
                }
                return image;
            }
            else
            {
                return texture.GetColorThumbnailCopy();
            }
        }

        private void UpdateTreeView()
        {
            if (materialSamplerNode_ != null)
            {
                materialSamplerNode_.Text =
                    string.Format("{0} [{1}]",
                        res.Strings.ObjectPropertyPanel_Material_Sampler,
                        (SamplerImport == MaterialTemplate.ImportType.Add) ?
                            res.Strings.MaterialTemplate_Add :
                            res.Strings.MaterialTemplate_Replace
                    );
            }

            if (materialShaderNode_ != null)
            {
                materialShaderNode_.Text = res.Strings.ObjectPropertyPanel_Material_Shader;
            }
        }

        public IEnumerable<Material> CheckedMaterials
        {
            get
            {
                foreach (ListViewItem item in uiListView1.Items)
                {
                    if (item.Checked)
                    {
                        yield return (Material)item.Tag;
                    }
                }
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool OnResultOk()
        {
            if (IsImporting)
            {
                var textureNames = new HashSet<string>();
                bool ok = false;
                bool applyShader = false;
                foreach (var node in TreeViewUtility.AllNodes(tvwTree))
                {
                    var item = node.Tag as MaterialTemplate.Item;
                    Debug.Assert(item == null || node.Checked == item.apply);

                    if (node.Checked && item != null && item.apply == true)
                    {
                        switch (item.category)
                        {
                            case MaterialTemplate.Category.Root:
                                break;
                            case MaterialTemplate.Category.Material:
                                break;
                            case MaterialTemplate.Category.Textures:
                                break;
                            case MaterialTemplate.Category.Animations:
                                break;
                            case MaterialTemplate.Category.Undef:
                                break;
                            case MaterialTemplate.Category.MaterialShader:
                            case MaterialTemplate.Category.ShaderAssignGroup:
                            case MaterialTemplate.Category.ShadingModel:
                                ok = true;
                                applyShader = true;
                                break;
                            case MaterialTemplate.Category.MaterialGeneral:
                            case MaterialTemplate.Category.MaterialRenderState:
                            case MaterialTemplate.Category.MaterialSampler:
                            case MaterialTemplate.Category.UserData:
                            case MaterialTemplate.Category.WorkData:
                            case MaterialTemplate.Category.CustomLabel:
                            case MaterialTemplate.Category.Curve:
                                ok = true;
                                break;
                            case MaterialTemplate.Category.Sampler:
                                textureNames.Add(source.material.sampler_array.sampler[item.index].tex_name);
                                ok = true;
                                break;
                            case MaterialTemplate.Category.File:
                                switch (item.guiObject.ObjectID)
                                {
                                    case GuiObjectID.Texture:
                                        {
                                            textureNames.Add(item.guiObject.Name);
                                            break;
                                        }
                                    case GuiObjectID.TexturePatternAnimation:
                                        {
                                            var anim = (TexturePatternAnimation)item.guiObject;
                                            foreach (var pattern in anim.TexPatterns)
                                            {
                                                textureNames.Add(pattern.tex_name);
                                            }
                                            break;
                                        }
                                }
                                ok = true;
                                break;
                            default:
                                break;
                        }
                    }
                }

                if (!ok)
                {
                    UIMessageBox.Warning(res.Strings.MaterialTemplate_InvalidCheckState);
                    return false;
                }

                if (applyShader)
                {
                    var message = source.Validate();
                    if (!string.IsNullOrEmpty(message))
                    {
                        UIMessageBox.Warning(message);
                        return false;
                    }
                }

                // 同名のテクスチャが存在する場合はモデルの追加参照パスの順に解決されるため、同名のテクスチャもインポート可能。
            }

            return true;
        }

        private bool upward = false;
        // チェック変更
        private void tvwTree_AfterCheck(object sender, TreeViewEventArgs e)
        {
            DebugConsole.WriteLine("Checked " + e.Node.Checked);
            Debug.Assert(e.Node.Tag is MaterialTemplate.Item);
            ((MaterialTemplate.Item)(e.Node.Tag)).apply = e.Node.Checked;
            if (upward == false)
            {
                foreach (TreeNode child in e.Node.Nodes)
                {
                    child.Checked = e.Node.Checked;
                }
            }
            upward = true;
            if (e.Node.Parent != null && e.Node.Checked)
            {
                e.Node.Parent.Checked = true;
            }
            else
            {
                upward = false;
            }
        }

        // 選択変更
        private void tvwTree_AfterSelect(object sender, TreeViewEventArgs e)
        {
            Debug.Assert(e.Node.Tag is MaterialTemplate.Item);
            prgProperty.SelectedObject = ((MaterialTemplate.Item)e.Node.Tag).PropertyGridItem;
        }

        // 全て展開
        private void btnExpandAll_Click(object sender, EventArgs e)
        {
            prgProperty.ExpandAllGridItems();
        }

        // 全て折畳
        private void btnCollapseAll_Click(object sender, EventArgs e)
        {
            prgProperty.CollapseAllGridItems();
        }

        // コメント変更
        private void tbxComment_TextChanged(object sender, EventArgs e)
        {
            source.Comment = tbxComment.Text;
        }

        private void uiListView1_ItemChecked(object sender, ItemCheckedEventArgs e)
        {
            if (uiListView1.Items.Count == 0 ||
                CheckedMaterials.Any())
            {
                btnOK.Enabled = true;
            }
            else
            {
                btnOK.Enabled = false;
            }
        }

        private void tsmSamplerImportType_Click(object sender, EventArgs e)
        {
            Debug.Assert(sender is ToolStripMenuItem);

            var item = sender as ToolStripMenuItem;

            Debug.Assert(item.Tag is MaterialTemplate.ImportType);

            SamplerImport = (MaterialTemplate.ImportType)item.Tag;
        }

        private void cmsSamplerImportType_Opening(object sender, CancelEventArgs e)
        {
            tsmSamplerImportTypeOverwrite.Checked	= SamplerImport == MaterialTemplate.ImportType.Overwrite;
            tsmSamplerImportTypeAdd.Checked			= SamplerImport == MaterialTemplate.ImportType.Add;
        }

        private void uiListView1_SizeChanged(object sender, EventArgs e)
        {
            uiListView1.BeginInvokeOrExecute(new Action(() =>
                {
                    clhMaterial.Width = uiListView1.ClientSize.Width - clhModel.Width - 2;
                }));
        }
    }
}
