﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Windows.Forms;
using System.Xml;
using System.Xml.Serialization;
using App.Command;
using App.ConfigData;
using App.PropertyEdit;
using App.Utility;
using ConfigCommon;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;

namespace App.Data
{
    public class MaterialTemplate
    {
        public enum ImportType
        {
            Overwrite,
            Add,
        };

        public string Comment = "";
        public Item RootItem;
        public Material material { private set; get; }

        public const string ext = "f3mt";
        //---------------------------------------------------------------------
        /// <summary>
        /// 書き出しパスの取得
        /// </summary>
        public string GetWritePath()
        {
            //Debug.Assert(this.Data != null);

            using(SaveFileDialog dialog = new SaveFileDialog())
            {
                while(true)
                {
                    dialog.Filter = PathUtility.MakeFileter(res.Strings.DocumentName_f3mt, Enumerable.Repeat(ext, 1));
                    dialog.DefaultExt = ext;
                    dialog.InitialDirectory = ConfigData.ApplicationConfig.Setting.MaterialTemplateDialog.Path;
                    dialog.FileName = material.Name;

                    if (dialog.ShowDialog() == DialogResult.OK)
                    {
                        Application.DoEvents();
                        string writePath = dialog.FileName;

                        // ファイル名に使えない文字がある
                        if (RegexMatch.Check(Path.GetFileName(writePath), "[0-9A-Za-z\\-\\._]+") == false)
                        {
                            Controls.UIMessageBox.Information(res.Strings.IO_InvalidMaterialTemplateFileName, writePath);
                            continue;
                        }

                        // 空ファイル名
                        if (Path.GetFileNameWithoutExtension(writePath) == string.Empty)
                        {
                            Controls.UIMessageBox.Information(res.Strings.IO_MaterialTemplateFilenameIsEmpty);
                            continue;
                        }

                        ConfigData.ApplicationConfig.Setting.MaterialTemplateDialog.Path = Path.GetDirectoryName(writePath);
                        return writePath;
                    }
                    else
                    {
                        return null;
                    }
                }
            }
        }

        /// <summary>
        /// マテリアル設定ファイル書き込み
        /// </summary>
        public bool Write(string filePath, ImportType samplerImportType)
        {
            if (File.Exists(filePath) && !PrePostIO.ExecutePreOpen(new [] { filePath }))
            {
                return false;
            }

            if (!DocumentSaver.PreSaveCommand(new[] { filePath }))
            {
                return false;
            }

            try
            {
                int offset = 0;
                foreach (var item in RootItem.items)
                {
                    if (item.category == Category.Material)
                    {
                        // マテリアルの書き込み
                        List<G3dStream> streams = new List<G3dStream>();
                        var data = material.UpdateData(streams, true);
                        using (var stream = new MemoryStream())
                        {
                            var serializer = new XmlSerializer(typeof(materialType));
                            serializer.Serialize(stream, data);
                            item.data = stream.ToArray().Concat(Enumerable.Repeat<byte>(0, 1)).Concat(IfBinaryWriteUtility.WriteStreamArray(streams)).ToArray();
                        }
                    }
                    else if (item.category == Category.MaterialSampler)
                    {
                        item.importType = samplerImportType;
                    }
                    else if (item.category == Category.File)
                    {
                        var intermediateFile = (IntermediateFileDocument)item.guiObject;
                        nw4f_3difType nw4f_3dif = intermediateFile.Create_nw4f_3difType();
                        // 付属ファイルの書き込み
                        if (G3dPath.IsTextPath(filePath))
                        {
                            if (!G3dStreamUtility.HasStreamArray(nw4f_3dif))
                            {
                                nw4f_3dif.RootElement.stream_array =
                                    G3dStreamUtility.ToStreamArray(intermediateFile.BinaryStreams);
                            }
                            item.data = IfTextFormatter.FormatBytes(nw4f_3dif, null);
                        }
                        else
                        {
                            List<G3dStream> streams = intermediateFile.BinaryStreams;
                            if (G3dStreamUtility.HasStreamArray(nw4f_3dif))
                            {
                                nw4f_3dif.RootElement.stream_array = null;
                            }
                            item.data = IfBinaryFormatter.FormatStream(nw4f_3dif, streams, null);
                        }
                    }

                    if (item.data != null)
                    {
                        item.length = item.data.Length;
                        item.offset = offset;
                        offset += item.length;
                    }
                }

                var header = new MaterialTemplateHeader()
                {
                    Comment = Comment,
                    Item = RootItem,
                };
                using (var stream = new MemoryStream())
                {
                    var serializer = new XmlSerializer(typeof(MaterialTemplateHeader));
                    XmlSerializerNamespaces ns = new XmlSerializerNamespaces(new[] { XmlQualifiedName.Empty });
                    XmlWriterSettings ws = new XmlWriterSettings();
                        ws.Indent = true;
                        ws.OmitXmlDeclaration = true;
                    using (var writer = XmlWriter.Create(stream, ws))
                    {
                        serializer.Serialize(writer, header, ns);
                    }
                    var bytes = stream.ToArray()
                        .Concat(Enumerable.Repeat<byte>(0, 1))
                        .Concat(RootItem.items.Where(x => x.data != null).SelectMany(x => x.data));
                    File.WriteAllBytes(filePath, bytes.ToArray());
                }
            }
            catch (Exception e)
            {
                Debug.Assert(false, e.Message);
                return false;
            }
            DocumentSaver.PostSaveCommand(new[] { filePath });
            return true;
        }

        // ファイルの読み込み
        private void ReadFileItem(Item item, byte[] bytes, int baseOffset, string filePath)
        {
            var sub = new byte[item.length];
            Array.Copy(bytes, baseOffset + item.offset, sub, 0, item.length);
            nw4f_3difType file;
            List<G3dStream> streams = null;
            if (G3dPath.IsTextPath(filePath))
            {
                file = IfTextReadUtility.Read(sub, DocumentManager.XsdBasePath);
            }
            else
            {
                streams = new List<G3dStream>();
                file = IfBinaryReadUtility.Read(streams, sub, DocumentManager.XsdBasePath);
            }

            item.guiObject = DocumentManager.Construct(item.name, file, streams);
        }

        /// <summary>
        /// マテリアル設定ファイル読み込み
        /// </summary>
        public bool Read(string filePath)
        {
            try
            {
                if (!PrePostIO.ExecutePreOpen(new[] { filePath }))
                {
                    return false;
                }

                var bytes = File.ReadAllBytes(filePath);
                int headerSize = Array.IndexOf<byte>(bytes, 0);
                MaterialTemplateHeader header;
                using (var stream = new MemoryStream(bytes, 0, headerSize))
                {
                    var serializer = new XmlSerializer(typeof(MaterialTemplateHeader));
                    header = (MaterialTemplateHeader)serializer.Deserialize(stream);
                    RootItem = header.Item;
                    Comment = header.Comment;

                    // バージョン
                    var comp = StringUtility.ParseVersion(header.Version).Zip(StringUtility.ParseVersion(MaterialTemplateVersion), (x, y) => (x.CompareTo(y)));

                    if (comp.Any(x => x != 0) && comp.First(x => x != 0) > 0)
                    {
                        Controls.UIMessageBox.Error(res.Strings.MaterialTemplate_VersionError, header.Version, MaterialTemplateVersion);
                        return false;
                    }
                }

                int samplerIndex = 0;
                int shaderAssignGroupIndex = 0;
                MaterialShaderAssign.GroupInfo[] shaderAssignGroups = null;

                int baseOffset = headerSize + 1;
                foreach (var item in header.Item.items)
                {
                    if (item.category == Category.Material)
                    {
                        // マテリアルの読み込み
                        int endOfText = Array.IndexOf<byte>(bytes, 0, baseOffset);
                        materialType data = null;
                        int textLength = endOfText - baseOffset;
                        using (var stream = new MemoryStream(bytes, baseOffset, textLength))
                        {
                            var serializer = new XmlSerializer(typeof(materialType));
                            data = (materialType)serializer.Deserialize(stream);
                        }

                        List<G3dStream> streams = null;
                        int binaryLength = item.length - textLength - 1;
                        if (binaryLength > 0)
                        {
                            streams = new List<G3dStream>();
                            var sub = new byte[binaryLength];
                            Array.Copy(bytes, baseOffset + textLength + 1, sub, 0, binaryLength);
                            IfBinaryReadUtility.ReadStreamArray(streams, sub);
                        }

                        // 一時的なモデル
                        var model = new Model(data, streams);
                        material = model.Materials[0];
                        item.guiObject = material;
                    }
                    else if (item.category == Category.File)
                    {
                        // ファイルの読み込み
                        ReadFileItem(item, bytes, baseOffset, filePath);
                    }
                    else if (item.category == Category.Sampler)
                    {
                        Debug.Assert(material != null);

                        item.guiObject	= material;
                        item.index		= samplerIndex;

                        ++ samplerIndex;
                    }
                    else if (item.category == Category.MaterialShader)
                    {
                        shaderAssignGroups = material.MaterialShaderAssign.GetGroups(true).ToArray();
                        var groups = item.children.Where(x => x.category == Category.ShaderAssignGroup).Select(x => x.name).OrderBy(x => x);
                        if (!shaderAssignGroups.Select(x => x.groupName).OrderBy(x => x).SequenceEqual(groups))
                        {
                            item.children.Clear();
                        }
                    }
                    else if (item.category == Category.ShadingModel)
                    {
                        item.guiObject = material;
                    }
                    else if (item.category == Category.ShaderAssignGroup)
                    {
                        Debug.Assert(material != null);
                        Debug.Assert(shaderAssignGroups != null);
                        Debug.Assert(shaderAssignGroupIndex < shaderAssignGroups.Length);

                        item.guiObject = material;
                        item.index = shaderAssignGroupIndex;
                        item.tag = shaderAssignGroups[shaderAssignGroupIndex];

                        ++shaderAssignGroupIndex;
                    }
                    else if (item.category == Category.Animations)
                    {
                        foreach (var fileItem in item.items)
                        {
                            if (fileItem.category == Category.File)
                            {
                                // ファイルの読み込み
                                ReadFileItem(fileItem, bytes, baseOffset, filePath);
                            }
                        }
                    }
                }
            }
            catch (Exception e)
            {
                Debug.Assert(false, e.Message);
                Controls.UIMessageBox.Error(res.Strings.MaterialTemplate_FileReadError, filePath);
                return false;
            }
            return true;
        }

        public static string MaterialTemplateVersion = G3dConstant.Version + ".0";

        /// <summary>
        /// マテリアル設定ファイルのヘッダ
        /// </summary>
        public class MaterialTemplateHeader
        {
            [XmlAttribute]
            public string Version = MaterialTemplateVersion;

            [XmlElement]
            public string Comment;

            public Item Item;
        }

        public class Item
        {
            [XmlAttribute("category")]
            public string categoryAttribute
            {
                get
                {
                    return category.ToString();
                }
                set
                {
                    if (!Enum.TryParse<Category>(value, out category))
                    {
                        category = Category.Undef;
                    }
                }
            }

            [XmlIgnore]
            public Category category;

            [XmlAttribute]
            public string name;

            [XmlAttribute]
            public int offset;

            [XmlAttribute]
            public int length;

            [XmlAttribute]
            public bool apply = true;

            [XmlAttribute]
            public MaterialTemplate.ImportType importType = MaterialTemplate.ImportType.Overwrite;

            [XmlIgnore]
            public byte[] data;

            [XmlIgnore]
            public GuiObject guiObject;

            [XmlIgnore]
            public object PropertyGridItem;

            [XmlIgnore]
            public int index;

            [XmlIgnore]
            public object tag;	// 好きに使っていい

            [XmlIgnore]
            public string label;

            public List<Item> children = new List<Item>();

            public IEnumerable<Item> items
            {
                get
                {
                    return Enumerable.Repeat(this, 1).Concat(children.SelectMany(x => x.items));
                }
            }
        }

        public enum Category
        {
            Root,
            Material,
            MaterialGeneral,
            MaterialRenderState,
            MaterialSampler,
            MaterialShader,
            MaterialOriginalInfo,
            UserData,
            WorkData,
            CustomLabel,
            Textures,
            Animations,
            File,
            Curve,
            Sampler,
            ShaderAssignGroup,
            ShadingModel,
            Undef,
        }

        /// <summary>
        /// エクスポート用のマテリアルの設定
        /// </summary>
        public bool SetSourceMaterial(Material material)
        {
            this.material = material;

            RootItem = new Item()
            {
                category = Category.Root,
            };

            // マテリアルノードの初期化

            var MaterialItem = new Item()
            {
                category = Category.Material,
                name = material.Name,
                guiObject = material,
            };
            RootItem.children.Add(MaterialItem);
            var categories = new Category[] {
                    Category.MaterialGeneral,
                    Category.MaterialRenderState,
                    Category.MaterialSampler,
                    Category.MaterialShader,
                    Category.UserData,
                    Category.WorkData,
                    Category.CustomLabel,
            };
            foreach (var category in categories)
            {
                MaterialItem.children.Add(new Item() { category = category, apply = category != Category.CustomLabel });
            }

            // 個々のサンプラ
            {
                var parentItem = MaterialItem.children.FirstOrDefault(x => x.category == Category.MaterialSampler);
                Debug.Assert(parentItem != null);

                int index = 0;
                foreach (var sampler in material.sampler_array.Items)
                {
                    var item = CreateSamplerItem(sampler, index);
                    Debug.Assert(item != null);
                    parentItem.children.Add(item);

                    ++ index;
                }
            }

            // 個々のシェーダーアサイングループ
            if (material.MaterialShaderAssign.ShadingModel != null)
            {
                var parentItem = MaterialItem.children.FirstOrDefault(x => x.category == Category.MaterialShader);
                Debug.Assert(parentItem != null);

                parentItem.children.Add(new Item() { category = Category.ShadingModel, guiObject = material });
                int index = 0;
                foreach (var info in material.MaterialShaderAssign.GetGroups(true).Where(x => x.group == null || string.IsNullOrEmpty(x.group.Group())))
                {
                    var item = CreateShaderAssignGroupItem(info, index);
                    Debug.Assert(item != null);
                    parentItem.children.Add(item);

                    ++ index;
                }
            }

            HashSet<string> textureNames = new HashSet<string>();

            // アニメーションノードの初期化
            var AnimationsItem = new Item()
            {
                category = Category.Animations
            };
            RootItem.children.Add(AnimationsItem);
            var referenceTextures = new List<IReferTexture>();
            referenceTextures.Add(material.Owner);
            foreach (var animation in material.Referrers.SelectMany(x => x.AllAnimations))
            {
                Item item = null;
                if (animation is ShaderParameterAnimation)
                {
                    item = CreateAnimationItem((ShaderParameterAnimation)animation);
                }
                else if (animation is TexturePatternAnimation)
                {
                    referenceTextures.Add((IReferTexture)animation);
                    item = CreateAnimationItem((TexturePatternAnimation)animation, textureNames);
                }
                else if (animation is MaterialVisibilityAnimation)
                {
                    item = CreateAnimationItem((MaterialVisibilityAnimation)animation);
                }
                else if (animation is MaterialAnimation)
                {
                    referenceTextures.Add((IReferTexture)animation);
                    item = CreateAnimationItem((MaterialAnimation)animation, textureNames);
                }
                if (item != null)
                {
                    AnimationsItem.children.Add(item);
                }
            }

            // テクスチャノードの初期化
            var TexturesItem = new Item()
            {
                category = Category.Textures
            };
            RootItem.children.Add(TexturesItem);
            textureNames.UnionWith(material.sampler_array.sampler.Select(x => x.tex_name));
            var ngTextures = new HashSet<string>();
            foreach (var textureName in textureNames)
            {
                foreach (var texture in DocumentManager.Textures.Where(x => x.Name == textureName))
                {
                    var paths = referenceTextures.Select(x => { string y; x.ReferenceTexturePaths.TryGetValue(texture.Name, out y); return y; })
                        .Where(x => x != null).Select(x => x.ToLower()).Distinct().ToArray();
                    if (paths.Length > 1)
                    {
                        ngTextures.Add(texture.Name);
                    }
                    else if (paths.Length == 1 && string.Compare(texture.FilePath, paths[0], true) == 0)
                    {
                        TexturesItem.children.Add(new Item() { category = Category.File, name = texture.FileName, guiObject = texture });
                    }
                }
            }

            if (ngTextures.Any())
            {
                using (var dialog = new App.Controls.OkListBoxDialog())
                {
                    dialog.lblDescription.Text = res.Strings.MaterialTemplate_TextureReference;
                    dialog.Text = res.Strings.MaterialTemplate;
                    foreach (var ng in ngTextures.OrderBy(x => x))
                    {
                        dialog.AddLine(ng);
                    }

                    dialog.ShowDialog();
                }

                return false;
            }

            return true;
        }

        private Item CreateSamplerItem(samplerType sampler, int index)
        {
            return
                new Item()
                {
                    category	= Category.Sampler,
                    name		= sampler.name,
                    guiObject	= material,
                    index		= index
                };
        }

        private Item CreateShaderAssignGroupItem(MaterialShaderAssign.GroupInfo info, int index)
        {
            return
                new Item()
                {
                    category	= Category.ShaderAssignGroup,
                    name		= info.groupName,
                    guiObject	= material,
                    index		= index,
                    tag			= info
                };
        }

        private Item CreateAnimationItem(ShaderParameterAnimation animation)
        {

            var item = new Item() { category = Category.File, name = animation.FileName, guiObject = animation };
            var curves = animation.ShaderParamAnims.Where(x => x.mat_name == material.Name).SelectMany(x => x.ParamAnims);

            var invisibleParams = new HashSet<string>();
            if (material.MaterialShaderAssign.ShadingModel != null)
            {
                foreach (var uniform in material.MaterialShaderAssign.ShadingModel.MaterialUniforms())
                {
                    if (!uniform.IsAnimVisible())
                    {
                        invisibleParams.Add(uniform.id);
                    }
                }
            }

            foreach (var curve in curves)
            {
                if (invisibleParams.Contains(curve.id))
                {
                    continue;
                }

                if (curve.ParamAnimTargets.Any(x => x.ExportType != CurveExportType.Ignored && x.ExportType != CurveExportType.Dependent))
                {
                    var curveItem = new Item() { category = Category.Curve, name = curve.id };
                    item.children.Add(curveItem);
                }
            }

            if (item.children.Any())
            {
                return item;
            }

            return null;
        }

        private Item CreateAnimationItem(TexturePatternAnimation animation, HashSet<string> textureNames)
        {
            var item = new Item() { category = Category.File, name = animation.FileName, guiObject = animation };
            var curves = animation.TexPatternMatAnims.Where(x => x.mat_name == material.Name).SelectMany(x => x.PatternAnimTargets);

            foreach (var curve in curves)
            {
                if (curve.KeyFrames.Any())
                {
                    var curveItem = new Item() { category = Category.Curve, name = curve.sampler_name };
                    item.children.Add(curveItem);
                    foreach (var value in curve.KeyFrames.Select(x => (int)x.Value))
                    {
                        if (0 <= value && value < animation.TexPatterns.Count)
                        {
                            textureNames.Add(animation.TexPatterns[value].tex_name);
                        }
                    }
                }
            }

            if (item.children.Any())
            {
                return item;
            }

            return null;
        }

        private Item CreateAnimationItem(MaterialVisibilityAnimation animation)
        {
            var item = new Item() { category = Category.File, name = animation.FileName, guiObject = animation };
            var curves = animation.MaterialVisibilityMatAnims.Where(x => x.mat_name == material.Name && x.KeyFrames.Any());

            if (curves.Any())
            {
                return item;
            }

            return null;
        }

        private Item CreateAnimationItem(MaterialAnimation animation, HashSet<string> textureNames)
        {
            var item = new Item() { category = Category.File, name = animation.FileName, guiObject = animation };
            var frames = 0;

            foreach (var perMaterialAnim in animation.PerMaterialAnims)
            {
                foreach (var paramAnim in perMaterialAnim.ParamAnims)
                {
                    foreach (var target in paramAnim.ParamAnimTargets)
                    {
                        frames += target.KeyFrames.Count;
                    }
                }

                foreach (var patternAnim in perMaterialAnim.PatternAnimTargets)
                {
                    foreach (var target in patternAnim.KeyFrames)
                    {
                        if ((int)target.Value < animation.TexPatterns.Count)
                        {
                            textureNames.Add(animation.TexPatterns[(int)target.Value].tex_name);
                        }
                    }
                }

                if (perMaterialAnim.MaterialVisibilityMatAnim != null)
                {
                    frames += perMaterialAnim.MaterialVisibilityMatAnim.KeyFrames.Count;
                }
            }

            if (frames > 0)
            {
                return item;
            }

            return null;
        }

        //---------------------------------------------------------------------
        // 読み込み
        //---------------------------------------------------------------------
        /// <summary>
        /// 読み込みパスの取得
        /// </summary>
        public string GetReadPath()
        {
            using(OpenFileDialog dialog = new OpenFileDialog())
            {
                dialog.Filter = PathUtility.MakeFileter(res.Strings.DocumentName_f3mt, Enumerable.Repeat(ext, 1));//StringResource.Get("IO_FILEFILTER_C3MT");
                dialog.DefaultExt = ext;
                dialog.InitialDirectory = ConfigData.ApplicationConfig.Setting.MaterialTemplateDialog.Path;

                if (dialog.ShowDialog() == DialogResult.OK)
                {
                    Application.DoEvents();
                    string readPath = dialog.FileName;
                    ConfigData.ApplicationConfig.Setting.MaterialTemplateDialog.Path = Path.GetDirectoryName(readPath);
                    return readPath;
                }
                return null;
            }
        }

        /// <summary>
        /// マテリアル設定の適用
        /// </summary>
        public void Apply(List<Material> targets, ImportType samplerImport)
        {
            var targetGroup = new GuiObjectGroup();
            foreach (var target in targets)
            {
                targetGroup.Add(target);
            }

            var texturePaths = new Dictionary<string, string>();
            foreach (var texture in DocumentManager.Textures)
            {
                // 同名のテクスチャでもモデルの追加参照パス設定で別々のディレクトリに配置できるので、プロジェクトパスで判別する。
                if (!string.IsNullOrEmpty(texture.ProjectPath))
                {
                    // 重複があるものは使われないので気にしない
                    texturePaths[texture.ProjectPath] = texture.FilePath;
                }
            }

            foreach (var item in RootItem.items.Where(x => x.category == Category.File && x.guiObject is Texture && x.apply))
            {
                Texture texture = (Texture)item.guiObject;
                if (!string.IsNullOrEmpty(texture.ProjectPath))
                {
                    if (!texturePaths.ContainsKey(texture.ProjectPath))
                    {
                        texturePaths[texture.ProjectPath] = string.Empty;
                    }
                }
            }

            using (var block = new App.AppContext.PropertyChangedSuppressBlock())
            {
                var commandSet = new EditCommandSet();
                var updatedAnimations = new List<AnimationDocument>();
                var createdAnimations = new List<AnimationDocument>();
                var updatedTextures = new List<Texture>();
                var createdTextures = new List<Texture>();
                HashSet<Model> notLoaded = new HashSet<Model>(targets.OfType<Material>().SelectMany(x => x.Referrers).Distinct().Where(x => !x.HioLoaded));
                var textureReferenceDocs = new List<IMaterialOwner>();
                using (var vdsb = new Viewer.ViewerDrawSuppressBlock(Viewer.ViewerDrawSuppressBlock.DiscardAllMessages))
                {
                    MaterialSamplerPage.CopyData samplerPageCopyData = null;
                    foreach (var item in RootItem.items.Where(x => x.items.Any(y => y.apply)))
                    {
                        switch (item.category)
                        {
                            case Category.MaterialGeneral:
                                {
                                    var data = MaterialGeneralPage.Copy(material);
                                    commandSet.Add(MaterialGeneralPage.Paste(targetGroup, data));
                                }
                                break;
                            case Category.MaterialRenderState:
                                {
                                    var data = MaterialRenderStatePage.Copy(material);
                                    commandSet.Add(MaterialRenderStatePage.Paste(targetGroup, data, false));
                                }
                                break;
                            case Category.MaterialSampler:
                                {
                                    var data = (MaterialSamplerPage.CopyData)MaterialSamplerPage.Copy(material, samplerImport == ImportType.Add);

                                    Debug.Assert(data is MaterialSamplerPage.CopyData);
                                    switch (samplerImport)
                                    {
                                        case ImportType.Overwrite:
                                            {
                                                commandSet.Add(CreateSamplerOverwriteCommand(targetGroup, data as MaterialSamplerPage.CopyData, item.children, texturePaths));
                                                if (ApplicationConfig.Preset.FollowDccSamplerNameRule)
                                                {
                                                    samplerPageCopyData = (MaterialSamplerPage.CopyData)data;
                                                }
                                                break;
                                            }
                                        case ImportType.Add: commandSet.Add(CreateSamplerAddCommand(targetGroup, data as MaterialSamplerPage.CopyData, item.children)); break;
                                        default: Debug.Assert(false); break;

                                    }

                                    textureReferenceDocs.AddRange(targetGroup.GetObjects(GuiObjectID.Material).OfType<Material>().Select(x => x.Owner).Distinct());

                                    foreach (var child in item.children)
                                    {
                                        if (child.apply)
                                        {
                                            var sampler = data.sampler_array.sampler.FirstOrDefault(x => x.name == child.name);
                                            if (sampler != null)
                                            {
                                                string path;
                                                if (texturePaths.TryGetValue(sampler.tex_name, out path))
                                                {
                                                    var models = textureReferenceDocs.OfType<Model>().ToArray();
                                                    var materialDocs = textureReferenceDocs.OfType<SeparateMaterial>().ToArray();
                                                    if (models.Any())
                                                    {
                                                        commandSet.Add(MaterialSamplerPage.CreateEditCommand_ReferenceTexturePath(new GuiObjectGroup(models), GuiObjectID.Model, sampler.tex_name, path).Execute());
                                                    }
                                                    if (materialDocs.Any())
                                                    {
                                                        commandSet.Add(MaterialSamplerPage.CreateEditCommand_ReferenceTexturePath(new GuiObjectGroup(materialDocs), GuiObjectID.SeparateMaterial, sampler.tex_name, path).Execute());
                                                    }
                                                }
                                            }
                                        }
                                    }
                                }
                                break;
                            case Category.MaterialShader:
                                {
                                    if (item.children.Count == 0)
                                    {
                                        var data = MaterialShaderPage.Copy(material);

                                        // サンプラ名の変更
                                        if (samplerPageCopyData != null)
                                        {
                                            foreach (var assign in ((MaterialShaderPage.CopyData)data).itemsCopyData.SamplerAssigns)
                                            {
                                                string converted;
                                                if (samplerPageCopyData.convertResult.TryGetValue(assign.Item2.sampler_name, out converted))
                                                {
                                                    assign.Item2.sampler_name = converted;
                                                }
                                            }
                                        }

                                        commandSet.Add(MaterialShaderPage.Paste(targetGroup, data, false));
                                    }
                                }
                                break;
                            case Category.ShadingModel:
                                commandSet.Add(CraeteShaderSettingCommand(targetGroup));
                                break;
                            case Category.ShaderAssignGroup:
                                commandSet.Add(CreateShaderAssignGroupCommand(targetGroup, item.name, false, samplerPageCopyData));
                                break;
                            case Category.UserData:
                                {
                                    var data = UserDataPage.Copy(material);
                                    commandSet.Add(UserDataPage.Paste(targetGroup, data));
                                }
                                break;
                            case Category.WorkData:
                                {
                                    commandSet.Add(ObjectPropertyPanel.CreateEditColorEditCommand(targetGroup, material.EditColor).Execute());
                                    commandSet.Add(ObjectPropertyPanel.CreateEditCommentEditCommand(targetGroup, material.Comment).Execute());
                                    commandSet.Add(ObjectPropertyPanel.CreateEditLabelEditCommand(targetGroup, material.Label).Execute());
                                }
                                break;
                            case Category.CustomLabel:
                                {
                                    var data = material.CreateNw3de_CustomUI();
                                    commandSet.Add(CreateCustomUICommand(targetGroup, data).Execute());
                                }
                                break;
                            case Category.File:
                                switch (item.guiObject.ObjectID)
                                {
                                    case GuiObjectID.Texture:
                                        {
                                            var newTexture = (Texture)item.guiObject;
                                            var old = targetGroup.Active.OwnerDocument.ReferenceDocuments.FirstOrDefault(x => x.Name == newTexture.Name);

                                            // モデルの参照先ドキュメントに存在しない場合は、パスが存在しない同名のドキュメントを上書き対象にする
                                            if (old == null)
                                            {
                                                old = DocumentManager.Textures.FirstOrDefault(x => x.Name == newTexture.Name && string.IsNullOrEmpty(x.ProjectPath));
                                            }

                                            if (old != null)
                                            {
                                                var dataChange = TextureGeneralPage.CreateTextureDataChangeCommand(new GuiObjectGroup(old), newTexture.Data);
                                                var streamChange = TextureGeneralPage.CreateTextureStreamChangeCommand(new GuiObjectGroup(old), newTexture.BinaryStreams);
                                                commandSet.Add(dataChange.Execute());
                                                commandSet.Add(streamChange.Execute());
                                                updatedTextures.Add((Texture)old);
                                            }
                                            else
                                            {
                                                commandSet.Add(DocumentManager.CreateAddOrRemoveDocumentCommand(Enumerable.Repeat(newTexture, 1), true).Execute());
                                                createdTextures.Add(newTexture);
                                            }
                                        }
                                        break;
                                    case GuiObjectID.ShaderParameterAnimation:
                                    case GuiObjectID.TextureSrtAnimation:
                                    case GuiObjectID.ColorAnimation:
                                        {
                                            ApplyShaderParameterAnimation(item, commandSet, targets, createdAnimations, updatedAnimations);
                                            break;
                                        }
                                    case GuiObjectID.TexturePatternAnimation:
                                        {
                                            ApplyTexturePatternAnimation(item, commandSet, targets, createdAnimations, updatedAnimations, texturePaths);

                                            break;
                                        }
                                    case GuiObjectID.MaterialVisibilityAnimation:
                                        {
                                            ApplyMaterialVisibilityAnimation(item, commandSet, targets, createdAnimations, updatedAnimations);
                                            break;
                                        }
                                    case GuiObjectID.MaterialAnimation:
                                        {
                                            ApplyMaterialAnimation(item, commandSet, targets, createdAnimations, updatedAnimations);
                                            break;
                                        }
                                }
                                break;
                        }
                    }

                    // テクスチャが新規作成された分についてはここで参照を解決する
                    // ただし、新規のパターンについては ApplyTexturePatternAnimation 内で既に行われている
                    var unresolved = from animation in updatedAnimations.OfType<TexturePatternAnimation>()
                                     from tex in createdTextures
                                     where animation.TexPatterns.Any(x => x.tex_name == tex.Name)
                                     select new { animation, tex };

                    foreach (var pair in unresolved)
                    {
                        commandSet.Add(TexturePatternAnimationPatternPage.CreateEditCommand_ReferenceTexturePath(
                            new GuiObjectGroup(pair.animation),
                            pair.tex.Name,
                            pair.tex.FilePath).Execute());
                    }


                    // 参照の削除はテクスチャの追加などが行われてから
                    if (textureReferenceDocs.Any())
                    {
                        var models = textureReferenceDocs.OfType<Model>().ToArray();
                        var materialDocs = textureReferenceDocs.OfType<SeparateMaterial>().ToArray();
                        if (models.Any())
                        {
                            commandSet.Add(MaterialSamplerPage.CreateEditCommand_RemoveTexturePaths(new GuiObjectGroup(models), GuiObjectID.Model).Execute());
                        }
                        if (materialDocs.Any())
                        {
                            commandSet.Add(MaterialSamplerPage.CreateEditCommand_RemoveTexturePaths(new GuiObjectGroup(materialDocs), GuiObjectID.SeparateMaterial).Execute());
                        }
                    }

                    // コマンドセットにもメッセージ抑制のデリゲートを設定する。
                    commandSet.SetViewerDrawSuppressBlockDelegate(Viewer.ViewerDrawSuppressBlock.DiscardAllMessages);
                }

                commandSet.Reverse();

                var commandSet2 = new EditCommandSet();
                bool undo = false;
                commandSet2.Add(commandSet);
                EventHandler postEdit = (s, e) =>
                    {
                        var referencingAnimations = DocumentManager.Animations.Where(
                            x => x.ReferenceDocuments.Intersect(updatedTextures.Concat(createdTextures)).Any());

                        var referencingModels = DocumentManager.Models.Where(
                            x => x.ReferenceDocuments.Intersect(updatedTextures.Concat(createdTextures)).Any());
                        if (!undo)
                        {
                            var animations = createdAnimations.Concat(updatedAnimations).Concat(referencingAnimations).Distinct();
                            foreach (var animation in animations)
                            {
                                Viewer.LoadOrReloadAnimation.Send(animation);
                            }

                            var models = Model.SortByPreviewDepth(targets.OfType<Material>().SelectMany(x => x.Referrers).Concat(referencingModels).Distinct());
                            foreach (var model in models)
                            {
                                Viewer.LoadOrReloadModel.Send(model);
                                Viewer.ViewerUtility.SendAnimationSet(model);

                                if (notLoaded.Contains(model))
                                {
                                    // プレビュー情報を送る
                                    model.SendEditBoneBind();
                                    model.SendEditModelLayout(false, sendChildren: true);
                                }
                            }
                        }
                        else
                        {
                            foreach (var animation in createdAnimations)
                            {
                                Viewer.Close.Send(animation);
                            }

                            var models = targets.OfType<Material>().SelectMany(x => x.Referrers).Concat(referencingModels).Distinct();
                            foreach (var model in models)
                            {
                                Viewer.LoadOrReloadModel.Send(model);
                                Viewer.ViewerUtility.SendAnimationSet(model);
                            }

                            var animations = updatedAnimations.Concat(referencingAnimations).Distinct();
                            foreach (var animation in animations)
                            {
                                Viewer.LoadOrReloadAnimation.Send(animation);
                            }
                        }
                        undo = !undo;
                    };
                postEdit(null, null);
                commandSet2.OnPostEdit += postEdit;
                TheApp.CommandManager.Add(commandSet2);
            }
        }

        private ICommand CreateSamplerOverwriteCommand(GuiObjectGroup targets, MaterialSamplerPage.CopyData pasteObject, List<Item> samplerItems, Dictionary<string, string> texturePaths)
        {
            // 仕様
            //
            // 各サンプラごとに適応させる・させないを選択
            // http://www-sdd.zelda.nintendo.co.jp/project/nintendoware3/kagemai/html/user.cgi?project=nw3_3de&action=view_report&id=989#4
            if (ApplicationConfig.Preset.FollowDccSamplerNameRule)
            {
                pasteObject.convertResult = pasteObject.convertResult.Where(x => samplerItems.Any(y => y.apply && y.name == x.Key)).ToDictionary(x => x.Key, x => x.Value);
                pasteObject.sampler_array.sampler = pasteObject.sampler_array.sampler.Where(x => samplerItems.Any(y => y.apply && pasteObject.convertResult[y.name] == x.name)).ToArray();
                pasteObject.texturePaths = pasteObject.sampler_array.sampler.Select(x => x.tex_name).Where(x => !string.IsNullOrEmpty(x)).Distinct()
                    .ToDictionary(x => x, x => { string y; texturePaths.TryGetValue(x, out y); return y; });

                return MaterialSamplerPage.Paste(targets, pasteObject, false);
            }
            else
            {
                var targetObjects = targets.GetObjects(GuiObjectID.Material);
                int targetCount = targetObjects.Count;
                var srcData = ObjectUtility.MultipleClone(pasteObject, targetCount);

                for (int i = 0; i != targetCount; ++i)
                {
                    Debug.Assert(targetObjects[i] is Material);
                    var targetMaterial = targetObjects[i] as Material;

                    // 操作しやすいように List にする
                    var targetSamplers = new List<samplerType>(ObjectUtility.Clone(targetMaterial.sampler_array.sampler));

                    var srcSamplers = srcData[i].sampler_array.Items.ToList();

                    // 各サンプラごとに適応させる・させないを選択
                    foreach (var sampler in samplerItems.Where(x => x.apply == false))
                    {
                        srcSamplers.RemoveAll(x => x.name == sampler.name);
                    }

                    // もとに戻す
                    srcData[i].sampler_array.Items = srcSamplers.ToArray();
                }

                return CreateSamplerCommandInternal(targets, srcData);
            }
        }

        private ICommand CreateSamplerAddCommand(GuiObjectGroup targets, MaterialSamplerPage.CopyData pasteObject, List<Item> samplerItems)
        {
            // 仕様
            //
            // name が同じ場合は上書きし、違う場合は追加
            // http://www-sdd.zelda.nintendo.co.jp/project/nintendoware3/kagemai/html/user.cgi?project=nw3_3de&action=view_report&id=989#1
            //
            // 各サンプラごとに適応させる・させないを選択
            // http://www-sdd.zelda.nintendo.co.jp/project/nintendoware3/kagemai/html/user.cgi?project=nw3_3de&action=view_report&id=989#4
            if (ApplicationConfig.Preset.FollowDccSamplerNameRule)
            {
                var targetObjects = targets.GetObjects(GuiObjectID.Material);
                int targetCount = targetObjects.Count;
                var sourceSamplers = pasteObject.sampler_array.sampler.Where(x => samplerItems.Any(y => y.apply && y.name == x.name)).ToArray();
                var targetSamplerLists = new List<samplerType[]>();
                for (int i = 0; i < targetCount; i++)
                {
                    var targetMaterial = (Material)targetObjects[i];
                    var targetSamplers = targetMaterial.sampler_array.sampler.ToList();

                    // 最初の変換候補は外しておく
                    var sourceSamplerNames = new HashSet<string>(sourceSamplers.Where(x => !x.name.StartsWith("_") || !targetSamplers.Any(y => y.name == x.name && y.hint == x.hint))
                        .Select(x => x.name.TrimStart(new[] { '_' })));
                    foreach (var sampler in sourceSamplers)
                    {
                        MaterialSamplerPage.InsertOrOverwriteSampler(targetSamplers, ObjectUtility.Clone(sampler), -1, sourceSamplerNames);
                    }
                    targetSamplerLists.Add(targetSamplers.ToArray());
                }
                return Material.ExecuteEditCommand_SamplerArray(targets, targetSamplerLists, false);
            }
            else
            {
                var targetObjects = targets.GetObjects(GuiObjectID.Material);
                int targetCount = targetObjects.Count;
                var srcData = ObjectUtility.MultipleClone(pasteObject, targetCount);

                for (int i = 0; i != targetCount; ++i)
                {
                    Debug.Assert(targetObjects[i] is Material);
                    var targetMaterial = targetObjects[i] as Material;

                    // 操作しやすいように List にする
                    var targetSamplers = new List<samplerType>(ObjectUtility.Clone(targetMaterial.sampler_array.sampler));

                    var srcSamplers = srcData[i].sampler_array.Items.ToDictionary(x => x.name);

                    // 各サンプラごとに適応させる・させないを選択
                    foreach (var sampler in samplerItems.Where(x => x.apply == false))
                    {
                        srcSamplers.Remove(sampler.name);
                    }

                    for (int j = 0; j != targetSamplers.Count; ++j)
                    {
                        // ソースから同じ同名のサンプラを探して、見つかれば上書きする
                        var targetSamplerName = targetSamplers[j].name;
                        samplerType srcSampler = null;

                        if (srcSamplers.TryGetValue(targetSamplers[j].name, out srcSampler))
                        {
                            targetSamplers[j] = srcSampler;
                            srcSamplers.Remove(targetSamplerName);	// 処理したので除く
                        }
                    }

                    // 残ったもの（＝同名のサンプラがないもの）を追加する
                    targetSamplers.AddRange(srcSamplers.Values);

                    // もとに戻す
                    srcData[i].sampler_array.Items = targetSamplers.ToArray();
                }

                return CreateSamplerCommandInternal(targets, srcData);
            }
        }

        private ICommand CreateSamplerCommandInternal(GuiObjectGroup targets, IEnumerable<MaterialSamplerPage.CopyData> srcData)
        {
            var commandSet = new EditCommandSet();
            {
                commandSet.Add((new GeneralGroupReferenceEditCommand<samplerType[]>(
                    targets,
                    GuiObjectID.Material,
                    srcData.Select(x => x.sampler_array.sampler),
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var material = (target as Material);

                        swap = ObjectUtility.Clone(material.sampler_array.sampler);
                        material.sampler_array.sampler = (samplerType[])data;
                    },
                    createEventArgsDelegate: (x, y) => new DocumentPropertyChangedSamplerCountArgs(x, y)
                )).Execute());

                // バインドを更新
                commandSet.Add(DocumentManager.CreateAnimationUpdateBindCommand(
                    targets.Objects.OfType<Material>().SelectMany(x => x.Referrers).Distinct().SelectMany(x => x.AllAnimations).Distinct()
                ).Execute());

                commandSet.Reverse();
            }

            return commandSet;
        }

        private ICommand CreateCustomUICommand(GuiObjectGroup targets, App.Data.ToolData.nw3de_CustomUI customUI)
        {
            return new GeneralGroupReferenceEditCommand<CustomUI>(
                targets,
                GuiObjectID.Material,
                targets.GetObjects(GuiObjectID.Material).Select(x => ToolData.convertToCustomUI(customUI)),
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    var material = target as Material;
                    swap = material.CustomUI;
                    material.CustomUI = (CustomUI)data;
                },
                createEventArgsDelegate: (x, y) => new DocumentPropertyChangedLabelArgs(x, y));
        }

        private ICommand CraeteShaderSettingCommand(GuiObjectGroup targets)
        {
            var commandSet = new EditCommandSet();
            commandSet.Add(MaterialShaderPage.CreateEditCommand_ShaderPool(targets).Execute());

            var materials = targets.Objects.OfType<Material>().ToArray();
            string shaderArchive = material.MaterialShaderAssign.ShaderDefinitionFileName;
            string shaderName = material.MaterialShaderAssign.ShaderName;

            var invisibleSamplerAssigns = new HashSet<string>(material.MaterialShaderAssign.ShadingModel.Samplers().Where(x => !x.SamplerVisible()).Select(x => x.id));
            var invisibleParams = new HashSet<string>(material.MaterialShaderAssign.ShadingModel.MaterialUniforms().Where(x => !x.IsParamVisible()).Select(x => x.id));
            var invisibleOptions = new HashSet<string>(material.MaterialShaderAssign.ShadingModel.Options().Where(x => !x.OptionVisible()).Select(x => x.id));
            var invisibleAttribAssigns = new HashSet<string>(material.MaterialShaderAssign.ShadingModel.Attributes().Where(x => !x.Editable()).Select(x => x.id));
            var assigns = materials.Select(x =>
            {
                bool sameShadingModel = x.MaterialShaderAssign.ShaderDefinitionFileName == shaderArchive && x.MaterialShaderAssign.ShaderName == shaderName;
                return new MaterialShaderAssign(shaderArchive, shaderName)
                {
                    ShaderOptions = sameShadingModel?
                        ObjectUtility.Clone(x.MaterialShaderAssign.ShaderOptions) :
                        ObjectUtility.Clone(x.MaterialShaderAssign.ShaderOptions.Where(y => !invisibleOptions.Contains(y.id)).ToList()),
                    SamplerAssigns = sameShadingModel ?
                        ObjectUtility.Clone(x.MaterialShaderAssign.SamplerAssigns) :
                        ObjectUtility.Clone(x.MaterialShaderAssign.SamplerAssigns.Where(y => !invisibleSamplerAssigns.Contains(y.id)).ToList()),
                    ShaderParams = sameShadingModel ?
                        ObjectUtility.Clone(x.MaterialShaderAssign.ShaderParams) :
                        ObjectUtility.Clone(x.MaterialShaderAssign.ShaderParams.Where(y => !invisibleParams.Contains(y.id)).ToList()), // 一致しない場合、見えないパラメーターは引き継がない
                    AttribAssigns = sameShadingModel ?
                        ObjectUtility.Clone(x.MaterialShaderAssign.AttribAssigns) :
                        ObjectUtility.Clone(x.MaterialShaderAssign.AttribAssigns.Where(y => !invisibleAttribAssigns.Contains(y.id)).ToList()),
                    RenderInfos = ObjectUtility.Clone(x.MaterialShaderAssign.RenderInfos),
                };
            }).ToArray();

            commandSet.Add(ShaderAssignUtility.CreateEditCommand_ShaderAssign(targets, assigns, false).Execute());

            var inconsistentMaterials = targets.GetObjects(GuiObjectID.Material).OfType<Material>().Where(x => !ShaderAssignUtility.IsConsistentWithDefinition(x)).ToArray();
            if (inconsistentMaterials.Any())
            {
                commandSet.Add(ShaderAssignUtility.ExecuteFixParameters(inconsistentMaterials, useHint:true, reload:false));
            }

            // バインドを更新
            commandSet.Add(DocumentManager.CreateAnimationUpdateBindCommand(
                targets.Objects.OfType<Material>().SelectMany(x => x.Referrers).Distinct().SelectMany(x => x.AllAnimations).Distinct()
            ).Execute());

            commandSet.Reverse();
            return commandSet;
        }

        private ICommand CreateShaderAssignGroupCommand(GuiObjectGroup targets, string group_name, bool reload, MaterialSamplerPage.CopyData samplerPage)
        {
            var copyData = PropertyEdit.ShaderParamControls.ShaderParamControlGroup.CreateGroupCopyData(
                material,
                group_name,
                true,
                true);

            // 名前を変換
            if (samplerPage != null)
            {
                foreach (var item in copyData.SamplerAssigns)
                {
                    string converted;
                    if (samplerPage.convertResult.TryGetValue(item.Item2.sampler_name, out converted))
                    {
                        item.Item2.sampler_name = converted;
                    }
                }
            }

            var command = PropertyEdit.ShaderParamControls.ShaderParamControlGroup.CreateItemsPasteCommand(targets, copyData, group_name, false, null, true, true, sendViewer:reload, checkVisible:true);
            if (command != null)
            {
                return command.Execute();
            }
            return null;
        }

        // マテリアルアニメーションの適用
        private void ApplyMaterialAnimation(Item item, EditCommandSet commandSet, List<Material> targets, List<AnimationDocument> createdAnimations, List<AnimationDocument> updatedAnimations)
        {
            var animation = (MaterialAnimation)item.guiObject;

            var sourceCurves = animation.PerMaterialAnims.FirstOrDefault(x => x.mat_name == material.Name);

            var pathNotSpecifiedAnimation = (MaterialAnimation)DocumentManager.Animations.FirstOrDefault(x =>
                x.ObjectID == animation.ObjectID && x.Name == animation.Name && string.IsNullOrEmpty(x.FileLocation));

            var materialReferrers = from material in targets
                                  group material by material.Referrers;

            var updated = new HashSet<AnimationDocument>();
            AnimationDocument createdAnimation = null;
            foreach (var group in materialReferrers)
            {
                foreach (var model in group.Key)
                {
                    var targetDocument = (MaterialAnimation)model.AllAnimations.FirstOrDefault(x => x.ObjectID == animation.ObjectID && x.Name == animation.Name);

                    if (targetDocument == null)
                    {
                        if (pathNotSpecifiedAnimation == null)
                        {
                            // アニメーションドキュメント作成
                            targetDocument = new MaterialAnimation(
                                new material_animType()
                                {
                                    material_anim_info = ObjectUtility.Clone(animation.Data.material_anim_info),
                                },
                                new List<G3dStream>())
                            {
                                Name = animation.Name,
                                FileDotExt = animation.FileDotExt,
                            };
                            targetDocument.UpdateSavedData();
                            commandSet.Add(DocumentManager.CreateAddOrRemoveDocumentCommand(Enumerable.Repeat(targetDocument, 1), true).Execute());
                            pathNotSpecifiedAnimation = targetDocument;
                            createdAnimation = targetDocument;
                            createdAnimations.Add(targetDocument);
                        }
                        else
                        {
                            targetDocument = pathNotSpecifiedAnimation;
                        }

                        // バインド
                        BindAnimation(targetDocument, group.ToList(), commandSet);
                    }

                    if (updated.Add(targetDocument))
                    {
                        if (createdAnimation != targetDocument)
                        {
                            updatedAnimations.Add(targetDocument);
                        }

                        // 編集情報
                        var targetGroup = new GuiObjectGroup(targetDocument);
                        commandSet.Add(ObjectPropertyPanel.CreateEditColorEditCommand(targetGroup, material.EditColor).Execute());
                        commandSet.Add(ObjectPropertyPanel.CreateEditCommentEditCommand(targetGroup, material.Comment).Execute());
                        commandSet.Add(ObjectPropertyPanel.CreateEditLabelEditCommand(targetGroup, material.Label).Execute());

                        // 全般
                        commandSet.Add(MaterialAnimationGeneralPage.Paste(new GuiObjectGroup(targetDocument), MaterialAnimationGeneralPage.Copy(animation)));

                        // ユーザーデータ
                        commandSet.Add(UserDataPage.Paste(new GuiObjectGroup(targetDocument), UserDataPage.Copy(animation)));
                    }

                    if (sourceCurves == null)
                    {
                        // インポートされた設定にカーブか存在しない
                        return;
                    }

                    // テクスチャファイルのパス
                    if (animation.TexPatterns.Count > 0)
                    {
                        targetDocument.TexPatterns = ObjectUtility.Clone(animation.TexPatterns);
                        targetDocument.ReferenceTexturePaths.Clear();
                        foreach (var texPattern in targetDocument.TexPatterns)
                        {
                            var texture = DocumentManager.Textures.FirstOrDefault(x => x.Name == texPattern.tex_name);
                            if (texture != null)
                            {
                                targetDocument.ReferenceTexturePaths[texPattern.tex_name] = texture.FilePath;
                            }
                            else
                            {
                                targetDocument.ReferenceTexturePaths[texPattern.tex_name] = string.Empty;
                            }
                        }
                    }

                    foreach (var target in targets)
                    {
                        List<IShaderParamMatAnim> targetAnims =
                            targetDocument.IShaderParamMatAnims.Where(x => x.mat_name == target.Name).ToList();

                        // テクスチャパターン
                        foreach (ITexPatternMatAnim targetAnim in targetAnims)
                        {
                            foreach (var anim in sourceCurves.PatternAnimTargets)
                            {
                                int index = targetAnim.PatternAnimTargets.FindIndex(x => x.sampler_name == anim.sampler_name);
                                if (index == -1)
                                {
                                    targetAnim.PatternAnimTargets.Add(ObjectUtility.Clone(anim));
                                }
                                else
                                {
                                    targetAnim.PatternAnimTargets[index] = ObjectUtility.Clone(anim);
                                }
                            }
                        }

                        // シェーダパラメータ
                        foreach (IShaderParamMatAnim targetAnim in targetAnims)
                        {
                            foreach (var anim in sourceCurves.ParamAnims)
                            {
                                // 対応の無いシェーダパラメータは対象外
                                int index = targetAnim.ParamAnims.FindIndex(x => x.id == anim.id);
                                if (index != -1)
                                {
                                    targetAnim.ParamAnims[index] = ObjectUtility.Clone(anim);
                                }
                            }
                        }

                        // ビジビリティ
                        if (sourceCurves.MaterialVisibilityMatAnim.HasKeyFrame)
                        {
                            foreach (MaterialAnimation.PerMaterialAnim targetAnim in targetAnims)
                            {
                                targetAnim.MaterialVisibilityMatAnim =
                                    ObjectUtility.Clone(sourceCurves.MaterialVisibilityMatAnim);
                            }
                        }
                    }
                }
            }
        }

        // ビジビリティーアニメーションの適用
        private void ApplyMaterialVisibilityAnimation(Item item, EditCommandSet commandSet, List<Material> targets, List<AnimationDocument> createdAnimations, List<AnimationDocument> updatedAnimations)
        {
            var animation = (MaterialVisibilityAnimation)item.guiObject;
            var sourceCurves = animation.MaterialVisibilityMatAnims.FirstOrDefault(x => x.mat_name == material.Name);

            var pathNotSpecifiedAnimation = (MaterialVisibilityAnimation)DocumentManager.Animations.FirstOrDefault(x =>
                x.ObjectID == animation.ObjectID && x.Name == animation.Name && string.IsNullOrEmpty(x.FileLocation));

            var materialReferrers = from material in targets
                                  group material by material.Referrers;

            var updated = new HashSet<AnimationDocument>();
            AnimationDocument createdAnimation = null;
            foreach (var group in materialReferrers)
            {
                foreach (var model in group.Key)
                {
                    var targetDocument = (MaterialVisibilityAnimation)model.AllAnimations.FirstOrDefault(x => x.ObjectID == animation.ObjectID && x.Name == animation.Name);

                    if (targetDocument == null)
                    {
                        if (pathNotSpecifiedAnimation == null)
                        {
                            // アニメーションドキュメント作成
                            targetDocument = new MaterialVisibilityAnimation(
                                new mat_visibility_animType()
                                {
                                    mat_visibility_anim_info = ObjectUtility.Clone(animation.Data.mat_visibility_anim_info),
                                },
                                new List<G3dStream>())
                            {
                                Name = animation.Name,
                                FileDotExt = animation.FileDotExt,
                                Comment = animation.Comment,
                            };
                            targetDocument.UpdateSavedData();
                            commandSet.Add(DocumentManager.CreateAddOrRemoveDocumentCommand(Enumerable.Repeat(targetDocument, 1), true).Execute());
                            pathNotSpecifiedAnimation = targetDocument;
                            createdAnimation = targetDocument;
                            createdAnimations.Add(targetDocument);
                        }
                        else
                        {
                            targetDocument = pathNotSpecifiedAnimation;
                        }

                        // バインド
                        BindAnimation(targetDocument, group.ToList(), commandSet);
                    }

                    if (updated.Add(targetDocument))
                    {
                        if (createdAnimation != targetDocument)
                        {
                            updatedAnimations.Add(targetDocument);
                        }

                        // 編集情報
                        var targetGroup = new GuiObjectGroup(targetDocument);
                        commandSet.Add(ObjectPropertyPanel.CreateEditColorEditCommand(targetGroup, targetDocument.EditColor).Execute());
                        commandSet.Add(ObjectPropertyPanel.CreateEditCommentEditCommand(targetGroup, targetDocument.Comment).Execute());
                        commandSet.Add(ObjectPropertyPanel.CreateEditLabelEditCommand(targetGroup, targetDocument.Label).Execute());

                        // 全般
                        commandSet.Add(MaterialVisibilityAnimationGeneralPage.Paste(new GuiObjectGroup(targetDocument), MaterialVisibilityAnimationGeneralPage.Copy(animation)));

                        // ユーザーデータ
                        commandSet.Add(UserDataPage.Paste(new GuiObjectGroup(targetDocument), UserDataPage.Copy(animation)));
                    }

                    var sourceAnim = animation.MaterialVisibilityMatAnims.FirstOrDefault(x => x.mat_name == material.Name);
                    // カーブの編集
                    foreach (var targetMaterial in group)
                    {
                        var curve = ObjectUtility.Clone(sourceAnim);
                        curve.mat_name = targetMaterial.Name;
                        var index = targetDocument.MaterialVisibilityMatAnims.FindIndex(x => x.mat_name == targetMaterial.Name);
                        if (index == -1)
                        {
                            // カーブの追加
                            var list = new List<MaterialVisibilityAnimation.MaterialVisibilityMatAnim>(
                                targetDocument.MaterialVisibilityMatAnims);
                            list.Add(curve);
                            commandSet.Add(CurveEditorPanel.CreateMaterialVisibilityMatAnimsEditCommand(
                                new GuiObjectGroup(targetDocument),
                                targetDocument.ObjectID,
                                list).Execute());
                            commandSet.Add(new GeneralGroupReferenceEditCommand<MaterialVisibilityAnimation.MaterialVisibilityMatAnim>(
                                new GuiObjectGroup(targetDocument),
                                targetDocument.ObjectID,
                                Enumerable.Repeat(curve, 1),
                                (ref GuiObject target, ref object data, ref object swap) =>
                                    targetDocument.UpdateIsModifiedAnimTargetAll()).Execute());
                        }
                        else
                        {
                            // カーブの書き換え
                            commandSet.Add(new GeneralGroupReferenceEditCommand<MaterialVisibilityAnimation.MaterialVisibilityMatAnim>(
                                new GuiObjectGroup(targetDocument),
                                targetDocument.ObjectID,
                                Enumerable.Repeat(curve, 1),
                                delegate (ref GuiObject target, ref object data, ref object swap)
                                {
                                    var visibilityAnimation = (MaterialVisibilityAnimation)target;
                                    swap = visibilityAnimation.MaterialVisibilityMatAnims[index];
                                    visibilityAnimation.MaterialVisibilityMatAnims[index] = (MaterialVisibilityAnimation.MaterialVisibilityMatAnim)data;
                                    targetDocument.UpdateIsModifiedAnimTargetAll();
                                }).Execute());
                        }
                    }
                }
            }
        }

        public void ApplyShaderParameterAnimation(Item item, EditCommandSet commandSet, List<Material> targets, List<AnimationDocument> createdAnimations, List<AnimationDocument> updatedAnimations)
        {
            var animation = (ShaderParameterAnimation)item.guiObject;
            var sourceCurves = animation.ShaderParamAnims.Where(x => x.mat_name == material.Name)
                .SelectMany(x => x.ParamAnims.Where(y => item.children.Any(z => z.apply && z.name == y.id)));

            var pathNotSpecifiedAnimation = (ShaderParameterAnimation)DocumentManager.Animations.FirstOrDefault(x =>
                x.ObjectID == animation.ObjectID && x.Name == animation.Name && string.IsNullOrEmpty(x.FileLocation));

            var materialReferrers = from material in targets
                                  group material by material.Referrers;

            var updated = new HashSet<AnimationDocument>();
            AnimationDocument createdAnimation = null;
            foreach (var group in materialReferrers)
            {
                foreach (var model in group.Key)
                {
                    var targetDocument = (ShaderParameterAnimation)model.AllAnimations.FirstOrDefault(x => x.ObjectID == animation.ObjectID && x.Name == animation.Name);

                    if (targetDocument == null)
                    {
                        if (pathNotSpecifiedAnimation == null)
                        {
                            // アニメーションドキュメント作成
                            var shader_param_anim = new shader_param_animType()
                            {
                                shader_param_anim_info = ObjectUtility.Clone(animation.Data.shader_param_anim_info),
                            };
                            var streams = new List<G3dStream>();
                            targetDocument = animation.ObjectID == GuiObjectID.ColorAnimation ? new ColorAnimation(shader_param_anim, streams) :
                                animation.ObjectID == GuiObjectID.TextureSrtAnimation ? new TextureSrtAnimation(shader_param_anim, streams) :
                                new ShaderParameterAnimation(shader_param_anim, streams);
                            targetDocument.Name = animation.Name;
                            targetDocument.FileDotExt = animation.FileDotExt;
                            targetDocument.Comment = animation.Comment;
                            targetDocument.UpdateSavedData();
                            commandSet.Add(DocumentManager.CreateAddOrRemoveDocumentCommand(Enumerable.Repeat(targetDocument, 1), true).Execute());
                            pathNotSpecifiedAnimation = targetDocument;
                            createdAnimation = targetDocument;
                            createdAnimations.Add(targetDocument);
                        }
                        else
                        {
                            targetDocument = pathNotSpecifiedAnimation;
                        }

                        // バインド
                        BindAnimation(targetDocument, group.ToList(), commandSet);
                    }

                    if (updated.Add(targetDocument))
                    {
                        if (createdAnimation != targetDocument)
                        {
                            updatedAnimations.Add(targetDocument);
                        }

                        // 編集情報
                        var targetGroup = new GuiObjectGroup(targetDocument);
                        commandSet.Add(ObjectPropertyPanel.CreateEditColorEditCommand(targetGroup, targetDocument.EditColor).Execute());
                        commandSet.Add(ObjectPropertyPanel.CreateEditCommentEditCommand(targetGroup, targetDocument.Comment).Execute());
                        commandSet.Add(ObjectPropertyPanel.CreateEditLabelEditCommand(targetGroup, targetDocument.Label).Execute());

                        // 全般
                        commandSet.Add(ShaderParameterAnimationGeneralPage.Paste(new GuiObjectGroup(targetDocument), ShaderParameterAnimationGeneralPage.Copy(animation)));

                        // ユーザーデータ
                        commandSet.Add(UserDataPage.Paste(new GuiObjectGroup(targetDocument), UserDataPage.Copy(animation)));
                    }

                    foreach (var targetMaterial in group)
                    {
                        var targetCurves = targetDocument.ShaderParamAnims.FirstOrDefault(x => x.mat_name == targetMaterial.Name);
                        if (targetCurves == null)
                        {
                            // ShaderParamAnim の追加
                            targetCurves = new ShaderParameterAnimation.ShaderParamAnim()
                            {
                                mat_name = targetMaterial.Name,
                            };

                            //var list = new ShaderParameterAnimation.ShaderParamAnimList(targetDocument.ShaderParamAnims);
                            var list = new ShaderParameterAnimation.ShaderParamAnimList();
                            list.AddRange(targetDocument.ShaderParamAnims);
                            list.Add(targetCurves);
                            commandSet.Add(CurveEditorPanel.CreateShaderParameterAnimsEditCommand(new GuiObjectGroup(targetDocument), targetDocument.ObjectID, list).Execute());

                            // クローンされるので取り直す必要がある
                            targetCurves = targetDocument.ShaderParamAnims.FirstOrDefault(x => x.mat_name == targetMaterial.Name);
                        }

                        var invisibleCurves = new HashSet<string>();
                        if (targetMaterial.MaterialShaderAssign.ShadingModel != null)
                        {
                            foreach (var uniform in targetMaterial.MaterialShaderAssign.ShadingModel.MaterialUniforms())
                            {
                                if (!uniform.IsAnimVisible())
                                {
                                    invisibleCurves.Add(uniform.id);
                                }
                            }
                        }

                        foreach (var curve in sourceCurves)
                        {
                            // 対象外
                            if (invisibleCurves.Contains(curve.id))
                            {
                                continue;
                            }

                            var index = targetCurves.ParamAnims.FindIndex(x => x.id == curve.id);
                            if (index == -1)
                            {
                                // カーブの追加
                                var list = new ShaderParameterAnimation.ParamAnimList();
                                list.AddRange(targetCurves.ParamAnims);
                                list.Add(ObjectUtility.Clone(curve));
                                commandSet.Add(new GeneralGroupReferenceEditCommand<ShaderParameterAnimation.ParamAnimList>(
                                    new GuiObjectGroup(targetDocument),
                                    targetDocument.ObjectID,
                                    Enumerable.Repeat(list, 1),
                                    delegate (ref GuiObject target, ref object data, ref object swap)
                                    {
                                        swap = targetCurves.ParamAnims;
                                        targetCurves.ParamAnims = (ShaderParameterAnimation.ParamAnimList)data;
                                        targetDocument.UpdateIsModifiedAnimTargetAll();
                                    }).Execute());
                            }
                            else
                            {
                                // カーブの書き換え
                                commandSet.Add(new GeneralGroupReferenceEditCommand<ShaderParameterAnimation.ParamAnim>(
                                    new GuiObjectGroup(targetDocument),
                                    targetDocument.ObjectID,
                                    Enumerable.Repeat(ObjectUtility.Clone(curve), 1),
                                    delegate (ref GuiObject target, ref object data, ref object swap)
                                    {
                                        swap = targetCurves.ParamAnims[index];
                                        targetCurves.ParamAnims[index] = (ShaderParameterAnimation.ParamAnim)data;
                                        targetDocument.UpdateIsModifiedAnimTargetAll();
                                    }).Execute());
                            }
                        }
                    }
                }
            }
        }

        public void BindAnimation(AnimationDocument animation, List<Material> targets, EditCommandSet commandSet)
        {
            foreach (var model in targets.SelectMany(x => x.Referrers).Distinct())
            {
                if (model.AllAnimations.Contains(animation))
                {
                    continue;
                }

                commandSet.Add(DocumentManager.CreateAnimationsEditCommand(model.DefaultAnimationSet,
                    model.DefaultAnimationSet.Animations.Concat(Enumerable.Repeat(new AnimationSetItem(animation.FileName, animation.FileLocation), 1)).ToArray(), false).Execute());
                commandSet.Add(DocumentManager.CreateAnimationUpdateBindCommand(model.AllAnimations).Execute());
            }
        }

        private void ApplyTexturePatternAnimation(Item item, EditCommandSet commandSet, List<Material> targets, List<AnimationDocument> createdAnimations, List<AnimationDocument> updatedAnimations, Dictionary<string, string> texturePaths)
        {
            var animation = (TexturePatternAnimation)item.guiObject;
            var sourceCurves = animation.TexPatternMatAnims.Where(x => x.mat_name == material.Name)
                .SelectMany(x => x.PatternAnimTargets.Where(y => item.children.Any(z => z.apply && z.name == y.sampler_name)));

            var pathNotSpecifiedAnimation = (TexturePatternAnimation)DocumentManager.Animations.FirstOrDefault(x =>
                x.ObjectID == animation.ObjectID && x.Name == animation.Name && string.IsNullOrEmpty(x.FileLocation));

            var materialReferrers = targets.GroupBy(material => material.Referrers);

            var updated = new HashSet<AnimationDocument>();
            AnimationDocument createdAnimation = null;
            foreach (var group in materialReferrers)
            {
                foreach (var model in group.Key)
                {
                    var targetDocument = (TexturePatternAnimation)model.AllAnimations.FirstOrDefault(x => x.ObjectID == animation.ObjectID && x.Name == animation.Name);

                    if (targetDocument == null)
                    {
                        if (pathNotSpecifiedAnimation == null)
                        {
                            // アニメーションドキュメント作成
                            targetDocument = new TexturePatternAnimation(
                                new tex_pattern_animType()
                                {
                                    tex_pattern_anim_info = ObjectUtility.Clone(animation.Data.tex_pattern_anim_info),
                                },
                                new List<G3dStream>())
                            {
                                Name = animation.Name,
                                FileDotExt = animation.FileDotExt,
                                Comment = animation.Comment,
                            };
                            targetDocument.UpdateSavedData();
                            commandSet.Add(DocumentManager.CreateAddOrRemoveDocumentCommand(Enumerable.Repeat(targetDocument, 1), true).Execute());
                            pathNotSpecifiedAnimation = targetDocument;
                            createdAnimation = targetDocument;
                            createdAnimations.Add(targetDocument);
                        }
                        else
                        {
                            targetDocument = pathNotSpecifiedAnimation;
                        }

                        // バインド
                        BindAnimation(targetDocument, group.ToList(), commandSet);
                    }

                    if (updated.Add(targetDocument))
                    {
                        if (createdAnimation != targetDocument)
                        {
                            updatedAnimations.Add(targetDocument);
                        }

                        // 編集情報
                        var targetGroup = new GuiObjectGroup(targetDocument);
                        commandSet.Add(ObjectPropertyPanel.CreateEditColorEditCommand(targetGroup, targetDocument.EditColor).Execute());
                        commandSet.Add(ObjectPropertyPanel.CreateEditCommentEditCommand(targetGroup, targetDocument.Comment).Execute());
                        commandSet.Add(ObjectPropertyPanel.CreateEditLabelEditCommand(targetGroup, targetDocument.Label).Execute());

                        // 全般
                        commandSet.Add(TexturePatternAnimationGeneralPage.Paste(new GuiObjectGroup(targetDocument), TexturePatternAnimationGeneralPage.Copy(animation)));

                        // ユーザーデータ
                        commandSet.Add(UserDataPage.Paste(new GuiObjectGroup(targetDocument), UserDataPage.Copy(animation)));
                    }

                    foreach (var targetMaterial in group)
                    {
                        var targetCurves = targetDocument.TexPatternMatAnims.FirstOrDefault(x => x.mat_name == targetMaterial.Name);
                        if (targetCurves == null)
                        {
                            // TexPatternMatAnim の追加
                            targetCurves = new TexturePatternAnimation.TexPatternMatAnim()
                            {
                                mat_name = targetMaterial.Name,
                            };

                            var list = new List<TexturePatternAnimation.TexPatternMatAnim>(targetDocument.TexPatternMatAnims);
                            list.Add(targetCurves);
                            commandSet.Add(CurveEditorPanel.CreateTexturePatternMatAnimsEditCommand(new GuiObjectGroup(targetDocument), list).Execute());

                            // クローンされるので取り直す必要がある
                            targetCurves = targetDocument.TexPatternMatAnims.FirstOrDefault(x => x.mat_name == targetMaterial.Name);
                        }

                        // テクスチャパターンの作成
                        var indices = sourceCurves.SelectMany(x => x.KeyFrames.Select(y => (int)y.Value)).Distinct().OrderBy(x => x);
                        var table = new Dictionary<int, int>();
                        List<TexturePatternAnimation.TexPattern> patterns = new List<TexturePatternAnimation.TexPattern>(targetDocument.TexPatterns);
                        foreach (var index in indices)
                        {
                            if (!table.ContainsKey(index))
                            {
                                if (0 <= index && index < animation.TexPatterns.Count)
                                {
                                    var texName = animation.TexPatterns[index].tex_name;
                                    var texIndex = patterns.FindIndex(x => x.tex_name == texName);
                                    if (texIndex == -1)
                                    {
                                        texIndex = patterns.Count;
                                        patterns.Add(ObjectUtility.Clone(animation.TexPatterns[index]));
                                    }
                                    table.Add(index, texIndex);
                                }
                            }
                        }

                        foreach (var index in indices)
                        {
                            if (!table.ContainsKey(index))
                            {
                                if (index >= 0)
                                {
                                    table.Add(index, table.Count);
                                }
                            }
                        }

                        foreach (var texName in patterns.Select(x => x.tex_name).Except(targetDocument.TexPatterns.Select(x => x.tex_name)))
                        {
                            string path;
                            if (texturePaths.TryGetValue(texName, out path))
                            {
                                commandSet.Add(
                                    TexturePatternAnimationPatternPage.CreateEditCommand_ReferenceTexturePath(new GuiObjectGroup(targetDocument), texName, path).Execute());
                            }
                        }
                        commandSet.Add(TexturePatternAnimationPatternPage.CreateEditCommand_TexPatterns(new GuiObjectGroup(targetDocument), patterns, true).Execute());
                        commandSet.Add(TexturePatternAnimationPatternPage.CreateEditCommand_RemoveTexturePaths(new GuiObjectGroup(targetDocument)).Execute());

                        // カーブの上書き
                        foreach (var curve in sourceCurves)
                        {
                            var modifiedCurve = ObjectUtility.Clone(curve);
                            for (int i = 0; i < modifiedCurve.KeyFrames.Count; i++)
                            {
                                int value = (int)modifiedCurve.KeyFrames[i].Value;
                                modifiedCurve.KeyFrames[i].Value = value < 0 ? value : table[value];
                            }

                            var index = targetCurves.PatternAnimTargets.FindIndex(x => x.sampler_name == curve.sampler_name);
                            if (index == -1)
                            {
                                // カーブの追加
                                var list = new List<TexturePatternAnimation.PatternAnimTarget>(
                                    targetCurves.PatternAnimTargets);
                                list.Add(modifiedCurve);
                                commandSet.Add(new GeneralGroupReferenceEditCommand<List<TexturePatternAnimation.PatternAnimTarget>>(
                                    new GuiObjectGroup(targetDocument),
                                    targetDocument.ObjectID,
                                    Enumerable.Repeat(list, 1),
                                    delegate (ref GuiObject target, ref object data, ref object swap)
                                    {
                                        swap = targetCurves.PatternAnimTargets;
                                        targetCurves.PatternAnimTargets = (List<TexturePatternAnimation.PatternAnimTarget>)data;
                                        targetDocument.UpdateIsModifiedAnimTargetAll();
                                    }).Execute());
                            }
                            else
                            {
                                // カーブの書き換え
                                commandSet.Add(new GeneralGroupReferenceEditCommand<TexturePatternAnimation.PatternAnimTarget>(
                                    new GuiObjectGroup(targetDocument),
                                    targetDocument.ObjectID,
                                    Enumerable.Repeat(modifiedCurve, 1),
                                    delegate (ref GuiObject target, ref object data, ref object swap)
                                    {
                                        swap = targetCurves.PatternAnimTargets[index];
                                        targetCurves.PatternAnimTargets[index] = (TexturePatternAnimation.PatternAnimTarget)data;
                                        targetDocument.UpdateIsModifiedAnimTargetAll();
                                    }).Execute());
                            }
                        }
                    }
                }
            }
        }

        // 適用可能性のチェック
        internal string Validate()
        {
            if (!string.IsNullOrEmpty(material.MaterialShaderAssign.ShaderDefinitionFileName))
            {
                var shaderDefinition = DocumentManager.ShaderDefinitions.FirstOrDefault(x => x.Name == material.MaterialShaderAssign.ShaderDefinitionFileName);
                if (shaderDefinition == null)
                {
                    return string.Format(res.Strings.MaterialTemplate_NoShaderDefinition, material.MaterialShaderAssign.ShaderDefinitionFileName);
                }

                var shadingModel = material.MaterialShaderAssign.ShadingModel;
                if (shadingModel == null)
                {
                    return string.Format(res.Strings.MaterialTemplate_NoShadingModel, material.MaterialShaderAssign.ShaderDefinitionFileName, material.MaterialShaderAssign.ShaderName);
                }
            }
            else
            {
                return res.Strings.MaterialTemplate_ShadingModelNotAssigned;
            }


            return string.Empty;
        }
    }
}
