﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text.RegularExpressions;
using System.Windows.Forms;
using App.Utility;
using App.res;
using nw.g3d.nw4f_3dif;

namespace App.Controls
{
    public partial class EnvRefDialog : OkCancelDialog
    {
        private readonly render_info_arrayType envRefArray_ = null;
        private render_infoType envRefData_ = null;
        private readonly render_infoType orgenvRefData_ = null;

        //// 文字列データの正規表現
        //private static readonly Regex _stringDataRegex =
        //    new Regex(@"\p{IsBasicLatin}+", RegexOptions.Compiled);
        //private static readonly Regex _stringDataRegex =
        //    new Regex("[ 0-9A-Za-z!#$%()*+,\\-./:;=?@\\[\\\\\\]^_{|}]+", RegexOptions.Compiled);
        private static readonly Regex _stringDataRegex =
            new Regex(@"[0-9A-Za-z\-\._]+", RegexOptions.Compiled);


        /// <summary>
        /// コンストラクタ
        /// </summary>
        public EnvRefDialog(render_info_arrayType envRefArray)
            : this(envRefArray, null)
        {
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public EnvRefDialog(render_info_arrayType envRefArray, render_infoType enfRefData)
        {
            InitializeComponent();

            envRefArray_ = envRefArray;

            chkCanEdit.Checked = enfRefData == null;
            pnlCanEdit.Visible = enfRefData != null;

            if (enfRefData == null)
            {
                Height -= pnlCanEdit.Height;
            }

            chkCanEdit_CheckedChanged(null, null);

            if (enfRefData == null)
            {
                envRefData_ = new render_infoType();
                orgenvRefData_ = null;
            }
            else
            {
                envRefData_ = ObjectUtility.Clone(enfRefData);
                orgenvRefData_ = ObjectUtility.Clone(enfRefData);
            }

            UpdateForm(envRefData_);
        }

        private void UpdateForm(render_infoType envRefData)
        {
            // 名前
            {
                tbxName.Text = envRefData.name;
            }

            // 値
            {
                List<string> values = new List<string>();
                {
                    if (envRefData.Value != null)
                    {

                        string[] targetParamArray = G3dDataParser.Tokenize(envRefData.Value);
                        if (targetParamArray != null)
                        {
                            values.AddRange(targetParamArray.ToArray());
                        }
                    }
                }
                tbxValue.Lines = values.ToArray();
            }
        }

        /// <summary>
        /// 結果の取得。
        /// </summary>
        public render_infoType GetResult()
        {
            // 新しくオブジェクトを作成して返す。
            string name = envRefData_.name;
            int dataIndex = envRefData_.index;
            int dataIndexHint = envRefData_.index_hint;

            envRefData_ = new render_infoType();
            envRefData_.name = name;
            envRefData_.index = dataIndex;
            envRefData_.index_hint = dataIndexHint;


            List<string> item = new List<string>();
            string errorMessage = null;
            int index = 0;
            foreach (string value in tbxValue.Lines)
            {
                if (value.Length > 0)
                {
                    string dst = null;
                    if (ParseStringData(value, ref dst, ref errorMessage))
                    {
                        item.Add(dst);
                    }
                }
                index++;
            }

            envRefData_.count = item.Count();
            envRefData_.Value = Utility.IfUtility.MakeArrayString(item);

            return envRefData_;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool OnResultOk()
        {
            // データが変わってなかったらキャンセルとみなす
            if (DataChanged() == false)
            {
                DialogResult = DialogResult.Cancel;
            }

            return true;
        }

        private bool IsValidData(ref string errorMessage)
        {
            foreach (string value in tbxValue.Lines)
            {
                if (value.Length > 0)
                {
                    string dst = null;
                    if (ParseStringData(value, ref dst, ref errorMessage) == false)
                    {
                        return false;
                    }
                }
            }

            return true;
        }

        private bool ParseStringData(string src, ref string dst, ref string errorMessage)
        {
            if (!RegexMatch.Check(src, _stringDataRegex))
            {
                errorMessage = Strings.UserData_Input_String_Invalid_Msg;
                return false;
            }
            else
            {
                dst = src;
                return true;
            }
        }

        /// <summary>
        /// データ変更チェック
        /// </summary>
        private bool DataChanged()
        {
            return GetResult().Equals(orgenvRefData_) == false;
        }

        private void tbxName_TextChanged(object sender, EventArgs e)
        {
            GetResult().name = tbxName.Text;
        }

        private void UserDataDialog_FormClosing(object sender, FormClosingEventArgs e)
        {
            if (DialogResult == DialogResult.OK)
            {
                render_infoType result = GetResult();

                // 名無しチェック
                {
                    if (string.IsNullOrEmpty(result.name))
                    {
                        UIMessageBox.Warning(Strings.EnvRef_Input_Name_Invalid_Msg);
                        e.Cancel = true;
                        return;
                    }
                }

                // 無効な文字を含んでいないかをチェック
                {
                    if (!StringUtility.CheckValidName(result.name, StringUtility.ValidNameRegExpType.NumberAlphabetMoreThan1))
                    {
                        UIMessageBox.Warning(Strings.EnvRef_Input_Name_Invalid_Msg);
                        e.Cancel = true;
                        return;
                    }
                }

                // 重複名チェック
                {
                    if (envRefArray_ != null && envRefArray_.render_info != null)
                    {
                        // 新規の時のみチェックする
                        foreach (render_infoType refEnvData in envRefArray_.render_info)
                        {
                            // 変更時で元々の名前であれば重複とは扱わない。
                            if ((orgenvRefData_ != null) && orgenvRefData_.name == refEnvData.name)
                            {
                                continue;
                            }

                            if (result.name == refEnvData.name)
                            {
                                UIMessageBox.Warning(Strings.EnvRef_Input_Name_Duplicate_Msg);
                                e.Cancel = true;
                                return;
                            }
                        }
                    }
                }

                // 正しいデータかのチェック
                {
                    string errorMessage = null;
                    if (IsValidData(ref errorMessage) == false)
                    {
                        UIMessageBox.Warning(errorMessage);
                        e.Cancel = true;
                        return;
                    }
                }
            }
        }

        private void chkCanEdit_CheckedChanged(object sender, EventArgs e)
        {
            lblName.Enabled = chkCanEdit.Checked;
            tbxName.Enabled = chkCanEdit.Checked;
        }
    }
}
