﻿using System;
using System.Collections.Generic;
using System.Drawing;
using System.Windows.Forms;
using System.Linq;
using ConfigCommon;
using App.Data;
using App.Utility;
using nw.g3d.nw4f_3dif;
using App.ConfigData;

namespace App.Controls
{
    /// <summary>
    /// ドキュメントを選択して閉じるダイアログクラス。
    /// </summary>
    public sealed partial class DocumentsCloseSelectionDialog : OkCancelDialog
    {
        // リスト初期化フラグ
        private bool loaded = false;

        // 閉じるドキュメントリスト
        private readonly List<Document> _closeDocuments = new List<Document>();
        // フォームサイズ
        private static Size _formSize = Size.Empty;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public DocumentsCloseSelectionDialog(IEnumerable<Document> documents, bool closeReference, bool showCloseReference = true)
        {
            InitializeComponent();

            DialogUtility.SetHelp(this, HelpUtility.PageKey.p_file_menu_a_file_close_select);

            foreach (Document document in documents)
            {
                var name = document.FileName + DocumentManager.GetSameNameIndexText(document, true);

                ListViewItem item = new ListViewItem(name);
                item.Tag = document;
                item.Checked = false;
                item.ImageKey = PathUtility.MakeNoFormatIfName(document.FileExt);
                item.SubItems.Add(document.FileDotExt.ToLower());
                lvwFileList.Items.Add(item);
            }
            lvwFileList.Sort(false);

            // チェックコンテキストメニュー
            InitializeContextMenuItem(cmiCheckF3pj, ProjectDocument.DefaultDotExt.Substring(1));
            InitializeContextMenuItem(cmiCheckFmd, G3dPath.ModelExtension.Substring(1));
            InitializeContextMenuItem(cmiCheckFtx, G3dPath.TextureExtension.Substring(1));
            InitializeContextMenuItem(cmiCheckFsk, G3dPath.SkeletalAnimExtension.Substring(1));
            InitializeContextMenuItem(cmiCheckFsp, G3dPath.ShaderParamAnimExtension.Substring(1));
            InitializeContextMenuItem(cmiCheckFcl, G3dPath.ColorAnimExtension.Substring(1));
            InitializeContextMenuItem(cmiCheckFts, G3dPath.TexSrtAnimExtension.Substring(1));
            InitializeContextMenuItem(cmiCheckFtp, G3dPath.TexPatternAnimExtension.Substring(1));
            InitializeContextMenuItem(cmiCheckFvb, G3dPath.BoneVisibilityAnimExtension.Substring(1));
            InitializeContextMenuItem(cmiCheckFvm, G3dPath.MatVisibilityAnimExtension.Substring(1));
            InitializeContextMenuItem(cmiCheckFsh, G3dPath.ShapeAnimExtension.Substring(1));
            InitializeContextMenuItem(cmiCheckFsn, G3dPath.SceneAnimExtension.Substring(1));
            InitializeContextMenuItem(cmiCheckFsd, G3dPath.ShaderDefinitionExtension.Substring(1));
            InitializeContextMenuItem(cmiCheckFmt, G3dPath.MaterialExtension.Substring(1));
            InitializeContextMenuItem(cmiCheckTga, "tga");

            // クリアコンテキストメニュー
            InitializeContextMenuItem(cmiClearF3pj, ProjectDocument.DefaultDotExt.Substring(1));
            InitializeContextMenuItem(cmiClearFmd, G3dPath.ModelExtension.Substring(1));
            InitializeContextMenuItem(cmiClearFtx, G3dPath.TextureExtension.Substring(1));
            InitializeContextMenuItem(cmiClearFsk, G3dPath.SkeletalAnimExtension.Substring(1));
            InitializeContextMenuItem(cmiClearFsp, G3dPath.ShaderParamAnimExtension.Substring(1));
            InitializeContextMenuItem(cmiClearFcl, G3dPath.ColorAnimExtension.Substring(1));
            InitializeContextMenuItem(cmiClearFts, G3dPath.TexSrtAnimExtension.Substring(1));
            InitializeContextMenuItem(cmiClearFtp, G3dPath.TexPatternAnimExtension.Substring(1));
            InitializeContextMenuItem(cmiClearFvb, G3dPath.BoneVisibilityAnimExtension.Substring(1));
            InitializeContextMenuItem(cmiClearFvm, G3dPath.MatVisibilityAnimExtension.Substring(1));
            InitializeContextMenuItem(cmiClearFsh, G3dPath.ShapeAnimExtension.Substring(1));
            InitializeContextMenuItem(cmiClearFsn, G3dPath.SceneAnimExtension.Substring(1));
            InitializeContextMenuItem(cmiClearFsd, G3dPath.ShaderDefinitionExtension.Substring(1));
            InitializeContextMenuItem(cmiClearFmt, G3dPath.MaterialExtension.Substring(1));
            InitializeContextMenuItem(cmiClearTga, "tga");

            lvwFileList_VisibleChanged(lvwFileList, EventArgs.Empty);

            // サイズ設定
            if (_formSize != Size.Empty) { Size = _formSize; }
            FormClosed += delegate { _formSize = Size; };

            cbxCloseReference.Checked = closeReference;
            cbxCloseReference.Visible = showCloseReference;
            UpdateState();
        }

        /// <summary>
        /// コンテキストメニュー項目初期化。
        /// </summary>
        private void InitializeContextMenuItem(ToolStripMenuItem menuItem, string fileExt)
        {
            menuItem.Tag = fileExt;
            menuItem.Enabled = lvwFileList.ContainsItem(fileExt) || lvwFileList.ContainsItem(fileExt + G3dPath.Text) || lvwFileList.ContainsItem(fileExt + G3dPath.Binary);
            menuItem.Image = Const.IconImages[fileExt];
        }

        /// <summary>
        /// 閉じるドキュメントリスト。
        /// </summary>
        public List<Document> CloseDocuments
        {
            get { return _closeDocuments; }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool OnResultOk()
        {
            _closeDocuments.Clear();
            foreach (ListViewItem item in lvwFileList.CheckedItems)
            {
                _closeDocuments.Add((Document)item.Tag);
            }
            return true;
        }

        public bool CloseReference
        {
            get
            {
                return cbxCloseReference.Checked;
            }
        }

        private void lvwFileList_ItemChecked(object sender, ItemCheckedEventArgs e)
        {
            if (loaded)
            {
                UpdateState();
            }
        }

        /// <summary>
        /// カウンタ更新。
        /// </summary>
        private void UpdateState()
        {
            // プロパティグリッドを更新するだけ
            pgdFileCount.SelectedObject = new CloseFileCountPropertyData(lvwFileList);

            // チェックされていなければ OK ボタンは有効にならない
            btnOK.Enabled = lvwFileList.CheckedItems.Count > 0;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnLoad(EventArgs e)
        {
            // 初期化終了
            // ListView.ItemChecked イベントが初回表示時に必ず呼ばれてしまうのを回避...
            loaded = true;
            base.OnLoad(e);
        }

        //---------------------------------------------------------------------
        // チェック状態変更
        private void cmiCheck_Click(object sender, EventArgs e)
        {
            string fileExt = (string)((ToolStripMenuItem)sender).Tag;

            // 全て
            if (fileExt == null)
            {
                lvwFileList.CheckAllItems();
            }
            // 種類別
            else
            {
                lvwFileList.CheckAllItems(fileExt);
                lvwFileList.CheckAllItems(fileExt + G3dPath.Text);
                lvwFileList.CheckAllItems(fileExt + G3dPath.Binary);
            }
            UpdateState();
        }

        private void cmiClear_Click(object sender, EventArgs e)
        {
            string fileExt = (string)((ToolStripMenuItem)sender).Tag;

            // 全て
            if (fileExt == null)
            {
                lvwFileList.ClearAllItems();
            }
            // 種類別
            else
            {
                lvwFileList.ClearAllItems(fileExt);
                lvwFileList.ClearAllItems(fileExt + G3dPath.Text);
                lvwFileList.ClearAllItems(fileExt + G3dPath.Binary);
            }
            UpdateState();
        }

        private void lvwFileList_VisibleChanged(object sender, EventArgs e)
        {
            lvwFileList.BeginInvokeOrExecute(new Action(() =>
                {
                    if (lvwFileList.ClientSize.Width > 0)
                    {
                        clhFileName.Width = lvwFileList.ClientSize.Width - clhFileType.Width;
                    }
                }));
        }
    }
}
