﻿using System;
using System.Collections.Generic;
using System.Drawing;
using System.Windows.Forms;
using System.Linq;
using ConfigCommon;
using App.Data;
using App.Utility;

namespace App.Controls
{
    /// <summary>
    /// ドキュメントを選択して閉じるダイアログクラス。
    /// </summary>
    public sealed partial class DocumentsCloseDialog : YesNoCancelDialog
    {
        // ダイアログの動作をカスタマイズするためのクラス
        public sealed class CustomizeBlock : IDisposable
        {
            public static bool IsDisableCancelButton { get { return params_.Peek().IsDisableCancelButton; } }
            public static bool IsEnabled { get { return params_.Count > 0; } }

            private class Param
            {
                public bool IsDisableCancelButton{ get; set; }
            }
            private static readonly Stack<Param>	params_ = new Stack<Param>();

            public CustomizeBlock(bool isDisableCancelButton = false)
            {
                params_.Push(
                    new Param()
                    {
                        IsDisableCancelButton = isDisableCancelButton
                    }
                );
            }

            public void Dispose()
            {
                params_.Pop();
            }
        }

        // 閉じるドキュメントリスト
        private readonly List<Document> _closeDocuments = new List<Document>();
        // フォームサイズ
        private static Size _formSize = Size.Empty;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public DocumentsCloseDialog(IEnumerable<Document> documents)
        {
            InitializeComponent();

            DialogUtility.SetHelp(this, HelpUtility.PageKey.p_file_menu_a_change_file);

            lvwFileList.SetupCheckAllCheckBox(cbxSelectAll);

            // ファイルリスト構築
            List<Image> images = new List<Image>();
            lvwFileList.SmallImageList = new ImageList();
            foreach (Document document in documents)
            {
                var name = document.FileName + DocumentManager.GetSameNameIndexText(document, true);

                ListViewItem item = new ListViewItem(name);
                item.Tag = document;
                item.Checked = false;
                item.ImageIndex = images.IndexOf(document.ObjectIcon);
                if (item.ImageIndex == -1)
                {
                    item.ImageIndex = images.Count;
                    images.Add(document.ObjectIcon);
                }
                item.SubItems.Add(document.FileDotExt.ToLower());
                lvwFileList.Items.Add(item);
            }
            lvwFileList.SmallImageList = new ImageList();
            lvwFileList.SmallImageList.Images.AddRange(images.ToArray());
            lvwFileList.Sort(false);
            lvwFileList_VisibleChanged(lvwFileList, EventArgs.Empty);
            lvwFileList.CheckBoxes = true;

            // サイズ設定
            if (_formSize != Size.Empty) { Size = _formSize; }
            FormClosed += delegate { _formSize = Size; };

            // カスタマイズ
            if (CustomizeBlock.IsEnabled)
            {
                if (CustomizeBlock.IsDisableCancelButton)
                {
                    btnCancel.Enabled = false;
                    ControlBox = false;
                }
            }
            else
            {
                // デフォルト動作
                btnCancel.Enabled = true;
                ControlBox = true;
            }
        }

        /// <summary>
        /// 閉じるドキュメントリスト。
        /// </summary>
        public List<Document> CloseDocuments
        {
            get { return _closeDocuments; }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool OnResultYes()
        {
            _closeDocuments.Clear();
            foreach (ListViewItem item in lvwFileList.CheckedItems)
            {
                _closeDocuments.Add((Document)item.Tag);
            }
            return true;
        }

        private void lvwFileList_ItemChecked(object sender, ItemCheckedEventArgs e)
        {
            // チェックされていなければ OK ボタンは有効にならない
            btnYes.Enabled = lvwFileList.CheckedItems.Count > 0;
        }

        private void lvwFileList_VisibleChanged(object sender, EventArgs e)
        {
            lvwFileList.BeginInvokeOrExecute(new Action(() =>
                {
                    if (lvwFileList.ClientSize.Width > 0)
                    {
                        clhFileName.Width = lvwFileList.ClientSize.Width - clhFileType.Width;
                    }
                }));
        }
    }
}
