﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using Microsoft.WindowsAPICodePack.Dialogs;

namespace App.Controls
{
    public partial class FileIoRuntimeUserScriptDialog : OkCancelDialog
    {
        /// <summary>
        /// Tag に設定するデータ
        /// </summary>
        public class TagData
        {
            public TeamConfig.FileIo.RuntimeUserScript RuntimeUserScript;

            public TagData(TeamConfig.FileIo.RuntimeUserScript runtimeUserScript)
            {
                RuntimeUserScript = runtimeUserScript;
            }
        }

        const int EM_SETCUEBANNER = 0x1500 + 1;

        private List<Func<bool>> acceptable = new List<Func<bool>>();

        public FileIoRuntimeUserScriptDialog()
        {
            InitializeComponent();
        }

        protected override void OnLoad(EventArgs e)
        {
            base.OnLoad(e);

            var data = (TagData)Tag;
            tbxRuntimeUserScriptName.Text = data.RuntimeUserScript.Name;
            tbxRuntimeUserScriptPath.Text = data.RuntimeUserScript.PathXml;

            // 入力必須項目に設定。
            if (tbxRuntimeUserScriptPath.IsHandleCreated)
            {
                var ptr = System.Runtime.InteropServices.Marshal.StringToBSTR(App.res.Strings.MandatoryField);
                Win32.NativeMethods.SendMessage(tbxRuntimeUserScriptPath.Handle, EM_SETCUEBANNER, IntPtr.Zero, ptr);
                acceptable.Add(() => { return !string.IsNullOrEmpty(tbxRuntimeUserScriptPath.Text); });
            }
        }

        protected override void OnClosing(CancelEventArgs e)
        {
            base.OnClosing(e);

            // e.Cancel 操作は base.OnClosing() 後に行う必要がある。
            if ((DialogResult == DialogResult.OK) && !acceptable.All(x => x()))
            {
                e.Cancel = true;
            }
        }

        protected override void OnClosed(EventArgs e)
        {
            base.OnClosed(e);

            // 変更を適用。
            if (DialogResult == DialogResult.OK)
            {
                // null 文字から空文字への変更を回避。
                Func<string, string, string> CorrectString = (oldValue, newValue) =>
                {
                    return ((oldValue == null) && string.IsNullOrEmpty(newValue)) ? null : newValue;
                };

                var data = (TagData)Tag;
                data.RuntimeUserScript.Name = CorrectString(data.RuntimeUserScript.Name, tbxRuntimeUserScriptName.Text);
                data.RuntimeUserScript.PathXml = CorrectString(data.RuntimeUserScript.PathXml, tbxRuntimeUserScriptPath.Text);
            }
        }

        private void btnRuntimeUserScriptPath_Click(object sender, EventArgs e)
        {
            using (var dialog = new CommonOpenFileDialog()
            {
                Title = res.Strings.IO_SpecifyTargetFile,
                IsFolderPicker = false,
                EnsureFileExists = true
            })
            {
                var initPath = System.Environment.ExpandEnvironmentVariables(tbxRuntimeUserScriptPath.Text);

                if (!string.IsNullOrEmpty(initPath))
                {
                    var initDir = System.IO.Path.GetDirectoryName(initPath);
                    if (!string.IsNullOrEmpty(initDir))
                    {
                        // 期待通りに動作させるにはディレクトリセパレータを正しておく必要がある。
                        dialog.InitialDirectory = initDir.Replace(System.IO.Path.AltDirectorySeparatorChar, System.IO.Path.DirectorySeparatorChar);
                    }

                    var initFile = System.IO.Path.GetFileName(initPath);
                    if (!string.IsNullOrEmpty(initFile))
                    {
                        dialog.DefaultFileName = initFile;
                    }
                }

                if (dialog.ShowDialog(Handle) == CommonFileDialogResult.Ok)
                {
                    tbxRuntimeUserScriptPath.Text = dialog.FileName;
                }
            }
        }
    }
}
