﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;

namespace ColorPicker
{
    public partial class ScaleSliderControl : UserControl
    {
        private bool m_inInit = false;
        private float m_value = 1.0f;
        private bool m_suppressTrackBarEvent = false;
        private bool m_editRight = false;

        // 外部から設定されたスケール非適用のカラー値 (読み取り専用)。
        private float m_colorValue = 1.0f;

        // 外部から設定されたカラーモード (読み取り専用)。
        private ConfigCommon.ColorMode m_colorMode = ConfigCommon.ColorMode.RGBA;

        // 外部から設定された HDR 上限値 (読み取り専用)。
        private float m_hdrUpperBound = float.MaxValue;

        public event ColorScaleChangedEventHandler ValueChanged;

        /// <summary>
        /// スケール値を設定または取得する。
        /// </summary>
        [Category("カスタム")]
        [Browsable(false)]
        public float Value
        {
            get
            {
                return m_value;
            }
        }

        /// <summary>
        /// スケール非適用のカラー値を設定する。
        /// </summary>
        public RGB Color
        {
            set
            {
                m_colorValue = Math.Max(Math.Max(value.R, value.G), value.B);
            }
        }

        /// <summary>
        /// カラーモードを設定する。
        /// </summary>
        public ConfigCommon.ColorMode ColorMode
        {
            set
            {
                m_colorMode = value;
                switch (m_colorMode)
                {
                    case ConfigCommon.ColorMode.RGB:
                    case ConfigCommon.ColorMode.RGBA:
                    case ConfigCommon.ColorMode.LinearRgb:
                    case ConfigCommon.ColorMode.LinearRgba:
                        lblScale.Text = "RGB Scale :";
                        break;
                    case ConfigCommon.ColorMode.HSV:
                    case ConfigCommon.ColorMode.HSVA:
                        lblScale.Text = "V Scale :";
                        break;
                }
            }
        }

        public bool IsLinearColorMode
        {
            get
            {
                return
                    (m_colorMode == ConfigCommon.ColorMode.LinearRgba) ||
                    (m_colorMode == ConfigCommon.ColorMode.LinearRgb);
            }
        }

        /// <summary>
        /// テキストボックスの表示位置を右側に設定する。
        /// </summary>
        [Category("カスタム")]
        [Description("テキストボックスの表示位置を右側に設定する。")]
        [DefaultValue(false)]
        public bool TextBoxIsRight
        {
            get
            {
                return m_editRight;
            }
            set
            {
                m_editRight = value;
                SuspendLayout();
                if (value)
                {
                    uitbScale.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Right)));
                    ftbScale.Location = new System.Drawing.Point(20, 6);
                    uitbScale.Location = new System.Drawing.Point(Size.Width - 52, 4);
                }
                else
                {
                    uitbScale.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left)));
                    ftbScale.Location = new System.Drawing.Point(72, 6);
                    uitbScale.Location = new System.Drawing.Point(16, 4);
                }
                ResumeLayout();
                Invalidate();
            }
        }

        public ScaleSliderControl()
        {
            InitializeComponent();
        }

        /// <summary>
        /// イベントを発生させず、コントロールを初期化する。
        /// </summary>
        /// <param name="color">正規化済みカラー</param>
        /// <param name="colorMode">カラーモード</param>
        /// <param name="value">スケール値</param>
        /// <param name="hdrUpperBound">HDR の上限値</param>
        public void Init(ConfigCommon.ColorMode colorMode, RGB color, float hdrUpperBound, float value)
        {
            m_inInit = true;
            m_colorValue = Math.Max(Math.Max(color.R, color.G), color.B);
            ColorMode = colorMode;
            m_hdrUpperBound = hdrUpperBound;

            // スケール値がトラックバーに収まるように範囲を調整。
            {
                var range = new App.Controls.FloatRangeProperty(0.0f, 2.0f, ftbScale.SmallChange, ftbScale.LargeChange, 0.0f, float.MaxValue, 0.0f, 2.0f);
                while (!range.Contains(value))
                {
                    range.Scale(true);
                }
                using (var block = new App.Controls.UIControlEventSuppressBlock())
                {
                    ftbScale.Minimum = range.Minimum;
                    ftbScale.Maximum = range.Maximum;
                    ftbScale.SmallChange = range.ChangeS;
                    ftbScale.LargeChange = range.ChangeL;
                }
            }

            m_value = value;
            ftbScale.Value = m_value;
            uitbScale.Text = Value.ToString();
            lblMinimum.Text = ftbScale.Minimum.ToString();
            lblMaximum.Text = ftbScale.Maximum.ToString();
            UpdateThumbColor();
            m_inInit = false;
        }

        /// <summary>
        /// スケール値がスライダーの上下限値に収まっているか否かで「つまみ」の色を更新する。
        /// </summary>
        private void UpdateThumbColor()
        {
            // 「つまみ」の位置調整のためだけに用いる ftbScale.Value を引数に渡してはならない。
            // 必ず実際の値である m_value を使用すること。
            UpdateThumbColor(m_value);
        }

        /// <summary>
        /// 引数値がスライダーの上下限値に収まっているか否かで「つまみ」の色を更新する。
        /// </summary>
        private void UpdateThumbColor(float value)
        {
            ftbScale.DrawRedThumb = (value < ftbScale.Minimum) || (value > ftbScale.Maximum);
        }

        #region イベント
        private void uitbScale_KeyPress(object sender, KeyPressEventArgs e)
        {
            TextBox obj = (TextBox)sender;
            if ((e.KeyChar < '0' || e.KeyChar > '9') && e.KeyChar != '\b')
            {
                if ((e.KeyChar != '.' && e.KeyChar != 'E' && e.KeyChar != 'e' && e.KeyChar != '-' && e.KeyChar != '+'))
                {
                    e.Handled = true;
                }
            }
        }

        private void uitbScale_KeyDown(object sender, KeyEventArgs e)
        {
            if (e.KeyCode == Keys.Enter || e.KeyCode == Keys.Return)
            {
                uitbScale_Leave(sender, EventArgs.Empty);
                uitbScale.SelectAll();
            }
        }

        private void uitbScale_Leave(object sender, EventArgs e)
        {
            if (m_inInit)
            {
                return;
            }

            float parsedValue;
            if (!float.TryParse(uitbScale.Text, out parsedValue))
            {
                uitbScale.Text = Value.ToString();
                return;
            }

            // 値確定。
            var value =
                (parsedValue < 0.0f) ? 0.0f :
                ((m_colorValue * parsedValue) > m_hdrUpperBound) ? (m_hdrUpperBound / m_colorValue) :
                parsedValue;

            // 確定値がテキスト値と異なる場合は
            // 確定値でテキストを更新。
            if (value != parsedValue)
            {
                uitbScale.Text = value.ToString();
            }

            if (value != m_value)
            {
                m_value = value;

                bool oSuppressTrackBarEvent = m_suppressTrackBarEvent;
                try
                {
                    m_suppressTrackBarEvent = true;

                    if (m_value > ftbScale.Maximum)
                    {
                        ftbScale.Maximum = m_value;
                        lblMaximum.Text = ftbScale.Maximum.ToString();
                    }
                    ftbScale.Value = m_value;
                }
                finally
                {
                    m_suppressTrackBarEvent = oSuppressTrackBarEvent;
                }
                if (ValueChanged != null)
                {
                    ValueChanged(this, new ColorScaleChangedEventArgs(Value, true));
                }
            }
            UpdateThumbColor();
        }

        private void ftbScale_SequentialValueChanged(object sender, App.Controls.SequentialValueChangedEventArgs e)
        {
            if (m_inInit || m_suppressTrackBarEvent)
            {
                return;
            }

            // 値確定。
            var value =
                (ftbScale.Value < 0.0f) ? 0.0f :
                ((m_colorValue * ftbScale.Value) > m_hdrUpperBound) ? (m_hdrUpperBound / m_colorValue) :
                ftbScale.Value;

            // 確定値でトラックバーを調整。
            if (value != ftbScale.Value)
            {
                bool oSuppressTrackBarEvent = m_suppressTrackBarEvent;
                try
                {
                    m_suppressTrackBarEvent = true;

                    if (value > ftbScale.Maximum)
                    {
                        ftbScale.Maximum = value;
                        lblMaximum.Text = ftbScale.Maximum.ToString();
                    }
                    ftbScale.Value = value;
                }
                finally
                {
                    m_suppressTrackBarEvent = oSuppressTrackBarEvent;
                }
            }

            // テキストボックスに確定値を反映。
            uitbScale.Text = value.ToString();

            // 変更を適用。
            {
                m_value = value;

                UpdateThumbColor();

                if (ValueChanged != null)
                {
                    ValueChanged(this, new ColorScaleChangedEventArgs(Value, !e.Changing));
                }
            }
        }

        private void btnRangeDown_Click(object sender, EventArgs e)
        {
            var range = new App.Controls.FloatRangeProperty(ftbScale.Minimum, ftbScale.Maximum, ftbScale.SmallChange, ftbScale.LargeChange, 0.0f, float.MaxValue, 0.0f, 2.0f);
            if (range.Scale(false))
            {
                using (var block = new App.Controls.UIControlEventSuppressBlock())
                {
                    ftbScale.Minimum = range.Minimum;
                    ftbScale.Maximum = range.Maximum;
                    ftbScale.SmallChange = range.ChangeS;
                    ftbScale.LargeChange = range.ChangeL;
                    lblMinimum.Text = ftbScale.Minimum.ToString();
                    lblMaximum.Text = ftbScale.Maximum.ToString();
                }

                // 内部値がスライダーバーの範囲外になった場合は、
                // 「つまみ」位置をスライダーバーの上下限位置に移動させる。
                // このときスライダーバーの値が実際のスケール値とは一致しなくなるが
                // 実際のスケール値は専用変数 m_value で管理しているため問題ない。
                var v = Math.Min(Math.Max(m_value, ftbScale.Minimum), ftbScale.Maximum);
                if (v != ftbScale.Value)
                {
                    bool oSuppressTrackBarEvent = m_suppressTrackBarEvent;
                    try
                    {
                        m_suppressTrackBarEvent = true;
                        ftbScale.Value = v;
                    }
                    finally
                    {
                        m_suppressTrackBarEvent = oSuppressTrackBarEvent;
                    }
                }

                UpdateThumbColor();
            }
        }

        private void btnRangeUp_Click(object sender, EventArgs e)
        {
            var range = new App.Controls.FloatRangeProperty(ftbScale.Minimum, ftbScale.Maximum, ftbScale.SmallChange, ftbScale.LargeChange, 0.0f, float.MaxValue, 0.0f, 2.0f);
            if (range.Scale(true))
            {
                using (var block = new App.Controls.UIControlEventSuppressBlock())
                {
                    ftbScale.Minimum = range.Minimum;
                    ftbScale.Maximum = range.Maximum;
                    ftbScale.SmallChange = range.ChangeS;
                    ftbScale.LargeChange = range.ChangeL;
                    lblMinimum.Text = ftbScale.Minimum.ToString();
                    lblMaximum.Text = ftbScale.Maximum.ToString();
                }

                // 内部値がスライダーバーの範囲外になった場合は、
                // 「つまみ」位置をスライダーバーの上下限位置に移動させる。
                // このときスライダーバーの値が実際のスケール値とは一致しなくなるが
                // 実際のスケール値は専用変数 m_value で管理しているため問題ない。
                var v = Math.Min(Math.Max(m_value, ftbScale.Minimum), ftbScale.Maximum);
                if (v != ftbScale.Value)
                {
                    bool oSuppressTrackBarEvent = m_suppressTrackBarEvent;
                    try
                    {
                        m_suppressTrackBarEvent = true;
                        ftbScale.Value = v;
                    }
                    finally
                    {
                        m_suppressTrackBarEvent = oSuppressTrackBarEvent;
                    }
                }

                UpdateThumbColor();
            }
        }
        #endregion
    }
}
