﻿using System;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;
using App.ConfigData;
using App.Data;
using App.Utility;
using ColorPicker;

namespace App.Controls
{
    /// <summary>
    /// カラーピッカーダイアログクラス。
    /// </summary>
    public sealed partial class ColorPickerDialog : ModelessDialog
    {
        // ダイアログインスタンス
        private static ColorPickerDialog _dialog = null;
        // カラーピッカーアダプタ
        private static ColorPickerAdapter _adapter = null;
        // ダイアログに設定したカラーパレット
        private static AppConfig.ColorPicker _config = null;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        private ColorPickerDialog()
        {
            InitializeComponent();

            // オーナー
            Owner = TheApp.MainFrame;

            // 状態復帰
            {
                // アダプタが未設定なので、ここでは通常の設定を用いる。
                // Location は ColorPicker と MarkColorPicker で共有している。
                _config = ApplicationConfig.Setting.ColorPicker;

                for (int i = 0; i < AppConfig.ColorPicker.CustomColorCount; i++)
                {
                    ctlColorPicker.CustomColors[i] = new RGB(
                        _config.CustomColor[i][0],
                        _config.CustomColor[i][1],
                        _config.CustomColor[i][2]);
                }
                ColorMode = _config.ColorMode;
                ValueMode = _config.ValueMode;

                if (_config.Location != null && Screen.AllScreens.Any(x => x.WorkingArea.Contains(_config.Location.Value)))
                {
                    DesktopLocation = (Point)_config.Location;
                }
                else
                {
                    StartPosition = FormStartPosition.CenterParent;
                }
            }

            // イベント登録
            ctlColorPicker.ColorChanged += Event_CtlColorPicker_ColorChanged;
            tmrDisconnect.Tick += Event_TmrDisconnect_Tick;
        }

        private bool _IsEditColor = false;
        public bool IsEditColor
        {
            set
            {
                _IsEditColor = value;
                btnNullify.Visible = value;

                ctlColorPicker.ShowCurrentColorComment = !value;
            }
        }

        /// <summary>
        /// ガンマ補正
        /// </summary>
        public bool IsDefaultLinear
        {
            get
            {
                return ctlColorPicker.IsDefaultLinear;
            }
            set
            {
                ctlColorPicker.IsDefaultLinear = value;
            }
        }

        /// <summary>
        /// ctlColorPicker と scaleSliderPanel に値を設定する
        /// </summary>
        /// <param name="color">デフォルトカラーモードでのカラー</param>
        /// <param name="hdrUpperBound">HDR 上限値</param>
        private void Setup(RGB color, float hdrUpperBound)
        {
            // 作業用カラーでは HDR を扱わない。
            // そのためスケール値スライダーが不要であり表示しないので正規化しない。
            var scale = _IsEditColor ? 1.0f : Math.Max(Math.Max(Math.Max(color.R, color.G), color.B), 1.0f);
            var ncolor = new RGB(
                color.A,
                color.R / scale,
                color.G / scale,
                color.B / scale);
            if ((!IsDefaultLinear && ctlColorPicker.SrgbColor != ncolor) || (IsDefaultLinear && ctlColorPicker.LinearColor != ncolor))
            {
                ctlColorPicker.SetColor(ncolor);
            }

            switch (ColorMode)
            {
                case ConfigCommon.ColorMode.LinearRgba:
                case ConfigCommon.ColorMode.LinearRgb:
                    if (IsDefaultLinear)
                    {
                        // Linear -> Linear なので変換不要。
                        scaleSliderPanel.Init(ColorMode, ctlColorPicker.LinearColor, hdrUpperBound, scale);
                    }
                    else
                    {
                        // Srtb -> Linear に変換。
                        var srgbColor = new RGB(
                            ctlColorPicker.SrgbColor.A,
                            ctlColorPicker.SrgbColor.R * scale,
                            ctlColorPicker.SrgbColor.G * scale,
                            ctlColorPicker.SrgbColor.B * scale);
                        var linearColor = srgbColor.ToLinearFromSrgb();
                        scale = Math.Max(Math.Max(Math.Max(linearColor.R, linearColor.G), linearColor.B), 1.0f);
                        var nlinearColor = new RGB(
                            linearColor.A,
                            linearColor.R / scale,
                            linearColor.G / scale,
                            linearColor.B / scale);
                        scaleSliderPanel.Init(ColorMode, nlinearColor, hdrUpperBound, scale);
                    }
                    break;
                default:
                    if (IsDefaultLinear)
                    {
                        // Linear -> Srgb に変換。
                        var linearColor = new RGB(
                            ctlColorPicker.LinearColor.A,
                            ctlColorPicker.LinearColor.R * scale,
                            ctlColorPicker.LinearColor.G * scale,
                            ctlColorPicker.LinearColor.B * scale);
                        var srgbColor = linearColor.ToSrgbFromLinear();
                        scale = Math.Max(Math.Max(Math.Max(srgbColor.R, srgbColor.G), srgbColor.B), 1.0f);
                        var nsrgbColor = new RGB(
                            srgbColor.A,
                            srgbColor.R / scale,
                            srgbColor.G / scale,
                            srgbColor.B / scale);
                        scaleSliderPanel.Init(ColorMode, nsrgbColor, hdrUpperBound, scale);
                    }
                    else
                    {
                        // Srgb -> Srgb なので変換不要。
                        scaleSliderPanel.Init(ColorMode, ctlColorPicker.SrgbColor, hdrUpperBound, scale);
                    }
                    break;
            }

            ctlColorPicker.HDRUpperBound = hdrUpperBound;
            ctlColorPicker.ColorScale = scale;
        }

        private Size initialSize;
        private Point initialScaleSliderPanelLocation;
        private bool sizeFitted = false;
        public void FitSize()
        {
            if (!sizeFitted)
            {
                initialSize = Size;
                initialScaleSliderPanelLocation = scaleSliderPanel.Location;
                sizeFitted = true;
            }

            if (ColorMode == ConfigCommon.ColorMode.A)
            {
                // アルファのみの場合はスケールを非表示＆無効化。
                Size = initialSize;
                scaleSliderPanel.Visible = false;
                scaleSliderPanel.Enabled = false;
                scaleSliderPanel.Location = initialScaleSliderPanelLocation;
                btnNullify.Visible = false;
                btnNullify.Enabled = false;
            }
            else if (_IsEditColor)
            {
                // 作業用カラーの場合はスケールを非表示＆無効化し、
                // 「作業用カラー削除」ボタンを表示＆有効化。
                Size = initialSize;
                scaleSliderPanel.Visible = false;
                scaleSliderPanel.Enabled = false;
                scaleSliderPanel.Location = initialScaleSliderPanelLocation;
                btnNullify.Visible = true;
                btnNullify.Enabled = true;
            }
            else
            {
                scaleSliderPanel.Visible = true;
                scaleSliderPanel.Enabled = true;
                btnNullify.Visible = false;
                btnNullify.Enabled = false;

                if (ColorPickerControl.HasAlpha(ColorMode))
                {
                    Size = new Size(initialSize.Width, initialSize.Height + scaleSliderPanel.Height);
                    scaleSliderPanel.Location = new Point(initialScaleSliderPanelLocation.X, initialScaleSliderPanelLocation.Y + 24);
                }
                else
                {
                    Size = new Size(initialSize.Width, initialSize.Height + scaleSliderPanel.Height - 24);
                    scaleSliderPanel.Location = new Point(initialScaleSliderPanelLocation.X, initialScaleSliderPanelLocation.Y);
                }
            }
        }

        /// <summary>
        /// カラーモード。
        /// </summary>
        private ConfigCommon.ColorMode ColorMode
        {
            get { return ctlColorPicker.ColorMode; }
            set { ctlColorPicker.ColorMode = value; }
        }

        /// <summary>
        /// 値モード。
        /// </summary>
        private ConfigCommon.ValueMode ValueMode
        {
            get { return ctlColorPicker.ValueMode; }
            set { ctlColorPicker.ValueMode = value; }
        }

        #region アダプタ接続
        /// <summary>
        /// 接続開始。
        /// </summary>
        public static void StartConnection(ColorPickerAdapter adapter, bool activate = true)
        {
            // ダイアログがなければ作成
            if (_dialog == null)
            {
                _dialog = new ColorPickerDialog();
            }

            // アダプタ設定
            SetAdapter(adapter);

            // 表示時にアダプタにロストフォーカス処理をさせない
            if (activate)
            {
                adapter.HandleLostFocus = false;
            }

            // 表示
            _dialog.Show();
            if (activate)
            {
                _dialog.Activate();
            }
        }

        /// <summary>
        /// 接続切り替え。
        /// </summary>
        public static void ChangeConnection(ColorPickerAdapter adapter)
        {
            // ダイアログ表示中なら
            if (_dialog != null)
            {
                // 接続アダプタを切り替える
                SetAdapter(adapter);
            }
        }

        /// <summary>
        /// 接続更新。
        /// </summary>
        public static void UpdateConnection(ColorPickerAdapter adapter)
        {
            // 接続済みのアダプタなら
            if (_dialog != null && _adapter == adapter)
            {
                // カラー更新
                _dialog.Setup(ToColor(adapter.Color, 1.0f), adapter.HDRUpperBound);
                _dialog.Text = adapter.ColorPickerText;
            }
        }

        private static RGB ToColor(RgbaColor color, float scale)
        {
            return new RGB(
                Math.Min(Math.Max(color.A, 0), 1),
                Math.Max(color.R / scale, 0),
                Math.Max(color.G / scale, 0),
                Math.Max(color.B / scale, 0));
        }

        /// <summary>
        /// 接続終了。
        /// </summary>
        public static void EndConnection(ColorPickerAdapter adapter)
        {
            // 接続済みのアダプタなら
            if (_dialog != null && _adapter == adapter)
            {
                // 接続解除開始
                if (!_dialog.tmrDisconnect.Enabled)
                {
                    if (_adapter != null)
                    {
                        // 切断通知
                        _adapter.OnBeforeDisconnect();

                        TimerStartedAdapter = _adapter;

                        // これ以降の変更が通知されないようにする
                        _adapter = null;
                    }


                    // すぐに再接続されるかもしれないので少し待つ
                    _dialog.tmrDisconnect.Start();
                }
            }
        }

        public static ColorPickerAdapter Adapter { get { return _adapter; } }

        public static ColorPickerAdapter TimerStartedAdapter { get; private set; }

        public static void StopTimer(ColorPickerAdapter adapter)
        {
            if (adapter == TimerStartedAdapter)
            {
                _dialog.tmrDisconnect.Stop();
                TimerStartedAdapter = null;
            }
        }

        /// <summary>
        /// アダプタ設定。
        /// </summary>
        private static void SetAdapter(ColorPickerAdapter adapter)
        {
            // アダプタ設定
            //if (_adapter != adapter)
            if (_adapter != adapter && _adapter != null)
            {
                // 切断通知
                _adapter.OnBeforeDisconnect();
            }

            if (adapter != null)
            {
                var config = adapter.IsMarkColor ? ApplicationConfig.Setting.MarkColorPicker : ApplicationConfig.Setting.ColorPicker;
                if ((config != _config) && (_dialog != null))
                {
                    // 設定が切り替わるので、古い設定を保存。
                    if (_config != null)
                    {
                        for (int i = 0; i < AppConfig.ColorPicker.CustomColorCount; i++)
                        {
                            // 切り捨てておけば誤差に悩まされない
                            _config.CustomColor[i] = ToRgbaColor(_dialog.ctlColorPicker.CustomColors[i], 1).ToArray();
                        }
                        _config.ColorMode = _dialog.ColorMode;
                        _config.ValueMode = _dialog.ValueMode;
                    }

                    // 新しい設定に切り替える。
                    _config = config;
                    _dialog.ColorMode = _config.ColorMode;
                    _dialog.ValueMode = _config.ValueMode;
                    for (int i = 0; i < AppConfig.ColorPicker.CustomColorCount; i++)
                    {
                        _dialog.ctlColorPicker.CustomColors[i] = new RGB(
                            _config.CustomColor[i][0],
                            _config.CustomColor[i][1],
                            _config.CustomColor[i][2]);
                    }

                    // パレットを更新。
                    _dialog.ctlColorPicker.Invalidate(true);
                }
            }

            {
                _adapter = adapter;
                ConfigCommon.ColorMode colorMode = _dialog.ColorMode;
                if (adapter.EnableAlpha)
                {
                    if (colorMode == ConfigCommon.ColorMode.RGB) { colorMode = ConfigCommon.ColorMode.RGBA; }
                    if (colorMode == ConfigCommon.ColorMode.HSV) { colorMode = ConfigCommon.ColorMode.HSVA; }
                    if (colorMode == ConfigCommon.ColorMode.LinearRgb) { colorMode = ConfigCommon.ColorMode.LinearRgba; }
                }
                else
                {
                    if (colorMode == ConfigCommon.ColorMode.RGBA) { colorMode = ConfigCommon.ColorMode.RGB; }
                    if (colorMode == ConfigCommon.ColorMode.HSVA) { colorMode = ConfigCommon.ColorMode.HSV; }
                    if (colorMode == ConfigCommon.ColorMode.LinearRgba) { colorMode = ConfigCommon.ColorMode.LinearRgb; }
                }
                _dialog.IsDefaultLinear = adapter.IsDefaultLinear;
                _dialog.ColorMode = colorMode;
                _dialog.Text = adapter.ColorPickerText;
                _dialog.IsEditColor = adapter.NullifyVisible;
                _dialog.Setup(ToColor(adapter.Color, 1.0f), adapter.HDRUpperBound);
                _dialog.FitSize();
            }

            // タイマ起動中なら停止
            if (_dialog.tmrDisconnect.Enabled)
            {
                _dialog.tmrDisconnect.Stop();
                TimerStartedAdapter = null;
            }
        }

        public static void StopTimer()
        {
            _dialog.tmrDisconnect.Stop();
            TimerStartedAdapter = null;
        }

        #endregion

        #region オーバーライド
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnDeactivate(EventArgs e)
        {
            // Close後も呼ばれるのでインスタンスの有無を見て判断
            if (_dialog != null)
            {
                if (UIMessageBox.DeactivatedForm != this)
                {
                    // 接続解除開始
                    if (!tmrDisconnect.Enabled)
                    {
                        if (_adapter != null)
                        {
                            // 切断通知
                            _adapter.OnBeforeDisconnect();

                            TimerStartedAdapter = _adapter;

                            // これ以降の変更が通知されないようにする
                            // _config は OnClosed で参照するので null を設定しない。
                            _adapter = null;
                        }

                        // すぐに再接続されるかもしれないので少し待つ
                        tmrDisconnect.Start();
                    }
                }
            }
            base.OnDeactivate(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnClosed(EventArgs e)
        {
            // 状態保存
            if (_config != null)
            {
                for (int i = 0; i < AppConfig.ColorPicker.CustomColorCount; i++)
                {
                    // 切り捨てておけば誤差に悩まされない
                    _config.CustomColor[i] = ToRgbaColor(ctlColorPicker.CustomColors[i], 1).ToArray();
                }
                _config.ColorMode = ColorMode;
                _config.ValueMode = ValueMode;

                // ロケーションは共有する。
                ApplicationConfig.Setting.ColorPicker.Location = DesktopLocation;
                ApplicationConfig.Setting.MarkColorPicker.Location = DesktopLocation;
            }

            // タイマー終了
            if (tmrDisconnect.Enabled)
            {
                tmrDisconnect.Stop();
                TimerStartedAdapter = null;
            }

            if (_adapter != null)
            {
                // 切断通知
                _adapter.OnBeforeDisconnect();

                // これ以降の変更が通知されないようにする
                _adapter = null;
            }

            // 切断してインスタンス破棄
            _dialog  = null;
            Debug.Assert(TimerStartedAdapter == null);

            base.OnClosed(e);
        }

        protected override bool ProcessCmdKey(ref Message msg, Keys keyData)
        {
            if (_adapter != null && _adapter.CanUndoRedo != null && !_adapter.CanUndoRedo() && ((keyData & Keys.Z) == Keys.Z || (keyData & Keys.Control) == Keys.Control))
            {
                return true;
            }

            return base.ProcessCmdKey(ref msg, keyData);
        }
        #endregion

        #region イベントハンドラ
        //---------------------------------------------------------------------------
        // カラー変更
        private void Event_CtlColorPicker_ColorChanged(object sender, ColorChangedEventArgs e)
        {
            // アダプタにイベントを発行させる
            if (_adapter != null)
            {
                var scale = scaleSliderPanel.Value;

                RGB color;
                if (_adapter.IsDefaultLinear == e.IsLinear)
                {
                    color = new RGB(
                        e.Color.A,
                        e.Color.R * scale,
                        e.Color.G * scale,
                        e.Color.B * scale);
                }
                else if (e.IsLinear)
                {
                    color = new RGB(
                        e.Color.A,
                        e.Color.R * scale,
                        e.Color.G * scale,
                        e.Color.B * scale).ToSrgbFromLinear();
                }
                else
                {
                    color = new RGB(
                        e.Color.A,
                        e.Color.R * scale,
                        e.Color.G * scale,
                        e.Color.B * scale).ToLinearFromSrgb();
                }

                var rgba = ToRgbaColor(color, 1.0f);
                _adapter.InvokeColorEdit(new ColorEditEventArgs(rgba, e.EditFixed, e.RedEdited, e.GreenEdited, e.BlueEdited, e.AlphaEdited));
            }
        }

        /// <summary>
        /// カラーモード変更イベントハンドラ
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void ctlColorPicker_ColorModeChanged(object sender, ColorModeChangedEventArgs e)
        {
            if (_adapter != null)
            {
                if (e.IsLinear)
                {
                    // Srgb -> Linear に変換。
                    if (!scaleSliderPanel.IsLinearColorMode)
                    {
                        // 計算誤差をできるだけ抑えるために、スケール値 1.0 の場合は不要な変換を行わない。
                        var scale = scaleSliderPanel.Value;
                        if (Math.Abs(scale - 1.0f) > float.Epsilon)
                        {
                            var srgbColor = new RGB(
                                ctlColorPicker.SrgbColor.A,
                                ctlColorPicker.SrgbColor.R * scale,
                                ctlColorPicker.SrgbColor.G * scale,
                                ctlColorPicker.SrgbColor.B * scale);
                            Setup(IsDefaultLinear ? srgbColor.ToLinearFromSrgb() : srgbColor, _adapter.HDRUpperBound);
                        }
                        else
                        {
                            Setup(IsDefaultLinear ? ctlColorPicker.LinearColor : ctlColorPicker.SrgbColor, _adapter.HDRUpperBound);
                        }
                    }
                }
                else
                {
                    // Linear -> Srgb に変換。
                    if (scaleSliderPanel.IsLinearColorMode)
                    {
                        // 計算誤差をできるだけ抑えるために、スケール値 1.0 の場合は不要な変換を行わない。
                        var scale = scaleSliderPanel.Value;
                        if (Math.Abs(scale - 1.0f) > float.Epsilon)
                        {
                            var linearColor = new RGB(
                                ctlColorPicker.LinearColor.A,
                                ctlColorPicker.LinearColor.R * scale,
                                ctlColorPicker.LinearColor.G * scale,
                                ctlColorPicker.LinearColor.B * scale);
                            Setup(IsDefaultLinear ? linearColor : linearColor.ToSrgbFromLinear(), _adapter.HDRUpperBound);
                        }
                        else
                        {
                            Setup(IsDefaultLinear ? ctlColorPicker.LinearColor : ctlColorPicker.SrgbColor, _adapter.HDRUpperBound);
                        }
                    }
                }
                scaleSliderPanel.ColorMode = e.ColorMode;
                FitSize();
            }
        }

        /// <summary>
        /// カラーピッカーでの色変更イベントハンドラ
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void ctlColorPicker_ColorChanged(object sender, ColorChangedEventArgs e)
        {
            // スライダー値の上限は HDRUpperBound を考慮する。
            // スケール非適用の e.Color をスライダーに渡して、スライダー値の上限調整時に参照する。
            if (scaleSliderPanel.IsLinearColorMode)
            {
                // スライダーがリニアモードなので、リニアカラーを設定する。
                scaleSliderPanel.Color = e.IsLinear ? e.Color : e.Color.ToLinearFromSrgb();
            }
            else
            {
                // スライダーが非リニアモードなので、非リニアカラーを設定する。
                scaleSliderPanel.Color = e.IsLinear ? e.Color.ToSrgbFromLinear() : e.Color;
            }
        }

        /// <summary>
        /// パレットカラー挿入イベントハンドラ
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void ctlColorPicker_PaletteColorInserted(object sender, PaletteColorInsertedEventArgs e)
        {
            if (_adapter != null)
            {
                // パレットには正規化された色が登録される。
                // それをユーザーに示すために、UI の値を正規化したものに更新する。
                if (scaleSliderPanel.IsLinearColorMode)
                {
                    if (Math.Max(Math.Max(ctlColorPicker.LinearColor.R, ctlColorPicker.LinearColor.G), ctlColorPicker.LinearColor.B) > 1.0f)
                    {
                        var scale = scaleSliderPanel.Value;
                        var linearColor = new RGB(
                            ctlColorPicker.LinearColor.A,
                            ctlColorPicker.LinearColor.R * scale,
                            ctlColorPicker.LinearColor.G * scale,
                            ctlColorPicker.LinearColor.B * scale);
                        var color = _adapter.IsDefaultLinear ? linearColor : linearColor.ToSrgbFromLinear();
                        Setup(color, _adapter.HDRUpperBound);
                    }
                }
                else
                {
                    if (Math.Max(Math.Max(ctlColorPicker.SrgbColor.R, ctlColorPicker.SrgbColor.G), ctlColorPicker.SrgbColor.B) > 1.0f)
                    {
                        var scale = scaleSliderPanel.Value;
                        var srgbColor = new RGB(
                            ctlColorPicker.SrgbColor.A,
                            ctlColorPicker.SrgbColor.R * scale,
                            ctlColorPicker.SrgbColor.G * scale,
                            ctlColorPicker.SrgbColor.B * scale);
                        var color = _adapter.IsDefaultLinear ? srgbColor.ToLinearFromSrgb() : srgbColor;
                        Setup(color, _adapter.HDRUpperBound);
                    }
                }
            }
        }

        private void scaleSliderPanel_ValueChanged(object sender, ColorScaleChangedEventArgs e)
        {
            ctlColorPicker.ColorScale = e.ColorScale;
            bool isLinear = scaleSliderPanel.IsLinearColorMode;
            var color = isLinear ? ctlColorPicker.LinearColor : ctlColorPicker.SrgbColor;
            Event_CtlColorPicker_ColorChanged(sender, new ColorChangedEventArgs(color, e.EditFixed, isLinear));
        }

        public static RgbaColor ToRgbaColor(RGB rgb, float scale)
        {
            var rgba = new RgbaColor()
            {
                R = Math.Max(rgb.R * scale, 0),
                G = Math.Max(rgb.G * scale, 0),
                B = Math.Max(rgb.B * scale, 0),
                A = Math.Max(Math.Min(1, rgb.A), 0),
            };
            return rgba;
        }

        //---------------------------------------------------------------------------
        // タイマー
        private void Event_TmrDisconnect_Tick(object sender, EventArgs e)
        {
            // ダイアログ終了
            tmrDisconnect.Stop();
            TimerStartedAdapter = null;

            Close();
        }

        private void btnNullify_Click(object sender, EventArgs e)
        {
            if (_adapter != null)
            {
                _adapter.InvokeNullify();
            }
            _dialog.Close();
        }
        #endregion
    }
}
