﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Drawing.Imaging;
using System.Windows.Forms;
using App.Utility;

namespace App.Controls
{
    /// <summary>
    /// カラー成分。
    /// </summary>
    public enum ColorComponent
    {
        /// <summary>Ｒ成分。</summary>
        R,
        /// <summary>Ｇ成分。</summary>
        G,
        /// <summary>Ｂ成分。</summary>
        B,
        /// <summary>Ａ成分。</summary>
        A,
    }

    /// <summary>
    /// カラー編集パネル用スライダークラス。
    /// </summary>
    [ToolboxItem(true)]
    [DefaultEvent("SequentialValueChanged")]
    public sealed class ColorEditSlider : UnselectableControl
    {
        // 成分
        private ColorComponent _component = ColorComponent.R;
        // 値
        private float _value = 0.0f;
        // 参照値０
        private float _refValue0 = 0.0f;
        // 参照値１
        private float _refValue1 = 0.0f;

        // ビットマップ
        private readonly Bitmap _bitmap = new Bitmap(256, 1, PixelFormat.Format24bppRgb);
        // ビットマップ更新停止中
        private bool _suspendUpdateBitmap = false;
        // ドラッグ中
        private bool _dragging = false;
        // ドラッグボタン
        private MouseButtons _draggingButton = MouseButtons.None;
        // つまみサイズ
        private readonly Size _thumbSize = new Size(10, 5);

        public bool IsLinear = false;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ColorEditSlider()
        {
            UpdateBitmap();
        }

        #region プロパティ
        /// <summary>
        /// 成分。
        /// </summary>
        [DefaultValue(ColorComponent.R)]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("扱うカラー成分を示します。")]
        public ColorComponent Component
        {
            get { return _component; }
            set
            {
                if (_component != value)
                {
                    _component = value;
                    UpdateBitmap();
                }
            }
        }

        /// <summary>
        /// 値。
        /// </summary>
        [DefaultValue(0)]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        public float Value
        {
            get { return _value; }
            set
            {
                _value = value;
                Invalidate();
            }
        }

        /// <summary>
        /// 参照値０。
        /// </summary>
        [DefaultValue(0)]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("ビットマップ作成用の参照値０。")]
        public float RefValue0
        {
            get { return _refValue0; }
            set
            {
                _refValue0 = value;
                UpdateBitmap();
            }
        }

        /// <summary>
        /// 参照値１。
        /// </summary>
        [DefaultValue(0)]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("ビットマップ作成用の参照値１。")]
        public float RefValue1
        {
            get { return _refValue1; }
            set
            {
                _refValue1 = value;
                UpdateBitmap();
            }
        }

        /// <summary>
        /// 最小値。
        /// </summary>
        private float minimum_ = 0.0f;
        public float Minimum
        {
            get
            {
                return minimum_;
            }

            set
            {
                //Debug.Assert(value < Maximum);

                minimum_ = Math.Min(Maximum, value);
                UpdateBitmap();
            }
        }

        /// <summary>
        /// 最大値。
        /// </summary>
        private float maximum_ = 1.0f;
        public float Maximum
        {
            get
            {
                return maximum_;
            }

            set
            {
                //Debug.Assert(value > Minimum);

                maximum_ = Math.Max(Minimum, value);
                UpdateBitmap();
            }
        }

        /// <summary>
        /// トラック領域。
        /// </summary>
        private Rectangle TrackBounds
        {
            get
            {
                Rectangle rcThumb = ClientRectangle;

                // 横幅を縮小
                rcThumb.Inflate((_thumbSize.Width / 2) * -1, 0);

                // 縦幅を縮小
                const int trackHeight = 8;
                rcThumb.Y = (DefaultSize.Height - trackHeight) / 2;
                rcThumb.Height = trackHeight;

                return rcThumb;
            }
        }

        public bool ReadOnly
        {
            get
            {
                return readOnly_;
            }

            set
            {
                readOnly_ = value;
                Invalidate();
            }
        }
        private bool readOnly_ = false;
        #endregion

        #region イベント
        //---------------------------------------------------------------------
        private static readonly object EVENT_SEQUENTIALVALUECHANGED = new object();

        //---------------------------------------------------------------------
        private static readonly object EVENT_DRAGBEGIN = new object();

        /// <summary>
        /// 値変更イベント。
        /// </summary>
        [Category(UIControlHelper.OriginalEventCategoryName)]
        [Description("値が変更された時に発生します。")]
        public event SequentialValueChangedEventHandler SequentialValueChanged
        {
            add { base.Events.AddHandler(EVENT_SEQUENTIALVALUECHANGED, value); }
            remove { base.Events.RemoveHandler(EVENT_SEQUENTIALVALUECHANGED, value); }
        }

        /// <summary>
        /// ドラッグ開始イベント。
        /// </summary>
        [Category(UIControlHelper.OriginalEventCategoryName)]
        [Description("ドラッグが開始された時に発生します。")]
        public event EventHandler DragBegin
        {
            add { base.Events.AddHandler(EVENT_DRAGBEGIN, value); }
            remove { base.Events.RemoveHandler(EVENT_DRAGBEGIN, value); }
        }

        /// <summary>
        /// 値変更ハンドラ。
        /// </summary>
        private void OnSequentialValueChanged(SequentialValueChangedEventArgs e)
        {
            var handler = (SequentialValueChangedEventHandler)base.Events[EVENT_SEQUENTIALVALUECHANGED];
            if (handler != null) { handler(this, e); }
        }

        private void OnDragBegin(EventArgs e)
        {
            var handler = (EventHandler)base.Events[EVENT_DRAGBEGIN];
            if (handler != null) { handler(this, e); }
        }

        #endregion

        /// <summary>
        /// ビットマップの更新処理を停止。
        /// </summary>
        public void SuspendUpdateBitmap()
        {
            if (!_suspendUpdateBitmap)
            {
                _suspendUpdateBitmap = true;
            }
        }

        /// <summary>
        /// ビットマップの更新処理を再開。
        /// </summary>
        public void ResumeUpdateBitmap()
        {
            if (_suspendUpdateBitmap)
            {
                _suspendUpdateBitmap = false;
                UpdateBitmap();
            }
        }

        /// <summary>
        /// ビットマップ更新。
        /// </summary>
        private void UpdateBitmap()
        {
            // 更新停止中なら何もしない
            if (_suspendUpdateBitmap) { return; }

            var refValue0 = (int)(255 * _refValue0);
            MathUtility.Clamp(ref refValue0, 0, 255);

            var refValue1 = (int)(255 * _refValue1);
            MathUtility.Clamp(ref refValue1, 0, 255);

            using (var bp = new BitmapProcess(_bitmap))
            {
                unsafe
                {
                    var width = _bitmap.Width;

                    // PixelFormat.24bppRgb は BGR の順にバイトが並ぶ
                    // bmpPtr[x * 3 + 2] -> R
                    // bmpPtr[x * 3 + 1] -> G
                    // bmpPtr[x * 3 + 0] -> B
                    var bmpPtr = (byte*)(void*)bp.BitmapData.Scan0;
                    {
                        switch (_component)
                        {
                            case ColorComponent.R:
                            {
                                var g = RefValue0;
                                var b = RefValue1;

                                for (var x = 0; x != width; ++ x)
                                {
                                    var r = (Maximum - Minimum) * ((float)x / width) + Minimum;

                                    // 位置 x における rgb の最大値を取得し、それを表示色の調整に用いる。
                                    var maxCompo = Math.Max(Math.Max(r, g), b);
                                    var scale = (maxCompo > 1.0f) ? (1 / maxCompo) : 1.0f;

                                    bmpPtr[x * 3 + 2] = (byte)Math.Max(0.0f, Math.Min(255.0f, r * scale * 255.0f));
                                    bmpPtr[x * 3 + 1] = (byte)Math.Max(0.0f, Math.Min(255.5f, g * scale * 255.0f));
                                    bmpPtr[x * 3 + 0] = (byte)Math.Max(0.0f, Math.Min(255.5f, b * scale * 255.0f));
                                }
                                break;
                            }

                            case ColorComponent.G:
                            {
                                var r = RefValue0;
                                var b = RefValue1;

                                for (var x = 0; x != width; ++ x)
                                {
                                    var g = (Maximum - Minimum) * ((float)x / width) + Minimum;

                                    // 位置 x における rgb の最大値を取得し、それを表示色の調整に用いる。
                                    var maxCompo = Math.Max(Math.Max(r, g), b);
                                    var scale = (maxCompo > 1.0f) ? (1 / maxCompo) : 1.0f;

                                    bmpPtr[x * 3 + 2] = (byte)Math.Max(0.0f, Math.Min(255.0f, r * scale * 255.0f));
                                    bmpPtr[x * 3 + 1] = (byte)Math.Max(0.0f, Math.Min(255.5f, g * scale * 255.0f));
                                    bmpPtr[x * 3 + 0] = (byte)Math.Max(0.0f, Math.Min(255.5f, b * scale * 255.0f));
                                }
                                break;
                            }

                            case ColorComponent.B:
                            {
                                var r = RefValue0;
                                var g = RefValue1;

                                for (var x = 0; x != width; ++ x)
                                {
                                    var b = (Maximum - Minimum) * ((float)x / width) + Minimum;

                                    // 位置 x における rgb の最大値を取得し、それを表示色の調整に用いる。
                                    var maxCompo = Math.Max(Math.Max(r, g), b);
                                    var scale = (maxCompo > 1.0f) ? (1 / maxCompo) : 1.0f;

                                    bmpPtr[x * 3 + 2] = (byte)Math.Max(0.0f, Math.Min(255.0f, r * scale * 255.0f));
                                    bmpPtr[x * 3 + 1] = (byte)Math.Max(0.0f, Math.Min(255.5f, g * scale * 255.0f));
                                    bmpPtr[x * 3 + 0] = (byte)Math.Max(0.0f, Math.Min(255.5f, b * scale * 255.0f));
                                }
                                break;
                            }

                            case ColorComponent.A:
                            {
                                for (var x = 0; x != width; ++ x)
                                {
                                    var a = (Maximum - Minimum) * ((float)x / width) + Minimum;

                                    bmpPtr[x * 3 + 2] = (byte)Math.Max(0.0f, Math.Min(255.0f, a * 255.0f));
                                    bmpPtr[x * 3 + 1] = (byte)Math.Max(0.0f, Math.Min(255.0f, a * 255.0f));
                                    bmpPtr[x * 3 + 0] = (byte)Math.Max(0.0f, Math.Min(255.0f, a * 255.0f));
                                }
                                break;
                            }
                        }
                    }
                }
            }

            // 再描画が必要
            Invalidate();
        }

        /// <summary>
        /// 位置→値への変換。
        /// </summary>
        private float PositionToValue(int position)
        {
            Rectangle rcTrack = TrackBounds;

            float min = Minimum;
            float max = Maximum;

            // 左端を含む外側
            if (position <= rcTrack.Left)
            {
                return min;
            }
            // 右端を含む外側
            if (position >= (rcTrack.Right -1))
            {
                return max;
            }

            // 内側
            // 適当な桁で丸める
            var range = max - min;
            if (range < 0.00001f)
            {
                range = 0.00001f;
            }

            var decimals = (int)Math.Ceiling(2 - Math.Log10(range));
            var value = (float)MathUtility.InterpolateRound(rcTrack.Left, rcTrack.Right, position, min, max, decimals);

            if (value >= max) { value = max; }
            return value;
        }

        /// <summary>
        /// 値→位置への変換。
        /// </summary>
        private int ValueToPosition(float value)
        {
            Rectangle rcTrack = TrackBounds;

            float min = Minimum;
            float max = Maximum;

            // 最小値
            if (value <= min)
            {
                return rcTrack.Left;
            }
            // 最大値
            if (value >= max)
            {
                return (rcTrack.Right - 1);
            }

            // 中間値
            var position = (int)MathUtility.Interpolate(min, max, value, rcTrack.Left, rcTrack.Right);
            if (position >= (rcTrack.Right - 1)) { position = (rcTrack.Right - 1); }
            return position;
        }

        #region オーバーライド
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override Size DefaultSize
        {
            get { return new Size(128, 21); }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void Dispose(bool disposing)
        {
            if (disposing)
            {
                _bitmap.Dispose();
            }
            base.Dispose(disposing);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnPaint(PaintEventArgs e)
        {
            Graphics g = e.Graphics;
            g.Clear(SystemColors.Control);

            if (Enabled || ReadOnly)
            {
                Rectangle rcTrack = TrackBounds;

                //-------------------------------------------------------------
                // トラック領域
                {
                    InterpolationMode imBack = g.InterpolationMode;
                    g.InterpolationMode = InterpolationMode.NearestNeighbor;
                    using (var attr = new ImageAttributes())
                    {
                        attr.SetGamma(IsLinear ? 1/2.2f: 1);
                        g.DrawImage(_bitmap, rcTrack, -0.5f, -0.5f, _bitmap.Width, _bitmap.Height, GraphicsUnit.Pixel, attr);
                    }
                    g.InterpolationMode = imBack;
                    GraphicsUtility.Draw3DRectangle(g, SystemPens.ControlDark, SystemPens.ControlLightLight, rcTrack);
                }

                //-------------------------------------------------------------
                // つまみ
                {
                    int position = ValueToPosition(Value);

                    var points = new Point[3];
                    points[0] = new Point(position, rcTrack.Bottom - 1);
                    points[1] = new Point(position - (_thumbSize.Width / 2), points[0].Y + _thumbSize.Height);
                    points[2] = new Point(position + (_thumbSize.Width / 2), points[0].Y + _thumbSize.Height);

                    g.FillPolygon(
                        ReadOnly								   ? SystemBrushes.Control :
                        ((Value >= Minimum) && (Value <= Maximum)) ? Brushes.White :
                                                                     Brushes.Red,
                        points
                    );

                    g.DrawLine(Pens.Black, points[0].X,     points[0].Y, points[1].X + 1, points[1].Y - 1);
                    g.DrawLine(Pens.Black, points[0].X,     points[0].Y, points[2].X - 1, points[2].Y - 1);
                    g.DrawLine(Pens.Black, points[1].X + 1, points[1].Y, points[2].X - 1, points[2].Y);
                }
            }
            else
            {
                GraphicsUtility.DrawRectangle(g, SystemPens.GrayText, TrackBounds);
            }
            base.OnPaint(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnMouseDown(MouseEventArgs e)
        {
            if (ReadOnly == false)
            {
                // 中ボタンでもドラッグ可能
                if (!_dragging && (e.Button == MouseButtons.Left || e.Button == MouseButtons.Middle))
                {
                    // ドラッグ開始
                    _dragging = true;
                    _draggingButton = e.Button;
                    Capture = true;

                    OnDragBegin(EventArgs.Empty);

                    // 値変更
                    Value = PositionToValue(e.X);
                    Invalidate();

                    var eventArgs = new SequentialValueChangedEventArgs(true)
                    {
                        MouseEventArgs = e
                    };
                    OnSequentialValueChanged(eventArgs);
                }
            }

            base.OnMouseDown(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnMouseMove(MouseEventArgs e)
        {
            if (ReadOnly == false)
            {
                // ドラッグ中
                if (_dragging)
                {
                    // 値変更（変更時のみ）
                    float value = PositionToValue(e.X);
                    if (Value != value)
                    {
                        Value = value;
                        Invalidate();

                        var eventArgs = new SequentialValueChangedEventArgs(true)
                        {
                            MouseEventArgs = e
                        };
                        OnSequentialValueChanged(eventArgs);
                    }
                }
            }

            base.OnMouseMove(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnMouseUp(MouseEventArgs e)
        {
            if (ReadOnly == false)
            {
                if (_dragging && e.Button == _draggingButton)
                {
                    // ドラッグ終了
                    _dragging = false;
                    _draggingButton = MouseButtons.None;
                    Capture = false;

                    // 変更確定イベント発行
                    var eventArgs = new SequentialValueChangedEventArgs(false)
                    {
                        MouseEventArgs = e
                    };
                    OnSequentialValueChanged(eventArgs);
                }
            }

            base.OnMouseUp(e);
        }
        #endregion
    }
}
