﻿using System;
using System.Linq;
using System.Windows.Forms;
using App.Data;
using App.Utility;
using App.res;
using ConfigCommon;
using System.Diagnostics;

namespace App.Controls
{
    /// <summary>
    /// アニメーション作成ダイアログクラス。
    /// </summary>
    public sealed partial class AnimationCreateDialog : OkCancelDialog
    {
        private readonly GuiObjectID id_;
        private AnimationSet animationSet;
        private Model ownerModel;
        private string postFix_;
        private string postFixTypeString_;
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public AnimationCreateDialog(AnimationSet document, GuiObject ownerDocument, GuiObjectID id, string postFix, string postFixTypeString)
        {
            InitializeComponent();

            DialogUtility.SetHelp(this, HelpUtility.PageKey.p_file_view_a_new_anim);

            id_ = id;
            postFix_ = postFix;
            postFixTypeString_ = postFixTypeString;

            animationSet = document;
            ownerModel = ownerDocument as Model;
            if (document != null)
            {
                var guiObject = document as GuiObject;

                var name =
                    ((guiObject.ObjectID == GuiObjectID.AnimationSet) ?
                        ownerDocument.Name :
                        guiObject.Name) + postFix;

                var nameBase = name;

                int i = 1;
                while (DocumentManager.Animations.Any(x => x.ObjectID == id && x.Name == name))
                {
                    name = nameBase + i + postFix;
                    ++ i;
                }

                cmbName.Items.Add(name);

                if (cmbName.Items.Count > 0)
                {
                    cmbName.SelectedIndex = 0;
                }
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool OnResultOk()
        {
            // アニメーション名の整合確認
            string name = AnimationName;

            // 空文字
            if (name == string.Empty)
            {
                UIMessageBox.Warning(Strings.AnimationCreateDialog_NameEmpty);
                return false;
            }

            if (!name.EndsWith(postFix_))
            {
                UIMessageBox.Warning(string.Format(Strings.Name_PostFix, postFixTypeString_, postFix_));
                AnimationName += postFix_;
                return false;
            }

            // 名前の重複チェック
            if (ObjectIDUtility.CanOpenSameNameFile(id_))
            {
                var exta = "." + ObjectIDUtility.IdToExt(id_).First(x => x.EndsWith("a"));
                var extb = "." + ObjectIDUtility.IdToExt(id_).First(x => x.EndsWith("b"));
                if (DocumentManager.Objects(id_).OfType<Document>().Any(
                    x => x.FileName == name + extb && string.IsNullOrEmpty(x.FileLocation)))
                {
                    UIMessageBox.Warning(Strings.Rename_NameDuplicateEmptyPath, name + extb);
                    return false;
                }

                if (ownerModel != null)
                {
                    var file = ownerModel.AnimationSetsWithDefault.SelectMany(x => x.Animations)
                        .FirstOrDefault(x => x.Name == name + exta || x.Name == name + extb);
                    if (file.Name != null)
                    {
                        UIMessageBox.Warning(Strings.Rename_NameDuplicateUnderModel, ownerModel.Name, file.Name);
                        return false;
                    }
                }
            }
            // 同名チェック
            else if (DocumentManager.Animations.Any(x => x.ObjectID == id_ && x.Name == name))
            {
                UIMessageBox.Warning(Strings.IO_Name_Collision, name);
                return false;
            }

            // 正規表現
            if (RegexMatch.Check(name, "[0-9A-Za-z\\-\\._]+") == false)
            {
                UIMessageBox.Warning(string.Format(Strings.IO_Name_Invalid, name));
                return false;
            }

            return true;
        }

        /// <summary>
        /// アニメーション名。
        /// </summary>
        public string AnimationName { get { return cmbName.Text; } set { cmbName.Text = value; } }

        /// <summary>
        /// フレーム数。
        /// </summary>
        public int FrameCount { get { return tbxFrameCount.Value; } }

        /// <summary>
        /// ループ。
        /// </summary>
        public bool IsLoop { get { return rbtLoopOn.Checked; } }

        private void ReturnKeyDown(object sender, EventArgs e)
        {
            DialogResult = DialogResult.OK;
        }
        private void cmiCopy_Click(object sender, EventArgs e)
        {
            App.Utility.ClipboardUtility.SetDataObject(tbxFrameCount.Value.ToString());
        }

        private void cmiPaste_Click(object sender, EventArgs e)
        {
            var iData = App.Utility.ClipboardUtility.GetDataObject();
            if (iData == null || !iData.GetDataPresent(DataFormats.Text)) return;
            int val;
            if (StringUtility.TrySafeIntParse((String)iData.GetData(DataFormats.Text), out val))
            {
                tbxFrameCount.Value = val;
            }
        }

        private void cmiMaxValue_Click(object sender, EventArgs e)
        {
            tbxFrameCount.Value = tbxFrameCount.Maximum;
        }
    }
}

